/* abuild.c
*
* analytic building routine
*
* use resultants to build atoms based on 3 or more distances
*
* use distance/planarity to build based on 1 or 2 distances
*
*  requires at least one atom not 0 0 0
* and active.
*/

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Constants ****/

#define   MAX_LOCAL             800


void AMMP_FASTCALL a_build(FILE *op, int nused, AMMP_VFUNC pot[], AMMP_FFUNC force[], int ncycles,
                           int low_serial, int high_serial, int echo)
{
  AMMP_ATOM     *ap, *bp, *bps[MAX_LOCAL];
  int           numatom;
  int           i, j, k, icycle;
  int           iatom, inbps;
  float         x, y, z, r;
  float         a, b, c, d;
  float         honker[MAX_LOCAL][3];
  float         m[82], v[MAX_LOCAL];
  float         m2[82], v2[10], m3[10];
  float         rvec[3], svec[3];
  int           (*lp[3])(), (*lf[3])(), ilocal;
  int           use_noel, use_bond,use_angle, use_hybrid, use_nonbon;

#ifdef VEGAZZ
  int           lastcycle;

  int           nupdat    = GetNupdat();
  int           supdat    = GetSupdat();
#endif

  /**** Check the number of atoms ****/

  numatom = a_number();
  if (numatom < 2) {
    aaerror("At least two atoms are required for abuild");
    return;
  }

  /**** Check the parameters ****/

  if (high_serial <= 0 ) {
    for(i = 0; i < numatom; i++) {
      ap = a_next(i);
      if (high_serial < ap -> serial) high_serial = ap -> serial;
    } /* End of for (i) */
  }

  if (low_serial > high_serial) {
    i = low_serial;
    low_serial = high_serial;
    high_serial = i;
  }

  if (ncycles < 1) ncycles = 1;
#ifdef VEGAZZ
  lastcycle = ncycles - 1;
#endif

  /**** Check if at least one atom isn't at (0, 0, 0) ****/

  j = 0;
  k = 0;
  for(i = 0; i < numatom; i++) {
    ap = a_next(i);
    if ((ap -> serial < low_serial) || (ap -> serial > high_serial)) continue;
    if ((ap -> x == 0.0f) && (ap -> y == 0.0f) && (ap -> z == 0.0f)) ++j;
    ++k;
  } /* End of for (i) */

  if (k > MAX_LOCAL) {
    aaerror("Too many atoms for abuild (max. %d)", MAX_LOCAL);
    return;
  }

  if (j == k) {
#ifndef VEGAZZ
    if (echo)
#endif
      fprintf(op, "  Abuild      : All atoms are at (0, 0, 0). Moving the first one at (1, 0, 0)\n");
    a_next(0) -> x = 1.0f;
  }

  use_bond   = FALSE;
  use_angle  = use_bond;
  use_noel   = use_bond;
  use_hybrid = use_bond;
  use_nonbon = use_bond;
  for(i = 0; i < nused; i++) {
    if (pot[i] == v_bond    ) use_bond   = (1==1);
    if (pot[i] == v_mmbond  ) use_bond   = (1==1);
    if (pot[i] == v_angle   ) use_angle  = (1==1);
    if (pot[i] == v_c_angle ) use_angle  = (1==1);
    if (pot[i] == v_mmangle ) use_angle  = (1==1);
    if (pot[i] == v_noel    ) use_noel   = (1==1);
    if (pot[i] == v_hybrid  ) use_hybrid = (1==1);
    if (pot[i] == v_hard    ) use_nonbon = (1==1);
    if (pot[i] == v_nonbon  ) use_nonbon = (1==1);
    if (pot[i] == u_v_nonbon) use_nonbon = (1==1);
  } /* End of for (i) */

  ilocal = 0;
  if (use_bond) {
    lf[ilocal  ] = (int (*)())f_bond;
    lp[ilocal++] = (int (*)())v_bond;
  }
  if (use_angle) {
    lf[ilocal  ] = (int (*)())f_angle;
    lp[ilocal++] = (int (*)())v_angle;
  }
  if (use_noel) {
    lf[ilocal  ] = (int (*)())f_noel;
    lp[ilocal++] = (int (*)())v_noel;
  }
  if (ilocal == 0) return; /* no distances will be generated*/

  for(icycle = 0; icycle < ncycles; icycle++) {
#ifdef VEGAZZ
    if ((supdat) && ((!(icycle % supdat)) || (icycle == lastcycle)))
#else
    if (echo)
#endif
      fprintf(op, "  Abuild %5d:\n", icycle);

    ap = a_next(-1);
    for(iatom = 0; iatom < numatom; iatom++) {
      if ((!ap -> active) || (ap -> serial < low_serial) ||
          (ap -> serial > high_serial)) goto SKIP;

      /**** Zero the gsdg return registers ****/

      for(j = 0; j < numatom; j++) {
        bp       = a_next(j);
        bp -> vx = 0.0f;
        bp -> vy = 0.0f;
      } /* End of for (j) */

      if (use_bond ) gsdg_bond(ap);
      if (use_angle) gsdg_angle(ap);
      if (use_noel ) gsdg_noel(ap);

      /**** Not-so abortive attempt to add contacts (oh well) ****/

      if (use_nonbon) {
        for(j = 0; j < numatom; j++) {
          bp = a_next(j);
          if (bp != ap) {
            if (bp -> vx == 0.0f) {
              x = ap -> x - bp -> x;
              y = ap -> y - bp -> y;
              z = ap -> z - bp -> z;
              r = x * x + y * y + z * z;
              if ( r < 8.0f) {          /* was 8 */
                bp -> vy = -10.0f;
                bp -> vx = 16.0f;
              }
            }
          }
        } /* End of for (j) */
      }
      inbps = 0;
      for(j = 0; j < numatom; j++) {
        bp = a_next(j);
        if ((bp -> vx > 0.0f) && (bp != ap) &&
            ((bp -> x != 0.0f) || (bp -> y != 0.0f) || (bp -> z != 0.0f) || (!bp -> active))) {
          bps[inbps++] = bp;
          if (inbps == MAX_LOCAL) break;
        } else {

          /**** It didn't make the cut so kill it ****/

          bp -> vx = -1.0f;
          bp -> vy = -1.0f;
        }
      } /* End of for (j) */

#ifdef VEGAZZ
      if ((supdat) && ((!(icycle % supdat)) || (icycle == lastcycle))) {
#else
      if (echo) {
#endif
        fprintf(op, "    %s %d inbps %d", ap -> name, ap -> serial, inbps);
        if (inbps < 4) fprintf(op, "\n");
      }

      switch(inbps){
      case 0: /* nada no distances to use */
        break;

      case 1:
        rand3(&x, &y, &z);
        bp = bps[0];
        r  = sqrt(bp -> vx);
        ap -> x = bp -> x + r * x;
        ap -> y = bp -> y + r * y;
        ap -> z = bp -> z + r * z;
        break;

      case 2: /* numerical solution is simplest because not unique */
      case 3:
        if ((ap -> x == 0.0f) && (ap -> y == 0.0f) && (ap -> z == 0.0f)) {
          ap -> z = bps[0] -> z;
          ap -> x = bps[1] -> x;
          ap -> y = bps[1] -> y - bps[0] -> y;
        }

        /**** Cut from gsdg ****/

        for(j = 0; j < 10; j++) {
          rvec[0]  = 0.0f;
          rvec[1]  = 0.0f;
          rvec[2]  = 0.0f;
          rand3(&svec[0], &svec[1], &svec[2]);

          x        = abuild_line_search(svec, &y, ap, bps, inbps);
          rvec[0] += y * svec[0];
          rvec[1] += y * svec[1];
          rvec[2] += y * svec[2];
          rand3(&svec[0], &svec[1], &svec[2]);

          x        = abuild_line_search(svec, &y, ap, bps, inbps);
          rvec[0] += y * svec[0];
          rvec[1] += y * svec[1];
          rvec[2] += y * svec[2];
          rand3(&svec[0], &svec[1], &svec[2]);

          x        = abuild_line_search(svec, &y, ap, bps, inbps);
          rvec[0] += y * svec[0];
          rvec[1] += y * svec[1];
          rvec[2] += y * svec[2];

          x        = abuild_line_search(rvec, &y, ap, bps, inbps);
          ap -> x += y * rvec[0];
          ap -> y += y * rvec[1];
          ap -> z += y * rvec[2];
        } /* End of for (j) */
        break;

//      case 4:
      default:
        a  = bps[0] -> vx;
        x  = bps[0] -> x;
        y  = bps[0] -> y;
        z  = bps[0] -> z;
        a -= x * x + y * y + z * z;
        for(i = 1; i < inbps; i++) {
          d = bps[i]->vx;
          x = bps[i]->x;
          y = bps[i]->y;
          z = bps[i]->z;
          d -= x*x + y*y + z*z;
          v[i - 1] = d-a;
          honker[i-1][0] = -2.0f * (bps[i] -> x - bps[0]-> x);
          honker[i-1][1] = -2.0f * (bps[i] -> y - bps[0]-> y);
          honker[i-1][2] = -2.0f * (bps[i] -> z - bps[0]-> z);
        } /* End of for (i) */

        for(i = 0; i < 9; i++) m2[i] = 0.0f;
        for(i = 0; i < 3; i++) v2[i] = 0.0f;
        for(i = 0; i < 3; i++) {
          for(j = 0; j < 3; j++) {
            for(k = 0; k < (inbps - 1); k++)
              m2[i * 3 + j] += honker[k][i] * honker[k][j];
          } /* End of for (j) */
        } /* End of for (i) */

        for(i = 0; i < 3; i++) {
          for(k = 0; k < (inbps - 1); k++)
            v2[i] += honker[k][i] * v[k];
        } /* End of for (i) */

        jacobi(m2, m, 3, 100, 1.e-10);

#ifdef VEGAZZ
        if ((supdat) && ((!(icycle % supdat)) || (icycle == lastcycle)))
#else
        if (echo)
#endif
          fprintf(op, " eig^2 %.4f %.4f %.4f\n", m2[0], m2[4], m2[8]);

        if (m2[0] > 0.1f) m2[0] = 1.0f / m2[0];
        else m2[0] = 0.0f;
        if (m2[4] > 0.1f) m2[4] = 1.0f / m2[4];
        else m2[4] = 0.0f;
        if (m2[8] > 0.1f) m2[8] = 1.0f / m2[8];
        else m2[8] = 0.0f;
        a = m2[0];
        b = m2[4];
        c = m2[8];

        for(i = 0; i < 9; i++) {
          m2[i] = 0.0f;
          m3[i] = 0.0f;
        } /* End of for (i) */

        for(i = 0; i < 3; i++) {
          for(j = 0; j < 3; j++) {
            m2[i] += m[j * 3 + i] * v2[j];
          } /* End of for (j) */
        } /* End of for (i) */

        m2[0] *= a;
        m2[1] *= b;
        m2[2] *= c;
        for(i = 0; i < 3; i++) {
          for(j = 0; j < 3; j++) {
            m3[i] += m[i * 3 + j] * m2[j];
          } /* End of for (j=) */
        } /* End of for (i) */

        ap -> x = m3[0];
        ap -> y = m3[1];
        ap -> z = m3[2];

        /**** Cut from gsdg ****/

        if (use_hybrid) gsdg_hybrid(ap);

        for(j = 0; j < 10; j++) {
          rvec[0]  = 0.0f;
          rvec[1]  = 0.0f;
          rvec[2]  = 0.0f;
          rand3(&svec[0], &svec[1], &svec[2]);

          x        = abuild_line_search(svec, &y, ap, bps, inbps);
          rvec[0] += y * svec[0];
          rvec[1] += y * svec[1];
          rvec[2] += y * svec[2];
          rand3(&svec[0], &svec[1], &svec[2]);

          x        = abuild_line_search(svec, &y, ap, bps, inbps);
          rvec[0] += y * svec[0];
          rvec[1] += y * svec[1];
          rvec[2] += y * svec[2];
          rand3(&svec[0], &svec[1], &svec[2]);

          x        = abuild_line_search(svec, &y, ap, bps, inbps);
          rvec[0] += y * svec[0];
          rvec[1] += y * svec[1];
          rvec[2] += y * svec[2];

          x        = abuild_line_search( rvec,&y,ap,bps,inbps);
          ap -> x += y * rvec[0];
          ap -> y += y * rvec[1];
          ap -> z += y * rvec[2];
        } /* End of dor (j) */
        break;
      } /* end of switch*/

SKIP:
      ap = ap -> next;
    } /* End of for (iatom) */
#ifdef VEGAZZ
    if ((nupdat) && ((!(icycle % nupdat)) || (icycle == lastcycle))) send_all_atoms();
#endif
  } /* End of for (icycle) */
}


float AMMP_FASTCALL abuild_line_search(float vect[3], float *step, AMMP_ATOM *who,
                                       AMMP_ATOM *bps[], int inbps)
{
  float         vt;
  int           i, j;

  float         dstep = -0.5f;
  float         lam = 0;
  float         val = abuild_dgeom(vect,0.,who,bps,inbps);

  *step = 0;

  for(i = 0; i < 3; i++) {
    dstep *= -0.5f;
    for(j = 0; j < 200; j++) {
      lam += dstep;
      vt   =  abuild_dgeom(vect, lam, who, bps, inbps);
      if (vt < val ) {
        *step = lam;
        val   = vt;
      } else break;
    } /* End of for (j) */
    if (j == 200) dstep *= -2.0f;
  } /* End of for (i) */

  return val;
}


float AMMP_FASTCALL abuild_dgeom(float vect[3], float lam, AMMP_ATOM *who, AMMP_ATOM *bps[], int inbps)
{
  AMMP_ATOM *   ap;
  float         dt;
  int           i;

  float         x      = who -> x + vect[0] * lam;
  float         y      = who -> y + vect[1] * lam;
  float         z      = who -> z + vect[2] * lam;
  float         dsum   = 0.0f;

  for(i = 0; i < inbps; i++) {
    ap = bps[i];
    if (ap != who) {
      if (ap -> vx > 0.0f) {
        if (ap -> vy > 0.0f) {
          dt    = (x - ap -> x) * (x - ap -> x);
          dt   += (y - ap -> y) * (y - ap -> y);
          dt   += (z - ap -> z) * (z - ap -> z);
          dsum += ap -> vy * (ap -> vx - dt) * (ap -> vx - dt);
        } else {
          dt  = (x - ap -> x) * (x - ap -> x);
          dt += (y - ap -> y) * (y - ap -> y);
          dt += (z - ap -> z) * (z - ap -> z);
          if (ap -> vx > dt)
            dsum -= ap -> vy * (ap -> vx - dt) *( ap -> vx - dt);
        }
      }
    }
  } /* End of for (i) */

  return dsum;
}

