/* angle.c
*
* collection of routines to service bond angle  potentials
*
* POOP (Poor-mans Object Oriented Programming) using scope rules
*
* these routines hold a data base (in terms of array indeces)
* of angle, with the associated length and force constant
*
* (this could be table driven but what the hell memories cheap)
*
* the routines for potential value, force and (eventually) second
* derivatives are here also
*
* force and 2nd derivative routines assume zero'd arrays for output
* this allows for parralellization if needed (on a PC?)
*
* forces are angle wise symmetric - so we don't have to fuck around with
* s matrices and the like.
*/
/*
*  copyright 1992 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify
*  this notice
*/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#ifdef WIN32
#  include <windows.h>
#endif

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"
#include "mthread.h"

/**** Local prototypes ****/

static int f_angle_thread(void *HdtGlobVars);
static int v_angle_thread(void *HdtGlobVars);
static int v_angle_thread_lambda(void *HdtGlobVars);


/**** Add a new angle ****/

int AMMP_FASTCALL angle(int p1, int p2, int p3, float fk, float bl)
{
  ANGLE         *new;
  AMMP_ATOM     *ap1, *ap2, *ap3;
  int           i;

  /**** Get the atom pointers for the two serial numbers ****/

  ap1 = a_m_serial_table(p1);
  ap2 = a_m_serial_table(p2);
  ap3 = a_m_serial_table(p3);

  if ((!ap1) || (!ap2) || (!ap3)) {
    aaerror("Undefined atom in angle %d %d %d", p1, p2, p3);
    return FALSE;
  }

  if ((new = (ANGLE *)Alloca(sizeof(ANGLE), "angle()")) == NULL)
    return FALSE;

  /**** Initialize the pointers ****/

  if (angle_first == NULL) angle_first = new;
  if (angle_last  == NULL) angle_last  = new;
  new -> atom1  = ap1;
  new -> atom2  = ap2;
  new -> atom3  = ap3;
  new -> target = bl;
  new -> k      = fk;
  new -> next   = new;

  /**** Update the exclude list in the atoms structure ****/

  if (ap1 -> dontuse < NEXCLUDE) {
    for(i = 0; i < ap1 -> dontuse; i++) {
      if(ap1 -> excluded[i] == ap3) goto excluded1;
    } /* End of for (i) */
    ap1 -> excluded[ap1 -> dontuse] = ap3;
    (ap1 -> dontuse)++;
  } else {
    aaerror("Too many bonds to an atom increase NEXCLUDE in ammp.h");
    return FALSE;
  }

excluded1:
  if (ap3 -> dontuse < NEXCLUDE) {
    for(i = 0; i < ap3->dontuse; i++) {
      if (ap3 -> excluded[i] == ap1) goto excluded3;
    } /* End of for (i) */
    ap3 -> excluded[ap3 -> dontuse] = ap1;
    (ap3 -> dontuse)++;
  } else {
    aaerror("Too many bonds to an atom increase NEXCLUDE in ammp.h");
    return FALSE;
  }

excluded3:
  angle_last -> next = new;
  angle_last         = new;

  return TRUE;
}


/**** Sum the angle potential for the atom range ****/

int AMMP_FASTCALL a_angle(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
  AMMP_ATOM     *a1, *a2, *a3;
  float         r, x1, y1, z1, x2, y2, z2;
  float         dp;

  ANGLE         *bp = angle_first;

  if (!bp) return TRUE;

  while(bp != NULL) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    a3 = bp -> atom3;
    if (((a1 -> serial >= ilow) && (a1 -> serial <= ihigh)) ||
        ((a2 -> serial >= ilow) && (a2 -> serial <= ihigh)) ||
        ((a3 -> serial >= ilow) && (a3 -> serial <= ihigh))) {
      if (lambda) {
        x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
        x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
        y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
        z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
      } else {
        x1 = a1 -> x - a2 -> x;
        y1 = a1 -> y - a2 -> y;
        z1 = a1 -> z - a2 -> z;
        x2 = a3 -> x - a2 -> x;
        y2 = a3 -> y - a2 -> y;
        z2 = a3 -> z - a2 -> z;
      }

      dp = x1 * x2 + y1 * y2 + z1 * z2;
      r  = ((x1 * x1 + y1 * y1 + z1 * z1) * (x2 * x2 + y2 * y2 + z2 * z2));
      if (r > 1.e-8f) {
        r  = sqrt(r);
        dp = dp / r;
        if (dp > one) dp = one;
        if (dp < - one) dp = - one;
        dp  = acos(dp);
        z2  = bp -> k * (bp -> target - dp) * (bp -> target - dp);
        *V += z2;
        fprintf(op, "Angle %s %d %s %d %s %d E %f value %f error %f\n",
                a1 -> name, a1 -> serial, a2 -> name, a2 -> serial, a3 -> name, a3 -> serial,
                z2, AMMP_RAD_TO_DEG(dp), AMMP_RAD_TO_DEG(dp - bp -> target));
      }
    }
    if (bp == bp -> next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}


/**** Dump the angles ****/

void AMMP_FASTCALL dump_angles(FILE *where)
{
  ANGLE         *b = angle_first;

  while(b) {
    fprintf(where, "angle %d %d %d %f %f;\n",
            b -> atom1 -> serial, b -> atom2 -> serial,
            b -> atom3 -> serial, b -> k, AMMP_RAD_TO_DEG(b -> target));
    if (b -> next == b) break;
    b = b -> next;
  } /* End of while */
}


/**** Increment the angle force (parallel) ****/

int AMMP_FASTCALL f_angle_p(float lambda)
{
  MCL_BEGIN
    float               lambda;
  MCL_END

  MCL_ALLOCGLOBVARS;

  if (!angle_first) return TRUE;

  MCL_GetVar(lambda) = lambda;
  MCL_CreateThreads(0, f_angle_thread, &HdtGlobVars, MCL_THREADS_WAIT);

  return TRUE;
}


/**** Increment the angle force ****/

int AMMP_FASTCALL f_angle(float lambda)
{
  AMMP_ATOM     *a1, *a2, *a3;
  float         r, ux1, uy1, uz1, ux2, uy2, uz2;
  float         x1, y1, z1, x2, y2, z2;
  float         r1, r2, dtheta, dp;
  float         r11, r22, sdth;

  ANGLE         *bp = angle_first;

  if (bp == NULL) return TRUE;

  while(bp != NULL) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    a3 = bp -> atom3;
    if ((a1 -> active) || (a2 -> active) || (a3 -> active)) {
      if (lambda) {
        x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
        x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
        y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
        z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
      } else {
        x1 = a1 -> x - a2 -> x;
        y1 = a1 -> y - a2 -> y;
        z1 = a1 -> z - a2 -> z;
        x2 = a3 -> x - a2 -> x;
        y2 = a3 -> y - a2 -> y;
        z2 = a3 -> z - a2 -> z;
      }
      r1 = sqrt(x1 * x1 + y1 * y1 + z1 * z1);
      if (r1 > 1.e-5f) {
        r2 = sqrt(x2 * x2 + y2 * y2 + z2 * z2);
        if (r2 > 1.e-5f) {
          r = r1 * r2;
          if (r > 1.e-8f) {
            dp = (x1 * x2 + y1 * y2 + z1 * z2) / r;
            if (dp > one) dp = one;
            else if (dp < - one) dp = - one;
            dtheta = acos(dp);
            sdth   = sin(dtheta);
            if (sdth < 1.e-3f) sdth = 1.e-3f;
            ux1     = x2 / r2 - dp * x1 / r1;
            uy1     = y2 / r2 - dp * y1 / r1;
            uz1     = z2 / r2 - dp * z1 / r1;
            ux2     = x1 / r1 - dp * x2 / r2;
            uy2     = y1 / r1 - dp * y2 / r2;
            uz2     = z1 / r1 - dp * z2 / r2;
            dtheta  = - two * bp -> k * (bp -> target - dtheta);
            r11     = dtheta / (r1 * sdth);
            r22     = dtheta / (r2 * sdth);
            ux1    *= r11;
            uy1    *= r11;
            uz1    *= r11;
            ux2    *= r22;
            uy2    *= r22;
            uz2    *= r22;

            if (a1 -> active) {
              a1 -> fx += ux1;
              a1 -> fy += uy1;
              a1 -> fz += uz1;
            }

            if (a2 -> active) {
              a2 -> fx -= ux1 + ux2;
              a2 -> fy -= uy1 + uy2;
              a2 -> fz -= uz1 + uz2;
            }

            if (a3 -> active) {
              a3 -> fx += ux2;
              a3 -> fy += uy2;
              a3 -> fz += uz2;
            }
          }
        }
      }
    }
    if (bp == bp -> next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}


/**** Increment the angle force (parallel thread) ****/

static int f_angle_thread(void *HdtGlobVars)
{
  MCL_BEGIN
    float               lambda;
  MCL_END

  AMMP_ATOM             *a1, *a2, *a3;
  float                 r, ux1, uy1, uz1, ux2, uy2, uz2;
  float                 x1, y1, z1, x2, y2, z2;
  float                 r1, r2, dtheta, dp;
  float                 r11, r22, sdth;

  unsigned int          ThreadID;

  ANGLE                 *bp    = angle_first;
  float                 lambda = MCL_GetVarTh(lambda);
  unsigned int          j      = 1;

  MCL_ObtainThID(ThreadID);

  while(bp != NULL) {
    if (j == ThreadID) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      a3 = bp -> atom3;
      if ((a1 -> active) || (a2 -> active) || (a3 -> active)) {
        if (lambda) {
          x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
          y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
          z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
          x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
          y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
          z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
        } else {
          x1 = a1 -> x - a2 -> x;
          y1 = a1 -> y - a2 -> y;
          z1 = a1 -> z - a2 -> z;
          x2 = a3 -> x - a2 -> x;
          y2 = a3 -> y - a2 -> y;
          z2 = a3 -> z - a2 -> z;
        }
        r1 = sqrt(x1 * x1 + y1 * y1 + z1 * z1);
        if (r1 > 1.e-5f) {
          r2 = sqrt(x2 * x2 + y2 * y2 + z2 * z2);
          if (r2 > 1.e-5f) {
            r = r1 * r2;
            if (r > 1.e-8f) {
              dp = (x1 * x2 + y1 * y2 + z1 * z2) / r;
              if (dp > one) dp = one;
              else if (dp < - one) dp = - one;
              dtheta = acos(dp);
              sdth   = sin(dtheta);
              if (sdth < 1.e-3f) sdth = 1.e-3f;
              ux1     = x2 / r2 - dp * x1 / r1;
              uy1     = y2 / r2 - dp * y1 / r1;
              uz1     = z2 / r2 - dp * z1 / r1;
              ux2     = x1 / r1 - dp * x2 / r2;
              uy2     = y1 / r1 - dp * y2 / r2;
              uz2     = z1 / r1 - dp * z2 / r2;
              dtheta  = - two * bp -> k * (bp -> target - dtheta);
              r11     = dtheta / (r1 * sdth);
              r22     = dtheta / (r2 * sdth);
              ux1    *= r11;
              uy1    *= r11;
              uz1    *= r11;
              ux2    *= r22;
              uy2    *= r22;
              uz2    *= r22;

//              MCL_MutexOn();
              if (a1 -> active) {
                a1 -> fx += ux1;
                a1 -> fy += uy1;
                a1 -> fz += uz1;
              }

              if (a2 -> active) {
                a2 -> fx += - ux1 - ux2;
                a2 -> fy += - uy1 - uy2;
                a2 -> fz += - uz1 - uz2;
              }

              if (a3 -> active) {
                a3 -> fx += ux2;
                a3 -> fy += uy2;
                a3 -> fz += uz2;
              }
//              MCL_MutexOff();

            }
          }
        }
      }
    }
    if (bp == bp -> next) break;
    if ((++j) > MCL_GetVarTh(Threads)) j = 1;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}


/**** Increment the angle force (homotopy version) ****/

int AMMP_FASTCALL f_ho_angle(float lambda)
{
  AMMP_ATOM     *a1,*a2,*a3;
  float         r, ux1, uy1, uz1, ux2, uy2, uz2;
  float         x1, y1, z1, x2, y2, z2;
  float         r1, r2, dtheta, dp;
  float         r11, r22, sdth;
  float         hol, target;

  ANGLE *       bp  = angle_first;

  if (!bp) return TRUE;
  hol = GetLambda();


  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    a3 = bp -> atom3;

    if ((a1 -> active) || (a2 -> active) || (a3 -> active)) {
      if (lambda) {
        x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
        x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
        y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
        z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
      } else {
        x1 = a1 -> x - a2 -> x;
        y1 = a1 -> y - a2 -> y;
        z1 = a1 -> z - a2 -> z;
        x2 = a3 -> x - a2 -> x;
        y2 = a3 -> y - a2 -> y;
        z2 = a3 -> z - a2 -> z;
      }

      dp = x1 * x2 + y1 * y2 + z1 * z2;
      r1 = sqrt(x1 * x1 + y1 * y1 + z1 * z1);
      r2 = sqrt(x2 * x2 + y2 * y2 + z2 * z2);
      if ((r1 > 1.e-5) && (r2 > 1.e-5)) {
        r = r1 * r2;
        if (r > 1.e-8) {
          dp = dp / r;
          if (dp > one) dtheta = zero;
          else if (dp < -one) dtheta = PI;
            else dtheta = acos(dp);
          target = hol * dtheta + (one - hol) * bp -> target;
          sdth   = sin(dtheta);
          if (sdth < 1.e-3) sdth = 1.e-3;

          ux1     = x2 / r2 - dp * x1 / r1;
          uy1     = y2 / r2 - dp * y1 / r1;
          uz1     = z2 / r2 - dp * z1 / r1;
          ux2     = x1 / r1 - dp * x2 / r2;
          uy2     = y1 / r1 - dp * y2 / r2;
          uz2     = z1 / r1 - dp * z2 / r2;
    dtheta  = - two * bp -> k * (target - dtheta) * (one - hol);
          r11     = dtheta / (r1 * sdth);
          r22     = dtheta / (r2 * sdth);
          ux1    *= r11;
          uy1    *= r11;
          uz1    *= r11;
          ux2    *= r22;
          uy2    *= r22;
          uz2    *= r22;

          if(a1 -> active) {
            a1 -> fx += ux1;
            a1 -> fy += uy1;
            a1 -> fz += uz1;
          }

          if (a2 -> active) {
            a2 -> fx -= ux1 + ux2;
            a2 -> fy -= uy1 + uy2;
            a2 -> fz -= uz1 + uz2;
          }

          if (a3 -> active) {
            a3 -> fx += ux2;
            a3 -> fy += uy2;
            a3 -> fz += uz2;
          }
        }
      }
    }
    if (bp == bp ->next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}


/**** Get the angle ****/

void AMMP_FASTCALL get_angle(AMMP_ATOM *a1, AMMP_ATOM *bonded[], int mbond, int *inbond)
{
  ANGLE         *mine = angle_first;

  *inbond = 0;
  while(mine != NULL) {
    if (mine -> atom1 == a1)
      bonded[(*inbond)++] = mine -> atom3;
    if (mine -> atom3 == a1)
      bonded[(*inbond)++] = mine -> atom1;

    if (mine == mine -> next) return;
    mine = mine -> next;
    if (*inbond == mbond) return;
  } /* Ene of while */
}


/**** Calculate the angle potential (parallel) ****/

int AMMP_FASTCALL v_angle_p(float *V, float lambda)
{
  MCL_BEGIN
    float               lambda;
    float               *V;
  MCL_END

  MCL_ALLOCGLOBVARS;

  if (!angle_first) return TRUE;

  MCL_GetVar(V) = V;
  if (lambda) {
    MCL_GetVar(lambda) = lambda;
    MCL_CreateThreads(0, v_angle_thread_lambda, &HdtGlobVars, MCL_THREADS_WAIT);
  } else {
    MCL_CreateThreads(0, v_angle_thread, &HdtGlobVars, MCL_THREADS_WAIT);
  }

  return TRUE;
}


/**** Calculate the angle potential ****/

int AMMP_FASTCALL v_angle(float *V, float lambda)
{
  AMMP_ATOM     *a1, *a2, *a3;
  float         r, x1, y1, z1, x2, y2, z2;
  float         dp;

  ANGLE         *bp = angle_first;

  if (!bp) return TRUE;

  if (lambda) {
    while(bp != NULL) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      a3 = bp -> atom3;
      if ((a1 -> active) || (a2 -> active) || (a3 -> active)) {
        x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
        x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
        y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
        z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
        r  = (x1 * x1 + y1 * y1 + z1 * z1) * (x2 * x2 + y2 * y2 + z2 * z2);
        if (r > 1.e-8f) {
          dp = (x1 * x2 + y1 * y2 + z1 * z2) / (float)sqrt(r);
          if (dp > one) dp = bp -> target;
          else if (dp < - one) dp = bp -> target - PI;
            else dp = bp -> target - (float)acos(dp);
          *V += bp -> k * dp * dp;
        }
      }
      if (bp == bp -> next) return TRUE;
      bp = bp -> next;
    } /* End of while */
  } else {
    while(bp != NULL) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      a3 = bp -> atom3;
      if ((a1 -> active) || (a2 -> active) || (a3 -> active)) {
        x1 = a1 -> x - a2 -> x;
        y1 = a1 -> y - a2 -> y;
        z1 = a1 -> z - a2 -> z;
        x2 = a3 -> x - a2 -> x;
        y2 = a3 -> y - a2 -> y;
        z2 = a3 -> z - a2 -> z;
        dp = x1 * x2 + y1 * y2 + z1 * z2;
        r  = (x1 * x1 + y1 * y1 + z1 * z1) * (x2 * x2 + y2 * y2 + z2 * z2);
        if (r > 1.e-8f) {
          dp /= (float)sqrt(r);
          if (dp > one) dp = bp -> target;
          else if (dp < - one) dp = bp -> target - PI;
            else dp = bp -> target - (float)acos(dp);
          *V += bp -> k * dp * dp;
        }
      }
      if (bp == bp -> next) return TRUE;
      bp = bp -> next;
    } /* End of while */
  }

  return FALSE;
}


/**** Calculate the angle potential (parallel, thread) ****/

static int v_angle_thread(void *HdtGlobVars)
{
  MCL_BEGIN
    float               lambda;
    float               *V;
  MCL_END

  AMMP_ATOM             *a1, *a2, *a3;
  float                 r, x1, y1, z1, x2, y2, z2;
  float                 dp;
  unsigned int          ThreadID;

  ANGLE                 *bp = angle_first;
  float                 V   = zero;
  unsigned int          j   = 1;

  MCL_ObtainThID(ThreadID);

  while(bp != NULL) {
    if (j == ThreadID) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      a3 = bp -> atom3;
      if ((a1 -> active) || (a2 -> active) || (a3 -> active)) {
        x1 = a1 -> x - a2 -> x;
        y1 = a1 -> y - a2 -> y;
        z1 = a1 -> z - a2 -> z;
        x2 = a3 -> x - a2 -> x;
        y2 = a3 -> y - a2 -> y;
        z2 = a3 -> z - a2 -> z;
        r  = (x1 * x1 + y1 * y1 + z1 * z1) * (x2 * x2 + y2 * y2 + z2 * z2);
        if (r > 1.e-8f) {
          dp = (x1 * x2 + y1 * y2 + z1 * z2) / (float)sqrt(r);
          if (dp > one) dp = bp -> target;
          else if (dp < - one) dp = bp -> target - PI;
            else dp = bp -> target - (float)acos(dp);
          V += bp -> k * dp * dp;
        }
      }
    }
    if (bp == bp -> next) break;
    if ((++j) > MCL_GetVarTh(Threads)) j = 1;
    bp = bp -> next;
  } /* End of while */

  MCL_MutexOn();
  *MCL_GetVarTh(V) += V;
  MCL_MutexOff();

  return FALSE;
}


/**** Calculate the angle potential (parallel thread) ****/

static int v_angle_thread_lambda(void *HdtGlobVars)
{
  MCL_BEGIN
    float               lambda;
    float               *V;
  MCL_END

  AMMP_ATOM             *a1, *a2, *a3;
  float                 r, x1, y1, z1, x2, y2, z2;
  float                 dp;
  unsigned int          ThreadID;

  ANGLE                 *bp    = angle_first;
  float                 lambda = MCL_GetVarTh(lambda);
  float                 V      = zero;
  unsigned int          j      = 1;

  MCL_ObtainThID(ThreadID);

  while(bp != NULL) {
    if (j == ThreadID) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      a3 = bp -> atom3;
      if ((a1 -> active) || (a2 -> active) || (a3 -> active)) {
        x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
        x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
        y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
        z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
        dp = x1 * x2 + y1 * y2 + z1 * z2;
        r  = (x1 * x1 + y1 * y1 + z1 * z1) * (x2 * x2 + y2 * y2 + z2 * z2);
        if (r > 1.e-8f) {
          dp /= (float)sqrt(r);
          if (dp > 1.0f) dp = bp -> target;
          else if (dp < -1.0f) dp = bp -> target - PI;
            else dp = bp -> target - (float)acos(dp);
          V += bp -> k * dp * dp;
        }
      }
    }
    if (bp == bp -> next) break;
    if ((++j) > MCL_GetVarTh(Threads)) j = 1;
    bp = bp -> next;
  } /* End of while */

  MCL_MutexOn();
  *MCL_GetVarTh(V) += V;
  MCL_MutexOff();

  return FALSE;
}


/* MM3 angle function  */

/* v_mmangle()
* this function sums up the potentials
* for the atoms defined in the angle data structure.
*/
/* standard returns 0 if error (any) 1 if ok
* V is the potential */

int AMMP_FASTCALL v_mmangle(float *V, float lambda)
{
  ANGLE *bp;
  float r,x1,y1,z1,x2,y2,z2;
  float dp;
  AMMP_ATOM *a1,*a2,*a3;


  bp = angle_first;
       if( bp == NULL ) return 1;
       while(1)
       {
  if( bp == NULL) return 0;
  a1 = bp->atom1; a2 = bp->atom2; a3 = bp->atom3;
  if( a1->active || a2->active || a3->active )
  {
  x1 = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
  y1 = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
  z1 = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
  x2 = (a3->x -a2->x +lambda*(a3->dx-a2->dx));
  y2 = (a3->y -a2->y +lambda*(a3->dy-a2->dy));
  z2 = (a3->z -a2->z +lambda*(a3->dz-a2->dz));
  dp = x1*x2+y1*y2+z1*z2;
  r = (( x1*x1+y1*y1+z1*z1)*(x2*x2+y2*y2+z2*z2));
  if( r > 1.e-8){
  r = sqrt(r);
  dp = dp/r;  if( dp > 1.) dp = 1.; if( dp < -1.) dp = -1.;
  dp = acos(dp);
/*  *V += bp->k * (bp->target-dp)*(bp->target-dp);
*/
  dp = dp - bp->target;
  *V += bp->k*dp*dp*(1.-.014*dp+5.6e-5*dp*dp
  -7.e-7*dp*dp*dp +9e-10*dp*dp*dp*dp);
  }
  }
  if( bp == bp->next ) return 1;
  bp = bp->next;
       }
}

/* f_mmangle()
*
* f_mmangle increments the forces in the atom structures by the force
* due to the angle components.  NOTE THE WORD increment.
* the forces should first be zero'd.
* if not then this code will be invalid.  THIS IS DELIBERATE.
* on bigger (and better?) machines the different potential terms
* may be updated at random or in parrellel, if we assume that this routine
* will initialize the forces then we can't do this.
*/

int AMMP_FASTCALL f_mmangle(float lambda)
/*  returns 0 if error, 1 if OK */
{
  ANGLE *bp;
  float r,k,ux1,uy1,uz1,ux2,uy2,uz2;
  AMMP_ATOM *a1,*a2,*a3;
  float x1,y1,z1,x2,y2,z2;
  float r1,r2,dtheta,dp;
  float r11,r22,sdth;

  bp = angle_first;
       if( bp == NULL ) return 1;
       while(1)
       {
  if( bp == NULL) return 0;
  k = bp->k;
  a1 = bp->atom1; a2 = bp->atom2; a3 = bp->atom3;
  if( a1->active|| a2->active || a3->active )
  {
  x1 = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
  y1 = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
  z1 = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
  x2 = (a3->x -a2->x +lambda*(a3->dx-a2->dx));
  y2 = (a3->y -a2->y +lambda*(a3->dy-a2->dy));
  z2 = (a3->z -a2->z +lambda*(a3->dz-a2->dz));
  dp = x1*x2+y1*y2+z1*z2;
  r1 = sqrt(x1*x1+y1*y1+z1*z1);
  r2 = sqrt(x2*x2+y2*y2+z2*z2);
  if( r1 < 1.e-5 || r2 < 1.e-5) goto SKIP;
  r = r1*r2;
  if( r > 1.e-8){

  dp = dp/r;  if( dp > 1.) dp = 1.; if( dp < -1.) dp = -1.;
  dtheta = acos(dp);
  sdth = sin(dtheta); if( sdth < 1.e-3) sdth = 1.e-3;
  r11 = r1*sdth; r22 = r2*sdth;
  ux1 = x2/r2 - dp*x1/r1;
  uy1 = y2/r2 - dp*y1/r1;
  uz1 = z2/r2 - dp*z1/r1;
  ux2 = x1/r1 - dp*x2/r2;
  uy2 = y1/r1 - dp*y2/r2;
  uz2 = z1/r1 - dp*z2/r2;
/*  *V += bp->k*dp*dp*(1.-.014*dp+5.6e-5*dp*dp
  -7.e-7*dp*dp*dp +9e-10*dp*dp*dp*dp);
*/
  dp = dtheta - bp->target;
  dtheta = k*dp*(2.-.014*3*dp+4*(5.6e-5)*dp*dp
  -5*(7.e-7)*dp*dp*dp +6*(9.e-10)*dp*dp*dp*dp);
  ux1 = ux1*dtheta/r11;
  uy1 = uy1*dtheta/r11;
  uz1 = uz1*dtheta/r11;
  ux2 = ux2*dtheta/r22;
  uy2 = uy2*dtheta/r22;
  uz2 = uz2*dtheta/r22;
  if( a1->active){
  a1->fx += ux1;
  a1->fy += uy1;
  a1->fz += uz1;
  }

  if( a2->active){
  a2->fx += -ux1 - ux2;
  a2->fy += -uy1 - uy2;
  a2->fz += -uz1 - uz2;
  }

  if( a3->active){
  a3->fx += ux2;
  a3->fy += uy2;
  a3->fz += uz2;
  }

  }
  }
SKIP:
  if( bp == bp->next ) return 1;
  bp = bp->next;
       }
}




/* MM3 angle function  */

/* a_mmangle()
* this function sums up the potentials
* for the atoms defined in the angle data structure.
*/
/* standard returns 0 if error (any) 1 if ok
* V is the potential */

int AMMP_FASTCALL a_mmangle(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
  ANGLE *bp;
  float r,x1,y1,z1,x2,y2,z2;
  float dp;
  AMMP_ATOM *a1,*a2,*a3;


  bp = angle_first;
       if( bp == NULL ) return 1;
       while(1)
       {
  if( bp == NULL) return 0;
  a1 = bp->atom1; a2 = bp->atom2; a3 = bp->atom3;
  if( (a1->serial >= ilow && a1->serial <= ihigh)
  ||  (a2->serial >= ilow && a2->serial <= ihigh)
  ||  (a3->serial >= ilow && a3->serial <= ihigh) )
  {
  x1 = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
  y1 = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
  z1 = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
  x2 = (a3->x -a2->x +lambda*(a3->dx-a2->dx));
  y2 = (a3->y -a2->y +lambda*(a3->dy-a2->dy));
  z2 = (a3->z -a2->z +lambda*(a3->dz-a2->dz));
  dp = x1*x2+y1*y2+z1*z2;
  r = (( x1*x1+y1*y1+z1*z1)*(x2*x2+y2*y2+z2*z2));
  if( r > 1.e-8){
  r = sqrt(r);
  dp = dp/r;  if( dp > 1.) dp = 1.; if( dp < -1.) dp = -1.;
  dp = acos(dp);
/*  *V += bp->k * (bp->target-dp)*(bp->target-dp);
*/
  dp = dp - bp->target;
  z2 = bp->k*dp*dp*(1.-.014*dp+5.6e-5*dp*dp
  -7.e-7*dp*dp*dp +9e-10*dp*dp*dp*dp);
  *V += z2;
  dp = dp + bp->target;
  fprintf(op,"mmAngle %s %d %s %d %s %d E %f value %f error %f\n",
    a1->name,a1->serial,a2->name,a2->serial,a3->name,a3->serial
    ,z2,AMMP_RAD_TO_DEG(dp),
    AMMP_RAD_TO_DEG(dp - bp -> target));
/*
  fprintf(op,"mm Angle %d %d %d E %f value %f error %f\n",
    a1->serial,a2->serial,a3->serial,z2,AMMP_RAD_TO_DEG(dp),
    AMMP_RAD_TO_DEG(dp - bp -> target));
*/
/*  fprintf(op,"mm Angle %d %d %d  E %f error %f\n",
    a1->serial,a2->serial,a3->serial,z2,AMMP_RAD_TO_DEG(dp));
*/
  }
  }
  if( bp == bp->next ) return 1;
  bp = bp->next;
       }
}

/* UFF cos angle types */
/* v_angle()
* this function sums up the potentials
* for the atoms defined in the angle data structure.
*/
/* standard returns 0 if error (any) 1 if ok
* V is the potential */

int AMMP_FASTCALL v_c_angle(float *V, float lambda)
{
  ANGLE *bp;
  float r,x1,y1,z1,x2,y2,z2;
  float dp;
  AMMP_ATOM *a1,*a2,*a3;
  float C0,C1,C2;


  bp = angle_first;
       if( bp == NULL ) return 1;
       while(1)
       {
  if( bp == NULL) return 0;
  a1 = bp->atom1; a2 = bp->atom2; a3 = bp->atom3;
  if( a1->active || a2->active || a3->active )
  {
  x1 = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
  y1 = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
  z1 = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
  x2 = (a3->x -a2->x +lambda*(a3->dx-a2->dx));
  y2 = (a3->y -a2->y +lambda*(a3->dy-a2->dy));
  z2 = (a3->z -a2->z +lambda*(a3->dz-a2->dz));
  dp = x1*x2+y1*y2+z1*z2;
  r = (( x1*x1+y1*y1+z1*z1)*(x2*x2+y2*y2+z2*z2));
  if( r > 1.e-8){
  r = sqrt(r);
  dp = dp/r;  if( dp > 1.) dp = 1.; if( dp < -1.) dp = -1.;
  r = dp;
  dp = acos(dp);
/*  *V += bp->k * (bp->target-dp)*(bp->target-dp); */
  C0 = cos( bp->target);
  C2 = 1./(4. - 4*C0*C0);
  C1 = -4.*C2*C0;
  C0 = C2*(2*C0*C0 + 1);
/* the 2* is extra because of unit differences between my quadratic and uff*/
  *V += 2*bp->k *( C0 + C1*r + C2*cos(dp*2));
  }
  }
  if( bp == bp->next ) return 1;
  bp = bp->next;
       }
}


/* f_angle()
*
* f_angle increments the forces in the atom structures by the force
* due to the angle components.  NOTE THE WORD increment.
* the forces should first be zero'd.
* if not then this code will be invalid.  THIS IS DELIBERATE.
* on bigger (and better?) machines the different potential terms
* may be updated at random or in parrellel, if we assume that this routine
* will initialize the forces then we can't do this.
*/

int AMMP_FASTCALL f_c_angle(float lambda)
/*  returns 0 if error, 1 if OK */
{
/* float        k; */
  float         r,ux1,uy1,uz1,ux2,uy2,uz2;
  AMMP_ATOM          *a1,*a2,*a3;
  float         x1,y1,z1,x2,y2,z2;
  float         r1,r2,dtheta,dp;
  float         r11,r22,sdth;
  float         C0,C1,C2;

  ANGLE *       bp = angle_first;

  if (bp == NULL) return 1;

  while(bp) {
/*  k = bp->k; */
  a1 = bp->atom1; a2 = bp->atom2; a3 = bp->atom3;
  if( a1->active|| a2->active || a3->active )
  {
  x1 = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
  y1 = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
  z1 = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
  x2 = (a3->x -a2->x +lambda*(a3->dx-a2->dx));
  y2 = (a3->y -a2->y +lambda*(a3->dy-a2->dy));
  z2 = (a3->z -a2->z +lambda*(a3->dz-a2->dz));
  dp = x1*x2+y1*y2+z1*z2;
  r1 = sqrt(x1*x1+y1*y1+z1*z1);
  r2 = sqrt(x2*x2+y2*y2+z2*z2);
  if( r1 < 1.e-5 || r2 < 1.e-5) goto SKIP;
  r = r1*r2;
  if( r > 1.e-8){

  dp = dp/r;  if( dp > 1.) dp = 1.; if( dp < -1.) dp = -1.;
  dtheta = acos(dp);
  sdth = sin(dtheta); if( sdth < 1.e-3) sdth = 1.e-3;
  r11 = r1*sdth; r22 = r2*sdth;
  ux1 = x2/r2 - dp*x1/r1;
  uy1 = y2/r2 - dp*y1/r1;
  uz1 = z2/r2 - dp*z1/r1;
  ux2 = x1/r1 - dp*x2/r2;
  uy2 = y1/r1 - dp*y2/r2;
  uz2 = z1/r1 - dp*z2/r2;
  C0 = cos( bp->target);
  C2 = 1.0f / (4.0f - 4.0f * C0 * C0);
  C1 = -4.0f * C2 * C0;
/*  C0 = C2*(2*C0*C0 + 1);
  *V += bp->k *( C0 + C1*r + C2*cos(dp*2)); */
  dtheta = -2.0f * bp -> k * (C1 * sdth + 2.0f * C2 * sin(dtheta * 2.0f));
/*  dtheta = -2.*k*(bp->target - dtheta); */
  ux1 = ux1*dtheta/r11;
  uy1 = uy1*dtheta/r11;
  uz1 = uz1*dtheta/r11;
  ux2 = ux2*dtheta/r22;
  uy2 = uy2*dtheta/r22;
  uz2 = uz2*dtheta/r22;
  if( a1->active){
  a1->fx += ux1;
  a1->fy += uy1;
  a1->fz += uz1;
  }

  if( a2->active){
  a2->fx += -ux1 - ux2;
  a2->fy += -uy1 - uy2;
  a2->fz += -uz1 - uz2;
  }

  if( a3->active){
  a3->fx += ux2;
  a3->fy += uy2;
  a3->fz += uz2;
  }

  }
  }
SKIP:

    if (bp == bp -> next) return 1;
    bp = bp -> next;
  } /* End of while */

  return 0;
}

int AMMP_FASTCALL a_c_angle(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
  ANGLE *bp;
  float r,x1,y1,z1,x2,y2,z2;
  float dp;
  AMMP_ATOM *a1,*a2,*a3;
  float C0,C1,C2;


  bp = angle_first;
       if( bp == NULL ) return 1;
       while(1)
       {
  if( bp == NULL) return 0;
  a1 = bp->atom1; a2 = bp->atom2; a3 = bp->atom3;
  if( (a1->serial >= ilow && a1->serial <= ihigh)
  ||  (a2->serial >= ilow && a2->serial <= ihigh)
  ||  (a3->serial >= ilow && a3->serial <= ihigh) )
  {
  x1 = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
  y1 = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
  z1 = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
  x2 = (a3->x -a2->x +lambda*(a3->dx-a2->dx));
  y2 = (a3->y -a2->y +lambda*(a3->dy-a2->dy));
  z2 = (a3->z -a2->z +lambda*(a3->dz-a2->dz));
  dp = x1*x2+y1*y2+z1*z2;
  r = (( x1*x1+y1*y1+z1*z1)*(x2*x2+y2*y2+z2*z2));
  if( r > 1.e-8){
  r = sqrt(r);
  dp = dp/r;  if( dp > 1.) dp = 1.; if( dp < -1.) dp = -1.;
  r = dp;
  dp = acos(dp);
/*  z2  = bp->k * (bp->target-dp)*(bp->target-dp);
  *V += z2;
*/
  C0 = cos( bp->target);
  C2 = 1./(4. - 4*C0*C0);
  C1 = -4.*C2*C0;
  C0 = C2*(2*C0*C0 + 1);
/* the 2* is extra because of unit differences between my quadratic and uff*/
  *V += 2*bp->k *( C0 + C1*r + C2*cos(dp*2));
  fprintf(op,"c Angle %s %d %s %d %s %d E %f value %f error %f\n",
    a1->name,a1->serial,a2->name,a2->serial,a3->name,a3->serial
    ,z2,AMMP_RAD_TO_DEG(dp),
    AMMP_RAD_TO_DEG(dp - bp -> target));
/*
  fprintf(op,"Angle %d %d %d E %f value %f error %f\n",
    a1->serial,a2->serial,a3->serial,z2, AMMP_RAD_TO_DEG(dp),
    AMMP_RAD_TO_DEG(dp - bp -> target));
*/
  }
  }
  if( bp == bp->next ) return 1;
  bp = bp->next;
       }
}

void AMMP_FASTCALL gsdg_angle(AMMP_ATOM *who)
{
  AMMP_ATOM          *ap, *ap1;
  float         r, r1, theta;

  ANGLE *       bp = angle_first;

  while(bp) {
    if (bp -> atom1 == who) {
  ap1 = bp->atom2;
  ap = bp->atom3;
  r = bond_length(who,ap1); r1 = bond_length(ap1,ap);
  theta = r*r + r1*r1 -2*cos( bp->target) *r*r1;
  ap->vx = theta;
  ap->vy = bp->k;
  }
  if( bp->atom3 == who ){
  ap1 = bp->atom2;
  ap = bp->atom1;
  r = bond_length(who,ap1); r1 = bond_length(ap1,ap);
  theta = r*r + r1*r1 -2*cos( bp->target) *r*r1;
  ap->vx = theta;
  ap->vy = bp->k;
    }

    if (bp == bp -> next) return;
    bp = bp -> next;
  } /* End of while */
}


/**** Calculate the angle potential (homotopy version) ****/

int AMMP_FASTCALL v_ho_angle(float *V, float lambda)
{
  AMMP_ATOM     *a1,*a2,*a3;
  float         r, x1, y1, z1, x2, y2, z2;
  float         dp, hol, target;

  ANGLE *       bp  = angle_first;

  if (!bp) return TRUE;

  hol = GetLambda();
  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    a3 = bp -> atom3;
    if ((a1 -> active) || (a2 -> active) || (a3 -> active)) {
      x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
      y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
      z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
      x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
      y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
      z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
      r  = (x1 * x1 + y1 * y1 + z1 * z1) * (x2 * x2 + y2 * y2 + z2 * z2);
      if (r > 1.e-8) {
        dp = (x1 * x2 + y1 * y2 + z1 * z2) / (float)sqrt(r);
  if (dp > one) dp = zero;
        else if (dp < - one) dp = PI;
    else dp = acos(dp);
  target  = (hol * dp + (one - hol) * bp -> target) - dp;
  *V     += bp -> k * target * target;
      }
    }

    if (bp == bp -> next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}



