/*  animate.c
*
* routines for performing molecular dynamics
*
* v_maxwell( float Temperature, driftx,drifty,driftz)  initialize velocities
*                           with maxwell distribution, assuming
*                           simple kinetic theory relating T and v
*                           driftx,drifty,driftz allow the use of a constant
*                           drift velocity.
*
* int v_rescale( float Temperature)
*  rescale velocities so that ke is 3RT/2M
*
* int verlet(forces,nforces, nstep,dtime)
*           perform verlet (forward Euler) dynamics
*
* int pac( forces,nforces, nstep,dtime)
*            predict and correct dynamics
*
* int pacpac( forces,nforces,nstep,dtime)
*             iterated pac dynamics
*
*
* int tpac( forces,nforces, nstep,dtime, T)
*  nose constrained dynamics
* int ppac( forces,nforces, nstep,dtime, P)
*   pressure only constrained
* int ptpac( forces,nforces, nstep,dtime,P, T)
*   pressure and temperature constrained
* int hpac( forces,nforces, nstep,dtime, H)
*  total energy  constrained dynamics
*/
/*
*  copyright 1992 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify
*  this notice
*/

#include <stdio.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"


int AMMP_FASTCALL v_maxwell(float T, float dx, float dy, float dz)
{
  AMMP_ATOM     *ap, *bonded[10];
  int           inbond;
  float         vmag;

  float         R     = 1.987f; /* kcal/mol/K */
  int           iflag = -1;

  while((ap = a_next(iflag++)) != NULL) {
    iflag = 1;
    if (ap->mass > 0.0f) {
      /* Convert from kcal to mks
       * 4.184 to joules
       * 1000 grams to kg
       */

      vmag = sqrt(3.0f * R * T / ap -> mass * 4.184f * 1000.0f )* randg();
      rand3(&ap -> vx, &ap -> vy, &ap -> vz);
      if (ap -> active) {
        ap -> vx = ap -> vx * vmag + dx;
        ap -> vy = ap -> vy * vmag + dy;
        ap -> vz = ap -> vz * vmag + dz;
      } else {
        ap -> vx = 0.0f;
        ap -> vy = 0.0f;
        ap -> vz = 0.0f;
      }
    }
  } /* End of while */

  /* Now check those who are zero mass
   * and give them the velocity of the first bonded atom
   */

  iflag = -1;
  while((ap = a_next(iflag)) != NULL) {
    iflag = 1;
    if (ap -> mass <= 0.0f) {
      get_bond(ap, bonded, 10, &inbond);
      if (inbond >= 0) {
        ap -> vx = bonded[0] -> vx;
        ap -> vy = bonded[0] -> vy;
        ap -> vz = bonded[0] -> vz;
      }
    }
  } /* End of while */

  return TRUE;
}


/* v_rescale(T)
*  rescale the velocities for constant KE  == Temperature
*/

int AMMP_FASTCALL v_rescale(float T)
{
  AMMP_ATOM *   ap;
  float         vmag;

  float         R      = 1.987f; /* kcal/mol/K */
  float         target = 0.5f * (3.0f * R * T) * 4.184f * 1000.0f * a_number();
  float         KE     = 0.0f;
  int           iflag  = -1;

  while((ap = a_next(iflag++)) != NULL) {
    iflag = 1;
    if (ap -> mass > 0.0f) {
      vmag  = ap -> vx * ap -> vx + ap -> vy * ap -> vy + ap -> vz * ap -> vz;
      KE   += ap -> mass * vmag;
    }
  } /* End of while */

  KE *= 0.5f;
  if (KE == 0.0f) {
    aaerror("Can't rescale a zero velocity field - use v_maxwell()");
    return FALSE;
  }
  vmag = sqrt(target / KE);

  iflag = -1;
  while((ap = a_next(iflag++)) != NULL) {
    iflag = 1;
    ap -> vx = ap -> vx * vmag;
    ap -> vy = ap -> vy * vmag;
    ap -> vz = ap -> vz * vmag;
  } /* End of while */

  return TRUE;
}

/* routine verlet( nstep,dtime)
*int verlet(forces,nforces, nstep,dtime)
*
* perform nstep leapfrogging dynamics with dtime
*/

int AMMP_FASTCALL verlet(AMMP_FFUNC forces[], int nforces, int nstep, float dtime)
{
  AMMP_ATOM     *bp, *ap, *bonded[10];
  float         t;
  int           inbond;
  int           istep, iforces;
  int           i, imax;

#ifdef AMMP_SENDTOGFX
  int           laststep;

  int           nupdat = GetNupdat();
#endif

  if (nstep <  1) nstep = 1;
  if (dtime <= 0) dtime = 0.00001f;

#ifdef AMMP_SENDTOGFX
  laststep = nstep - 1;
#endif

  for(istep = 0; istep < nstep; istep++) {

#ifdef AMMP_SENDTOGFX
    if ((nupdat) && ((!(istep % nupdat)) || (istep == laststep))) send_all_atoms();
#endif

    /**** Find the force at the midpoint ****/

    a_f_zero();
    for(iforces = 0; iforces < nforces; iforces++)
      (*forces[iforces])( 0.);

    /**** Update velocities ****/

    imax = a_number();
    ap   = a_next(-1);
    for(i = 0; i < imax; i++, ap = bp) {
      bp = a_next(1);
      if (ap -> mass > 0.0f) {

        /**** The magic number takes kcal/molA to mks ****/

        t = ap -> mass * dtime * 4.184e6f;

        ap -> vx += ap -> fx / t;
        ap -> vy += ap -> fy / t;
        ap -> vz += ap -> fz / t;
      }
    } /* End of for (i) */

    imax = a_number();
    ap   = a_next(-1);
    for(i = 0; i < imax; i++, ap = bp) {
      bp = a_next(1);
      if (ap -> mass <= 0.0f) {
        get_bond(ap,bonded,10,&inbond);
        if (inbond >= 0) {
          ap -> vx = bonded[0] -> vx;
          ap -> vy = bonded[0] -> vy;
          ap -> vz = bonded[0] -> vz;
        }
      }
    } /* End of for (i) */

    /**** Update positions ****/

    a_inc_v(dtime);
  } /* End of for (istep) */

  return TRUE;
}


/* routine pac( nstep,dtime)
*int pac(forces,nforces, nstep,dtime)
*
* perform nstep pac dynamics with dtime
*
* predict the path given current velocity
* integrate the force (simpson's rule)
*  predict the final velocity
*  update the position using trapezoidal correction
*
*  ideally several cycles are good
*/

int AMMP_FASTCALL pac(AMMP_FFUNC forces[], int nforces, int nstep, float dtime)
{
  AMMP_ATOM     *ap, *bp, *bonded[10];
  float         Tmp;
  int           inbond;
  int           istep, iforces;
  int           i, imax;

#ifdef AMMP_SENDTOGFX
  int           laststep;

  int           nupdat = GetNupdat();
#endif

  if (nstep <  1) nstep = 1;
  if (dtime <= 0) dtime = 0.00001f;

#ifdef AMMP_SENDTOGFX
  laststep = nstep - 1;
#endif

  for(istep = 0; istep < nstep; istep++) {

#ifdef AMMP_SENDTOGFX
    if ((nupdat) && ((!(istep % nupdat)) || (istep == laststep))) send_all_atoms();
#endif

    /**** Move the velocity vector into the displacment slot ****/

    imax = a_number();
    ap   = a_next(-1);
    for(i = 0; i < imax; i++, ap = bp) {
      bp = a_next(1);
      ap -> dx = ap -> vx;
      ap -> dy = ap -> vy;
      ap -> dz = ap -> vz;
    } /* End of for (i) */

    /**** Find the force at the midpoint ****/

    a_f_zero();
    Tmp = dtime * 0.5f;
    for(iforces = 0; iforces < nforces; iforces++)
      (*forces[iforces])(Tmp);

    /**** Update velocities ****/

    imax = a_number();
    ap   = a_next(-1);

    for(i = 0; i < imax; i++, ap = bp) {
      bp = a_next(1);
      if (ap -> mass > 0.0f) {
        Tmp = ap -> mass * dtime * 4.184e6f;
        ap -> vx = ap -> dx + ap -> fx / Tmp;
        ap -> vy = ap -> dy + ap -> fy / Tmp;
        ap -> vz = ap -> dz + ap -> fz / Tmp;
      }
    } /* End of for (i) */

    imax = a_number();
    ap   = a_next(-1);

    for(i = 0; i < imax; i++, ap = bp) {
      bp = a_next(1);
      if (ap -> mass <= 0.0f) {
        get_bond(ap, bonded, 10, &inbond);
        if (inbond >= 0) {
          ap -> vx = bonded[0] -> vx;
          ap -> vy = bonded[0] -> vy;
          ap -> vz = bonded[0] -> vz;
        }
      }
    } /* End of for (i) */

    /**** Update positions ****/

    imax = a_number();
    ap   = a_next(-1);

    Tmp = 0.5f * dtime;
    for(i = 0; i < imax; i++, ap = bp) {
      bp = a_next(1);
      ap -> x += (ap -> vx + ap -> dx) * Tmp;
      ap -> y += (ap -> vy + ap -> dy) * Tmp;
      ap -> z += (ap -> vz + ap -> dz) * Tmp;
    } /* End of for (i) */
  } /* End of for (istep) */

  return TRUE;
}


/* routine tpac( nstep,dtime)
*int tpac(forces,nforces, nstep,dtime,T)
*
* perform nstep pac dynamics with dtime
* kinetic energy constraint to (3*natom-1) kT/2
*
* predict the path given current velocity
* integrate the force (simpson's rule)
*  predict the final velocity
*  update the position using trapezoidal correction
*
*  ideally several cycles are good
*
* adaptive steps (6/19/96)
*  if the rescale is too large (i.e. > 2) do two half steps
*
*/

int AMMP_FASTCALL tpac(AMMP_FFUNC forces[], int nforces, int nstep, float dtime_real, float T)
{
  AMMP_ATOM     *ap, *bp, *bonded[10];
  float         ke, Tke;
  float         dtime;
  int           imax, inbond;
  int           i, istep, iforces;

  float         R       = 1.987f; /* kcal/mol/K */

#ifdef AMMP_SENDTOGFX
  int           laststep;

  int           nupdat = GetNupdat();
#endif

  if (nstep      <  1) nstep      = 1;
  if (dtime_real <= 0) dtime_real = 0.00001f;

#ifdef AMMP_SENDTOGFX
  laststep = nstep - 1;
#endif

  for(istep = 0; istep < nstep; istep++) {

#ifdef AMMP_SENDTOGFX
    if ((nupdat) && ((!(istep % nupdat)) || (istep == laststep))) send_all_atoms();
#endif

/*  move the velocity vector into the displacment slot */
  ke = 0.;
  imax = a_number();
  ap = a_next(-1);
/*  bp = ap; */
  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
  ke += ap->mass*(
  ap->vx*ap->vx + ap->vy*ap->vy + ap->vz*ap->vz);
  ap->dx = ap->vx;
  ap->dy = ap->vy;
  ap->dz = ap->vz;
  }
  Tke = 3*(imax)*R*4.184*1000;  /* converted into MKS */
  Tke = ke/Tke;  /* Tke is now the current temperature */
/* scale the current velocities */
  dtime = dtime_real;
  if( Tke > 1.e-6)
  {
  ke = sqrt(T/Tke); /* ke is the scaled shift value */
  dtime = dtime_real/ke;
/* 0.00002 is 2fs, this is near the limit so don't use it */
  if( dtime > 0.000020 ){
    tpac(forces,nforces,1,dtime_real*0.5,T);
    tpac(forces,nforces,1,dtime_real*0.5,T);
    goto SKIP;
      }
  ap = a_next(-1);
/*  bp =  ap; */
  for( i=0; i< imax;  i++, ap = bp)
  {
  bp = a_next(1);
  ap->dx *= ke;
  ap->dy *= ke;
  ap->dz *= ke;
  }
  }

/*  find the force at the midpoint */
  a_f_zero();
  for( iforces=0;iforces<nforces; iforces++)
    (*forces[iforces])( dtime/2.);
/* update velocities */
  imax = a_number();
  ap = a_next(-1);
/*  bp = ap; */

  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
  if( ap->mass > 0.)
  {
    ap->vx = ap->dx  + ap->fx/ap->mass*dtime*4.184e6;
    ap->vy = ap->dy  + ap->fy/ap->mass*dtime*4.184e6;
    ap->vz = ap->dz  + ap->fz/ap->mass*dtime*4.184e6;
  }
  }
  imax = a_number();
  ap = a_next(-1);
/*  bp = ap; */

  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
  if( ap->mass <= 0.)
    {
                get_bond(ap,bonded,10,&inbond);
                        if( inbond >= 0)
                        {
                        ap->vx = bonded[0]->vx;
                        ap->vy = bonded[0]->vy;
                        ap->vz = bonded[0]->vz;
                        }
    }
  }
/* update positions */
  imax = a_number();
  ap = a_next(-1);
/*  bp = ap; */

  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
//    iflag = 1;
    ap->x += .5*(ap->vx + ap->dx)*dtime;
    ap->y += .5*(ap->vy + ap->dy)*dtime;
    ap->z += .5*(ap->vz + ap->dz)*dtime;
  }
SKIP: ; /* if we are here from goto we have done two half steps (or more)*/

  }/* end of istep loop */

  return TRUE;
}

/* routine pacpac( nstep,dtime)
*int pacpac(forces,nforces, nstep,dtime)
*
* perform nstep pac dynamics with dtime
*
* predict the path given current velocity
* integrate the force (simpson's rule)
*  predict the final velocity
*  update the position using trapezoidal correction
*
*  ideally several cycles are good
*/

int AMMP_FASTCALL pacpac(AMMP_FFUNC forces[], int nforces, int nstep, float dtime)
{
  AMMP_ATOM     *ap, *bp, *bonded[10];
  float         t;
  int           inbond, iflag;
  int           istep, iforces, icorrect;
  int           i, imax;

#ifdef AMMP_SENDTOGFX
  int           laststep;

  int           nupdat = GetNupdat();
#endif

  if (nstep <  1) nstep = 1;
  if (dtime <= 0) dtime = 0.00001f;

#ifdef AMMP_SENDTOGFX
  laststep = nstep - 1;
#endif

  for(istep = 0; istep < nstep; istep++) {

#ifdef AMMP_SENDTOGFX
    if ((nupdat) && ((!(istep % nupdat)) || (istep == laststep))) send_all_atoms();
#endif

    /**** Move the velocity vector into the displacment slot ****/

    iflag = -1;
    while((ap = a_next(iflag)) != NULL) {
      iflag = 1;
      ap -> dx = ap -> vx;
      ap -> dy = ap -> vy;
      ap -> dz = ap -> vz;
    } /* End of while */

    /**** Find the force at the midpoint ****/

    a_f_zero();
    t = dtime * 0.5f;
    for(iforces = 0; iforces < nforces; iforces++)
      (*forces[iforces])(t);

    /**** Update velocities ****/

    iflag = -1;
    while((ap = a_next(iflag)) != NULL) {
      iflag = 1;
      if (ap -> mass > 0.0f) {

        /**** The magic number takes kcal/molA to mks ****/

        ap -> gx  = ap -> vx;
        ap -> gy  = ap -> vy;
        ap -> gz  = ap -> vz;
        t         = ap -> mass * dtime * 4.184e6f;
        ap -> vx += ap -> fx / t;
        ap -> vy += ap -> fy / t;
        ap -> vz += ap -> fz / t;
      }
    } /* End of while */

    iflag = -1;
    while((ap = a_next(iflag)) != NULL) {
      iflag = 1;
      if (ap -> mass <= 0.0f) {
        ap -> gx = ap -> vx;
  ap -> gy = ap -> vy;
  ap -> gz = ap -> vz;
        get_bond(ap, bonded, 10, &inbond);
        if (inbond >= 0) {
          ap -> vx = bonded[0] -> vx;
          ap -> vy = bonded[0] -> vy;
          ap -> vz = bonded[0] -> vz;
        }
      }
    } /* End of while */

    /**** Make up the new prediction direction ****/

    imax = a_number();
    ap = a_next(-1);

    for(i = 0; i < imax; i++, ap = bp) {
      bp       = a_next(1);
      ap -> dx = ap -> vx + ap -> gx;
      ap -> dy = ap -> vy + ap -> gy;
      ap -> dz = ap -> vz + ap -> gz;
    } /* End of for (i) */

    for(icorrect = 0; icorrect < 2; icorrect++) {
      a_f_zero();
      t = dtime * 0.25f;
      for(iforces = 0; iforces < nforces; iforces++)
        (*forces[iforces])(t);

      /**** Update velocities ****/

      imax = a_number();
      ap = a_next(-1);

      for(i = 0; i < imax; i++, ap = bp) {
        bp = a_next(1);
        if (ap -> mass > 0.0f) {

          /**** The magic number takes kcal/molA to mks ****/

          t        = 1.0f / (ap -> mass * dtime * 4.184e6f);
          ap -> vx = ap -> gx + ap -> fx * t;
          ap -> vy = ap -> gy + ap -> fy * t;
          ap -> vz = ap -> gz + ap -> fz * t;
        }
      } /* End of for (i) */

      imax = a_number();
      ap   = a_next(-1);

      for(i = 0; i < imax; i++, ap = bp) {
        bp = a_next(1);
        if (ap -> mass <= 0.0f) {
          get_bond(ap, bonded, 10, &inbond);
          if (inbond >= 0) {
            ap -> vx = bonded[0] -> vx;
            ap -> vy = bonded[0] -> vy;
            ap -> vz = bonded[0] -> vz;
          }
        }
      } /* End of for (i) */

      /**** Make up the new prediction direction ****/

      imax = a_number();
      ap = a_next(-1);

      for(i = 0; i < imax; i++, ap = bp) {
        bp       = a_next(1);
        ap -> dx = ap -> vx + ap -> gx;
        ap -> dy = ap -> vy + ap -> gy;
        ap -> dz = ap -> vz + ap -> gz;
      } /* End of for (i) */
    }

    /**** Update positions ****/

    imax = a_number();
    ap   = a_next(-1);

    t = 0.5f * dtime;
    for(i = 0; i < imax; i++, ap = bp) {
      bp = a_next(1);
      ap->x += (ap -> vx + ap -> gx) * t;
      ap->y += (ap -> vy + ap -> gy) * t;
      ap->z += (ap -> vz + ap -> gz) * t;
    } /* End of for (i) */
  } /* End of for istep loop */

  return TRUE;
}


/* routine hpac( nstep,dtime)
*int hpac(forces,nforces, nstep,dtime,H)
*
* perform nstep pac dynamics with dtime
* kinetic energy adusted for constant H
*
* predict the path given current velocity
* integrate the force (simpson's rule)
*  predict the final velocity
*  update the position using trapezoidal correction
*
*  ideally several cycles are good
*/

int AMMP_FASTCALL hpac(AMMP_FFUNC forces[], AMMP_VFUNC poten[], int nforces, int nstep,
                       float dtime_real, float H)
{
  AMMP_ATOM     *ap, *bp, *bonded[10];
  float         ke, Tke, Tmp;
  float         dtime;
  int           inbond;
  int           istep, iforces;
  int           i, imax;

#ifdef AMMP_SENDTOGFX
  int           laststep;

  int           nupdat = GetNupdat();
#endif

  if (nstep      <  1) nstep      = 1;
  if (dtime_real <= 0) dtime_real = 0.00001f;

#ifdef AMMP_SENDTOGFX
  laststep = nstep - 1;
#endif


  for(istep = 0; istep < nstep; istep++) {

#ifdef AMMP_SENDTOGFX
    if ((nupdat) && ((!(istep % nupdat)) || (istep == laststep))) send_all_atoms();
#endif

  /**** Move the velocity vector into the displacment slot ****/

  ke   = 0.0f;
  imax = a_number();
  ap   = a_next(-1);
  for(i = 0; i < imax; i++, ap = bp) {
    bp        = a_next(1);
    ke       += ap -> mass * (ap -> vx * ap -> vx + ap -> vy * ap -> vy + ap -> vz * ap -> vz);
    ap -> dx  = ap -> vx;
    ap -> dy  = ap -> vy;
    ap -> dz  = ap -> vz;
  } /* End of for (i) */
  ke = ke * 0.5f / 4.184f / 1000.0f / 1000.0f;  /* ke in kcal/mol */

  /**** Get the current potential ****/

  Tke = 0.0f;
  for(i = 0; i < nforces; i++) (*poten[i])(&Tke, 0.0f);

  /**** Scale the current velocities ****/

  dtime = dtime_real;
  if (Tke < H) {
    ke    = sqrt((H - Tke) / ke); /* ke is the scaled shift value */
    dtime = dtime_real / ke;

    /**** 0.00002 is 2fs, this is near the limit so don't use it ****/

    if (dtime > 0.000020f) {
      hpac(forces, poten, nforces, 1, dtime_real * 0.5f, H);
      hpac(forces, poten, nforces, 1, dtime_real * 0.5f, H);
      goto SKIP;
    }

    ap = a_next(-1);
    for(i = 0; i < imax;  i++, ap = bp) {
      bp        = a_next(1);
      ap -> dx *= ke;
      ap -> dy *= ke;
      ap -> dz *= ke;
    } /* End of for (i) */
  } else {
    aawarning("Hpac potential energy higher than target");
    a_v_zero();
    a_d_zero();
  }

  /****  Find the force at the midpoint ****/

  a_f_zero();
  Tmp = dtime * 0.5f;
  for(iforces = 0; iforces < nforces; iforces++)
    (*forces[iforces])(Tmp);

  /**** Update the velocities ****/

  imax = a_number();
  ap   = a_next(-1);

  for(i = 0; i < imax; i++, ap = bp) {
    bp = a_next(1);
    if (ap -> mass > 0.0f) {
      Tmp = 1.0f / (ap -> mass * dtime * 4.184e6);
      ap -> vx = ap -> dx + ap -> fx * Tmp;
      ap -> vy = ap -> dy + ap -> fy * Tmp;
      ap -> vz = ap -> dz + ap -> fz * Tmp;
    }
  } /* End of for (i) */

  imax = a_number();
  ap   = a_next(-1);

  for(i = 0; i < imax; i++,ap = bp) {
    bp = a_next(1);
    if (ap -> mass <= 0.0f) {
      get_bond(ap, bonded, 10, &inbond);
      if (inbond >= 0) {
        ap -> vx = bonded[0] -> vx;
        ap -> vy = bonded[0] -> vy;
        ap -> vz = bonded[0] -> vz;
      }
    }
  } /* End of for (i) */

  /**** Update positions ****/

  imax = a_number();
  ap   = a_next(-1);

  Tmp = 0.5f * dtime;
  for(i = 0; i < imax; i++, ap = bp) {
    bp = a_next(1);
    ap -> x += (ap -> vx + ap -> dx) * Tmp;
    ap -> y += (ap -> vy + ap -> dy) * Tmp;
    ap -> z += (ap -> vz + ap -> dz) * Tmp;
  } /* End of for (i) */

SKIP:;
  } /* End of for (istep) */

  return TRUE;
}


/* routine ppac( nstep,dtime)
*int ppac(forces,nforces, nstep,dtime,P)
*
* force the pressure to be constant
* use P = integral ( f . r )dV as
* the basis for a diffeomorphism
*   P => kP or Integral( kf.r)dV
*         to enforce pressure
*   r => r/k to enforce physical reality
*   may need to damp this.
*
*
* perform nstep pac dynamics with dtime
*
* predict the path given current velocity
* integrate the force (simpson's rule)
*  predict the final velocity
*  update the position using trapezoidal correction
*
*  ideally several cycles are good
*/

int AMMP_FASTCALL ppac(AMMP_FFUNC forces[], int nforces, int nstep, float dtime_real, float P)
{
  AMMP_ATOM     *ap, *bp, *bonded[10];
  float         p, Tp;
  float         dtime, cx, cy, cz;
  int           inbond;
  int           istep, iforces;
  int           i;

  int           imax    = a_number();
#ifdef AMMP_SENDTOGFX
  int           laststep;

  int           nupdat = GetNupdat();
#endif

  if (imax <= 0) return FALSE;

  if (nstep      <  1) nstep      = 1;
  if (dtime_real <= 0) dtime_real = 0.00001f;

#ifdef AMMP_SENDTOGFX
  laststep = nstep - 1;
#endif


  for(istep = 0; istep < nstep; istep++) {

#ifdef AMMP_SENDTOGFX
    if ((nupdat) && ((!(istep % nupdat)) || (istep == laststep))) send_all_atoms();
#endif

  cx = 0.; cy = 0.; cz = 0.;
/*  move the velocity vector into the displacment slot */
  ap = a_next(-1);
/*  bp = ap; */
  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
  ap->dx = ap->vx;
  ap->dy = ap->vy;
  ap->dz = ap->vz;
  cx += ap->x;
  cy += ap->y;
  cz += ap->z;
  }
  cx /= imax;
  cy /= imax;
  cz /= imax;

/* calculate the pressure */

  p = 0.;
  Tp = 0.;
  ap = a_next(-1);
/*  bp = ap; */
  for( i=0; i< imax; i++,ap = bp)
  {
    bp = a_next(1);
    p += ap->vx*ap->vx*ap->mass;
    p += ap->vy*ap->vy*ap->mass;
    p += ap->vz*ap->vz*ap->mass;
    Tp += (ap->x-cx)*(ap->x-cx);
    Tp += (ap->y-cy)*(ap->y-cy);
    Tp += (ap->z-cz)*(ap->z-cz);
  }
  Tp = sqrt(Tp/imax);
  Tp = 4*PI/3*Tp*Tp*Tp;
    p = p/imax/Tp*.5; /* now mks molar */
  printf("P %f p %f Tp %f\n",P,p,Tp);
/* moment shift
  p = sqrt( P/p);
  dtime = dtime_real/p;
*/
  dtime = dtime_real;
/* this is about the steepest volume correction which works !!
  1. + .2/1.2 and 1 + .5/1.5 fail
*/
  p = (1.+.1*pow( p/P, 1./3.))/1.1;

/* temporary kludge to understand problem */
  ap = a_next(-1);
/*  bp = ap; */
  for( i=0; i< imax; i++,ap = bp)
  {
    bp = a_next(1);
/*
    ap->vx *= p;
    ap->vy *= p;
    ap->vz *= p;
    ap->dx *= p;
    ap->dy *= p;
    ap->dz *= p;
*/

    ap->x *= p;
    ap->y *= p;
    ap->z *= p;
  }
/*  find the force at the midpoint */
  a_f_zero();
  for( iforces=0;iforces<nforces; iforces++)
    (*forces[iforces])( dtime/2.);

/* update velocities */
  imax = a_number();
  ap = a_next(-1);
/*  bp = ap; */

  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
  if( ap->mass > 0.)
  {
    ap->vx = ap->dx  + ap->fx/ap->mass*dtime*4.184e6;
    ap->vy = ap->dy  + ap->fy/ap->mass*dtime*4.184e6;
    ap->vz = ap->dz  + ap->fz/ap->mass*dtime*4.184e6;
  }
  }
  imax = a_number();
  ap = a_next(-1);
/*  bp = ap; */

  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
  if( ap->mass <= 0.)
    {
                get_bond(ap,bonded,10,&inbond);
                        if( inbond >= 0)
                        {
                        ap->vx = bonded[0]->vx;
                        ap->vy = bonded[0]->vy;
                        ap->vz = bonded[0]->vz;
                        }
    }
  }
/* update positions */
  imax = a_number();
  ap = a_next(-1);
/*  bp = ap; */

  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
//    iflag = 1;
    ap->x += .5*(ap->vx + ap->dx)*dtime;
    ap->y += .5*(ap->vy + ap->dy)*dtime;
    ap->z += .5*(ap->vz + ap->dz)*dtime;
  }

  } /* End of for (i) */

  return TRUE;
}

/* routine ptpac( nstep,dtime)
*int ptpac(forces,nforces, nstep,dtime,P,T)
*
* force the pressure to be constant
*
*
* perform nstep pac dynamics with dtime
*
* predict the path given current velocity
* integrate the force (simpson's rule)
*  predict the final velocity
*  update the position using trapezoidal correction
*
*  ideally several cycles are good
*/

int AMMP_FASTCALL ptpac(AMMP_FFUNC forces[], int nforces, int nstep, float dtime_real,
                        float P, float T)
{
  AMMP_ATOM     *ap, *bp, *bonded[10];
  float         p, Tp;
  float         Tk;
  float         dtime, cx, cy, cz;
  int           inbond;
  int           istep, iforces;
  int           i;

  float         R       = 1.987f; /* kcal/mol/K */
  int           imax    = a_number();

#ifdef AMMP_SENDTOGFX
  int           laststep;
  int           nupdat = GetNupdat();
#endif

  if (imax <= 0) return FALSE;

  if (nstep      <  1) nstep      = 1;
  if (dtime_real <= 0) dtime_real = 0.00001f;

#ifdef AMMP_SENDTOGFX
  laststep = nstep - 1;
#endif

  for(istep = 0; istep < nstep; istep++) {

#ifdef AMMP_SENDTOGFX
    if ((nupdat) && ((!(istep % nupdat)) || (istep == laststep))) send_all_atoms();
#endif
  cx = 0.; cy = 0.; cz = 0.;
/*  move the velocity vector into the displacment slot */
  ap = a_next(-1);
/*  bp = ap; */
  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
  ap->dx = ap->vx;
  ap->dy = ap->vy;
  ap->dz = ap->vz;
  cx += ap->x;
  cy += ap->y;
  cz += ap->z;
  }
  cx /= imax;
  cy /= imax;
  cz /= imax;

/* calculate the pressure */

  p = 0.;
  Tp = 0.;
  ap = a_next(-1);
/*  bp = ap; */
  for( i=0; i< imax; i++,ap = bp)
  {
    bp = a_next(1);
    p += ap->vx*ap->vx*ap->mass;
    p += ap->vy*ap->vy*ap->mass;
    p += ap->vz*ap->vz*ap->mass;
    Tp += (ap->x-cx)*(ap->x-cx);
    Tp += (ap->y-cy)*(ap->y-cy);
    Tp += (ap->z-cz)*(ap->z-cz);
  }
  Tp = sqrt(Tp/imax);
  Tp = 4*PI/3*Tp*Tp*Tp;
  Tk = 3*imax*R*4.184*1000;
  Tk = p/Tk;  /* Tk is now the temperature */
  if( Tk < 1.e-5) Tk = 1.;
    p = p/imax/Tp*.5; /* now mks molar  ( kilopascal's because of grams)*/
  printf("P %f p %f Tp %f\n",P,p,Tp);
/* momentum shift */
  Tk = sqrt(T/Tk);
  dtime = dtime_real/Tk;
/* 0.00002 is 2fs, this is near the limit so don't use it */
  if( dtime > 0.000020 ){
    ptpac(forces,nforces,1,dtime_real*0.5,P,T);
    ptpac(forces,nforces,1,dtime_real*0.5,P,T);
    goto SKIP;
      }
/* this is about the steepest volume correction which works !!
  1. + .2/1.2 and 1 + .5/1.5 fail
also checked that the current 'pressure' is the best to use
for stable running
*/
  p = (1.+.1*pow( p/P, 1./3.))/1.1;

/* temporary kludge to understand problem */
  ap = a_next(-1);
/*  bp = ap; */
  for( i=0; i< imax; i++,ap = bp)
  {
    bp = a_next(1);
    ap->vx *= Tk;
    ap->vy *= Tk;
    ap->vz *= Tk;
    ap->dx *= Tk;
    ap->dy *= Tk;
    ap->dz *= Tk;

    ap->x *= p;
    ap->y *= p;
    ap->z *= p;
  }
/*  find the force at the midpoint */
  a_f_zero();
  for( iforces=0;iforces<nforces; iforces++)
    (*forces[iforces])( dtime/2.);

/* update velocities */
  imax = a_number();
  ap = a_next(-1);
/*  bp = ap; */
  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
  if( ap->mass > 0.)
  {
    ap->vx = ap->dx  + ap->fx/ap->mass*dtime*4.184e6;
    ap->vy = ap->dy  + ap->fy/ap->mass*dtime*4.184e6;
    ap->vz = ap->dz  + ap->fz/ap->mass*dtime*4.184e6;
  }
  }
  imax = a_number();
  ap = a_next(-1);
/*  bp = ap; */
  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
  if( ap->mass <= 0.)
    {
                get_bond(ap,bonded,10,&inbond);
                        if( inbond >= 0)
                        {
                        ap->vx = bonded[0]->vx;
                        ap->vy = bonded[0]->vy;
                        ap->vz = bonded[0]->vz;
                        }
    }
  }
/* update positions */
  imax = a_number();
  ap = a_next(-1);
/*  bp = ap; */

  for( i=0; i< imax; i++,ap = bp)
  {
  bp = a_next(1);
//    iflag = 1;
    ap->x += .5*(ap->vx + ap->dx)*dtime;
    ap->y += .5*(ap->vy + ap->dy)*dtime;
    ap->z += .5*(ap->vz + ap->dz)*dtime;
  }

SKIP: ; /* if goto here we've had too large a step and used half steps */
  }/* end of istep loop */

  return TRUE;
}

