/* atoms.c
*
* collection of routines to service atom memory storage
*
* POOP (Poor-mans Object Oriented Programming) using scope rules
*
* these routines hold a data base (in terms of array indeces)
* of atoms, with the associated forces, and misc consts
*
* routines
*  atom - adds an atom to the table
*  a_m_serial returns pointer to ATOM structure for matching serial
*  a_next  gets next atom in line
*  a_f_zero  zeros force (fx..) entries
*  a_d_zero  zeros dx entries
*  a_v_zero zeros velocity entries 
*  a_number  returns number of atoms
* (this could be table driven but what the hell memories cheap)
*
*  a_readvelocity( serial,vx,vy,vz) sets the velocities
*  dump_atom,dump_velocity,dump_force dump the information
*/
/*
*  copyright 1992 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#ifdef WIN32
#  include <windows.h>
#endif

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

#ifdef GRAMMP
#  include "dsc.h"
#endif

/**** Local variables ****/

static AMMP_ATOM *      a_m_serial_ap;

static AMMP_ATOM *      a_m_serial_lastmatched = NULL;
static AMMP_ATOM *      a_next_ap              = NULL;
static AMMP_ATOM **     atom_table             = NULL;
static int              atom_highest           = -1;
static int              atom_lowest            = -1;
static int              atom_table_highest     = -1;
static int              atom_table_lowest      = -1;

/**** Local prototypes ****/

static void             AMMP_FASTCALL UpdateAtomTable(void);


/**** Reset local variables ****/

void AMMP_FASTCALL ResetAtom(void)
{
  AMMP_ATOM *           Atm;

  if (atomfirst) {
    Atm = atomfirst;
    while(Atm -> next) {
      if (Atm -> Close) free(Atm -> Close);
      if (Atm -> next == Atm) break;
      Atm = Atm -> next;
    } /* End of while */

    ListFree((void **)&atomfirst, (void **)&atomlast);
  }

  SafeFree(atom_table);

  a_m_serial_lastmatched = NULL;
  a_next_ap              = NULL;
  atom_table             = NULL;
  atom_highest           = -1;
  atom_lowest            = -1;
  atom_table_highest     = -1;
  atom_table_lowest      = -1;
}

/* activate(i1, i2, Val)
*  i2  == 0 just do i 1
*  else do the range from i1 to i2
*
* set the active flag in the atoms to Val (1 or 0)
*/

int AMMP_FASTCALL activate(int i1, int i2, int Val)
{
  int           upper, lower;
  AMMP_ATOM *   ap;
  int           i;

  int           numatm = a_number();

  if (!numatm) {
    aaerror("No atoms");
    return FALSE;
  }

  if (!i2) {
    ap = a_m_serial(i1) ;
    if (ap != NULL) ap -> active = Val;
    return TRUE;
  }

  if (i2 < i1) {
    lower = i2;
    upper = i1;
  } else {
    upper = i2;
    lower = i1;
  }

  for(i = 0; i < numatm; i++) {
    ap = a_next(i);
    if ((ap -> serial >= lower) && (ap -> serial <= upper))
      ap -> active = Val;
  } /* End of for (i) */

  return TRUE;
}


/* function atom adds an atom to the atom list
* returns 1 if ok
* returns 0 if not
* passed the atomic data (at least the initial values )
* allocates the new memory, initializes it and
* returns
*/

int AMMP_FASTCALL atom(float x, float y, float z, int serial, float q, float a,
                       float b, float mass, char *name)
{
  int           i;

  AMMP_ATOM *   new     = NULL;

  if ((atom_highest >= serial) && (atom_lowest <= serial)) new = a_m_serial(serial);
  if (new == NULL) {
    if ((new = (AMMP_ATOM *)Alloca(sizeof(AMMP_ATOM), "atom()")) == NULL) return FALSE;
    new -> dontuse     = 0;
    new -> dontuse     = 1;
    new -> excluded[0] = new;
    new -> active      = 1;
    for(i = 0; i < NEXCLUDE; i++)
      new -> exkind[i] = 0;
    new -> next = NULL;
  }

  /**** Initialize the pointers ****/

  if (!atomfirst) {
    atomfirst      = new;
    atom_highest   = serial;
    atom_lowest    = serial;
  }
  if (!atomlast) atomlast = new;
  new -> x   = x;
  new -> y   = y;
  new -> z   = z;
  new -> w   = 0.0f;
  new -> fx  = 0.0;
  new -> fy  = 0.0;
  new -> fz  = 0.0;
  new -> fw  = 0.0;
  new -> dx  = 0.0f;
  new -> dy  = 0.0f;
  new -> dz  = 0.0f;
  new -> dw  = 0.0f;
  new -> vx  = 0.0f;
  new -> vy  = 0.0f;
  new -> vz  = 0.0f;
  new -> vw  = 0.0f;
  new -> jaa = -1.0f;
  new -> chi = -1.0f;
  new -> Close = NULL;

 /**** for reinterpolates ****/

  new -> px  = 10e10;
  new -> py  = 10e10;
  new -> pz  = 10e10;
  new -> pw  = 10e10;
/*
  new ->qx = 10e10;
  new ->qy = 10e10;
  new ->qz = 10e10;
*/
  new -> q      = q;
  new -> a      = a;
  new -> b      = b;
  new -> serial = serial;
  new -> mass  = mass;
  for(i = 0; i < 8; i++) {
    new -> name[i    ] = *name;
    new -> name[i + 1] = '\0';
    if (*name == '\0') break;
    name++;
  } /* End of for (i) */
  if (new -> next == NULL) {
    new      -> next = new;
    atomlast -> next = new;
    atomlast         = new;
  }
  atomUPDATE = TRUE;

#ifdef GRAMMP
  send_an_atom(new);
#endif

  if (atom_highest < serial ) atom_highest = serial;
  if (atom_lowest  > serial ) atom_lowest  = serial;

  return TRUE;
}

/* function a_number()
 * returns number of atoms defined
 * this is just atomNUMBER if atomUPDATE == 0
 * other wise just figure it out
 */

int AMMP_FASTCALL a_number(void)
{
  AMMP_ATOM *        ap;

  if (atomUPDATE) {
    if (atomfirst == NULL) return 0;
    atomUPDATE = FALSE;
    atomNUMBER = 0;
    ap         = atomfirst;

    while(ap -> next) {
      atomNUMBER++;
      if(ap -> next == ap) break;
      ap = ap -> next;
    } /* End of while */
  }

  return atomNUMBER;
}

/* function a_m_serial( serial )
* returns NULL on error or returns the address of the AMMP_ATOM
* which matches serial
* cute?
*/

AMMP_ATOM * AMMP_FASTCALL a_m_serial(int serial)
{
  int                   i, n;

  if (atomUPDATE) n = a_number();
  else n = atomNUMBER;

  a_m_serial_ap = atomfirst;
  if (!a_m_serial_ap) return NULL;
  if (!a_m_serial_lastmatched) a_m_serial_lastmatched = atomfirst;
  if (serial == a_m_serial_lastmatched -> serial) return a_m_serial_lastmatched;
  if (serial >  a_m_serial_lastmatched -> serial) a_m_serial_ap = a_m_serial_lastmatched;

  for(i = 0; i < n; i++) {
    if(a_m_serial_ap -> serial == serial) {
      a_m_serial_lastmatched = a_m_serial_ap;
      return a_m_serial_ap;
    }
    if (a_m_serial_ap == a_m_serial_ap -> next) a_m_serial_ap = atomfirst;
    else a_m_serial_ap = a_m_serial_ap -> next;
  } /* End of for (i) */

  return NULL;
}


/**** a_m_serial() table accelerated ****/

AMMP_ATOM * AMMP_FASTCALL a_m_serial_table(int serial)
{
  if ((serial < atom_lowest) || (serial > atom_highest)) return NULL;

  if ((atom_table_highest != atom_highest) || (atom_table_lowest != atom_lowest))
    UpdateAtomTable();

  return atom_table[serial - atom_table_lowest];
}


/* function a_next( flag )
* returns NULL on error or last atom
* then steps to the next
* cute?
* flag <= 0 starts it off
*/

AMMP_ATOM * AMMP_FASTCALL a_next(int flag)
{
  if (a_next_ap == NULL) a_next_ap = atomfirst;
  if (a_next_ap == NULL) return NULL;
  if (flag <= 0) {
    a_next_ap = atomfirst;
    return a_next_ap;
  }
  if(a_next_ap == a_next_ap -> next) return NULL;
  a_next_ap = a_next_ap -> next;

  return a_next_ap;
}

/* function a_f_zero()
* zeros the forces in each atom element
* return is 0 on error 1 iff OK */

int AMMP_FASTCALL a_f_zero(void)
{
  AMMP_ATOM *        ap = atomfirst;

  while(ap -> next) {
    ap -> fx = 0.0;
    ap -> fy = 0.0;
    ap -> fz = 0.0;
    ap -> fw = 0.0;

    if (ap == ap -> next) return 1;
    ap = ap -> next;
  } /* End of while */

  return 0;
}

/* function a_d_zero()
* zeros the dx,dy,dz storage for each atom element 
*/
/* return is 0 on error 1 iff OK */

int AMMP_FASTCALL a_d_zero(void)
{
  AMMP_ATOM *        ap = atomfirst;

  if (ap == NULL) return 0;

  while(ap -> next) {
    ap -> dx = 0.0f;
    ap -> dy = 0.0f;
    ap -> dz = 0.0f;
    ap -> dw = 0.0f;

    if (ap == ap -> next) return 1;
    ap = ap -> next;
  } /* End of while */

  return 0;
}

/* function a_g_zero()
* zeros the velocities in each atom element
*/
/* return is 0 on error 1 iff OK */

int AMMP_FASTCALL a_g_zero(void)
{
  AMMP_ATOM *        ap = atomfirst;
  if (ap == NULL) return 0;

  while(ap -> next) {
    ap -> gx = 0.0f;
    ap -> gy = 0.0f;
    ap -> gz = 0.0f;
    ap -> gw = 0.0f;

    if (ap == ap -> next) return 1;
    ap = ap -> next;
  } /* End of while */
  return 0;
}

/* function a_v_zero()
* zeros the velocities in each atom element
* return is 0 on error 1 iff OK */

int AMMP_FASTCALL a_v_zero(void)
{
  AMMP_ATOM *        ap = atomfirst;

  if (ap == NULL) return 0;

  while(ap -> next) {
    ap -> vx = 0.0f;
    ap -> vy = 0.0f;
    ap -> vz = 0.0f;
    ap -> vw = 0.0f;
    if (ap == ap -> next) return 1;
    ap = ap -> next;
  } /* End of while */

  return 0;
}

/* function a_inc_f( lambda )
*  moves the atoms lambda times the forces
*/

int AMMP_FASTCALL a_inc_f(float lambda)
{
  AMMP_ATOM *        ap = atomfirst;

  if (ap == NULL) return FALSE;

  while(ap -> next) {
    ap -> x += ap -> fx * lambda;
    ap -> y += ap -> fy * lambda;
    ap -> z += ap -> fz * lambda;
    ap -> w += ap -> fw * lambda;
    if(ap == ap -> next) return TRUE;
    ap = ap -> next;
  }

  return FALSE;
}

/* function a_inc_d( lambda )
*  moves the atoms lambda times the dx
*/

int AMMP_FASTCALL a_inc_d(float lambda)
{
  AMMP_ATOM *        ap = atomfirst;

  if (ap == NULL) return 0;

  while(ap -> next) {
    ap -> x += ap -> dx * lambda;
    ap -> y += ap -> dy * lambda;
    ap -> z += ap -> dz * lambda;
    ap -> w += ap -> dw * lambda;
    if (ap == ap -> next) return TRUE;
    ap = ap -> next;
  } /* End of while */

  return FALSE;
}

/* function a_inc_v( lambda )
*  moves the atoms lambda times the velocities
*/

int AMMP_FASTCALL a_inc_v(float lambda)
{
  AMMP_ATOM *        ap = atomfirst;

  if( ap == NULL) return 0;

  while(ap -> next) {
    ap -> x += ap -> vx * lambda;
    ap -> y += ap -> vy * lambda;
    ap -> z += ap -> vz * lambda;
    ap -> w += ap -> vw * lambda;
    if (ap == ap -> next) return 1;
    ap = ap -> next;
  } /* End of while */

  return 0;
}

/* function a_ftodx( lambda )
*  moves the atom force components to the dx slots
* with a scale factor.
*/

int AMMP_FASTCALL a_ftodx(float lambda, float lamold)
{
  AMMP_ATOM *        ap = atomfirst;

  if (ap == NULL) return 0;

  while(ap -> next) {
    ap -> dx = ap -> dx * lamold + ap -> fx * lambda;
    ap -> dy = ap -> dy * lamold + ap -> fy * lambda;
    ap -> dz = ap -> dz * lamold + ap -> fz * lambda;
    ap -> dw = ap -> dw * lamold + ap -> fw * lambda;
    if (ap == ap -> next) return 1;
    ap = ap->next;
  }

  return 0;
}

/* function a_ftogx( lambda,lamold )
*  moves the atom force components to the gx slots
* with a scale factor.
*/

int AMMP_FASTCALL a_ftogx(float lambda, float lamold)
{
  AMMP_ATOM *        ap = atomfirst;

  if (ap == NULL) return 0;

  while(ap -> next) {
    ap -> gx = ap -> gx * lamold + ap -> fx * lambda;
    ap -> gy = ap -> gy * lamold + ap -> fy * lambda;
    ap -> gz = ap -> gz * lamold + ap -> fz * lambda;
    ap -> gw = ap -> gw * lamold + ap -> fw * lambda;
    if (ap == ap -> next) return 1;
    ap = ap->next;
  }

  return 0;
}

/* function a_ftovx( lambda,lamold )
*  moves the atom force components to the vx slots
* with a scale factor.
*/

int AMMP_FASTCALL a_ftovx(float lambda, float lamold)
{
  AMMP_ATOM *        ap = atomfirst;

  if (ap == NULL) return 0;

  while(ap -> next) {
    ap -> vx = ap -> vx * lamold + ap -> fx * lambda;
    ap -> vy = ap -> vy * lamold + ap -> fy * lambda;
    ap -> vz = ap -> vz * lamold + ap -> fz * lambda;
    ap -> vw = ap -> vw * lamold + ap -> fw * lambda;
    if (ap == ap->next) return 1;
    ap = ap -> next;
  }

  return 0;
}

/* function a_max_f()
* returns the maximum l2 metric of a force on an atom
*/

float AMMP_FASTCALL a_max_f(void)
{
  float         l2norm;

  AMMP_ATOM *   ap    = atomfirst;
  float         l2max = -1.0f;

  if (ap == NULL) return l2max;

  while(ap -> next) {
    l2norm  = ap -> fx * ap -> fx;
    l2norm += ap -> fy * ap -> fy;
    l2norm += ap -> fz * ap -> fz;
    l2norm += ap -> fw * ap -> fw;
    if (l2norm > l2max) l2max = l2norm;
    if (ap == ap -> next) return l2max;
    ap = ap -> next;
  } /* End of while */

  return l2max;
}

/* function a_max_d()
* returns the maximum l2 metric of a displacement of an atom
*/

float AMMP_FASTCALL a_max_d(void)
{
  float         l2norm;

  AMMP_ATOM *   ap    = atomfirst;
  float         l2max = -1.0f;

  if (ap == NULL) return l2max;

  while(ap -> next) {
    l2norm  = ap -> dx * ap -> dx;
    l2norm += ap -> dy * ap -> dy;
    l2norm += ap -> dz * ap -> dz;
    l2norm += ap -> dw * ap -> dw;
    if (l2norm > l2max) l2max = l2norm;
    if (ap == ap -> next) return l2max;
    ap = ap -> next;
  } /* End of while */

  return l2max;
}

/* function a_l2_f(  )
*  return l2 norm of the forces
*/

float AMMP_FASTCALL a_l2_f(void)
{
  AMMP_ATOM *   ap = atomfirst;
  float         l2 = 0.0f;

  if (ap == NULL) return 0.0f;

  while(ap -> next) {
    l2 += ap -> fx * ap -> fx;
    l2 += ap -> fy * ap -> fy;
    l2 += ap -> fz * ap -> fz;
    l2 += ap -> fw * ap -> fw;
    if (ap == ap -> next) return l2;
    ap = ap -> next;
  } /* End of while */

  return -l2;
}

/* function a_l2_g(  )
*  return l2 norm of the velocities
*/

float AMMP_FASTCALL a_l2_g(void)
{
  AMMP_ATOM *   ap = atomfirst;
  float         l2 = 0.0f;

  if (ap == NULL) return 0.0f;

  while(ap -> next) {
    l2 += ap -> gx * ap -> gx;
    l2 += ap -> gy * ap -> gy;
    l2 += ap -> gz * ap -> gz;
    l2 += ap -> gw * ap -> gw;
    if(ap == ap -> next) return l2;
    ap = ap->next;
  } /* End of while */

  return -l2;
}

/* function a_l2_v(  )
*  return l2 norm of the velocities
*/

float AMMP_FASTCALL a_l2_v(void)
{
  AMMP_ATOM *   ap = atomfirst;
  float         l2 = 0.0f;

  if (ap == NULL) return 0.0f;

  while(ap -> next) {
    l2 += ap -> vx * ap -> vx;
    l2 += ap -> vy * ap -> vy;
    l2 += ap -> vz * ap -> vz;
    l2 += ap -> vw * ap -> vw;
    if (ap == ap -> next) return l2;
    ap = ap -> next;
  } /* End of while */

  return -l2;
}

/* function a_l2_d(  )
*  return l2 norm of the dx terms
*/

float AMMP_FASTCALL a_l2_d(void)
{
  AMMP_ATOM *   ap = atomfirst;
  float         l2 = 0.0f;

  if (ap == NULL) return 0.0f;
  while(ap -> next) {
    l2 += ap -> dx * ap -> dx;
    l2 += ap -> dy * ap -> dy;
    l2 += ap -> dz * ap -> dz;
    l2 += ap -> dw * ap -> dw;
    if (ap == ap -> next) return l2;
    ap = ap -> next;
  } /* End of while */

  return -l2;
}


/**** Dump the atoms ****/

void AMMP_FASTCALL dump_atoms(FILE *where)
{
  register AMMP_ATOM *  a;

  if (!CheckAtoms()) return;

  a = atomfirst;
  while(a) {
    fprintf(where, "atom %f %f %f %d %s %f %f %f %f;\n",
            a -> x, a -> y, a -> z, a -> serial, a -> name, a -> q, a -> a,
            a -> b, a -> mass);
    if ((a -> chi > 0.0f) && (a -> jaa > 0.0f))
      fprintf(where, "mompar %d %f %f;\n", a -> serial, a -> chi, a -> jaa);
    if (!a -> active)
      fprintf(where, "inactive %d;\n", a -> serial);
    if (a -> next == a) break;
    a = a -> next;
  } /* End of while */

  dump_excludes(where);
}


/**** Dump the excludes ****/

void AMMP_FASTCALL dump_excludes(FILE *where)
{
  AMMP_ATOM     *a, *ap;
  int           numatm;
  int           i, j;

  numatm = a_number();
  if (numatm <= 0) return;

  for(i = 0; i < numatm; i++) {
    a = a_next(i);
    for(j = 0; j < a -> dontuse; j++) {
      if (a -> exkind[j] > 0) {
        ap = a -> excluded[j];
        if (ap -> serial > a -> serial)
          fprintf(where, "tailor exclude %d %d;\n",
                  a -> serial, ap -> serial);
      }
    } /* End of for (j) */
  } /* End of for (i) */
}


/**** Dump the forces ****/

void AMMP_FASTCALL dump_force(FILE *where)
{
  AMMP_ATOM     *a = atomfirst;

  while(a) {
    fprintf(where, "force %d %f %f %f %f %f %f;\n",
	    a -> serial, a -> x, a -> y, a -> z, a -> fx, a -> fy, a -> fz);
    if (a -> next == a) break;
    a = a -> next;
  } /* End of while */
}

/**** Dump the PDB file ****/

void AMMP_FASTCALL dump_pdb(FILE *where, int res_mod)
{
  char          *np, resid[5], atid[5];
  int           i, ires;

  AMMP_ATOM     *a    = atomfirst;
  int           iatom = 0;

  if (!res_mod) {
    aaerror("Need a non-zero residue modulus in dump_pdb");
    return;
  }

  while(a) {
    iatom++;
    ires = a -> serial / res_mod;
    np   = a -> name;
    while(!strcmp(np, "sna.rkq")) {
      a = a -> next;
      if (a -> next == NULL ) return;
      ires = a -> serial / res_mod;
      np   = a -> name;
    } /* End of while */

    for(i = 0; i < 5; i++) {
      if (*np == '.') {
        resid[i] = 0;
        break;
      } else resid[i] = toupper(*np);
      if (!*np) break;
      np++;
    } /* End of for (i) */
    if (*np == '.') np++;

    for(i = 0; i < 5; i++) {
      if (*np == '.') {
        atid[i] = 0;
        break;
      } else atid[i] = toupper(*np);
      if (!*np) break;
      np++;
    } /* End of for (i) */

    if (atid[0] == 'H') {
      fprintf(where, "ATOM  %5d %-4s%c%-3s  %4d    %8.3f%8.3f%8.3f%6.2f%6.2f\n",
	      iatom, atid, ' ', resid, ires, a -> x, a -> y, a -> z, 1.0f, 10.0f);
    } else {
      fprintf(where, "ATOM  %5d  %-4s%-3s  %4d    %8.3f%8.3f%8.3f%6.2f%6.2f\n",
	      iatom, atid, resid, ires, a -> x, a -> y, a -> z, 1.0f, 10.0f);
    }
    if (a -> next == a) break;
    a = a -> next;
  } /* End of while */
  fprintf(where, "END\n");
}


/**** Dump the velocities ****/

void AMMP_FASTCALL dump_velocity(FILE *where)
{
  AMMP_ATOM     *a = atomfirst;

  while(a) {
    fprintf(where, "velocity %d %f %f %f ;\n",
            a -> serial, a -> vx, a -> vy, a -> vz);
    if (a -> next == a) break;
    a = a -> next;
  } /* End of while */
}


/* int a_readvelocity( serial,vx,vy,vz)
*  update the velocity field of the atom structure
*/

int AMMP_FASTCALL a_readvelocity(int serial, float vx, float vy, float vz)
{
  AMMP_ATOM *ap = a_m_serial( serial);
  if (ap == NULL) return FALSE;

  ap -> vx = vx;
  ap -> vy = vy;
  ap -> vz = vz;

  return TRUE;
}




/* function a_pr_beta()
* a_pr_beta() returns the Poliak Ribeire beta
* for conjugate gradients
*
*/

float AMMP_FASTCALL a_pr_beta(void)
{
  AMMP_ATOM *   ap = atomfirst;
  float         a  = 0.0f;
  float         b  = 0.0f;

  if( ap == NULL) return 0.0f;

  while(ap -> next) {
    a += ap -> fx * ap -> fx;
    a += ap -> fy * ap -> fy;
    a += ap -> fz * ap -> fz;
    a += ap -> fw * ap -> fw;
    a -= ap -> gx * ap -> fx;
    a -= ap -> gy * ap -> fy;
    a -= ap -> gz * ap -> fz;
    a -= ap -> gw * ap -> fw;
    b += ap -> gx * ap -> gx;
    b += ap -> gy * ap -> gy;
    b += ap -> gz * ap -> gz;
    b += ap -> gw * ap -> gw;
    if (ap == ap -> next) {
      if (b <= 1.e-5) {
        a = 0.0f;
        b = 1.0f;
      }
      return a/b;
    }
    ap = ap -> next;
  } /* End of while */

  return 0.0f;
}

/* a_inactive_f_zero()
* loop through the atoms and zero the inactive forces
*/

int AMMP_FASTCALL a_inactive_f_zero(void)
{
  AMMP_ATOM     *ap;
  int           i;
  int           numatom = a_number();

  if (!numatom) {
    aaerror("No atoms");
    return FALSE;
  }

  for(i = 0; i < numatom; i++) {
    ap = a_next(i);
    if (!ap -> active) {
      ap -> fx = 0.0f;
      ap -> fy = 0.0f;
      ap -> fz = 0.0f;
      ap -> fw = 0.0f;
    }
  } /* End of for (i) */

  return TRUE;
}


/* inactivate(i1, i2)
*  i2  == 0 just do i 1
*  else do the range from i1 to i2
*
* set the active flag in the atoms to 0 and
* turn them off
*/

void AMMP_FASTCALL inactivate_non_zero(int i1, int i2)
{
  int           upper, lower;
  AMMP_ATOM *        ap;
  int           i, numatm;

  if (i2 == 0) {
    ap = a_m_serial(i1);
    if (ap != NULL)
      ap -> active = 0;
    return;
  }

  if (i2 < i1) {
    lower = i2;
    upper = i1;
  } else {
    lower = i1;
    upper = i2;
  }

  numatm = a_number();
  for(i = 0; i < numatm; i++) {
    ap = a_next(i);
    if ((ap -> serial >= lower) && (ap -> serial <= upper) &&
	(ap -> x != 0.0f) && (ap -> y != 0.0f) && (ap -> z != 0.0f))
      ap -> active = FALSE;
  } /* End of for (i) */
}


/**** Update the atom table ****/

static void AMMP_FASTCALL UpdateAtomTable(void)
{
  AMMP_ATOM *           ap;
  int                   Size;

  SafeFree(atom_table);
  atom_table_highest = atom_highest;
  atom_table_lowest  = atom_lowest;
  Size               = sizeof(AMMP_ATOM *) * (atom_table_highest - atom_table_lowest + 1);

  if ((atom_table = Alloca(Size, "UpdateAtomTable()")) == NULL)
    exit(0);

#ifdef WIN32
  ZeroMemory(atom_table, Size);
#else
  memset(atom_table, 0, Size);
#endif

  ap = atomfirst;
  while(ap -> next) {
    atom_table[ap -> serial - atom_table_lowest] = ap;
    if(ap -> next == ap) break;
    ap = ap -> next;
  } /* End of while */
}


/* grAMMP calls */
/* send_an_atom( AP* )
*  sends an atom to the drawing process
*  
* send_all_atoms(void)
*  sends all of the atoms
*/

#ifdef GRAMMP
int AMMP_FASTCALL send_an_atom(AMMP_ATOM *who)
{

	int i;
	void send_xyz(),send_color(),send_label();

	if( who == NULL ) return;
		i = 3;
		WriteToGrfx(&i, sizeof(int));
		send_xyz(who);
		send_label(who);
		send_color(who);
}/* end of send_an_atom */


int AMMP_FASTCALL send_all_atoms(void)
{
  AMMP_ATOM *        who;
  int           ip;

  int           i = 2 * a_number();

  if (i < 1) return ;

  WriteToGrfx(&i, sizeof(int));

  i = i/2;
  who = first;
  for(ip = 0; ip < i; ip++) {
    send_xyz(who);
    send_color(who);
    who = who -> next;
    if( who == NULL ) break;
  } /* End of for (ip) */
}
#endif
