/* bonds.c
*
* collection of routines to service bond length potentials
*
* POOP (Poor-mans Object Oriented Programming) using scope rules
*
* these routines hold a data base (in terms of array indeces)
* of bonds, with the associated length and force constant
*
* (this could be table driven but what the hell memories cheap)
*
* the routines for potential value, force and (eventually) second
* derivatives are here also
*
* force and 2nd derivative routines assume zero'd arrays for output
* this allows for parralellization if needed (on a PC?)
*
* forces are bond wise symmetric - so we don't have to fuck around with
* s matrices and the like.
*/
/*
*  copyright 1992,1993,1994,1995 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify
*  this notice
*/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"
#include "mthread.h"


/**** Sum the bond potential by the atom range ****/

int AMMP_FASTCALL a_bond(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
  AMMP_ATOM     *a1,*a2;
  float         r, xt, yt, zt;

  BOND          *bp = bond_first;

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    if (((a1 -> active) || (a2 -> active)) &&
        (((a1 -> serial >= ilow) && (a1 -> serial <= ihigh)) ||
         ((a2 -> serial >= ilow) && (a2 -> serial <= ihigh)))) {
      if (lambda) {
        xt = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        yt = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        zt = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
      } else {
        xt = a1 -> x - a2 -> x;
        yt = a1 -> y - a2 -> y;
        zt = a1 -> z - a2 -> z;
      }
      r  = (float)sqrt(xt * xt + yt * yt + zt * zt) - bp -> length;
      zt = bp -> k * r * r;
      *V += zt;
      fprintf(op, "Bond %s %d %s %d E %f value %f error %f\n",
              a1 -> name, a1 -> serial, a2 -> name, a2 -> serial,
              zt, r + bp -> length, r);
    }
    bp = bp -> next;
  } /* End of while */

  return TRUE;
}


/**** Sum the bond potential by the atom range (mm3 version) ****/

int AMMP_FASTCALL a_mmbond(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
  AMMP_ATOM     *a1,*a2;
  float         r, xt, yt, zt;

  BOND          *bp = bond_first;

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    if (((a1 -> active) || (a2 -> active)) &&
        (((a1 -> serial >= ilow) && (a1 -> serial <= ihigh)) ||
         ((a2 -> serial >= ilow) && (a2 -> serial <= ihigh)))) {
      if (lambda) {
        xt = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        yt = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        zt = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
      } else {
        xt = a1 -> x - a2 -> x;
        yt = a1 -> y - a2 -> y;
        zt = a1 -> z - a2 -> z;
      }
      r   = (float)sqrt(xt * xt + yt * yt + zt * zt) - bp -> length;
      zt  =  bp -> k * r * r * (one - 2.55f * r + seven * 2.55f / twelve * r * r);
      *V += zt;
      fprintf(op, "mmBond %s %d %s %d E %f value %f error %f\n",
              a1 -> name, a1 -> serial, a2 -> name, a2 -> serial, zt,
              r + bp -> length, r);
    }
    bp = bp -> next;
  } /* End of while */

  return TRUE;
}


/**** Add a new bond ****/

int AMMP_FASTCALL bond(int p1, int p2, float bl, float fk, float order)
{
  BOND *        NewBond;
  int           i;

  AMMP_ATOM *   ap1 = a_m_serial_table(p1);
  AMMP_ATOM *   ap2 = a_m_serial_table(p2);
  const char *  Err = "Too many bonds to an atom increase NEXCLUDE in ammp.h";

  if ((!ap1) || (!ap2)) {
    aaerror("Undefined atom in bond %d %d", p1, p2);
    return FALSE;
  }

  if ((NewBond = (BOND *)ListAdd((void **)&bond_first, (void **)&bond_last, sizeof(BOND))) == NULL)
    return FALSE;

  if (!order) order = 1.0;

  /**** Initialize the pointers ****/

  NewBond -> atom1  = ap1;
  NewBond -> atom2  = ap2;
  NewBond -> length = bl;
  NewBond -> k      = fk;
  NewBond -> order  = order;

  /**** Update the exclude list in the atoms structure ****/

  if (ap1 -> dontuse < NEXCLUDE) {
    for(i = 0; i < ap1 -> dontuse; i++)
      if(ap1 -> excluded[i] == ap2) goto excluded1;
    ap1 -> excluded[ap1 -> dontuse] = ap2;
    (ap1 -> dontuse)++;
  } else {
    aaerror(Err);
    return FALSE;
  }

excluded1:

  if (ap2 -> dontuse < NEXCLUDE) {
    for(i = 0; i < ap2 -> dontuse; i++)
      if(ap2 -> excluded[i] == ap1) return TRUE;
    ap2 -> excluded[ap2 -> dontuse] = ap1;
    (ap2 -> dontuse)++;
  } else {
    aaerror(Err);
    return FALSE;
  }

  return TRUE;
}


/**** Bond length ****/

float AMMP_FASTCALL bond_length(AMMP_ATOM *a1, AMMP_ATOM *a2)
{
  BOND          *bp = bond_first;

  while(bp) {
    if ((a1 == bp -> atom1) && (a2 == bp -> atom2)) return bp -> length;
    if ((a2 == bp -> atom1) && (a1 == bp -> atom2)) return bp -> length;
    bp = bp -> next;
  } /* End of while */

  return -one;
}


/**** Get the next bond ****/

int AMMP_FASTCALL bond_next(int i, AMMP_ATOM **ap1, AMMP_ATOM **ap2)
{
  static BOND *         bp;

  if (i <= 0) {
    bp = bond_first;
    if (bp == NULL) {
      *ap1 = NULL;
      *ap2 = NULL;
      return FALSE;
    }
  }
  *ap1 = bp -> atom1;
  *ap2 = bp -> atom2;
  if (bp == NULL) return FALSE;
  if (bp -> next) bp = bp -> next;

  return TRUE;
}


/**** Number of bonds ****/

int AMMP_FASTCALL bond_number(void)
{
  register BOND *       Bnd = bond_first;
  register int          Number = 0;

  while(Bnd) {
    ++Number;
    Bnd = Bnd -> next;
  } /* End of while */

  return Number;
}


/**** Dump the bonds ****/

void AMMP_FASTCALL dump_bonds(FILE *where)
{
  AMMP_ATOM     *a1,*a2;

  BOND          *b = bond_first;

  while(b) {
    a1 = b -> atom1;
    a2 = b -> atom2;
    fprintf(where, "bond %d %d %f %f %.2f;\n",
            a1 -> serial, a2 -> serial, b -> length, b -> k, b -> order);
    b = b -> next;
  } /* End of while */
}


/**** Increment the bond force ****/

int AMMP_FASTCALL f_bond(float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         r, k, ux, uy, uz;

  BOND          *bp = bond_first;

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    if ((a1 -> active) || (a2 -> active)) {
      if (lambda) {
        ux = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
        uy = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
        uz = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
      } else {
        ux = a2 -> x - a1 -> x;
        uy = a2 -> y - a1 -> y;
        uz = a2 -> z - a1 -> z;
      }
      r = ux * ux + uy * uy + uz * uz;
      if (r <= 1.0e-15f) {
        ux = - two * bp -> k * bp -> length;
        uy = zero;
        uz = zero;
      } else {
        r   = sqrt(r);
        k   = (two * bp -> k * (r - bp -> length)) / r;
        ux *= k;
        uy *= k;
        uz *= k;
      }

      if (a1 -> active) {
        a1 -> fx += ux;
        a1 -> fy += uy;
        a1 -> fz += uz;
      }

      if (a2 -> active) {
        a2 -> fx -= ux;
        a2 -> fy -= uy;
        a2 -> fz -= uz;
      }
    }
    bp = bp -> next;
  }

  return TRUE;
}


/**** Increment the bond force (homotopy version) ****/

int AMMP_FASTCALL f_ho_bond(float lambda)
{
  AMMP_ATOM     *a1,*a2;
  float         r, k, ux, uy, uz;

  BOND *        bp  = bond_first;
  float         hol = GetLambda();

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    if ((a1 -> active) || (a2 -> active)) {
      if (lambda) {
        ux = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
        uy = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
        uz = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
      } else {
        ux = a2 -> x - a1 -> x;
        uy = a2 -> y - a1 -> y;
        uz = a2 -> z - a1 -> z;
      }
      r = ux * ux + uy * uy + uz * uz;
      if (r <= 1.0e-15f) {
        ux = - two * bp -> k * (one - hol) * bp -> length;
        uy = zero;
        uz = zero;
      } else {
        r   = sqrt(r);
        k   = (two * bp -> k * (r - hol * r - (one - hol) * bp -> length)) / r;
        ux *= k;
        uy *= k;
        uz *= k;
      }
      if (a1 -> active) {
        a1 -> fx += ux;
        a1 -> fy += uy;
        a1 -> fz += uz;
      }

      if (a2 -> active) {
        a2 -> fx -= ux;
        a2 -> fy -= uy;
        a2 -> fz -= uz;
      }
    }
    bp = bp -> next;
  } /* End of while */

  return TRUE;
}


/**** Increment the bond force (mm3 version) ****/

int AMMP_FASTCALL f_mmbond(float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         r, k, ux, uy, uz;

  BOND *        bp = bond_first;

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    if ((a1 -> active) || (a2 -> active)) {
      if (lambda) {
        ux = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
        uy = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
        uz = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
      } else {
        ux = a2 -> x - a1 -> x;
        uy = a2 -> y - a1 -> y;
        uz = a2 -> z - a1 -> z;
      }
      r = ux * ux + uy * uy + uz * uz;
      if (r <= 1.0e-5f) {
        r  = 0;
        ux = one;
        uy = zero;
        uz = zero;
      } else {
        r   = sqrt(r);
        ux /= r;
        uy /= r;
        uz /= r;
      }
      r  -= bp -> length;
      k   = bp -> k * r * (two - three * 2.55f * r + four * seven * 2.55f * r * r / twelve);
      ux *= k;
      uy *= k;
      uz *= k;

      if (a1 -> active) {
        a1 -> fx += ux;
        a1 -> fy += uy;
        a1 -> fz += uz;
      }

      if (a2 -> active) {
        a2 -> fx -= ux;
        a2 -> fy -= uy;
        a2 -> fz -= uz;
      }
    }
    bp = bp -> next;
  } /* End of while */

  return TRUE;
}


/**** Get the bond ****/

void AMMP_FASTCALL get_bond(AMMP_ATOM *a1, AMMP_ATOM *bonded[], int mbond, int *inbond)
{
  BOND *        mine = bond_first;

  *inbond = 0;
  while(mine) {
    if (mine -> atom1 == a1) bonded[(*inbond)++] = mine -> atom2;
    else if (mine -> atom2 == a1) bonded[(*inbond)++] = mine -> atom1;
    if (*inbond == mbond) return;
    mine = mine -> next;
  } /* End of while */
}


/**** Get bond & length ****/

void AMMP_FASTCALL get_bond_and_length(AMMP_ATOM *a1, AMMP_ATOM *bonded[], float r[], int mbond, int *inbond)
{
  BOND *        mine = bond_first;

  *inbond = 0;
  while(mine) {
    if (mine -> atom1 == a1) {
      r[*inbond]          = mine -> length;
      bonded[(*inbond)++] = mine -> atom2;
    } else if (mine -> atom2 == a1) {
      r[*inbond]          = mine -> length;
      bonded[(*inbond)++] = mine -> atom1;
    }
    if (*inbond == mbond) return;
    mine = mine -> next;
  } /* End of while */
}


/**** Get the bond pointer ****/

BOND * AMMP_FASTCALL get_bond_pointer(AMMP_ATOM *a1, AMMP_ATOM *a2)
{
  BOND *        bp = bond_first;

  while(bp) {
    if (((a1 == bp -> atom1) && (a2 == bp -> atom2)) ||
        ((a1 == bp -> atom2) && (a2 == bp -> atom1)))
      return bp;

    bp = bp -> next;
  } /* End of while */

  return NULL;
}


/**** Get the bond (gsdg version) ****/

void AMMP_FASTCALL gsdg_bond(AMMP_ATOM *who)
{
  AMMP_ATOM *   ap;

  BOND *        bp = bond_first;

  while(bp) {
    if (bp -> atom1 == who) {
      ap       = bp -> atom2;
      ap -> vx = bp -> length * bp -> length;
      ap -> vy = bp -> k;
    }

    if (bp -> atom2 == who) {
      ap       = bp -> atom1;
      ap -> vx = bp -> length * bp -> length;
      ap -> vy = bp -> k;
    }
    bp = bp -> next;
  } /* End of while */
}


/**** Calculate the bond potential ****/

int AMMP_FASTCALL v_bond(float *V, float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         r, xt, yt, zt;

  BOND *        bp = bond_first;

  if (lambda) {
    while(bp != NULL) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      if ((a1 -> active) || (a2 -> active)) {
        xt = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        yt = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        zt = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
        r  = (float)sqrt(xt * xt + yt * yt + zt * zt) - bp -> length;
        *V += bp -> k * r * r;
      }
      bp = bp -> next;
    } /* End of while */
  } else {
    while(bp != NULL) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      if ((a1 -> active) || (a2 -> active)) {
        xt = a1 -> x - a2 -> x;
        yt = a1 -> y - a2 -> y;
        zt = a1 -> z - a2 -> z;
        r  = (float)sqrt(xt * xt + yt * yt + zt * zt) - bp -> length;
        *V += bp -> k * r * r;
      }
      bp = bp -> next;
    } /* End of while */
  }

  return TRUE;
}


/**** Calculate the bond potential (homotopy version) ****/

int AMMP_FASTCALL v_ho_bond(float *V, float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         r, xt, yt, zt;
  float         target;

  BOND *        bp  = bond_first;
  float         hol = GetLambda();

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    if ((a1 -> active) || (a2 -> active)) {
      xt      = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
      yt      = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
      zt      = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
      r       = sqrt(xt * xt + yt * yt + zt * zt);
      target  = r - (hol * r + (one - hol) * bp -> length);
      *V     += bp -> k * target * target;
    }
    bp = bp -> next;
  } /* End of while */

  return TRUE;
}


/**** Calculate the bond potential (mm3 version) ****/

int AMMP_FASTCALL v_mmbond(float *V, float lambda)
{
  AMMP_ATOM     *a1,*a2;
  float         r, r2, xt, yt, zt;

  BOND          *bp = bond_first;


  if (lambda) {
    while(bp) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      if ((a1 -> active) || (a2 -> active)) {
        xt  = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        yt  = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        zt  = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
        r   = (float)sqrt(xt * xt + yt * yt + zt * zt) - bp -> length;
        r2  = r * r;
        *V += bp -> k * r2 * (one - 2.55f * r + seven * 2.55f / twelve * r2);
      }
      bp = bp -> next;
    } /* End of while */
  } else {
    while(bp) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      if ((a1 -> active) || (a2 -> active)) {
        xt  = a1 -> x - a2 -> x;
        yt  = a1 -> y - a2 -> y;
        zt  = a1 -> z - a2 -> z;
        r   = (float)sqrt(xt * xt + yt * yt + zt * zt) - bp -> length;
        r2  = r * r;
        *V += bp -> k * r2 * (one - 2.55f * r + seven * 2.55f / twelve * r2);
      }
      bp = bp -> next;
    } /* End of while */
  }
  return TRUE;
}

