/* ammp.c
* Another Molecular Mechanics Program
*
*  this essentially runs the intermediate code for
*  a molecular mechanics program
*
* instructions are of the form
*  ident <parameters> ;
*  # <stuff> ; is a comment
*   "<stuff>" is  a literal string
*  most instructions can be nested, but NOT loop<if> and labels
*
*  allowed idents
*
* atom   - atom record
* bond   - bond record
*       morse  - morse record
* angle  - angle record
*   torsion - torsion record
*       hybrid  - hybrid (pyramid height) record
* abc    - angle bond correlation record
*                  i1 i2 i3 angle zero_angle  dr/da dk12/da dk23/da
* av5  - tetrahedral 'volume' or centroid
*   i1,i2,i3,i4,i5, k, value
*       ttarget torsion target i j k l angle fk;
*       swarm k n (o,end),,... n/2 pairs  add a distance restrain to the
*                mean distance
*       box  box potential; controlled with variable bbox
* noel -noe distance restraint
*       step - step distance term (3 point noel)
*       noegen dm dp; set the noels to the current geometry -dm + dp
*       velocity  - velocity record
* read <file>  open and read from file untill done
* output <file> <vers>  open and use for output file
* dump <atom,bond,angle,abc,torsion,hybrid,av5,morse,pdb,variable,velocity,force>
*                         write out the results
* analyze ilow,ihigh  write out the errors in the current potential for atoms
*       ilow to ihigh. if ilow > ihigh ilow to ilow
* close     close the current output file if not stdout
* steep  niter,toler   steepest descents
* bfgs  niter,toler  bfgs quasi newton
* cngdel  niter,ncut,toler  conjugate del
* trust   niter,dtoler,toler   trust optimizer
*       polytope imin,imax,niter,vstart,vfinal  polytope of a range
* rigid imin,imax,niter, vstart,vfinal  polytope rigid body solver
* echo <off>   echo to the user (turn off when dumping !!)
*       use  < none,bond,angle,abc,torsion,nonbon,morse,restrain,tether
*   ,periodic,mmbond,mmangle,cangle,screen,debye,shadow,fourd
*   hobond hoangle trace honoel hotether >
*               flag on potentials
*       restrain    - restrain a distance
* tether      - tether an atom to a positon
*          tether serial fk x y z
*          tether all fk x y z  do all of them
* tailor  qab   number q a b  - set the qab parameters of an atom
*       tailor  exclude  number number  - add an interaction to the nonbon exclude list
*       tailor  include number number  - delete an interaction from the nonbon exclude list
* setf name value  set a float into the variable store
*       seti name value   set an int into the variable store
*       loopi label init max delta  loop to label while init < max integer vers.
*       loopf label init max delta  loop to label while init < max float vers.
*       label:
* monitor    find potential energy and kinetic energy, and calculate the forces
*       v_maxwell  temperature,dx,dy,dz
* v_rescale   temperature
*       verlet       nstep,dtime (dtime is in m/s = .01A/ps)
*       pac          nstep,dtime (dtime is in m/s = .01A/ps)
*       tpac          nstep,dtime,Temp (dtime in m/s = .01A/ps,1fs = .00001)
*       ppac          nstep,dtime,pressure (dtime in m/s = .01A/ps,1fs = .00001)
*       ptpac          nstep,dtime,pressure,Temp (dtime in m/s = .01A/ps,1fs = .00001)
*       hpac          nstep,dtime,Htarget (dtime in m/s = .01A/ps,1fs = .00001)
*       pacpac       nstep,dtime (dtime is in m/s = .01A/ps)
*       richard <pac ... verlet> nstep mdparams; WFK integral set the parameter
*                                 occup
* doubletime   nstep,dlong,dshort,temper  double time scale dynamics
* dipole first,last  calculate the dipole moment for atoms first to last
*                          assumes sequential atom numbers...
* tgroup id serial1 serial2 serial3 serial4 base number
*            define a tgroup( torsion by serial numbers) base = zeropoint
*      number == number of steps.  The group of atoms is everything bonded to
*       serial3 that isn't serial 2.
* tsearch id id id id (up to 8  - terminated by 0 or ; )
*             search the tgroups defined
*
*       tset i1 i2 i3 i4 where
*            set the torsion angle defined by i1...i4 to where
*            unlike tgroup,tsearch only one angle at a time, and
*            no limit to the number of atoms rotated
* tmin i1 i2 i3 i4 nstep
*            search the torsion angle  i1...i4 for the minimum
*            energy in nsteps
* tmap i1 i2 i3 i4 j1 j2 j3 j4 ni nj;  map in ni nj steps
*              the i j atoms over all 360 degrees;
*
*   mompar  serial,chi,jaa  add electronegativity and self colomb to atom serial
* momadd  serial serial  adds atoms to the MOM stack( can just be called with one)
*       mom   tq, niter   solves current mom stack for charges
*     tq = total charge, niter = number of iterations (20 default)
*
*       time  return time of day and elapsed time (not on all machines)
*
* math routines  see math.c
*   add a b ;
*   sub a b ;
*   mul a b;
*   div a b;
*   nop a;  these routines can work with atomic parameters
*   mov a b;  variables, and imeadiate values.
*   max a b;
*   min a b;
*   randf a ;
*
*   serial a i atomid;  put the serial number or residue i, atom atomid
*                   into a
* index a i;  put the serial number of the ith atom into a;
*
*        je a b label: ;   jump a == b
*        jl a b label: ;   jump a < b
*        jg a b label: ;   jump a > b
* jes a string label: ; dump to label if a->name == string
* jnes a string label: ; dump to label if a->name != string
*           jumps are restricted to the current file
*
* exit         - exit the routine - in case EOF is not defined
*
*   active i1 i2; <i2 optional> active atoms i1 to i2 (default is active)
*       inactive i1 i2; < i2 optional> inactivate atoms i1 to i2
*       nzinactive i1 i2; < i2 optional> inactivate atoms i1 to i2 that
*                               are not 0 0 0
*
*
*   grasp nstep nopt imin imax atom;  GRASP in torsion space
* genetic nstep ndeep sigma target n_opt_steps ; genetic optimizer
* gsdg  niter min_atom max_atom; iterative distance geometry bounded by
*                                       serial numbers
* abuild  niter min_atom max_atom; analytic geometry solver bounded by
*                                       serial numbers
*       bell niter min_atom Max_atom; iterative distance geometry using
*                                      the bellman-ford-fulkerson algorithm
*       kohonen niter radius <1 init -1 continue> < r | r z| rx ry rz>
*                                    kohonen search
*
* dgeom niter origin shift;  standard distance geometry
*                             implemented with the power method
*                              origin is the atom to use as the key
*                              shift is the amount of eigenvalue shift
*
* normal damp    ;    calculate the normal modes  if damp > 0 output them
*
* table id n ; create empty sorted table
*       tableent id who r v ; add the who'th element to the table it
*       access with use tbond
*
* direct SCF terms
*       orbit <o1,o1o,o2,o3,o4s,o4p,om> i1,<i2-i5>,osn, parameters, ipair ;
*               ipair == 2 (doublet) ipair == 1 (singlet)
*       expand osn,n,a,r,a,r (up to 6)  ;
*                          these define an orbital
*
*       dscf <coef,expo,xyz,geom,anal> n toler;  optimize the orbitals
*            <coefficients, exponents, atom center, orbital geometry>;
*
* others like fix,and... TBD
*   first nonblank == '#' is a comment and the line is skipped
*/
/*
*  copyright 1992,1993 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify
*  this notice
*/
#define MAXTOKEN 20
#define TOKENLENGTH 80

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>

#ifdef ESV
#  define NOTIME
#endif
#ifndef NOTIME
#  define TIME
#endif
#ifdef GRACELESS
#  include graceless.h
#endif

#include <sys/types.h>

#ifndef WIN32
#  include <sys/time.h>
#  include <unistd.h>
#endif

#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"
#include "orbit.h"

#ifdef GRAMMP
#  include "dsc.h"

/**** Prototypes ****/

void send_connect(int,int);
void send_terminate(void);
void send_eof(void);
void send_tether(int,float,float,float,float);
void send_noel(int,int,float,float,float,float,float);
#endif


/**** Local variables ****/

static int      ConfirmEnd;


/**** Read the commands ****/

void AMMP_FASTCALL read_eval_do(FILE *ip, FILE *op)
{
  char                  line[AMMP_LINEBUF];
  char                  StringSep;
  char *                LinePtr;
  char *                StringSepPtr;
  int                   inliteral;

#ifdef GRAMMP
  char *                lp;                     /* Char pointer for parsing commands */
  int                   fd;                     /* File descriptor */
  int                   fduse;                  /* Current file descriptor */
  int                   retval;                 /* Return value */

#  ifndef WIN32
  fd_set                rfds;                   /* Array of fds being watched for input */
  fd_set                rfduse;
  struct timeval        tv;
#  endif
#endif

#ifndef WIN32
  struct timespec       sleep_time, dummy;

  sleep_time.tv_sec  = 0.0;               /* 0 seconds */
  sleep_time.tv_nsec = 100e6;             /* 1 millisecond */

#  ifdef GRAMMP
  tv.tv_sec          = 0;                 /* 0 seconds */
  tv.tv_usec         = 100;               /* 100 milliseconds */
#  endif
#endif

  ConfirmEnd         = FALSE;
  inliteral          = FALSE;

  if (!ip) {
    aaerror("Can't use input file");
    return;
  }
  if (!op) {
    aaerror("Can't use output file");
    return;
  }

#ifdef GRAMMP
  if (ip != stdin) goto THE_OLD_WAY;

  lp    = &line[0];
  fduse = fileno(stdin); /* default test */
  while(TRUE) {
    while (TRUE) {
      FD_ZERO(&rfds);
      FD_SET(fileno(stdin),&rfds);
      FD_SET(grfx_to_ammp_pipe[0],&rfds);
      retval = select(grfx_to_ammp_pipe[0] + 1, &rfds, NULL, NULL, &tv);
      if (retval < 0) aaerror("read_eval_do() error with select()");
      fduse = fileno(stdin);
      if (FD_ISSET(fileno(stdin), &rfds)) {
        fduse = fileno(stdin);
        break;
      } else if (FD_ISSET(grfx_to_ammp_pipe[0], &rfds))  {
        fduse = grfx_to_ammp_pipe[0];
        break;
      } else nanosleep(&sleep_time, &dummy);
    }
    FD_ZERO(&rfduse);
    FD_SET(fduse,&rfduse);
    while(select(fduse + 1, &rfduse, NULL, NULL, &tv)) {
      read(fduse, lp, sizeof(char));
      if (!inliteral && *lp == '"')  inliteral = TRUE;
      if ( inliteral && *lp == '"')  inliteral = FALSE;
      if (!inliteral) {
        if (*lp == ',')  *lp = ' ';
        if (*lp == '\t') *lp = ' ';
        if (*lp == '\n') *lp = ' ';
        if (*lp == ';') {
          *lp = '\0';
          if(eval(ip,op,line) < 0) return;
          lp = &line[0];
          break;
        } else if (*lp != '\n') lp++;
      }
    }
  } /* While forever */

THE_OLD_WAY:
#endif

  LinePtr      = line;
  StringSep    = 0;
  StringSepPtr = NULL;
  while((*LinePtr = fgetc(ip)) != EOF) {
    if (!inliteral) {
      if ((*LinePtr == '"') || (*LinePtr == '\'')) {
        inliteral    = TRUE;
        StringSep    = *LinePtr;
        StringSepPtr = LinePtr;
      } else {
        if (strchr(",\t\n\r", *LinePtr)) *LinePtr = ' ';
        else if (*LinePtr == ';') {
          *LinePtr = 0;
          if (eval(ip, op, line) < 0) return;
          LinePtr = line;
          continue;
        }
      }
    } else if (*LinePtr == StringSep) {
      *LinePtr      = 1;
      *StringSepPtr = 1;
      inliteral     = FALSE;
    }
    if (++LinePtr == (line + AMMP_LINEBUF)) {
      aaerror("Command buffer overflow");
      return;
    }
  } /* End of the lex while */
}


/* eval actually parses the line */
/* original version used sscanf *
*  current version lexes tokens and if numeric
*  converts them to integer and floating point versions
*/

int AMMP_FASTCALL eval(FILE *ip, FILE *op, char *line)
{
  FILE *        newfile;
  char          token[MAXTOKEN][TOKENLENGTH], *ap, *ap1;
  char          errmes[80];
  char          istring[MAXTOKEN];
  int           itemp[MAXTOKEN], itoken;
  float         ftemp[MAXTOKEN];
  VARIABLE *    ivar[MAXTOKEN];

  static int    echo   = TRUE;
  static int    inloop = 1;

  /**** For safety and to avoid side effects the token arrays are zero'd ****/

  for(itoken = 0; itoken < MAXTOKEN; itoken++) {
    ftemp[itoken]    = zero;
    istring[itoken]  = FALSE;
    itemp[itoken]    = 0;
    ivar[itoken]     = NULL;
    token[itoken][0] = 0;
  } /* End of for (itoken) */

  /**** Now extract tokens and prepare to match it ****/

#ifndef VEGAZZ
  if (echo) fprintf(op, "%s;\n", line);
#endif

  ap = line;
  for(itoken = 0; itoken < MAXTOKEN; itoken++) {
    ap1  = token[itoken];
    *ap1 = '\0';
    while(*ap == ' ') ap++;
    if (*ap == 1) {
      istring[itoken] = TRUE;
      ap++;
      while((*ap != 1) && (*ap != '\0')) *(ap1++) = *(ap++);
      if (*ap == 1) ap++;
    } else {
      if ((itoken == 0) && (*ap == '#')) return TRUE;
      while((*ap != ' ') && (*ap != '\0')) {
        if ((!itoken) ||
            ((strcmp(token[0], "read"  )) &&
             (strcmp(token[0], "output")) &&
             (strcmp(token[0], "print" )))) {
          *ap1++ = tolower(*ap++);
        } else *ap1++ = *ap++;
      } /* End of while */
    }
    *ap1 = 0;

    /**** If the token is a number atof or atoi it ****/

    ap1 = &token[itoken][0];
    if (tisvariable(ap1))
      ivar[itoken] = get_fi_variable(ap1, ftemp + itoken, itemp + itoken);
    else if (tisint(ap1) == 1) {
      itemp[itoken] = atoi(ap1);
      ftemp[itoken] = itemp[itoken];
    } else {
      ftemp[itoken] = atof(ap1);
      itemp[itoken] = (int)ftemp[itoken];
    }
    if (!*ap) break;
  } /* End of for (itoken) */

  if (!token[0][0]) return TRUE;

  /**** Atom ****/

  if (!strcmp(token[0], "atom")) {
    if (!atom(ftemp[1], ftemp[2], ftemp[3], itemp[4], ftemp[6],
              ftemp[7], ftemp[8], ftemp[9], token[5]))
      aaerror("Can't add to atom structure - data structure error");
    goto DONE;
  }

  /**** MomPar ****/

  if (!strcmp(token[0], "mompar")) {
    mom_param(itemp[1], ftemp[2], ftemp[3]);
    goto DONE;
  }

  /**** Angle ****/

  if (!strcmp(token[0], "angle")) {
    if (!angle(itemp[1], itemp[2], itemp[3], ftemp[4], AMMP_DEG_TO_RAD(ftemp[5])))
      aaerror("Can't add to angle structure - data structure error");
    goto DONE;
  }

  /**** Bond ****/

  if (!strcmp(token[0], "bond")) {
    if (!bond(itemp[1], itemp[2], ftemp[3], ftemp[4], ftemp[5]))
      aaerror("Can't add to bond structure - data structure error");
#ifdef GRAMMP
    else {
      itemp[3] = 1;
      WriteToGrfx(&itemp[3], sizeof(int));
      send_connect(itemp[1], itemp[2]);
    }
#endif
    goto DONE;
  }

  /**** Hybrid ****/

  if (!strcmp(token[0], "hybrid")) {
    if (!hybrid(itemp[1], itemp[2], itemp[3], itemp[4], ftemp[5], ftemp[6]))
      aaerror("Can't add to hybrid structure - data structure error");
    goto DONE;
  }

  /**** Torsion ****/

  if (!strcmp(token[0], "torsion")) {
    if (!torsion(itemp[1], itemp[2], itemp[3], itemp[4], ftemp[5], itemp[6],
                 AMMP_DEG_TO_RAD(ftemp[7])))
      aaerror("Can't add to torsion structure - data structure error");
    goto DONE;
  }

  /**** Abc ****/

  if (!strcmp(token[0], "abc")) {
    if (!abc(itemp[1], itemp[2], itemp[3], AMMP_DEG_TO_RAD(ftemp[4]),
             AMMP_DEG_TO_RAD(ftemp[5]), ftemp[6], ftemp[7], ftemp[8]))
      aaerror("Can't add to abc structure - data structure error");
    goto DONE;
  }

  /**** Abuild ****/

  if (!strcmp(token[0], "abuild" )) {
    if (nused <= 0) goto DONE;
    a_build(op, nused, potentials, forces, itemp[1], itemp[2], itemp[3], echo);

#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Active ****/

  if (!strcmp(token[0], "active")) {
    activate(itemp[1], itemp[2], TRUE);
    goto DONE;
  }

  /**** Analyze ****/

  if (!strcmp(token[0], "analyze")) {
    analyze(potentials, nused, itemp[1], itemp[2], op);
    goto DONE;
  }

  /**** Av5 ****/

  if (!strcmp(token[0], "av5")) {
    if (!av5(itemp[1], itemp[2], itemp[3], itemp[4], itemp[5], ftemp[6], ftemp[7]))
      aaerror("Can't add to av5 structure - data structure error");
     goto DONE;
  }

  /**** Clone ****/

  if (!strcmp(token[0], "clone")) {
#ifdef GTK_VERSION
    AMMP_READY_go_for_it = FALSE;
#endif
    Clone(NULL, abs(itemp[1]));
#ifdef GTK_VERSION
    if (!AMMP_in_fileselect) AMMP_READY_go_for_it = TRUE;
#endif
    goto DONE;
  }

  /**** DelClone ****/

  if (!strcmp(token[0], "delclone")) {
    if (!CloneDel(NULL, itemp[1]))
      aaerror("Clone %d not found", itemp[1]);
    goto DONE;
  }

  /**** Dipole ****/

  if (!strcmp(token[0], "dipole")) {
    dipole(op, itemp[1], itemp[2]);
    goto DONE;
  }

  /**** Doubletime ****/

  if (!strcmp(token[0], "doubletime")) {
    doubletime(forces, nused, itemp[1], ftemp[2], ftemp[3], ftemp[4]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Dump ****/

  if (!strcmp(token[0], "dump")) {
    for(itoken = 1; (itoken < MAXTOKEN) && (*token[itoken]); itoken++) {
      if (!strcmp(token[itoken], "angle")) {
        dump_angles(op);
        continue;
      }
      if (!strcmp(token[itoken], "atom")) {
        dump_atoms(op);
        continue;
      }
      if (!strcmp(token[itoken], "bond")) {
        dump_bonds(op);
        continue;
      }
      if (!strcmp(token[itoken], "hybrid")) {
        dump_hybrids(op);
        continue;
      }
      if (!strcmp(token[itoken], "pdb")) {
        dump_pdb(op, 100);
        continue;
      }
      if (!strcmp(token[itoken], "torsion")) {
        dump_torsions(op);
        continue;
      }

      if (!strcmp(token[itoken], "abc")) {
        dump_abcs(op);
        continue;
      }
      if (!strcmp(token[itoken], "av5")) {
        dump_av5s(op);
        continue;
      }
      if (!strcmp(token[itoken], "force")) {
        dump_force(op);
        continue;
      }
      if (!strcmp(token[itoken], "morse")) {
        dump_morse(op);
        continue;
      }
      if (!strcmp(token[itoken], "noel")) {
        dump_noels(op);
        continue;
      }
      if (!strcmp(token[itoken], "orbit")) {
        dump_orbit(op);
        continue;
      }
      if (!strcmp(token[itoken], "restrain")) {
        dump_restrains(op);
        continue;
      }
      if (!strcmp(token[itoken], "step")) {
        dump_steps(op);
        continue;
      }
      if (!strcmp(token[itoken], "swarm")) {
        dump_swarms(op);
        continue;
      }
      if (!strcmp(token[itoken], "table")) {
        dump_table(op);
        continue;
      }
      if (!strcmp(token[itoken], "tbond")) {
        dump_tbond(op);
        continue;
      }
      if (!strcmp(token[itoken], "tether")) {
        dump_tethers(op);
        continue;
      }
      if (!strcmp(token[itoken], "tgroup")) {
        TgroupDump(op);
        continue;
      }
      if (!strcmp(token[itoken], "ttarget")) {
        dump_ttargets(op);
        continue;
      }
      if (!strcmp(token[itoken], "variable")) {
        dump_variable(op);
        continue;
      }
      if (!strcmp(token[itoken], "velocity")) {
        dump_velocity(op);
        continue;
      }
    } /* End of for (itoken) */
    goto DONE;
  }

  /**** Kdock ****/

  if (!strcmp(token[0], "kdock")) {
    if (nused)
      kdock(potentials, forces, nused, itemp[1], ftemp[2], itemp[3], ftemp[4],
            ftemp[5], ftemp[6]);
    goto DONE;
}

  /**** Expand ****/

  if (!strcmp(token[0], "expand")) {
    expand(itemp[1 ], itemp[2 ], ftemp[3 ], ftemp[4 ], ftemp[5 ], ftemp[6 ],
           ftemp[7 ], ftemp[8 ], ftemp[9 ], ftemp[10], ftemp[11], ftemp[12],
           ftemp[13], ftemp[14]);
    goto DONE;
  }
  /**** Inactive ****/

  if (!strcmp(token[0], "inactive")) {
    activate(itemp[1], itemp[2], FALSE);
    goto DONE;
  }

  /**** Init4d ****/

  if (!strcmp(token[0], "init4d")) {
    init_fourd(ftemp[1]);
    goto DONE;
  }

  /**** Mom ****/

  if (!strcmp(token[0], "mom")) {
    mom(op, ftemp[1], itemp[2], echo);
    goto DONE;
  }

  /**** MomAdd ****/

  if (!strcmp(token[0], "momadd")) {
    mom_add(itemp[1], itemp[2]);
    goto DONE;
  }

  /**** Monitor ****/

  if (!strcmp(token[0], "monitor")) {
    monitor(potentials, forces, nused, op);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Morse ****/

  if (!strcmp(token[0], "morse")) {
    if (!morse(itemp[1], itemp[2], ftemp[3], ftemp[4], ftemp[5]))
      aaerror("Can't add to morse structure - data structure error");
    goto DONE;
  }

  /**** Noel ****/

  if (!strcmp(token[0], "noel")) {
    if (!noel(itemp[1], itemp[2], ftemp[3], ftemp[4], ftemp[5], ftemp[6], ftemp[7]))
      aaerror("Can't add to noel structure - data structure error");
#ifdef GRAMMP
    else
      send_noel(itemp[1], itemp[2], ftemp[3], ftemp[4], ftemp[5], ftemp[6], ftemp[7]);
#endif
    goto DONE;
  }

  /**** Normal ****/

  if (!strcmp(token[0], "normal")) {
    FDnormal(forces, nused, echo, op, ftemp[1]);
    goto DONE;
  }

  /**** Nzinactive ****/

  if (!strcmp(token[0], "nzinactive")) {
    inactivate_non_zero(itemp[1], itemp[2]);
    goto DONE;
  }

  /**** Orbit ****/

  if (!strcmp(token[0], "orbit")) {
    if (!strcmp(token[1], "1")) {
      orbital(Or1, itemp[2], -1, -1, -1, -1, -1, itemp[3], zero, zero, zero, zero,
              itemp[4], itemp[5]);
      goto DONE;
    }

    if (!strcmp(token[1], "1o")) {
      orbital(Or1o, itemp[2], -1, -1, -1, -1, -1, itemp[3], ftemp[4], ftemp[5],
              ftemp[6], ftemp[7], itemp[8], itemp[9]);
      goto DONE;
    }

    if (!strcmp(token[1], "2")) {
      orbital(Or2, itemp[2], itemp[3], -1, -1, -1, -1, itemp[4], ftemp[5],
              zero, zero, zero, itemp[6], itemp[7]);
      goto DONE;
    }

    if (!strcmp(token[1], "3")) {
      orbital(Or3, itemp[2], itemp[3], itemp[4], -1, -1, -1, itemp[5], ftemp[6],
              ftemp[7], zero, zero, itemp[8], itemp[9]);
      goto DONE;
    }

    if (!strcmp(token[1], "4s")) {
      orbital(Or4s, itemp[2], itemp[3], itemp[4], itemp[5], -1, -1, itemp[6],
              ftemp[7], zero, zero, zero, itemp[8], itemp[9]);
      goto DONE;
    }

    if (!strcmp(token[1], "4p")) {
      orbital(Or4p, itemp[2], itemp[3], itemp[4], itemp[5], -1, -1, itemp[6],
              ftemp[7], zero, zero, zero, itemp[8], itemp[9]);
      goto DONE;
    }

    if (!strcmp(token[1], "m")) {
      orbital(Orm, itemp[2], itemp[3], itemp[4], itemp[5], itemp[6], itemp[7],
              itemp[8], zero, zero, zero, zero, itemp[9], itemp[10]);
      goto DONE;
    }

    aaerror("Unknown orbital type");
    goto DONE;
  }

  /**** Output ****/

  if (!strcmp(token[0], "output")) {

    /**** Of a non-zero version then write it out ****/

    if (itemp[2] > 0) {
      sprintf(errmes, "%s.%d", token[1], itemp[2]);
      newfile = fopen(errmes, "w");
    } else newfile = fopen(token[1], "w");
    if (newfile == NULL)
      aaerror("Can't open the file \"%s\" for write", token[1]);
    else
      read_eval_do(ip, newfile);
    goto DONE;
  }

  /**** Pac ****/

  if (!strcmp(token[0], "pac")) {
    pac(forces, nused, itemp[1], ftemp[2]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Print ****/

  if (!strcmp(token[0], "print")) {
    if (ivar[1]) {
      if (ivar[1] -> type == AMMP_VAR_TYPE_INTEGER) fprintf(op, "%d", itemp[1]);
      else fprintf(op, "%f", ftemp[1]);
    } else PrintStr(op, token[1]);
    if (!itemp[2]) fprintf(op, "\n");
    goto DONE;
  }

  /**** Read ****/

  if (!strcmp(token[0], "read")) {
    if ((newfile = fopen(token[1], "r")) == NULL) {
      aaerror("Can't open the file \"%s\" for read", token[1]);
    } else {
      read_eval_do(newfile, op);
      fclose(newfile);
#ifdef GRAMMP
      send_eof();
#endif
    }
    goto DONE;
  }

  /**** Restore ****/

  if (!strcmp(token[0], "restore")) {
    if (!CloneRestore(NULL, abs(itemp[1])))
      aaerror("Unable to restore from clone %d", itemp[1]);
    goto DONE;
  }

  /**** Restrain ****/

  if (!strcmp(token[0], "restrain")) {
    if (!restrain(itemp[1], itemp[2], ftemp[3], ftemp[4]))
      aaerror("Can't add to restrain structure - data structure error");
    goto DONE;
  }

  /**** Statclone ****/

  if (!strcmp(token[0], "statclone")) {
    statclone(op);
    goto DONE;
  }

  /**** Step ****/

  if (!strcmp(token[0], "step")) {
    step(itemp[1], itemp[2], ftemp[3], ftemp[4], ftemp[5], ftemp[6],
         ftemp[7], ftemp[8]);
    goto DONE;
  }

  /**** Subclone ****/

  if (!strcmp(token[0], "subclone")) {
    for(itoken = 1; (itoken < MAXTOKEN) && (itemp[itoken]); itoken++)
    subclone(op, itemp + 1, itoken - 1);
    goto DONE;
  }

  /**** Swarm ****/

  if (!strcmp(token[0], "swarm")) {
    if (itemp[2] > 8) itemp[2] = 8;
    swarm(ftemp[1], itemp[2], itemp[3], itemp[4], itemp[5 ],
          itemp[6], itemp[7], itemp[8], itemp[9], itemp[10]);
    goto DONE;
  }

  /**** Table ****/

  if (!strcmp(token[0], "table")) {
    create_table(itemp[1], itemp[2]);
    goto DONE;
  }

  /**** Tableent ****/

  if (!strcmp(token[0], "tableent")) {
    add_pair_to_table(itemp[1], itemp[2], ftemp[3], ftemp[4]);
    goto DONE;
  }

  /**** Tailor ****/

  if (!strcmp(token[0], "tailor")) {
    if (!strcmp(token[1], "qab")) {
      tailor_qab(itemp[2], ftemp[3], ftemp[4], ftemp[5]);
      goto DONE;
    }

    if (!strcmp(token[1], "include")) {
      tailor_include(itemp[2], itemp[3]);
      goto DONE;
    }

    if (!strcmp(token[1], "exclude")) {
      tailor_exclude(itemp[2], itemp[3]);
      goto DONE;
    }
    aaerror("Undefined tailor option (%s)", token[1]);
    goto DONE;
  }

  /**** Tbond ****/

  if (!strcmp(token[0], "tbond")) {
    if (!tbond(itemp[1], itemp[2], itemp[3], ftemp[4]))
      aaerror("Can't add to tbond structure - data structure error");
    goto DONE;
  }

  /**** Tether ****/

  if (!strcmp(token[0], "tether")) {
    if (!strcmp(token[1], "all")) {
      if (!alltether(ftemp[2]))
        aaerror("Can't add to tether structure - data structure error");
    } else {
      if (!tether(itemp[1], ftemp[2], ftemp[3], ftemp[4], ftemp[5]))
        aaerror("Can't add to tether structure - data structure error");
#ifdef GRAMMP
      else
        send_tether(itemp[1], ftemp[2], ftemp[3], ftemp[4], ftemp[5]);
#endif
    }
    goto DONE;
  }

  /**** Tgroup ****/

  if (!strcmp(token[0], "tgroup")) {
    Tgroup(itemp[1], itemp[2], itemp[3], itemp[4], itemp[5], ftemp[6], itemp[7], ftemp[8]);
    goto DONE;
  }

  /**** Tjump ****/

  if (!strcmp(token[0], "tjump")) {
    Tjump(potentials, forces, nused, itemp[1], itemp[2], ftemp[3], ftemp[4], ftemp[5], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Tmap ****/

  if (!strcmp(token[0], "tmap")) {
    tmap(op, echo, potentials, nused, itemp[1], itemp[2], itemp[3], itemp[4],
         itemp[5], itemp[6], itemp[7], itemp[8], itemp[9], itemp[10]);
    goto DONE;
  }

  /**** Tmin ****/

  if (!strcmp(token[0], "tmin")) {
    tmin(op, echo, itemp[1], itemp[2], itemp[3], itemp[4], itemp[5], potentials, nused);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE ;
  }

  /**** Trandom ****/

  if (!strcmp(token[0], "trandom")) {
    Trandom(potentials, forces, nused, itemp[1], itemp[2], ftemp[3], itemp[4],
            itemp[5], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE ;
  }

  /**** Tsearch ****/

  if (!strcmp(token[0], "tsearch")) {
    Tsearch(itemp[1], ftemp[2], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Tset ****/

  if (!strcmp(token[0], "tset")) {
    tset(op, echo, itemp[1], itemp[2], itemp[3], itemp[4], AMMP_DEG_TO_RAD(ftemp[5]));
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE ;
  }

  /**** Ttarget ****/

  if (!strcmp(token[0], "ttarget")) {
    ttarget(itemp[1], itemp[2], itemp[3], itemp[4], AMMP_DEG_TO_RAD(ftemp[5]), ftemp[6]);
    goto DONE;
  }

  /**** Velocity ****/

  if (!strcmp(token[0], "velocity")) {
    if (!a_readvelocity(itemp[1], ftemp[2], ftemp[3], ftemp[4]))
      aaerror("Can't update velocity - is this atom defined ?");
    goto DONE;
  }

  /**** Use ****/

  if (!strcmp(token[0], "use")) {
    for(itoken = 1; itoken < MAXTOKEN; itoken++) {
      if (token[itoken][0] == '\0') goto DONE;

      if (!strcmp(token[itoken], "angle")) {
        forces[nused]       = f_angle;
        potentials[nused++] = v_angle;
        continue;
      }

      if (!strcmp(token[itoken], "bond")) {
        forces[nused]       = f_bond;
        potentials[nused++] = v_bond;
        continue;
      }

      if (!strcmp(token[itoken], "hybrid")) {
        forces[nused]       = f_hybrid;
        potentials[nused++] = v_hybrid;
        continue;
      }

      if (!strcmp(token[itoken], "nonbon")) {
        forces[nused]       = u_f_nonbon;
        potentials[nused++] = u_v_nonbon;
        continue;
      }

      if (!strcmp(token[itoken], "none")) {
        nused = 0;
        continue;
      }

      if (!strcmp(token[itoken], "torsion")) {
        forces[nused]       = f_torsion;
        potentials[nused++] = v_torsion;
        continue;
      }

      if (!strcmp(token[itoken], "abc")) {
        forces[nused]       = f_abc;
        potentials[nused++] = v_abc;
        continue;
      }

      if (!strcmp(token[itoken], "av5")) {
        forces[nused]       = f_av5;
        potentials[nused++] = v_av5;
        continue;
      }

      if (!strcmp(token[itoken], "box")) {
        forces[nused]       = f_box;
        potentials[nused++] = v_box;
        continue;
      }

      if (!strcmp(token[itoken], "cangle")) {
        forces[nused]       = f_c_angle;
        potentials[nused++] = v_c_angle;
        continue;
      }

      if (!strcmp(token[itoken], "debye")) {
        forces[nused]       = f_debye;
        potentials[nused++] = v_debye;
        continue;
      }

      if (!strcmp(token[itoken], "fourd")) {
        forces[nused] = f_fourd;
        potentials[nused++] = v_fourd;
        continue;
      }

      if (!strcmp(token[itoken], "hard")) {
        forces[nused]       = f_hard;
        potentials[nused++] = v_hard;
        continue;
      }

      if (!strcmp(token[itoken], "hoangle")) {
        forces[nused]       = f_ho_angle;
        potentials[nused++] = v_ho_angle;
        continue;
      }

      if (!strcmp(token[itoken], "hoav5")) {
        forces[nused]       = f_ho_av5;
        potentials[nused++] = v_ho_av5;
        continue;
      }

      if (!strcmp(token[itoken], "hobond")) {
        forces[nused]       = f_ho_bond;
        potentials[nused++] = v_ho_bond;
        continue;
      }

      if (!strcmp(token[itoken], "honoel")) {
        forces[nused]       = f_ho_noel;
        potentials[nused++] = v_ho_noel;
        continue;
      }

      if (!strcmp(token[itoken], "hohybrid")) {
        forces[nused]       = f_ho_hybrid;
        potentials[nused++] = v_ho_hybrid;
        continue;
      }

      if (!strcmp(token[itoken], "hotether")) {
        forces[nused]       = f_ho_tether;
        potentials[nused++] = v_ho_tether;
        continue;
      }

      if (!strcmp(token[itoken], "mmangle")) {
        forces[nused]       = f_mmangle;
        potentials[nused++] = v_mmangle;
        continue;
      }

      if (!strcmp(token[itoken], "mmbond")) {
        forces[nused]       = f_mmbond;
        potentials[nused++] = v_mmbond;
        continue;
      }

      if (!strcmp(token[itoken], "morse")) {
        forces[nused]       = f_morse;
        potentials[nused++] = v_morse;
        continue;
      }

      if (!strcmp(token[itoken], "noel")) {
        forces[nused]       = f_noel;
        potentials[nused++] = v_noel;
        continue;
      }

      if (!strcmp(token[itoken], "periodic")) {
        forces[nused]       = f_periodic;
        potentials[nused++] = v_periodic;
        continue;
      }

      if (!strcmp(token[itoken], "react")) {
        forces[nused]       = f_react;
        potentials[nused++] = v_react;
        continue;
      }

      if (!strcmp(token[itoken], "restrain")) {
        forces[nused]       = f_restrain;
        potentials[nused++] = v_restrain;
        continue;
      }

      if (!strcmp(token[itoken], "screen")) {
        forces[nused]       = f_screen;
        potentials[nused++] = v_screen;
        continue;
      }

      if (!strcmp(token[itoken], "shadow")) {
        forces[nused]       = f_shadow;
        potentials[nused++] = v_shadow;
        continue;
      }

      if (!strcmp(token[itoken], "step")) {
        forces[nused]       = f_step;
        potentials[nused++] = v_step;
        continue;
      }

      if (!strcmp(token[itoken], "swarm")) {
        forces[nused]       = f_swarm;
        potentials[nused++] = v_swarm;
        continue;
      }

      if (!strcmp(token[itoken], "tbond")) {
        forces[nused]       = f_tbond;
        potentials[nused++] = v_tbond;
        continue;
      }

      if (!strcmp(token[itoken], "tether")) {
        forces[nused]       = f_tether;
        potentials[nused++] = v_tether;
        continue;
      }

      if (!strcmp(token[itoken], "trace")) {
        forces[nused]       = f_trace;
        potentials[nused++] = v_trace;
        continue;
      }

      if (!strcmp(token[itoken], "ttarget")) {
        forces[nused]       = f_ttarget;
        potentials[nused++] = v_ttarget;
        continue;
      }
    } /* End of for (itoken) */
    goto DONE;
  }

  /**** Close ****/

  if (!strcmp(token[0], "close")) {
    if (op != stdout) {
      fclose(op);
      return -1;
    }
    goto DONE;
  }

  /**** Seti ****/

  if (!strcmp(token[0], "seti" )) {
    if (token[1][0] == '\0') {
      aaerror("Seti requires a variable name: seti <name> value");
      goto DONE;
    }
    set_i_variable(token[1], itemp[2]);
    goto DONE;
  }

  /**** Setf ****/

  if (!strcmp(token[0], "setf" )) {
    if (token[1][0] == '\0') {
      aaerror("Setf requires a variable name: setf <name> value");
      goto DONE;
    }
    set_f_variable(token[1], ftemp[2]);
    goto DONE;
  }

  /**** Math commands ****/

  if (math(token, ftemp, itemp, ip, op, echo) > 0 ) goto DONE;

  /**** Trunc ****/

  if (!strcmp(token[0], "trunc" )) {
    if (nused <= 0) goto DONE;
    tncnewt(op, potentials, forces, nused, itemp[1], ftemp[2], ftemp[3], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** V_maxwell ****/

  if (!strcmp(token[0], "v_maxwell")) {
    v_maxwell( ftemp[1],ftemp[2],ftemp[3],ftemp[4]);
    goto DONE;
  }

  /**** V_rescale ****/

  if (!strcmp(token[0], "v_rescale")) {
    v_rescale( ftemp[1]);
    goto DONE;
  }

  /**** Verlet ****/

  if (!strcmp(token[0], "verlet")) {
    verlet(forces, nused, itemp[1], ftemp[2]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Tpac ****/

  if (!strcmp(token[0], "tpac")) {
    tpac(forces, nused, itemp[1], ftemp[2], ftemp[3]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Ppac ****/

  if (!strcmp(token[0], "ppac")) {
    ppac(forces, nused, itemp[1], ftemp[2], ftemp[3]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Ptpac ****/

  if (!strcmp(token[0], "ptpac")) {
    ptpac( forces,nused, itemp[1],ftemp[2],ftemp[3],ftemp[4]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Hpac ****/

  if (!strcmp(token[0], "hpac")) {
    hpac(forces, potentials, nused, itemp[1], ftemp[2], ftemp[3]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Pacpac ****/

  if (!strcmp(token[0], "pacpac")) {
    pacpac(forces, nused, itemp[1], ftemp[2]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Richard ****/

  if (!strcmp(token[0], "richard")) {
    richard(op, echo, potentials, forces, nused, token[1], itemp[2],
            ftemp[3], ftemp[4], ftemp[5]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Trust ****/

  if (!strcmp(token[0], "trust")) {
    if (nused <= 0) goto DONE;
    trust(potentials, forces, nused, itemp[1], ftemp[2], ftemp[3]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Steep ****/

  if (!strcmp(token[0], "steep")) {
    if (nused <= 0) goto DONE;
    steep(potentials, forces, nused, itemp[1], ftemp[2], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Bgfs ****/

  if (!strcmp(token[0], "bfgs")) {
    if (nused <= 0) goto DONE;
    bfgs(potentials, forces, nused, itemp[1], ftemp[2], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Dgeom ****/

  if (!strcmp(token[0], "dgeom")) {
    if( nused <= 0) goto DONE;
    dgeom(op, potentials, nused, itemp[1], itemp[2], ftemp[3], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Noegen ****/

  if (!strcmp(token[0], "noegen")) {
    noel_generate(ftemp[1], ftemp[2]);
    goto DONE;
  }

  /**** Bell ****/

  if (!strcmp(token[0], "bell")) {
    if (nused <= 0) goto DONE;
    bellman(op, potentials, nused, itemp[1], itemp[2], itemp[3], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Kohonen ****/

  if (!strcmp(token[0], "kohonen")) {
    if (nused <= 0) goto DONE;
    kohonen(op, potentials, forces, nused, itemp[1], ftemp[2], itemp[3],
            ftemp[4], ftemp[5], ftemp[6], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Gsdg ****/

  if (!strcmp(token[0], "gsdg")) {
    if (nused <= 0) goto DONE;
    gsdg(op, potentials, nused, itemp[1], itemp[2], itemp[3], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Dscf ****/

  if (!strcmp(token[0], "dscf")) {
    direct_scf(op, itemp[2], ftemp[3], token[1]);
    goto DONE;
  }

  /**** Grasp ****/

  if (!strcmp(token[0], "grasp")) {
    if (nused <= 0) goto DONE;
    grasp(op, echo, potentials, forces, nused, itemp[1], 0, itemp[2], itemp[3],
    itemp[4], token[5]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Genetic ****/

  if (!strcmp(token[0], "genetic")) {
    if (nused <= 0) goto DONE;
    gene(op, potentials, forces, nused, itemp[1], itemp[2], ftemp[3], ftemp[4],itemp[5]);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Cngdel ****/

  if (!strcmp(token[0], "cngdel" )) {
    if (nused > 0) {
      cngdel(potentials, forces, nused, itemp[1], itemp[2], ftemp[3], echo, GetNupdat());
#ifdef GRAMMP
    send_all_atoms();
#endif
    }
    goto DONE;
  }

  /**** Polytope ****/

  if (!strcmp(token[0], "polytope")) {
    if (nused <= 0) goto DONE;
    simplex(op, ftemp[5], itemp[3], ftemp[4], potentials, nused, itemp[1],
            itemp[2], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Rigid ****/

  if (!strcmp(token[0], "rigid")) {
    if (nused <= 0) goto DONE;
    rigid(op, ftemp[5], itemp[3], ftemp[4], potentials, nused, itemp[1],
          itemp[2], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Gdock ****/

  if (!strcmp(token[0], "gdock")) {
    if (nused <= 0) goto DONE;
    gdock(op, ftemp[1], itemp[2], itemp[3], ftemp[4], ftemp[5], potentials,
          nused, itemp[6], itemp[7], echo);
#ifdef GRAMMP
    send_all_atoms();
#endif
    goto DONE;
  }

  /**** Time ****/

#ifdef TIME
  if (!strcmp(token[0], "time")) {
    fprintf(op, "%f CPU\n", ((float)clock()) / CLOCKS_PER_SEC);
    goto DONE;
  }
#endif

  /**** Echo ****/

  if (!strcmp(token[0], "echo")) {
    echo = (strcmp(token[1], "off") != 0);
    goto DONE;
  }

  /**** Exit ****/

  if (!strcmp(token[0], "exit")) {
#ifdef GRAMMP
    itemp[1] = 1;
    WriteToGrfx(&itemp[1],sizeof(int));
    send_terminate();
#endif
    Reset();
    exit(0);
  }

  /**** ConfirmEnd ****/

  if (!strcmp(token[0], "confirmend")) {
    ConfirmEnd = TRUE;
    return TRUE;
  }

  /**** Flush ****/

  if (!strcmp(token[0], "flush")) {
    fflush(op);
    goto DONE;
  }

  /**** Reset ****/

  if (!strcmp(token[0], "reset")) {
    Reset();
    Default();
    goto DONE;
  }

  /**** Send the coordinates to VEGA ZZ ****/

#ifdef AMMP_SENDTOGFX
  if (!strcmp(token[0], "bindump")) {
    if (!strcmp(token[1], "charge")) {
      BinDumpCharge();
    } else if (!strcmp(token[1], "coord")) {
      send_all_atoms();
    } else aaerror("Unknown dump mode. It can be CHARGE and COORD");
    goto DONE;
  }
#endif

  /**** Loopi ****/

  if (!strcmp(token[0], "loopi")) {
    if (token[1][0] == '\0') {
      aaerror("Must have a label to loop to");
      goto DONE;
    }
    if (itemp[4] == 0) itemp[4] = 1;
    newfile = tmpfile();
    if (newfile == NULL) {
      aaerror("Can't open the temporary file in loopi");
      goto DONE;
    }

    /**** Scan the input data until the label is found ****/

    loadloop(ip, newfile, token[1]);

    /**** Now do the loop ****/

    if (itemp[4] > 0) {
      for(itemp[0] = itemp[2]; itemp[0] < itemp[3]; itemp[0] += itemp[4]) {
        inloop = -1;
        if (tisvariable(token[2]))
          set_i_variable(token[2], itemp[0]);
        rewind(newfile);
        read_eval_do(newfile,op);
      } /* End of for (itemp[0]) */
    } else{
      for(itemp[0] = itemp[2]; itemp[0] < itemp[3]; itemp[0] += itemp[4]) {
        inloop = -1;
        if (tisvariable(token[2]))
          set_i_variable(token[2], itemp[0]);
        rewind(newfile);
        read_eval_do(newfile, op);
      } /* End of for (itemp[0]) */
    }
    inloop = 1;
    fclose(newfile);
    goto DONE;
  }

  /**** Loopf ****/

  if (!strcmp(token[0], "loopf")) {
    if (!token[1][0]) {
      aaerror("Must have a label to loop to");
      goto DONE;
    }

    if (ftemp[4] == zero) ftemp[4] = one;

    if ((newfile = tmpfile()) == NULL ) {
      aaerror("Can't open the temporary file in loopi");
      goto DONE;
    }

    /**** Scan the input data until the label is found ****/

    loadloop(ip, newfile, token[1]);

    /**** Now do the loop ****/

    if (ftemp[4] > zero) {
      for(ftemp[0] = ftemp[2]; ftemp[0] < ftemp[3]; ftemp[0] += ftemp[4]) {
        inloop = -1;
        if (tisvariable(token[2]))
          set_f_variable(token[2], ftemp[0]);
        rewind(newfile);
        read_eval_do(newfile, op);
      } /* End of for (ftemp[0]) */
    } else  {
      for(ftemp[0] = ftemp[2]; ftemp[0] > ftemp[3]; ftemp[0] += ftemp[4]) {
        inloop = -1;
        if( tisvariable(token[2]))
          set_f_variable(token[2], ftemp[0]);
        rewind(newfile);
        read_eval_do(newfile, op);
      } /* End of for (ftemp[0]) */
    }
    inloop = 1;
    goto DONE;
  }

  /* Check if its a label and return. Inloop returns -1 if in a loop which
   * causes read_eval_do() to return and activates the loop routine
   */

  for(itemp[0] = 0; itemp[0] < TOKENLENGTH; itemp[0]++) {
    if ((token[0][itemp[0]] == '\0') || (token[0][itemp[0]] == ' ')) {
      if (itemp[0] == 0) break;
      if (token[0][itemp[0] - 1] == ':') return inloop;
    }
  } /* End of for (itemp[0]) */

  /**** Default unrecognized token ****/

  aaerror("Unrecognized token \"%s\"", token[0]);
DONE:

  if (ConfirmEnd) {
    printf("#DONE\n");
    ConfirmEnd = FALSE;
  }

  return TRUE;
}


/**** General error call function ****/

void aaerror(const char *line, ...)
{
  va_list       vl;

  va_start(vl, line);
  fprintf(stderr ,"ERROR: ");
  vfprintf(stderr, line, vl);
  fprintf(stderr, ".\n");
  va_end(vl);
}


/**** General warning call function ****/

void aawarning(const char *line, ...)
{
  va_list       vl;

  va_start(vl, line);
  fprintf(stderr ,"WARNING: ");
  vfprintf(stderr, line, vl);
  fprintf(stderr, ".\n");
  va_end(vl);
}


/* function tisvariable( char *p )
*
* returns 1 if the character string contains anything other than
* <+-><0-9>.<0-9><e><+-><0-9>
* works on a tolowered string !!
*/

int AMMP_FASTCALL tisvariable(char * p)
{
  if( (*p != '+')&&(*p != '-')&& !(isdigit( (int) *p)) &&(*p != '.') )
   return 1;
/* now for the rest we check until either '\0' or not a digit */
  p++;
  while( (*p != '\0') && (isdigit( (int) *p) ) ) p++;
  if( *p == '\0') return 0;
  if( (*p != '.') && (*p != 'e') ) return 1;
  p++;
  if( !(isdigit( (int) *p)) ){
  if( *p == '\0' ) return 0;
  if( (*p != '.') && (*p != 'e') ) return 1;
  p++;
    }
  if( *p == '\0') return 0;
  if( (*p != '+')&&(*p != '-')&& !(isdigit( (int) *p)) &&(*p != '.') )
   return 1;
  p++;
  if( *p == '\0') return 0;
  while( (*p != '\0') && ((isdigit( (int) *p))||(*p=='.')) ) p++;
  if( *p == '\0') return 0;
  return 1;
}


/* function tisint( char *p )
*
* check that a string is <+-><0-9>
* return 1 if true
* return 0 if not
*/

int AMMP_FASTCALL tisint(char *p)
{
  char *pp;
  pp = p;
  while( *pp != '\0')
  { if( *pp == '.') return 0; pp++;}
  if( (*p != '+')&&(*p != '-')&& !(isdigit( (int) *p)) ) return 0;
  p++;
  while (*p != '\0')
  {
    if( !(isdigit( (int) *p )) ) return 0;
    p++;
  }
  return 1;
}


/* routine loadloop( FILE *ip, FILE *tp, char *label)
*
* read lines from ip and write to tp
* when the line begins with label  stop (after writing it )
*/

void AMMP_FASTCALL loadloop(FILE *ip, FILE *tp, char *label)
{
  char line[256], *fgets() ;
  char *sp,*wp;

/*  printf( " the target label >%s<\n" , label);
*/
  while( fgets(  line,256,ip) != NULL )
  {
  fputs( line,tp );
  fputs("\n",tp);
  sp = line;
  while( *sp == ' ' && *sp != '\0') sp++;
  if( *sp != '\0' )
    {
    wp = sp;
    while(*wp != ';' && *wp != ' ' && *wp != '\0')
      { if( isupper(*wp)){*wp = (char)tolower((int)*wp);}
       wp++;}
  if( *wp == ' ' ) *wp = '\0';
  if( *wp == ';' ) *wp = '\0';
    if( strcmp(sp,label) == 0 ) return;
    }
  }
  aaerror("Must have a label for looping. Where is \"%s\" label ?",label);
  return;
}
