/* math.c 
*  perform basic mathematical operations on data
* return values to memory locations
*  uses floating point contagion
*  defined ops
*   add  a b  a <= a+b
*   sub  a b  a <= a - b
*   mul  a b  a <= a*b
*   div  a b  a <= a/b
*   fix  a    a <= (int) a   (drops the fraction for atomdata)
*   sqrt a    a <= sqrt a  (becomes a float !!!)
*   nop  a    just echo the value
*   mov  a b  a <= b
*   randf a   put a random variable 0-<1. in a
*   max  a b   a <= max( a b )
*   min  a b   b <= min( a b )
*   serial a res atom   move the serial number of the atom
*                       into a  (res, atom are like 100 ca)
*   index a i    move the serial number of the i'th atom into a
*
*   linmin  search the atom.<dx,dy,dz> direction for a minimum
*   je a b label:  jump to label if equal
*   jne a b label:  jump to label if  not equal
*   jl a b label:  jump to label if  a < b
*   jg a b label:  jump to label if  a > b
*   jes a string label: jump to label: if a->label == string  
*   jnes a string label: jump to label: if a->label != string  
*        label:  is a label within the current script file
*                which may be within the current loop
*                the j<elg> commands will rewind the input file
*                and search for label: 
*
*
*   data types
*   imeadiate    e.g. a number like 3.14159
*   variable     e.g. a name  like pi
*   atomdata    serial.<x,y,z,fx,fy,fz,dx,dy,dz,vx,vy,vz,q,a,b,m,chi,jaa>
*		serial may be a variable
*		valid format atoms with non-extant serial numbers will be
*		ignored and the operation will silently not happen
*		this allows sums over discontinous atom ranges
*
*
*  routines defined in this module
*
*	math()  does the work
* 	getatomdata() returns a float * or null for an atomdata
*	validatom()  returns nonzero when atomdata format is valid 
*   
*/
/* header from variable.c
*
* variable storage and retreival routines for AMMP
*
* using scope rules for structuring
*
*
*  variables are stored in linked list form
*
*   get_f_variable( char *name, float *fvalue )
*   get_i_variable( char *name,  int *ivalue )
*	returns variable who matches name (all lower case )
*   set_f_variable( char *name, float fvalue )
*   set_i_variable( char *name, int ivalue )
*	sets variable who matches name (all lower case )
*   match_variable( char *name ) returns pointer to name if there NULL if not
*   dump_variable(FILE *output  )
*	 dumps variables to  file 
*/
/*
*  copyright 1992 1993 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#ifdef WIN32
#  include  <windows.h>
#else
#  include <sys/time.h>
#endif

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

#define  ATOMDATA               0
#define  IMEADIATE              1
#define  VARDATA                2

/**** Global variables ****/

extern AMMP_VFUNC       potentials[];
extern AMMP_FFUNC       forces[];
extern int              nused;


/**** Evaluate a math expression ****/

#ifdef ESV
#  define  tokens               (*tokens)
#endif

int AMMP_FASTCALL math(char tokens[][TOKENLENGTH], float fvalue[], int *ivalue, FILE *ip, FILE *op, int echo)

{
  AMMP_ATOM             *ap;
  float                 *foutpointer, *fp, fa, fb;
  int                   adata;
  int                   atype, btype;
  int                   *ioutpointer, i, ia, ib, j;
  VARIABLE              *vp, *vos;

#ifdef WIN32
  LARGE_INTEGER         liDiff;
  LARGE_INTEGER         liFreq;
#else
  struct timeval        tnow;
#endif

  const char *          LabelReq = "Label: required for a jump";

	if( (vp = match_variable(&tokens[1][0])) != NULL)
	{
	adata = VARDATA;
	atype = vp->type;
	vos = vp;
	if( atype == AMMP_VAR_TYPE_FLOAT){ fa = vp->value.f;}
	foutpointer  = &vp->value.f;
	if( atype == AMMP_VAR_TYPE_INTEGER){ia = vp->value.i; }
	ioutpointer = &vp->value.i;
	goto AFOUND;
	}
	if( (foutpointer = getatomdata(&tokens[1][0])) != NULL)
	{
	adata = ATOMDATA;
	atype = AMMP_VAR_TYPE_FLOAT;
	fa = *foutpointer;
	goto AFOUND;
	}
	if( validatom(&tokens[1][0]) != 0 ) return( 1);
	foutpointer = NULL; /* shouldn't change it but this is safe */
	adata = IMEADIATE;
	fa = fvalue[1];
	ia = ivalue[1];
	atype = AMMP_VAR_TYPE_FLOAT;
	if( tisint( &tokens[1][0]) == 1) atype = AMMP_VAR_TYPE_INTEGER;
/*
	if( tisint( &tokens[1][0]) == 1)printf(" INTEGER "); 
	printf(">%s<\n",&tokens[1][0]);
*/
AFOUND:
	if( (vp = match_variable(&tokens[2][0])) != NULL)
	{
/*	bdata = VARDATA;
*/
	btype = vp->type;
	if(btype == AMMP_VAR_TYPE_FLOAT  ) fb = vp->value.f;
	if(btype == AMMP_VAR_TYPE_INTEGER) ib = vp->value.i;
	goto BFOUND;
	}
	if( (fp = getatomdata(&tokens[2][0])) != NULL)
	{
/*	bdata = ATOMDATA;
*/
	btype = AMMP_VAR_TYPE_FLOAT;
	fb = *fp;
	goto BFOUND;
	}
	if( validatom(&tokens[2][0]) != 0 ) return( 1);
 /*	bdata = IMEADIATE;
 */
	fb = fvalue[2];
	ib = ivalue[2];
	btype = AMMP_VAR_TYPE_FLOAT;
	if( tisint( &tokens[2][0]) == 1) btype = AMMP_VAR_TYPE_INTEGER;
/*
	if( tisint( &tokens[2][0]) == 1)printf(" INTEGER "); 
	printf(">%s<\n",&tokens[2][0]);
*/
BFOUND:

    /**** Make the converted data ****/

    if (atype == AMMP_VAR_TYPE_FLOAT) ia = (int)fa;
    else if (atype == AMMP_VAR_TYPE_INTEGER) fa = (float)ia;
    if (btype == AMMP_VAR_TYPE_FLOAT) ib = (int)fb;
    else if (btype == AMMP_VAR_TYPE_INTEGER) fb = (float)ib;

    /**** Add ****/

    if (!strcmp(tokens[0], "add")) {
      if (atype == AMMP_VAR_TYPE_FLOAT) fa = fa + fb;
      else if (atype == AMMP_VAR_TYPE_INTEGER) ia = ia + ib;
      goto GOOD_OP;
    }

    /**** Cos ****/

    if (!strcmp(tokens[0], "cos")) {
      if (adata == ATOMDATA) {
        *foutpointer = cos(fa);
        return TRUE;
      }
      atype = AMMP_VAR_TYPE_FLOAT;
      fa = cos(fa);
      goto GOOD_OP;
    }

    /**** Div ****/

    if (!strcmp(tokens[0], "div")) {
      if (atype == AMMP_VAR_TYPE_FLOAT) fa = fa / fb;
      else if(atype == AMMP_VAR_TYPE_INTEGER) ia = ia / ib;
      goto GOOD_OP;
    }

    /**** Fix ****/

    if (!strcmp(tokens[0], "fix" )) {
      if (adata == ATOMDATA) {
        ia           = (int)fa;
        *foutpointer = (float)ia;
        if (echo) fprintf(op, "%d\n", ia);
        return TRUE;
      }
      if (atype == AMMP_VAR_TYPE_FLOAT) {
        atype = AMMP_VAR_TYPE_INTEGER;
        ia    = (int)fa;
      }
      goto GOOD_OP;
    }

    /**** Index ****/

    if (!strcmp(tokens[0], "index")) {
      ap = a_next(-1);
      for(i = 0; i < ib; i++) ap = a_next(i);
      ia    = ap -> serial;
      atype = AMMP_VAR_TYPE_INTEGER;
      goto GOOD_OP;
    }

    /**** Je ****/

    if (!strcmp(tokens[0], "je")) {
      if (!tokens[3][0]) {
        aaerror(LabelReq);
        goto GOOD_OP;
      }
      if (((atype == AMMP_VAR_TYPE_INTEGER) && (ia == ib)) ||
          ((atype == AMMP_VAR_TYPE_FLOAT  ) && (fa == fb))) {
        rewind(ip);
        math_findlabel( ip,&tokens[3][0]);
      }
      goto GOOD_OP;
    }

    /**** Jes ****/

    if (!strcmp(tokens[0], "jes")) {
      if (!tokens[3][0]) {
        aaerror(LabelReq);
        goto GOOD_OP;
      }
      if ((ap = a_m_serial(ia)) == NULL) return TRUE;
      if (!strcmp(ap -> name, tokens[2])) {
        rewind(ip);
	math_findlabel(ip, tokens[3]);
      }
      goto GOOD_OP;
    }

    /**** Jg ****/

    if (!strcmp(tokens[0], "jg" )) {
      if (!tokens[3][0]) {
        aaerror(LabelReq);
        goto GOOD_OP;
      }
      if (((atype == AMMP_VAR_TYPE_INTEGER) && (ia > ib)) ||
          ((atype == AMMP_VAR_TYPE_FLOAT  ) && (fa > fb))) {
        rewind(ip);
        math_findlabel(ip, &tokens[3][0]);
      }
      goto GOOD_OP;
    }

    /**** Jl ****/

    if (!strcmp(tokens[0], "jl")) {
      if (!tokens[3][0]) {
        aaerror(LabelReq);
        goto GOOD_OP;
      }

      if (((atype == AMMP_VAR_TYPE_INTEGER) && (ia < ib)) ||
          ((atype == AMMP_VAR_TYPE_FLOAT  ) && (fa < fb))) {
        rewind(ip);
        math_findlabel( ip,&tokens[3][0]);
      }
      goto GOOD_OP;
    }

    /**** Jnes ****/

    if (!strcmp(tokens[0], "jnes")) {
      if (!tokens[3][0]) {
        aaerror(LabelReq);
        goto GOOD_OP;
      }
      if ((ap = a_m_serial(ia)) == NULL) {
        rewind(ip);
	math_findlabel(ip, tokens[3]);
        goto GOOD_OP;
      }
      if (strcmp(ap -> name, tokens[2])) {
        rewind(ip);
	math_findlabel(ip, &tokens[3][0]);
      }
      goto GOOD_OP;
    }

    /**** Jne ****/

    if (!strcmp(tokens[0], "jne" )) {
      if (!tokens[3][0]) {
        aaerror(LabelReq);
        goto GOOD_OP;
      }
      if (((atype == AMMP_VAR_TYPE_INTEGER) && (ia != ib)) ||
          ((atype == AMMP_VAR_TYPE_FLOAT  ) && (fa != fb))) {
        rewind(ip);
        math_findlabel( ip,&tokens[3][0]);
      }
      goto GOOD_OP;
    }

    /**** Linmin ****/

    if (!strcmp(tokens[0], "linmin")) {
      fa = linmin(potentials, nused, sqrt(a_max_d()));
      if (echo) fprintf(op, "%f step to minimum\n", fa);
      a_inc_d(fa);
      goto GOOD_OP;
    }

    /**** Max ****/

    if (!strcmp(tokens[0], "max")) {
      if (atype == AMMP_VAR_TYPE_FLOAT) {
        if (fa < fb) fa = fb;
      } else if (atype == AMMP_VAR_TYPE_INTEGER) {
        if (ia < ib) ia = ib;
      }
      goto GOOD_OP;
    }

    /**** Min ****/

    if (!strcmp(tokens[0], "min")) {
      if (atype == AMMP_VAR_TYPE_FLOAT) {
        if (fa > fb) fa = fb;
      } else if (atype == AMMP_VAR_TYPE_INTEGER) {
        if (ia > ib) ia = ib;
      }
      goto GOOD_OP;
    }

    /**** Mov ****/

    if (!strcmp(tokens[0], "mov")) {
      ia    = ib;
      fa    = fb;
      atype = btype;
      goto GOOD_OP;
    }

    /**** Mul ****/

    if (!strcmp(tokens[0], "mul")) {
      if (atype == AMMP_VAR_TYPE_FLOAT) fa = fa * fb;
      else if (atype == AMMP_VAR_TYPE_INTEGER) ia = ia * ib;
      goto GOOD_OP;
    }

    /**** Nop ****/

    if (!strcmp(tokens[0], "nop")) {
      goto GOOD_OP;
    }

    /**** Randf ****/

    if (!strcmp(tokens[0], "randf")) {
      atype = AMMP_VAR_TYPE_FLOAT;
      fa    = randf();
      goto GOOD_OP;
    }

    /**** Serial ****/

    if (!strcmp(tokens[0], "serial")) {
      ia = 100 * ib - 1;
      j  = ia + 100;
      i  = -1;
      while((ap = a_next(i)) != NULL) {
        i = 1;
        if ((ap -> serial > ia) && (ap -> serial < j) &&
            (math_match_atom(tokens[3], ap))) {
          atype = AMMP_VAR_TYPE_INTEGER;
          ia    = ap->serial;
          goto GOOD_OP;
        }
      } /* End of while */

      /**** Never a serial number ****/

      ia    = -1;
      atype = AMMP_VAR_TYPE_INTEGER;
      goto GOOD_OP;
    }

    /**** Sin ****/

    if (!strcmp(tokens[0], "sin")) {
      if (adata == ATOMDATA) {
        *foutpointer = sin(fa);
        return TRUE;
      }
      atype = AMMP_VAR_TYPE_FLOAT;
      fa = sin(fa);
      goto GOOD_OP;
    }

    /**** Sqrt ****/

    if (!strcmp(tokens[0], "sqrt")) {
      if (adata == ATOMDATA) {
        if (fa > one) *foutpointer = sqrt(fa);
        else *foutpointer = -sqrt(-fa);
        if (echo) fprintf(op, "%f\n", *foutpointer);
        return TRUE;
      }
      atype = AMMP_VAR_TYPE_FLOAT;
      if (fa > 0) fa = sqrt(fa);
      else fa = -sqrt(-fa);
      goto GOOD_OP;
    }

    /**** Sub ****/

    if (!strcmp(tokens[0], "sub")) {
      if (atype == AMMP_VAR_TYPE_FLOAT) fa = fa - fb;
      else if (atype == AMMP_VAR_TYPE_INTEGER) ia = ia - ib;
      goto GOOD_OP;
    }

    /**** Tan ****/

    if (!strcmp(tokens[0], "tan")) {
      if (adata == ATOMDATA) {
        *foutpointer = tan(fa);
        return TRUE;
      }
      atype = AMMP_VAR_TYPE_FLOAT;
      fa = tan(fa);
      goto GOOD_OP;
    }

    /**** Time ****/

    if (!strcmp(tokens[0], "time")) {
      atype = AMMP_VAR_TYPE_FLOAT;
#ifdef WIN32
      QueryPerformanceCounter(&liDiff);
      QueryPerformanceFrequency(&liFreq);
      fa = ((double)liDiff.QuadPart / (double)liFreq.QuadPart);
#else
      gettimeofday(&tnow, NULL);
      fa = ((double)tnow.tv_sec + (double)tnow.tv_usec * 1.0e-06);
#endif
      goto GOOD_OP;
    }
    return(-1);

  /**** If found we jump here ****/

GOOD_OP:

  if ((tisvariable(&tokens[1][0])) && (tokens[1][0] != '\0') && (adata == IMEADIATE)) {
    set_i_variable(&tokens[1][0],0);
    adata       = VARDATA;
    vos         = match_variable(&tokens[1][0]);
    foutpointer = &vos -> value.f;
    ioutpointer = &vos -> value.i;
  }

  if (adata != IMEADIATE ) {
    if ((adata == ATOMDATA) && (atype == AMMP_VAR_TYPE_INTEGER))
      atype = AMMP_VAR_TYPE_FLOAT;
    if (atype == AMMP_VAR_TYPE_FLOAT) {
      if (foutpointer) *foutpointer = fa;
    } else if ((atype == AMMP_VAR_TYPE_INTEGER) && (ioutpointer)) *ioutpointer = ia;
  }
  if (echo) {
    if (atype == AMMP_VAR_TYPE_INTEGER) fprintf(op, "%d\n", ia);
    else if (atype == AMMP_VAR_TYPE_FLOAT) fprintf(op, "%f\n", fa);
  }

  return TRUE;
}

#ifdef ESV
#  undef  tokens
#endif

/* int validatom()
*  given a string return 0 if it is not of the form
*  stuff.<x y z fx fy fz dx dy dz vx vy vz q a b m chi jaa na>
*  these being the valid atomdata parameters 
*  returns 1 to 19 for x y z fx fy fz dx dy dz vx vy vz q a b m chi jaa na
*  so we don't have to lex again
*/

int AMMP_FASTCALL validatom(char *who)
{
char *cp,*pp,*cp1,*cp2,*cp3;
int i;
	cp = who;
	i = 0;
	while ( *cp != '\0')
	{
	if( *cp == '.') {i++;  pp = cp;}
	cp++;
	}
	if( i != 1) return (0 ); /* there can only be one '.' */
	cp = pp ; cp++;
/* now we check the tailing characters */
	cp1 = cp; cp1++;
	cp2 = cp1; cp2++;  
	cp3 = cp2; cp3++;  
	if( *cp1 == '\0')
	{
	if( *cp == 'x') return (1);
	if( *cp == 'y') return (2);
	if( *cp == 'z') return (3);
	if( *cp == 'q') return (13);
	if( *cp == 'a') return (14);
	if( *cp == 'b') return (15);
	if( *cp == 'm') return (16);
	return (0);
	}
	if( *cp2 == '\0')
	{
	if( *cp == 'f') {
	if( *cp1 == 'x') return (4);
	if( *cp1 == 'y') return (5);
	if( *cp1 == 'z') return (6);
			}
	if( *cp == 'd' ){
	if( *cp1 == 'x') return (7);
	if( *cp1 == 'y') return (8);
	if( *cp1 == 'z') return (9);
			}
	if( *cp == 'v' ){
	if( *cp1 == 'x') return (10);
	if( *cp1 == 'y') return (11);
	if( *cp1 == 'z') return (12);
			}
	if( *cp == 'n' ) {
		if( *cp1 =='a') return 19;
			}
	}
	if( *cp3 == '\0')
	{
	if( *cp == 'c' && *cp1 == 'h' && *cp2 == 'i') return 17;
	if( *cp == 'j' && *cp1 == 'a' && *cp2 == 'a') return 18;
	}
	return (0);	
}

/* float *getatomdata() returns a float * or null for an atomdata
*/

float * AMMP_FASTCALL getatomdata(char *who)
{

int i,j;
char aser[TOKENLENGTH],*cp;
AMMP_ATOM *ap;
VARIABLE *vp;
static float fx,fy,fz;
	i = validatom( who );
	if( i == 0 ) return ( NULL ); /* if not the right format it aint one */
	
	cp = who; j = 0;
	while( *cp != '.') 
	{ aser[j++] = *cp; cp++; }
	aser[j] = '\0';
	if( (vp = match_variable(aser)) == NULL)
	{ j = atoi(aser); } else { 
		if( vp ->type == AMMP_VAR_TYPE_INTEGER) j = vp->value.i;
		if( vp ->type == AMMP_VAR_TYPE_FLOAT) j = (int)vp->value.f;
				}
	ap = a_m_serial(j);
	if( ap == NULL) return( NULL );
	if( i == 1 ) return ( &ap->x );
	if( i == 2 ) return ( &ap->y );
	if( i == 3 ) return ( &ap->z );
	if( i == 4 ) { fx = ap->fx ; return( &fx);}
	if( i == 5 ) { fy = ap->fy ; return(&fy);}
	if( i == 6 ) { fz = ap->fz ; return(&fz);}
	if( i == 7 ) return ( &ap->dx );
	if( i == 8 ) return ( &ap->dy );
	if( i == 9 ) return ( &ap->dz );
	if( i == 10 ) return ( &ap->vx );
	if( i == 11 ) return ( &ap->vy );
	if( i == 12 ) return ( &ap->vz );
	if( i == 13 ) return ( &ap->q );
	if( i == 14 ) return ( &ap->a );
	if( i == 15 ) return ( &ap->b );
	if( i == 16 ) return ( &ap->mass );
	if( i == 17 ) return ( &ap->chi );
	if( i == 18 ) return ( &ap->jaa );
	if( i == 19 ) return ( &ap->na );
	return (NULL );
}
/*	math_match_atom(char* atomname, AMMP_ATOM * ap)
*
*  find if the atomname is part of the atom .name field
*  as in atomname = "ca" and ap->name = arg.ca
*
*/

int AMMP_FASTCALL math_match_atom(char *who, AMMP_ATOM *ap)
{
  char *        cp = ap -> name;

  while(*cp != '.') {
    if (!*cp) return FALSE;
    cp++;
  } /* End of while */
  cp++;

  return (strcmp(who, cp) == 0);
}

/*
*	    math_findlabel(FILE * ip, char *label);
* search a file for label:
*  if label doesn't end in : add it
*
*/

void AMMP_FASTCALL math_findlabel(FILE *fp, char *label)
{
char *cp,*lp;
char llabel[TOKENLENGTH];
char myline[TOKENLENGTH]; /* since label is no longer than TOKENLENGTH we can skip such lines */
int  inmyline;
int i;
char ac;

cp = label;
lp  = &llabel[0];
	while( *cp != '\0')
	{
		*lp = *cp ; lp++; cp ++;
	}
	cp = lp; cp--;
	if( *cp != ':'){ *lp = ':'; lp++;}
	*lp = '\0';
/* now one char at a time scan ip for a blank */
	inmyline = 0;
	lp  = &llabel[0];
	while( (i= fgetc( fp )) != EOF )
	{
	ac = (char)i;
	if( !isspace((int) ac) )
	{
	if( ac == ';')
	{
	myline[inmyline] = '\0';
/*	printf(">%s<\n",&myline[0]);
*/
	if( strcmp( lp,&myline[0]) == 0 ) return ;
	inmyline = 0;
	}else{
	if( inmyline > TOKENLENGTH) inmyline = 0;
	myline[inmyline++] = ac;
	}
	}
	}

}
