/*  gdock.c
*
*  genetic algorithm docker
*
* AMMP version
*  given a range of atom ID's and uses them to define hull
*  also uses the total potential,
*
*
*  copyright 1998 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify
*  this notice
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/* Specific definitions to pre-tune algorithm
 * change these and re-compile
 */

#define  MAX_PROGENY            4
#define  MIN_GENE               8


/**** Genetic algorithm/quarternion rigid body solver ****/

void AMMP_FASTCALL gdock(FILE *op, float toler, int ngene, int niter, float vara, float varx,
                         AMMP_VFUNC potent[], int inpotent, int imin, int imax, int echo)
{
  AMMP_ATOM     *(*ap)[];
  int           i, j, k, natom;

#ifdef VEGAZZ
  int           lastiter;

  int           nupdat = GetNupdat();
  int           supdat = GetSupdat();
#endif

  /* int nargs */
  /* quarternion_rot_tran() normalizes the quarternion and applies the
   * translation */

  /* store the genome here */

  float         *gene  = NULL;
  float         *genet;

  /* store the energy and "fitness" */

  float         *fvals   = NULL;
  float         *fitness = NULL;
  float         fbest, best[7];

  /* stud register */

  int           *studbook;

  /* use these to analyze the values */

  int           ibest;
  int           worst;

  /**** float sigma ****/

  float         x, y, z;

  const char *  Routine = "gdock()";

  if (!CheckAtoms()) return;
  if (ngene < MIN_GENE) ngene = MIN_GENE;
  if (niter < 1) niter = 1;
  if (imin <= 0) imin = 1;
  if (imax <= 0) imax = a_number();
  if (vara <= zero) vara = one;
  if (varx <= zero) varx = one;

#ifdef VEGAZZ
  lastiter = niter - 1;
#endif

  /**** Allocate the genome q1..q4, x, y, z X ngene storage ****/

  if (((studbook = (int   *)Alloca(ngene * sizeof(int  ), Routine)) == NULL) ||
      ((fvals    = (float *)Alloca(ngene * sizeof(float), Routine)) == NULL) ||
      ((fitness  = (float *)Alloca(ngene * sizeof(float), Routine)) == NULL) ||
      ((gene     = (float *)Alloca(ngene * 7 * sizeof(float), Routine)) == NULL) ||
      ((genet    = (float *)Alloca(ngene * 7 * sizeof(float), Routine)) == NULL)) {
    SafeFree(studbook);
    SafeFree(fvals   );
    SafeFree(fitness );
    SafeFree(gene    );
    return;
  }

  /* initialize the geneome
   *  always include the current position */

  gene[0] = zero;
  gene[1] = zero;
  gene[2] = zero;
  gene[3] = one;
  gene[4] = zero;
  gene[5] = zero;
  gene[6] = zero;

  for(i = 1; i < ngene; i++) {
    for(j = 0; j < 3; j++) {
      k = i * 7 + j;
      gene[k] = gene[j] + (two * randf() - one) * varx;
    } /* End of for (j) */
    for(j = 3; j < 7; j++) {
      k = i * 7 + j;
      gene[k] = gene[j] + (two * randf() - one) * vara;
    } /* End of for (j) */
  } /* End of for (i) */

  /* figure out how many atoms there are in the range given */

  if (imin > imax) {
    i    = imin;
    imin = imax;
    imax = i;
  }

  natom = 0;
  for(i = imin; i <= imax; i ++)
    if (a_m_serial(i) != NULL) natom++;

  if (!natom) {
    free(studbook);
    free(fvals   );
    free(fitness);
    free(gene    );
    free(genet   );
    return;
  }

/*      nargs = 7; */

  if ((ap = Alloca(natom * sizeof(AMMP_ATOM *), Routine)) == NULL) {
    free(studbook);
    free(fvals   );
    free(fitness);
    free(gene    );
    free(genet   );
    return;
  }

  /**** Now gather up the atoms ****/

  for(i = 0; i < natom; i++) (*ap)[i] = NULL;
  j = 0;
  for(i = imin; i <= imax; i ++) {
    if (((*ap)[j] = a_m_serial(i)) != NULL) {
      (*ap)[j]->gx = (*ap)[j]->x;
      (*ap)[j]->gy = (*ap)[j]->y;
      (*ap)[j]->gz = (*ap)[j]->z;
      j++;
    }
    if (j == natom) break;
  } /* End of for (i) */

  /**** Initialize the fvals data structure ****/

  fbest   = 10.e10;
  best[0] = zero;
  best[1] = zero;
  best[2] = zero;
  best[3] = one;
  best[4] = zero;
  best[5] = zero;
  best[6] = one;

  /**** How to do the potential ****/

  for(i = 0; i < ngene; i++) {
    quarternion_rot_tran(gene + (i * 7), ap, natom);
    fvals[i] = zero;
    for(j = 0; j < inpotent; j++) {
      if ((*potent[j]) != v_nonbon && (*potent[j]) != u_v_nonbon)
          (*potent[j])(fvals + i, 0.0f);
      else
        zone_nonbon(fvals + i, 0.0f, ap, natom);
    } /* End of for (j) */

    if (fvals[i] < fbest){
      fbest = fvals[i];
      for(j = 0; j < 7; j++) best[j] = gene[7 * i + j];
    }
  } /* End of for (i) */

  /**** Main loop ****/

  for(k = 0; k < niter; k++) {
    ibest = simplex_get_best(fvals, ngene - 1);

    if (simplex_get_var(fvals, ngene - 1) < toler) goto DONE;
    worst = simplex_get_worst(fvals, ngene - 1);


#ifdef VEGAZZ
    if ((supdat) && ((!(k % supdat)) || (k == lastiter)))
#else
    if (echo)
#endif
      fprintf(op, "  Gdock  %5d: best %d (energy %f), worst %d (energy %f)\n",
              k, ibest, fvals[ibest], worst, fvals[worst]);


        /* make up the new genome */
        /* evaluate "fitness" */

        x = fvals[ibest];
	y = fvals[worst];
	z = fabs(x - y);
	if (z > 0.0f) z = 1.0f / z;

        /* linearly scale */

        x = 0.0f;
        for(i = 0; i < ngene; i++) {
          fitness[i] = fabs(fvals[i] - y) * z;
          x += fitness[i];
          studbook[i] = 0; /* might as well here */
        }

        /* normalize */

	if (x > 0.0f) x = 1.0f / x;
        for(i = 0; i < ngene; i++)
          fitness[i] *= x;

        /* convert to cumulative */
        /* dont , we just need deltas
	x = 0.;
	for( i=0; i< ngene; i++)
	{
		x += (*fitness)[i];
		(*fitness)[i] = x;
	}
        */
        /* now select new trials */

	for(i = 0; i < ngene; i++) {

          /* setup with replacement */

REDO_THE_GENE:
	  worst = 0;
	  for(j = 0; j < ngene; j++) {
            if (studbook[j] < MAX_PROGENY) {
              if (randf() <= fitness[j]) {
                worst++;
                genet[i * 7    ] = gene[j * 7    ] + (2.0f * randf() - 1.0f) * varx * 0.1f;
		genet[i * 7 + 1] = gene[j * 7 + 1] + (2.0f * randf() - 1.0f) * varx * 0.1f;
		genet[i * 7 + 2] = gene[j * 7 + 2] + (2.0f * randf() - 1.0f) * varx * 0.1f;
		genet[i * 7 + 3] = gene[j * 7 + 3] + (2.0f * randf() - 1.0f) * vara * 0.1f;
		genet[i * 7 + 4] = gene[j * 7 + 4] + (2.0f * randf() - 1.0f) * vara * 0.1f;
		genet[i * 7 + 5] = gene[j * 7 + 5] + (2.0f * randf() - 1.0f) * vara * 0.1f;
		genet[i * 7 + 6] = gene[j * 7 + 6] + (2.0f * randf() - 1.0f) * vara * 0.1f;
		studbook[j] += 1;
		goto REPLACED;
              }
            }
          } /* End of for (j) */
          if (worst == 0) goto REDO_THE_GENE;
REPLACED:;
	} /* End of for (i) */

        /* now recombine a few */

	for(worst = 0; worst < 2; worst ++) {
          i = randf() * ngene;
	  j = randf() * ngene;
	  if( i == j) continue; /* don't bother if i'm me */

          /* only recombine orientations vs translations */

	  for(ibest = 3; ibest < 7; ibest++) {
            x = genet[i * 7 + ibest];
	    genet[i * 7 + ibest] = genet[j * 7 + ibest];
	    genet[j * 7 + ibest] = x;
          }
        }

        /* copy the best over - this aids the convergence */

	ibest = simplex_get_best(fvals, ngene - 1);
        for(i = 0; i < 7; i++)
          genet[ibest * 7 + i] = gene[ibest * 7 + i];

        /* lazy,lazy,lazy programming - copy the new genome over */

        for(i = 0; i < (7 * ngene); i++)
          gene[i] = genet[i];

        for(i = 0; i < ngene; i++) {

        /* how to do the potential */

        quarternion_rot_tran(gene + (i * 7), ap, natom);
#ifdef VEGAZZ
        if ((nupdat) && ((!(k % nupdat)) || (k == lastiter))) send_all_atoms();
#endif

#ifdef GRAMMP
	send_all_atoms();
#endif
        fvals[i] = 0.0f;
        for(j = 0; j < inpotent; j++) {
          if ((*potent[j]) != v_nonbon && (*potent[j]) != u_v_nonbon)
            (*potent[j])(fvals + i, 0.0f);
          else
            zone_nonbon(fvals + i, 0.0f, ap, natom);
        }
	if (fvals[i] < fbest) {
          fbest = fvals[i];
          for(j = 0; j < 7; j++)
            best[j] = gene[7 * i + j];
        }
      }
    } /*k */

DONE:
#ifndef VEGAZZ
  if (echo)
#endif
    fprintf(op, "  Gdock       : Putting best %d (enegy %f) into coordinates\n\n",
            ibest, fbest);
  quarternion_rot_tran(best, ap, natom);

  free(ap      );
  free(gene    );
  free(genet   );
  free(fitness );
  free(fvals   );
  free(studbook);
}


