/* gene.c
*
* genetic optimizer for AMMP
*
* given potentials use the genetic algorithm to find an optimum
*
*/
/*
*  copyright 1993,1994 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

#define MAX_PROGENY 4


int AMMP_FASTCALL gene(FILE *op, AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int nstep,
                       int ndeep, float sigma, float target, int nos)
{
  AMMP_ATOM     *ap;
  float         (*xvs)[];
  float         vmax, vmin;
  float         sigo, mean, t, x, y;
  int           (*studbook)[];
  int           thebest, theworst;
  int           k, l;

  const char *  Routine       = "gene()";
  const char *  StepStr       = "  Genetic %4d: %f %f %f\n";
  float         (*yvs)[]      = NULL;
  float         (*zvs)[]      = NULL;
  float         (*value)[]    = NULL;
  float         (*cumvalue)[] = NULL;
  int           numatm        = a_number();
  int           j             = ndeep * sizeof(float);
  int           i             = j * numatm;

  if (!CheckAtoms()) return FALSE;
  if (nstep < 1) nstep = 1;

  if (((xvs      = Alloca(i                  , Routine)) == NULL) ||
      ((yvs      = Alloca(i                  , Routine)) == NULL) ||
      ((zvs      = Alloca(i                  , Routine)) == NULL) ||
      ((value    = Alloca(j                  , Routine)) == NULL) ||
      ((cumvalue = Alloca(j                  , Routine)) == NULL) ||
      ((studbook = Alloca(ndeep * sizeof(int), Routine)) == NULL)) {
    SafeFree(xvs     );
    SafeFree(yvs     );
    SafeFree(zvs     );
    SafeFree(value   );
    SafeFree(cumvalue);
    return FALSE;
  }

  for(i = 0; i < numatm; i++) {
    ap = a_next(i);
    k  = 0;
    for(j = 0; j < ndeep; j++) {
      (*xvs)[k + i] = ap -> x ;
      if (ap -> active) (*xvs)[k + i] += randg() * sigma;
      (*yvs)[k + i] = ap -> y ;
      if (ap -> active) (*yvs)[k + i] += randg() * sigma;
      (*zvs)[k + i] = ap -> z ;
      if (ap -> active) (*zvs)[k + i] += randg() * sigma;
      k += numatm;
    } /* End of for (j) */
  } /* End of for (i) */

  k = 0;
  for(j = 0; j < ndeep; j++) {
    (*value)[j] = 0;
    for(i = 0; i < numatm; i++) {
      ap      = a_next(i);
      ap -> x = (*xvs)[k + i];
      ap -> y = (*yvs)[k + i];
      ap -> z = (*zvs)[k + i];
    } /* End of for (i) */
    k += numatm;
    gene_valid_coordinate();
    for(i = 0; i < nfs; i++)
      (*vfs[i])( &(*value)[j], 0.0f);
#ifndef VEGAZZ
    if (echo)
#endif
      fprintf(op, "  Genetic     : Initialization %d to %f\n", j, (*value)[j]);
  } /* End of for (j) */

  /**** Do the work ****/

  for(l = 0; l < nstep; l++) {
    sigo     = 0.0f;
    mean     = 0.0f;
    vmin     = 10e10;
    thebest  = -1;
    vmax     = -10e10;
    theworst = -1;
    for(j = 0; j < ndeep; j++) {
      if ((*value)[j] < vmin) {
        vmin    = (*value)[j];
        thebest = j;
      }
      if ((*value)[j] > vmax) {
        vmax     = (*value)[j];
        theworst = j;
      }
      sigo += (*value)[j] * (*value)[j];
      mean += (*value)[j];
    } /* End of for (l) */

    mean = mean / ndeep;
    sigo = sqrt(sigo / ndeep - mean * mean);
    if (sigo < target) goto DONE;
    for(j = 0; j < ndeep; j++) (*studbook)[j] = 0;

RE_BUILD:
    x = 0.0f;
    for(j = 0; j < ndeep; j++) {
      if ((*studbook)[j] < MAX_PROGENY){
        y = vmax -(*value)[j] ;
        t = vmax - mean - 2.0f * sigo;
        if (y < t) y = t;
        x += y;
        (*cumvalue)[j] = x;
      }
    } /* End of for (j) */

    if (x <= 1.e-4) goto DONE;
    for(j = 0; j< ndeep; j++) {
      if ((*studbook)[j] < MAX_PROGENY)
        (*cumvalue)[j] /= x;
    } /* End of for (j) */

    mean = randf();
    for(j = 1; j < ndeep; j++) {
      if (((*studbook)[j] < MAX_PROGENY) &&
          ((*cumvalue)[j - 1] < mean) &&
          ((*cumvalue)[j] > mean)) {
        thebest = j;
        break;
      }
    } /* End of for (j) */

    (*studbook)[thebest] += 1;
    if ((*studbook)[thebest] > MAX_PROGENY) goto RE_BUILD;

    /**** Make up the coords ****/

    k = thebest * numatm;
    if ((nos > 0) && (l < ndeep)) k = l * numatm;
    for(i = 0; i < numatm; i++) {
      ap   = a_next(i);
      mean = randf();
      if (mean < (2.0f / (float)numatm)) {
        mean = randf();
        if ((*cumvalue)[0] > mean) {
          k = 0;
        } else {
          for(j = 1; j < ndeep ; j++) {
            if (((*studbook)[j] < MAX_PROGENY) &&
                ((*cumvalue)[j-1] < mean && (*cumvalue)[j] > mean)) {
              k = j * numatm;
              break;
            }
          } /* End of for (j) */
        }
        if ((nos > 0) && (l < ndeep)) k = l * numatm;
      }

      /**** Make up the coords ****/

#if 0
      if (ap -> active) {
        ap -> x = (*xvs)[k + i] /* + 0.4f * randf() - 0.2f */;
        ap -> y = (*yvs)[k + i] /* + 0.4f * randf() - 0.2f */;
        ap -> z = (*zvs)[k + i] /* + 0.4f * randf() - 0.2f */;
      } else {
        ap -> x = (*xvs)[k + i];
        ap -> y = (*yvs)[k + i];
        ap -> z = (*zvs)[k + i];
      }
#else
      ap -> x = (*xvs)[k + i];
      ap -> y = (*yvs)[k + i];
      ap -> z = (*zvs)[k + i];
#endif
    } /* End of for (i) */

    /**** Minimize a little ****/

    gene_valid_coordinate();
    cngdel(vfs, ffs, nfs, nos, nos, 0.0f, FALSE, GetNupdat());

    /**** Insert into the queue ****/

    x       = (*value)[0];
    thebest = 0;
    for(i = 1; i < ndeep; i++) {
      if ((*value)[i] < x) {
        x       = (*value)[i];
        thebest = i;
      }
    } /* End of for (i) */

    x = 0.0f;
    for(i = 0; i < nfs; i++) (*vfs[i])(&x, 0.0f);

#ifndef VEGAZZ
    if (echo)
#endif
      fprintf(op, StepStr, l,
             (*value)[theworst], x, (*value)[thebest]);

    if (x < vmax) {
      k = theworst * numatm;
      (*value)[theworst] = x;
      for(i = 0; i < numatm; i++) {
        ap            = a_next(i);
	(*xvs)[k + i] = ap -> x;
	(*yvs)[k + i] = ap -> y;
	(*zvs)[k + i] = ap -> z;
      } /* End of for (i) */
    }
  } /* End of for (l) */

  /**** Clean up ****/

DONE:
  vmin    = 10e10;
  thebest = -1;

  for(j = 0; j < ndeep; j++) {
    if ((*value)[j] < vmin ) {
      vmin = (*value)[j];
      thebest = j;
    }
  } /* End of for (j) */

  thebest *= numatm;
  for(i = 0; i < numatm; i++) {
    ap      = a_next(i);
    ap -> x = (*xvs)[thebest + i];
    ap -> y = (*yvs)[thebest + i];
    ap -> z = (*zvs)[thebest + i];
  } /* End of for (i) */

  free(studbook);
  free(cumvalue);
  free(value   );
  free(zvs     );
  free(yvs     );
  free(xvs     );

#ifndef VEGAZZ
  if (echo)
#endif
    fprintf(op, "\n");

  return TRUE;
}


void AMMP_FASTCALL gene_valid_coordinate(void)
{
  AMMP_ATOM     *ap1, *ap2;
  float         x, y, z;
  int           na;
  int           i, j;

  na = a_number();
  if (na < 1) return;

  ap1 = a_next(-1);
  ap1 = ap1 -> next;
  for(i = 1; i < na; i++) {
    for(j = 0; j < i; j++) {
      ap2 = a_next(j);
      x   = fabs(ap1 -> x - ap2 -> x);
      if (x < 1.e-5) {
        y = fabs(ap1 -> y - ap2 -> y);
        if (y < 1.e-5) {
          z = fabs(ap1 -> z - ap2 -> z);
          if (z < 1.e-5) {
            ap2 -> x += 1.e-4;
            ap2 -> y += 1.e-4;
            ap2 -> z += 1.e-4;
          }
        }
      }
    }
    ap1 = ap1->next;
  } /* End of for (i) */
}
