
/*************************************************
****    AMMP - Grid-based nonbond potential   ****
**** Copyright 2006-2012, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <ctype.h>
#include <string.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Cell grid structure ****/

typedef struct __ammp_gcell {
  AMMP_ATOM **          Atoms;          /* Atom array                         */
  int                   AtomsMax;       /* Atom array size                    */
  int                   Num;            /* Number of atoms in the Atoms array */
  struct __ammp_gcell **NbrCells;       /* Neighbor cell array                */
  float *               AtmCoord;       /* Atom coordinates                   */
} AMMP_GCELL;

/**** Grid structure ****/

typedef struct {
  AMMP_GCELL *          Cells;          /* Cell array pointer                 */
  float                 Cutoff;         /* Cutoff distance                    */
  float                 Cutoff2;        /* Cutoff distance square             */
  int                   StartNumInBox;  /* Number of starting atoms for cell  */
  int                   Xb, Yb, Zb;     /* Number of cell for axis            */
  int                   NumAtoms;       /* Number of atoms                    */
  int                   TotCell;        /* Total number of cells              */
  int                   TotXYCell;      /* Number of cell for row             */
} AMMP_GGRID;

/**** Local variables ****/

static AMMP_GGRID *     NbGrid = NULL;


/**** Reset local variables ****/

void AMMP_FASTCALL ResetGNonBon(void)
{
  register AMMP_GCELL * Cell;
  register int          k;

  if (!NbGrid) return;

  if (NbGrid -> Cells) {
    for(k = 0; k < NbGrid -> TotCell; ++k) {
      Cell = NbGrid -> Cells + k;
      if (Cell -> Atoms   ) free(Cell -> Atoms   );
      if (Cell -> AtmCoord) free(Cell -> AtmCoord);
      if (Cell -> NbrCells) free(Cell -> NbrCells);
    } /* End of for (k) */
    free(NbGrid -> Cells);
  }

  free(NbGrid);
  NbGrid = NULL;
}


/**** Initialize the grid ****/

int AMMP_FASTCALL GNonBonInit(void)
{
  AMMP_ATOM *   Atm;
  AMMP_GCELL *  Cell;
  float         Xmin, Xmax, Ymin, Ymax, Zmin, Zmax;
  int           Nbrs[14];
  int           i, k, n, Xi, Yi, Zi;
  int           TotXYCell, Xb, Yb, Zb;
  int           Aindex, Axb, Ayb, Azb;

  const char *  Routine = "GNonBonInit()";

  if ((NbGrid = (AMMP_GGRID *)AllocaC(sizeof(AMMP_GGRID), Routine)) == NULL)
    return FALSE;

  NbGrid -> Cutoff = get_f_variable("cutoff");
  NbGrid -> NumAtoms = a_number();

  if (NbGrid -> Cutoff <= 0) {
    aaerror("Invalid cutoff value");
    ResetGNonBon();
    return FALSE;
  }

  NbGrid -> Cutoff2 = NbGrid -> Cutoff * NbGrid -> Cutoff;

  /**** Evaluate the molecule dimensions ****/

  Atm  = a_next(0);
  Xmin = Xmax = Atm -> x;
  Ymin = Ymax = Atm -> y;
  Zmin = Zmax = Atm -> z;
  for(i = 1; i < NbGrid -> NumAtoms; i++) {
    Atm = a_next(i);
    if (Atm -> x < Xmin) Xmin = Atm -> x;
    else if (Atm -> x > Xmax) Xmax = Atm -> x;

    if (Atm -> y < Ymin) Ymin = Atm -> y;
    else if(Atm -> y > Ymax) Ymax = Atm -> y;

    if(Atm -> z < Zmin) Zmin = Atm -> z;
    else if(Atm -> z > Zmax) Zmax = Atm -> z;
  } /* End of for (i) */

  Xb                  = (int)((Xmax - Xmin) / NbGrid -> Cutoff) + 1;
  Yb                  = (int)((Ymax - Ymin) / NbGrid -> Cutoff) + 1;
  Zb                  = (int)((Zmax - Zmin) / NbGrid -> Cutoff) + 1;
  TotXYCell           = Yb * Xb;
  NbGrid -> NumAtoms  = a_number();
  NbGrid -> Xb        = Xb;
  NbGrid -> Yb        = Yb;
  NbGrid -> Zb        = Zb;
  NbGrid -> TotXYCell = TotXYCell;
  NbGrid -> TotCell   = TotXYCell * Zb;

  if ((NbGrid -> Cells = (AMMP_GCELL *)AllocaC(sizeof(AMMP_GCELL) * NbGrid -> TotCell, Routine)) == NULL) {
    ResetGNonBon();
    return FALSE;
  }

  /**** Put the atoms in the cells ****/

  NbGrid -> StartNumInBox = CeilF((NbGrid -> Cutoff * NbGrid -> Cutoff * NbGrid -> Cutoff) / 8.0);

  for(i = 0; i < NbGrid -> NumAtoms; i++) {
    Atm    = a_next(i);
    Axb    = (int)((Atm -> x - Xmin) / NbGrid -> Cutoff);
    Ayb    = (int)((Atm -> y - Ymin) / NbGrid -> Cutoff);
    Azb    = (int)((Atm -> z - Zmin) / NbGrid -> Cutoff);
    Cell   = NbGrid -> Cells + (Azb * TotXYCell + Ayb * Xb + Axb);
    if (!Cell -> Atoms) {
      if ((Cell -> Atoms = (AMMP_ATOM **)Alloca(sizeof(AMMP_ATOM *) * NbGrid -> StartNumInBox, Routine)) == NULL) {
        ResetGNonBon();
        return FALSE;
      }
      Cell -> AtomsMax = NbGrid -> StartNumInBox;
    } else if (Cell -> Num == Cell -> AtomsMax) {
      Cell -> AtomsMax *= 2;
      if ((Cell -> Atoms = (AMMP_ATOM **)ReAlloca(Cell -> Atoms, sizeof(AMMP_ATOM *) * Cell -> AtomsMax, Routine)) == NULL) {
        ResetGNonBon();
        return FALSE;
      }
    }
    Cell -> Atoms[Cell -> Num++] = Atm;
  } /* End of for (i) */

  /**** Create neighbor list for each box ****/

  Aindex = 0;
  for(Zi = 0; Zi < Zb; Zi++) {
    for(Yi = 0; Yi < Yb; Yi++) {
      for(Xi = 0; Xi < Xb; Xi++) {
        Cell      = NbGrid -> Cells + Aindex;
        n         = 0;
        Nbrs[n++] = Aindex;
        if (Xi < Xb - 1) Nbrs[n++] = Aindex + 1;
        if (Yi < Yb - 1) Nbrs[n++] = Aindex + Xb;
        if (Zi < Zb - 1) Nbrs[n++] = Aindex + TotXYCell;
        if (Xi < (Xb - 1) && Yi < (Yb - 1)) Nbrs[n++] = Aindex + Xb + 1;
        if (Xi < (Xb - 1) && Zi < (Zb - 1)) Nbrs[n++] = Aindex + TotXYCell + 1;
        if (Yi < (Yb - 1) && Zi < (Zb - 1)) Nbrs[n++] = Aindex + TotXYCell + Xb;
        if (Xi < (Xb - 1) && Yi > 0) Nbrs[n++] = Aindex - Xb + 1;
        if (Xi > 0 && Zi < (Zb - 1)) Nbrs[n++] = Aindex + TotXYCell - 1;
        if (Yi > 0 && Zi < (Zb - 1)) Nbrs[n++] = Aindex + TotXYCell - Xb;
        if (Xi < (Xb - 1) && Yi < (Yb - 1) && Zi < (Zb - 1))
          Nbrs[n++] = Aindex + TotXYCell + Xb + 1;
        if (Xi > 0 && Yi < (Yb - 1) && Zi < (Zb - 1))
          Nbrs[n++] = Aindex + TotXYCell + Xb - 1;
        if (Xi < (Xb - 1) && Yi > 0 && Zi < (Zb - 1))
          Nbrs[n++] = Aindex + TotXYCell - Xb + 1;
        if (Xi > 0 && Yi > 0 && Zi < (Zb - 1))
          Nbrs[n++] = Aindex + TotXYCell - Xb - 1;

        if (((Cell -> NbrCells = (AMMP_GCELL **)Alloca(sizeof(AMMP_GCELL *) * (n + 1), Routine)) == NULL) ||
            ((Cell -> Num) &&
             ((Cell -> AtmCoord = (float *)Alloca(sizeof(float) * 3 * Cell -> Num, Routine)) == NULL))) {
          ResetGNonBon();
          return FALSE;
        }
        for(k = 0; k < n; ++k) Cell -> NbrCells[k] = NbGrid -> Cells + Nbrs[k];
        Cell -> NbrCells[n] = 0;
        Aindex++;
      } /* End of for (Xi) */
    } /* End of for (Yi) */
  } /* End of for (Zi) */

  return TRUE;
}


/**** Sum the potential ****/

int AMMP_FASTCALL g_v_nonbon(float *V, float lambda)
{
  AMMP_ATOM             *Atm1, *Atm2;
  AMMP_GCELL *          Cell;
  AMMP_GCELL **         NbrCell;
  float                 *Coord1, *Coord2;
  float                 Cutoff2, Dielectric, Rdebye;
  float                 En, r, r6, t;
  int                   j, k, Aindex, Offset;

  if ((!NbGrid) && (!GNonBonInit())) return FALSE;

  Cutoff2    = NbGrid -> Cutoff2;
  Dielectric = Q_CONST / GetDielectric();
  En         = 0.0f;

  Cell = NbGrid -> Cells;
  if (lambda) {
    for(Aindex = 0; Aindex < NbGrid -> TotCell; ++Aindex) {
      Coord1 = Cell -> AtmCoord;
      for(k = 0; k < Cell -> Num; ++k) {
        Atm1 = Cell -> Atoms[k];
        Coord1[0] = Atm1 -> x + Atm1 -> dx * lambda;
        Coord1[1] = Atm1 -> y + Atm1 -> dy * lambda;
        Coord1[2] = Atm1 -> z + Atm1 -> dz * lambda;
        Coord1 += 3;
      } /* End of for (k) */
      ++Cell;
    } /* End of for (Aindex) */
  } else {
    for(Aindex = 0; Aindex < NbGrid -> TotCell; ++Aindex) {
      Coord1 = Cell -> AtmCoord;
      for(k = 0; k < Cell -> Num; ++k) {
        Atm1 = Cell -> Atoms[k];
        Coord1[0] = Atm1 -> x;
        Coord1[1] = Atm1 -> y;
        Coord1[2] = Atm1 -> z;
        Coord1 += 3;
      } /* End of for (k) */
      ++Cell;
    } /* End of for (Aindex) */
  }

  for(Aindex = 0, Cell = NbGrid -> Cells; Aindex < NbGrid -> TotCell; ++Aindex, ++Cell) {
    if (!Cell -> Atoms) continue;
    for(NbrCell = Cell -> NbrCells; *NbrCell; ++NbrCell) {
      if (!(*NbrCell) -> Atoms) continue;
      for(k = 0, Coord1 = Cell -> AtmCoord; k < Cell -> Num; ++k, Coord1 += 3) {
        Atm1 = Cell -> Atoms[k];
        if (Cell == *NbrCell) Offset = k + 1;
        else Offset = 0;
        for(j = Offset, Coord2 = (*NbrCell) -> AtmCoord + Offset * 3; j < (*NbrCell) -> Num; ++j, Coord2 += 3) {
          Atm2 = (*NbrCell) -> Atoms[j];
          r  = Coord1[0] - Coord2[0];
          r *= r;
          t  = Coord1[1] - Coord2[1];
          r += t * t;
          t  = Coord1[2] - Coord2[2];
          r += t * t;

          if ((!r) || (r > Cutoff2)) continue;
//          r6 = r * r * r;
          r  = sqrt(r);
          En += Atm1 -> q * Atm2 -> q / r * Dielectric;
/*
          En += Atm1 -> q * Atm2 -> q / r * Dielectric -
                Atm1 -> a * Atm2 -> a / r6 +
                Atm1 -> b * Atm2 -> b / (r6 * r6);
*/
        } /* End of for (j) */
      } /* End of for (k) */
    } /* End of for (NbrCells) */
  } /* End of for (Aindex) */
  *V += En;


  return TRUE;
}


/**** Increment the forces ****/

int AMMP_FASTCALL g_f_nonbon(float lambda)
{
  AMMP_ATOM             *Atm1, *Atm2;
  AMMP_GCELL *          Cell;
  AMMP_GCELL **         NbrCell;
  float *               Coord;
  float                 Cutoff2, Dielectric, Rdebye;
  float                 k0, r, r2, r6, t;
  float                 aa, bb, qq, xx, yy, zz;
  float                 fx, fy, fz;
  float                 xt, yt, zt;
  int                   j, k, Aindex;


  if ((!NbGrid) && (!GNonBonInit())) return FALSE;

  Cutoff2    = NbGrid -> Cutoff2;
  Dielectric = Q_CONST / GetDielectric();
  Rdebye     = NbGrid -> Cutoff * 0.5f;

  Cell = NbGrid -> Cells;
  if (lambda) {
    for(Aindex = 0; Aindex < NbGrid -> TotCell; ++Aindex) {
      Coord = Cell -> AtmCoord;
      for(k = 0; k < Cell -> Num; ++k) {
        Atm1     = Cell -> Atoms[k];
        Coord[0] = Atm1 -> x + Atm1 -> dx * lambda;
        Coord[1] = Atm1 -> y + Atm1 -> dy * lambda;
        Coord[2] = Atm1 -> z + Atm1 -> dz * lambda;
        Coord += 3;
      } /* End of for (k) */
      ++Cell;
    } /* End of for (Aindex) */
  } else {
    for(Aindex = 0; Aindex < NbGrid -> TotCell; ++Aindex) {
      Coord = Cell -> AtmCoord;
      for(k = 0; k < Cell -> Num; ++k) {
        Atm1     = Cell -> Atoms[k];
        Coord[0] = Atm1 -> x;
        Coord[1] = Atm1 -> y;
        Coord[2] = Atm1 -> z;
        Coord += 3;
      } /* End of for (AtmPtr) */
      ++Cell;
    } /* End of for (Aindex) */
  }

  for(Aindex = 0, Cell = NbGrid -> Cells; Aindex < NbGrid -> TotCell; ++Aindex, ++Cell, Coord += 3) {
    if (!Cell -> Atoms) continue;
    for(k = 0; k < Cell -> Num; ++k) {
      Atm1 = Cell -> Atoms[k];
      aa = Atm1 -> a;
      bb = Atm1 -> b;
      xx = Atm1 -> x + lambda * Cell -> Atoms[k] -> dx;
      yy = Atm1 -> y + lambda * Cell -> Atoms[k] -> dy;
      zz = Atm1 -> z + lambda * Cell -> Atoms[k] -> dz;
      qq = Atm1 -> q;
      for(NbrCell = Cell -> NbrCells; *NbrCell; ++NbrCell) {
        Coord = (*NbrCell) -> AtmCoord;
        for(j = 0; j < (*NbrCell) -> Num; ++j) {
          Atm2 = (*NbrCell) -> Atoms[j];
          if (Atm1 == Atm2) continue;
          xt = xx - Coord[0];
          yt = yy - Coord[1];
          zt = zz - Coord[2];
          r2 = xx * xx + yy * yy + zz * zz;
          if (r2 > Cutoff2) continue;
          if (r2 < 1.0f) {
            r6 = 1.0f;
            r2 = 1.0f;
            r  = 1.0f;
          } else {
            r6 = r2 * r2 * r2 ;
            r  = sqrt(r2);
          }
          xt /= r;
          yt /= r;
          zt /= r;

          k0  = -qq * Atm2 -> q * Dielectric * exp(-r / Rdebye) *
                (1.0f /(Rdebye * r) + 1.0f / r2) ;
          k0 += aa * Atm2 -> a/r6/r * 6.0f;
    k0 -= bb * Atm2 -> b/r6/r6/r * 12.0f;
          xt *= k0;
          yt *= k0;
          zt *= k0;

          /**** Increment the forces ****/

          Atm1 -> fx -= xt;
          Atm1 -> fy -= yt;
          Atm1 -> fz -= zt;
          Atm2 -> fx += xt;
          Atm2 -> fy += yt;
          Atm2 -> fz += zt;
        } /* End of for (j) */
      } /* End of for (NbrCells) */
    } /* End of for (k) */
  } /* End of for (Aindex) */

  return TRUE;
}

