/* gsdg.c
*
*  Gauss-Siedel Distance Geometry
*  
*  iteratively solve distance geometry equations
*  one atom at a time, but update the calculated distance estimates
*  each time.  (we know from the PMDG experiments that this is not
*  too expensive)
*/
/*
*  copyright 1993,1994 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

void AMMP_FASTCALL gsdg(FILE *op, AMMP_VFUNC vfs[], int nfs, int niter, int low_serial,
                        int high_serial, int echo)
{
  AMMP_ATOM             *ap, *bp;
  int                   numatm;
  int                   iter, i, j, k;
  float                 svec[3], rvec[3]; /* storage for search vectors */
  float                 y;

#ifdef VEGAZZ
  int                   supdat;
#endif

 const char *  StepStr = "  Gsdg %7d\n";

  if (!CheckAtoms()) return;

  numatm = a_number();

  if (low_serial > high_serial) {
    i           = low_serial;
    low_serial  = high_serial;
    high_serial = i;
  }

  if (high_serial <= 0) {
    for(i = 0; i < numatm; i++) {
      ap = a_next(i);
      if (high_serial < ap -> serial) high_serial = ap->serial;
    } /* End of for (i) */
  }

#ifdef VEGAZZ
  if (!echo) supdat = 0;
  else supdat = GetSupdat();
#endif

  for(iter = 0; iter < niter; iter++) {

#ifdef VEGAZZ
    if ((supdat) && (!(iter % supdat)))
#else
    if (echo)
#endif
      printf(StepStr, iter);

    ap = a_next(-1);
    for(i = 0; i < numatm; i++) {
      if (ap == NULL) break;
      if ((ap -> active) &&
          (ap -> serial >= low_serial) && (ap->serial <= high_serial)) {
        for(j = 0; j < numatm; j++) {
          bp       = a_next(j);
          bp -> vx = 16.0f;                     /* Default to 4A separation  */
          bp -> vy = 0.0f;                      /* but only as a lower bound */
        } /* End of for (j) */

        for(j = 0; j < nfs; j++) {
          if ((vfs[j] == v_nonbon) || (vfs[j] == u_v_nonbon)) {
            for(k = 0; k < numatm; k++) {
              bp       = a_next(k);
              bp -> vy = -10.0f;
            } /* End of for (k) */
            break;
          }
        } /* End of for (j) */

        for(j = 0; j < nfs; j++) {
          if ((vfs[j] == v_bond ) || (vfs[j] == v_mmbond ) || (vfs[j] == v_abc    )) gsdg_bond(ap);
          if ((vfs[j] == v_angle) || (vfs[j] == v_mmangle) || (vfs[j] == v_c_angle)) gsdg_angle(ap);
          if (vfs[j] == v_noel) gsdg_noel(ap);
/*
          if (vfs[j] == v_hard) gsdg_nonbon(ap)
          if ((vfs[j] == v_nonbon) || (vfs[j] == u_v_nonbon)) gsdg_nonbon(ap)
*/
        } /* End of for (j) */

        rvec[0] = 0.0f;
        rvec[1] = 0.0f;
        rvec[2] = 0.0f;
        rand3(&svec[0], &svec[1], &svec[2]);
        gsdg_line_search(svec, &y, ap);
        rvec[0] += y * svec[0];
        rvec[1] += y * svec[1];
        rvec[2] += y * svec[2];
        rand3(&svec[0], &svec[1], &svec[2]);
        gsdg_line_search(svec, &y, ap);
        rvec[0] += y * svec[0];
        rvec[1] += y * svec[1];
        rvec[2] += y * svec[2];
        rand3(&svec[0], &svec[1], &svec[2]);
        gsdg_line_search(svec, &y, ap);
        rvec[0] += y * svec[0];
        rvec[1] += y * svec[1];
        rvec[2] += y * svec[2];
        gsdg_line_search(rvec, &y, ap);

        ap -> x += y * rvec[0];
        ap -> y += y * rvec[1];
        ap -> z += y * rvec[2];
      }
      if (ap == ap -> next ) break;
      ap = ap -> next;
    } /* End of for (i) */

#ifdef GRAMMP
    send_all_atoms();
#endif
  } /* End of for (iter)*/

#ifdef VEGAZZ
  if (supdat) {
    if ((iter % supdat) || (iter == niter))
      printf(StepStr, iter);
    printf("\n");
  }
#else
  if (echo) printf("\n");
#endif
}


/**** Line search ****/

float AMMP_FASTCALL gsdg_line_search(float vect[3], float *step, AMMP_ATOM *who)
{
  float         val;
  float         vt, lam;
  float         dstep;
  int           i, j;

  val   = gsdg_dgeom(vect, 0.0f, who);
  lam   =  0.0f;
  *step =  0.0f;
  dstep = -0.5f;

  for(i = 0; i < 3; i++) {
    dstep *= -0.5f;
    for(j = 0; j < 200 ; j++) {
      lam += dstep;
      vt   = gsdg_dgeom(vect, lam, who);
      if (vt < val) {
        *step = lam;
        val   = vt;
      } else break;
    } /* End of for (j) */

    if (j == 200) dstep *= -2.0f;
  }

  return val;
}


float AMMP_FASTCALL gsdg_dgeom(float vect[3], float lam, AMMP_ATOM *who)
{
  AMMP_ATOM *           ap;
  float                 x, y, z;
  float                 dt;
  float                 dsum;
  int                   numatm;
  int                   i;

  numatm = a_number();
  x = who -> x + vect[0] * lam;
  y = who -> y + vect[1] * lam;
  z = who -> z + vect[2] * lam;

  dsum = 0.0f;
  for(i = 0; i < numatm; i++) {
    ap = a_next(i);
    if (ap != who) {
      dt  = (x - ap -> x) * (x - ap -> x);
      dt += (y - ap -> y) * (y - ap -> y);
      dt += (z - ap -> z) * (z - ap -> z);
      if (ap -> vy > 0.0f)
        dsum += ap -> vy * (ap -> vx - dt) * (ap -> vx - dt);
      else if (ap -> vx > dt)
        dsum -= ap -> vy * (ap -> vx - dt) * (ap -> vx - dt);
    }
  } /* End of for (i) */

  return dsum;
}


/**** Dgeom trace routines ****/

int AMMP_FASTCALL v_trace(float *V , float lambda)
{
  AMMP_ATOM *           ap;
  float                 xc, yc, zc;
  float                 xt, yt, zt;
  float                 fnumatm, l_trace;
  int                   numatm, i;

  numatm = a_number();
  if (numatm < 2) return FALSE;

  l_trace = GetTrace();
  fnumatm = 1.0f / (float)numatm;

  if (!l_trace) l_trace = fnumatm;
  xc = 0.0f;
  yc = 0.0f;
  zc = 0.0f;

  for(i = 0; i < numatm; i++) {
    ap  = a_next(i);
    xc += ap -> x + lambda * ap -> dx;
    yc += ap -> y + lambda * ap -> dy;
    zc += ap -> z + lambda * ap -> dz;
  } /* End of for (i) */

  xc *= fnumatm;
  yc *= fnumatm;
  zc *= fnumatm;
  for(i = 0; i < numatm; i++) {
    ap  = a_next(i);
    xt  = ap -> x + lambda * ap -> dx - xc;
    yt  = ap -> y + lambda * ap -> dy - yc;
    zt  = ap -> z + lambda * ap -> dz - zc;
    *V -= l_trace * (xt * xt + yt * yt + zt * zt);
  } /* End of for (i) */

  return TRUE;
}


int AMMP_FASTCALL f_trace(float lambda)
{
  AMMP_ATOM *           ap;
  float                 xc, yc, zc;
  float                 xt, yt, zt;
  float                 fnumatm, l_trace;
  int                   numatm, i;

  numatm = a_number();
  if (numatm < 2) return FALSE;

  l_trace = GetTrace();
  fnumatm = 1.0f / (float)numatm;

  if (!l_trace) l_trace = fnumatm;

  xc = 0.0f;
  yc = 0.0f;
  zc = 0.0f;

  for(i = 0; i < numatm; i++) {
    ap  = a_next(i);
    xc += ap -> x + lambda * ap -> dx;
    yc += ap -> y + lambda * ap -> dy;
    zc += ap -> z + lambda * ap -> dz;
  } /* End of for (i) */
  xc *= fnumatm;
  yc *= fnumatm;
  zc *= fnumatm;

  l_trace = 2.0f * l_trace * (1.0f - fnumatm);
  for(i = 0; i < numatm; i++) {
    ap        = a_next(i);
    xt        = ap -> x + lambda * ap -> dx - xc;
    yt        = ap -> y + lambda * ap -> dy - yc;
    zt        = ap -> z + lambda * ap -> dz - zc;
    ap -> fx += l_trace * xt;
    ap -> fy += l_trace * yt;
    ap -> fz += l_trace * zt;
  } /* End of for (i) */

  return TRUE;
}
