/* hybrid.c
*
* collection of routines to service bond hybrid potentials
*
* these are the planar and pyrmid height potentials
*
*
*
* POOP (Poor-mans Object Oriented Programming) using scope rules
*
* these routines hold a data base (in terms of array indeces)
* of hybrid, with the associated length and force constant
*
* (this could be table driven but what the hell memories cheap)
*
* the routines for potential value, force and (eventually) second
* derivatives are here also
*
* force and 2nd derivative routines assume zero'd arrays for output
* this allows for parralellization if needed (on a PC?)
*
* forces are bond wise symmetric - so we don't have to fuck around with
* s matrices and the like.
*/
/*
*  copyright 1992 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify
*  this notice
*/

#include <stdio.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"


/**** Sum the hybrid potential by the atom range ****/

int AMMP_FASTCALL a_hybrid(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
  AMMP_ATOM     *a1, *a2, *a3, *a4;
  float         x1, y1, z1, x2, y2, z2, x3, y3, z3;
  float         r, cx1, cy1, cz1;
  float         hite;

  HYBRID        *bp = hybrid_first;

  if (!bp) return TRUE;

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    a3 = bp -> atom3;
    a4 = bp -> atom4;
    if (((a1 -> serial >= ilow) && (a1 -> serial <= ihigh)) ||
        ((a2 -> serial >= ilow) && (a2 -> serial <= ihigh)) ||
        ((a3 -> serial >= ilow) && (a3 -> serial <= ihigh)) ||
        ((a4 -> serial >= ilow) && (a4 -> serial <= ihigh))) {
      if (lambda) {
        x1 = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
        y1 = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
        z1 = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
        x2 = a3 -> x - a1 -> x + lambda * (a3 -> dx - a1 -> dx);
        y2 = a3 -> y - a1 -> y + lambda * (a3 -> dy - a1 -> dy);
        z2 = a3 -> z - a1 -> z + lambda * (a3 -> dz - a1 -> dz);
        x3 = a4 -> x - a1 -> x + lambda * (a4 -> dx - a1 -> dx);
        y3 = a4 -> y - a1 -> y + lambda * (a4 -> dy - a1 -> dy);
        z3 = a4 -> z - a1 -> z + lambda * (a4 -> dz - a1 -> dz);
      } else {
        x1 = a2 -> x - a1 -> x;
        y1 = a2 -> y - a1 -> y;
        z1 = a2 -> z - a1 -> z;
        x2 = a3 -> x - a1 -> x;
        y2 = a3 -> y - a1 -> y;
        z2 = a3 -> z - a1 -> z;
        x3 = a4 -> x - a1 -> x;
        y3 = a4 -> y - a1 -> y;
        z3 = a4 -> z - a1 -> z;
      }

      /**** 1 cross 2 ****/

      cx1 = y1 * z2 - y2 * z1;
      cy1 = x2 * z1 - x1 * z2;
      cz1 = x1 * y2 - x2 * y1;
      r = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;
      if (r > 1.e-16) {
        r = sqrt(r);

        /**** Height is x3 vectordot cx1 (normalized) ****/

        hite = ((cx1 * x3 + cy1 * y3 + cz1 * z3) / r) - bp -> offset;
        z2   = bp -> k * hite * hite;
        *V  += z2;
        fprintf(op, "Hybrid %s %d %s %d %s %d %s %d E %f value %f error %f\n",
                a1 -> name, a1 -> serial, a2 -> name, a2 -> serial,
                a3 -> name, a3 -> serial, a4 -> name, a4 -> serial,
                z2, hite, hite);
      }
    }

    if (bp == bp -> next) return TRUE;
    bp = bp->next;
  }

  return TRUE;
}


/**** Dump the hybrids ****/

void AMMP_FASTCALL dump_hybrids(FILE *where)
{
  HYBRID        *b = hybrid_first;

  while(b) {
    fprintf(where, "hybrid %d %d %d %d %f %f;\n",
            b -> atom1 -> serial, b -> atom2 -> serial,
            b -> atom3 -> serial, b -> atom4 -> serial,
            b -> k, b -> offset);
    if (b -> next == b) break;
    b = b -> next;
  } /* End of while */
}


/**** Increment the hybrid force (homotopy version) ****/

int AMMP_FASTCALL f_ho_hybrid(float lambda)
{
  AMMP_ATOM     *a1, *a2, *a3, *a4, *at;
  float         x1, y1, z1, x2, y2, z2, x3, y3, z3;
  float         r, cx1, cy1, cz1;
  float         dx,dy,dz;
  float         hite, hol;
  float         c, df, r3;
  int           i;

  HYBRID        *bp = hybrid_first;

  if (!bp) return TRUE;

  hol = GetLambda();

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    a3 = bp -> atom3;
    a4 = bp -> atom4;
    if ((a1 -> active) || (a2 -> active) ||
        (a3 -> active) || (a4 -> active)) {
      for(i = 0; i < 3; i++) {
        if (lambda) {
          x1 = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
          y1 = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
          z1 = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
          x2 = a3 -> x - a1 -> x + lambda * (a3 -> dx - a1 -> dx);
          y2 = a3 -> y - a1 -> y + lambda * (a3 -> dy - a1 -> dy);
          z2 = a3 -> z - a1 -> z + lambda * (a3 -> dz - a1 -> dz);
          x3 = a4 -> x - a1 -> x + lambda * (a4 -> dx - a1 -> dx);
          y3 = a4 -> y - a1 -> y + lambda * (a4 -> dy - a1 -> dy);
          z3 = a4 -> z - a1 -> z + lambda * (a4 -> dz - a1 -> dz);
        } else {
          x1 = a2 -> x - a1 -> x;
          y1 = a2 -> y - a1 -> y;
          z1 = a2 -> z - a1 -> z;
          x2 = a3 -> x - a1 -> x;
          y2 = a3 -> y - a1 -> y;
          z2 = a3 -> z - a1 -> z;
          x3 = a4 -> x - a1 -> x;
          y3 = a4 -> y - a1 -> y;
          z3 = a4 -> z - a1 -> z;
        }

        /**** 1 cross 2 ****/

        cx1 = y1 * z2 - y2 * z1;
        cy1 = x2 * z1 - x1 * z2;
        cz1 = x1 * y2 - x2 * y1;
        r = cx1 * cx1 + cy1 * y1 + cz1 * cz1;
        if (r > 1.e-16) {
          r  = sqrt(r);
          r3 = r * r * r;

          /**** Height is x3 vectordot cx1 (normalized) ****/

          hite = (cx1 * x3 + cy1 * y3 + cz1 * z3) / r;
          df   = two * bp -> k * (one - hol) *
                 ((one - hol) * bp -> offset - (one + hol) * hite) / three;

          /**** Do the apex derivatives now (easy) ****/

          c         = df / r;
          a4 -> fx += c * cx1;
          a4 -> fy += c * cy1;
          a4 -> fz += c * cz1;
          a1 -> fx -= c * cx1;
          a1 -> fy -= c * cy1;
          a1 -> fz -= c * cz1;

          /**** Now the side derivative (messy) dot product of r3 and cx1 components ****/

          c  = df / r3;
          dx = - cx1 * x3 * c;
          dy = - cy1 * y3 * c;
          dz = - cz1 * z3 * c;

          c         = df * ((- y3 * z2 + z3 * y2) / r) +
                      dx * (y2 * (x1 * y2 - x2 * y1) - z2 * (x2 * z1 - x1 * z2));
          a2 -> fx += c;
          a1 -> fx -= c;

          c         = df * ((- z3 * x2 + x3 * z2) / r) +
                      dy * (z2 * (y1 * z2 - y2 * z1) - x2 * (x1 * y2 - x2 * y1));
          a2 -> fy += c;
          a1 -> fy -= c;

          c = df * ((- x3 * y2 + y3 * x2) / r) +
              dz * (x2 * (x2 * z1 - x1 * z2) - y2 *(y1 * z2 - y2 * z1));
          a2 -> fz += c;
          a1 -> fz -= c;

          c = df * ((- z3 * y1 + y3 * z1) / r) -
              dx * (y1 * (x1 * y2 - x2 * y1) - z1 * (x2 * z1 - x1 * z2));
          a3 -> fx += c;
          a1 -> fx -= c;

          c         = df * ((- x3 * z1 + z3 * x1) / r) -
                      dy * (z1 * (y1 * z2 - y2 * z1) - x1 * (x1 * y2 - x2 * y1));
          a3 -> fy += c;
          a1 -> fy -= c;

          c         = df * ((- y3 * x1 + x3 * y1) / r) -
                      dz * (x1 * (x2 * z1 - x1 * z2) - y1 * (y1 * z2 - y2 * z1));
          a3 -> fz += c;
          a1 -> fz -= c;

          /**** Circularly shift the base atoms ****/

          at = a1;
          a1 = a2;
          a2 = a3;
          a3 = at;
        }
        if (!a1 -> active) {
          a1 -> fx = zero;
          a1 -> fy = zero;
          a1 -> fz = zero;
        }

        if (!a2 -> active) {
          a2 -> fx = zero;
          a2 -> fy = zero;
          a2 -> fz = zero;
        }

        if (!a3 -> active) {
          a3 -> fx = zero;
          a3 -> fy = zero;
          a3 -> fz = zero;
        }

        if (!a4 -> active) {
          a4 -> fx = zero;
          a4 -> fy = zero;
          a4 -> fz = zero;
        }
      }
    }
    if (bp == bp -> next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}


/**** Increment the hybrid force ****/

int AMMP_FASTCALL f_hybrid(float lambda)
{
  AMMP_ATOM     *a1, *a2, *a3,  *a4, *at;
  float         x1 ,y1, z1, x2, y2, z2, x3, y3, z3;
  float         r, cx1, cy1, cz1;
  float         dx, dy, dz;
  float         df;
  float         r3, c;
  int           i;


  HYBRID *      bp = hybrid_first;
  if (!bp) return TRUE;

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    a3 = bp -> atom3;
    a4 = bp -> atom4;
    if ((a1 -> active) || (a2 -> active) ||
        (a3 -> active) || (a4 -> active)) {
      for(i = 0; i < 3; i++) {
        if (lambda) {
          x1 = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
          y1 = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
          z1 = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
          x2 = a3 -> x - a1 -> x + lambda * (a3 -> dx - a1 -> dx);
          y2 = a3 -> y - a1 -> y + lambda * (a3 -> dy - a1 -> dy);
          z2 = a3 -> z - a1 -> z + lambda * (a3 -> dz - a1 -> dz);
          x3 = a4 -> x - a1 -> x + lambda * (a4 -> dx - a1 -> dx);
          y3 = a4 -> y - a1 -> y + lambda * (a4 -> dy - a1 -> dy);
          z3 = a4 -> z - a1 -> z + lambda * (a4 -> dz - a1 -> dz);
        } else {
          x1 = a2 -> x - a1 -> x;
          y1 = a2 -> y - a1 -> y;
          z1 = a2 -> z - a1 -> z;
          x2 = a3 -> x - a1 -> x;
          y2 = a3 -> y - a1 -> y;
          z2 = a3 -> z - a1 -> z;
          x3 = a4 -> x - a1 -> x;
          y3 = a4 -> y - a1 -> y;
          z3 = a4 -> z - a1 -> z;
        }

        /**** 1 cross 2 ****/

        cx1 =  y1 * z2 - y2 * z1;
        cy1 = -x1 * z2 + x2 * z1;
        cz1 =  x1 * y2 - x2 * y1;
        r = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;

        if (r > 1.e-16) {
          r  = sqrt(r);
          r3 = r * r * r;

          /**** Height is x3 vectordot cx1 (normalized) ****/

          df =  two * bp -> k * (bp -> offset - (cx1 * x3 + cy1 * y3 + cz1 * z3) / r) / three;

          /**** Do the apex derivatives now (easy) ****/

          c         = df / r;
          a4 -> fx += c * cx1;
          a4 -> fy += c * cy1;
          a4 -> fz += c * cz1;
          a1 -> fx -= c * cx1;
          a1 -> fy -= c * cy1;
          a1 -> fz -= c * cz1;

          /**** Now the side derivative (messy) dot product of r3 and cx1 components ****/

          c  = df / r3;
          dx = - cx1 * x3 * c;
          dy = - cy1 * y3 * c;
          dz = - cz1 * z3 * c;

          c         = df * ((z3 * y2 - y3 * z2) / r) +
                      dx * (y2 * (x1 * y2 - x2 * y1) - z2 * (x2 * z1 - x1 * z2));
          a2 -> fx += c;
          a1 -> fx -= c;


          c         = df * ((x3 * z2 - z3 * x2) / r) +
                      dy * (z2 * (y1 * z2 - y2 * z1) - x2 * (x1 * y2 - x2 * y1));
          a2 -> fy += c;
          a1 -> fy -= c;

          c         = df * ((y3 * x2 - x3 * y2) / r) +
                      dz * (x2 * (x2 * z1 - x1 * z2) -  y2 * (y1 * z2 - y2 * z1));
          a2 -> fz += c;
          a1 -> fz -= c;

          c         = df * ((y3 * z1 - z3 * y1) / r) -
                      dx * (y1 * (x1 * y2 - x2 * y1) - z1 * (x2 * z1 - x1 * z2));
          a3 -> fx += c;
          a1 -> fx -= c;

          c         = df * ((z3 * x1 - x3 * z1) / r) -
                      dy * (z1 * (y1 * z2 - y2 * z1) - x1 * (x1 * y2 - x2 * y1));
          a3 -> fy += c;
          a1 -> fy -= c;

          c         = df * ((x3 * y1 - y3 * x1) / r) -
                      dz * (x1 * (x2 * z1 - x1 * z2) - y1 * (y1 * z2 - y2 * z1));
          a3 -> fz += c;
          a1 -> fz -= c;

          /**** Circularly shift the base atoms ****/

          at = a1;
          a1 = a2;
          a2 = a3;
          a3 = at;
        } /* End of for (i) */

        if (!a1 -> active) {
          a1 -> fx = zero;
          a1 -> fy = zero;
          a1 -> fz = zero;
        }

        if (!a2 -> active) {
          a2 -> fx = zero;
          a2 -> fy = zero;
          a2 -> fz = zero;
        }

        if (!a3 -> active) {
          a3 -> fx = zero;
          a3 -> fy = zero;
          a3 -> fz = zero;
        }

        if (!a4 -> active) {
          a4 -> fx = zero;
          a4 -> fy = zero;
          a4 -> fz = zero;
        }
      }
    }

    if (bp == bp -> next) return TRUE;
    bp = bp->next;
  } /* End of while */

  return FALSE;
}


/**** Get the hybrid ****/

void AMMP_FASTCALL get_hybrid(AMMP_ATOM *a1, AMMP_ATOM *bonded[], int mbond, int *inbond)
{
  HYBRID        *mine = hybrid_first;

  *inbond = 0;

  while(mine) {
    if (mine -> atom1 == a1)
      bonded[(*inbond)++] = mine -> atom4;
    if (mine -> atom4 == a1)
      bonded[(*inbond)++] = mine -> atom1;

    if ((mine == mine -> next) ||
        (*inbond == mbond)) return;
    mine = mine -> next;
  } /* End of while */
}


/**** Calculate the hybrid potential of the selected atom ****/

void AMMP_FASTCALL gsdg_hybrid(AMMP_ATOM *who)
{
  AMMP_ATOM     *a1, *a2, *a3, *a4;
  float         x3, y3, z3;
  float         x1, y1, z1;
  float         x2, y2, z2;
  float         hite, r, cx1, cy1, cz1;

  HYBRID        *hp = hybrid_first;

  if ((!hp) || (!who -> active)) return;

  while(hp) {
    a4 = hp -> atom4;
    if (a4 == who) {
      a1 = hp -> atom1;
      a2 = hp -> atom2;
      a3 = hp -> atom3;
      x1 = a2 -> x - a1 -> x;
      y1 = a2 -> y - a1 -> y;
      z1 = a2 -> z - a1 -> z;
      x2 = a3 -> x - a1 -> x;
      y2 = a3 -> y - a1 -> y;
      z2 = a3 -> z - a1 -> z;
      x3 = a4 -> x - a1 -> x;
      y3 = a4 -> y - a1 -> y;
      z3 = a4 -> z - a1 -> z;

      /**** 1 cross 2 ****/

      cx1 = y1 * z2 - y2 * z1;
      cy1 = x2 * z1 - x1 * z2;
      cz1 = x1 * y2 - x2 * y1;
      r   = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;
      if (r > 1.e-16) {
        r    = (1.0 / sqrt(r));
        cx1 *= r;
        cy1 *= r;
        cz1 *= r;

        /**** Height is x3 vectordot cx1 (normalized) ****/

        hite     = hp -> offset - cx1 * x3 - cy1 * y3 - cz1 * z3;
        a4 -> x += cx1 * hite;
        a4 -> y += cy1 * hite;
        a4 -> z += cz1 * hite;
      }
    }

    if (hp -> next == hp) break;
    hp = hp -> next;
  } /* End of while */
}


/**** Add a new hybrid ****/

int AMMP_FASTCALL hybrid(int p1, int p2, int p3, int p4, float fk, float off)
{
  HYBRID *      new;

  AMMP_ATOM *   ap1 = a_m_serial_table(p1);
  AMMP_ATOM *   ap2 = a_m_serial_table(p2);
  AMMP_ATOM *   ap3 = a_m_serial_table(p3);
  AMMP_ATOM *   ap4 = a_m_serial_table(p4);

  if ((!ap1) || (!ap2) || (!ap3) || (!ap4)) {
    aaerror("Undefined atom in hybrid %d %d %d %d", p1, p2, p3, p4);
    return FALSE;
  }

  if ((new = (HYBRID *)Alloca(sizeof(HYBRID), "hybrid()")) == NULL)
    return FALSE;

  /**** Initialize the pointers ****/

  if (hybrid_first == NULL) hybrid_first = new;
  if (hybrid_last  == NULL) hybrid_last  = new;

  new -> atom1        = ap1;
  new -> atom2        = ap2;
  new -> atom3        = ap3;
  new -> atom4        = ap4;
  new -> offset       = off;
  new -> k            = fk;
  new -> next         = new;
  hybrid_last -> next = new;
  hybrid_last         = new;

  return TRUE;
}


/**** Calculate the hybrid potential (homotopy version) ****/

int AMMP_FASTCALL v_ho_hybrid(float *V, float lambda)
{
  AMMP_ATOM     *a1, *a2, *a3, *a4;
  float         x1, y1, z1, x2, y2, z2, x3, y3, z3;
  float         r, cx1, cy1, cz1;
  float         hite, hol;

  HYBRID        *bp = hybrid_first;

  if (!bp) return TRUE;

  hol = GetLambda();

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    a3 = bp -> atom3;
    a4 = bp -> atom4;
    if ((a1 -> active) || (a2 -> active) ||
        (a3 -> active) || (a4 -> active)) {
      if (lambda) {
        x1 = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
        y1 = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
        z1 = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
        x2 = a3 -> x - a1 -> x + lambda * (a3 -> dx - a1 -> dx);
        y2 = a3 -> y - a1 -> y + lambda * (a3 -> dy - a1 -> dy);
        z2 = a3 -> z - a1 -> z + lambda * (a3 -> dz - a1 -> dz);
        x3 = a4 -> x - a1 -> x + lambda * (a4 -> dx - a1 -> dx);
        y3 = a4 -> y - a1 -> y + lambda * (a4 -> dy - a1 -> dy);
        z3 = a4 -> z - a1 -> z + lambda * (a4 -> dz - a1 -> dz);
      } else {
        x1 = a2 -> x - a1 -> x;
        y1 = a2 -> y - a1 -> y;
        z1 = a2 -> z - a1 -> z;
        x2 = a3 -> x - a1 -> x;
        y2 = a3 -> y - a1 -> y;
        z2 = a3 -> z - a1 -> z;
        x3 = a4 -> x - a1 -> x;
        y3 = a4 -> y - a1 -> y;
        z3 = a4 -> z - a1 -> z;
      }

      /**** 1 cross 2 ****/
      cx1 = y1 * z2 - y2 * z1;
      cy1 = x2 * z1 - x1 * z2;
      cz1 = x1 * y2 - x2 * y1;
      r   = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;

      if (r > 1.e-16) {
        r = sqrt(r);

        /**** Height is x3 vectordot cx1 (normalized) ****/

        hite  = (cx1 * x3 + cy1 * y3 + cz1 * z3) / r;
        r     =  hite * (one + hol) - bp -> offset * (one - hol);
        *V   += bp -> k * r * r;
      }
    }
    if (bp == bp -> next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}


/**** Calculate the hybrid potential ****/

int AMMP_FASTCALL v_hybrid(float *V, float lambda)
{
  AMMP_ATOM     *a1, *a2, *a3, *a4;
  float         x1, y1, z1, x2, y2, z2, x3, y3, z3;
  float         r, cx1, cy1, cz1;
  float         hite;

  HYBRID        *bp = hybrid_first;

  if (!bp) return TRUE;

  if (lambda) {
    while(bp != NULL) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      a3 = bp -> atom3;
      a4 = bp -> atom4;
      if ((a1 -> active) || (a2 -> active) || (a3 -> active) || (a4 -> active)) {
        x1 = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
        y1 = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
        z1 = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
        x2 = a3 -> x - a1 -> x + lambda * (a3 -> dx - a1 -> dx);
        y2 = a3 -> y - a1 -> y + lambda * (a3 -> dy - a1 -> dy);
        z2 = a3 -> z - a1 -> z + lambda * (a3 -> dz - a1 -> dz);
        x3 = a4 -> x - a1 -> x + lambda * (a4 -> dx - a1 -> dx);
        y3 = a4 -> y - a1 -> y + lambda * (a4 -> dy - a1 -> dy);
        z3 = a4 -> z - a1 -> z + lambda * (a4 -> dz - a1 -> dz);

        /**** Cross product ****/

        cx1 =  y1 * z2 - y2 * z1;
        cy1 = -x1 * z2 + x2 * z1;
        cz1 =  x1 * y2 - x2 * y1;
        r   = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;
        if (r > 1.e-16f) {

          /**** Height is x3 vectordot cx1 (normalized) ****/

          hite = cx1 * x3 + cy1 * y3 + cz1 * z3;
          hite = (hite / (float)sqrt(r)) - bp -> offset;
          *V  += bp -> k * hite * hite;
        }
      }
      if (bp == bp -> next) return TRUE;
      bp = bp->next;
    } /* End of while */
  } else {
    while(bp != NULL) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      a3 = bp -> atom3;
      a4 = bp -> atom4;
      if ((a1 -> active) || (a2 -> active) || (a3 -> active) || (a4 -> active)) {
        x1 = a2 -> x - a1 -> x;
        y1 = a2 -> y - a1 -> y;
        z1 = a2 -> z - a1 -> z;
        x2 = a3 -> x - a1 -> x;
        y2 = a3 -> y - a1 -> y;
        z2 = a3 -> z - a1 -> z;
        x3 = a4 -> x - a1 -> x;
        y3 = a4 -> y - a1 -> y;
        z3 = a4 -> z - a1 -> z;

        /**** Cross product ****/

        cx1 =  y1 * z2 - y2 * z1;
        cy1 = -x1 * z2 + x2 * z1;
        cz1 =  x1 * y2 - x2 * y1;
        r   = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;
        if (r > 1.e-16f) {

          /**** Height is x3 vectordot cx1 (normalized) ****/

          hite = cx1 * x3 + cy1 * y3 + cz1 * z3;
          hite = (hite / (float)sqrt(r)) - bp -> offset;
          *V  += bp -> k * hite * hite;
        }
      }
      if (bp == bp -> next) return TRUE;
      bp = bp->next;
    } /* End of while */
  }

  return FALSE;
}




