/* kdock.c
*  modified version of kohonen.c for docking
*
* the idea is to use the algorithm in a spatially limited 
*  region on an active zone.
*  use mxdq as a limit on step size and skip if the closest
*  atom is inactive.
*
*/
/*
*  kohonen.c
*
*  implement a kohonen neural net to find a 
* 3-d space filling curve corresponding to the structure;
*
*  net links correspond to chemical bonds, angle distances, noel distances
*  eventually the distance update will also include vdw interactions
*  and chirality.
*
*/
/*
*  copyright 1993-1997 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/


#include <stdio.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Constants ****/

#define  MAX_LOCAL                      200
#define  noel_target

/* Define either STEP_METRIC for metrical usage or STEP_NEIGHBORHOOD
 * for neighborhood usage
 */

#define   STEP_NEIGHBORHOOD
//#define  STEP_METRIC


/**** Kohonen docking ****/

int AMMP_FASTCALL kdock(AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int niter,
                         float radius, int initialize, float rx, float ry, float rz)
{
  AMMP_ATOM             *ap,*bp[MAX_LOCAL],*cp;
  AMMP_ATOM             *actp[600];
  float                 x, y, z, r, rc;
  float                 cx, cy, cz, dx, dy, dz;
  float                 upper[MAX_LOCAL], middle[MAX_LOCAL];
  float                 target[MAX_LOCAL];
  float                 k, t, mxdq, mxdq2;
  int                   inactp;
  int                   iter, i, j;
  int                   use_noel, use_hybrid, use_step;

  int                   numatm = a_number();

  if (numatm < 2) {
    aaerror("At least two atoms are required");
    return FALSE;
  }

  if (radius < one) radius = sqrt((double)numatm);

  mxdq = get_f_variable("mxdq");
  if (mxdq < 0.1f) mxdq = 0.1f;
  mxdq2 = mxdq * mxdq;

  cx = rx;
  cy = ry;
  cz = rz;

  /**** Initialization ****/

  if (initialize) {
    for(i = 0; i < numatm; i++) {
      ap = a_next(i);

REDO_INITIAL_VALUE:
      x = two * randf() - one;
      y = two * randf() - one;
      z = two * randf() - one;
      if ((x * x + y * y + z * z) > one) goto REDO_INITIAL_VALUE;

      if (ap -> active) {
        ap -> x = x * radius + cx;
        ap -> y = y * radius + cy;
        ap -> z = z * radius + cz;
      }
    } /* End of for (i) */
  }

  use_noel   = FALSE;
  use_hybrid = use_noel;
  use_step   = use_noel;

  for(i = 0; i < nfs; i++) {
    if (vfs[i] == v_noel  ) use_noel   = TRUE;
    if (vfs[i] == v_hybrid) use_hybrid = TRUE;
    if (vfs[i] == v_step  ) use_step   = TRUE;
  } /* End of for (i) */


  for(iter = 0; iter < niter; iter++) {

#ifdef GRAMMP
    if (iter % 10) force_screen_update();
    BE_NICE();
#endif

    k = 0.2f;

RE_RANDOM:;

    x  = two * randf() - one;
    y  = two * randf() - one;
    z  = two * randf() - one;
    rc = x * x + y * y + z * z;
    if (rc > one) goto RE_RANDOM;
    x = x * radius + cx;
    y = y * radius + cy;
    z = z * radius + cz;

    /**** Find the closest atom ****/

    inactp = 0;
    cp     = a_next(-1);
    rc     = 10.e10;
    for(i = 0;i < numatm; i++) {
      ap = a_next(i);
      r  = ap -> x - x;
      r *= r;
      t  = ap -> y - y;
      r += t * t;
      t  = ap -> z - z;
      r += t * t;
      if (r < rc) {
        cp = ap;
        rc = r;
      }
    } /* End of for (i) */

    if ((!cp -> active) && (rc < 9.0f)) goto RE_RANDOM;

    /**** Now find the closest active atom ****/

    cp = a_next(-1);
    rc = 10.e10;
    for(i = 0; i < numatm; i++) {
      ap = a_next(i);
      if (ap -> active) {
        ap = a_next(i);
        r  = ap -> x - x;
        r *= r;
        t  = ap -> y - y;
        r += t * t;
        t  = ap -> z - z;
        r += t * t;
        if (r < rc) {
          cp = ap;
          rc = r;
        }
      }
    } /* End of for (i) */

    /**** Truncate the step to no more than mxdq ****/

    if (rc > mxdq2) {
      rc = mxdq / sqrt(rc);
      x = (x - cp -> x) * rc + cp -> x;
      y = (y - cp -> y) * rc + cp -> y;
      z = (z - cp -> z) * rc + cp -> z;
    }

    /**** Update it and its neighbors ****/

    for(i = 0; i < numatm; i++) {
      ap = a_next(i);
      if ((ap -> active) && (ap != cp)) {
        r  = ap -> x - cp -> x;
        r *= r;
        t  = ap -> y - cp -> y;
        r += t * t;
        t  =  ap -> z - cp -> z;
        r += t * t;
        if (r < 25.0f) { /* 25 (r==5) works well with good noe */
          ap -> x += (x - cp -> x) * k;
          ap -> y += (y - cp -> y) * k;
          ap -> z += (z - cp -> z) * k;
          actp[inactp++] = ap;
        }
      }
    } /* End of for (i) */

    if (cp -> active){
      dx       = x - cp -> x;
      dy       = y - cp -> y;
      dz       = z - cp -> z;
      cp -> x += dx * k;
      cp -> y += dy * k;
      cp -> z += dz * k;
      actp[inactp++] = cp;
    } else { /* put the center at the atom */
      x = cp -> x;
      y = cp -> y;
      z = cp -> z;
      dx = zero;
      dy = zero;
      dz = zero;
    }

    get_bond_and_length(cp, bp, target, 20, &j);
    for(i = 0; i < j; i++) {
      ap = bp[i];
      if (ap -> active) {
        ap -> x += (x - ap -> x) * k;
        ap -> y += (y - ap -> y) * k;
        ap -> z += (z - ap -> z) * k;
        actp[inactp++] = ap;
        r  = x - ap -> x;
        r *= r;
        t  = y - ap -> y;
        r += t * t;
        t  = z - ap -> z;
        r += t * t;
        if (r > zero) {
          r = target[i] / sqrt(r);
          ap -> x = x + (ap -> x - x) * r;
          ap -> y = y + (ap -> y - y) * r;
          ap -> z = z + (ap -> z - z) * r;
        }
      }
    } /* End of for (i) */

    if (use_step) {
      get_step_and_bounds(cp, bp, target, middle, upper, MAX_LOCAL, &j);
      for(i = 0; i < j; i++) {
        ap = bp[i];
        if (ap -> active) {
#ifdef STEP_METRIC
          r  = x - ap -> x;
          r *= r;
          t  = y - ap -> y;
          r += t * t;
          t  = z - ap -> z;
          r += t * t;
          if (r <= (upper[i] * upper[i]) && (r > zero)) {
            if (r > (middle[i] * middle[i])) r = middle[i] / sqrt(r);
            else if (r < (target[i] * target[i])) r = target[i] / sqrt(r);
              else r = one;
            ap -> x = x + (ap -> x - x) * r;
            ap -> y = y + (ap -> y - y) * r;
            ap -> z = z + (ap -> z - z) * r;
#endif

#ifdef STEP_NEIGHBORHOOD
          r  =  cp -> x - ap -> x;
          r *= r;
          t  = cp -> y - ap -> y;
          r += t * t;
          t  = cp -> z - ap -> z;
          r += t * t;
          if (r < (upper[i] * upper[i])) {
            ap -> x += dx * k;
            ap -> y += dy * k;
            ap -> z += dz * k;
#endif
            actp[inactp++] = ap;
          }
        }
      } /* End of for (i) */
    }

    if (use_noel) {

#ifdef noel_target
      get_noel_and_length(cp,bp,target,MAX_LOCAL,&j);
#else
      get_noel_and_bounds(cp,bp,target,upper,MAX_LOCAL,&j);
#endif

      for(i = 0; i < j ; i++) {
        ap = bp[i];
        if (ap -> active) {
          ap -> x += dx * k;
          ap -> y += dy * k;
          ap -> z += dz * k;
          actp[inactp++] = ap;

          r  = x - ap -> x;
          r *= r;
          t  = y - ap -> y;
          r += t * t;
          t  =  z - ap -> z;
          r += t * t;
          if (r > zero) {

#ifdef noel_target
            r = target[i] / sqrt(r);
#else
            r = sqrt(r);
            if (r < target[i]) {
              r = target[i] / r;
              ap -> x = x + (ap -> x - x) * r;
              ap -> y = y + (ap -> y - y) * r;
              ap -> z = z + (ap -> z - z) * r;
            } else if (r > upper[i]) {
              r = upper[i]/r;
              ap -> x = x + (ap -> x - x) * r;
              ap -> y = y + (ap -> y - y) * r;
              ap -> z = z + (ap -> z - z) * r;
            }
#endif
          }
        }
      }
    }

    printf("%d\n", iter);

    if (inactp > 0) {
      if (use_hybrid) {
        for(i = 0; i < inactp; i++)
          gsdg_hybrid(actp[i]);
      }
      for(i = 0; i < inactp; i++)
        actp[i] -> active = FALSE;
      kohonen_minimizer(vfs, ffs, nfs, 1);
      for(i = 0; i < inactp; i++)
        actp[i] -> active = TRUE;
    }
  } /* End of for (iter) */

  return TRUE;
}

