/*
*  kohonen.c
*
*  implement a kohonen neural net to find a 
*  3D space filling curve corresponding to the structure;
*
*  net links correspond to chemical bonds, angle distances, noel distances
*  eventually the distance update will also include vdw interactions
*  and chirality.
*
*/
/*
*  copyright 1993-1997 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Constants ****/

#define  noel_target
#define  MAX_LOCAL              700

//#define  STEP_METRIC
#define  STEP_NEIGHBORHOOD

#ifdef WINDOWS
void force_screen_update(); /* in animate.c */
#endif



void AMMP_FASTCALL kohonen(FILE *op, AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int niter,
                           float radius, int initialize, float rx, float ry, float rz, int echo)
{
  AMMP_ATOM     *ap, *bp[MAX_LOCAL], *cp;
  AMMP_ATOM     *actp[100 + MAX_LOCAL]; /* pointers which are frozen */
  float         upper[MAX_LOCAL];
  float         middle[MAX_LOCAL];
  float         target[MAX_LOCAL];
  float         x, y, z, r, rc;
  float         cx, cy, cz;
  float         k, t;
  int           inactp, numatm, prior_type;
  int           iter, local_iter, i, j;
  int           use_noel, use_hybrid, use_step;

  const char *  StepStr = "  Kohonen %4d: Computing\n";
#ifdef VEGAZZ
  int           lastiter;

  int           nupdat = GetNupdat();
  int           supdat = GetSupdat();
#endif

  if (!CheckAtoms()) return;
  if (niter < 1) niter = 1;

#ifdef VEGAZZ
  lastiter = niter - 1;
#endif

  numatm = a_number();
  if (numatm < 2   ) return;
  if (radius < 1.0f) {
    radius = sqrt((float)numatm);
#ifndef VEGAZZ
    if (echo)
#endif
      fprintf(op, "  Kohonen     : Radius automatically assigned to %.4f\n", radius);
  }

  cx = 0.0f;
  cy = 0.0f;
  cz = 0.0f;
  j  = 0;

  for(i = 0; i < numatm; i++) {
    ap = a_next(i);
    if (!ap -> active) {
      j++;
      cx += ap -> x;
      cy += ap -> y;
      cz += ap -> z;
    }
  } /* End of for (j) */

  if (j > 0) {
    r = 1.0f / (float)j;
    cx *= r;
    cy *= r;
    cz *= r;
  }

  /**** Initialization with random coordinates ****/

  if (initialize > 0) {
#ifndef VEGAZZ
    if (echo)
#endif
      fprintf(op, "  Kohonen     : Initialization for %d atoms\n", numatm);

    for(i = 0; i < numatm; i++) {
      ap = a_next(i);
      x = 2.0f * randf() - 1.0f;
      y = 2.0f * randf() - 1.0f;
      z = 2.0f * randf() - 1.0f;
      if (ap -> active) {
        ap -> x = x * radius + cx;
	ap -> y = y * radius + cy;
        ap -> z = z * radius + cz;
      }
    } /* End of for (i) */
  }

  /*  Figure out what kind of prior to use
   *  rx >= 0 ry == 0 rz == 0 spherical
   *  rx >  0 ry >  0 rz == 0 cylindrical
   *  rx >  0 ry >  0 rz >  0 ellipsoidal
   *
   *  Cylindrical and elipsoidal ignore the radius.
   *  Spherical uses the radius value.
   */

  prior_type = 0;  /* default to sphere */

  if ((rx >= 0.0f) && (ry < 1.e-7) && (rz < 1.e-7)) prior_type = 0;
  if ((rx >  0.0f) && (ry > 0.0f ) && (rz < 1.e-7)) prior_type = 1;
  if ((rx >  0.0f) && (ry > 0.0f ) && (rz > 0.0f )) prior_type = 2;

#ifndef VEGAZZ
  if (echo) {
#endif
    fprintf(op, "  Kohonen     : ");
    switch(prior_type) {
    case 1:
      fprintf(op, "Cylindrical symmetry\n");
      break;
    case 2:
      fprintf(op, "Ellipsoidal symmetry\n");
      break;
    default:
      fprintf(op, "Spherical symmetry\n");
    } /* End of switch */
#ifndef VEGAZZ
  }
#endif

  use_noel   = FALSE;
  use_hybrid = use_noel;
  use_step   = use_noel;

  /**** Check the potential to enable specific functions ****/

  for(i = 0; i < nfs; i++) {
    if (vfs[i] == v_noel) use_noel = TRUE;
    else if (vfs[i] == v_hybrid) use_hybrid = TRUE;
    else if (vfs[i] == v_step  ) use_step   = TRUE;
  } /* End of for (i) */

  /**** Main loop ****/

  for(iter = 0; iter < niter; iter++) {
#ifdef VEGAZZ
    if ((nupdat) && ((!(iter % nupdat)) || (iter == lastiter))) send_all_atoms();
    if ((supdat) && ((!(iter % supdat)) || (iter == lastiter)))
#else
    if (echo)
#endif
      fprintf(op, StepStr, iter);

#ifdef WINDOWS
    force_screen_update();
#endif

#ifdef GRAMMP
    send_all_atoms();
#endif

    k = 0.01f + (float)(niter - iter) / (float)niter * 0.2f;

    for(local_iter = 0; local_iter < numatm; local_iter++) {
      k = 0.2f;

      /**** Spherical ****/

      if (prior_type == 0) {
        r = radius;
RE_RANDOM:
        x  = 2.0f * randf() - 1.0f;
        y  = 2.0f * randf() - 1.0f;
        z  = 2.0f * randf() - 1.0f;
        rc = x * x + y * y + z * z;
        if (rc > 1.0f) goto RE_RANDOM;
        x = x * r + cx;
        y = y * r + cy;
        z = z * r + cz;

        /**** Cylindrical ****/

      } else if (prior_type == 1) {
RE_RANDOM_2D:
        x  = 2.0f * randf() - 1.0f;
        y  = 2.0f * randf() - 1.0f;
        rc = x * x + y * y;
        if (rc > 1.0f) goto RE_RANDOM_2D;
        z = 2.0f * randf() - 1.0f;
        x = x * rx + cx;
        y = y * rx + cy;
        z = z * ry;

        /**** Elipsoidal ****/

      } else if( prior_type == 2){
RE_RANDOM_ELIPSE:
        x  = 2.0f * randf() - 1.0f;
        y  = 2.0f * randf() - 1.0f;
        z  = 2.0f * randf() - 1.0f;
        rc = x * x + y * y + z * z;
        if (rc > 1.0f) goto RE_RANDOM_ELIPSE;
        x = x * rx + cx;
        y = y * ry + cy;
        z = z * rz + cz;
      }

      /**** Find the closest atom ****/

      inactp = 0;
      rc     = 10.e10;
      for(i = 0; i < numatm; i++) {
        ap = a_next(i);
        if (ap -> active) {
          r  = ap -> x - x;
          r *= r;
          t  = ap -> y - y;
          r += t * t;
          t  = ap -> z - z;
          r += t * t;
          if (r < rc) {
            cp = ap;
            rc = r;
          }
        }
      } /* End of for (i) */

      /**** Update it and its neighbors ****/

      if (cp -> active) {
        cp -> x += (x - cp -> x) * k;
	cp -> y += (y - cp -> y) * k;
	cp -> z += (z - cp -> z) * k;
        actp[inactp++] = cp;
      } else {

        /**** Put the center at the atom ****/

        x = cp -> x;
        y = cp -> y;
        z = cp -> z;
      }

      get_bond_and_length(cp, bp, target, 20, &j);
      for(i = 0; i < j ; i++) {
        ap = bp[i];
        if (ap -> active) {
          ap -> x        += (x - ap -> x) * k;
          ap -> y        += (y - ap -> y) * k;
          ap -> z        += (z - ap -> z) * k;
          actp[inactp++]  = ap;
          r               = x - ap -> x;
          r              *= r;
          t               = y - ap -> y;
          r              += t * t;
          t               = z - ap -> z;
          r              += t * t;
          if (r > 0.0f) {
            r       = target[i] / sqrt(r);
            ap -> x = x + (ap -> x - x) * r;
            ap -> y = y + (ap -> y - y) * r;
            ap -> z = z + (ap -> z - z) * r;
          }
        }
      } /* End of for (i) */

      if (use_step) {
        get_step_and_bounds(cp, bp, target, middle, upper, MAX_LOCAL, &j);
        for(i = 0; i < j; i++) {
          ap = bp[i];
          if (ap -> active) {
#ifdef STEP_METRIC
            r  = (x - ap -> x) * (x - ap -> x);
            r += (y - ap -> y) * (y - ap -> y);
            r += (z - ap -> z) * (z - ap -> z);
            if ((r <= (upper[i] * upper[i])) && (r > 0.0f)) {
              if (r > (middle[i] * middle[i])) r = middle[i] / sqrt(r);
              else if (r < (target[i] * target[i])) r = target[i] / sqrt(r);
              else r = 1.0f;
            ap -> x = x + (ap -> x - x) * r;
	    ap -> y = y + (ap -> y - y) * r;
	    ap -> z = z + (ap -> z - z) * r;
#endif

#ifdef STEP_NEIGHBORHOOD
          r  = cp -> x - ap -> x;
          r *= r;
          t  = cp -> y - ap -> y;
          r += t * t;
          t  = cp -> z - ap -> z;
          r += t * t;
          t  = upper[i];
          if (r <= (t * t)) {
            ap -> x += (x - cp -> x) * k;
            ap -> y += (y - cp -> y) * k;
            ap -> z += (z - cp -> z) * k;
#endif
            actp[inactp++] = ap;
          }
        }
      } /* End of for (i) */
    }

    if (use_noel) {
#ifdef noel_target
      get_noel_and_length(cp, bp, target, MAX_LOCAL, &j);
#else
      get_noel_and_bounds(cp, bp, target, upper, MAX_LOCAL, &j);
#endif
      for(i = 0; i < j; i++) {
        ap = bp[i];
        if (ap -> active) {
          ap -> x        += (x - ap -> x) * k;
          ap -> y        += (y - ap -> y) * k;
          ap -> z        += (z - ap -> z) * k;
          actp[inactp++]  = ap;
          r               = x - ap -> x;
          r              *= r;
          t               = y - ap -> y;
          r              += t * t;
          t               = z - ap -> z;
          r              += t * t;
          if (r > 0.0f) {
#ifdef noel_target
            r       = target[i] / sqrt(r);
            ap -> x = x + (ap -> x - x) * r;
            ap -> y = y + (ap -> y - y) * r;
            ap -> z = z + (ap -> z - z) * r;
#else
            r = sqrt(r);
            if (r < target[i]) {
              r       = target[i] / r;
              ap -> x = x + (ap -> x - x) * r;
              ap -> y = y + (ap -> y - y) * r;
              ap -> z = z + (ap -> z - z) * r;
            } else if (r > upper[i]) {
              r = upper[i] / r;
              ap -> x = x + (ap -> x - x) * r;
              ap -> y = y + (ap -> y - y) * r;
              ap -> z = z + (ap -> z - z) * r;
            }
#endif
          }
        }
      }
    }

    if (inactp > 0) {
      if (use_hybrid) {
        for(i = 0; i < inactp; i++)
          gsdg_hybrid(actp[i]);
      }
      for(i = 0; i < inactp; i++)
        actp[i] -> active = FALSE;
        kohonen_minimizer(vfs, ffs, nfs, 1);
#ifdef GRAMMP
        send_all_atoms();
#endif

#ifdef VEGAZZ
        if ((nupdat) && ((!(iter % nupdat)) || (iter == lastiter))) send_all_atoms();
#endif
        for(i = 0; i < inactp; i++)
          actp[i] -> active = TRUE;
      }
    } /* End of for (local_iter) */
  } /* End of for (iter) */

#ifndef VEGAZZ
  if (echo)
#endif
    fprintf(op, "\n");
}


/**** Copy of cngdel, buggerized for one step ****/

void AMMP_FASTCALL kohonen_minimizer(AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int nstep)
{
  float         lam, vb;
  int           i,ifs;

  /**** Do at most nstep steps ****/

  a_g_zero();
  a_d_zero();
  for(i = 0; i < nstep; i++) {
    vb = 0.0f;
    for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vb, 0.0f);
    a_f_zero();
    for(ifs = 0; ifs < nfs; ifs++) (*ffs[ifs])(0.0f);
    a_ftodx(1.0f, 0.0f);
    lam = linmin(vfs, nfs, sqrt(a_max_d()));
    a_inc_d(lam);
  } /* End of for (i) */

#ifdef WINDOWS
  force_screen_update();
#endif
}


