/* implementation of Monte-Carlo /minimization
 * i.e. score for simulated annealling after minimization
 *  depends on the clone routines
 *  
* also depends on tset  actually using tset_bond_build and set_torsion
 */

#include <stdio.h>
#include <stdlib.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

#ifdef graphic
void force_screen_update();
#endif

/**** Constants ****/

#define  AMMP_MCMIN_CLONEID1            -1
#define  AMMP_MCMIN_CLONEID2            -2


/**** Monte-Carlo minimization ****/

int AMMP_FASTCALL mcmin(FILE *op, AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs,
                        char *at1, char *at2, float temper, int nstep, int echo)
{
  AMMP_ATOM             *(*aps)[];
  AMMP_ATOM             *a1, *a2, *a3, *a4;
  AMMP_ATOM             *bonded[10];
  float                 dx, V, Vbest, dV,dt;
  int                   i, j, in_bond, istep, laststep;

  CLONE *               Clone1 = NULL;
  CLONE *               Clone2 = NULL;
  int                   numatom = a_number();

#ifdef VEGAZZ
  int                   nupdat  = GetNupdat();
  int                   supdat  = GetSupdat();
#endif

  if (!CheckAtoms()) return FALSE;
  if (numatom < 4) return FALSE;

  if (nstep < 1) nstep = 1;
  laststep = nstep - 1;

  if ((aps = (void *)Alloca(numatom * sizeof(AMMP_ATOM *), "mcmin()")) == NULL)
    return FALSE;

  for(i = 0; i < numatom; i++) (*aps)[i] = a_next(i);
//    dt = 3.1415926535f * 0.5f;
    // special version for 2pi/3 torsion shifts;
  dt = 3.1415926535f * 2.0f / 3.0f;

  Clone1 = Clone(Clone1, AMMP_MCMIN_CLONEID1);
  for(istep = 0; istep < nstep; istep++) {

    /**** Compute the potential ****/

    V = 0.0f;
    for(i = 0; i < nfs; i++) (*vfs[i])(&V, zero);
    Vbest = V;


    for(j = 0; j < 1; j++) {
      Clone2 = Clone(Clone2, AMMP_MCMIN_CLONEID2);

RE_CHOOSE:

      i = numatom * randf();
      if (i < 0) i = 0;
      if (i >= numatom) i = numatom - 1;
      a2 = NULL;
      for(; i< numatom; i++)
        if (math_match_atom(at1, (*aps)[i]) != 0) {
          a2 = (*aps)[i];
          break;
        }
      if (!a2) goto RE_CHOOSE;

      get_bond(a2, bonded, 10, &in_bond);
      a3 = NULL;
      for(i = 0; i < in_bond; i++)
        if (math_match_atom(at2, bonded[i]) != 0) {
          a3 = bonded[i];
          break;
        }
      if (!a3) goto RE_CHOOSE;

      a1 = NULL;
      for(i = 0; i < in_bond; i++)
        if (math_match_atom(at2, bonded[i]) == 0) {
          a1 = bonded[i];
          break;
        }

      if (!a1) goto RE_CHOOSE;

      get_bond( a3,bonded, 10, &in_bond);
      a4 = NULL;
      for(i = 0; i < in_bond; i++)
        if (math_match_atom(at1, bonded[i]) == 0) {
          a4 = bonded[i];
          break;
        }
      if (a4 == a1) goto RE_CHOOSE; /* its a loop */
      if (!a4) goto RE_CHOOSE;

      get_bond(a4, bonded, 10, &in_bond);
      for(i = 0; i < in_bond; i++)
        if (bonded[i] == a1) goto RE_CHOOSE;

#ifdef VEGAZZ
      if ((supdat) && ((!(istep % supdat))) || (istep == laststep))
#else
      if (echo)
#endif
        fprintf(op, "  mcmin %6d: torsion %s %s %s %s\n"
                    "              : before %.2f, ",
                    istep + 1, a1 -> name, a2 -> name, a3 -> name, a4 -> name,
                    get_torsion_value(a1, a2, a3, a4));

      /**** So now we have a1, a2, a3, a4 which define a valid torsion ****/

//      dV = dt*(2.*randf()-1.);
      dV = dt;
      if (randf() < 0.5f) dV = -dV;
      tset_bond_build(a1, a2, a3, a4);
      set_torsion(a1, a2, a3, a4, dV);

#ifdef VEGAZZ
      if ((supdat) && ((!(istep % supdat))) || (istep == laststep))
#else
      if (echo)
#endif
        fprintf(op, "after %.3f, delta %.3f\n", get_torsion_value(a1, a2, a3, a4), dV);
    } /* End of for (j) */

    cngdel(vfs, ffs, nfs, 10, 10, zero, FALSE, nupdat);

#ifdef graphic
    force_screen_update();
#endif

#ifdef VEGAZZ
    if ((nupdat) && (!(i % nupdat))) send_all_atoms();
#endif

    dV = zero;
    for(i = 0; i < nfs; i++) (*vfs[i])(&dV, zero);
    dx = exp((V - dV) / temper * 1.987f);

#ifdef VEGAZZ
      if ((supdat) && ((!(istep % supdat))) || (istep == laststep))
#else
    if (echo)
#endif
      fprintf(op, "  energy      : V %f, dV %f, delta %f, dx %f\n\n", V, dV, V - dV, dx);

    if (dV < Vbest) {
      Clone(Clone1, AMMP_MCMIN_CLONEID1);
      Vbest = dV;
    }
    if (dV < V) Clone(Clone2, AMMP_MCMIN_CLONEID2);
    else if (dx < randf()) CloneRestore(Clone2, AMMP_MCMIN_CLONEID2);
  } /* End of for (istep) */

  CloneRestore(Clone1, AMMP_MCMIN_CLONEID1);

#ifdef VEGAZZ
  if (nupdat) send_all_atoms();
#endif

  CloneDel(Clone1, AMMP_MCMIN_CLONEID1);
  CloneDel(Clone2, AMMP_MCMIN_CLONEID2);
  free(aps);

  return TRUE;
}

