/* mom.c
*
* evaluate charges by the method of moments
*
* largely cadged from Rappe and Goddard JPC 95 3358-3363
* cleaned up the solver
*
* modifies the atom structure to have a jaa and chi field
*  (self colomb energy and electronegativity )
*/
/*
*  copyright 1993 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify
*  this notice
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Local variables ****/

static int      MaxMom;


/**** Calculate the atomic charges ****/

int AMMP_FASTCALL mom(FILE *op, float tq, int niter, int echo)
{
  float         (*v)[];
  float         (*Zk)[];
  float         r, dx,dy, dz;
  float         dv, tv;
  int           row;

  const char *  Routine = "mom()";
  float         (*Z)[]  = NULL;
  int           i       = in_mom_list * sizeof(float);
  int           j       = i * in_mom_list;
#ifdef VEGAZZ
  int           lastiter;

  int           supdat = GetSupdat();
#endif

  if (!CheckAtoms()) return FALSE;
  if (!in_mom_list) {
    aaerror("No atoms selected for MoM calculation");
    return FALSE;
  }

  if (niter <= 0) niter = 20; /* default number of trials */

#ifdef VEGAZZ
  lastiter = niter - 1;
#else
  if (echo)
#endif
    fprintf(op, "  MoM         : Initialization\n");

  if (((v  = Alloca(i, Routine)) == NULL) ||
      ((Z  = Alloca(j, Routine)) == NULL) ||
      ((Zk = Alloca(j, Routine)) == NULL)) {
    SafeFree(v);
    SafeFree(Z);
    return FALSE;
  }

  row = in_mom_list;

  /**** Build the impedance matrix in Zk ****/

  for(i = 0; i < row; i++)
    for(j = i + 1; j < row; j++) {
      dx = mom_list[j] -> x - mom_list[i] -> x;
      dy = mom_list[j] -> y - mom_list[i] -> y;
      dz = mom_list[j] -> z - mom_list[i] -> z;
      r  = sqrt(dx * dx + dy * dy + dz * dz);
      (*Zk)[i + j * row] = mom_jab(r, mom_list[i] -> jaa, mom_list[j] -> jaa);
      (*Zk)[j + i * row] = (*Zk)[i + j * row];
    } /* End of for (j) */

  for(i = 0; i < row; i++) {
    (*Zk)[i + i * row] = mom_list[i] -> jaa * 0.5f;
  } /* End of for (i) */

  /**** Now we're ready to do it ****/

  dv = 0.0f; /* the offset charge */

  for(j = 1; j <= niter; j++) {
    for(i = 0; i < row * row; i++) (*Z)[i] = (*Zk)[i];
    for(i = 0; i < row      ; i++) (*v)[i] =  -mom_list[i] -> chi;

    tv = 0.0f;
    for(i = 0; i < row; i++) tv += (*v)[i];
    tv = tv / row + dv;
    for(i = 0; i < row; i++) (*v)[i] -= tv;

    mom_solve(Z, v, row, row);

    tv = 0.0f;
    for(i = 0; i < row; i++) tv += (*v)[i];

/*
    dv += 2*(tv - tq);
    dv += 14.4*(tv - tq)/row*1.5;
*/

    dv += 14.4f * (tv - tq) / row * 0.25f;
#ifdef VEGAZZ
    if ((supdat) && ((!(j % supdat)) || (j == lastiter)))
#else
    if (echo)
#endif
      fprintf(op, "  MoM    %5d: Error %e\n", j, tv - tq);
  } /* End of for (j) */

#ifndef VEGAZZ
  if (echo)
#endif
    fprintf(op, "\n");

  /**** Copy the result ****/

  for(i = 0; i < row; i++) mom_list[i] -> q = (*v)[i];

  /**** Cleanup ****/

  free(mom_list);
  free(v       );
  free(Z       );
  free(Zk      );

  mom_list    = NULL;
  in_mom_list = 0;

  return TRUE;
}


/**** Add atoms to the mom list ****/

int AMMP_FASTCALL mom_add(int s1, int s2)
{
  AMMP_ATOM     *ap;
  int           i, j;

  const char *  Routine = "mom_add()";
  int           numatm  = a_number();
  int           locmom  = numatm + 1;

  if (!CheckAtoms()) return FALSE;

  if (!mom_list) {
    MaxMom = locmom;
    if ((mom_list = (AMMP_ATOM **)Alloca(sizeof(AMMP_ATOM *) * locmom, Routine)) == NULL)
      return FALSE;
  } else if (MaxMom <= locmom) {
    MaxMom = locmom;
    if ((mom_list = (AMMP_ATOM **)ReAlloca(mom_list, sizeof(AMMP_ATOM *) * locmom, Routine)) == NULL)
      return FALSE;
  }

  if (s1 <= 0) s1 = 1;
  if (s2 <= 0) {
    for(i = 0; i < numatm; i++) {
      ap = a_next(i);
      if (s2 < ap -> serial) s2 = ap -> serial;
    } /* End of for (i) */
  }

  if ((s2 > 0) && (s1 > s2)) {
    i  = s1;
    s1 = s2;
    s2 = i;
  }

  for(i = 0; i < numatm; i++) {
    ap = a_next(i);
    if ((ap -> serial >= s1) && (ap -> serial <= s2)) {
      if ((ap -> chi > 0.0f) && (ap -> jaa > 0.0f)) {
        for(j = 0; j < in_mom_list; j++)
          if (mom_list[j] == ap) goto THERE_NOW;
        mom_list[in_mom_list++] = ap;
THERE_NOW:;
      }
    }
  } /* End of for (i) */

  return TRUE;
}


void AMMP_FASTCALL mom_param(int serial, float chi, float jaa)
{
  AMMP_ATOM          *ap;

  if ((ap = a_m_serial(serial)) == NULL) {
    aaerror("MOM> Can't modify the atom %d because it doesn't exist", serial);
    return;
  }
  ap -> chi = chi;
  ap -> jaa = jaa;
}


float AMMP_FASTCALL mom_jab(float r, float j1, float j2)
{
  float         a, b, b2;
/* fit by guess to the repulsion curve
*  tested on methane
*/
/*	if( r < 4.) return 9/(1+ .09375*r*r);
*/
/*
	if( r < 4.) return 8.5/(1+ .085069444*r*r)/2;
*/
  if (r < 30.0f) {
		/* taken from wallace h,h interaction with a small fudge */
		/* the energy of h(1s) h(1s) is given, and we scale it
		   by a term depending on atom type */
/*		a = (j1+j2)/4.;
*/
    a = sqrt(j1 * j2) / 2.0f;
/*		r = 1.08*r;
*/

    /**** Effective radius is adjusted here  (1. is ok this is a little better) ****/

    r = 1.1f * r;
    b = exp(-r);
/*		b2 = 1. +  (33*r + 9*r*r + r*r*r)/48.;
*/
/*		b2 = 1. +  (20*r - 9*r*r - r*r*r)/48.;
*/
    b2 = 1.0f + (15.0f * r - 9.0f * r * r - r * r * r) / 48.0f;

    return a / r * (b * b2);
  }

  return 14.4f / r / 2.0f;
}


/*  this is a routine to solve a linear equation by
    guassian elimination.  (basically solve.for translated) */
/* in order to have the  array matrix be of any length it must be passed as
   a linear array.  Since C has the opposite convention for array packing from
   FORTRAN ( row fastest rather than column fastest) the leading dimension
   ilead is the row size of the array to which matrix points */

int AMMP_FASTCALL mom_solve(float (*matrix)[], float (*vector)[], int irow, int ilead)
{
  float         quotient;
  int           i, j, k;
  int           mpj, mpk;

  int           mpi = 0;

  for(i = 0 ;i < irow - 1 ; i++) {
    j   = i;
    mpj = mpi;
    while((*matrix)[mpi + i] == 0) {
      if (j == irow) return -1;
      j++;
      mpj          += ilead;
      (*vector)[i] += (*vector)[j];
      for(k = i; k < irow; k++)
        (*matrix)[mpi + k] += (*matrix)[mpj + k];
    } /* End of while */

    /**** If here then the diagonal element is not zero so we can do the division ****/

    mpj = mpi + ilead ;
    for( j = i + 1; j < irow ; j++) {
      if ((*matrix)[mpj + i] != 0) {
        quotient      = (*matrix)[mpj + i] / (*matrix)[mpi + i];
        (*vector)[j] -= (*vector)[i] * quotient;
        for(k = i ; k < irow ; k++)
          (*matrix)[mpj + k] -= (*matrix)[mpi + k]*quotient;
      }
      mpj += ilead;
    } /* End of for (j) */
    mpi += ilead;
  } /* End of for (i) */

  /**** Now start the back substitution loop ****/

  /* mpi = 0; */

  for(i = 0; i < irow - 1 ; i++ ) {
    k   = irow - i - 1;
    mpj = 0;
    mpk =  k * ilead;
    for(j = 0; j < k ; j++) {
      (*vector)[j] -= (*matrix)[mpj+k] / (*matrix)[mpk+k] * (*vector)[k];
      mpj          += ilead;
    } /* End of for (j) */
  } /* End of for (i) */

  /**** Finally divide by the diagonal elements ****/

  mpi = 0;
  for(i = 0; i < irow ; i++) {
    (*vector)[i] /= (*matrix)[mpi + i];
    mpi          += ilead;
  } /* End of for (i) */

  return 0;
}


/**** Reset all MoM variables ****/

void AMMP_FASTCALL ResetMom(void)
{
  SafeFree(mom_list);
  in_mom_list = 0;
}

