/* monitor.c
*
*  routine to monitor energy and force for AMMP.
*
*  monitors the potential due to each kind of potential used
*
*  reports kinetic energy and total potential and action (T-V)
*
*  reports maximum l_infinity force
*/
/*
*  copyright 1992 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <string.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Constants ****/

#define  AMMP_MONITOR_FMT               20

/**** Local prototype ****/

static void AMMP_FASTCALL MonitorPrint(FILE *FH, const char *Str, float Val);


/**** Evaluate and show the energies ****/

void AMMP_FASTCALL monitor(AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, FILE *op)
{
  AMMP_ATOM  *  ap;
  AMMP_VFUNC *  Vfunc;
  const char *  Str;
  float         mxdq;
  float         V, T, vt;
  int           ifs;

  if (!CheckAtoms()) return;

  V = 0.0f;
  T = 0.0f;
  a_f_zero();

  fprintf(op, "Energy:\n");

  for(ifs = 0, Vfunc = vfs; ifs < nfs; ++ifs, ++Vfunc) {
    vt = 0.0f;
    (*vfs[ifs])(&vt, 0.0f);
    mxdq = get_f_variable("mxdq");
    set_f_variable("mxdq", 100.);
    (*ffs[ifs])(0.0f);
    set_f_variable("mxdq", mxdq);
    V += vt;

    if      (*Vfunc == u_v_nonbon ) Str = "Non-bond";
    else if (*Vfunc == v_abc      ) Str = "Abc";
    else if (*Vfunc == v_bond     ) Str = "Bond";
    else if (*Vfunc == v_mmbond   ) Str = "Bond MM3";
    else if (*Vfunc == v_morse    ) Str = "Bond morse";
    else if (*Vfunc == v_restrain ) Str = "Bond restraint";
    else if (*Vfunc == v_angle    ) Str = "Angle";
    else if (*Vfunc == v_c_angle  ) Str = "Angle UFF cosine";
    else if (*Vfunc == v_mmangle  ) Str = "Angle MM3";
    else if (*Vfunc == v_noel     ) Str = "Noel";
    else if (*Vfunc == v_nonbon   ) Str = "Non-bond";
    else if (*Vfunc == v_debye    ) Str = "Non-bond debye screened";
    else if (*Vfunc == v_hard     ) Str = "Non-bond hard shell";
    else if (*Vfunc == v_periodic ) Str = "Non-bond periodic";
    else if (*Vfunc == v_screen   ) Str = "Non-bond screened";
    else if (*Vfunc == v_screen   ) Str = "Non-bond shadow";
    else if (*Vfunc == v_torsion  ) Str = "Torsion";
    else if (*Vfunc == v_hybrid   ) Str = "Hybrid";
    else if (*Vfunc == v_tether   ) Str = "Tether";
    else if (*Vfunc == v_fourd    ) Str = "4-d to 3-d projection";
    else if (*Vfunc == v_step     ) Str = "Step pseudo-bond";
    else if (*Vfunc == v_swarm    ) Str = "Swarm";
    else Str = "Unknown";

    MonitorPrint(op, Str, vt);
  } /* End of for (ifs) */

  MonitorPrint(op, "Total potential", V);

  ifs = -1;
  while ((ap = a_next(ifs)) != NULL) {
    ifs = 1;
    T += ap -> vx * ap -> vx * ap -> mass;
    T += ap -> vy * ap -> vy * ap -> mass;
    T += ap -> vz * ap -> vz * ap -> mass;
  } /* End of while */
  T = T * 0.5f / 4.184f / 1000.0f / 1000.0f;

  MonitorPrint(op, "Total kinetic", T    );
  MonitorPrint(op, "Total energy" , T + V);
  MonitorPrint(op, "Total action"  ,T - V);
  fprintf(op, "\n");
}


/**** Print the monitor value ****/

static void AMMP_FASTCALL MonitorPrint(FILE *FH, const char *Str, float Val)
{
  int           k = AMMP_MONITOR_FMT - strlen(Str);

  fprintf(FH, "  %s", Str);

  while(k-- > 0) fputc('.', FH);

  fprintf(FH, ": %12.6f\n", Val);
}
