/*****************************************************************************
 * motif.c                                                                   *
 *     The files 'motif.c' and 'opengl.c' contain all of the functions       *
 *     necessary for the GUI portion of GRAMMP.  This file contains          *
 *     functions handling the Motif functionality of GRAMMP.                 *
 *                                                                           *
 * Written by David Cavanaugh (4/99)                                         *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* include files for Motif and X Window functions */
#include <Xm/Text.h>
#include <Xm/MessageB.h>
#include <Xm/SelectioB.h>
#include <Xm/LabelG.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/PushBG.h>
#include <Xm/PanedW.h>
#include <Xm/Separator.h>
#include <Xm/ToggleBG.h>
#include <Xm/MainW.h>
#include <Xm/RowColumn.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>
#include <Xm/CascadeB.h>
#include <Xm/Frame.h>
#include <Xm/DrawingA.h>
#include <Xm/Label.h>
#include <Xm/ScrollBar.h>
#include <Xm/FileSB.h>
#include <Xm/TextF.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <time.h>
#include "dsc.h"

void CreateActionButtons(Widget, char *, XtCallbackProc, XtPointer, int, int);
Widget CreateLabeledTextField(Widget, char *, char *);
Widget GetTopShell(Widget);
void dscDestroyWidget(Widget, Widget, XtPointer);
void GetOnlineHelp();
void RunAmmpCommand(Widget,Widget);
void Minimize(Widget, Widget *);
void Monitor(void);
void MinimizeCallback(Widget);
void SelectAtomCallback(Widget);
void TorsionSearchCallback(Widget);
void SelectAtom(Widget, Widget *);
void MDCallback(Widget);
void AnalyzeCallback(Widget);
void Analyze(Widget, Widget *);
void draw_COM(void);
void draw_ONLAST(void);
void AddMenuItem(Widget, char *, int, XtCallbackProc, XtPointer);
void AddSeparator(Widget);
Widget AddMenuPane(Widget, char *, int);
void RunMD(Widget, Widget *);
void WriteCallback (Widget,XtPointer, XtPointer);
void SetOptionMode(Widget, int);
void GetAmmpBrowser(Widget);
void SetAmmpBrowser(Widget, XtPointer, XmFileSelectionBoxCallbackStruct *);
void MouseTracking(Widget, XButtonEvent *, String *, int *);
void GrfxInit(void);
void graphic_do(int *, char **);
void IO_poller(void);
void draw_walker( void);
int PipePoller(void);
void PickCallback(Widget, int);
void CloseGraphicsWindow(void);
void Monitor(void);
void Exit(void);
void LoadCallback (Widget);
void OutputCallback (Widget, XtPointer, XtPointer);
void HSBCallback (Widget, XtPointer, XmScrollBarCallbackStruct *);
void VSBCallback (Widget, XtPointer, XmScrollBarCallbackStruct *);
void ResizeScene(Widget);
void LabelsOffCallback(Widget);
void SetMolecularColor(Widget, char *);
void load_atom(void);
void load_color(void);
void load_connect(void);
void load_label(void);
void load_tether(void);
void load_noel(void);


/* variables which require scope within this entire file */
GLboolean doubleBuffer;
GLXContext cx;
int New_angle = TRUE;
int New_distance = TRUE;
typedef struct { int num_fields;  Widget *fields; } TextForm;
char *AMMP_BROWSER = NULL;   /* defines Web browser for Help pages */
char *AMMP_URL = NULL;            /* defines url for Help pages */

/***************************************************************************** 
 * void graphic_do() - initialize graphics.                                  *
 *         1. start X Window application                                     *
 *         2. define Motif functionality: menus, callbacks, buttons, etc ... *
 *         3. develop OpenGL context and bind to X Window widget             *
 *         4. start event loop                                               *
 *****************************************************************************/

void graphic_do(argc,argv) 

char **argv;
int *argc;
{
	#define INITIAL_WIDTH  500   /* Initial width of X Window */
	#define INITIAL_HEIGHT 500   /* Initial height of X Window */

	Widget ToggleBox;            /* Toggle button box */
	Widget RadioBox;             /* Radio button box */
	Widget MenuPane;             /* pull-down menu pane */
	Widget PullRightMenu;        /* pull-right menu pane */
	Widget MenuBar;              /* menu-bar widget */
	Widget Btn;                  /* push button widgets */
	Widget HelpWidget;           /* help pulldown menu */
	Widget MainWindow;           /* X window widget */
	Widget Toplevel;             /* highest level widget */
	XtAppContext App;            /* X application context */
	Colormap cmap;               /* colormap */
	Arg BtnArgs[5];              /* Arguments for Btn widget */
	Arg ScrollArgs[5];           /* Arguments for VSB & HSB widgets */

	int i,j;
	/* define callbacks for mouse events */
	String translations = 
	     "<Btn1Down>: MouseTracking(down) ManagerGadgetArm() \n\
	      <Btn1Up>: MouseTracking(up) ManagerGadgetActivate() \n\
	      <Btn1Motion>: MouseTracking(motion) ManagerGadgetButtonMotion()";
	XtActionsRec actions;

	char *Options[] = { "Rotate (Y-X)", "Rotate (Y-Z)", "Translate", "Scale-Slab", "Autocenter", "Autoscale" };
	char *Colors[] = { "Monochrome", "CPK", "Force" };
	char *Pick[] = { "Atom", "Distance", "Angle", "Dihedral" , "Show Tethers","Show NOELS", "Show Violations"};

    Toplevel = XtVaAppInitialize(&App, "GRAMMP", NULL, 0, argc, argv,NULL,
			       XmNminWidth, INITIAL_WIDTH, XmNminHeight,INITIAL_HEIGHT,NULL);
    display = XtDisplay(Toplevel);

	doubleBuffer = GL_TRUE; made_current = GL_FALSE; 

    /* find an OpenGL-capable RGB visual with depth buffer */
    vi = glXChooseVisual(display, DefaultScreen(display), VisualArgs);
    if (vi == NULL)
	{
		vi = glXChooseVisual(display, DefaultScreen(display), snglBuf);
		if (vi == NULL)
		{
			XtAppError(App, "no RGB visual with depth buffer");
			doubleBuffer = GL_FALSE;
		}	
	}

	/* Create an OpenGL rendering context.  No sharing of display lists (NULL) 
	   and favor direct connection (GL_TRUE) */
	cx = glXCreateContext(display, vi, NULL, GL_TRUE);
    if (cx == NULL)  XtAppError(App, "could not create rendering context");

    /* create an X colormap since probably not using default visual */
    cmap = XCreateColormap(display, RootWindow(display, vi->screen),vi->visual, AllocNone);

    /* Establish the visual, depth, and colormap of the Toplevel
       widget before the widget is realized.  */
    XtVaSetValues(Toplevel, XtNvisual, vi->visual, XtNdepth, vi->depth, XtNcolormap, cmap, NULL);
    MainWindow = XmCreateMainWindow(Toplevel, "MainWindow", NULL, 0);
    XtManageChild(MainWindow);

	/* add the MouseTracking function used with translation table */
	actions.string = "MouseTracking";
	actions.proc = (XtActionProc) MouseTracking; 
	XtAppAddActions(App,&actions,1);

   /***************************************************************************** 
    * Create Vertical and Horizontal Scollbars                                  *
    *****************************************************************************/
    XtSetArg(ScrollArgs[0], XmNorientation, XmVERTICAL);
    XtSetArg(ScrollArgs[1], XmNincrement, SBincrement);
    XtSetArg(ScrollArgs[2], XmNmaximum, SBmax);
    XtSetArg(ScrollArgs[3], XmNsliderSize, SBincrement);
	VSB = XmCreateScrollBar(MainWindow,"VSB", ScrollArgs, 4);
    XtAddCallback(VSB, XmNvalueChangedCallback,(XtCallbackProc) VSBCallback, NULL); 
    XtAddCallback(VSB, XmNincrementCallback,(XtCallbackProc) VSBCallback, NULL); 
    XtAddCallback(VSB, XmNdecrementCallback,(XtCallbackProc) VSBCallback, NULL); 
    XtManageChild(VSB);

    XtSetArg(ScrollArgs[0], XmNorientation, XmHORIZONTAL);
	HSB = XmCreateScrollBar(MainWindow,"HSB", ScrollArgs, 4);
    XtAddCallback(HSB, XmNvalueChangedCallback,(XtCallbackProc) HSBCallback, NULL); 
    XtAddCallback(HSB, XmNincrementCallback,(XtCallbackProc) HSBCallback, NULL); 
    XtAddCallback(HSB, XmNdecrementCallback,(XtCallbackProc) HSBCallback, NULL); 
    XtManageChild(HSB);

	for (i=0;i<10;i++)
	{
		SB[i].hlv = 0;
		SB[i].vlv = 0;
	}

    /* Create horizontal menu bar along the top of the X Window */
    MenuBar = XmCreateMenuBar(MainWindow, "MenuBar", NULL, 0);
    XtManageChild(MenuBar);

   /***************************************************************************** 
    * Create "File" pull-down menu                                              * 
    *****************************************************************************/
	MenuPane = AddMenuPane(MenuBar,"File",'F');
	AddMenuItem(MenuPane, "Load AMMP file", 'L', (XtCallbackProc) LoadCallback, NULL);
	AddMenuItem(MenuPane, "Output AMMP file", 'O', (XtCallbackProc) OutputCallback, NULL);
	AddMenuItem(MenuPane, "Close Window", 'C', (XtCallbackProc) CloseGraphicsWindow, NULL);

	AddSeparator(MenuPane);

	AddMenuItem(MenuPane, "Exit", 'x', (XtCallbackProc) Exit, NULL);

   /***************************************************************************** 
    * Create "View" pull-down menu                                              * 
    *****************************************************************************/
	MenuPane = AddMenuPane(MenuBar,"View",'V');

	PullRightMenu = AddMenuPane(MenuPane,"Axis View",'A');
	AddMenuItem(PullRightMenu, "X", 'X', (XtCallbackProc) LookDownAxis, "x");
	AddMenuItem(PullRightMenu, "Y", 'Y', (XtCallbackProc) LookDownAxis, "y");
	AddMenuItem(PullRightMenu, "Z", 'Z', (XtCallbackProc) LookDownAxis, "z");

	PullRightMenu = AddMenuPane(MenuPane,"Colors",'C');
	RadioBox = XmCreateRadioBox(PullRightMenu,"RadioBox",NULL,0);
	for (i=0;i<3;i++) 
	{
		Btn = XtVaCreateManagedWidget(Colors[i],xmToggleButtonGadgetClass,
									  RadioBox,NULL);
		XtAddCallback(Btn,XmNvalueChangedCallback,(XtCallbackProc) SetMolecularColor,Colors[i]);
		/* set CPK coloring to default */
		if (i==1)  XmToggleButtonSetState(Btn,TRUE,TRUE);
	}
	XtManageChild(RadioBox);

   /***************************************************************************** 
    * Create "Controls" pull-down menu                                          * 
    *****************************************************************************/
	MenuPane = AddMenuPane(MenuBar,"Controls",'C');

	RadioBox = XmCreateRadioBox(MenuPane,"RadioBox",NULL,0);
	for (i=0;i<4;i++)  
	{
		Btn = XtVaCreateManagedWidget(Options[i],xmToggleButtonGadgetClass,
									  RadioBox,NULL);
		XtAddCallback(Btn,XmNvalueChangedCallback,(XtCallbackProc) SetOptionMode, 
					  (XtPointer) i);
		if (i==0)  XmToggleButtonSetState(Btn,TRUE,TRUE);
	}
	XtManageChild(RadioBox);

	AddSeparator(MenuPane);

	AddMenuItem(MenuPane, "Center on Last Atom", 'L', (XtCallbackProc) draw_ONLAST, NULL);
	AddMenuItem(MenuPane, "Center on COM", 'O', (XtCallbackProc) CenterMolecule, NULL);
	AddMenuItem(MenuPane, "Select Atom for Center", 'S', (XtCallbackProc) SelectAtomCallback, NULL);

	AddSeparator(MenuPane);

    ToggleBox = XtVaCreateWidget("ToggleBox",xmRowColumnWidgetClass,MenuPane,XmNpacking, XmPACK_COLUMN, 
	                             XmNnumColumns,1,NULL);
	for (i=4;i<6;i++)  
	{
		Btn = XtVaCreateManagedWidget(Options[i],xmToggleButtonGadgetClass,ToggleBox,NULL);
		XtAddCallback(Btn, XmNvalueChangedCallback, (XtCallbackProc) SetOptionMode, (XtPointer) i);
		XmToggleButtonSetState(Btn,TRUE,TRUE);
	}
	XtManageChild(ToggleBox);

   /***************************************************************************** 
    * Create "Geometry" pull-down menu                                          * 
    *****************************************************************************/
	MenuPane = AddMenuPane(MenuBar,"Geometry",'G');

    RadioBox = XmCreateRadioBox(MenuPane,"RadioBox",NULL,0);
    for (i=0;i<4;i++)  
	{
        Btn = XtVaCreateManagedWidget(Pick[i],xmToggleButtonGadgetClass,RadioBox,NULL);
        XtAddCallback(Btn,XmNvalueChangedCallback, (XtCallbackProc) PickCallback,(XtPointer) i);
        if (i==0)  XmToggleButtonSetState(Btn,TRUE,TRUE);
    }
	XtManageChild(RadioBox);

	AddSeparator(MenuPane);
	AddMenuItem(MenuPane, "Clear Labels", 'C', (XtCallbackProc) LabelsOffCallback, NULL);

	AddSeparator(MenuPane);
    ToggleBox = XtVaCreateWidget("ToggleBox",xmRowColumnWidgetClass,MenuPane,XmNpacking, XmPACK_COLUMN, 
	                             XmNnumColumns,1,NULL);
	for (i=4;i<7;i++)  
	{
		Btn = XtVaCreateManagedWidget(Pick[i],xmToggleButtonGadgetClass,ToggleBox,NULL);
		XtAddCallback(Btn, XmNvalueChangedCallback, (XtCallbackProc) PickCallback, (XtPointer) i);
		XmToggleButtonSetState(Btn,FALSE,TRUE);
	}
	XtManageChild(ToggleBox);


   /***************************************************************************** 
    * Create "Ammp" pull-down menu                                              * 
    *****************************************************************************/
	MenuPane = AddMenuPane(MenuBar,"Ammp",'A');

	AddMenuItem(MenuPane, "Dynamics", 'D', (XtCallbackProc) MDCallback, NULL);
	AddMenuItem(MenuPane, "Minimize", 'M', (XtCallbackProc) MinimizeCallback, NULL);
	AddMenuItem(MenuPane, "Torsion Search", 'T', (XtCallbackProc) TorsionSearchCallback, NULL);

	AddSeparator(MenuPane);

	AddMenuItem(MenuPane, "Analyze", 'n', (XtCallbackProc) AnalyzeCallback, NULL);
	AddMenuItem(MenuPane, "Monitor", 'o', (XtCallbackProc) Monitor, NULL);

   /***************************************************************************** 
    * Create "Help" pull-down menu                                              * 
    *****************************************************************************/
	MenuPane = XmCreatePulldownMenu(MenuBar,"HelpPulldown",NULL,0);
	HelpWidget = XtVaCreateManagedWidget("Help",xmCascadeButtonWidgetClass, MenuBar,
   	             XmNsubMenuId, MenuPane,XmNmnemonic, 'H', NULL);
	AddMenuItem(MenuPane, "On-line", 'O', (XtCallbackProc) GetOnlineHelp, NULL);
	/* define this widget as a help widget (i.e. right justify this menu-pane on the menubar) */
	XtVaSetValues(MenuBar,XmNmenuHelpWidget,HelpWidget,NULL);

	/* Glxarea is the "screen" for all of the OpenGL commands */
    Glxarea = XtVaCreateManagedWidget("Glxarea", xmDrawingAreaWidgetClass, MainWindow,
		XmNtranslations,XtParseTranslationTable(translations),NULL);

	XtAddCallback(Glxarea, XmNinputCallback,(XtCallbackProc) MouseTracking,NULL);
	XtAddCallback(Glxarea, XmNresizeCallback, (XtCallbackProc) ResizeScene, NULL);
	XtAddCallback(Glxarea, XmNexposeCallback, (XtCallbackProc) draw_walker, NULL); 

    /* set up Application's window layout */
    XmMainWindowSetAreas(MainWindow, MenuBar, NULL, NULL, NULL, NULL);  

    XtRealizeWidget(Toplevel);

	/* Now that the Toplevel is realized, we can bind the OpenGL rendering
	   context to the window */
    glXMakeCurrent(display, XtWindow(Glxarea), cx);
    made_current = GL_TRUE;

	GrfxInit();

    dscIdleFunc(IO_poller);
	dscExec(App); 

}  /* end of graphic_do */


/*****************************************************************************
 * GrfxInit() - Initialize important global variables for the grfx process   *
 *****************************************************************************/
void GrfxInit()
{
int i;


    Font draw_font;
    GLfloat light_ambient[]  = { 1.0, 1.0, 1.0, 1.0 };
    GLfloat light_diffuse[]  = { 1.0, 1.0, 1.0, 1.0 };
    GLfloat light_specular[] = { 1.0, 1.0, 1.0, 1.0 };
    GLfloat light_position[] = { 1.0, 1.0, 1.0, 0.0 };

	/* OpenGL initialization */
    glLightfv (GL_LIGHT0, GL_AMBIENT,  light_ambient);
    glLightfv (GL_LIGHT0, GL_DIFFUSE,  light_diffuse);
    glLightfv (GL_LIGHT0, GL_SPECULAR, light_specular);
    glLightfv (GL_LIGHT0, GL_POSITION, light_position);

    glClearColor(0.0,0.0,0.0,1.0);
    glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    draw_font = XLoadFont(display,"9x15");
    glXUseXFont(draw_font,0,127,0);

    glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    glOrtho(-1.0,1.0,-1.0,1.0,-1.0,1.0);
    glDepthFunc(GL_LESS);
    glEnable(GL_DEPTH_TEST);

	/* global variable initializatoins */
	last_picked_draw_ATOM;
    first_draw_ATOM = NULL;
    last_draw_ATOM = NULL;
    first_draw_BOND = NULL;
    last_draw_BOND = NULL;
	draw_center[0] = 0.0; draw_center[1] = 0.0; draw_center[2] = 0.0;
	show_tethers = FALSE;
	show_noels = FALSE;
	show_violations = FALSE;
    draw_winx = INITIAL_WIDTH; draw_winy = INITIAL_HEIGHT;

	LOADING = TRUE;
	PickList[DISTANCE].num = 0;
	PickList[DISTANCE].show = FALSE;
	PickList[DISTANCE].calc = FALSE;
	PickList[ANGLE].num = 0;
	PickList[ANGLE].show = FALSE;
	PickList[ANGLE].calc = FALSE;
	PickList[DIHEDRAL].num = 0;
	PickList[DIHEDRAL].show = FALSE;
	PickList[DIHEDRAL].calc = FALSE;
	for (i=0;i<MAX_PICKED_ATOMS;i++)  
	{
		PickList[DISTANCE].atoms[i] = NULL;
		PickList[ANGLE].atoms[i] = NULL;
		PickList[DIHEDRAL].atoms[i] = NULL;
	}
}

/******************************************************************************
 * void MouseTracking() - called when the mouse is moved across the graphics  *
 *                        window.  When mouse button is pushed down and       *
 *                        moved, simulataneously update the scrollbars.  Let  *
 *                        the scrollbar callback functions do the appropriate *
 *                        appropriate work                                    *
 *                                                                            *
 * Input:                                                                     *
 *     Widget parent - parent widget                                          *
 *     XButtonEvent *event - structure containing XEvent information          *
 *     String *args - arguments (e.g. "up","down","motion"                    *
 *     int *num_args - number arguments                                       *
 ******************************************************************************/
void MouseTracking(parent, event, args, num_args)
Widget parent;
XButtonEvent *event;
String *args;
int *num_args;
{
	int x,y;
	int xx,yy;
	float start,finish;
	static int old_x,old_y;
	static int last_vsb, last_hsb;
	static int count;
	int new_x, new_y;
	char message[256];

	if (!strcmp(args[0],"down"))
	{
		if (PickAtom(event))  return;
		LEFT_BUTTON_DOWN = TRUE; 
		old_x = (int) (event->x/draw_winx*SBmax);
		old_y = (int) (event->y/draw_winy*SBmax);
		count = 0;
		return;
	}
	else if (!strcmp(args[0],"up"))
	{
		LEFT_BUTTON_DOWN = FALSE; 
	}
	else if (!strcmp(args[0],"motion"))
	{
		if (!LEFT_BUTTON_DOWN)  return;
		new_x = (int) (event->x/draw_winx*SBmax);
		new_y = (int) (event->y/draw_winy*SBmax);
		x = new_x - old_x; y = new_y - old_y;

		old_x = new_x;  old_y = new_y;

		x += SB[OptionMode].hlv; 
		y += SB[OptionMode].vlv;
		xx = x;
		yy = y;
		if (OptionMode == SCALE_SLAB)
		{
			if (yy > (SBmax - SBincrement))   y = yy = SBmax-SBincrement;
			if (yy < 0)                      y = yy = 0;
			if ((x > (SBmax - SBincrement)) || (x < 0))
			{
				while (x > (SBmax - SBincrement))  x -= SBmax;
				while (x < 0)                      x += SBmax;
				xx = SB[OptionMode].hlv;
				SB[OptionMode].hlv = x;
			}
		}
		else
		{
			if ((x > (SBmax - SBincrement)) || (x < 0))
			{
				while (x > (SBmax - SBincrement))  x -= SBmax;
				while (x < 0)                      x += SBmax;
				xx = SB[OptionMode].hlv;
				SB[OptionMode].hlv = x;
			}
			if ((y > (SBmax - SBincrement)) || (y < 0))
			{
				while (y > (SBmax - SBincrement))  y -= SBmax;
				while (y < 0)                      y += SBmax;
				yy = SB[OptionMode].vlv;
				SB[OptionMode].vlv = y;
			}
		}	
#ifdef DEBUG
		fprintf(stderr,"xx,yy: (%d,%d)\n",xx,yy);
		fprintf(stderr,"hlv,vlv: (%d,%d)\n\n",SB[OptionMode].hlv,SB[OptionMode].vlv);
#endif
		XmScrollBarSetValues(HSB,xx,SBincrement,SBincrement,0,TRUE);
		XmScrollBarSetValues(VSB,yy,SBincrement,SBincrement,0,TRUE);
		SB[OptionMode].hlv = x; 
		SB[OptionMode].vlv = y;
	}
	else  
	{
		sprintf(message,"Unknown Mode (%s) in MouseTracking",args[0]);
		GrammpError(message,FATAL);
	}	
}


/*****************************************************************************
 * IO_poller - poll the ammp_to_grfx_pipe for available data.  parse the     *
 *             data and execute the cooresponding commands                   *
 *****************************************************************************/
void IO_poller()
{
char string[256];
int i,n_message,message;
int loadme;

/* check whether any data in the ammp_to_grfx_pipe, if no then return.
   otherwise parse the data and execute the appropriate commands */
	if (!PipePoller())  return;

	i=ReadFromAmmp((char *) &n_message,sizeof(int));
	if (i < 1)  return;
	for (i=0; i< n_message; i++)
	{
		ReadFromAmmp(&message,sizeof(int));
#ifdef DEBUG
		fprintf(stderr,"message = %d\n",message);
#endif
		switch (message)
		{
			case (-1): 
				exit(0);
				break;	
			case (0): 
				load_atom();
				break;	
			case (1):
				load_color(); 
				break;
			case (2):
				load_connect();
				loadme = FALSE;
				break;
			case (3):
				load_label();
				break;
			case (4):
				LOADING = FALSE;
				break;
			case (5):
				load_tether();
				break;
			case (6):
				load_noel();
				break;
			default:
				sprintf(string,"Unrecognizable message (%s) in IO_poller()",message);
				GrammpError(string,FATAL);
				break;
		}  /* end of switch */
	}   /* i */
	if (!LOADING)  draw_walker(); 
}  /* end of IO_poller */


/*****************************************************************************
 * void VSBCallback() - vertical scrollbar callback, action depends on the   *
 *                      OptionMode variable                                  *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     XmScrollBarCallback *cbs - structure containing info. about scrollbar *
 *****************************************************************************/
void VSBCallback (parent, clientData, cbs)  
Widget parent;
XtPointer clientData;
XmScrollBarCallbackStruct *cbs;
{
	int vlv = SB[OptionMode].vlv;  /* last VSB value */
	int vcv = cbs->value;      /* current VSB value */
	GLdouble foreground[4] = { 0.0, 0.0,  1.0,  1.0 };
	GLdouble background[4] = { 0.0, 0.0,  1.0, -1.0 };
	float value;

/* No scrollbar wrapping (see explanation below) b/c slab scrollbar goes from
   infinite slab to zero slab over the span of the scrollbar */
    if (OptionMode == SCALE_SLAB)
	{
		value = 1 - vcv/((float) SBmax - 1);

		foreground[3] = value;
		glClipPlane(GL_CLIP_PLANE0, foreground);
		glEnable(GL_CLIP_PLANE0); 

		background[3] = value;
		glClipPlane(GL_CLIP_PLANE1, background);
		glEnable(GL_CLIP_PLANE1); 
	}
	else if (OptionMode == ROTATE_Y_X)
	{
		value = (vcv-vlv)*(360.0/(float) SBmax);
		Rotate('x',value);
	}
	else if (OptionMode == ROTATE_Y_Z)
	{
		value = (vcv-vlv)*(360.0/(float) SBmax);
		Rotate('z',value);
	}
	else if (OptionMode == TRANSLATE)
	{
		value = 0.05 * (float) (vcv-vlv);
		Translate('y',value);
	}
	vlv = vcv;
	if (OptionMode != SCALE_SLAB)
	{
/* Enable scrollbar wrapping: if at SBmax and incrementing then move scrollbar to
   zero; if at zero and decrementing then move scrollbar to SBmax.  Change hlv
   so that scaling will be done correctly  */
		if ((vcv == SBmax-SBincrement) && (cbs->reason == XmCR_INCREMENT))
		{
			vlv = -1;
			XmScrollBarSetValues(VSB,0,SBincrement,SBincrement,0,TRUE);
		}
		else if ((vcv == 0) && (cbs->reason == XmCR_DECREMENT))
		{
			vlv = SBmax;
			XmScrollBarSetValues(VSB,SBmax-SBincrement,SBincrement,SBincrement,0,TRUE);
		}
	}
	SB[OptionMode].vlv = vlv;
	draw_walker();
}


/*****************************************************************************
 * void HSBCallback() - horizontal scrollbar callback, action depends on the *
 *                      OptionMode variable                                  *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     XmScrollBarCallback *cbs - structure containing info. about scrollbar * 
 *****************************************************************************/
void HSBCallback (parent, clientData, cbs)  
Widget parent;
XtPointer clientData;
XmScrollBarCallbackStruct *cbs;
{
	int hlv = SB[OptionMode].hlv;  /* last value */
	int hcv = cbs->value;          /* current value */
	float value;
	float scaling;
	float factor;

    if (OptionMode == SCALE_SLAB)
	{
		factor = 1 + ((float)(hcv-hlv))/SBmax;
		ScaleMolecule(&factor); draw_walker();
	}
    else if ((OptionMode == ROTATE_Y_X) || (OptionMode == ROTATE_Y_Z))
	{
		/* convert to degrees from increment */
		value = (hcv-hlv)*(360.0/(float) SBmax);
		Rotate('y',value);
	}
	else if (OptionMode == TRANSLATE)
	{
		value = 0.05 * (float) (hcv-hlv);
		Translate('x',-value);
	}
	hlv = hcv;

/* Enable scrollbar wrapping: if at SBmax and incrementing then move scrollbar to
   zero; if at zero and decrementing then move scrollbar to SBmax.  Change hlv
   so that scaling will be done correctly */
	if ((hcv == SBmax-SBincrement) && (cbs->reason == XmCR_INCREMENT))
	{
		hlv = -1;
		XmScrollBarSetValues(HSB,0,SBincrement,SBincrement,0,FALSE);
	}
	else if ((hcv == 0) && (cbs->reason == XmCR_DECREMENT))
	{
		hlv = SBmax;
		XmScrollBarSetValues(HSB,SBmax-SBincrement,SBincrement,SBincrement,0,FALSE);
	}
	SB[OptionMode].hlv = hlv;
	draw_walker();
}


/******************************************************************************
 * void SetOptionMode() - Set the OptionMode variable (controls the scrollbar *
 *                        actions), Autocenter, and Autoscale                 *
 *                                                                            *
 * Input:                                                                     *
 *     Widget parent - parent widget                                          *
 *     int option - option to set                                             *
 ******************************************************************************/
void SetOptionMode(parent, option)  

Widget parent;
int option;
{
	if      (option == 0)  OptionMode = ROTATE_Y_X;
	else if (option == 1)  OptionMode = ROTATE_Y_Z;
	else if (option == 2)  OptionMode = TRANSLATE;
	else if (option == 3)  OptionMode = SCALE_SLAB;

	XmScrollBarSetValues(HSB,SB[OptionMode].hlv,SBincrement,SBincrement,0,FALSE);
	XmScrollBarSetValues(VSB,SB[OptionMode].vlv,SBincrement,SBincrement,0,FALSE);

	if (option == 4)  Autocenter = !Autocenter;
	if (option == 5)  Autoscale = !Autoscale;
}


/*****************************************************************************
 * void AddMenuItem() - Add menu item to a parent widget                     *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     char *name - name of menu-item                                        *
 *     int mnemonic - mnemonic for name                                      *
 *     XtCallbackProc callback - callback for clicking on menu item          *
 *     XtPointer - pointer to possible callback data                         *
 *****************************************************************************/
void AddMenuItem(parent, name, mnemonic, callback, data)  
Widget parent;
char *name;
int mnemonic;
XtCallbackProc callback;
XtPointer data;
{
	Widget btn;
	Arg btn_args[1];

    XtSetArg(btn_args[0], XmNmnemonic, mnemonic);
	btn = XmCreatePushButton(parent, name, btn_args, 1);
   	if (callback != NULL)  XtAddCallback(btn, XmNactivateCallback, callback, data); 
    XtManageChild(btn);
}


/*****************************************************************************
 * void AddMenuPane() - Add a menupane to a parent.  Can be used to build    *
 *                      drop-down menus from a menu-bar or pull-right menus  *
 *                      from a drop-down menu.                               *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     char *name - name of menupane                                         *
 *     int mnemonic - mnemonic for name                                      *
 * Returns:                                                                  *
 *    Widget - menupane widget is returned                                   *
 *****************************************************************************/
Widget AddMenuPane(parent, name, mnemonic)  
Widget parent;
char *name;
int mnemonic;
{
	Widget menupane;
	Widget cascade;
	Arg menupane_args[3];

    XtSetArg(menupane_args[0], XmNvisual, vi->visual);
    menupane = XmCreatePulldownMenu(parent, "menupane", menupane_args, 1); 
    XtSetArg(menupane_args[0], XmNsubMenuId, menupane);
    XtSetArg(menupane_args[1], XmNmnemonic, mnemonic);
    cascade = XmCreateCascadeButton(parent, name, menupane_args, 2);
    XtManageChild(cascade);
	return(menupane);
}


/*****************************************************************************
 * void AddSeparator() - Add separator bar to parent                         *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *****************************************************************************/
void AddSeparator(Widget parent)  
{
	Widget btn;
	btn = XmCreateSeparator(parent, "Separator", NULL, 0);
	XtManageChild(btn);
}


/*****************************************************************************
 * void PickCallback() - callback for selecting atoms, distances, etc...     *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     int option - defines which picking option has been selected           * 
 *****************************************************************************/
void PickCallback(Widget parent, int option)  
{  
	char string[256];
	if (option == 0)  {}

	/* pick distances */
	else if (option == 1)  {
		PickList[DISTANCE].show = TRUE;
		PickList[DISTANCE].calc = TRUE;
		PickList[ANGLE].calc = FALSE;
		PickList[DIHEDRAL].calc = FALSE;
	}
	/* pick angles */
	else if (option == 2)  {
		PickList[ANGLE].show = TRUE;
		PickList[ANGLE].calc = TRUE;
		PickList[DIHEDRAL].calc = FALSE;
		PickList[DISTANCE].calc = FALSE;
	}
	/* pick dihedrals */
	else if (option == 3)  {
		PickList[DIHEDRAL].show = TRUE;
		PickList[DIHEDRAL].calc = TRUE;
		PickList[DISTANCE].calc = FALSE;
		PickList[ANGLE].calc = FALSE;
	}
	/* toggle show tethers */
	else if (option == 4)  {
		show_tethers = !show_tethers;
	}
	/* toggle show noels */
	else if (option == 5)  {
		show_noels = !show_noels;
	}
	/* toggle show violations */
	else if (option == 6)  {
		show_violations = !show_violations;
	}
	else
	{
		sprintf(string,"Unknown option (%d) in PickCallback()\n",option);
		GrammpError(string,FATAL);
	}
	draw_walker();
}


/*****************************************************************************
 * RunMD() - callback from MDCallback.  parses pertient MD information       *               
 *           from the text fields and then sends an MD command to ammp       *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     Widget *widget_list - pointers to text field widgets containing info. *
 *****************************************************************************/
void RunMD(parent, widget_list)
Widget parent;
Widget *widget_list;
{
	char *temp;
	char *holder;
	char *nsteps;
	float time_step;
	char string[256];
	Widget grandparent;

	XtPopdown(GetTopShell(parent));

	temp =  XmTextFieldGetString(widget_list[0]);
	holder =  XmTextFieldGetString(widget_list[1]);
	nsteps =  XmTextFieldGetString(widget_list[2]);
	time_step = atof(holder)/100000;
	sprintf(string,"tpac %s %f %s;",nsteps,time_step,temp);
    WriteToAmmp(string,strlen(string)); 
	/* XtFree(temp); XtFree(time_step); XtFree(nsteps); */
}



/*****************************************************************************
 * Analyze() - callback from AnalyzeCallback().  parses pertient info.       *
 *             and sends an analyze command to ammp                          *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     Widget *widget_list - pointers to text field widgets containing info. *
 *****************************************************************************/
void Analyze(parent, widget_list)
Widget parent;
Widget *widget_list;
{
    char *start;
    char *end;
    char string[256];

    XtPopdown(GetTopShell(parent));
    start =  XmTextFieldGetString(widget_list[0]);
    end =  XmTextFieldGetString(widget_list[1]);
    sprintf(string,"analyze %s %s;",start,end);
    WriteToAmmp(string,strlen(string));
}


/*****************************************************************************
 * MDCallback() - display a custom dialog for molecular dynamics. prompt     *
 *                user for temperature, time-step, and number of steps       *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *****************************************************************************/
void MDCallback(parent)  
Widget parent;
{
	static Widget dialog = NULL;
	Widget pb,form,pane,temp,nsteps,time_step;
	Widget *widget_list;
	Arg args[3];
	XmString title;

	if (!dialog)  
	{
		widget_list = (Widget *) malloc (3 * sizeof(Widget)); 
		XtSetArg(args[0],XmNvisual, vi->visual);
		XtSetArg(args[1],XmNautoUnmanage, False);
		dialog = XmCreateFormDialog(parent,"dialog",args,2);
		pane = XtVaCreateManagedWidget("pane",xmPanedWindowWidgetClass,dialog,XmNsashWidth,1,XmNsashHeight,1,NULL);
		form = XtVaCreateManagedWidget("upper_form",xmRowColumnWidgetClass,pane, NULL);

		title = XmStringCreateSimple("Molecular Dynamics (Temperature Constrained)");
		XtVaCreateManagedWidget("label",xmLabelWidgetClass,form,XmNlabelString,title,XmNmarginLeft,0,NULL);
		XmStringFree(title);
		AddSeparator(form);

		/* Prompt user for simulation temperature */
		temp = CreateLabeledTextField(form,"Temperature (K): ","300");

		/* Prompt user the simulation time step */
		time_step = CreateLabeledTextField(form,"Time Step (fs):  ","1");

		/* Prompt user for number of dynamics steps */
		nsteps = CreateLabeledTextField(form,"Number of Steps: ",NULL);

		/* Create action area with OK, Clear, and Cancel buttons */
		form = XtVaCreateManagedWidget("lower_form",xmFormWidgetClass,pane, XmNfractionBase, 5, NULL);
		CreateActionButtons(form,"Ok",(XtCallbackProc) RunMD, (XtPointer) widget_list,0,1);
		/* CreateActionButtons(form,"Clear",NULL,NULL,2,3); */
		CreateActionButtons(form,"Cancel",(XtCallbackProc) dscDestroyWidget, (XtPointer) dialog,4,5);
		widget_list[0] = temp;
		widget_list[1] = time_step;
		widget_list[2] = nsteps;
	}
	XtManageChild(dialog);
	XtPopup(XtParent(dialog),XtGrabNone);
}


/*****************************************************************************
 * ClearTextFields() - clear text fields                                     *
 *****************************************************************************/
void ClearTextFields(num_fields,fields)
int num_fields;
Widget *fields;
{
int i;

for (i=0;i<num_fields;i++)  
	XmTextFieldSetString(fields[i],"");

return;
}


/******************************************************************************
 * MinimizeCallback() - open a dialog box to specify minimization information *
 *                      then send a command to ammp to minimize using the     *
 *                      specified information.                                *
 *                                                                            *
 * Input:                                                                     *
 *     Widget parent - parent widget                                          *
 ******************************************************************************/
void MinimizeCallback(parent)  
Widget parent;
{
	static Widget dialog = NULL;
	Widget pb,form,pane,nsteps,ncycles;
	Widget *widget_list;
	Arg args[3];
	XmString title;

	if (!dialog)
	{
		widget_list = (Widget *) malloc (3 * sizeof(Widget)); 
		XtSetArg(args[0],XmNvisual, vi->visual);
		XtSetArg(args[1],XmNautoUnmanage, False);
		dialog = XmCreateFormDialog(parent,"dialog",args,2);
		pane = XtVaCreateManagedWidget("pane",xmPanedWindowWidgetClass,dialog,XmNsashWidth,1,XmNsashHeight,1,NULL);
		form = XtVaCreateManagedWidget("upper_form",xmRowColumnWidgetClass,pane, NULL);

		title = XmStringCreateSimple("Conjugate Gradient Minimization");
		XtVaCreateManagedWidget("label",xmLabelWidgetClass,form,XmNlabelString,title,XmNmarginLeft,55,NULL);
		XmStringFree(title);
		AddSeparator(form);

		/* Prompt user for number of minimization steps */
		nsteps = CreateLabeledTextField(form, "Number of Steps (Per Cycle): ","20");

		/* Prompt user the number of cycles */
		ncycles = CreateLabeledTextField(form,"Number of Total Cycles:      ","1");

		form = XtVaCreateManagedWidget("lower_form",xmFormWidgetClass,pane, XmNfractionBase, 5, NULL);
		CreateActionButtons(form,"Ok",(XtCallbackProc) Minimize,widget_list,0,1);
		/* CreateActionButtons(form,"Clear",NULL,NULL,2,3); */
		CreateActionButtons(form,"Cancel",(XtCallbackProc) dscDestroyWidget,dialog,4,5);
		widget_list[0] = nsteps;
		widget_list[1] = ncycles;
	}
	XtManageChild(dialog);
	XtPopup(XtParent(dialog),XtGrabNone);
}


/*****************************************************************************
 * SelectAtomCallback() - open a dialog to select an atom to center on       *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *****************************************************************************/
void SelectAtomCallback(parent)
Widget parent;
{
    static Widget dialog = NULL;
    Widget pb,form,pane,atom_num,res_num,atom_name;
    Widget *widget_list;
    Arg args[3];
    XmString title;

    if (!dialog)
    {
        widget_list = (Widget *) malloc (3 * sizeof(Widget));
        XtSetArg(args[0],XmNvisual, vi->visual);
        XtSetArg(args[1],XmNautoUnmanage, False);
        dialog = XmCreateFormDialog(parent,"dialog",args,2);
        pane = XtVaCreateManagedWidget("pane",xmPanedWindowWidgetClass,dialog,XmNsashWidth,1,XmNsashHeight,1,NULL);
        form = XtVaCreateManagedWidget("upper_form",xmRowColumnWidgetClass,pane, NULL);

        title = XmStringCreateSimple("       Select Atom");
        XtVaCreateManagedWidget("label",xmLabelWidgetClass,form,XmNlabelString,title,XmNmarginLeft,55,NULL);
        XmStringFree(title);
        AddSeparator(form);

        /* Prompt user for atom number */
        title = XmStringCreateSimple("         Specify:");
        XtVaCreateManagedWidget("label",xmLabelWidgetClass,form,XmNlabelString,title,XmNmarginLeft,55,NULL);
        XmStringFree(title);
        atom_num = CreateLabeledTextField(form,"Atom Number:   ","");

        AddSeparator(form);
        title = XmStringCreateSimple("           OR:");
        XtVaCreateManagedWidget("label",xmLabelWidgetClass,form,XmNlabelString,title,XmNmarginLeft,55,NULL);
        XmStringFree(title);
        AddSeparator(form);
		

        /* Prompt user for residue number */
        res_num = CreateLabeledTextField(form,"Residue Number:","");

        /* Prompt user for atom name */
        atom_name = CreateLabeledTextField(form,"Atom Name:     ","");

        form = XtVaCreateManagedWidget("lower_form",xmFormWidgetClass,pane, XmNfractionBase, 5, NULL);
        CreateActionButtons(form,"Ok",(XtCallbackProc) SelectAtom,widget_list,0,1);
        /* CreateActionButtons(form,"Clear",NULL,NULL,2,3); */
        CreateActionButtons(form,"Cancel",(XtCallbackProc) dscDestroyWidget,dialog,4,5);
        widget_list[0] = atom_num;
        widget_list[1] = res_num;
        widget_list[2] = atom_name;
    }
    XtManageChild(dialog);
    XtPopup(XtParent(dialog),XtGrabNone);
}


/*****************************************************************************
 * TorsionSearchCallback() - open a dialog to select atoms for a torsion     *
 *                           search                                          *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *****************************************************************************/
void TorsionSearchCallback(parent)
Widget parent;
{
	char string[256];
    static Widget dialog = NULL;
	int i;
    Widget pb,form,pane,atom_num[4];
    Widget *widget_list;
    Arg args[3];
    XmString title;

    if (!dialog)
    {
        widget_list = (Widget *) malloc (3 * sizeof(Widget));
        XtSetArg(args[0],XmNvisual, vi->visual);
        XtSetArg(args[1],XmNautoUnmanage, False);
        dialog = XmCreateFormDialog(parent,"dialog",args,2);
        pane = XtVaCreateManagedWidget("pane",xmPanedWindowWidgetClass,dialog,XmNsashWidth,1,XmNsashHeight,1,NULL);
        form = XtVaCreateManagedWidget("upper_form",xmRowColumnWidgetClass,pane, NULL);

        title = XmStringCreateSimple("      Torsion Search");
        XtVaCreateManagedWidget("label",xmLabelWidgetClass,form,XmNlabelString,title,XmNmarginLeft,55,NULL);
        XmStringFree(title);
        AddSeparator(form);

		for (i=0;i<4;i++)
		{
			sprintf(string,"Atom Number: %d",i);
			title = XmStringCreateSimple(string);
			XtVaCreateManagedWidget("label",xmLabelWidgetClass,form,XmNlabelString,title,XmNmarginLeft,55,NULL);
			XmStringFree(title);
			atom_num[0] = CreateLabeledTextField(form,"Atom Number:","");
		}

        form = XtVaCreateManagedWidget("lower_form",xmFormWidgetClass,pane, XmNfractionBase, 5, NULL);
        CreateActionButtons(form,"Ok",(XtCallbackProc) SelectAtom,widget_list,0,1);
        /* CreateActionButtons(form,"Clear",NULL,NULL,2,3); */
        CreateActionButtons(form,"Cancel",(XtCallbackProc) dscDestroyWidget,dialog,4,5);
		for (i=0;i<4;i++)
			widget_list[i] = atom_num[i];
    }
    XtManageChild(dialog);
    XtPopup(XtParent(dialog),XtGrabNone);
}


/******************************************************************************
 * SelectAtom() - callback from SelectAtomCallback().  parse atom information *
 *                from text fields of parent and then search atom structures  *
 *                for a match.                                                *
 *                                                                            *
 * Input:                                                                     *
 *     Widget parent - parent widget                                          *
 *     Widget *widget_list - pointers to text field widgets containing info.  *
 ******************************************************************************/
void SelectAtom(parent, widget_list)
Widget parent;
Widget *widget_list;
{
	char *element;
    draw_ATOM *ap;
    list_link *lp;
	char *atom_name = NULL;
	int atom_num;
	int res_num;
	
	XtPopdown(GetTopShell(parent));
	atom_num =  atoi(XmTextFieldGetString(widget_list[0]));
	res_num =  atoi(XmTextFieldGetString(widget_list[1]));
	atom_name =  XmTextFieldGetString(widget_list[2]);

	if (atom_num != 0)
	{
		lp = first_draw_ATOM;
		if (lp == NULL) return ;
		while (TRUE)
		{
			ap = lp->what;
			 
			if (ap->serial == atom_num)
			{
				draw_center[0] = ap->x;
				draw_center[1] = ap->y;
				draw_center[2] = ap->z;
				break;
			}
			if( lp == last_draw_ATOM ) break;
			if( lp->next == lp) break;
			if( lp->next == NULL ) break;
			lp = lp->next;
		}

	}
	else
	{
		lp = first_draw_ATOM;
		if (lp == NULL) return ;
		while (TRUE)
		{
			ap = lp->what;
			 
			if (res_num == (int)(ap->serial/100))
			{
				element = strchr(ap->name,'.');
				element++;
				if (!strcmp(atom_name,element))
				{
					draw_center[0] = ap->x;
					draw_center[1] = ap->y;
					draw_center[2] = ap->z;
					break;
				}
			}

			if( lp == last_draw_ATOM ) break;
			if( lp->next == lp) break;
			if( lp->next == NULL ) break;
			lp = lp->next;
		}
	}
}

/*****************************************************************************
 * Minimize() - callback from MinimizeCallback().  parse info. from text     *
 *              fields of parent and then send command to ammp               *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     Widget *widget_list - pointers to text field widgets containing info. *
 *****************************************************************************/
void Minimize(parent, widget_list)
Widget parent;
Widget *widget_list;
{
	char *nsteps;
	char *dummy;
	char string[256];
	int i,ncycles;

	XtPopdown(GetTopShell(parent));
	nsteps =  XmTextFieldGetString(widget_list[0]);
	dummy =  XmTextFieldGetString(widget_list[1]);
	ncycles = atoi(dummy);
	sprintf(string,"cngdel %s;",nsteps);
	for (i=0;i<ncycles;i++) 
	{
		WriteToAmmp(string,strlen(string)); 
	}
	/* XtFree(nsteps); XtFree(ncycles);*/
}


/*****************************************************************************
 * Monitor() - send a monitor command to ammp                                *
 *****************************************************************************/
 void Monitor()  
{
	char *string = "monitor;";
	WriteToAmmp(string,strlen(string));
}


/*****************************************************************************
 * GetOnlineHelp() - launch Web browser to view Help pages.  First get       *
 *                   AMMP_BROWSER global variable.  If not set internally,   *
 *                   then check environment variables, else use              *
 *                   FileSelectionBoxDialog to browse for browser executable *
 *****************************************************************************/
void GetOnlineHelp()  
{
    Widget dialog,form,pane;
    Arg args[3];
	char string[256];
    XmString title;
	char command[256];

	if (!AMMP_URL)  
	{
		AMMP_URL = getenv("AMMP_URL");
		if (!AMMP_URL)  
		{
			AMMP_URL = (char *) malloc (256 * sizeof(char));
			strcpy(AMMP_URL,"http://asterix.jci.tju.edu/ammphelp/Index.htm");
		}
	}
	if (!AMMP_BROWSER)  
	{
		AMMP_BROWSER = getenv("AMMP_BROWSER");
		if (!AMMP_BROWSER)
		{
			XtSetArg(args[0],XmNvisual, vi->visual);
			XtSetArg(args[1],XmNautoUnmanage, False);
			dialog = XmCreateFormDialog(Glxarea,"dialog",args,2);
			pane = XtVaCreateManagedWidget("pane",xmPanedWindowWidgetClass,dialog,XmNsashWidth,1,XmNsashHeight,1,NULL);
			form = XtVaCreateManagedWidget("upper_form",xmRowColumnWidgetClass,pane, NULL);

			sprintf(string,"         AMMP_BROWSER Environment Variable NOT SET!\n");
			strcat(string, "Browse Filesystem for Suitable Browser or Cancel On-line Help");
			title = XmStringCreateLtoR(string,XmFONTLIST_DEFAULT_TAG);
			XtVaCreateManagedWidget("label2",xmLabelGadgetClass,form,XmNlabelString,title,NULL);
			XmStringFree(title);

			/* Create action area with Cancel and Browse */
			form = XtVaCreateManagedWidget("lower_form",xmFormWidgetClass,pane, XmNfractionBase, 5, NULL);
			CreateActionButtons(form,"Browse",(XtCallbackProc) GetAmmpBrowser, NULL,1,2);
			CreateActionButtons(form,"Cancel",(XtCallbackProc) dscDestroyWidget, (XtPointer) dialog,3,4);
			XtManageChild(dialog);
			XtPopup(XtParent(dialog),XtGrabNone);
			return;
		}
	}
	sprintf(command,"%s %s",AMMP_BROWSER,AMMP_URL);
	dscSystem(command);
}


/*****************************************************************************
 * GetAmmpBrowser() - Launch a FileSelectionDialog to search for a browser.  *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *****************************************************************************/
void GetAmmpBrowser(Widget parent)
{
		static Widget dialog = NULL;
		Arg args[2];

		XtPopdown(GetTopShell(parent));
		if (!dialog)
		{
			XtSetArg(args[0], XmNvisual, vi->visual);
			dialog = XmCreateFileSelectionDialog(Glxarea,"GetAmmpBrowserDialog",args,1);
			XtAddCallback(dialog,XmNokCallback,(XtCallbackProc) SetAmmpBrowser,NULL);
			XtAddCallback(dialog,XmNcancelCallback,(XtCallbackProc) dscDestroyWidget,dialog);
		}
		XtManageChild(dialog);
		XtPopup(XtParent(dialog),XtGrabNone);
}


/*******************************************************************************
 * SetAmmpBrowser() - Callback from GetAmmpBrowser().  Grab filename for       *
 *                    FileSelectionDialog and copy into AMMP_BROWSER variable. *  
 *                    Then call GetOnlineHelp() to laucnh this browser with    *
 *                    AMMP Help pages                                          *
 *                                                                             *
 * Input:                                                                      *
 *     Widget parent - parent widget                                           *
 *     XtPointer clientData - pointer to possible information                  *
 *     XmFileSelectionBoxCallbackStruct *cbs - pointer to FileSelection struct * 
 ******************************************************************************/
void SetAmmpBrowser(parent, clientData, cbs)
Widget parent;
XtPointer clientData;
XmFileSelectionBoxCallbackStruct *cbs;
{
	char *filename;

	XtPopdown(GetTopShell(parent));
	XmStringGetLtoR(cbs->value,XmFONTLIST_DEFAULT_TAG,&filename);
	AMMP_BROWSER = (char *) malloc (256 * sizeof(char));
	strcpy(AMMP_BROWSER,filename);
	GetOnlineHelp();
}


/*****************************************************************************
 * void CloseGraphicsWindow() - close the graphics window  (i.e. kill        *
 *                              graphics process                             *
 *****************************************************************************/
void CloseGraphicsWindow()
{
	exit(0);
}


/*****************************************************************************
 * Exit() - kill graphics window and exit ammp                               *
 *****************************************************************************/
void Exit()  
{
	char *string = "exit;";
	WriteToAmmp(string,strlen(string));
	CloseGraphicsWindow();
}


/*****************************************************************************
 * GetTopShell() - search for higher level widget given another widget       *
 *                                                                           *
 * Input:                                                                    *
 *     Widget widget - lower level widget                                    *
 * Return:                                                                   *
 *     Widget - higher level widget                                          *
 *****************************************************************************/
Widget GetTopShell(Widget widget)
{
	while (widget && !XtIsSubclass(widget,xmDialogShellWidgetClass))
	{
		widget = XtParent(widget);
	}
	return (widget);
}


/*****************************************************************************
 * CreateLabeledTextField() - add a labeled text field to a parent widget.   *
 *                            align the label on the left-side of the text   *
 *                            field.                                         *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     char *text - text for label adjacent to text field                    *
 *     char *default_value - default value for text field                    *
 * Return:                                                                   *
 *     Widget - labeled text field widget                                    *
 *****************************************************************************/
Widget CreateLabeledTextField(parent, text, default_value)
Widget parent;
char *text;
char *default_value;
{
	Widget form, label, field;
	Arg args[5];
	int num_args;

	form = XtVaCreateWidget(NULL,xmFormWidgetClass,parent,
							XmNorientation, XmHORIZONTAL, NULL);
	XtSetArg(args[0],XmNleftAttachment, XmATTACH_FORM);
	XtSetArg(args[1],XmNtopAttachment,XmATTACH_FORM);
	XtSetArg(args[2],XmNbottomAttachment,XmATTACH_FORM);
	label = XtCreateManagedWidget(text,xmLabelGadgetClass,form,args,3);

	if (default_value != NULL)  
	{
		XtSetArg(args[4],XmNvalue,default_value);
		num_args = 5;
	}
	else  num_args = 4;
	XtSetArg(args[0],XmNleftAttachment,XmATTACH_WIDGET);
	XtSetArg(args[3],XmNleftWidget,label);
	field = XtCreateManagedWidget("dummy",xmTextFieldWidgetClass,form,args,num_args);

	XtManageChild(form);
	return (field);
}


/*****************************************************************************
 * CreateActionButtons() - Create action buttons (Ok,Cancel, etc...).  The   *
 *                         action area should be composed of multiple        *
 *                         sections.  The arguments left and right define    *
 *                         where to "attach" the button.  For example, if    *
 *                         the action area has five sections and this        *
 *                         function is called for an "OK" button left==0 and *
 *                         right==1 then the action area will look as        *
 *                         follows:                                          *
 *                                                                           * 
 *                         0     1     2     3     4                         *
 *                         | OK  |     |     |     |                         *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     char *name - action button text                                       *
 *     XtCallbackProc callback - callback for button "activation"            *
 *     XtPointer callData - data to pass to callback routine                 *
 *     int left,right - defines location of button (see above for info.)     *
 *****************************************************************************/
void CreateActionButtons(parent,name,callback,callData,left,right)
Widget parent;
char *name;
XtCallbackProc callback;
XtPointer callData;
int left,right;
{
	Widget pb;
	pb = XtVaCreateManagedWidget(name,xmPushButtonGadgetClass,parent,
	                              XmNshowAsDefault, TRUE, 
	                              XmNrightPosition, right,
	                              XmNleftPosition, left,
								  XmNtopAttachment, XmATTACH_FORM,
								  XmNbottomAttachment, XmATTACH_FORM,
								  XmNleftAttachment, XmATTACH_POSITION,
								  XmNrightAttachment, XmATTACH_POSITION,
								  XmNdefaultButtonShadowThickness, 1,
								  NULL);
	if (callback != NULL)
		XtAddCallback(pb,XmNactivateCallback,(XtCallbackProc) callback, callData);
}



/*****************************************************************************
 * AnalyzeCallback() - display a custom dialog to analyze the molecule       *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *****************************************************************************/
void AnalyzeCallback(parent)
Widget parent;
{
	static Widget dialog = NULL;
    Widget pb,form,pane,start,end;
    Widget *widget_list;
    Arg args[3];
    XmString title;

	if (!dialog)
	{
		widget_list = (Widget *) malloc (2 * sizeof(Widget));
		XtSetArg(args[0],XmNvisual, vi->visual);
		XtSetArg(args[1],XmNautoUnmanage, False);
		dialog = XmCreateFormDialog(parent,"dialog",args,2);
		pane = XtVaCreateManagedWidget("pane",xmPanedWindowWidgetClass,dialog,XmNsashWidth,1,XmNsashHeight,1,NULL);
		form = XtVaCreateManagedWidget("upper_form",xmRowColumnWidgetClass,pane, NULL);

		title = XmStringCreateSimple("Analyze Range of Atoms");
		XtVaCreateManagedWidget("label",xmLabelGadgetClass,form,XmNlabelString,title,XmNmarginLeft,80,NULL);
		XmStringFree(title);

		/* Prompt user for start of analysis range */ 
		start = CreateLabeledTextField(form,"Start Atom: ",NULL);

		/* Prompt user the end of analysis range */
		end = CreateLabeledTextField(form,"Last Atom:  ",NULL);

		/* Create action area with OK, Clear, and Cancel buttons */
		form = XtVaCreateManagedWidget("lower_form",xmFormWidgetClass,pane, XmNfractionBase, 5, NULL);
		CreateActionButtons(form,"Ok",(XtCallbackProc) Analyze, (XtPointer) widget_list,0,1);
		/* CreateActionButtons(form,"Clear",NULL,NULL,2,3); */
		CreateActionButtons(form,"Cancel",(XtCallbackProc) dscDestroyWidget, (XtPointer) dialog,4,5);
		widget_list[0] = start;
		widget_list[1] = end;
	}
	XtManageChild(dialog);
	XtPopup(XtParent(dialog),XtGrabNone);

}


/*******************************************************************************
 * GrammpError() - pop-up an error dialog with a message and then exit grammp  *
 *    if (mode == FATAL) then force user to exit grammp (remove cancel button) *
 *    if (mode == NON_FATAL) then allow user to exit or ignore error           *
 *                                                                             *
 * Input:                                                                      *
 *     char *message - message to be displayed                                 *
 *     int mode - error mode (see above for more info.)                        *
 *******************************************************************************/
void GrammpError(char *message, int mode)
{
	static Widget dialog = NULL;
	XmString text;
	XmString exit_lbl;
	XmString ignore_lbl;
	char string [1000];
	Arg args[4];
	int num_args;

	exit_lbl = XmStringCreateSimple("Exit");
	ignore_lbl = XmStringCreateSimple("Ignore");
	XtSetArg(args[0], XmNvisual, vi->visual);
	if (mode == FATAL)  
	{
		sprintf(string,"FATAL ERROR: %s\n",message);
		text = XmStringCreateLtoR(string,XmFONTLIST_DEFAULT_TAG);
		XtSetArg(args[1], XmNmessageString, text);
		XtSetArg(args[2], XmNokLabelString, exit_lbl);
		dialog = XmCreateErrorDialog(Glxarea,"GrammpError",args,3);
		XtAddCallback(dialog,XmNokCallback,(XtCallbackProc) Exit,NULL);
		XtUnmanageChild(XmMessageBoxGetChild(dialog,XmDIALOG_CANCEL_BUTTON));
	}
	else  
	{
		sprintf(string,"NON-FATAL ERROR: %s\n",message);
		text = XmStringCreateLtoR(string,XmFONTLIST_DEFAULT_TAG);
		XtSetArg(args[1], XmNmessageString, text);
		XtSetArg(args[2], XmNokLabelString, exit_lbl);
		XtSetArg(args[3], XmNcancelLabelString, ignore_lbl);
		XtSetArg(args[4], XmNdefaultButtonType, XmDIALOG_CANCEL_BUTTON);
		dialog = XmCreateErrorDialog(Glxarea,"GrammpError",args,5);
		XtAddCallback(dialog,XmNokCallback,(XtCallbackProc) Exit,NULL);
		XtAddCallback(dialog,XmNcancelCallback,(XtCallbackProc) dscDestroyWidget,dialog);
	}
	XtUnmanageChild(XmMessageBoxGetChild(dialog,XmDIALOG_HELP_BUTTON));
	XtManageChild(dialog);
	XtPopup(XtParent(dialog),XtGrabNone);
}


/******************************************************************************
 * GetInputFileName() - callback from LoadCallback().  parse the file         *
 *                      selection box for a filename.  tell ammp to open      *
 *                      this file.                                            *
 *                                                                            *
 * Input:                                                                     *
 *     Widget parent - parent widget                                          *
 *     XtPointer clientData - pointer to data (this is a dummy variable)      *
 *     XmFileSelectionBoxCallbackStruct *cbs - pointer to file selection data *
 ******************************************************************************/
void GetInputFileName(w, clientData, cbs)
Widget w;
XtPointer clientData;
XmFileSelectionBoxCallbackStruct *cbs;
{
	char *filename;
    char string[256];

	XtUnmanageChild(w);
	XmStringGetLtoR(cbs->value,XmFONTLIST_DEFAULT_TAG,&filename);
	sprintf(string,"echo off; read %s; echo on;",filename);
	WriteToAmmp(string,strlen(string));
	XtFree(filename);
}


/******************************************************************************
 * GetOutputFileName() - callback from OutputCallback().  parse the file      *
 *                       selection box for a filename.  tell ammp to dump     *
 *                       ammp values into this file                           *
 *                                                                            *
 * Input:                                                                     *
 *     Widget parent - parent widget                                          *
 *     XtPointer clientData - pointer to data (this is a dummy variable)      *
 *     XmFileSelectionBoxCallbackStruct *cbs - pointer to file selection data *
 ******************************************************************************/
void GetOutputFileName(w, clientData, cbs)
Widget w;
XtPointer clientData;
XmFileSelectionBoxCallbackStruct *cbs;
{
    char *filename;
    char string[256];

    XtUnmanageChild(w);
    XmStringGetLtoR(cbs->value,XmFONTLIST_DEFAULT_TAG,&filename);
    sprintf(string,"echo off; output %s; dump atom bond angle hybrid torsion nonbon;",filename);
    WriteToAmmp(string,strlen(string));
    XtFree(filename);
}


/*****************************************************************************
 * LoadCallback() - callback from File menu - open a file selection dialog   *
 *                  box to select an input ammp file.                        *
 *                                                                            *
 * Input:                                                                     *
 *     Widget parent - parent widget                                          *
 *****************************************************************************/
void LoadCallback (Widget w)
{
    static Widget dialog = NULL;
    Arg args[2];
    XmString pattern = XmStringCreateSimple("*.ammp");

    if (!dialog)  
    {
		XtSetArg(args[0], XmNpattern,pattern);
		XtSetArg(args[1], XmNvisual, vi->visual);
		dialog = XmCreateFileSelectionDialog(w,"openFileDialog",args,2);
		XtAddCallback(dialog,XmNokCallback,(XtCallbackProc) GetInputFileName,NULL);
		XtAddCallback(dialog,XmNcancelCallback,(XtCallbackProc) dscDestroyWidget,dialog);
	}
    XtManageChild(dialog);
}


/*****************************************************************************
 * OutputCallback() - callback from File menu - open a file selection dialog *
 *                    box to select an output ammp file                      *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *****************************************************************************/
void OutputCallback (Widget w,XtPointer clientData, XtPointer calldata)  
{
    static Widget dialog = NULL;
    Arg args[2];
    XmString pattern = XmStringCreateSimple("*.ammp");

    if (!dialog)  
    {
		XtSetArg(args[0], XmNpattern,pattern);
		XtSetArg(args[1], XmNvisual, vi->visual);
		dialog = XmCreateFileSelectionDialog(w,"openFileDialog",args,2);
		XtAddCallback(dialog,XmNokCallback,(XtCallbackProc) GetOutputFileName,NULL);
		XtAddCallback(dialog,XmNcancelCallback,(XtCallbackProc) dscDestroyWidget,dialog);
	}
    XtManageChild(dialog);
}


/*****************************************************************************
 * ResizeScene() - Redraw the scene when the X Window is resized.  Get the   *
 *                 current width and height and then change the Viewport     *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *****************************************************************************/
void ResizeScene(Widget parent)
{
    Dimension width, height;
    float min[3],max[3];

	/* resize the window only if it has been created */
    if (made_current)
    {
        XtVaGetValues(Glxarea, XmNwidth, &width, XmNheight, &height, NULL);
        draw_winx = width; draw_winy = height;

        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
        glViewport(0, 0, draw_winx, draw_winy);
    }
}


/*****************************************************************************
 * LabelsOffCallback() - remove atom, distance, angle, and dihedral labels   *
 *                       from the screen                                     *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *****************************************************************************/
void LabelsOffCallback(Widget parent)
{
    draw_ATOM *ap;
    list_link *lp;
    int natom;

    lp = first_draw_ATOM;
    if (lp == NULL) return ;
    while(TRUE)
    {
        ap = lp->what;
        ap->ispicked = FALSE;
        if( lp == last_draw_ATOM ) break;
        if( lp->next == lp) break;
        if( lp->next == NULL ) break;
        lp = lp->next;
    }
    PickList[DISTANCE].show = FALSE;
    PickList[DISTANCE].num = 0;
    PickList[ANGLE].show = FALSE;
    PickList[ANGLE].num = 0;
    PickList[DIHEDRAL].show = FALSE;
    PickList[DIHEDRAL].num = 0;
    draw_walker();
}


/*****************************************************************************
 * void SetMolecularColor() - Set the value of the MolecularColor.  This     *
 *                            variable determines the coloring scheme of the *
 *                            displayed molecule.                            *
 *                                                                           *
 * Input:                                                                    *
 *     Widget parent - parent widget                                         *
 *     char *color - new color                                               *
 *****************************************************************************/
void SetMolecularColor(parent, color)

Widget parent;
char *color;
{
    strcpy(MolecularColor,color);
    draw_walker();
}
