
/*************************************************
**** Universal Multithreading Control Library ****
****    (c) 2003-2012, Alessandro Pedretti    ****
*************************************************/


#include "ammp.h"
#include "mthread.h"

#ifdef MCL_PTHREAD
#  ifndef __USE_XOPEN2K
#    define  __USE_XOPEN2K
#  endif
#  include <sys/types.h>
#  include <pthread.h>

typedef struct {
  int                   InitCount;      /* Initial count value                */
  pthread_barrier_t     hEvent;         /* Handle of the event                */
} MCL_BAR;

static pthread_mutex_t  MthH;
static pthread_cond_t * StartEvent = NULL;
#endif

#ifdef MCL_LINUX
#  include <stdio.h>
#endif

#ifdef MCL_SGI
#  include <unistd.h>
#  include <invent.h>
#  include <abi_mutex.h>
#  include <ulocks.h>
#  include <sys/prctl.h>
#  include <sys/types.h>
#  include <sys/wait.h>

typedef struct {
  int                   InitCount;      /* Initial count value                */
  barrier_t *           hEvent;         /* Handle of the events               */
} MCL_BAR;

static abilock_t        MthH;
static barrier_t        *BarrierH;
static int              ThreadsToWait;
static usptr_t          *UsH;
#endif

#ifdef MCL_SUN
#  include <unistd.h>
#endif

#ifdef MCL_WIN32
#  include <windows.h>

typedef struct {
  int                   InitCount;      /* Initial count value                */
  int                   Ndx;            /* Event index                        */
  int                   WaitCount;      /* Wait count                         */
  HANDLE                hEvent[2];      /* Handle of the events               */
} MCL_BAR;

static HANDLE           MthH       = NULL;
static HANDLE           StartEvent = NULL;
#else
static int              MthH_Init  = FALSE;
#endif

/**** Local prototypes ****/

static int              MCL_BurstThreadLoop(MCL_BURSTTHREAD hCtx);


/**** Close mutex ****/

void AMMP_FASTCALL MCL_CloseMutex(void)
{
#ifdef MCL_PTHREAD
  if (MthH_Init) {
    pthread_mutex_destroy(&MthH);
    MthH_Init = FALSE;
  }
#endif

#ifdef MCL_SGI
  if (MthH_Init) {
    release_lock(&MthH);
    MthH_Init = FALSE;
  }
#endif

#ifdef WIN32
  if (!MthH) return;
  CloseHandle(MthH);
  MthH = NULL;
#endif
}


/**** Create mutex ****/

int AMMP_FASTCALL MCL_CreateMutex(void)
{

#ifdef MCL_PTHREAD
  pthread_mutex_init(&MthH, NULL);
  MthH_Init = TRUE;
  return TRUE;
#endif

#ifdef sgi
  if (init_lock(&MthH)) return FALSE;
  MthH_Init = TRUE;
  return TRUE;
#endif

#ifdef WIN32
  return ((MthH = CreateMutex(NULL, FALSE, NULL)) != NULL);
#endif
}


/**** Creates parallel threads ****/

int AMMP_FASTCALL MCL_CreateThreads(int Cpu, int (*Routine)(void *), void *Args, int Flags)
{
  MCL_BEGIN
  MCL_END

  int                           Err = 0;

#ifdef MCL_PTHREAD
  int                           k, ThreadId;
  pthread_t                     ThreadHandle[MCL_MAX_CPU];
#endif

#ifdef MCL_SGI
  int                           k, WaitAddr;
  int                           ThreadHandle[MCL_MAX_CPU];
#endif

#ifdef MCL_WIN32
  LPTHREAD_START_ROUTINE        MyRoutine = (LPTHREAD_START_ROUTINE)Routine;
  DWORD                         ThreadId;
  int                           k;
  HANDLE                        ThreadHandle[MCL_MAX_CPU];
#else
  int                           (*MyRoutine)(void *) = (void *)Routine;
#endif


#ifdef WIN32
  if ((!MthH) && (!MCL_CreateMutex())) return TRUE;
#else
  if ((!MthH_Init) && (!MCL_CreateMutex())) return TRUE;
#endif

  if (!Cpu) Cpu = GetThreads();
  ((MCL_GLOBVARS *)Args) -> ThreadID = 0;
  ((MCL_GLOBVARS *)Args) -> Threads  = Cpu;
  if (Cpu > 1) {

#ifdef MCL_PTHREAD
    for(k = 0; k < Cpu; ++k) {
      if ((ThreadId = pthread_create(&ThreadHandle[k], NULL, (void *(*)(void *))Routine, Args)) != 0) {
        return 1;
      }
    } /* End of for */

    if (Flags & MCL_THREADS_WAIT) {
      for(k = 0; k < Cpu; ++k) Err |= pthread_join(ThreadHandle[k], NULL);
    }

#endif

#ifdef MCL_SGI
    BarrierH      = NULL;
    ThreadsToWait = 0;
    UsH           = NULL;
    if (Flags & MCL_THREADS_SYNC) {
      UsH = usinit("dumb1");
      BarrierH = new_barrier(UsH);
      init_barrier(BarrierH);
      ThreadsToWait = Cpu;
    }

    for(k = 0; k < Cpu; ++k) {
      if ((ThreadHandle[k] = sproc((void (*)(void *))Routine, PR_SADDR, Args)) == 0) {
        return 1;
      }
    } /* End of for */

    if (Flags & MCL_THREADS_WAIT) while(wait(&WaitAddr) > 0);

#endif

#ifdef MCL_WIN32
    if (Flags & MCL_THREADS_SYNC) {
      if ((StartEvent = CreateEvent(NULL, TRUE, FALSE, NULL)) == NULL)
        return 1;
      ResetEvent(StartEvent);
    }
    for(k = 0; k < Cpu; ++k) {
      if ((ThreadHandle[k] = CreateThread(NULL,
                                          0,
                                          (LPTHREAD_START_ROUTINE)Routine,
                                          Args, 0, &ThreadId)) == NULL) {
        return 1;
      }
    } /* End of for */
    if (Flags & MCL_THREADS_SYNC)
      SetEvent(StartEvent);
    if (Flags & MCL_THREADS_WAIT) {
      WaitForMultipleObjects(Cpu, ThreadHandle, TRUE, INFINITE);
      for(k = 0; k < Cpu; ++k) {
        GetExitCodeThread(ThreadHandle[k], &ThreadId);
        Err |= ThreadId;
        CloseHandle(ThreadHandle[k]);
      } /* End of for */
    }
#endif

  } else Err = MyRoutine(Args);

  return Err;
}


/**** Get the number of CPUs ****/

int AMMP_FASTCALL MCL_GetCPUs(void)
{
#ifdef MCL_WIN32
  SYSTEM_INFO   SysInfo;
#endif

  if (MthThreads) return MthThreads;

#ifdef MCL_WIN32
  ZeroMemory(&SysInfo, sizeof(SYSTEM_INFO));
  GetSystemInfo(&SysInfo);

  MthThreads = SysInfo.dwNumberOfProcessors;
#else

#  ifdef MCL_SGI
  MthThreads = sysconf(_SC_NPROC_ONLN);
#  else

#    ifdef MCL_SUN
  return sysconf(_SC_NPROCESSORS_ONLN);
#    else
#      ifdef MCL_LINUX
  FILE          *FH;
  int           Cpus = 1;

  if ((FH = popen("grep '^processor' /proc/cpuinfo| wc -l", "r")) != NULL) {
    fscanf(FH, "%d", &Cpus);
    if (Cpus < 1) Cpus = 1;
    pclose(FH);
  }

  MthThreads = Cpus;
#      else
  MthThreads = MCL_MAX_CPU;
#      endif
#    endif
#  endif
#endif

  return MthThreads;
}


/**** Disable mutex ****/

void AMMP_FASTCALL MCL_MutexOff(void)
{
#ifdef MCL_PTHREAD
  pthread_mutex_unlock(&MthH);
#endif

#ifdef MCL_SGI
  release_lock(&MthH);
#endif

#ifdef MCL_WIN32
  if (MthH) ReleaseMutex(MthH);
#endif
}


/**** Activate mutex ****/

void AMMP_FASTCALL MCL_MutexOn(void)
{
#ifdef MCL_PTHREAD
  pthread_mutex_lock(&MthH);
#endif

#ifdef MCL_SGI
  spin_lock(&MthH);
#endif

#ifdef MCL_WIN32
  if (MthH) WaitForSingleObject(MthH, INFINITE);
#endif
}


/**** Wait the start signal ****/

void AMMP_FASTCALL MCL_StartWait(void)
{

#ifdef MCL_SGI
  if (ThreadsToWait) barrier(BarrierH, ThreadsToWait);
#endif

#ifdef MCL_WIN32
  WaitForSingleObject(StartEvent, INFINITE);
#endif
}

/**** Close thread ****/

void AMMP_FASTCALL MCL_ThreadClose(MCL_THREAD ThreadHandle)
{
  if (!ThreadHandle) return;

#ifdef MCL_WIN32
  CloseHandle((HANDLE)ThreadHandle);
#endif
}


/**** Reset the event ****/

void AMMP_FASTCALL MCL_EventClose(MCL_EVENT hEvent)
{
#ifdef MCL_WIN32
  if (hEvent) CloseHandle((HANDLE)hEvent);
#endif
}


/**** Create a new event ****/

MCL_EVENT AMMP_FASTCALL MCL_EventCreate(int ManualReset, int InitialState)
{
#ifdef MCL_WIN32
  return (MCL_EVENT)CreateEvent(NULL, ManualReset, InitialState, NULL);
#endif
}


/**** Reset the event ****/

void AMMP_FASTCALL MCL_EventReset(MCL_EVENT hEvent)
{
#ifdef MCL_WIN32
  ResetEvent((HANDLE)hEvent);
#endif
}


/**** Set the event ****/

void AMMP_FASTCALL MCL_EventSet(MCL_EVENT hEvent)
{
#ifdef MCL_WIN32
  SetEvent((HANDLE)hEvent);
#endif
}


/**** Wait the event ****/

void AMMP_FASTCALL MCL_EventWait(MCL_EVENT hEvent)
{
#ifdef MCL_WIN32
  WaitForSingleObject((HANDLE)hEvent, INFINITE);
#endif
}


/**** Close barrier ****/

void AMMP_FASTCALL MCL_BarrierClose(MCL_BARRIER Bh)
{
#ifndef MCL_AMIGA
  if ((GetThreads() == 1) || (!Bh) || (Bh == (MCL_BARRIER)-1)) return;

#  ifdef MCL_PTHREAD
  pthread_barrier_destroy(&((MCL_BAR *)Bh) -> hEvent);
  free(Bh);
#  endif

#  ifdef MCL_SGI
  free_barrier(((MCL_BAR *)Bh) -> hEvent);
  free(Bh);
#  endif

#  ifdef MCL_WIN32
/*
 * See: http://softwareforums.intel.com/ids/board/message?board.id=42&message.id=599
 */

  if (((MCL_BAR *)Bh) -> hEvent[0]) CloseHandle(((MCL_BAR *)Bh) -> hEvent[0]);
  if (((MCL_BAR *)Bh) -> hEvent[1]) CloseHandle(((MCL_BAR *)Bh) -> hEvent[1]);
  free(Bh);
#  endif
#endif
}


/**** Create barrier ****/

MCL_BARRIER AMMP_FASTCALL MCL_BarrierCreate(int Count)
{
#ifdef MCL_SGI
  int                   k;
#endif

#if defined(MCL_PTHREAD) || defined(MCL_SGI) || defined(MCL_WIN32)
  MCL_BAR *             Bh;
#endif

#ifdef MCL_AMIGA
  return (MCL_BARRIER)-1;
#endif

#if defined(MCL_PTHREAD) || defined(MCL_SGI) || defined(MCL_WIN32)
  if (MthThreads == 1) return (MCL_BARRIER)-1;
  if (!Count) Count = MthThreads;
  if ((Bh = (MCL_BAR *)AllocaC(sizeof(MCL_BAR), "MCL_BarrierCreate()")) == NULL)
    return NULL;

  Bh -> InitCount = Count;
#endif

#ifdef MCL_PTHREAD
  if (pthread_barrier_init(&Bh -> hEvent, NULL, Count))
    return NULL;
  return (MCL_BARRIER)Bh;
#endif

#ifdef MCL_SGI
  if (!UsH) UsH = usinit("dumb1");
  for(k = 0; k < Count; ++k) {
    if (Bh -> hEvent[k] = new_barrier(UsH)) == NULL) {
      return NULL;
    } else init_barrier(Bh -> hEvent[k]);
  } /* End of for (k) */

  return (MCL_BARRIER)Bh;
#endif

#ifdef MCL_WIN32
  Bh -> WaitCount = Count;

  if ((Bh -> hEvent[0] = CreateEvent(NULL, TRUE, FALSE, NULL)) == NULL) {
    free(Bh);
    return NULL;
  }
  if ((Bh -> hEvent[1] = CreateEvent(NULL, TRUE, FALSE, NULL)) == NULL) {
    CloseHandle(Bh -> hEvent[0]);
    free(Bh);
    return NULL;
  }

  return (MCL_BARRIER)Bh;
#endif
}


/**** Wait barrier ****/

void AMMP_FASTCALL MCL_BarrierWait(MCL_BARRIER Bh)
{
#ifdef MCL_WIN32
  int           Ndx;
#endif

#ifndef MCL_AMIGA
  if ((!Bh) || (Bh == (MCL_BARRIER)-1)) return;
#endif

#ifdef MCL_SGI
  if (((MCL_BAR *)Bh) -> InitCount == 1) return;
  barrier(((MCL_BAR *)Bh) -> hEvent, ((MCL_BAR *)Bh) -> InitCount);
#endif

#ifdef MCL_PTHREAD
  if (((MCL_BAR *)Bh) -> InitCount == 1) return;
  pthread_barrier_wait(&((MCL_BAR *)Bh) -> hEvent);
#endif

#ifdef MCL_WIN32
  Ndx = ((MCL_BAR *)Bh) -> Ndx;
  if (InterlockedDecrement((LPLONG)&(((MCL_BAR *)Bh) -> WaitCount)) == 0) {
    ((MCL_BAR *)Bh) -> WaitCount = ((MCL_BAR *)Bh) -> InitCount;
    ((MCL_BAR *)Bh) -> Ndx       = 1 - Ndx;
    ResetEvent(((MCL_BAR *)Bh) -> hEvent[((MCL_BAR *)Bh) -> Ndx]);
    SetEvent(((MCL_BAR *)Bh) -> hEvent[Ndx]);
  } else WaitForSingleObject(((MCL_BAR *)Bh) -> hEvent[Ndx], INFINITE);
#endif
}


/**** Close burst threads ****/

void AMMP_FASTCALL MCL_BurstThreadClose(MCL_BURSTTHREAD hCtx)
{
  if (!hCtx) return;

  if (hCtx -> Flags & MCL_BTFLAG_RUNNING) {
    int           k;

    hCtx -> Flags |= MCL_BTFLAG_DESTROYING;
    MCL_EventSet(hCtx -> hStart);
    MCL_EventWait(hCtx -> hEnd);

    for(k = 0; k < hCtx -> NumThreads; ++k)
      MCL_ThreadClose(hCtx -> Threads[k]);
  }
  SafeFree(hCtx -> Threads);
  MCL_BarrierClose(hCtx -> hSyncBar);
  MCL_EventClose(hCtx -> hEnd);
  MCL_EventClose(hCtx -> hStart);
  MCL_MutexClose(hCtx -> hMutex);
  free(hCtx);
}


/**** Create burst threads ****/

MCL_BURSTTHREAD AMMP_FASTCALL MCL_BurstThreadCreate(int Cpus)
{
#ifdef MCL_WIN32
  DWORD                 ThreadId;
#endif

#ifdef MCL_PTHREAD
  int                   ThreadId;
#endif

  int                   k;
  MCL_BURSTTHREAD       hCtx;

  const char *          Routine = "MCL_BurstThreadCreate()";

  if ((hCtx = AllocaC(sizeof(struct __mcl_burstthread), Routine)) == NULL)
    return NULL;

  if (!Cpus) Cpus = GetThreads();

  if ((hCtx -> Threads = AllocaC(sizeof(MCL_THREAD) * Cpus, Routine)) == NULL) {
    free(hCtx);
    return NULL;
  }

  hCtx -> Flags      = MCL_BTFLAG_NONE;
  hCtx -> NumThreads = Cpus;
  hCtx -> Routine    = NULL;
  hCtx -> hEnd       = MCL_EventCreate(FALSE, FALSE);
  hCtx -> hMutex     = MCL_MutexCreate();
  hCtx -> hStart     = MCL_EventCreate(TRUE , FALSE);
  hCtx -> hSyncBar   = MCL_BarrierCreate(Cpus);

  /**** Create the threads ****/

#ifdef MCL_WIN32
  if ((!MthH) && (!MCL_CreateMutex())) {
#else
  if ((!MthH_Init) && (!MCL_CreateMutex())) {
#endif
    MCL_BurstThreadClose(hCtx);
    return NULL;
  }

#ifdef MCL_PTHREAD
    for(k = 0; k < Cpus; ++k) {
      if ((ThreadId = pthread_create(&hCtx -> Threads[k], NULL, (void *(*)(void *))MCL_BurstThreadLoop, hCtx)) != 0) {
        MCL_BurstThreadClose(hCtx);
        return NULL;
      }
    } /* End of for (k) */
#endif

#ifdef MCL_WIN32
    for(k = 0; k < Cpus; ++k) {
      if ((hCtx -> Threads[k] = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)MCL_BurstThreadLoop,
                                             hCtx, 0, &ThreadId)) == NULL) {
        MCL_BurstThreadClose(hCtx);
        return NULL;
      }
    } /* End of for (k) */
#endif

  hCtx -> Flags |= MCL_BTFLAG_RUNNING;

  return hCtx;
}


/**** Burst thread loop ****/

static int MCL_BurstThreadLoop(MCL_BURSTTHREAD hCtx)
{

  for(;;) {
    MCL_EventWait(hCtx -> hStart);
    if ((!(hCtx -> Flags & MCL_BTFLAG_DESTROYING)) &&
        (hCtx -> Routine)) hCtx -> Routine(hCtx -> Args);

    MCL_BarrierWait(hCtx -> hSyncBar);
    MCL_MutexOnEx(hCtx -> hMutex);
    if (hCtx -> SetSignal) {
      MCL_EventReset(hCtx -> hStart);
      MCL_EventSet(hCtx -> hEnd);
      hCtx -> SetSignal = FALSE;
    }
    MCL_MutexOffEx(hCtx -> hMutex);
    if (hCtx -> Flags & MCL_BTFLAG_DESTROYING) break;
  } /* End of for */

  return FALSE;
}


/**** Burst thread run ****/

void AMMP_FASTCALL MCL_BurstThreadRun(MCL_BURSTTHREAD hCtx, int (*Routine)(void *), void *Args)
{
  MCL_BEGIN
  MCL_END
  if (Args) {
    ((MCL_GLOBVARS *)Args) -> ThreadID = 0;
    ((MCL_GLOBVARS *)Args) -> Threads  = hCtx -> NumThreads;
  }

  hCtx -> Args     = Args;
  hCtx -> Routine  = Routine;
  hCtx -> SetSignal = TRUE;
  MCL_EventSet(hCtx -> hStart);
  MCL_EventWait(hCtx -> hEnd);
}


/**** Close mutex ****/

void AMMP_FASTCALL MCL_MutexClose(MCL_MUTEX Mtx)
{
#ifndef MCL_AMIGA
  if ((MthThreads == 1) || (!Mtx) || (Mtx == (MCL_MUTEX)-1)) return;

#  ifdef MCL_PTHREAD
  pthread_mutex_destroy((pthread_mutex_t *)Mtx);
  free(Mtx);
#  endif

#  ifdef MCL_SGI
  release_lock((abilock_t *)Mtx);
  free(Mtx);
#  endif

#  ifdef MCL_WIN32
  CloseHandle((HANDLE)Mtx);
#  endif
#endif
}


/**** Create mutex  ****/

MCL_MUTEX AMMP_FASTCALL MCL_MutexCreate(void)
{
  MCL_MUTEX     NewMutex;


  if (MthThreads == 1) return (void *)-1;

#ifdef MCL_PTHREAD
  if ((NewMutex = (MCL_MUTEX)AllocaC(sizeof(pthread_mutex_t), "MCL_MutexCreate()")) != NULL)
    pthread_mutex_init((pthread_mutex_t *)NewMutex, NULL);
#endif

#ifdef MCL_SGI
  if ((NewMutex = (MCL_MUTEX)AllocaC(sizeof(abilock_t), "MCL_MutexCreate()")) != NULL)
    if (init_lock((abilock_t *)NewMutex)) {
      free(NewMutex);
      NewMutex = NULL;
    }
  }
#endif

#ifdef MCL_WIN32
  NewMutex = (MCL_MUTEX)CreateMutex(NULL, FALSE, NULL);
#endif

  return NewMutex;
}


/**** Disable mutex extended ****/

void AMMP_FASTCALL MCL_MutexOffEx(MCL_MUTEX Mtx)
{
  if (MthThreads == 1) return;

  if (Mtx) {
#ifdef MCL_PTHREAD
    pthread_mutex_unlock((pthread_mutex_t *)Mtx);
#endif

#ifdef MCL_SGI
    release_lock((abilock_t *)Mtx);
#endif

#ifdef MCL_WIN32
    ReleaseMutex((HANDLE)Mtx);
#endif
  }
}


/**** Activate mutex extended ****/

void AMMP_FASTCALL MCL_MutexOnEx(MCL_MUTEX Mtx)
{
  if (MthThreads == 1) return;

  if (Mtx) {
#ifdef MCL_PTHREAD
    pthread_mutex_lock((pthread_mutex_t *)Mtx);
#endif

#ifdef MCL_SGI
    spin_lock((abilock_t *)Mtx);
#endif

#ifdef MCL_WIN32
    WaitForSingleObject((HANDLE)Mtx, INFINITE);
#endif
  }
}
