/* noel.c
*
* collection of routines to service noel potentials
*
* noel  == NOE Length
*
*  stepped bonds
*  noel i j d d- d+ kl kh
*  v =  (r - ( d - d-) )**2 *kl r < d-
*  v =  (r - ( d + d+) )**2 *kh r > d+
*
*  kl,kh can be of any value 
*
* POOP (Poor-mans Object Oriented Programming) using scope rules
*
* these routines hold a data base (in terms of array indeces)
* of noelts, with the associated length and force constant
* These are updateable - unlike bonds which are "permanent"
*
* (this could be table driven but what the hell memories cheap)
*
* the routines for potential value, force and (eventually) second
* derivatives are here also
*
* force and 2nd derivative routines assume zero'd arrays for output
* this allows for parralellization if needed (on a PC?)
*
* forces are symmetric 
*/
/*
*  copyright 1992,1994 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Local variables ****/

static NOEL *   noel_next_np = NULL;


/**** Reset local variables ****/

void AMMP_FASTCALL ResetNoel(void)
{
  noel_next_np = NULL;
}


/**** Dump the noels ****/

void AMMP_FASTCALL dump_noels(FILE *where)
{
  NOEL          *b = noel_first;

  while(b) {
    fprintf(where, "noel %d %d %f %f %f %f %f;\n",
            b -> atom1 -> serial, b -> atom2 -> serial,
            b -> d, b -> dm, b -> dh, b -> km, b -> kh);
    if (b -> next == b) break;
    b = b -> next;
  } /* End of while */
}


/**** Add a new noel ****/

int AMMP_FASTCALL noel(int p1, int p2, float d, float dm, float dh, float km, float kh)
{
  NOEL          *new;

  /**** Get the atom pointers for the two serial numbers ****/

  AMMP_ATOM *   ap1 = a_m_serial(p1);
  AMMP_ATOM *   ap2 = a_m_serial(p2);

  if ((!ap1) || (!ap2)) {
    aaerror("Undefined atom in noel %d %d", p1, p2);
    return FALSE;
  }

  /**** Check to see if a noelt is already defined ****/

  if ((new = noel_first) != NULL) {
    while(1) {
      if (new == NULL) break;
      if (((new -> atom1 == ap1) && (new -> atom2 == ap2)) ||
          ((new -> atom1 == ap2) && (new -> atom2 == ap1))) {
      new -> ismet = TRUE;
      new -> d     = d;
      new -> dm    = dm;
      new -> dh    = dh;
      new -> km    = km;
      new -> kh    = kh;
      return TRUE;
    }
    if (new == new -> next) break;
    new = new -> next;
    } /* End of while */
  }

  if ((new = (NOEL *)Alloca(sizeof(NOEL), "noel()")) == NULL)
    return FALSE;

  /**** Initialize the pointers ****/

  if (noel_first == NULL) noel_first = new;
  if (noel_last  == NULL) noel_last  = new;
  new -> atom1      = ap1;
  new -> atom2      = ap2;
  new -> ismet      = TRUE;
  new -> d          = d;
  new -> dm         = dm;
  new -> dh         = dh;
  new -> km         = km;
  new -> kh         = kh;
  new -> next       = new;
  noel_last -> next = new;
  noel_last         = new;

  return TRUE;
}


/**** Set the noels to the current geometry ****/

int AMMP_FASTCALL noel_generate(float dm, float dh)
{
  AMMP_ATOM     *ap1,*ap2;
  float         r, t;
  NOEL          *np;

  if (!noel_first) return TRUE;

  np = noel_first;
  while(np != NULL) {
    ap1 = np -> atom1;
    ap2 = np -> atom2;
    if ((ap1 -> active) && (ap2 -> active) &&
        (ap1 -> x) && (ap1 -> y) && (ap1 -> z) &&
        (ap2 -> x) && (ap2 -> y) && (ap2 -> z)) {
       r  = ap1 -> x - ap2 -> x;
       r *= r;
       t  = ap1 -> y - ap2 -> y;
       r += t * t;
       t  = ap1 -> z - ap2 -> z;
       r += t * t;
       np -> d = sqrt(r);
       if (dm > zero) np -> dm = dm;
       if (dh > zero) np -> dh = dh;
    }
    if (np == np -> next) return TRUE;
    np = np -> next;
  } /* End of while (np) */

  return FALSE;
}



/* v_noel()
* this function sums up the potentials
* for the atoms defined in the NOEL data structure.
*/
/* standard returns 0 if error (any) 1 if ok
* V is the potential */

int AMMP_FASTCALL v_noel(float *V, float lambda)
{
	NOEL *bp;
	float r,xt,yt,zt;
	AMMP_ATOM *a1,*a2;


	bp = noel_first;
		 if( bp == NULL ) return 1;
		 while(1)
		 {
	if( bp == NULL) return 0;
	a1 = bp->atom1; a2 = bp->atom2;
	if( a1->active || a2->active ){
	if( lambda == 0.)
	{
	r = (a1->x - a2->x)*(a1->x - a2->x);
	r = r + (a1->y - a2->y)*(a1->y - a2->y);
	r = r + (a1->z - a2->z)*(a1->z - a2->z);
	} else
	{
	xt = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
	yt = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
	zt = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
	r = xt*xt+yt*yt+zt*zt;
	}
	r = sqrt(r);
	bp->ismet = (1==1);
	if( r < bp->d -bp->dm)
	{
	bp->ismet = (1==0);
	r = r - bp->d + bp->dm;
	*V += bp->km * r*r;
	} else if( r > bp->d+ bp->dh) {
	bp->ismet = (1==0);
	r = r - bp->d - bp->dh;
	*V += bp->kh * r*r;
	}
	}
	if( bp == bp->next ) return 1;
	bp = bp->next;
		 }
}
/* f_noel()
*
* f_noel increments the forces in the atom structures by the force
* due to the noel components.  NOTE THE WORD increment.
* the forces should first be zero'd.
* if not then this code will be invalid.  THIS IS DELIBERATE.
* on bigger (and better?) machines the different potential terms
* may be updated at random or in parrellel, if we assume that this routine
* will initialize the forces then we can't do this.
*/

int AMMP_FASTCALL f_noel(float lambda)
/*  returns 0 if error, 1 if OK */
{
	NOEL *bp;
	float r,t,ux,uy,uz;
	AMMP_ATOM *a1,*a2;


	bp = noel_first;
		 if( bp == NULL ) return 1;
		 while(1)
		 {
	if( bp == NULL) return 0;
	a1 = bp->atom1; a2 = bp->atom2;
	if( a1->active || a2->active){
	if( lambda == 0.)
	{
	ux = (a2->x - a1->x);
	uy = (a2->y - a1->y);
	uz = (a2->z - a1->z);
	}else{
	ux = (a2->x -a1->x +lambda*(a2->dx-a1->dx));
	uy = (a2->y -a1->y +lambda*(a2->dy-a1->dy));
	uz = (a2->z -a1->z +lambda*(a2->dz-a1->dz));
	}
	r = ux*ux + uy*uy + uz*uz ;
	 /* watch for FP errors*/
	 if( r <= 1.e-5)
	 { r = 0; ux = 1.; uy = 0.; uz = 0.;}else{
	r = sqrt(r); t = 1/r; ux = ux*t; uy = uy*t; uz = uz*t;
	}
	bp->ismet = (1==1);
	if( r < bp->d -bp->dm)
	{
	bp->ismet = (1==0);
	r = r - bp->d + bp->dm;
	ux = 2*bp->km * r *ux;
	uy = 2*bp->km * r *uy;
	uz = 2*bp->km * r *uz;
	} else if( r > bp->d+ bp->dh) {
	bp->ismet = (1==0);
	r = r - bp->d - bp->dh;
	ux = 2*bp->kh * r *ux;
	uy = 2*bp->kh * r *uy;
	uz = 2*bp->kh * r *uz;
	}else{
	ux = 0.; uy = 0.; uz = 0.;
	}
	if( a1->active){
	a1->fx += ux;
	a1->fy += uy;
	a1->fz += uz;
	}

	if( a2->active) {
	a2->fx -= ux;
	a2->fy -= uy;
	a2->fz -= uz;
	}
	}
	if( bp == bp->next ) return 1;
	bp = bp->next;
		 }
}
/* function get_noel( a1,noeled,10,innoel);
* check the NOELS list for atoms noeled to a1
*/

void AMMP_FASTCALL get_noel(AMMP_ATOM *a1, AMMP_ATOM *noeled[], int mnoel, int *innoel)
{
	NOEL *mine;
	mine = noel_first;
	*innoel = 0;
	while(1)
	{
	if( (mine == NULL) )
	{
		return;
	}
	if( mine->atom1 == a1)
	{
		noeled[(*innoel)++] = mine->atom2;
	}
	if( mine->atom2 == a1)
	{
		noeled[(*innoel)++] = mine->atom1;
	}
	if( mine == mine->next) return;
	mine = mine->next;
	if( *innoel == mnoel ) return;
	}
}

/* function get_noel( a1,noeled,10,innoel);
* check the NOELS list for atoms noeled to a1
*/

void AMMP_FASTCALL get_noel_and_length(AMMP_ATOM *a1, AMMP_ATOM *noeled[], float r[],
                                       int mnoel, int *innoel)
{
	NOEL *mine;
	mine = noel_first;
	*innoel = 0;
	while(1)
	{
	if( (mine == NULL) )
	{
		return;
	}
	if( mine->atom1 == a1)
	{
		r[*innoel] = mine->d;
		noeled[(*innoel)++] = mine->atom2;
	}
	if( mine->atom2 == a1)
	{
		r[*innoel] = mine->d;
		noeled[(*innoel)++] = mine->atom1;
	}
	if( mine == mine->next) return;
	mine = mine->next;
	if( *innoel == mnoel ) return;
	}
}

void AMMP_FASTCALL get_noel_and_bounds(AMMP_ATOM *a1, AMMP_ATOM *noeled[], float r[], float rmax[], int mnoel, int *innoel)
{
	NOEL *mine;
	mine = noel_first;
	*innoel = 0;
	while(1)
	{
	if( (mine == NULL) )
	{
		return;
	}
	if( mine->atom1 == a1)
	{
		r[*innoel] = mine->d-mine->dm;
		rmax[*innoel] = mine->d+mine->dh;
		noeled[(*innoel)++] = mine->atom2;
	}
	if( mine->atom2 == a1)
	{
		r[*innoel] = mine->d-mine->dm;
		rmax[*innoel] = mine->d+mine->dh;
		noeled[(*innoel)++] = mine->atom1;
	}
	if( mine == mine->next) return;
	mine = mine->next;
	if( *innoel == mnoel ) return;
	}
}

/* noel_next()
*  like bond_next() but for NOEl structures
*/

int AMMP_FASTCALL noel_next(int i, AMMP_ATOM **n1, AMMP_ATOM **n2, int *ismet)
{
  *n1 = NULL;
  *n2 = NULL;

  if (noel_first == NULL) return FALSE;
  if ((noel_next_np == NULL) || (i <= 0)) {
    noel_next_np = noel_first;
  } else {
    noel_next_np = noel_next_np -> next;
  }
  *n1    = noel_next_np -> atom1;
  *n2    = noel_next_np -> atom2;
  *ismet = noel_next_np -> ismet;
  if (noel_next_np -> next != noel_next_np) return TRUE;

  return FALSE;
}


/* a_noel()
* this function sums up the potentials
* for the atoms defined in the NOEL data structure.
*/
/* standard returns 0 if error (any) 1 if ok
* V is the potential */

int AMMP_FASTCALL a_noel(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
	NOEL *bp;
	float r,xt,yt,zt;
	AMMP_ATOM *a1,*a2;


	bp = noel_first;
		 if( bp == NULL ) return 1;
		 while(1)
		 {
	if( bp == NULL) return 0;
	a1 = bp->atom1; a2 = bp->atom2;
	if(( a1->serial >= ilow && a1->serial <=ihigh)
	 ||( a2->serial >= ilow && a2->serial <=ihigh))
	{
	if( lambda == 0.)
	{
	r = (a1->x - a2->x)*(a1->x - a2->x);
	r = r + (a1->y - a2->y)*(a1->y - a2->y);
	r = r + (a1->z - a2->z)*(a1->z - a2->z);
	} else
	{
	xt = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
	yt = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
	zt = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
	r = xt*xt+yt*yt+zt*zt;
	}
	r = sqrt(r); 
	zt = 0;
	if( r < bp->d -bp->dm)
	zt= bp->km*( r - bp->d+ bp->dm)*(r - bp->d+ bp->dm);
	if( r > bp->d +bp->dh)
	zt= bp->kh*( r - bp->d- bp->dh)*(r - bp->d- bp->dh);
	*V += zt;
	fprintf(op,"NOEl %s %d %s %d E %f value %f error %f\n"
	,a1->name,a1->serial,a2->name,a2->serial,zt,r,r-bp->d);
	}
	if( bp == bp->next ) return 1;
	bp = bp->next;
       }
}
/* gsdg_noel( AMMP_ATOM *ap )
*  
* setup the distances for NOEL terms
*/

void AMMP_FASTCALL gsdg_noel(AMMP_ATOM *ap)
{
	AMMP_ATOM *bp;
	NOEL *np;

	np = noel_first;
	while(1)
	{ if( np == NULL ) return;
	if( np->atom1 == ap )
	{  bp = np->atom2; bp->vx = (np->d*np->d );
	   bp->vy = np->km; }
	if( np->atom2 == ap )
	{  bp = np->atom1; bp->vx = (np->d*np->d ); 
	bp->vy = np->km; }

	if( np == np->next ) return;
	np = np->next;
	}
}


/**** Increment the noel force (homotopy version) ****/

int AMMP_FASTCALL f_ho_noel(float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         r, ux, uy, uz;
  float         hol;

  NOEL          *bp = noel_first;

  if (!bp) return TRUE;

  hol = GetLambda();

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    if ((a1 -> active) || (a2 -> active)) {
      if (lambda) {
        ux = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
        uy = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
        uz = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
      } else {
        ux = a2 -> x - a1 -> x;
        uy = a2 -> y - a1 -> y;
        uz = a2 -> z - a1 -> z;
      }

      r = ux * ux + uy * uy + uz * uz;
      bp -> ismet = TRUE;
      if (r <= 1.e-5f) {
        r  = zero;
        ux = one;
        uy = zero;
        uz = zero;
      } else {
	r   = sqrt(r);
        ux /= r;
        uy /= r;
        uz /= r;
      }
      if (r < (bp -> d - bp -> dm)) {
        bp -> ismet = FALSE;
        r   = two * bp -> km * (r - (hol * r + (one - hol) * (bp -> d - bp -> dm))) * (one - hol);
        ux *= r;
        uy *= r;
        uz *= r;
      } else if (r > (bp -> d + bp -> dh)) {
        bp -> ismet = FALSE;
	r   = two * bp -> kh * (r - (hol * r + (one - hol) * (bp -> d + bp -> dh))) * (one - hol);
        ux *= r;
        uy *= r;
        uz *= r;
      } else {
        ux = zero;
        uy = zero;
        uz = zero;
      }

      if(a1 -> active) {
        a1 -> fx += ux;
        a1 -> fy += uy;
        a1 -> fz += uz;
      }

      if (a2 -> active) {
        a2 -> fx -= ux;
        a2 -> fy -= uy;
        a2 -> fz -= uz;
      }
    }
    if (bp == bp -> next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}


/**** Calculate the noel potential (homotopy version) ****/

int AMMP_FASTCALL v_ho_noel(float *V, float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         r, xt, yt, zt;
  float         hol;

  NOEL          *bp = noel_first;

  if (!bp) return TRUE;

  hol = GetLambda();

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    if ((a1 -> active) || (a2 -> active)) {
      if (lambda) {
	xt = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
	yt = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
	zt = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
      } else {
	xt = a1 -> x - a2 -> x;
	yt = a1 -> y - a2 -> y;
	zt = a1 -> z - a2 -> z;
      }
      r = sqrt(xt * xt + yt * yt + zt * zt);
      bp -> ismet = TRUE;

      if (r < (bp -> d - bp -> dm)) {
        bp -> ismet = FALSE;
	r   = r - (hol * r + (one - hol) * (bp -> d - bp -> dm));
	*V += bp -> km * r * r;
      } else if (r > (bp -> d + bp -> dh)) {
        bp -> ismet = FALSE;
	r   = r - (hol * r + (one - hol) * (bp -> d + bp -> dh));
	*V += bp -> kh * r * r;
      }
    }
    if (bp == bp -> next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}

