/*  normal.c
*
*   calculate the normal modes of a molecule
*   this is best for small molecules
*
*  use finite differences for d2V/dxdy
*  use jacobi method to solve the equations
*
*  report the spectrum and modes, but what to do with them?
*/
/*
*  copyright 1993,1994,1995 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify
*  this notice
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

#define  STEP           1.e-6
#define  DSTEP          one / (STEP + STEP)


int AMMP_FASTCALL FDnormal(AMMP_FFUNC forces[], int nfs, int echo, FILE *op, float normal)
{
  AMMP_ATOM     *ap, *bp;
  char          resid[5], atid[5], *np;
  float         *norm, *vect;
  float         x;
  int           i, j, ii, numatm;
  int           iindex, jindex;
  int           iforce, iatom;

  const char *  Routine = "FDnormal()";
  const char *  PdbRem  = "REMARK   8 ";

  if (!CheckAtoms()) return FALSE;

  /**** Don't if there are no potentials ****/

  if (nfs < 1) return FALSE;

  /**** Get the space ****/

  numatm = a_number();
  if (numatm < 2) return FALSE;
  i    = 3 * numatm;
  j    = i * i;
  if (((norm = (float *)Alloca(j * sizeof(float), Routine)) == NULL) ||
      ((vect = (float *)Alloca(j * sizeof(float), Routine)) == NULL)) {
    SafeFree(norm);
    return FALSE;
  }

  /*  now do the finite differences */
  bp = a_next(-1); /* bp loops over the atoms with iatom */
                   /* bp cannot be done with a_next because ap will */
  a_d_zero();
  for(iatom = 0; iatom < numatm; iatom++) {
    a_f_zero();
    bp -> dx = 1.0f;
    for(iforce = 0; iforce < nfs; iforce++)
      (*forces[iforce])(STEP);
    a_ftogx(1.0f, 0.0f);
    a_f_zero();
    for(iforce = 0; iforce < nfs; iforce++)
      (*forces[iforce])(-STEP);
    a_ftogx(-1.0f, 1.0f);
    ap = a_next(-1);
    for(i = 0; i < numatm; i++) {
      j           = (iatom * 3) * 3 * numatm + i * 3;
      norm[j    ] = -0.5f * ap -> gx * DSTEP;
      norm[j + 1] = -0.5f * ap -> gy * DSTEP;
      norm[j + 2] = -0.5f * ap -> gz * DSTEP;
      ap = ap -> next;
    } /* End of for (i) */
    bp -> dx = 0.0f;
    a_f_zero();
    bp -> dy = 1.0f;
    for(iforce = 0; iforce < nfs; iforce++)
      (*forces[iforce])(STEP);
    a_ftogx(1.0f, 0.0f);
    a_f_zero();
    for(iforce = 0; iforce < nfs; iforce++)
      (*forces[iforce])(-STEP);
    a_ftogx(-1.0f, 1.0f);
    ap = a_next(-1);
    for(i = 0; i < numatm; i++) {
      j           = (iatom * 3 + 1 ) * 3 * numatm + i * 3;
      norm[j    ] = -0.5f * ap -> gx * DSTEP;
      norm[j + 1] = -0.5f * ap -> gy * DSTEP;
      norm[j + 2] = -0.5f * ap -> gz * DSTEP;
      ap = ap -> next;
    } /* End of for (j) */
    bp -> dy = 0.0f;
    a_f_zero();
    bp -> dz = 1.0f;
    for(iforce = 0; iforce < nfs; iforce++)
      (*forces[iforce])(STEP);
    a_ftogx(1.0f, 0.0f);
    a_f_zero();
    for(iforce = 0; iforce < nfs; iforce++)
      (*forces[iforce])(-STEP);
    a_ftogx(-1.0f, 1.0f);
    ap = a_next(-1);
    for(i = 0; i < numatm; i++) {
      j =  (iatom * 3 + 2) * 3 * numatm + i * 3;
      norm[j    ] = -0.5f * ap -> gx * DSTEP;
      norm[j + 1] = -0.5f * ap -> gy * DSTEP;
      norm[j + 2] = -0.5f * ap -> gz * DSTEP;
      ap = ap -> next;
    } /* End of for (i) */
    bp -> dz = 0.0f;
    bp       = bp -> next;
  }

  /**** Now symmetrize the matrix ****/

  for(i = 0; i < (numatm * 3); i++) {
    for(j = i; j < (numatm * 3); j++) {
      iindex        = i * 3 * numatm + j;
      jindex        = j * 3 * numatm + i;
      x             = norm[iindex] + norm[jindex];
      x            *= 0.5f;
      norm[iindex]  = x;
      norm[jindex]  = x;
    } /* End of for (j) */
  } /* End of for (i) */

  /**** Force the diagonal to be the sum of the others ****/

  for(i = 0; i < (numatm * 3); i++) {
    x = 0.0f;
    for(j = 0; j < i; j++) {
      jindex  = i * 3 * numatm + j;
      x      += norm[jindex];
    }
    for(j = i + 1; j < (numatm * 3); j++) {
      jindex  = i * 3 * numatm + j;
      x      += norm[jindex];
    }
    norm[i * 3 * numatm + i] = -x;
  } /* End of for (i) */

  /**** Now mass-weight it ****/

  bp = a_next(-1);
  for(i = 0; i < (numatm * 3); i+=3) {
    for(j = 0; j < numatm; j++) {
      ap     = a_next(j);
      x      = one / sqrt(ap -> mass * bp -> mass);
      iindex = i * 3 * numatm + j * 3;
      norm[iindex    ] *= x;
      norm[iindex + 1] *= x;
      norm[iindex + 2] *= x;
      iindex = i * 3 * numatm + 3 * numatm + j * 3;
      norm[iindex    ] *= x;
      norm[iindex + 1] *= x;
      norm[iindex + 2] *= x;
      iindex = i * 3 * numatm + 6 * numatm + j * 3;
      norm[iindex    ] *= x;
      norm[iindex + 1] *= x;
      norm[iindex + 2] *= x;
    } /* End of for (j) */
    bp = bp->next;
  } /* End of for (i) */

/*
	if( echo )
	{
		fprintf(op,"The mass-weighted Force Matrix \n");
		for( i=0; i< numatm*3; i++)
		{
		for( j=0; j< numatm*3; j++)
		{
			fprintf(op,"%f ",(*norm)[i*numatm*3 + j ]);
		}
		fprintf(op,"\n");
		}
	}
*/

  if (jacobi(norm, vect, numatm * 3, 100 * numatm * numatm, 1.e-10) != 0) {
    aaerror("Jacobi in FDnormal() returns an error");
  }

  /* check orthogonal */
  /*
	for( i=0; i< numatm; i++)
	{
	x1 =0; x2 = 0; x3 = 0;
	for (j=0; j< numatm; j++)
	{
	x1 +=(*vect)[ j*3]*(*vect)[j*3];
	x1 +=(*vect)[ j*3+1]*(*vect)[j*3+1];
	x1 +=(*vect)[ j*3+2]*(*vect)[j*3+2];
	x2 +=(*vect)[i*numatm*3 +j*3]*(*vect)[i*numatm*3+j*3];
	x2 +=(*vect)[i*numatm*3 +j*3+1]*(*vect)[i*numatm*3+j*3+1];
	x2 +=(*vect)[i*numatm*3 +j*3+2]*(*vect)[i*numatm*3+j*3+2];
	x3 +=(*vect)[j*3]*(*vect)[i*numatm*3+j*3];
	x3 +=(*vect)[j*3+1]*(*vect)[i*numatm*3+j*3+1];
	x3 +=(*vect)[j*3+2]*(*vect)[i*numatm*3+j*3+2];
	}
	fprintf(op,"normality check %f %d %f >%f< should be zero\n",x1,i,x2,x3);
	}
*/
/* end check */
  if (echo) {
    if (normal > 0.0f) {
      fprintf(op, "REMARK   4\n" \
                  "REMARK   4 AMMP normal mode analysis results\n"\
                  "REMARK   4\n"
                  "%s", PdbRem);
    }
    fprintf(op, "%d Eigenvalues found:\n", 3 * numatm);
    for(i = 0; i < (numatm * 3); i++) {
      if (normal > 0.0f) fprintf(op, PdbRem);
      fprintf(op, "%12.6f kcal/A^2g", norm[i * 3 * numatm + i]);
      if (norm[i * 3 * numatm + i] > 0 ) {
        fprintf(op, " %12.6f cm-1\n",
                sqrt(4.184E26 * norm[i * 3 * numatm + i] * 0.5f) / 2.997924e10 / 3.14159265);
      } else {
        fprintf(op, " ***\n");
      }
    }
  }

  if ((echo) && (normal > 0.0f)) {
    fprintf(op, "%sThe Eigenvectors\n", PdbRem);
    for(i = 0; i < (numatm * 3); i++) {
      if (norm[i * numatm * 3 + i] > 0 ) {
        fprintf(op, "%s%f cm-1\n"
                    "MODEL %8d\n",
                    PdbRem,
                    sqrt(4.184E26 * norm[i * 3 * numatm + i] * 0.5f) / 2.997924e10 / 3.14159265,
                    i + 1);

      for(j = 0; j < numatm; j++) {
        ap = a_next(j);
        np = ap -> name;
        for(ii = 0; ii < 5; ii++) {
        if (*np != '.') {
          if (islower(*np)) {
            resid[ii] = toupper(*np);
          } else {
            resid[ii] = *np;
          }
        } else {
          resid[ii] = '\0';
          break;
        }
	if (*np == '\0') break;
        np++;
      } /* End of for (j) */
      np++;

      for(ii = 0; ii < 5; ii++) {
        if (*np != '.') {
          if(islower(*np)) {
            atid[ii] = toupper(*np);
          } else {
            atid[ii] = *np;
          }
        } else {
          atid[ii] = '\0';
          break;
        }
	if (*np == '\0') break;
        np++;
      } /* End of for (ii) */

      fprintf(op, "ATOM  %5d  %-3.3s", j + 1, atid);

      fprintf(op, "%c%-3s  %4d    %8.3f%8.3f%8.3f%6.2f%6.2f\n",
              ' ', resid, ap -> serial / 100 + i + 1,
              ap -> x + vect[i * numatm * 3 + j * 3] * normal,
              ap -> y + vect[i * numatm * 3 + j * 3 + 1] * normal,
              ap -> z + vect[i * numatm * 3 + j * 3 + 2] * normal, 1.0f, 0.0f);
    } /* End of for (i) */
    fprintf(op, "ENDMDL\n");
  } else {
/*
		fprintf(op," ***** cm-1\n" );
		for( j=0; j< numatm*3; j++)
		{
			fprintf(op,"%f ",(*vect)[i*numatm*3 + j ]);
		}
		fprintf(op,"\n");
*/
      }
    }
  }

  free(vect);
  free(norm);

  return TRUE;
}


/**** Jacobi method for eigenvalue/eigenvector calculations ****/

int AMMP_FASTCALL jacobi(float *am, float *em, int n, int maxit, float toler)
{
  float         *s1, *s2;
  int           iindex, jindex, j;
  int           imax, jmax;
  int           iter;
  float         emax, r, sa, ca;

  const char *  Routine = "jacobi()";
  int           i       = n * sizeof(float);

  if (((s1 = (float *)Alloca(i, Routine)) == NULL) ||
      ((s2 = (float *)Alloca(i, Routine)) == NULL)) {
    SafeFree(s1);
    return FALSE;
  }

  /* set em (the eigenvector matrix)  to I */

  for(i = 0; i < n; i++) {
    for(j = 0; j< n; j++) {
      em[i *n + j] = 0.0f;
    } /* End of for (j) */
    em[i * n + i] = 1.0f;
  } /* End of for (i) */

  for(iter = 0; iter < maxit; iter++) {
    emax = -1;
    imax = 0;
    jmax = 0;
    for(i =0; i < n; i++) {
      for(j = i + 1; j< n; j++) {
        if (fabs(am[i*n+j]) > emax) {
          emax = fabs(am[i * n + j]);
          imax = i;
          jmax = j;
        }
      } /* End of for (j) */
    } /* End of for (i) */

    if (emax < toler) {
      free(s1);
      free(s2);
      return FALSE;
    }

    r = am[imax * n + imax] - am[jmax * n + jmax];
    r = r * r + 4.0f * am[imax * n + jmax] * am[imax * n + jmax];
    if (r <= 0.0f) {
      free(s1);
      free(s2);
      return TRUE;
    }
    r = sqrt(r);
    iindex = imax * n + imax;
    jindex = jmax * n + jmax;
    if (am[iindex] > am[jindex]) {
      ca = 0.5f * (1.0f + (am[iindex] - am[jindex]) / r);
      ca = sqrt(ca);
      if (am[imax * n + jmax] < 0.0f) ca = -ca;
      sa = am[imax * n + jmax] / r / ca;
    } else {
      sa = 0.5f * (1.0f - (am[iindex] - am[jindex]) / r);
      sa = sqrt(sa);
      ca = am[imax * n + jmax] / r / sa;
    }

    /* use the transformation */
    /* the rows */

    for(i = 0; i < n; i++) {
      iindex = i * n;
      s1[i] =  ca * am[iindex + imax] + sa * am[iindex + jmax];
      s2[i] = -sa * am[iindex + imax] + ca * am[iindex + jmax];
    }
    for(i = 0; i < n; i++) {
      iindex = i*n;
      am[iindex + imax] = s1[i];
      am[iindex + jmax] = s2[i];
    }

    /* the columns */

    for(i = 0; i < n; i++) {
      iindex = imax * n;
      jindex = jmax * n;
      s1[i]  =  ca * am[iindex + i] + sa * am[jindex + i];
      s2[i]  = -sa * am[iindex + i] + ca * am[jindex + i];
    }
    for(i = 0; i < n; i++) {
      iindex         = imax * n;
      jindex         = jmax * n;
      am[iindex + i] = s1[i];
      am[jindex + i] = s2[i];
    }

    /* and finally update v */

    for(i = 0; i < n; i++) {
      iindex = i * n;
      s1[i]  =  ca * em[iindex + imax] + sa * em[iindex + jmax];
      s2[i]  = -sa * em[iindex + imax] + ca * em[iindex + jmax];
    }
    for(i = 0; i < n; i++) {
      iindex = i * n;
      em[iindex + imax] = s1[i];
      em[iindex + jmax] = s2[i];
    }
  } /* end of iter loop */

  free(s1);
  free(s2);

  return FALSE;
}

