/*****************************************************************************
 * opengl.c                                                                  *
 *     The files 'motif.c' and 'opengl.c' contain all of the functions       *
 *     necessary for the GUI portion of GRAMMP.  This file contains          *
 *     functions handling the OpenGL commands for GRAMMP                     *    
 *                                                                           *
 * Written by David Cavanaugh (4/99)                                         *
 *****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "dsc.h"
#include "ammp.h"

list_link *first_draw_ATOM = NULL;
list_link *last_draw_ATOM = NULL;
list_link *first_draw_BOND = NULL;
list_link *last_draw_BOND = NULL;
list_link *first_TETHER = NULL;
list_link *last_TETHER = NULL;
list_link *first_NOEL = NULL;
list_link *last_NOEL = NULL;
GLboolean doubleBuffer;
GLXContext cx;
float draw_matrix[3][3] = {1.,0.,0., 0.,1.,0., 0.,0.,1.};
float draw_scale[3] = {1.0,1.0,1.0};
float draw_winz_plus = 10.,draw_winz_minus = -10.;

void SetAtomColor(draw_ATOM *);
void load_atom();
void load_color();
void load_connect();
void load_label();
void load_tether();
void load_noel();
list_link *match_label( list_link *, int);
void draw_walker(void);
void draw_atom_walker(void);
void draw_COM(void);
void draw_ONLAST(void);
void DrawAxis(void);
void DrawVectors(Vector *, float *);
void ShowDistanceLabel(draw_ATOM *, draw_ATOM *);
void ShowAngleLabel(draw_ATOM *, draw_ATOM *, draw_ATOM *);
void ShowDihedralLabel(draw_ATOM *, draw_ATOM *, draw_ATOM *, draw_ATOM *);
void Normalize3v(Vector *);
double ScalarProduct(Vector, Vector);
void SetDrawScale(float);
typedef struct{
	draw_ATOM *ap;
	float k,x,y,z;
	void *next;
}  TETHER;
typedef struct{
    draw_ATOM *atom1,*atom2;
    float d,dm,dh,km,kh;
    int ismet;
    void *next;
}  NOEL;


/*****************************************************************************
 * load_atom() - read ammp_to_grfx pipe for atoms                            *
 *****************************************************************************/
void load_atom()
{
	int id;
	float x,y,z;
	list_link *mylink;
	draw_ATOM *myatom;

	ReadFromAmmp(&id,sizeof(int));
	ReadFromAmmp(&x,sizeof(float));
	ReadFromAmmp(&y,sizeof(float));
	ReadFromAmmp(&z,sizeof(float));
	mylink = match_label( first_draw_ATOM, id); 

	if (mylink == NULL)
	{
		myatom = (draw_ATOM *) malloc( sizeof( draw_ATOM) );
		mylink = (list_link *) malloc( sizeof( list_link) );
		if( myatom == NULL || mylink == NULL )
			{aaerror(" cannot allocate memory in load_atom\n"); return; }
		mylink->what = myatom;
		mylink->id = id;
		myatom->serial = id;
		myatom->ispicked = 1==0;
		if( first_draw_ATOM == NULL ) 
			{ first_draw_ATOM = mylink;last_draw_ATOM = mylink;}
		if( last_draw_ATOM != mylink )
			{  last_draw_ATOM->next = mylink; last_draw_ATOM = mylink;}
			 mylink->next = NULL ;
			myatom->red   = 1.;
			myatom->green = 1.;
			myatom->blue  = 1.;
	}
	myatom = mylink->what;
	if (myatom == NULL)  
		GrammpError("Unable to Allocate Memory in LoadAtom()",FATAL);
	myatom->x = x;
	myatom->y = y;
	myatom->z = z;	
}


/*****************************************************************************
 * load_connect() - read ammp_to_grfx pipe for bonds                         *
 *****************************************************************************/
void load_connect()  
{
	int i1,i2;
	static int id = 0;
	list_link *mylink,*lp;
	draw_BOND *mybond;
	draw_ATOM *ap;

	ReadFromAmmp(&i1,sizeof(int));
	ReadFromAmmp(&i2,sizeof(int));

	mybond = (draw_BOND *) malloc( sizeof( draw_BOND) );
	mylink = (list_link *) malloc( sizeof( list_link) );
	if( mybond == NULL || mylink == NULL )
		{aaerror(" cannot allocate memory in load_connect\n"); return; }
	mylink->what = mybond;
	mylink->id = id++;
	lp = match_label( first_draw_ATOM,i1);
	mybond->first = lp->what;
	lp = match_label( first_draw_ATOM,i2);
	mybond->second = lp->what;
/* don't append non-extent bonds */
	if( mybond->first == NULL || mybond->second == NULL )
	{
		aaerror("did not match bond ids\n");
		free(mybond);
		free(mylink);
		return;
	}
	
	ap = mybond->first;
	ap = mybond->second;
	
	if( first_draw_BOND == NULL ) 
		{ first_draw_BOND = mylink;last_draw_BOND = mylink;}
	if( last_draw_BOND != mylink )
		{ last_draw_BOND->next = mylink; last_draw_BOND = mylink;}
	mylink->next = NULL ;

	return;
}


/*****************************************************************************
 * load_color() - read ammp_to_grfx pipe for atom colors                     *
 *****************************************************************************/
void load_color() 
{
	int id;
	int i;
	float red,green,blue;
	list_link *mylink;
	draw_ATOM *myatom;


	ReadFromAmmp(&id,sizeof(int));
	ReadFromAmmp(&red,sizeof(float));
	ReadFromAmmp(&green,sizeof(float));
	ReadFromAmmp(&blue,sizeof(float));
	mylink = match_label( first_draw_ATOM, id); 

	if( mylink == NULL ) return ;

	myatom = mylink->what;
	if( myatom == NULL )  
		GrammpError("Unable to Allocate Memory in LoadColor()",FATAL);

	glShadeModel(GL_SMOOTH);
	myatom->red = red; myatom->green = green; myatom->blue = blue;
}


/*****************************************************************************
 * load_tether() - read ammp_to_grfx pipe for tethers                        *
 *****************************************************************************/
void load_tether()
{
    int id;
    float k,x,y,z;
	list_link *mylink;
	draw_ATOM *ap;
	TETHER *new;

    ReadFromAmmp(&id,sizeof(int));
    ReadFromAmmp(&k,sizeof(float));
    ReadFromAmmp(&x,sizeof(float));
    ReadFromAmmp(&y,sizeof(float));
    ReadFromAmmp(&z,sizeof(float));

	mylink = match_label(first_draw_ATOM, id); 
    if (mylink == NULL)
    {
		fprintf(stderr,"atom %d not in draw_ATOM link list\n",id);
		return;
	}
	new = (TETHER *) malloc (sizeof(TETHER));
    new->ap = mylink->what;
    new->k = k;
    new->x = x;
    new->y = y;
    new->z = z;

	mylink = match_label(first_TETHER,id);
	if (mylink == NULL)
	{
		mylink = (list_link *) malloc(sizeof(list_link));
        if (new == NULL || mylink == NULL)
			GrammpError("Unable to Allocate Memory in LoadAtom()",FATAL);
        if (first_TETHER == NULL)
            { first_TETHER = mylink;last_TETHER = mylink;}
        if (last_TETHER != mylink)
            {  last_TETHER->next = mylink; last_TETHER = mylink;}
		mylink->next = NULL;
    }
	mylink->what = new;

#ifdef DEBUG
	printf("load_tether(%d,%f,%f,%f,%f)\n",id,k,x,y,z);
#endif
}

void load_noel()
{
    int id1,id2;
    float d,dm,dh,km,kh;
	list_link *mylink;
	draw_ATOM *ap1;
	draw_ATOM *ap2;
	NOEL *new;

    ReadFromAmmp(&id1,sizeof(int));
    ReadFromAmmp(&id2,sizeof(int));
    ReadFromAmmp(&d,sizeof(float));
    ReadFromAmmp(&dm,sizeof(float));
    ReadFromAmmp(&dh,sizeof(float));
    ReadFromAmmp(&km,sizeof(float));
    ReadFromAmmp(&kh,sizeof(float));

#ifdef DEBUG
	printf("load_noel(%d,%d,%f,%f,%f,%f,%f)\n",id1,id2,d,dm,dh,km,kh);
#endif

	mylink = match_label(first_draw_ATOM, id1); 
    if (mylink == NULL)
    {
		fprintf(stderr,"atom %d not in draw_ATOM link list\n",id1);
		return;
	}
	new = (NOEL *) malloc (sizeof(NOEL));
    new->atom1 = mylink->what;

	mylink = match_label(first_draw_ATOM, id2); 
    if (mylink == NULL)
    {
		fprintf(stderr,"atom %d not in draw_ATOM link list\n",id2);
		return;
	}
    new->atom2 = mylink->what;
    new->d = d;
    new->dm = dm;
    new->dh = dh;
    new->km = km;
    new->kh = kh;

	mylink = match_label(first_TETHER,id1);
	if (mylink == NULL)
	{
		mylink = (list_link *) malloc(sizeof(list_link));
        if (new == NULL || mylink == NULL)
			GrammpError("Unable to Allocate Memory in LoadAtom()",FATAL);
        if (first_NOEL == NULL)
            { first_NOEL = mylink;last_NOEL = mylink;}
        if (last_NOEL != mylink)
            {  last_NOEL->next = mylink; last_NOEL = mylink;}
		mylink->next = NULL;
    }
	mylink->what = new;
}

/*****************************************************************************
 * load_label() - read ammp_to_grfx pipe for atom labels                     *
 *****************************************************************************/
void load_label() 
{
	int id,i,ilong;
	char mess[10];
	list_link *mylink;
	draw_ATOM *myatom;

	ReadFromAmmp(&id,sizeof(int));
	ReadFromAmmp(&ilong,sizeof(int));
	mylink = match_label( first_draw_ATOM, id); 
	for( i=0;i< ilong; i++)
	{
		if( i < 9)
		{
			ReadFromAmmp(&mess[i],sizeof(char));
			mess[i+1] = '\0';
		}
		else
			ReadFromAmmp(&mess[9],sizeof(char));
	}	
	mess[9] = '\0';

	if (mylink == NULL)  return;

	myatom = mylink->what;
	if (myatom == NULL)
		GrammpError("Unable to Allocate Memory in LoadLabel()",FATAL);

	for( i=0; i< 10; i++)
		myatom->name[i] = mess[i];
}


/*****************************************************************************
 * match_label() - find id in list_link, return NULL if not found
 *****************************************************************************/
list_link *match_label(lroot, id)
list_link *lroot;
int id;
{
	if (lroot == NULL)  return (NULL);
	while (lroot->id != id)
	{
		if( lroot->next == NULL ) return (NULL);
		if( lroot->next == lroot) return (NULL);
		lroot = lroot->next;
	}
	if (lroot->id == id)  return (lroot);
	return (NULL);
}


/*****************************************************************************
 * draw_walker() - draw atoms and bonds with appropriate colors              *
 *****************************************************************************/
void draw_walker(void)
{
    Vector mid;
	Vector tview;
	TETHER *tp;
	NOEL *np;
	Vector ab;
	draw_BOND *bp;
	draw_ATOM *ap,*ap2;
	float distance;
	list_link *lp;
	int i;
	int num;
	char line[80];

	lp = first_draw_BOND;
	if( lp == NULL ) return; 
	bp = lp->what;

	if (LOADING)  return;
	draw_atom_walker();
	if (Autocenter)  CenterMolecule();
	if (Autoscale)   ScaleMolecule(NULL);

	glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glClearColor(0.0, 0.0, 0.0, 1.0); 

	lp = first_draw_ATOM;
	if( lp == NULL ) return ;
	glPushAttrib(GL_LIST_BIT);
	glColor3f(1.0,0.0,0.0);
	glListBase(0);

	/* output a label on the screen for each picked atom */
	while (TRUE)
	{
		ap = lp->what;
		if (ap->ispicked) 
		{
			sprintf( line,"%d %s",ap->serial,ap->name);
			glRasterPos3f(ap->dx,ap->dy,ap->dz);
			glCallLists(strlen(line),GL_UNSIGNED_BYTE, (GLubyte *) line);
		}
/* check for termination */
		if( lp == last_draw_ATOM ) break;
		if( lp->next == lp) break;
		if( lp->next == NULL ) break;
		lp = lp->next;
	}	
	glPopAttrib();
	glColor3f(0.0,0.0,1.0);
	glLineWidth(2.0);
	lp = first_draw_BOND;
	if( lp == NULL ) return; 

	/* Create bonds */
	glPushMatrix();
	glBegin(GL_LINES);
	while (TRUE)  
	{
		bp = lp->what;
		ap = bp->first;
		ap2 = bp->second;
		/* skip bonds if there length is almost zero - otherwise OpenGL will
		   crash */
		if ((fabs(ap->x-ap2->x) + fabs(ap->y-ap2->y) + fabs(ap->z-ap2->z)) > 1e-4) 
		{
			SetAtomColor(ap); SetAtomColor(ap2);
			/* CPK colors - use solid colors (glShade(GL_FLAT)) and CPK colors for
			   atoms.  These colors are then used to color halfway down the bonds. */
			if (!strcmp(MolecularColor,"Monochrome"))
			{
				glColor3f(ap->red,ap->green,ap->blue);
				glVertex3f(ap->dx,ap->dy,ap->dz);
				glColor3f(ap2->red,ap2->green,ap2->blue);
				glVertex3f(ap2->dx,ap2->dy,ap2->dz);
			}
			else if (!strcmp(MolecularColor,"CPK"))
			{
				mid.xyz[0] = (ap2->dx - ap->dx);
				mid.xyz[1] = (ap2->dy - ap->dy);
				mid.xyz[2] = (ap2->dz - ap->dz);
				Normalize3v(&mid);

				mid.xyz[0] = ap->dx + 0.5*mid.length*mid.xyz[0];
				mid.xyz[1] = ap->dy + 0.5*mid.length*mid.xyz[1];
				mid.xyz[2] = ap->dz + 0.5*mid.length*mid.xyz[2];

				glColor3f(ap->red,ap->green,ap->blue);
				glVertex3f(ap->dx,ap->dy,ap->dz);
				glVertex3fv(mid.xyz);

				glColor3f(ap2->red,ap2->green,ap2->blue);
				glVertex3fv(mid.xyz);
				glVertex3f(ap2->dx,ap2->dy,ap2->dz);
			}
			/* color each atom and then use glShade(GL_SMOOTH) to interpoliate 
			   between the two atoms for the bond color */
			else  
			{
				glColor3f(ap->red,ap->green,ap->blue);
				glVertex3f(ap->dx,ap->dy,ap->dz);
				glColor3f(ap2->red,ap2->green,ap2->blue);
				glVertex3f(ap2->dx,ap2->dy,ap2->dz);
			}
		}  /* end of non-zero bond length drawing */
/* check for termination */
		if (lp == last_draw_BOND) break;
		if (lp->next == lp) break;
		if (lp->next == NULL ) break;
		lp = lp->next;
	}
	glEnd();
	glPopMatrix();

	/* draw tethers */
	if (show_tethers && (first_TETHER != NULL))
	{
		lp = first_TETHER;

		glColor3f(1.0,0.0,1.0);
		glLineStipple(2,0xAAAA);
		glEnable(GL_LINE_STIPPLE);
		while (TRUE)
		{
			tp = lp->what;
			ap = tp->ap;
			tview.xyz[0] = (tp->x);
			tview.xyz[1] = (tp->y);
			tview.xyz[2] = (tp->z);
			DrawVectors(&tview,draw_center);
			for (i=0;i<3;i++)  tview.xyz[i] -= draw_center[i];
			for (i=0;i<3;i++)  tview.xyz[i] *= draw_scale[i];
			glBegin(GL_LINES);
				glVertex3f(ap->dx,ap->dy,ap->dz);
				glVertex3fv(tview.xyz);
			glEnd();

	/* check for termination */
			if (lp == last_draw_ATOM)  break;
			if (lp->next == lp)  break;
			if (lp->next == NULL)  break;
			lp = lp->next;
		}
		glDisable(GL_LINE_STIPPLE);
	}
    if ((show_noels || show_violations) && (first_NOEL != NULL))
    {
        lp = first_NOEL;

        glLineStipple(2,0xAAAA);
        glEnable(GL_LINE_STIPPLE);
        while (TRUE)
        {
            np = lp->what;
            ap = np->atom1;
            ap2 = np->atom2;
			/* create vector from atom a to b (real coordinates) */
			ab.xyz[0] = (ap2->x - ap->x);
			ab.xyz[1] = (ap2->y - ap->y);
			ab.xyz[2] = (ap2->z - ap->z);
			Normalize3v(&ab);
			distance = ab.length;
			if ((distance > (np->d-np->dm)) && (distance < (np->d+np->dh)))  { glColor3f(0.0,1.0,0.0); np->ismet=TRUE; }
			else  { glColor3f(1.0,0.0,0.0); np->ismet=FALSE; }
			if (show_noels || (show_violations && !np->ismet))
			{
				glBegin(GL_LINES);
					glVertex3f(ap->dx,ap->dy,ap->dz);
					glVertex3f(ap2->dx,ap2->dy,ap2->dz);
				glEnd();
			}

    /* check for termination */
            if (lp == last_draw_ATOM)  break;
            if (lp->next == lp)  break;
            if (lp->next == NULL)  break;
            lp = lp->next;
        }
        glDisable(GL_LINE_STIPPLE);
	}





/*****************************************************************************
 * If geometric calculations are "on" (defined by boolean 'show') then
 * call the appropriate function for displaying the calculation.  Send the
 * appropriate number of atoms (i.e. distance: two, angle: three, and 
 * dihedral: four) and ignore the rest.
 *****************************************************************************/

	if (PickList[DISTANCE].show)  
	{
		num = PickList[DISTANCE].num - PickList[DISTANCE].num%2;
		if (num != 0)
			for (i=0;i<num;i+=2) ShowDistanceLabel(PickList[DISTANCE].atoms[i],PickList[DISTANCE].atoms[i+1]);
	}
	if (PickList[ANGLE].show)  
	{
		num = PickList[ANGLE].num - PickList[ANGLE].num%3;
		if (num != 0)
			for (i=0;i<num;i+=3) ShowAngleLabel(PickList[ANGLE].atoms[i],PickList[ANGLE].atoms[i+1],PickList[ANGLE].atoms[i+2]);
	}
	if (PickList[DIHEDRAL].show)  
	{
		num = PickList[DIHEDRAL].num - PickList[DIHEDRAL].num%4;
		if (num != 0)
			for (i=0;i<num;i+=4) ShowDihedralLabel(PickList[DIHEDRAL].atoms[i],PickList[DIHEDRAL].atoms[i+1],PickList[DIHEDRAL].atoms[i+2],
									PickList[DIHEDRAL].atoms[i+3]);
	}
	glPushMatrix();
		glDisable(GL_CLIP_PLANE0);
		glDisable(GL_CLIP_PLANE1);
		DrawAxis();
		glEnable(GL_CLIP_PLANE0);
		glEnable(GL_CLIP_PLANE1);
	glPopMatrix();
	SendToScreen(Glxarea);
}


/********************************************************************************
 * draw_atom_walker() - transform real atomic coordinates to screen coordinates *
 ********************************************************************************/
void draw_atom_walker()
{
	Vector dummy;
	draw_ATOM *dp;
	list_link *lp;
	int i;

	lp = first_draw_ATOM;
	if( lp == NULL ) return ;
	while (TRUE)/* do for ever */
	{
		dp = lp->what;

		dummy.xyz[0] = dp->x; dummy.xyz[1] = dp->y; dummy.xyz[2] = dp->z;
		DrawVectors(&dummy,draw_center);
		dp->dx = dummy.xyz[0]; dp->dy = dummy.xyz[1]; dp->dz = dummy.xyz[2];
		dp->dx -= draw_center[0]; dp->dy -= draw_center[1]; dp->dz -= draw_center[2];
		dp->dx *= draw_scale[0]; dp->dy *= draw_scale[1]; dp->dz *= draw_scale[2];

		dp->isslabed = FALSE;
		if (dp->dz > draw_winz_plus)   dp->isslabed = TRUE;
		if (dp->dz < draw_winz_minus)  dp->isslabed = TRUE;

		/* check for termination */
		if( lp == last_draw_ATOM ) break;
		if( lp->next == lp) break;
		if( lp->next == NULL ) break;
		lp = lp->next;
	}
}


/********************************************************************************
 * CenterMolecule() - force center (atom or COM) to center of coordinate system *
 ********************************************************************************/
void CenterMolecule()  
{
	draw_ATOM *ap;
	list_link *lp;
	int natom;
	
	lp = first_draw_ATOM;
	if (lp == NULL)  return ;
	draw_center[0] = 0.;
	draw_center[1] = 0.;
	draw_center[2] = 0.;
	natom = 0;
	while(TRUE)
	{
		ap = lp->what;
		draw_center[0] += ap->x;
		draw_center[1] += ap->y;
		draw_center[2] += ap->z;
		natom += 1;
	/* check for termination */
		if( lp == last_draw_ATOM ) break;
		if( lp->next == lp) break;
		if( lp->next == NULL ) break;
		lp = lp->next;
	}

	draw_center[0] /= natom;
	draw_center[1] /= natom;
	draw_center[2] /= natom;
}


/*****************************************************************************
 * draw_ONLAST() - set draw_center to last picked atom                       *
 *****************************************************************************/
void draw_ONLAST(void)
{
	if (last_picked_draw_ATOM == NULL) return ;
	draw_center[0] = last_picked_draw_ATOM ->x;
	draw_center[1] = last_picked_draw_ATOM ->y;
	draw_center[2] = last_picked_draw_ATOM ->z;
	draw_walker();
}


/*****************************************************************************
 * concatview() - makes the r1.rout product for xview = rout x (rout is the
 *                last left concat) and then graham-scmidts it to be 
 *                orthogonal matrix
 *****************************************************************************/
void concatview( r1,rout)
float r1[3][3],rout[3][3];
{
	float rs[3][3],a,b,c;
	int i ,j,k;

	for( i=0; i< 3; i++)  {
		for( j = 0; j<3; j++)  {
			rs[i][j] = 0.;
			for( k=0; k<3; k++)
			rs[i][j] += r1[i][k]*rout[k][j];
		}
	}

/* now orthogonalize */
/* normalize the first row */
	a = rs[0][0]*rs[0][0] + rs[1][0]*rs[1][0] + rs[2][0]*rs[2][0];
	c = 1./sqrt(a);
	rs[0][0]*=c; rs[1][0] *= c; rs[2][0] *= c;
	for( i=0; i<2; i++)  
	{
		a = rs[0][i]*rs[0][i] + rs[1][i]*rs[1][i] + rs[2][i]*rs[2][i];
		for( j=i+1; j< 3; j++) 
		{
			b = rs[0][j]*rs[0][j] + rs[1][j]*rs[1][j] + rs[2][j]*rs[2][j];
			c = rs[0][j]*rs[0][i] + rs[1][j]*rs[1][i] + rs[2][j]*rs[2][i];
			c = c/ sqrt(a*b);
			rs[0][j] -= c*rs[0][i]; rs[1][j] -= c*rs[1][i]; rs[2][j] -= c*rs[2][i];
			b = rs[0][j]*rs[0][j] + rs[1][j]*rs[1][j] + rs[2][j]*rs[2][j];
			c = 1./sqrt(b);
			rs[0][j] *= c; rs[1][j] *= c; rs[0][j] *= c;
		}/* j */
	}/* i */
	rout[0][0] = rs[0][0];
	rout[0][1] = rs[0][1];
	rout[0][2] = rs[0][2];
	rout[1][0] = rs[1][0];
	rout[1][1] = rs[1][1];
	rout[1][2] = rs[1][2];
	rout[2][0] = rs[2][0];
	rout[2][1] = rs[2][1];
	rout[2][2] = rs[2][2];
}


/*****************************************************************************
 * PickAtom() - attempt to pick an atom very close to the mouse click        *
 *                                                                           *
 * Input:                                                                    *
 *     XButtonEvent *event - contains info. about button event               *
 * Returns:                                                                  *
 *    int - TRUE or FALSE depending on whether an atom was picked or not     * 
 *****************************************************************************/
int PickAtom(XButtonEvent *event)  
{
	char string[256];
	draw_ATOM *ap, *apc;
	list_link *lp;
	float x, y;
	float dist2, dist2_min;
	int i;
	int success = FALSE;

/* Transform X coordinates to OpenGL coordinates.  The OpenGL coordinates range from
   -1 to 1 (defined with glOrtho) */
	x = (event->x - draw_winx*0.5);
	x /= draw_winx*0.5;
	y = (event->y - draw_winy*0.5);
	y /= -draw_winy*0.5;
	lp = first_draw_ATOM;
	if (lp == NULL)  return;
	dist2_min = 1e10;
	while (TRUE)  {
		ap = lp->what;
		if (!ap->isslabed)  
		{
			dist2 = (ap->dx - x)*(ap->dx - x) + (ap->dy - y)*(ap->dy - y);
			if (dist2 < dist2_min)  { dist2_min = dist2; apc = ap; }
		}
		if (lp == last_draw_ATOM )  break;
		if (lp->next == lp)  break;
		if (lp->next == NULL)  break;
		lp = lp->next;
	}
	/* close enough! */
	if (dist2_min < 0.0005)  
	{
		success = TRUE;
		if (apc->ispicked)  apc->ispicked = FALSE;
		else                apc->ispicked = TRUE; 
		last_picked_draw_ATOM = apc;
		if (PickList[DISTANCE].calc)  
		{
			if (PickList[DISTANCE].num == (MAX_PICKED_ATOMS-1))
			{
				sprintf(string,"Reached Maximium Allowed Picked Atoms for Distance Calculation (%d).  Hit Clear Labels",MAX_PICKED_ATOMS);
				GrammpError(string,NON_FATAL);
				return;
			}
			PickList[DISTANCE].atoms[PickList[DISTANCE].num++] = apc;
		}
		else if (PickList[ANGLE].calc)  
		{
			if (PickList[ANGLE].num == (MAX_PICKED_ATOMS-1))
			{
				sprintf(string,"Reached Maximium Allowed Picked Atoms for Angle Calculation (%d).  Hit Clear Labels",MAX_PICKED_ATOMS);
				GrammpError(string,NON_FATAL);
				return;
			}
			PickList[ANGLE].atoms[PickList[ANGLE].num++] = apc;
		}
		else if (PickList[DIHEDRAL].calc)  
		{
			if (PickList[DIHEDRAL].num == (MAX_PICKED_ATOMS-1))
			{
				sprintf(string,"Reached Maximium Allowed Picked Atoms for Dihedral Calculation (%d).  Hit Clear Labels",MAX_PICKED_ATOMS);
				GrammpError(string,NON_FATAL);
				return;
			}
			PickList[DIHEDRAL].atoms[PickList[DIHEDRAL].num++] = apc;
		}
	}	
	draw_walker();
	return (success);
}


/*****************************************************************************
 * ScaleMolecule() - scale the molecule.                                     *
 *                                                                           *
 *  Input:                                                                   *
 *     float *scale_ratio:                                                   *
 *         if NULL - roughly scale molecule within screen                    *
 *         else    - scale molecule by this ratio (range: 0..Inf)            *
 *****************************************************************************/
void ScaleMolecule(float *scale_ratio)  
{
    draw_ATOM *ap;
    list_link *lp;
    float xmax,xmin,ymax,ymin;
	float scale = draw_scale[0];

	/* roughly scale molecule within screen */
	if (scale_ratio == NULL)  
	{
		lp = first_draw_ATOM;
		if (lp == NULL)  return;
		xmax = 0.; xmin = 0.;
		ymax = 0.; ymin = 0.;
		while (TRUE)  
		{
			ap = lp->what;
			if (ap->dx > xmax)  xmax = ap->dx;
			if (ap->dy > ymax)  ymax = ap->dy;
			if (ap->dx < xmin)  xmin = ap->dx;
			if (ap->dy < ymin)  ymin = ap->dy;
/* check for termination */
			if (lp == last_draw_ATOM )  break;
			if (lp->next == lp)  break;
			if (lp->next == NULL)  break;
			lp = lp->next;
		}
		xmax = xmax -xmin;
		ymax = ymax-ymin;
		if (xmax < 0.01 && ymax < 0.01)  { SetDrawScale(10.0); return; }
		if (xmin == 0.)  xmax = 0.2/xmax; 
		else  xmax = 0.9/xmax;

		if (ymin == 0.)  ymax = 0.2/ymax;
		else  ymax = 0.9/ymax;

		if (ymax < xmax)  scale *= ymax;
		else   scale *= xmax;

		SetDrawScale(scale);
	}	
	/* use argument to scale molecule - make sure scale_ratio is >0 */
	else  
	{ 
		if (*scale_ratio < 0)  *scale_ratio = -(*scale_ratio);
		if (*scale_ratio == 0)
			GrammpError("ScaleMolecule() Argument equal to Zero",FATAL);
		SetDrawScale(draw_scale[0]*(*scale_ratio));
	}
}


/*****************************************************************************
 * void Rotate() - rotate the molecule around either the x, y, or z axis     *
 *                                                                           *
 *  Input:                                                                   *
 *     char axis - defines rotation axis                                     *
 *     float angle - rotation angle in degrees                               *
 *****************************************************************************/
void Rotate(char axis, float angle)  
{
	char string[256];
    float rot[3][3];
    float sth,cth;

	sth = sin(angle*PI/180.0); cth = cos(angle*PI/180.0);

	switch (axis)  
	{
		case ('x'):
			rot[0][0] =  1.0; rot[0][1] =  0.0; rot[0][2] =  0.0;
			rot[1][0] =  0.0; rot[1][1] =  cth; rot[1][2] = -sth;
			rot[2][0] =  0.0; rot[2][1] =  sth; rot[2][2] =  cth;
			break;
		case ('y'):
			rot[0][0] =  cth; rot[0][1] =  0.0; rot[0][2] = -sth;
			rot[1][0] =  0.0; rot[1][1] =  1.0; rot[1][2] =  0.0;
			rot[2][0] =  sth; rot[2][1] =  0.0; rot[2][2] =  cth;
			break;
		case ('z'):
			rot[0][0] =  cth; rot[0][1] = -sth; rot[0][2] =  0.0;
			rot[1][0] =  sth; rot[1][1] =  cth; rot[1][2] =  0.0;
			rot[2][0] =  0.0; rot[2][1] =  0.0; rot[2][2] =  1.0;
			break;
		default:
			sprintf(string,"Unrecognizable axis (%c) in Rotate()",axis);
			GrammpError(string,FATAL);
            break;
	}
    concatview(rot,draw_matrix);
    draw_walker();
}


/*****************************************************************************
 * void Translate() - translate the molecule down either the x, y, or z axis *
 *                                                                           *
 *  Input:                                                                   *
 *     char axis - defines rotation axis                                     *
 *     float dist - translation distance                                     *
 *****************************************************************************/
void Translate(char axis, float dist)
{
	int i,j;
	char string[256];
	float zscale;

	#define draw_matrix_t(i,j) draw_matrix[j][i]
	dist *= 0.25;
	switch (axis)
    {
        case ('x'):
			dist /= draw_scale[0];
			draw_center[0] += dist*draw_matrix_t(0,0);
			draw_center[1] += dist*draw_matrix_t(1,0);
			draw_center[2] += dist*draw_matrix_t(2,0);
            break;
        case ('y'):
			dist /= draw_scale[1];
			draw_center[0] += dist*draw_matrix_t(0,1);
			draw_center[1] += dist*draw_matrix_t(1,1);
			draw_center[2] += dist*draw_matrix_t(2,1);
            break;
        case ('z'):
			dist /= draw_scale[2];
			draw_center[0] += dist*draw_matrix_t(0,2);
			draw_center[1] += dist*draw_matrix_t(1,2);
			draw_center[2] += dist*draw_matrix_t(2,2);
            break;
		default:
			sprintf(string,"Unrecognizable axis (%c) in Translate()",axis);
			GrammpError(string,FATAL);
            break;
    }
    draw_walker();
}


/*****************************************************************************
 * void LookDownAxis() - View molecule along x, y, or z axis                 *
 *                                                                           *
 *  Input:                                                                   *
 *     Widget parent - parent widget                                         *
 *     char axis - defines axis to view down                                 *
 *****************************************************************************/
void LookDownAxis(Widget parent, char *axis)  
{
	switch(*axis)  
	{
		case ('x'):
			draw_matrix[0][0] =  0.; draw_matrix[0][1] = 0.; draw_matrix[0][2] = 1.;
			draw_matrix[1][0] =  0.; draw_matrix[1][1] = 1.; draw_matrix[1][2] = 0.;
			draw_matrix[2][0] = -1.; draw_matrix[2][1] = 0.; draw_matrix[2][2] = 0.;
			break;
		case ('y'):
			draw_matrix[0][0] = 1.; draw_matrix[0][1] =  0.; draw_matrix[0][2] = 0.;
			draw_matrix[1][0] = 0.; draw_matrix[1][1] =  0.; draw_matrix[1][2] = 1.;
			draw_matrix[2][0] = 0.; draw_matrix[2][1] = -1.; draw_matrix[2][2] = 0.;
			break;
		case ('z'):
			draw_matrix[0][0] = 1.; draw_matrix[0][1] = 0.; draw_matrix[0][2] = 0.;
			draw_matrix[1][0] = 0.; draw_matrix[1][1] = 1.; draw_matrix[1][2] = 0.;
			draw_matrix[2][0] = 0.; draw_matrix[2][1] = 0.; draw_matrix[2][2] = 1.;
			break;
	}
	draw_walker();
}


/*****************************************************************************
 * void SetAtomColor() - set the color for an atom.  use MolecularColor      *
 *                       to define coloring scheme                           *              
 *                                                                           *
 *  Input:                                                                   *
 *     draw_ATOM *ap - pointer to atom structure                             *
 *****************************************************************************/
void SetAtomColor(draw_ATOM *ap)  
{
	float red, green, blue;
	char *element;
	char message[256];

	if (!strcmp(MolecularColor,"Monochrome"))
	{
		red = 255.0/255.0; green = 255.0/255.0; blue = 255.0/255.0;
		ap->red = red; ap->green = green; ap->blue = blue;
	}
	else if (!strcmp(MolecularColor,"CPK"))
	{
		element = strchr(ap->name,'.');
		element++;
		if (!strncmp(element,"h",1) || !strncmp(element,"1",1))
			{ red=255.0/255.0; green=255.0/255.0; blue=255.0/255.0; }
		else if (!strncmp(element,"c",1))  
			{ red=175.0/255.0; green=175.0/255.0; blue=175.0/255.0; }
		else if (!strncmp(element,"o",1))  
			{ red=255.0/255.0; green=0.0/255.0;   blue=0.0/255.0; }
		else if (!strncmp(element,"n",1))  
			{ red=143.0/255.0; green=143.0/255.0; blue=255.0/255.0; }
		else if (!strncmp(element,"s",1))  
			{ red=255.0/255.0; green=200.0/255.0; blue=50.0/255.0; }
		else if (!strncmp(element,"p",1))  
			{ red=255.0/255.0; green=165.0/255.0; blue=0.0/255.0; }
		else  
			{ red=0.0/255.0; green=255.0/255.0; blue=0.0/255.0; }
		ap->red = red; ap->green = green; ap->blue = blue;
	}	
	else if (!strcmp(MolecularColor,"Force"))
	{
			return;
	}
	else
	{
		sprintf(message,"Unknown MolecularColor (%s) in SetAtomColor() - Using White",MolecularColor);
		GrammpError(message,NON_FATAL);
		red = 255.0/255.0; green = 255.0/255.0; blue = 255.0/255.0;
	}
	return;
}


/*****************************************************************************
 * void DrawAxis() - draw x,y,z axis                                         *
 *****************************************************************************/
void DrawAxis(void)
{
	static Vector origin;
	static Vector x_unit;
	static Vector y_unit;
	static Vector z_unit;
	Vector color;
	Vector *vp;
	float length;
	char line[1056];
	int need_init = TRUE;
	int i,j;

	if (need_init)
	{
		origin.xyz[0] = -0.80; origin.xyz[1] = -0.80; origin.xyz[2] = 0.0;
		length = 0.15;

		for (i=0;i<3;i++)  
		{
			x_unit.xyz[i] = origin.xyz[i];
			y_unit.xyz[i] = origin.xyz[i];
			z_unit.xyz[i] = origin.xyz[i];
		}
		x_unit.xyz[0] += length;
		y_unit.xyz[1] += length;
		z_unit.xyz[2] += length;
		need_init = FALSE;
	}
	DrawVectors(&x_unit,origin.xyz);
	DrawVectors(&y_unit,origin.xyz);
	DrawVectors(&z_unit,origin.xyz);

	glPushMatrix();
	glLineWidth(2.0);
	glBegin(GL_LINES);
		glColor3f(1.0,0.0,0.0);
		glVertex3fv(origin.xyz);
		glVertex3fv(x_unit.xyz);
		glColor3f(0.0,1.0,0.0);
		glVertex3fv(origin.xyz);
		glVertex3fv(y_unit.xyz);
		glColor3f(0.5,0.5,1.0);
		glVertex3fv(origin.xyz);
		glVertex3fv(z_unit.xyz);
	glEnd();

	glColor3f(1.0,0.0,0.0);
	sprintf(line,"X");
	glRasterPos3fv(x_unit.xyz);
	glCallLists(strlen(line),GL_UNSIGNED_BYTE, (GLubyte *) line);

	glColor3f(0.0,1.0,0.0);
	sprintf(line,"Y");
	glRasterPos3fv(y_unit.xyz);
	glCallLists(strlen(line),GL_UNSIGNED_BYTE, (GLubyte *) line);

	glColor3f(0.5,0.5,1.0);
	sprintf(line,"Z");
	glRasterPos3fv(z_unit.xyz);
	glCallLists(strlen(line),GL_UNSIGNED_BYTE, (GLubyte *) line);
	glPopMatrix();
}


/*****************************************************************************
 * void DrawVectors() - rotate a vector in space                             *
 *                                                                           *
 *  Input:                                                                   *
 *     Vector *vp - pointer to vector structure                              *
 *     float *origin - coordinates of origin                                 *
 *****************************************************************************/
void DrawVectors(Vector *vp,float *origin)
{
	Vector dummy;
	int i,j;

	/* translate the coordinates to the origin */
	for (i=0;i<3;i++)  
		dummy.xyz[i] = vp->xyz[i] - origin[i];

	/* perform necessary rotation and translate the coordinates back */
	for (i=0;i<3;i++)  
	{
		vp->xyz[i] = 0.0;
		for (j=0;j<3;j++)  vp->xyz[i] += dummy.xyz[j]*draw_matrix[i][j];
		vp->xyz[i] += origin[i];
	}
}


/*****************************************************************************
 * void draw_COM: set the draw_center array to the geometric center of the   *
 *                molecule                                                   *
 *****************************************************************************/
void draw_COM(void)  
{ 
    draw_ATOM *ap;
	float COM[] = {0.0,0.0,0.0};
	int i;
    list_link *lp;

	lp = first_draw_ATOM;
	if (lp == NULL) return ;
	while (TRUE)
	{
		ap = lp->what;

		COM[0] += ap->x;
		COM[1] += ap->y;
		COM[2] += ap->z;

		if( lp == last_draw_ATOM ) break;
		if( lp->next == lp) break;
		if( lp->next == NULL ) break;
		lp = lp->next;
	}
	for (i=0;i<3;i++)  draw_center[i] = COM[i];
	draw_walker();
}


/*******************************************************************************
 * void ShowDistanceLabel() - 1. calculate the distance between the two atoms  *
 *                            2. draw a stippled line between these two atoms  *
 *                            3. put a label with the distance halfway between *
 *                                                                             *
 *  Input:                                                                     *
 *     draw_ATOM a - first atom                                                *
 *     draw_ATOM b - second atom                                               *
 *******************************************************************************/
void ShowDistanceLabel(a,b) 
draw_ATOM *a,*b; 
{
	Vector mid;
	Vector ab;
	float distance;
	GLfloat dummy[3];
	char line[40];

	/* return if a and b have not been "picked" and if a and b are the same atom. */
	if ((a == NULL) || (b == NULL))  return;
	if (a == b)  return;

	/* create vector from atom a to b (real coordinates) */
	ab.xyz[0] = (b->x - a->x);
	ab.xyz[1] = (b->y - a->y);
	ab.xyz[2] = (b->z - a->z);
	Normalize3v(&ab);
	distance = ab.length;
	sprintf(line,"%.2f",distance);

	/* create vector from atom a to b (screen coordinates) */
	ab.xyz[0] = (b->dx - a->dx);
	ab.xyz[1] = (b->dy - a->dy);
	ab.xyz[2] = (b->dz - a->dz);
	Normalize3v(&ab);

	/* find midpoint of a to b bond (screen coordinates) */
	mid.xyz[0] = (ab.xyz[0]*0.5*ab.length + a->dx);
	mid.xyz[1] = (ab.xyz[1]*0.5*ab.length + a->dy);
	mid.xyz[2] = (ab.xyz[2]*0.5*ab.length + a->dz);

	glColor3f(0.0,1.0,0.0);

	/* Create distance "bond" */
	glLineWidth(2.0);
	glLineStipple(2,0xAAAA);
	glEnable(GL_LINE_STIPPLE);
	glPushMatrix();
	glBegin(GL_LINES);
		glVertex3f(a->dx,a->dy,a->dz);
		glVertex3fv(mid.xyz);
		glVertex3fv(mid.xyz);
		glVertex3f(b->dx,b->dy,b->dz);
	glEnd();
	glPopMatrix();
	glDisable(GL_LINE_STIPPLE);

	/* Write distance on screen */
	glPushAttrib(GL_LIST_BIT);
	glRasterPos3fv(mid.xyz);
	glCallLists(strlen(line),GL_UNSIGNED_BYTE, (GLubyte *) line);
	glPopAttrib();
}


/*******************************************************************************
 * void ShowDihedralLabel() - 1. calculate the dihedral angle given four atoms *
 *                            2. draw a stippled line between these four atoms *
 *                            3. put a label with the angle (in degrees) near  *
 *                               the second atom                               *
 *                                                                             *
 *  Input:                                                                     *
 *     draw_ATOM a - first atom                                                *
 *     draw_ATOM b - second atom                                               *
 *     draw_ATOM c - third atom                                                *
 *     draw_ATOM d - fourth atom                                               *
 *******************************************************************************/
void ShowDihedralLabel(a,b,c,d) 
draw_ATOM *a,*b,*c,*d;
{
	int i;
	extern float get_torsion_value(ATOM *, ATOM *, ATOM *, ATOM *);
	float dihedral;
	ATOM a1,b1,c1,d1;
    Vector ab,bc,cd,abc,place;
	char line[256];

    /* return if a, b, c and d have not been "picked" and if a, b, c, and d are the same atom. */
	if ((a == NULL) || (b == NULL) || (c == NULL) || (d == NULL))  return;
	if ((a == b) || (a == c) || (a == d))  return;

	/* use ammp function get_torsion_value() to calculate the dihedral angle.  first, must
	   massage the draw_ATOM struct data into ATOM struct data */
	a1.x = a->x;	a1.y = a->y;	a1.z = a->z;	
	b1.x = b->x;	b1.y = b->y;	b1.z = b->z;	
	c1.x = c->x;	c1.y = c->y;	c1.z = c->z;	
	d1.x = d->x;	d1.y = d->y;	d1.z = d->z;	
	dihedral = get_torsion_value(&a1,&b1,&c1,&d1);
	dihedral *= 180/PI;

    ab.xyz[0] = (b->x - a->x); ab.xyz[1] = (b->y - a->y); ab.xyz[2] = (b->z - a->z);
    bc.xyz[0] = (c->x - b->x); bc.xyz[1] = (c->y - b->y); bc.xyz[2] = (c->z - b->z);
    cd.xyz[0] = (d->x - c->x); cd.xyz[1] = (d->y - c->y); cd.xyz[2] = (d->z - c->z);

    glColor3f(1.0,1.0,0.0);
    glLineWidth(2.0);
    glLineStipple(2,0xAAAA);
    glEnable(GL_LINE_STIPPLE);
    glPushMatrix();
    glBegin(GL_LINES);
        glVertex3f(a->dx,a->dy,a->dz);
        glVertex3f(b->dx,b->dy,b->dz);
        glVertex3f(b->dx,b->dy,b->dz);
        glVertex3f(c->dx,c->dy,c->dz);
        glVertex3f(c->dx,c->dy,c->dz);
        glVertex3f(d->dx,d->dy,d->dz);
    glEnd();
    glPopMatrix();
    glDisable(GL_LINE_STIPPLE);

    Normalize3v(&ab);
    Normalize3v(&cd);
    for (i=0;i<3;i++)  abc.xyz[i] = ab.xyz[i] + cd.xyz[i];
    Normalize3v(&abc);

    /* place the angle label on the vector bisecting the other two
       vectors (screen coordinates).  it should be close to the second
       atom but no too close that it overlaps the atom label */
    place.xyz[0] = abc.xyz[0]*0.1*abc.length + b->dx;
    place.xyz[1] = abc.xyz[1]*0.1*abc.length + b->dy;
    place.xyz[2] = abc.xyz[2]*0.1*abc.length + b->dz;

	glPushAttrib(GL_LIST_BIT);
    glRasterPos3fv(place.xyz);
	sprintf(line,"%.1f",dihedral);
    glCallLists(strlen(line),GL_UNSIGNED_BYTE, (GLubyte *) line);
	glPopAttrib();
}


/********************************************************************************  
 * void ShowAngleLabel() - 1. calculate the angle between the three atoms       *
 *                         2. draw a stippled line between these three atoms    *
 *                         3. atoms and put a label with the angle (in degrees) *
 *                            near the middle atom                              *
 *                                                                              *
 *  Input:                                                                      *
 *     draw_ATOM a - first atom                                                 *
 *     draw_ATOM b - second atom                                                *
 *     draw_ATOM c - third atom                                                 *
 ********************************************************************************/
void ShowAngleLabel(a,b,c) 
draw_ATOM *a,*b,*c;
{
	Vector ba,bc,abc,place;
	double sp, theta;
	char line[10];
	int i;

    /* return if a, b, and c have not been "picked" and if a, b, and c are the same atom. */
	if ((a == NULL) || (b == NULL) || (c == NULL))  return;
	if ((a == b) || (a == c))  return;

	/* create vectors from second atom to first and third (real coordinates),
	   calculate scalar product and then determine angle */
	ba.xyz[0] = (a->x - b->x);
	ba.xyz[1] = (a->y - b->y);
	ba.xyz[2] = (a->z - b->z);
	bc.xyz[0] = (c->x - b->x);
	bc.xyz[1] = (c->y - b->y);
	bc.xyz[2] = (c->z - b->z);
	sp = ScalarProduct(ba,bc);
	Normalize3v(&ba);
	Normalize3v(&bc);
	theta = acos(sp/(ba.length*bc.length))*180/PI;

	/* create stippled lines to each of three atoms (screen coordinates) */
	ba.xyz[0] = (a->dx - b->dx);
	ba.xyz[1] = (a->dy - b->dy);
	ba.xyz[2] = (a->dz - b->dz);
	bc.xyz[0] = (c->dx - b->dx);
	bc.xyz[1] = (c->dy - b->dy);
	bc.xyz[2] = (c->dz - b->dz);
	Normalize3v(&ba);
	Normalize3v(&bc);
	for (i=0;i<3;i++)  abc.xyz[i] = ba.xyz[i] + bc.xyz[i];
	Normalize3v(&abc);

	/* place the angle label on the vector bisecting the other two
	   vectors (screen coordinates).  it should be close to the second
	   atom but no too close that it overlaps the atom label */
	place.xyz[0] = abc.xyz[0]*0.1*abc.length + b->dx;
	place.xyz[1] = abc.xyz[1]*0.1*abc.length + b->dy;
	place.xyz[2] = abc.xyz[2]*0.1*abc.length + b->dz;
    glColor3f(0.6,0.6,1.0);

    glLineWidth(2.0);
    glLineStipple(2,0xAAAA);
    glEnable(GL_LINE_STIPPLE);
	glPushMatrix();
	glBegin(GL_LINES);
		glVertex3f(a->dx,a->dy,a->dz);
		glVertex3f(b->dx,b->dy,b->dz);
		glVertex3f(b->dx,b->dy,b->dz);
		glVertex3f(c->dx,c->dy,c->dz);
	glEnd();
	glPopMatrix();
    glDisable(GL_LINE_STIPPLE);

	glPushAttrib(GL_LIST_BIT);
    glRasterPos3fv(place.xyz);
	sprintf(line,"%.1f",theta);
    glCallLists(strlen(line),GL_UNSIGNED_BYTE, (GLubyte *) line);
	glPopAttrib();
}


/*****************************************************************************
 * Normalize3v() - normalize a three coordinate vector                       *
 *                                                                           *
 * Input:                                                                    *
 *     Vector *a - pointer to a three coordinate vector                      *
 *****************************************************************************/
void Normalize3v(Vector *a)
{
    double length = 0.0;
    int i;

    for (i=0;i<3;i++)  length += a->xyz[i]*a->xyz[i];
    length = sqrt(length);
    for (i=0;i<3;i++)  a->xyz[i] /= length;

    a->length = (float) length;
}


/*****************************************************************************
 * ScalarProduct() - return the scalar product of two vectors                *
 *                                                                           *
 * Input:                                                                    *
 *     Vector a - first vector                                               *
 *     Vector b - second vector                                              *
 *****************************************************************************/
double ScalarProduct(Vector a, Vector b)
{
    double sp = 0.0;
    int i;

    for (i=0;i<3;i++)  sp += a.xyz[i]*b.xyz[i];
    return(sp);
}


/*****************************************************************************
 * SendToScreen() - Output window matrix to a Drawing Area widget            *
 *                                                                           *
 * Input:                                                                    *
 *     Widget w - send the grfx matrix to this widget                        *
 *****************************************************************************/
void SendToScreen(Widget w)
{
    if (doubleBuffer) glXSwapBuffers(display, XtWindow(w));
    if(!glXIsDirect(display, cx))
        glFinish(); /* avoid indirect rendering latency from queuing */
}


/*****************************************************************************
 * SetDrawScale() - set the DrawScale matrix                                 *
 *                                                                           *
 * Input:                                                                    *
 *     int scale - scaling for x direction                                   *
 *****************************************************************************/
void SetDrawScale(float scale)
{
	float zscale = draw_winz_plus - draw_winz_minus;
	if (zscale > 0.01)  zscale = 1./zscale;

    draw_scale[0] = scale;
	draw_scale[1] = scale*draw_winx/draw_winy;
	draw_scale[2] = zscale;
}
