/*
* optimist.c
*
* optimiser routines for AMMP
*
*  steepest descents
*  conjugate gradients
*  line search    routines
*
*/
/*
*  copyright 1992 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/* function steep()
* steepest descents optimiser
*
* passed a set of functions vfs for the potential
* passed a set of functions ffs for the force
*  how many functions  nfs
*  how many steps to try
*  when to stop
*/

int AMMP_FASTCALL steep(AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int nstep, float toler, int echo)
{
  float         vb, vt, vto;
  int           i, j, ifs;
  int           Abort;

  const char *  StepStr = "  Steep  %5d: vb %f, maxf %f\n";
  float         lam     = 0.0f;
  float         lastvb  = 0.0f;
  int           vbcheck = 0;

#ifdef VEGAZZ
  int           nupdat = GetNupdat();
  int           supdat;
#endif

  if (!CheckAtoms()) return FALSE;
  if (nstep < 1) nstep = 1;

#ifdef VEGAZZ
  if (!echo) supdat = 0;
  else supdat = GetSupdat();
#endif

  Abort = FALSE;
  for(i = 0; i < nstep; i++) {

#ifdef VEGAZZ
    if ((nupdat) && (!(i % nupdat))) send_all_atoms();
#endif

    vb = 0.0f;
    for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vb, 0.0f);
    a_f_zero();
    for(ifs = 0; ifs < nfs; ifs++) (*ffs[ifs])(0.0f);
    lam = a_max_f();

#ifdef VEGAZZ
    if ((supdat) && (!(i % supdat)))
#else
    if (echo)
#endif
      printf(StepStr, i, vb, lam);

    if (lam <= toler) break;
    if (vb == lastvb) {
      if (vbcheck++ == 5) {
        Abort = TRUE;
        break;
      }
    } else vbcheck = 0;
    lastvb = vb;

    if (lam <= 1.0f ) a_ftodx(1.0, 0.0f);
    else a_ftodx(1.0 / sqrt(lam), 0.0f);

    vto = vb;
    lam = 0.0f;

    /* This is a Fibonacci line search rather odd, the steps keep getting
     * bigger until either the end or we go up
     */

    for(j = 0; j < 200; j++) {
      vt = 0.0f;
      lam = lam + 0.01f * (float)j;
      for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vt, lam);
      if ((vt > vto) && (j == 0)) a_ftodx(0.0f, 0.1f);
      else if (vt > vto) {
        lam = lam - 0.01f * (float)j;
        break;
      }
      vto = vt;
    } /* End of for (j) */
    a_inc_d(lam);

#ifdef GRAMMP
    send_all_atoms();
#endif
  } /* End of for (i) */

#ifdef VEGAZZ
  if ((supdat) && ((i % supdat) || (i == nstep)))
    printf(StepStr, i, vb, lam);
#endif

  if (Abort)
    printf("  Steep       : Unable to converge. Minimization stopped\n");

#ifdef VEGAZZ
  if (supdat) printf("\n");
#else
  if (echo  ) printf("\n");
#endif

  return TRUE;
}


/* function cngdel()
* conjugate gradients optimiser
*
* passed a set of functions vfs for the potential
* passed a set of functions ffs for the force
*  how many functions  nfs
*  how many steps to try
*  when to stop
*/

int AMMP_FASTCALL cngdel(AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int nstep,
                         int nreset, float toler, int echo, int nupdat)
{
  int           i, ifs;
  float         vb;
  float         lam;

  const char *  StepStr = "  Cngdel %5d: vb %f, maxf %f, beta %f\n";
  float         beta    = 0.0f;
  float         lastlam = 0.0f;

#ifdef VEGAZZ
  int           supdat;
#endif

  if (!CheckAtoms()) return FALSE;
  if (nstep < 1) nstep = 1;

#ifdef VEGAZZ
  if (!echo) supdat = 0;
  else supdat = GetSupdat();
#endif

  /**** Do at most nstep steps ****/

  if (nreset < 1) nreset = nstep;
  a_g_zero();
  a_d_zero();
  for(i = 0; i < nstep ; i++) {

#ifdef VEGAZZ
    if ((nupdat) && (!(i % nupdat))) send_all_atoms();
#endif

    vb = 0.0f;
    for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vb, 0.0f);
    a_f_zero();
    for(ifs = 0; ifs < nfs; ifs++) (*ffs[ifs])(0.0f);
    lam = a_max_f();

    /**** Make up the beta use just the simple one ****/

/*
    beta = a_l2_f();
    betad = a_l2_g();
    if (betad < 1.e-4f) {
      betad = 1.0f;
      beta = 0.0f;
    }
    beta = -beta/betad;
*/

    beta = a_pr_beta();

    if ((i % nreset) == 0) beta = 0.0f;
#ifdef VEGAZZ
    if ((supdat) && (!(i % supdat)))
#else
    if(echo)
#endif
      printf(StepStr, i, vb, lam, beta);
    lastlam = lam;

    if (lam <= toler) break;

/*	lam = a_max_f(); */
    a_ftodx(1.0f, beta);
    a_ftogx(1.0f, 0.0f);
    lam = linmin(vfs, nfs, sqrt(a_max_d()) );
    if (lam < 1.e-6) {
      a_f_zero();
      for(ifs = 0; ifs < nfs; ifs++) (*ffs[ifs])(0.0f);
      lam = sqrt(a_max_f());
      if (lam < 1.0f) lam = 1.0f;
      a_ftodx(1.0f / lam, 0.0f);
      lam = linmin(vfs, nfs, 1.0f );
      if (lam < 1.e-6) break;
    }
    a_inc_d(lam);
#ifdef GRAMMP
    send_all_atoms();
#endif
  } /* End of for (i) */

#ifdef VEGAZZ
  if (supdat) {
    if ((i % supdat) || (i == nstep))
      printf(StepStr, i, vb, lastlam, beta);
    printf("\n");
  }
#else
  if (echo) printf("\n");
#endif

  return TRUE;
}


/* Line minimization routine linmin( x, search, n, f )
 *
 * returns step size for approximate line minimizer
 * recursive line minimizer
 */

float AMMP_FASTCALL linmin(AMMP_VFUNC ffs[], int nfs, float damp)
{
  int           i, iter, jter, imin;
  float         alpha[401], fval[401], dstep, step, stpmin, fvt ,fmin, fold;

  float         mxdq   = GetMxdq();
  int           nostep = GetNostep();

  if (nostep < 1) nostep = 8;

  SetMxdq(hundred);

  step     = 0;
  stpmin   = zero;
  imin     = 0;
  alpha[0] = zero;
  fval[0]  = zero;

  if (damp < one) dstep = fourth;
  else dstep = one / damp;

  for(i = 0; i < nfs; i++) (*ffs[i])(&fval[0], zero);
  fmin = fval[0];
  fold = fmin;

  /**** Recheck is to find a descent stp length ****/

  imin++;

recheck:
  alpha[imin] = dstep;
  fval[imin]  = zero;
  for(i = 0; i < nfs; i++) (*ffs[i])(&fval[imin], dstep);
  if (fval[imin] > fval[0]) {
    dstep = dstep * fourth;
    if (dstep > 1.e-8) goto recheck;
    goto cleanup;
  }

  /**** If here we have dstep small enough to use ****/

  for(iter = 0; iter < nostep; iter++) {
    for(jter = 1; jter < 100; jter++) {
      step += dstep;

      /**** Get the function value ****/

      for(i = 0; i < imin; i++) {
        if (alpha[i] == step) {
          fvt = fval[i];
          goto FOUND;
        }
      } /* End of for (i) */
      fvt = 0.0f;
      for(i = 0; i < nfs; i++) (*ffs[i])(&fvt, step);
      alpha[imin]  = step;
      fval[imin++] = fvt;
      if (imin > 400) imin = 400;
FOUND:
/*    printf("\n %f %f %f %f", step,fvt,fmin,fold);  */
      if (fvt < fmin) {
        fmin   = fvt;
        stpmin = step;
      }
      if (fvt > fold) {
        dstep = - dstep / two;
        break;
      }
      fold = fvt;
    }
  }
cleanup:

  /* do not allow 0 steps */
  /*	if( stpmin == 0.) stpmin = dstep/2; */
  SetMxdq(mxdq);

  return stpmin;
}
