/* orbit.c
*
* routines to use floating gaussian orbitals for QM calculations
*
*/
/*
*  copyright 1993,1994,1995 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <assert.h>
#include <stdio.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"
#include "orbit.h"

/**** Macros ****/

#define  o_first()              firstORBITAL;

/**** Global variables ****/

ORBITAL *               firstORBITAL = NULL;
ORBITAL *               lastORBITAL  = NULL;

float                   total_ex, total_col, total_nuc;
float                   total_enuc, total_kinet, total_norm;

/**** Local variables ****/

static ORBITAL *        o_m_serial_op;

static int              orbitNUMBER            = 0;
static int              orbitUPDATE            = 0;
static int              orbital_highest        = -1;
static int              orbital_lowest         = -1;
static ORBITAL *        o_m_serial_lastmatched = NULL;
static ORBITAL *        o_next_op              = NULL;


/**** Reset local variables ****/

void AMMP_FASTCALL ResetOrbit(void)
{
  orbitNUMBER            = 0;
  orbitUPDATE            = 0;
  orbital_highest        = -1;
  orbital_lowest         = -1;
  o_m_serial_lastmatched = NULL;
  o_next_op              = NULL;
}

/**** Add a new orbital ****/

int AMMP_FASTCALL orbital(int type, int i1, int i2, int i3, int i4, int i5, int i6,
                          int osn, float along, float x, float y, float z , int spin,
                          int pair)
{
  AMMP_ATOM     *ap;
  ORBITAL       *new;

  if ((ap = a_m_serial(i1)) == NULL) {
    aaerror("No atom for an orbital");
    return FALSE;
  }
  if (!pair) pair = 2;

  new = NULL;
  if ((orbital_highest >= osn) && (orbital_lowest <= osn)) new = o_m_serial(osn);
  if (orbital_highest < osn) orbital_highest = osn;
  if (orbital_lowest  > osn) orbital_lowest  = osn;
  if (!new) {
    orbitUPDATE = 1;
    if ((new = Alloca(sizeof(ORBITAL), "orbital()")) == NULL)
      return FALSE;

    new -> next = NULL;
  }

  if (firstORBITAL == NULL) {
    firstORBITAL    = new;
    orbital_highest = osn;
    orbital_lowest  = osn;
  }

  if (lastORBITAL == NULL) lastORBITAL = new;
  if (new -> next == NULL) {
    new -> gang         = NULL;
    new -> next         = new;
    new -> n            = 0;
    new -> active       = TRUE;
    lastORBITAL -> next = new;
    lastORBITAL         = new;
  }

  new -> type    = type;
  new -> myatom  = ap;
  new -> ncenter = 1;
  new -> a1      = NULL;
  new -> a2      = NULL;
  new -> a3      = NULL;
  new -> a4      = NULL;
  new -> a5      = NULL;

  if (i2 >= 0) new -> a1 = a_m_serial(i2);
  if (i3 >= 0) new -> a2 = a_m_serial(i3);
  if (i4 >= 0) new -> a3 = a_m_serial(i4);
  if (i5 >= 0) new -> a4 = a_m_serial(i5);
  if (i6 >= 0) new -> a5 = a_m_serial(i6);

  /**** Multi atom center ****/

  if (type == Orm) {
   if (i2 >= 0) new -> ncenter += 1;
   if (i3 >= 0) new -> ncenter += 1;
   if (i4 >= 0) new -> ncenter += 1;
   if (i5 >= 0) new -> ncenter += 1;
   if (i6 >= 0) new -> ncenter += 1;
  }

  new -> osn    = osn;
  new -> along  = along;
  new -> along2 = x;
  new -> x      = x;
  new -> y      = y;
  new -> z      = z;
  new -> spin   = spin;
  new -> ipair  = pair;

  return TRUE;
}


int AMMP_FASTCALL expand(int osn, int n, float a1, float b1, float a2, float b2,
                         float a3, float b3, float a4, float b4, float a5,
                         float b5, float a6, float b6)
{
  ORBITAL       *op = o_m_serial(osn);

  if (!op) {
    aaerror("No orbital for expansion");
    return FALSE;
  }

  if (n > 6) n = 6;
  else if(n < 0) n = 0;

  op -> n = n;
  op -> a[0] = a1;
  op -> a[1] = a2;
  op -> a[2] = a3;
  op -> a[3] = a4;
  op -> a[4] = a5;
  op -> a[5] = a6;

  if (b1 > zero) op -> rl[0] = log(b1);
  else op -> rl[0] = b1;

  if (b2 > zero) op -> rl[1] = log(b2);
  else op -> rl[1] = b2;

  if (b3 > zero) op -> rl[2] = log(b3);
  else op -> rl[2] = b3;

  if (b4 > zero) op -> rl[3] = log(b4);
  else op -> rl[3] = b4;

  if (b5 > zero) op -> rl[5] = log(b5);
  else op -> rl[5] = b5;

  return TRUE;
}


/**** Dump orbit ****/

void AMMP_FASTCALL dump_orbit(FILE *outp)
{
  AMMP_ATOM     *ap;
  ORBITAL       *op;
  int           i, j, natom;
  int           s1, s2, s3, s4, s5;

  int no = o_number();
  if (no <= 0) return;

  for(i = 0; i < no; i++) {
    op = o_next(i);
    switch(op -> type) {
    case Or1:
      fprintf(outp, "orbit 1 %d %d %d %d ;\n",
              op -> myatom -> serial, op -> osn, op -> spin, op -> ipair);
      break;

    case Or1o:
      fprintf(outp, "orbit 1o %d %d %f %f %f %f %d %d;\n",
              op -> myatom -> serial, op -> osn, op -> along,
              op -> x, op -> y,op -> z, op -> spin, op -> ipair);
      break;

    case Or2:
      fprintf(outp, "orbit 2 %d %d %d %f %d %d;\n",
              op -> myatom -> serial, op -> a1 -> serial,
              op -> osn, op -> along, op -> spin, op -> ipair);
      break;

    case Or3:
      fprintf(outp, "orbit 3 %d %d %d %d %f %f %d %d;\n",
              op -> myatom -> serial, op -> a1 -> serial, op -> a2 -> serial,
              op -> osn, op -> along, op -> along2, op -> spin, op -> ipair);
      break;

    case Or4s:
      fprintf(outp, "orbit 4s %d %d %d %d %d %f %d %d;\n",
              op -> myatom -> serial, op -> a1 -> serial, op -> a2 -> serial,
              op -> a3 -> serial, op -> osn, op -> along, op -> spin, op -> ipair);
      break;

    case Or4p:
      fprintf(outp, "orbit 4p %d %d %d %d %d %f %d %d;\n",
              op -> myatom -> serial, op -> a1 -> serial, op -> a2 -> serial,
              op -> a3 -> serial, op -> osn, op -> along, op -> spin, op -> ipair);
      break;

    case Orm:
      if (op -> a1) s1 = op -> a1 -> serial;
      else s1 = -1;
      if (op -> a2) s2 = op -> a2 -> serial;
      else s2 = -1;
      if (op -> a3) s3 = op -> a3 -> serial;
      else s3 = -1;
      if (op -> a4) s4 = op -> a4 -> serial;
      else s4 = -1;
      if (op -> a5) s5 = op -> a5 -> serial;
      else s5 = -1;

      fprintf(outp, "orbit m %d %d %d %d %d %d  %d %d %d;\n",
              op -> myatom -> serial, s1, s2, s3, s4, s5, op -> osn,
              op -> spin, op -> ipair);
      break;
    } /* End of switch */
  } /* End of for (i) */

  for(i = 0; i < no; i++) {
    op = o_next(i);
    fprintf(outp, "expand %d %d ", op -> osn, op -> n);
    for(j = 0; j < op -> n; j++)
      if(op -> rl[j] > zero)
        fprintf(outp, "%g %g ", op -> a[j], exp(op -> rl[j]));
      else
        fprintf(outp, "%g %g ", op -> a[j], op -> rl[j]);
    fprintf(outp, ";\n");
  } /* End of for (i) */

  natom = a_number();
  for(i = 0; i < natom; i++) {
    ap = a_next(i) ;
    if (ap -> na > 0)
      fprintf(outp, "mov %d.na %f;\n", ap -> serial, ap -> na);
  } /* End of for (i) */

  for(i = 0; i < no; i++) {
    op = o_next(i);
    if (op -> gang)
      fprintf(outp, "dscf gang %d %d ;\n", op -> osn, ((ORBITAL *)op -> gang) -> osn);
    if (!op -> active)
      fprintf(outp, "dscf freeze %d ;\n", op -> osn);
  } /* End of for (i) */
}


float AMMP_FASTCALL Loose_Ritz_product(ORBITAL *odummy)
{
  return Basic_Ritz_product(odummy);

/*
  H = Basic_Ritz_product(odummy);
  x = fabs( H +total_kinet/total_norm);
  no = o_number();
  return H + x*2/no;
*/
}

float AMMP_FASTCALL Just_Ritz_product(ORBITAL *odummy)

{
  return Basic_Ritz_product(odummy);
}

float AMMP_FASTCALL Ritz_product(ORBITAL *odummy)
{
  float         H = Basic_Ritz_product(odummy);
  float         x = fabs(H + total_kinet / total_norm);

  return (H + x);
}

float AMMP_FASTCALL INDO_Ritz_product(ORBITAL *odummy)
{
  return Basic_INDO_Ritz_product(odummy);
/*
	float H,x;
	H = Basic_INDO_Ritz_product(odummy);
*/

/*	x = fabs( H +total_kinet/total_norm);
*/
/*
	return H;
*/
/*	return H + x;
*/
}

float AMMP_FASTCALL tight_INDO_Ritz_product(ORBITAL *odummy)
{
  float         H = Basic_INDO_Ritz_product(odummy);
  float         x = fabs( H + total_kinet / total_norm);

  return (H + x);
}

/* form the full ritz product
* other routines will do the denominator and partial products */

float AMMP_FASTCALL Basic_Ritz_product(ORBITAL *odummy)
{
  float         ritzn, ritzd;
  int           i, no;
  ORBITAL       *op1;


  /**** Warning phiH4phi calls o_next() ****/

  no = o_number();
  if (no < 1) return 0.0f;
/*
float total_ex,total_col,total_nuc,total_enuc,total_kinet,total_norm;
*/
	total_ex = 0.;
	total_col = 0.;
	total_nuc = 0.;
	total_enuc = 0.;
	total_kinet = 0.;
	total_norm = 0.;
	o_update_normals();
	ritzn = 0.;
	ritzd = 1.;
	op1 = o_first() ;
	for( i=0; i< no; i++)
	{ /*op1 = o_next(i); */
	 op1->normal = phiphi(op1,op1);
	 ritzd *= op1->normal;
	op1 = op1->next;
	}
/*
	printf( " ritzd %f\n",ritzd);
	if( ritzd <= 0.) return 1000000.;
	if( ritzd >= 10000000.) return 100000000.;
*/
	if( ritzd <= 0.) ritzd = 1.e-6;
	assert( ritzd > 0.);
	op1 = o_first();
	for( i=0; i< no; i++)
	{
		ritzn += phiH2phi(op1,op1,ritzd);
		ritzn += phiH4phi(op1,op1,ritzd);
	op1 = op1->next; 
	}/* i */
	total_norm = ritzd;
	assert( total_norm > 0.);
	total_nuc = nuclear();

  return ritzn/ritzd + total_nuc;
}

/* form the full ritz product
* other routines will do the denominator and partial products */

float AMMP_FASTCALL Basic_INDO_Ritz_product(ORBITAL *odummy)
{
  float         ritzn,ritzd;
  int           i, no;
  ORBITAL       *op1;

/* warning phiH4phi calls o_next() */

	no = o_number();
	if( no <1 ) return 0.;
/*
float total_ex,total_col,total_nuc,total_enuc,total_kinet,total_norm;
*/
	total_ex = 0.;
	total_col = 0.;
	total_nuc = 0.;
	total_enuc = 0.;
	total_kinet = 0.;
	total_norm = 0.;
	o_update_normals();
RETRY: 
/*	ritzn = 0.; */
/*	ritzd = 1.; */
	op1 = firstORBITAL;
	if( op1 == NULL ) return 0.0f;
	for( i=0; i< no; i++)
	{ /*op1 = o_next(i);*/
	 op1->normal = phiphi(op1,op1);
	op1 = op1->next;
	}
	op1 = o_first();
	for( i=0; i< no; i++)
	{
		if( op1->normal < .1 || op1->normal > 10.)
		{
			renormalize();
			goto RETRY;

		}
		op1 = op1->next;
	}
	op1   = o_first() ;
	ritzd = 1.0f;
	for( i=0; i< no; i++)
	{
	ritzn = op1->normal;
	 ritzd *= ritzn;
	op1 = op1->next;
	}
	ritzn = 0.;
/*
	printf( " ritzd %f\n",ritzd);
	if( ritzd <= 0.) return 1000000.;
	if( ritzd >= 10000000.) return 100000000.;
*/
	if( ritzd <= 0.) ritzd = 1.e-6;
	assert( ritzd > 0.);
	op1 = o_first();
	for( i=0; i< no; i++)
	{
		ritzn += phiH2phi(op1,op1,ritzd);
		ritzn += INDOphiH4phi(op1,op1,ritzd,1.e-3);
/* the cutoff is int( oi oj)/sqrt( int(oi,oi),int(oj,oj))
* 1.e-2 is on the hairy edge (numerical tests)
* 1.e-3 is almost indistinquishable from 0 
*/
	op1 = op1->next; 
	}/* i */
	total_norm = ritzd;
	assert( total_norm > 0.);
	total_nuc = nuclear();

	return ritzn/ritzd + total_nuc;


}

void AMMP_FASTCALL renormalize(void)
{
  float         mag;
  int           i,j,no;
  ORBITAL       *op;


	no = o_number();
	if( no < 1) return;
	op = o_first();
	for( i=0; i< no; i++)
	{ 
		mag = phiphi(op,op);
		mag = 1./sqrt(mag);
		for(j=0; j< op->n; j++)
			op->a[j] *= mag;
		op = op->next;
	}
}

int AMMP_FASTCALL report(FILE *out)
{
	fprintf(out," sums \n");
	fprintf(out," nuclear    %f\n",total_nuc);
	fprintf(out," kinetic    %f\n",total_kinet/total_norm);
	fprintf(out," e-/nuclear %f\n",total_enuc/total_norm);
	fprintf(out," coloumb    %f\n",total_col/total_norm);
	fprintf(out," Normalizer %f\n",total_norm);
	fprintf(out," Energy H   %f\n", total_nuc+
		(total_kinet+total_enuc+total_col)/total_norm);
	fprintf(out," Virial error  %f\n", fabs(total_nuc+
		(total_kinet+total_kinet+total_enuc+total_col)/total_norm));

  return TRUE;
}


float AMMP_FASTCALL phiH2phi(ORBITAL *o1, ORBITAL *o2, float prenorm)
{
  AMMP_ATOM *   ac;
  float         x, y, z, r;
  float         x1, y1, z1;
  float         x2, y2, z2;
  float         ke, nuc;
  float         rab, rcp, re, rs;
  float         norm, ntemp;
  float         s1, s2, sign;
  int           i, j, kk, l, m;
  int           k, numatm;
	
  ke   = 0.0f;
  nuc  = 0.0f;
  norm = prenorm / o1 -> normal / o2 -> normal * phiphi(o1, o2);
	
	s2 = 1;
	for( l = 0; l< o2->ncenter; l++)
	{
	if( o2->type == Or4p) s2 = -s2;
	s1 = 1;
	for( k = 0; k< o1->ncenter; k++)
	{
	if( o1->type == Or4p) s1 = -s1;
	sign = s1*s2;
	x = o2->rx[l] - o1->rx[k];
	y = o2->ry[l] - o1->ry[k];
	z = o2->rz[l] - o1->rz[k];
	rab = (x*x + y*y + z*z)*INVBOHR*INVBOHR;
	numatm = a_number();	
	for( i=0; i< o1->n; i++)
	{
		for( j=0; j< o2->n; j++) 
		{

		r = o1->r[i]*o2->r[j]/( o1->r[i] + o2->r[j]);
		re = exp(-r*rab);
		rs = (o1->r[i] + o2->r[j]);
/* x1,y1,z1 are in ANGSTROMS */
		ke = ke +  sign*o1->a[i]*o2->a[j]*
			r*(three-two*r*rab)*pow( PI/rs,1.5)*re;
/* nuclear integrals including partial charges for at a distance */
		x1 = (o2->rx[l]*o2->r[j]+o1->rx[k]*o1->r[i])/rs;
		y1 = (o2->ry[l]*o2->r[j]+o1->ry[k]*o1->r[i])/rs;
		z1 = (o2->rz[l]*o2->r[j]+o1->rz[k]*o1->r[i])/rs;
		ntemp =  sign*o1->a[i]*o2->a[j]*TWOPI/rs*re;
		for(kk=0; kk< numatm; kk++)
		{ ac = a_next(kk); 
/* ANGSTROM difference */
		x2 = x1 - ac->x;
		y2 = y1 - ac->y;
		z2 = z1 - ac->z;
/* converted to BOHR here */
		rcp = (x2*x2 +y2*y2 + z2*z2)*INVBOHR*INVBOHR;
		if( ac->na > 0 )
		nuc = nuc + ntemp*ac->na *Fzero(rs*rcp); 
		else 
		{
		for( m=0; m< o1->myatom->dontuse; m++)
			if( ac == o1->myatom->excluded[m] )  goto SKIP;
		for( m=0; m< o2->myatom->dontuse; m++)
			if( ac == o2->myatom->excluded[m] )  goto SKIP;
		nuc = nuc - ntemp*ac->q *Fzero(rs*rcp); 
SKIP:		;
		}/*else */
		} /* kk */
		}
	}
	}} /* k,l loops */
	if( o1->ipair == 2 && o2->ipair == 2) {
	nuc = 2*nuc*norm;
	ke = 2*ke*norm;
	} else {
	nuc = nuc*norm;
	ke = ke*norm;
	}

/*
float total_ex,total_col,total_nuc,total_enuc,total_kinet,total_norm;
*/
	total_kinet += ke;
	total_enuc -= nuc;
	return (ke-nuc);
}

float AMMP_FASTCALL QandDphiH4phi(ORBITAL *o1, ORBITAL *o4, float prenorm)
{
/* coloumb and exchange integrals are 4-center uugh!! */
/* do all the integrals 
* o1 is assumed to be o4 
o1   o2 o3 o4(4)
and
o1  o2 o3(4) o4(3)
*/
/* just Hartree version !!! */

  float         x1, y1, z1, r1;
  float         x2, y2, z2, r2;
  float         coloumb, exchange;
  float         rab, rac, rpq;
  float         s4, s12, s34;
  float         norm_col;
  int           i, j, m, n, o, p;
  int           k, l, no;
  int           io3, io2;
  ORBITAL       *o3, *o2;

	
	no = o_number();
	if( no == 1  && o1->ipair == 1 ) return 0.;
	if( o1 != o4)
	{
		fprintf(stderr,"invalid call to phiH4phi\n"); 
		return 0.;
	}
	coloumb = 0.;
	exchange = 0.;
	
	o3 = firstORBITAL;
	for( io3 = 0; io3 <no; io3++)
	{
	o2= firstORBITAL;
	for( io2 = 0; io2 <no ; io2++)
	{ 
	/* o1 == o4 so only one overlap */

	norm_col = prenorm/o1->normal/o2->normal/o3->normal*phiphi(o2,o3);

	if( o1->ipair == 2) norm_col *= two/o1->ncenter;
/* this is an approximate overlap correction */
	if( o2 != o1 || o3 != o1) norm_col *= half;
	if( o2->ipair == 1) norm_col *= half;
	if( o3->ipair == 1) norm_col *= half;
	for( m = 0 ; m < o1->ncenter; m++)
	for( n = 0 ; n < o2->ncenter; n++)
	for( o = 0 ; o < o3->ncenter; o++)
	for( p = 0 ; p < o4->ncenter; p++)
	{
	

		x1 = o1->rx[m] -o4->rx[p];
		y1 = o1->ry[m] -o4->ry[p];
		z1 = o1->rz[m] -o4->rz[p];
		rab = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
		x1 = o2->rx[n] -o3->rx[o];
		y1 = o2->ry[n] -o3->ry[o];
		z1 = o2->rz[n] -o3->rz[o];
		rac = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
/*
		x1 = o1->rx[m] -o3->rx[o];
		y1 = o1->ry[m] -o3->ry[o];
		z1 = o1->rz[m] -o3->rz[o];
		rbc = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
		x1 = o4->rx[p] -o2->rx[n];
		y1 = o4->ry[p] -o2->ry[n];
		z1 = o4->rz[p] -o2->rz[n];
		rcd = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
*/
	for( i=0; i< o1->n; i++)
	{
	for( j=0; j< o2->n; j++) 
	{
		for( k=0; k< o3->n; k++)
		{
		for( l=0; l< o4->n; l++) 
		{
/*  1,2  3,4 coloumb */
/* this is the correct form at least for 2 e- */

		s12 = o1->r[i] + o4->r[l];
		s34 = o3->r[k] + o2->r[j];
		s4  = s12 + s34;
		r1 = o1->r[i]*o4->r[l]/( o1->r[i]+o4->r[l]);
		r2 = o3->r[k]*o2->r[j]/( o3->r[k]+o2->r[j]);
		x1 = (o1->r[i]*o1->rx[m] + o4->r[l]*o4->rx[p])/s12;
		y1 = (o1->r[i]*o1->ry[m] + o4->r[l]*o4->ry[p])/s12;
		z1 = (o1->r[i]*o1->rz[m] + o4->r[l]*o4->rz[p])/s12;
		x2 = (o3->r[k]*o3->rx[o] + o2->r[j]*o2->rx[n])/s34;
		y2 = (o3->r[k]*o3->ry[o] + o2->r[j]*o2->ry[n])/s34;
		z2 = (o3->r[k]*o3->rz[o] + o2->r[j]*o2->rz[n])/s34;
		x2 -= x1;
		y2 -= y1;
		z2 -= z1;
		rpq = (x2*x2 + y2*y2 + z2*z2)*INVBOHR*INVBOHR;
/* rab == 0 , rcd == 0 so just need rac or rbd or ... */
		coloumb = coloumb + 
		norm_col*o1->a[i]*o2->a[j]*o3->a[k]*o4->a[l] *
		34.986837 /(s12*s34*sqrt(s4))*Fzero( s12*s34/s4*rpq)
		*exp( -r1*rab-r2*rac);
/* rab = rcd = 0. */

		} /* l */
	}/*k */
		} /* j */
	} /* i */
	} /* m,n,o,p loops */
	o2 = o2->next;
	} /* io2 */
	o3 = o3->next;
	} /* io3 */

/*
float total_ex,total_col,total_nuc,total_enuc,total_kinet,total_norm;
*/
	total_ex -= exchange;
	total_col += coloumb;
	return (coloumb  - exchange);
}


float AMMP_FASTCALL phiH4phi(ORBITAL *o1, ORBITAL *o4, float prenorm)
{

/* coloumb and exchange integrals are 4-center uugh!! */
/* do all the integrals 
* o1 is assumed to be o4 
o1   o2 o3 o4(4)
and
o1  o2 o3(4) o4(3)
*/
/* just Hartree version !!! */
	int i,j,m,n,o,p;
	float x1,y1,z1,r1;
	float x2,y2,z2,r2;
	float coloumb,exchange;
	float rab,rbc,rac,rcd,rpq;
	float sa4,s12,s34;
	int k,l,no;
	int io3,io2;
	ORBITAL *o3,*o2;
	float norm_col;
	float s1,s2,s3,s4,sign;
	
	no = o_number();
	if( no == 1  && o1->ipair == 1 ) return 0.;
	if( o1 != o4)
	{
		fprintf(stderr,"invalid call to phiH4phi\n"); 
		return 0.;
	}
	coloumb = 0.;
	exchange = 0.;
	
	o3 = firstORBITAL;
	for( io3 = 0; io3 <no; io3++)
	{
	o2= firstORBITAL;
	for( io2 = 0; io2 < no  ; io2++)
	{ 
	/* o1 == o4 so only one overlap */

	norm_col = prenorm/o1->normal/o2->normal/o3->normal*phiphi(o2,o3);

/*
	if there is one function then the coloumb and exchange
	integrals are the same and the 2*c -ex = 1*c
*/
	if( no == 1 ) norm_col *= half;
	if( o1->ipair == 2) norm_col *= two;
/* this is an approximate overlap correction 
	if( o2 != o1 || o3 != o1) norm_col *= half;
*/
	if( o2->ipair == 1) norm_col *= half;
	if( o3->ipair == 1) norm_col *= half;
	s1 = 1;
	for( m = 0 ; m < o1->ncenter; m++)
	{
	if( o1->type == Or4p) s1 = -s1;
	s2 = 1;
	for( n = 0 ; n < o2->ncenter; n++)
	{
	s3 = 1;
	if( o2->type == Or4p) s2 = -s2;
	for( o = 0 ; o < o3->ncenter; o++)
	{
	if( o3->type == Or4p) s3 = -s3;
	s4 = 1;
	for( p = 0 ; p < o4->ncenter; p++)
	{
	if( o4->type == Or4p) s4 = -s4;

	sign = s1*s2*s3*s4;
		x1 = o1->rx[m] -o4->rx[p];
		y1 = o1->ry[m] -o4->ry[p];
		z1 = o1->rz[m] -o4->rz[p];
		rab = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
		x1 = o2->rx[n] -o3->rx[o];
		y1 = o2->ry[n] -o3->ry[o];
		z1 = o2->rz[n] -o3->rz[o];
		rac = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
		x1 = o1->rx[m] -o3->rx[o];
		y1 = o1->ry[m] -o3->ry[o];
		z1 = o1->rz[m] -o3->rz[o];
		rbc = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
		x1 = o4->rx[p] -o2->rx[n];
		y1 = o4->ry[p] -o2->ry[n];
		z1 = o4->rz[p] -o2->rz[n];
		rcd = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
	for( i=0; i< o1->n; i++)
	{
	for( j=0; j< o2->n; j++) 
	{
		for( k=0; k< o3->n; k++)
		{
		for( l=0; l< o4->n; l++) 
		{
/*  1,2  3,4 coloumb */
/* this is the correct form at least for 2 e- */

		s12 = o1->r[i] + o4->r[l];
		s34 = o3->r[k] + o2->r[j];
		sa4  = s12 + s34;
		r1 = o1->r[i]*o4->r[l]/( o1->r[i]+o4->r[l]);
		r2 = o3->r[k]*o2->r[j]/( o3->r[k]+o2->r[j]);
		x1 = (o1->r[i]*o1->rx[m] + o4->r[l]*o4->rx[p])/s12;
		y1 = (o1->r[i]*o1->ry[m] + o4->r[l]*o4->ry[p])/s12;
		z1 = (o1->r[i]*o1->rz[m] + o4->r[l]*o4->rz[p])/s12;
		x2 = (o3->r[k]*o3->rx[o] + o2->r[j]*o2->rx[n])/s34;
		y2 = (o3->r[k]*o3->ry[o] + o2->r[j]*o2->ry[n])/s34;
		z2 = (o3->r[k]*o3->rz[o] + o2->r[j]*o2->rz[n])/s34;
		x2 -= x1;
		y2 -= y1;
		z2 -= z1;
		rpq = (x2*x2 + y2*y2 + z2*z2)*INVBOHR*INVBOHR;
/* rab == 0 , rcd == 0 so just need rac or rbd or ... */
		x2 = r1*rab+r2*rac;
		if( x2 < 70.)
		coloumb = coloumb + sign*
		norm_col*o1->a[i]*o2->a[j]*o3->a[k]*o4->a[l] *
		34.986837 /(s12*s34*sqrt(sa4))*Fzero( s12*s34/sa4*rpq)
		*exp( -x2);
/* rab = rcd = 0. */
/*	if( o2 != o1 || o3 != o1) */
		if( no > 1 )
		{ /* minimum det is order 2 */
		s12 = o1->r[i] + o3->r[k];
		s34 = o4->r[l] + o2->r[j];
		sa4  = s12 + s34;
		x1 = (o1->r[i]*o1->rx[m] + o3->r[k]*o3->rx[o])/s12;
		y1 = (o1->r[i]*o1->ry[m] + o3->r[k]*o3->ry[o])/s12;
		z1 = (o1->r[i]*o1->rz[m] + o3->r[k]*o3->rz[o])/s12;
		x2 = (o4->r[l]*o4->rx[p] + o2->r[j]*o2->rx[n])/s34;
		y2 = (o4->r[l]*o4->ry[p] + o2->r[j]*o2->ry[n])/s34;
		z2 = (o4->r[l]*o4->rz[p] + o2->r[j]*o2->rz[n])/s34;
		x2 -= x1;
		y2 -= y1;
		z2 -= z1;
		rpq = (x2*x2 + y2*y2 + z2*z2)*INVBOHR*INVBOHR;
		r1 = o1->r[i]*o3->r[k]/( o1->r[i]+o3->r[k]);
		r2 = o4->r[l]*o2->r[j]/( o4->r[l]+o2->r[j]);
		x2 = r1*rbc + r2*rcd;
		if( x2 < 70.)
		coloumb = coloumb - sign*
		0.5*norm_col*o1->a[i]*o2->a[j]*o3->a[k]*o4->a[l] *
		34.986837 /(s12*s34*sqrt(sa4))*Fzero( s12*s34/sa4*rpq)
		*exp( -x2);

/*
		coloumb = coloumb - sign*
		0.5*norm_col*o1->a[i]*o2->a[j]*o3->a[k]*o4->a[l] *
		34.986837 /(s12*s34*sqrt(sa4))*Fzero( s12*s34/sa4*rpq)
		*exp( -r1*rbc-r2*rcd);
*/

				}

		} /* l */
	}/*k */
		} /* j */
	} /* i */
	}}}} /* m,n,o,p loops */
	o2 = o2->next;
	} /* io2 */
	o3 = o3->next;
	} /* io3 */

/*
float total_ex,total_col,total_nuc,total_enuc,total_kinet,total_norm;
*/
	total_ex -= exchange;
	total_col += coloumb;
	return (coloumb  - exchange);
}

float AMMP_FASTCALL INDOphiH4phi(ORBITAL *o1, ORBITAL *o4, float prenorm, float cutoff)
{

/* coloumb and exchange integrals are 4-center uugh!! */
/* do all the integrals 
* o1 is assumed to be o4
o1   o2 o3 o4(4)
and
o1  o2 o3(4) o4(3)
*/
/* just Hartree version !!! */
	int i,j,m,n,o,p;
	float x1,y1,z1,r1;
	float x2,y2,z2,r2;
	float coloumb,exchange;
	float rab,rbc,rac,rcd,rpq;
	float sa4,s12,s34;
	int k,l,no;
	int io3,io2;
	ORBITAL *o3,*o2;
	float norm_col;
	float s1,s2,s3,s4,sign;
	
	no = o_number();
	if( no == 1  && o1->ipair == 1 ) return 0.;
	if( o1 != o4)
	{
		fprintf(stderr,"invalid call to phiH4phi\n"); 
		return 0.;
	}
	coloumb = 0.;
	exchange = 0.;
	
	o3 = firstORBITAL;
	for( io3 = 0; io3 <no; io3++)
	{
	o2= firstORBITAL;
	for( io2 = 0; io2 < no  ; io2++)
	{ 
	/* o1 == o4 so only one overlap */

	norm_col = phiphi(o2,o3);
	if( norm_col/o2->normal/o3->normal < cutoff) goto SKIP;
	norm_col = prenorm/o1->normal/o2->normal/o3->normal*norm_col;

/*
	if there is one function then the coloumb and exchange
	integrals are the same and the 2*c -ex = 1*c
*/
	if( no == 1 ) norm_col *= half;
	if( o1->ipair == 2) norm_col *= two;
/* this is an approximate overlap correction 
	if( o2 != o1 || o3 != o1) norm_col *= half;
*/
	if( o2->ipair == 1) norm_col *= half;
	if( o3->ipair == 1) norm_col *= half;
	s1 = 1;
	for( m = 0 ; m < o1->ncenter; m++)
	{
	if( o1->type == Or4p) s1 = -s1;
	s2 = 1;
	for( n = 0 ; n < o2->ncenter; n++)
	{
	s3 = 1;
	if( o2->type == Or4p) s2 = -s2;
	for( o = 0 ; o < o3->ncenter; o++)
	{
	if( o3->type == Or4p) s3 = -s3;
	s4 = 1;
	for( p = 0 ; p < o4->ncenter; p++)
	{
	if( o4->type == Or4p) s4 = -s4;

	sign = s1*s2*s3*s4;
		x1 = o1->rx[m] -o4->rx[p];
		y1 = o1->ry[m] -o4->ry[p];
		z1 = o1->rz[m] -o4->rz[p];
		rab = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
		x1 = o2->rx[n] -o3->rx[o];
		y1 = o2->ry[n] -o3->ry[o];
		z1 = o2->rz[n] -o3->rz[o];
		rac = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
		x1 = o1->rx[m] -o3->rx[o];
		y1 = o1->ry[m] -o3->ry[o];
		z1 = o1->rz[m] -o3->rz[o];
		rbc = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
		x1 = o4->rx[p] -o2->rx[n];
		y1 = o4->ry[p] -o2->ry[n];
		z1 = o4->rz[p] -o2->rz[n];
		rcd = (x1*x1 + y1*y1 + z1*z1)*INVBOHR*INVBOHR;
	for( i=0; i< o1->n; i++)
	{
	for( j=0; j< o2->n; j++) 
	{
		for( k=0; k< o3->n; k++)
		{
		for( l=0; l< o4->n; l++) 
		{
/*  1,2  3,4 coloumb */
/* this is the correct form at least for 2 e- */

		s12 = o1->r[i] + o4->r[l];
		s34 = o3->r[k] + o2->r[j];
		sa4  = s12 + s34;
		r1 = o1->r[i]*o4->r[l]/( o1->r[i]+o4->r[l]);
		r2 = o3->r[k]*o2->r[j]/( o3->r[k]+o2->r[j]);
		x1 = (o1->r[i]*o1->rx[m] + o4->r[l]*o4->rx[p])/s12;
		y1 = (o1->r[i]*o1->ry[m] + o4->r[l]*o4->ry[p])/s12;
		z1 = (o1->r[i]*o1->rz[m] + o4->r[l]*o4->rz[p])/s12;
		x2 = (o3->r[k]*o3->rx[o] + o2->r[j]*o2->rx[n])/s34;
		y2 = (o3->r[k]*o3->ry[o] + o2->r[j]*o2->ry[n])/s34;
		z2 = (o3->r[k]*o3->rz[o] + o2->r[j]*o2->rz[n])/s34;
		x2 -= x1;
		y2 -= y1;
		z2 -= z1;
		rpq = (x2*x2 + y2*y2 + z2*z2)*INVBOHR*INVBOHR;
/* rab == 0 , rcd == 0 so just need rac or rbd or ... */
		x2 = rab*r1 + rac*r2;
		if( x2 < 70.)
		coloumb = coloumb + sign*
		norm_col*o1->a[i]*o2->a[j]*o3->a[k]*o4->a[l] *
		34.986837 /(s12*s34*sqrt(sa4))*Fzero( s12*s34/sa4*rpq)
		*exp( -r1*rab-r2*rac);
/* rab = rcd = 0. */
/*	if( o2 != o1 || o3 != o1) */
		if( no > 1 )
		{ /* minimum det is order 2 */
		s12 = o1->r[i] + o3->r[k];
		s34 = o4->r[l] + o2->r[j];
		sa4  = s12 + s34;
		x1 = (o1->r[i]*o1->rx[m] + o3->r[k]*o3->rx[o])/s12;
		y1 = (o1->r[i]*o1->ry[m] + o3->r[k]*o3->ry[o])/s12;
		z1 = (o1->r[i]*o1->rz[m] + o3->r[k]*o3->rz[o])/s12;
		x2 = (o4->r[l]*o4->rx[p] + o2->r[j]*o2->rx[n])/s34;
		y2 = (o4->r[l]*o4->ry[p] + o2->r[j]*o2->ry[n])/s34;
		z2 = (o4->r[l]*o4->rz[p] + o2->r[j]*o2->rz[n])/s34;
		x2 -= x1;
		y2 -= y1;
		z2 -= z1;
		rpq = (x2*x2 + y2*y2 + z2*z2)*INVBOHR*INVBOHR;
		r1 = o1->r[i]*o3->r[k]/( o1->r[i]+o3->r[k]);
		r2 = o4->r[l]*o2->r[j]/( o4->r[l]+o2->r[j]);
		x2 = rbc*r1 + rcd*r2;
		if( x2 < 70.)
		coloumb = coloumb - sign*
		0.5*norm_col*o1->a[i]*o2->a[j]*o3->a[k]*o4->a[l] *
		34.986837 /(s12*s34*sqrt(sa4))*Fzero( s12*s34/sa4*rpq)
		*exp( -r1*rbc-r2*rcd);

				}

		} /* l */
	}/*k */
		} /* j */
	} /* i */
	}}}} /* m,n,o,p loops */
SKIP:
	o2 = o2->next;
	} /* io2 */
	o3 = o3->next;
	} /* io3 */

/*
float total_ex,total_col,total_nuc,total_enuc,total_kinet,total_norm;
*/
	total_ex -= exchange;
	total_col += coloumb;
	return (coloumb  - exchange);
}


float AMMP_FASTCALL phiphi(ORBITAL *o1, ORBITAL *o2)
{
	int i,j,k,l;
	float x,y,z,r;
	float rab,rs,re;
	float overlap;
	float s1,s2,sign;
	overlap = 0.;

	s1 = 1.;
	for( k = 0; k < o1->ncenter; k++)
	{
	if( o1->type == Or4p) s1 = -s1;
	s2 = 1.;
	for( l = 0; l < o2->ncenter; l++)
	{
	if( o2->type == Or4p) s2 = -s2;
	sign = s1*s2;
	x = o2->rx[l] - o1->rx[k];
	y = o2->ry[l] - o1->ry[k];
	z = o2->rz[l] - o1->rz[k];
	rab = (x*x + y*y + z*z)*INVBOHR*INVBOHR;
	for( i=0; i< o1->n; i++)
	{
		for( j=0; j< o2->n; j++) 
		{
		rs = (o1->r[i] + o2->r[j]);
/* not the error
		if( rs < 1.e-6) rs = 1.e-6;
*/
		r = o1->r[i]*o2->r[j]/rs;
		r = r*rab;
		if( r > 70. ) r = 70.;
		if( r < -70.) r = -70.;
		re = exp(-r);
		overlap += sign*o1->a[i]*o2->a[j]*pow(PI/rs,1.5)*re;
		}
	}
	}} /* k,l */
	return overlap;
}


float AMMP_FASTCALL nuclear(void)
{
	AMMP_ATOM *a1,*a2;
	int i,j,natom;
	int l;
	float x,y,z,r,r2,r6,accum;
	float q1,q2;
	natom = a_number();
	if( natom < 2) return 0.;

	accum = 0.;
	a1 = a_next(-1);
	a1 = a1->next;
	for( i=1; i< natom ; i++)
	{
		q1 = a1->na;
/*
		if( q1 == 0 ) q1 = a1->q;
*/
		if( q1 != 0 ){
		for( j= 0; j< i; j++)
		{
		a2 = a_next(j);
		q2 = a2->na;
		if( q2 == 0.  ){ 
		q2 = a2->q;
		for(l=0; l< a1->dontuse; l++)
		{
			if(a2 == a1->excluded[l]) q2 = 0.;
		}
		x = a1->x - a2->x;
		y = a1->y - a2->y;
		z = a1->z - a2->z;
		r = 1./sqrt(x*x+y*y+z*z);
		r2 = r*r;
		r6 = r2*r2*r2;
		r2 = r6*r6;
		r = q1*q2*r * Q_CONST - a1->a*a2->a*r6 + a1->b*a2->b*r2 ;
/* accum = accum + r / 627.51; */
		accum = accum + r * 0.0015936001;
		} else {
		x = a1->x - a2->x;
		y = a1->y - a2->y;
		z = a1->z - a2->z;
		r = sqrt(x*x+y*y+z*z)*INVBOHR;
		accum = accum + q1*q2/r;
		}
		}
		}/* q1 != 0 */
		a1 = a1->next;
	}
	return accum;
}


float AMMP_FASTCALL Fzero(float x)
{
#ifdef notanyoldSGI
	float accum,etox;
	accum = 0.;
	etox = exp(-x);
/*
        accum = 1./(23.)*(2.*x*accum + etox);
        accum = 1./(21.)*(2.*x*accum + etox);
*/
        accum = 1./(19.)*(2.*x*accum + etox);
        accum = 1./(17.)*(2.*x*accum + etox);
        accum = 1./(15.)*(2.*x*accum + etox);
        accum = 1./(13.)*(2.*x*accum + etox);
        accum = 1./(11.)*(2.*x*accum + etox);
        accum = 1./( 9.)*(2.*x*accum + etox);
        accum = 1./( 7.)*(2.*x*accum + etox);
        accum = 1./( 5.)*(2.*x*accum + etox);
        accum = 1./( 3.)*(2.*x*accum + etox);
        accum = 2.*x*accum + etox;
	return accum;
#else
	if( x < 1.e-7) return 1.;
	x = sqrt(x);
	return .5*ROOTPI/x*erf(x);
	
#endif

}



/* go through the list and force each normal to be the current one */
/* since i can't imagine just doing this also do the orbital origin */

void AMMP_FASTCALL o_update_normals(void)
{
	ORBITAL *op,*op1;
	AMMP_ATOM *ap;
	int i,no;
	int j;
	float x,y,z,r;
	float x1,y1,z1,r1;
	float x2,y2,z2;
	
	no = o_number();
	if( no < 1) return;
	op = o_first();
	for( i=0; i< no; i++)
	{
/*	op = o_next(i);
*/
	if( op->gang != NULL && op->gang != op )
	{ /* if i'm ganged */
	op1 = op->gang;
	while(op1->gang != NULL ) op1 = op1->gang;
		op->x = op1->x;	
		op->y = op1->y;	
		op->z = op1->z;	
		op->along = op1->along;
		op->along2 = op1->along2;
		op->n = op1->n;
		for( j=0; j< op->n; j++)
		{
			op->rl[j] = op1->rl[j];
			op->a[j] = op1->a[j];
		}
	}

	for( j=0; j< op->n; j++)
		if( op->rl[j] < -6.) op->rl[j] = -6.;
/*
		if( op->rl[j] < -10.) op->rl[j] = -10.;
		if( op->rl[j] < -9.) op->rl[j] = -9.;
		if( op->rl[j] < -4.) op->rl[j] = -4.;
		if( op->rl[j] < -3.) op->rl[j] = -3.;
*/
	for( j=0 ; j < op->n; j++)
		op->r[j] = exp( op->rl[j]);

	switch( op->type)
	{
	case Or1:
	ap = op->myatom;
	op->rx[0] = ap->x;
	op->ry[0] = ap->y;
	op->rz[0] = ap->z;
	break;

	case Or1o:
	ap = op->myatom;
	op->rx[0] = ap->x + op->x;
	op->ry[0] = ap->y + op->y;
	op->rz[0] = ap->z + op->z;
	break;

	case Or2:  /* direction along a bond */
	ap = op->a1;
	x = ap->x;
	y = ap->y;
	z = ap->z;
	ap = op->myatom;
	x -= ap->x;
	y -= ap->y;
	z -= ap->z;
/*
	r = sqrt(x * x + y * y + z * z);
	if( r < 1.e-7 ) r = 1.;

	op->x = x/r;	
	op->y = y/r;	
	op->z = z/r;	
*/
	op->x = x;	
	op->y = y;	
	op->z = z;	
	if( op->along < -.5 ) op->along = -.5;
	if( op->along > 2.0 ) op->along = 2.0;
	op->rx[0] =   ap->x + op->along*op->x;
	op->ry[0] =   ap->y + op->along*op->y;
	op->rz[0] =   ap->z + op->along*op->z;
	break;

	case Or3: /* along a angle diagonal  and the cross product*/
	ap = op->a1;
	x = ap->x;
	y = ap->y;
	z = ap->z;
	ap = op->a2;
	x1 = ap->x;
	y1 = ap->y;
	z1 = ap->z;
	ap = op->myatom;
	x -= ap->x;
	y -= ap->y;
	z -= ap->z;
	x1 -= ap->x;
	y1 -= ap->y;
	z1 -= ap->z;
	r = sqrt(x*x + y*y + z*z);	
	r1 = sqrt(x1*x1 + y1*y1 + z1*z1);	
	if( r < 1.e-7) r = 1.;
	if( r1 < 1.e-7) r1 = 1.;
	op->x = (x/r +x1/r1)*ROOTHALF;
	op->y = (x/r +x1/r1)*ROOTHALF;
	op->z = (x/r +x1/r1)*ROOTHALF;
/*
	if( op->along > .6) op->along = .6;
	if( op->along < -.6) op->along = -.6;
*/
	op->rx[0] =   ap->x + op->along*op->x;
	op->ry[0] =   ap->y + op->along*op->y;
	op->rz[0] =   ap->z + op->along*op->z;
/* the cross product */
	x2 = y*z1 - z*y1; 
	y2 = z*x1 - x*z1;
	z2 = x*y1 - y*x1;
	r =  sqrt(x2*x2 + y2*y2 + z2*z2);
/*
	if( op->along2 > .4) op->along2 = .4;
	if( op->along2 < -.4) op->along2 = -.4;
*/
	op->rx[0] += op->along2*x2/r;
	op->ry[0] += op->along2*y2/r;
	op->rz[0] += op->along2*z2/r;
	break;

	case Or4s: /* perp to a plane */
	case Or4p: /* perp to a plane */
	if( op->type == Or4p && op->along < .05)  op->along = .05;
	ap = op->a2;
	x = ap->x;
	y = ap->y;
	z = ap->z;
	ap = op->a3;
	x1 = ap->x;
	y1 = ap->y;
	z1 = ap->z;
	ap = op->a1;
	x -= ap->x;
	y -= ap->y;
	z -= ap->z;
	x1 -= ap->x;
	y1 -= ap->y;
	z1 -= ap->z;
	x2 = y*z1 - z*y1; 
	y2 = z*x1 - x*z1;
	z2 = x*y1 - y*x1;
	r =  sqrt(x2*x2 + y2*y2 + z2*z2);
	if( r > 0.){
	op->x = x2/r;
	op->y = y2/r;
	op->z = z2/r;
	} else { 
	op->x = 0.;
	op->y = 0.;
	op->z = 0.;
	}
	ap = op->myatom;
	op->rx[0] =   ap->x + op->along*op->x;
	op->ry[0] =   ap->y + op->along*op->y;
	op->rz[0] =   ap->z + op->along*op->z;
	if( op->type == Or4p)
	{
	op->ncenter = 2;
	op->rx[1] =   ap->x - op->along*op->x;
	op->ry[1] =   ap->y - op->along*op->y;
	op->rz[1] =   ap->z - op->along*op->z;
	}

	break;
	case Orm:
		op->rx[0] = (op->myatom)->x;	
		op->ry[0] = (op->myatom)->y;	
		op->rz[0] = (op->myatom)->z;	
	if( op->ncenter >1 ){	
		op->rx[1] = (op->a1)->x;	
		op->ry[1] = (op->a1)->y;	
		op->rz[1] = (op->a1)->z;	
			}
	if( op->ncenter >2 ){	
		op->rx[2] = (op->a2)->x;	
		op->ry[2] = (op->a2)->y;	
		op->rz[2] = (op->a2)->z;	
			}
	if( op->ncenter >3 ){	
		op->rx[3] = (op->a3)->x;	
		op->ry[3] = (op->a3)->y;	
		op->rz[3] = (op->a3)->z;	
			}
	if( op->ncenter >4 ){	
		op->rx[4] = (op->a4)->x;	
		op->ry[4] = (op->a4)->y;	
		op->rz[4] = (op->a4)->z;	
			}
	if( op->ncenter >5 ){	
		op->rx[5] = (op->a5)->x;	
		op->ry[5] = (op->a5)->y;	
		op->rz[5] = (op->a5)->z;	
			}
	break;
	}
	op = op->next;
	}
/*
	renormalize();
*/

}

/* useful functions from atom.c */
/* function o_number()
* returns number of orbits defined
*  this is just orbitNUMBER if orbitUPDATE == 0
*  other wise just figure it out
*/

int AMMP_FASTCALL o_number(void)
{
        ORBITAL *op;
        if( orbitUPDATE )
        {
        orbitUPDATE = 0;
        orbitNUMBER = 0;
        if( firstORBITAL == NULL ) return 0 ;
        op = firstORBITAL;
        while(1)
        {
                if( op->next == NULL) break;
                orbitNUMBER++;
                if( op->next == op ) break;
                op = op->next;
        }
        }
        return orbitNUMBER;
}


/* function o_m_serial( serial )
* returns NULL on error or returns the address of the ORBITAL
* which matches serial
* cute?
*/

ORBITAL * AMMP_FASTCALL o_m_serial(int serial)
{
  int                   i, n;

  if (orbitUPDATE) n = o_number();
  else n = orbitNUMBER;

  o_m_serial_op = firstORBITAL;

  if (o_m_serial_op == NULL) return NULL;
  if (o_m_serial_lastmatched == NULL) o_m_serial_lastmatched = firstORBITAL;

  if (serial == o_m_serial_lastmatched -> osn) return o_m_serial_lastmatched;
  if (serial  > o_m_serial_lastmatched -> osn) o_m_serial_op = o_m_serial_lastmatched;

  for(i = 0; i < n; i++) {
    if (o_m_serial_op -> osn == serial) {
      o_m_serial_lastmatched = o_m_serial_op;
      return o_m_serial_op;
    }
    if (o_m_serial_op == o_m_serial_op->next) o_m_serial_op = firstORBITAL;
    else o_m_serial_op = o_m_serial_op->next;
  } /* End of for (i) */

  return NULL;
}


/* function o_next( flag )
* returns NULL on error or last orbital
* then steps to the next
* cute?
* flag <= 0 starts it off
*/

ORBITAL * AMMP_FASTCALL o_next(int flag)
{
  if (o_next_op == NULL) o_next_op = firstORBITAL;
  if (o_next_op == NULL) return NULL;
  if (flag <= 0) {
    o_next_op = firstORBITAL;
    return o_next_op;
  }
  if(o_next_op == o_next_op -> next) return NULL;
  o_next_op = o_next_op -> next;

  return o_next_op;
}

float AMMP_FASTCALL total_orbit_norm(void)
{
	float accum_over,accum_elec;
	float x;
	int no, i,j;
	ORBITAL *o1,*o2;

	no = o_number();
	if( no == 0 ) return 1.;
	accum_over = 0.;
	accum_elec = 0.;
	o1 = o_first();
	for( i = 0 ; i< no; i++)
	{
	if( o1->ipair == 2) accum_elec += 1.;
	accum_elec += 1.;
	for( j=0; j<no; j++)
	{ 
		o2 = o_next(j);
		x = phiphi(o1,o2);
		if( o1->ipair ==2) accum_over += x;
		accum_over += x;
	}
	o1 = o1->next;
	}

	return accum_elec/accum_over;
}
