
/*************************************************
****        AMMP - Non-bond potential         ****
**** Copyright 1992-2012, Robert W. Harrison  ****
****    VEGA edition by Alessandro Pedretti   ****
*************************************************/

/*
 * Integrated multipole method with the amortized
 * standard nonbonded program.
 * rectangular multipole expansion to the r^-6 order (5th order expansion)
 *
 * From Eyges "The Classical Electromagnetic Field"
 * note that we use the oposite convention for sign of
 * expansion so use + for all the cumulants, while he uses
 * -1^n.   This is solely due to choice of origin and for
 * other applications (rxn feild ) the -1^n is correct.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/*
#define  FOURTH
#define  FIFTH
*/

#ifdef FIFTH
#  define  FOURTH
#endif

/**** Global variables ****/

extern AMMP_ATOM **     NbAtomAll;
extern float *          NbVector;
extern AMMP_MMATOM *    NbAtomList;
extern int              NbAtomAllNum;
extern int *            NbIndexes;

/**** Update the non-bond parameters ****/

int AMMP_FASTCALL mm_fv_update_nonbon(float lambda)
{
  AMMP_ATOM **  Close;
  AMMP_ATOM     *ap, *a1, *a2;
  float         r, r0, xt, yt, zt;
  float         xt2 , xt3, yt2, yt3, zt2, zt3;
  float         k, k1, k2;
  float         ka2, kb2;
  float         c1, c2, c3; /* constants for the mm expansion */
  float *       VectorPtr;
  float         xmax, xmin, ymax, ymin, zmax, zmin;
  float         t1, t2, t3;
  int           CloseSize;
  int           i, ii, j, imax, inclose;
  int           nx, ny, nz;
  int           ix, iy, iz,inode;
  int           naybor[27];
  AMMP_MMNODE * nodelist;

#ifdef CUBIC
  float         k3, ka3, kb3;
#endif

#ifdef FOURTH
  float         c4;
  float         xt4, yt4, zt4;
#endif

#ifdef FIFTH
  float         c5;
#endif

  const char *  Routine       = "mm_fv_update_nonbon()";
  float         dielectric    = Q_CONST / GetDielectric();
  float         mmbox         = GetMmbox();
  float         mxcut         = GetMxcut();

  /* First check if anyone's moved and update the lists.
   * Note that this must be a look-ahead rather than
   * look back search because
   * we cannot update -> px until we've used that atom !!!
   */

  imax = a_number();
  for(ii = 0, VectorPtr = NbVector; ii < imax; ii++, VectorPtr += 4) {
    a1            = NbAtomAll[ii];
    VectorPtr[0] = a1 -> dx * lambda + a1 -> x ;
    VectorPtr[1] = a1 -> dy * lambda + a1 -> y;
    VectorPtr[2] = a1 -> dz * lambda + a1 -> z;
  } /* End of for (ii) */

  /**** Determine the bounds of box which surrounds all of the atoms ****/

  xmax = -10e10;
  ymax = -10e10;
  zmax = -10e10;
  xmin =  10e10;
  ymin =  10e10;
  zmin =  10e10;
  for(ii = 0, VectorPtr = NbVector; ii < imax; ii++, VectorPtr += 4) {
    if (xmax < VectorPtr[0]) xmax = VectorPtr[0];
    if (ymax < VectorPtr[1]) ymax = VectorPtr[1];
    if (zmax < VectorPtr[2]) zmax = VectorPtr[2];
    if (xmin > VectorPtr[0]) xmin = VectorPtr[0];
    if (ymin > VectorPtr[1]) ymin = VectorPtr[1];
    if (zmin > VectorPtr[2]) zmin = VectorPtr[2];
  } /* End of for (ii) */

  nx = (xmax - xmin) / mmbox + 1;
  ny = (ymax - ymin) / mmbox + 1;
  nz = (zmax - zmin) / mmbox + 1;

  /**** Now try to allocate the mmnodes ****/

  if ((nodelist = (AMMP_MMNODE *)Alloca(nx * ny * nz * sizeof(AMMP_MMNODE), Routine)) == NULL) {
    aaerror("mmbox %f nx %d ny %d nz %d\n"
            "xmin xmax %f %f ymin ymax %f %f zmin zmax %f %f",
            mmbox, nx, ny, nz,
            xmin, xmax, ymin, ymax, zmin, zmax);
    mmbox = mmbox * 2.0f;
    set_f_variable("mmbox", mmbox);
    nx = (xmax - xmin) / mmbox + 1;
    ny = (ymax - ymin) / mmbox + 1;
    nz = (zmax - zmin) / mmbox + 1;
    if ((nodelist = (AMMP_MMNODE *)Alloca(nx * ny * nz * sizeof(AMMP_MMNODE), Routine)) == NULL) {
      NonBondFreeTable();
      return FALSE;
    }
  }

  for(ix = 0; ix < nx; ix++)
    for(iy = 0; iy < ny; iy++)
      for(iz = 0; iz < nz; iz++) {
  inode              = ((iz * ny) + iy) * nx + ix;
  nodelist[inode].xc = ix * mmbox + 0.5f * mmbox + xmin;
  nodelist[inode].yc = iy * mmbox + 0.5f * mmbox + ymin;
  nodelist[inode].zc = iz * mmbox + 0.5f * mmbox + zmin;
      } /* End of for (iz) */

  for(ii = 0; ii < nx * ny * nz; ii++) {
    nodelist[ii].sqp  = zero;
    nodelist[ii].sa   = zero;
    nodelist[ii].xa   = zero;
    nodelist[ii].ya   = zero;
    nodelist[ii].za   = zero;
    nodelist[ii].q100 = zero;
    nodelist[ii].q010 = zero;
    nodelist[ii].q001 = zero;
    nodelist[ii].q200 = zero;
    nodelist[ii].q020 = zero;
    nodelist[ii].q002 = zero;
    nodelist[ii].q101 = zero;
    nodelist[ii].q110 = zero;
    nodelist[ii].q011 = zero;
    nodelist[ii].q300 = zero;
    nodelist[ii].q030 = zero;
    nodelist[ii].q003 = zero;
    nodelist[ii].q210 = zero;
    nodelist[ii].q120 = zero;
    nodelist[ii].q201 = zero;
    nodelist[ii].q102 = zero;
    nodelist[ii].q021 = zero;
    nodelist[ii].q012 = zero;
    nodelist[ii].q111 = zero;
#ifdef FOURTH
    nodelist[ii].q400 = zero;
    nodelist[ii].q040 = zero;
    nodelist[ii].q004 = zero;
    nodelist[ii].q310 = zero;
    nodelist[ii].q130 = zero;
    nodelist[ii].q301 = zero;
    nodelist[ii].q103 = zero;
    nodelist[ii].q031 = zero;
    nodelist[ii].q013 = zero;
    nodelist[ii].q220 = zero;
    nodelist[ii].q202 = zero;
    nodelist[ii].q022 = zero;
    nodelist[ii].q211 = zero;
    nodelist[ii].q121 = zero;
    nodelist[ii].q112 = zero;
#endif
#ifdef FIFTH
    nodelist[ii].q500 = zero;
    nodelist[ii].q050 = zero;
    nodelist[ii].q005 = zero;
    nodelist[ii].q410 = zero;
    nodelist[ii].q140 = zero;
    nodelist[ii].q401 = zero;
    nodelist[ii].q104 = zero;
    nodelist[ii].q041 = zero;
    nodelist[ii].q014 = zero;
    nodelist[ii].q320 = zero;
    nodelist[ii].q230 = zero;
    nodelist[ii].q302 = zero;
    nodelist[ii].q203 = zero;
    nodelist[ii].q032 = zero;
    nodelist[ii].q023 = zero;
    nodelist[ii].q221 = zero;
    nodelist[ii].q212 = zero;
    nodelist[ii].q122 = zero;
    nodelist[ii].q311 = zero;
    nodelist[ii].q131 = zero;
    nodelist[ii].q113 = zero;
#endif
    nodelist[ii].first  = -1;
    nodelist[ii].last   = -1;
    nodelist[ii].innode = 0;
  } /* End of for (ii) */

  /**** Now decide for each atom who he belongs to ****/

  for(ii = 0, VectorPtr = NbVector; ii < imax; ii++, VectorPtr += 4) {
    ix    = (VectorPtr[0] - xmin) / mmbox;
    iy    = (VectorPtr[1] - ymin) / mmbox;
    iz    = (VectorPtr[2] - zmin) / mmbox;
    inode = ((iz * ny) + iy) * nx + ix;
    NbAtomList[ii].which = inode;
  } /* End of for (ii) */

  /**** Generate the links ****/

  for(inode = 0; inode < nx*ny*nz; inode++) {

    /**** First find the first atom which belongs to me ****/

    for(ii = 0; ii < imax; ii++) {
      if (NbAtomList[ii].which == inode) {
        nodelist[inode].first   = ii;
        nodelist[inode].last    = ii;
  nodelist[inode].innode += 1;
        ap = NbAtomList[ii].who;
        break;
      }
    } /* End of for (ii) */

    /**** Only if i'm not null ****/

    if (ii != imax) {
      for (ii = nodelist[inode].first; ii < imax; ii++) {
        if (NbAtomList[ii].which == inode) {
          NbAtomList[nodelist[inode].last].next = ii;
          nodelist[inode].last                   = ii;
    nodelist[inode].innode                += 1;
          ap                                     = NbAtomList[ii].who;
          xt                    = ap -> x + lambda * ap -> dx - nodelist[inode].xc;
    yt                    = ap -> y + lambda * ap -> dy - nodelist[inode].yc;
    zt                    = ap -> z + lambda * ap -> dz - nodelist[inode].zc;
          nodelist[inode].sqp  += ap-> q;
    nodelist[inode].sa   += ap-> a;
    nodelist[inode].xa   += ap-> a * xt;
    nodelist[inode].ya   += ap-> a * yt;
    nodelist[inode].za   += ap-> a * zt;
    xt2                   = xt  * xt;
    xt3                   = xt2 * xt;
#ifdef FOURTH
    xt4                   = xt3 * xt;
#endif
    yt2                   = yt  * yt;
    yt3                   = yt2 * yt;
#ifdef FOURTH
    yt4                   = yt3 * yt;
#endif
    zt2                   = zt  * zt;
    zt3                   = zt2 * zt;
#ifdef FOURTH
    zt4                   = zt3 * zt;
#endif
          nodelist[inode].q100 += ap -> q * xt;
          nodelist[inode].q010 += ap -> q * yt;
          nodelist[inode].q001 += ap -> q * zt;
          nodelist[inode].q200 += ap -> q * xt2;
          nodelist[inode].q020 += ap -> q * yt2;
          nodelist[inode].q002 += ap -> q * zt2;
          nodelist[inode].q101 += ap -> q * xt * zt;
          nodelist[inode].q110 += ap -> q * xt * yt;
          nodelist[inode].q011 += ap -> q * yt * zt;
          nodelist[inode].q300 += ap -> q * xt3;
          nodelist[inode].q030 += ap -> q * yt3;
          nodelist[inode].q003 += ap -> q * zt3;
          nodelist[inode].q210 += ap -> q * xt2 * yt;
          nodelist[inode].q120 += ap -> q * xt  * yt2;
          nodelist[inode].q201 += ap -> q * xt2 * zt;
          nodelist[inode].q102 += ap -> q * xt  * zt2;
          nodelist[inode].q021 += ap -> q * yt2 * zt;
          nodelist[inode].q012 += ap -> q * yt  * zt2;
          nodelist[inode].q111 += ap -> q * xt * yt * zt;
#ifdef FOURTH
          nodelist[inode].q400 += ap -> q * xt4;
          nodelist[inode].q040 += ap -> q * yt4;
          nodelist[inode].q004 += ap -> q * zt4;
          nodelist[inode].q310 += ap -> q * xt3 * yt;
          nodelist[inode].q130 += ap -> q * xt  * yt3;
          nodelist[inode].q301 += ap -> q * xt3 * zt;
          nodelist[inode].q103 += ap -> q * xt  * zt3;
          nodelist[inode].q031 += ap -> q * yt3 * zt;
          nodelist[inode].q013 += ap -> q * yt  * zt3;
          nodelist[inode].q220 += ap -> q * xt2 * yt2;
          nodelist[inode].q202 += ap -> q * xt2 * zt2;
          nodelist[inode].q022 += ap -> q * yt2 * zt2;
          nodelist[inode].q211 += ap -> q * xt2 * yt  * zt;
          nodelist[inode].q121 += ap -> q * xt  * yt2 * zt;
          nodelist[inode].q112 += ap -> q * xt  * yt  * zt2;
#endif
#ifdef FIFTH
          nodelist[inode].q500 += ap -> q * xt4 * xt;
          nodelist[inode].q050 += ap -> q * yt4 * yt;
          nodelist[inode].q005 += ap -> q * zt4 * zt;
          nodelist[inode].q410 += ap -> q * xt4 * yt;
          nodelist[inode].q140 += ap -> q * yt4 * xt;
          nodelist[inode].q401 += ap -> q * xt4 * zt;
          nodelist[inode].q104 += ap -> q * zt4 * xt;
          nodelist[inode].q041 += ap -> q * yt4 * zt;
          nodelist[inode].q014 += ap -> q * zt4 * yt;
          nodelist[inode].q320 += ap -> q * xt3 * yt2;
          nodelist[inode].q230 += ap -> q * yt3 * xt2;
          nodelist[inode].q302 += ap -> q * xt3 * zt2;
          nodelist[inode].q203 += ap -> q * zt3 * xt2;
          nodelist[inode].q032 += ap -> q * yt3 * zt2;
          nodelist[inode].q023 += ap -> q * zt3 * yt2;
          nodelist[inode].q221 += ap -> q * xt2 * yt2 * zt;
          nodelist[inode].q212 += ap -> q * xt2 * yt  * zt2;
          nodelist[inode].q122 += ap -> q * xt  * yt2 * zt2;
          nodelist[inode].q311 += ap -> q * xt3 * yt  * zt;
          nodelist[inode].q131 += ap -> q * xt  * yt3 * zt;
          nodelist[inode].q113 += ap -> q * xt  * yt  * zt3;
#endif
        }
      } /* End of for (ii) */
    }
  } /* End of for (inode) */

  /**** Now (almost done with the MM setup) normalize the accumulated nodal data ****/


  k  = dielectric * 0.5f; /* multiplied by .5 to correct for double counting */
  xt = 0.5f / 3.0f;
  yt = xt / 4.0f;
  zt = yt / 5.0f;
  for(ii = 0; ii < nx * ny * nz; ii++) {
    nodelist[ii].sqp  *= k;
    nodelist[ii].q100 *= k;
    nodelist[ii].q010 *= k;
    nodelist[ii].q001 *= k;
    nodelist[ii].q200 *= 0.5f * k;
    nodelist[ii].q020 *= 0.5f * k;
    nodelist[ii].q002 *= 0.5f * k;
    nodelist[ii].q101 *= k;
    nodelist[ii].q110 *= k;
    nodelist[ii].q011 *= k;
    nodelist[ii].q300 *= xt * k;
    nodelist[ii].q030 *= xt * k;
    nodelist[ii].q003 *= xt * k;
    nodelist[ii].q210 *= 0.5f * k;
    nodelist[ii].q120 *= 0.5f * k;
    nodelist[ii].q201 *= 0.5f * k;
    nodelist[ii].q102 *= 0.5f * k;
    nodelist[ii].q021 *= 0.5f * k;
    nodelist[ii].q012 *= 0.5f * k;
    nodelist[ii].q111 *= k;
#ifdef FOURTH
    nodelist[ii].q400 *= yt * k;
    nodelist[ii].q040 *= yt * k;
    nodelist[ii].q004 *= yt * k;
    nodelist[ii].q310 *= xt * k;
    nodelist[ii].q130 *= xt * k;
    nodelist[ii].q301 *= xt * k;
    nodelist[ii].q103 *= xt * k;
    nodelist[ii].q031 *= xt * k;
    nodelist[ii].q013 *= xt * k;
    nodelist[ii].q220 *= 0.25f * k;
    nodelist[ii].q202 *= 0.25f * k;
    nodelist[ii].q022 *= 0.25f * k;
    nodelist[ii].q211 *= 0.5f * k;
    nodelist[ii].q121 *= 0.5f * k;
    nodelist[ii].q112 *= 0.5f * k;
#endif
#ifdef FIFTH
    nodelist[ii].q500 *= zt * k;
    nodelist[ii].q050 *= zt * k;
    nodelist[ii].q005 *= zt * k;
    nodelist[ii].q410 *= yt * k;
    nodelist[ii].q140 *= yt * k;
    nodelist[ii].q401 *= yt * k;
    nodelist[ii].q104 *= yt * k;
    nodelist[ii].q041 *= yt * k;
    nodelist[ii].q014 *= yt * k;
    nodelist[ii].q320 *= 0.5f * xt * k;
    nodelist[ii].q230 *= 0.5f * xt * k;
    nodelist[ii].q302 *= 0.5f * xt * k;
    nodelist[ii].q203 *= 0.5f * xt * k;
    nodelist[ii].q032 *= 0.5f * xt * k;
    nodelist[ii].q023 *= 0.5f * xt * k;
    nodelist[ii].q221 *= 0.25f * k;
    nodelist[ii].q212 *= 0.25f * k;
    nodelist[ii].q122 *= 0.25f * k;
    nodelist[ii].q311 *= xt * k;
    nodelist[ii].q131 *= xt * k;
    nodelist[ii].q113 *= xt * k;
#endif

    if (nodelist[ii].sa) {
      nodelist[ii].xa = nodelist[ii].xa / nodelist[ii].sa;
      nodelist[ii].ya = nodelist[ii].ya / nodelist[ii].sa;
      nodelist[ii].za = nodelist[ii].za / nodelist[ii].sa;
    }
    nodelist[ii].xa += nodelist[ii].xc;
    nodelist[ii].ya += nodelist[ii].yc;
    nodelist[ii].za += nodelist[ii].zc;

    /**** Correct for double counting */

    nodelist[ii].sa *= 0.5f;
  } /* End of for (ii) */

  CloseSize = NCLOSE;
  if ((Close = (AMMP_ATOM **)Alloca(sizeof(AMMP_ATOM *) * CloseSize, Routine)) == NULL)
    return FALSE;

  /**** Initialize the data for every atom ****/

  for(ii = 0; ii < imax; ii++) {
    a1 = NbAtomAll[ii];
    a1 -> px  = a1 -> x + lambda * a1 -> dx;
    a1 -> py  = a1 -> y + lambda * a1 -> dy;
    a1 -> pz  = a1 -> z + lambda * a1 -> dz;
    a1 -> VP  = zero;
    a1 -> dpx = zero;
    a1 -> dpy = zero;
    a1 -> dpz = zero;
    a1 -> qxx = zero;
    a1 -> qxy = zero;
    a1 -> qxz = zero;
    a1 -> qyy = zero;
    a1 -> qyz = zero;
    a1 -> qzz = zero;

    /**** Clear the forces of inactive atoms ****/

    if (!a1 -> active) {
      a1 -> fx = zero;
      a1 -> fy = zero;
      a1 -> fz = zero;
      a1 -> fw = zero;
    }

#ifdef CUBIC
    a1 -> qxxx = zero;
    a1 -> qxxy = zero;
    a1 -> qxxz = zero;
    a1 -> qxyy = zero;
    a1 -> qxyz = zero;
    a1 -> qxzz = zero;
    a1 -> qyyy = zero;
    a1 -> qyyz = zero;
    a1 -> qyzz = zero;
    a1 -> qzzz = zero;
#endif
  } /* End of for (ii) */

  for(ii = 0; ii < imax; ii++) {
    a1 = NbAtomList[ii].who;
    inclose = 0;

    /*
     * Loop over the nodes and if the node is mine or a neighbor then use an
     * explicit summation otherwise use the MM node.
     */

    ix = (a1 -> px - xmin) / mmbox;
    iy = (a1 -> py - ymin) / mmbox;
    iz = (a1 -> pz - zmin) / mmbox;
    naybor[0 ] = ((iz * ny) + iy) * nx + ix;
    naybor[1 ] = ((iz * ny) + iy) * nx + ix + 1;
    naybor[2 ] = ((iz * ny) + iy) * nx + ix - 1;
    naybor[3 ] = ((iz * ny) + iy) * nx +nx + ix;
    naybor[4 ] = ((iz * ny) + iy) * nx - nx + ix;
    naybor[5 ] = ((iz * ny) + iy) * nx + nx + ix + 1;
    naybor[6 ] = ((iz * ny) + iy) * nx + nx + ix - 1;
    naybor[7 ] = ((iz * ny) + iy) * nx - nx + ix + 1;
    naybor[8 ] = ((iz * ny) + iy) * nx - nx + ix - 1;
    naybor[9 ] = ((iz * ny) + ny + iy) * nx + ix;
    naybor[10] = ((iz * ny) + ny + iy) * nx + ix+1;
    naybor[11] = ((iz * ny) + ny + iy) * nx + ix-1;
    naybor[12] = ((iz * ny) + ny + iy) * nx + nx + ix;
    naybor[13] = ((iz * ny) + ny + iy) * nx - nx + ix;
    naybor[14] = ((iz * ny) + ny + iy) * nx + nx + ix + 1;
    naybor[15] = ((iz * ny) + ny + iy) * nx + nx + ix - 1;
    naybor[16] = ((iz * ny) + ny + iy) * nx - nx + ix + 1;
    naybor[17] = ((iz * ny) + ny + iy) * nx - nx + ix - 1;
    naybor[18] = ((iz * ny) - ny + iy) * nx + ix;
    naybor[19] = ((iz * ny) - ny + iy) * nx + ix + 1;
    naybor[20] = ((iz * ny) - ny + iy) * nx + ix - 1;
    naybor[21] = ((iz * ny) - ny + iy) * nx + nx + ix;
    naybor[22] = ((iz * ny) - ny + iy) * nx - nx + ix;
    naybor[23] = ((iz * ny) - ny + iy) * nx + nx + ix + 1;
    naybor[24] = ((iz * ny) - ny + iy) * nx + nx + ix - 1;
    naybor[25] = ((iz * ny) - ny + iy) * nx - nx + ix + 1;
    naybor[26] = ((iz * ny) - ny + iy) * nx - nx + ix - 1;

    for(inode = 0; inode < nx * ny * nz; inode++) {
      for(j = 0; j < 27; j++)
        if (inode == naybor[j]) break;
      if (j == 27 ) {
  if (nodelist[inode].innode > 0) {

          /*
           * Now do the multipole expansion for the electrostatic terms
           * note that dielectric is included in the multipole expansion
           */

  xt         = nodelist[inode].xc - a1 -> px;
  yt         = nodelist[inode].yc - a1 -> py;
  zt         = nodelist[inode].zc - a1 -> pz;
  r          = one / (xt * xt + yt * yt + zt * zt);
  r0         = sqrt(r);
  c1         =  -r * r0;
  c2         = -three * c1 * r;
  c3         = -five  * c2 * r;

#ifdef FOURTH
  c4         = -seven * c3 * r;
#endif

#ifdef FIFTH
  c5         = -nine  * c4 * r;
#endif
  xt2        = xt  * xt;
  xt3        = xt2 * xt;
#ifdef FOURTH
  xt4        = xt3 * xt;
#endif
  yt2        = yt  * yt;
  yt3        = yt2 * yt;
#ifdef FOURTH
  yt4        = yt3 * yt;
#endif
  zt2        = zt  * zt;
  zt3        = zt2 * zt;
#ifdef FOURTH
  zt4        = zt3 * zt;
#endif

  a1 -> VP  += nodelist[inode].sqp * a1 -> q * r0;
  k          = c1 * a1 -> q * xt;
  a1 -> VP  += k * nodelist[inode].q100;
  a1 -> dpx += k * nodelist[inode].sqp;
  k          = c1 * a1 -> q * yt;
  a1 -> VP  += k * nodelist[inode].q010;
  a1 -> dpy += k * nodelist[inode].sqp;
  k          = c1 * a1 -> q * zt;
  a1 -> VP  += k * nodelist[inode].q001;
  a1 -> dpz += k * nodelist[inode].sqp;

        /**** n = 2 ****/

  k          = (c2 * xt2 + c1) * a1 -> q;
  a1 -> VP  += k * nodelist[inode].q200;
  a1 -> dpx += k * nodelist[inode].q100;
  a1 -> qxx += k * nodelist[inode].sqp;
  k          = (c2 * yt2 + c1) * a1 -> q;
  a1 -> VP  += k * nodelist[inode].q020;
  a1 -> dpy += k * nodelist[inode].q010;
  a1 -> qyy += k * nodelist[inode].sqp;
  k = (c2*zt2 +c1) * a1 -> q;
  a1->VP += k*nodelist[inode].q002;
  a1->dpz += k*nodelist[inode].q001;
  a1->qzz += k*nodelist[inode].sqp;
  k = c2*xt*yt*a1->q;
  a1->VP += k*nodelist[inode].q110;
  a1->dpx += k*nodelist[inode].q010;
  a1->dpy += k*nodelist[inode].q100;
  a1->qxy += k*nodelist[inode].sqp;
  k = c2*xt*zt*a1->q;
  a1->VP += k*nodelist[inode].q101;
  a1->dpx += k*nodelist[inode].q001;
  a1->dpz += k*nodelist[inode].q100;
  a1->qxz += k*nodelist[inode].sqp;
  k = c2*yt*zt*a1->q;
  a1->VP += k*nodelist[inode].q011;
  a1->dpy += k*nodelist[inode].q001;
  a1->dpz += k*nodelist[inode].q010;
  a1->qyz += k*nodelist[inode].sqp;
/* n=3 */
  k = (c3*xt3 +3*c2*xt)*a1->q;
  a1->VP += k*nodelist[inode].q300;
  a1->dpx += k*nodelist[inode].q200;
  a1->qxx += k*nodelist[inode].q100;
  k = (c3*yt3 +3*c2*yt)*a1->q;
  a1->VP += k*nodelist[inode].q030;
  a1->dpy += k*nodelist[inode].q020;
  a1->qyy += k*nodelist[inode].q010;
  k = (c3*zt3 +3*c2*zt)*a1->q;
  a1->VP += k*nodelist[inode].q003;
  a1->dpz += k*nodelist[inode].q002;
  a1->qzz += k*nodelist[inode].q001;
  k = (c3*xt2*yt+c2*yt)*a1->q;
  a1->VP += k*nodelist[inode].q210;
  a1->dpx += k*nodelist[inode].q110;
  a1->dpy += k*nodelist[inode].q200;
  a1->qxx += k*nodelist[inode].q010;
  a1->qxy += k*nodelist[inode].q100;
  k = (c3*yt2*xt+c2*xt)*a1->q;
  a1->VP += k*nodelist[inode].q120;
  a1->dpx += k*nodelist[inode].q020;
  a1->dpy += k*nodelist[inode].q110;
  a1->qyy += k*nodelist[inode].q100;
  a1->qxy += k*nodelist[inode].q010;
  k = (c3*xt2*zt+c2*zt)*a1->q;
  a1->VP += k*nodelist[inode].q201;
  a1->dpx += k*nodelist[inode].q101;
  a1->dpz += k*nodelist[inode].q200;
  a1->qxx += k*nodelist[inode].q001;
  a1->qxz += k*nodelist[inode].q100;
  k = (c3*zt2*xt+c2*xt)*a1->q;
  a1->VP += k*nodelist[inode].q102;
  a1->dpx += k*nodelist[inode].q002;
  a1->dpz += k*nodelist[inode].q101;
  a1->qzz += k*nodelist[inode].q100;
  a1->qxz += k*nodelist[inode].q001;
  k = (c3*yt2*zt+c2*zt)*a1->q;
  a1->VP += k*nodelist[inode].q021;
  a1->dpy += k*nodelist[inode].q011;
  a1->dpz += k*nodelist[inode].q020;
  a1->qyy += k*nodelist[inode].q001;
  a1->qyz += k*nodelist[inode].q010;
  k = (c3*zt2*yt+c2*yt)*a1->q;
  a1->VP += k*nodelist[inode].q012;
  a1->dpy += k*nodelist[inode].q002;
  a1->dpz += k*nodelist[inode].q011;
  a1->qzz += k*nodelist[inode].q010;
  a1->qyz += k*nodelist[inode].q001;
  k = (c3*zt*yt*xt)*a1->q;
  a1->VP += k*nodelist[inode].q111;
  a1->dpx += k*nodelist[inode].q011;
  a1->dpy += k*nodelist[inode].q101;
  a1->dpz += k*nodelist[inode].q110;
/* n=4 */
#ifdef FOURTH
  k = (c4*xt4 +six*c3*(xt2) +three*c2)*a1->q;
  a1->VP += k*nodelist[inode].q400;
  a1->dpx += k*nodelist[inode].q300;
  a1->qxx += k*nodelist[inode].q200;
  k = (c4*yt4 +six*c3*(yt2) +three*c2)*a1->q;
  a1->VP += k*nodelist[inode].q040;
  a1->dpy += k*nodelist[inode].q030;
  a1->qyy += k*nodelist[inode].q020;
  k = (c4*zt4 +six*c3*(zt2) +three*c2)*a1->q;
  a1->VP += k*nodelist[inode].q004;
  a1->dpz += k*nodelist[inode].q003;
  a1->qzz += k*nodelist[inode].q002;
  k = (c4*xt3*yt + three*c3*xt*yt)*a1->q;
  a1->VP += k*nodelist[inode].q310;
  a1->dpx += k*nodelist[inode].q210;
  a1->dpy += k*nodelist[inode].q300;
  a1->qxx += k*nodelist[inode].q110;
  a1->qxy += k*nodelist[inode].q200;
  k = (c4*yt3*xt + three*c3*xt*yt)*a1->q;
  a1->VP += k*nodelist[inode].q130;
  a1->dpx += k*nodelist[inode].q030;
  a1->dpy += k*nodelist[inode].q120;
  a1->qyy += k*nodelist[inode].q110;
  a1->qxy += k*nodelist[inode].q020;
  k = (c4*xt3*zt + three*c3*xt*zt)*a1->q;
  a1->VP += k*nodelist[inode].q301;
  a1->dpx += k*nodelist[inode].q201;
  a1->dpz += k*nodelist[inode].q300;
  a1->qxx += k*nodelist[inode].q101;
  a1->qxz += k*nodelist[inode].q200;
  k = (c4*zt3*yt + three*c3*xt*yt)*a1->q;
  a1->VP += k*nodelist[inode].q103;
  a1->dpx += k*nodelist[inode].q003;
  a1->dpz += k*nodelist[inode].q102;
  a1->qzz += k*nodelist[inode].q101;
  a1->qxz += k*nodelist[inode].q002;
  k = (c4*yt3*zt + three*c3*zt*yt)*a1->q;
  a1->VP += k*nodelist[inode].q031;
  a1->dpz += k*nodelist[inode].q030;
  a1->dpy += k*nodelist[inode].q021;
  a1->qyy += k*nodelist[inode].q011;
  a1->qyz += k*nodelist[inode].q020;
  k = (c4*zt3*yt + three*c3*zt*yt)*a1->q;
  a1->VP += k*nodelist[inode].q013;
  a1->dpz += k*nodelist[inode].q012;
  a1->dpy += k*nodelist[inode].q003;
  a1->qzz += k*nodelist[inode].q011;
  a1->qyz += k*nodelist[inode].q002;
  k = (c4*xt2*yt2 + c3*(xt2+yt2) +c2)*a1->q;
  a1->VP += k*nodelist[inode].q220;
  a1->dpx += k*nodelist[inode].q120;
  a1->dpy += k*nodelist[inode].q210;
  a1->qxx += k*nodelist[inode].q020;
  a1->qyy += k*nodelist[inode].q200;
  a1->qxy += k*nodelist[inode].q110;
  k = (c4*xt2*zt2 + c3*(xt2+zt2) +c2)*a1->q;
  a1->VP += k*nodelist[inode].q202;
  a1->dpx += k*nodelist[inode].q102;
  a1->dpz += k*nodelist[inode].q201;
  a1->qxx += k*nodelist[inode].q002;
  a1->qzz += k*nodelist[inode].q200;
  a1->qxz += k*nodelist[inode].q101;
  k = (c4*zt2*yt2 + c3*(zt2+yt2) +c2)*a1->q;
  a1->VP += k*nodelist[inode].q022;
  a1->dpz += k*nodelist[inode].q021;
  a1->dpy += k*nodelist[inode].q012;
  a1->qzz += k*nodelist[inode].q020;
  a1->qyy += k*nodelist[inode].q002;
  a1->qyz += k*nodelist[inode].q011;
  k = (c4*xt2*yt*zt +c3*yt*zt)*a1->q;
  a1->VP += k*nodelist[inode].q211;
  a1->dpz += k*nodelist[inode].q210;
  a1->dpy += k*nodelist[inode].q201;
  a1->dpx += k*nodelist[inode].q111;
  a1->qxx += k*nodelist[inode].q011;
  a1->qxy += k*nodelist[inode].q101;
  a1->qyz += k*nodelist[inode].q200;
  a1->qxz += k*nodelist[inode].q110;
  k = (c4*xt*yt2*zt +c3*xt*zt)*a1->q;
  a1->VP += k*nodelist[inode].q121;
  a1->dpz += k*nodelist[inode].q120;
  a1->dpy += k*nodelist[inode].q111;
  a1->dpx += k*nodelist[inode].q021;
  a1->qyy += k*nodelist[inode].q101;
  a1->qxy += k*nodelist[inode].q011;
  a1->qyz += k*nodelist[inode].q110;
  a1->qxz += k*nodelist[inode].q020;
  k = (c4*xt*yt*zt2 +c3*yt*xt)*a1->q;
  a1->VP += k*nodelist[inode].q112;
  a1->dpz += k*nodelist[inode].q111;
  a1->dpy += k*nodelist[inode].q102;
  a1->dpx += k*nodelist[inode].q012;
  a1->qzz += k*nodelist[inode].q110;
  a1->qxy += k*nodelist[inode].q002;
  a1->qxz += k*nodelist[inode].q011;
  a1->qyz += k*nodelist[inode].q101;
#endif
/* n=5 */
#ifdef FIFTH
  k = ((c5*xt+9*c4)*xt4  +15*c3*xt)*a1->q;
  a1->VP += k*nodelist[inode].q500;
  a1->dpx += k*nodelist[inode].q400;
  a1->qxx += k*nodelist[inode].q300;
  k = ((c5*yt+9*c4)*yt4  +15*c3*yt)*a1->q;
  a1->VP += k*nodelist[inode].q050;
  a1->dpy += k*nodelist[inode].q040;
  a1->qyy += k*nodelist[inode].q030;
  k = ((c5*zt+9*c4)*zt4  +15*c3*zt)*a1->q;
  a1->VP += k*nodelist[inode].q005;
  a1->dpz += k*nodelist[inode].q004;
  a1->qzz += k*nodelist[inode].q003;
  k = (c5*xt4+six*c4*xt2 +three*c3)*yt*a1->q;
  a1->VP += k*nodelist[inode].q410;
  a1->dpx += k*nodelist[inode].q310;
  a1->dpy += k*nodelist[inode].q400;
  a1->qxx += k*nodelist[inode].q210;
  a1->qxy += k*nodelist[inode].q300;
  k = (c5*yt4+six*c4*yt2 +three*c3)*xt*a1->q;
  a1->VP += k*nodelist[inode].q140;
  a1->dpx += k*nodelist[inode].q040;
  a1->dpy += k*nodelist[inode].q130;
  a1->qyy += k*nodelist[inode].q120;
  a1->qxy += k*nodelist[inode].q030;
  k = (c5*xt4+six*c4*xt2 +three*c3)*zt*a1->q;
  a1->VP += k*nodelist[inode].q401;
  a1->dpx += k*nodelist[inode].q301;
  a1->dpz += k*nodelist[inode].q400;
  a1->qxx += k*nodelist[inode].q201;
  a1->qxz += k*nodelist[inode].q300;
  k = (c5*zt4+six*c4*zt2 +three*c3)*xt*a1->q;
  a1->VP += k*nodelist[inode].q104;
  a1->dpx += k*nodelist[inode].q004;
  a1->dpz += k*nodelist[inode].q103;
  a1->qzz += k*nodelist[inode].q102;
  a1->qxz += k*nodelist[inode].q003;
  k = (c5*yt4+six*c4*yt2 +three*c3)*zt*a1->q;
  a1->VP += k*nodelist[inode].q041;
  a1->dpy += k*nodelist[inode].q031;
  a1->dpz += k*nodelist[inode].q040;
  a1->qyy += k*nodelist[inode].q021;
  a1->qyz += k*nodelist[inode].q030;
  k = (c5*zt4+six*c4*zt2 +three*c3)*yt*a1->q;
  a1->VP += k*nodelist[inode].q014;
  a1->dpy += k*nodelist[inode].q004;
  a1->dpz += k*nodelist[inode].q013;
  a1->qzz += k*nodelist[inode].q012;
  a1->qyz += k*nodelist[inode].q003;
  k = (c5*xt3*yt2 +c4*(three*xt*yt2-xt3) +three*c3*xt)*a1->q;
  a1->VP += k*nodelist[inode].q320;
  a1->dpx += k*nodelist[inode].q220;
  a1->dpy += k*nodelist[inode].q310;
  a1->qxx += k*nodelist[inode].q120;
  a1->qxy += k*nodelist[inode].q210;
  a1->qyy += k*nodelist[inode].q300;
  k = (c5*yt3*xt2 +c4*(three*yt*xt2-yt3) +three*c3*yt)*a1->q;
  a1->VP += k*nodelist[inode].q230;
  a1->dpx += k*nodelist[inode].q130;
  a1->dpy += k*nodelist[inode].q220;
  a1->qxx += k*nodelist[inode].q030;
  a1->qxy += k*nodelist[inode].q120;
  a1->qyy += k*nodelist[inode].q210;
  k = (c5*xt3*zt2 +c4*(three*xt*zt2-xt3) +three*c3*xt)*a1->q;
  a1->VP += k*nodelist[inode].q302;
  a1->dpx += k*nodelist[inode].q202;
  a1->dpz += k*nodelist[inode].q301;
  a1->qxx += k*nodelist[inode].q102;
  a1->qxz += k*nodelist[inode].q201;
  a1->qzz += k*nodelist[inode].q300;
  k = (c5*zt3*xt2 +c4*(three*zt*xt2-zt3) +three*c3*zt)*a1->q;
  a1->VP += k*nodelist[inode].q203;
  a1->dpx += k*nodelist[inode].q103;
  a1->dpz += k*nodelist[inode].q202;
  a1->qxx += k*nodelist[inode].q003;
  a1->qxz += k*nodelist[inode].q102;
  a1->qzz += k*nodelist[inode].q201;
  k = (c5*yt3*zt2 +c4*(three*yt*zt2-yt3) +three*c3*yt)*a1->q;
  a1->VP += k*nodelist[inode].q032;
  a1->dpy += k*nodelist[inode].q022;
  a1->dpz += k*nodelist[inode].q031;
  a1->qyy += k*nodelist[inode].q012;
  a1->qyz += k*nodelist[inode].q021;
  a1->qzz += k*nodelist[inode].q030;
  k = (c5*zt3*yt2 +c4*(three*zt*yt2-zt3) +three*c3*zt)*a1->q;
  a1->VP += k*nodelist[inode].q023;
  a1->dpy += k*nodelist[inode].q013;
  a1->dpz += k*nodelist[inode].q022;
  a1->qyy += k*nodelist[inode].q003;
  a1->qyz += k*nodelist[inode].q012;
  a1->qzz += k*nodelist[inode].q021;
  k = (c5*xt2*yt2 +c4*(xt2+yt2) +c3)*zt*a1->q;
  a1->VP += k*nodelist[inode].q221;
  a1->dpx += k*nodelist[inode].q121;
  a1->dpy += k*nodelist[inode].q211;
  a1->dpz += k*nodelist[inode].q220;
  a1->qxx += k*nodelist[inode].q021;
  a1->qyy += k*nodelist[inode].q201;
  a1->qxz += k*nodelist[inode].q120;
  a1->qyz += k*nodelist[inode].q210;
  k = (c5*xt2*zt2 +c4*(xt2+zt2) +c3)*yt*a1->q;
  a1->VP += k*nodelist[inode].q212;
  a1->dpx += k*nodelist[inode].q112;
  a1->dpy += k*nodelist[inode].q202;
  a1->dpz += k*nodelist[inode].q211;
  a1->qxx += k*nodelist[inode].q012;
  a1->qzz += k*nodelist[inode].q210;
  a1->qxz += k*nodelist[inode].q111;
  a1->qyz += k*nodelist[inode].q201;
  k = (c5*zt2*yt2 +c4*(zt2+yt2) +c3)*xt*a1->q;
  a1->VP += k*nodelist[inode].q122;
  a1->dpx += k*nodelist[inode].q022;
  a1->dpy += k*nodelist[inode].q112;
  a1->dpz += k*nodelist[inode].q121;
  a1->qzz += k*nodelist[inode].q120;
  a1->qyy += k*nodelist[inode].q102;
  a1->qxy += k*nodelist[inode].q022;
  a1->qxz += k*nodelist[inode].q022;
  k = (c5*xt3+three*c4*xt)*yt*zt*a1->q;
  a1->VP += k*nodelist[inode].q311;
  a1->dpx += k*nodelist[inode].q211;
  a1->dpy += k*nodelist[inode].q301;
  a1->dpz += k*nodelist[inode].q310;
  a1->qxx += k*nodelist[inode].q211;
  a1->qxy += k*nodelist[inode].q201;
  a1->qxz += k*nodelist[inode].q210;
  k = (c5*yt3+three*c4*yt)*xt*zt*a1->q;
  a1->VP += k*nodelist[inode].q131;
  a1->dpx += k*nodelist[inode].q031;
  a1->dpy += k*nodelist[inode].q121;
  a1->dpz += k*nodelist[inode].q130;
  a1->qyy += k*nodelist[inode].q111;
  a1->qxy += k*nodelist[inode].q021;
  a1->qyz += k*nodelist[inode].q120;
  k = (c5*zt3+three*c4*zt)*yt*xt*a1->q;
  a1->VP += k*nodelist[inode].q113;
  a1->dpx += k*nodelist[inode].q013;
  a1->dpy += k*nodelist[inode].q103;
  a1->dpz += k*nodelist[inode].q112;
  a1->qzz += k*nodelist[inode].q111;
  a1->qyz += k*nodelist[inode].q102;
  a1->qxz += k*nodelist[inode].q012;
#endif
      }
    } else if (nodelist[inode].innode > 0) {
      imax = 0;
      i    = nodelist[inode].first;
      if ((nodelist[inode].innode) > 0  &&
    (NbAtomList[i].who -> serial > a1 -> serial))
        NbAtomAll[imax++] = NbAtomList[i].who;
      for(j = 1; j < nodelist[inode].innode - 1; j++) {
        i = NbAtomList[i].next;
  if (NbAtomList[i].who -> serial > a1 -> serial)
          NbAtomAll[imax++] = NbAtomList[i].who;
      } /* End of for (j) */

      for(i = 0, VectorPtr = NbVector; i < imax; i++, VectorPtr += 4) {
  a2 = NbAtomAll[i];
  VectorPtr[0] = a2 -> px - a1 -> px;
  VectorPtr[1] = a2 -> py - a1 -> py;
  VectorPtr[2] = a2 -> pz - a1 -> pz;
        VectorPtr[3] = sqrt(VectorPtr[0] * VectorPtr[0] +
          VectorPtr[1] * VectorPtr[1] +
          VectorPtr[2] * VectorPtr[2]);
      } /* End of for (i) */

      /**** Add the new components ****/

      for(i = 0, VectorPtr = NbVector; i < imax; i++, VectorPtr += 4) {
  a2 = NbAtomAll[i];
  for(j = 0; j < a1 -> dontuse; j++)
    if (a2 == a1 -> excluded[j]) goto SKIPNEW;

  if (VectorPtr[3] > mxcut) {
    r0  = one / VectorPtr[3];
    r   = r0 * r0;
    r   = r * r * r; /* r0^-6 */
    xt  = a1 -> q * a2 -> q * dielectric * r0;
    yt  = a1 -> a * a2 -> a * r;
    zt  = a1 -> b * a2 -> b * r * r;
    k   = xt - yt + zt;
    xt *= r0;
          yt *= r0;
          zt *= r0;
    k1  = xt - yt * six + zt * twelve;
    xt *= r0;
          yt *= r0;
          zt *= r0;
    k2  = xt * three;
          ka2 = - yt * six * eight;
          kb2 =   zt * twelve * 14.0f;
#ifdef CUBIC
    xt *= r0;
          yt *= r0;
          zt *= r0;
    k3  = -xt *  5.0f *  3.0f;
          ka3 =  yt *  6.0f *  8.0f * 10.0f;
          kb3 = -zt * 12.0f * 14.0f * 16.0f;
#endif
    k1  = -k1;
          xt  = VectorPtr[0] * r0;
          yt  = VectorPtr[1] * r0;
          zt  = VectorPtr[2] * r0;

          a1 -> VP  += k;

          t1         = k1 * xt;
          a2 -> dpx -= t1;
          a1 -> dpx += t1;
          t1         = k1 * yt;
          a2 -> dpy -= t1;
          a1 -> dpy += t1;
          t1         = k1 * zt;
          a2 -> dpz -= t1;
          a1 -> dpz += t1;

          /****  Note that xt has the 1/r in it so k2*xt*xt is 1/r^5 ****/

          t1         = xt * xt;
          t2         = k2 * (t1 - third) + ka2 * (t1 - eightth) + kb2 * (t1 - fourteenth);
          a2 -> qxx -= t2;
          a1 -> qxx -= t2;
          t3         = k2 + ka2 + kb2;
          t2         = t3 * yt * xt;
          a2 -> qxy -= t2;
          a1 -> qxy -= t2;
          t2         = t3 * zt * xt;
          a2 -> qxz -= t2;
          a1 -> qxz -= t2;
          t1         = yt * yt;
          t2         = k2 * (t1 - third) + ka2 * (t1 - eightth) + kb2 * (t1 - fourteenth);
          a2 -> qyy -= t2;
          a1 -> qyy -= t2;
          t2         = t3 * yt * zt;
          a2 -> qyz -= t2;
          a1 -> qyz -= t2;
          t1         = zt * zt;
          t2         = k2 * (t1 - third) + ka2 * (t1 - eightth) + kb2 * (t1 - fourteenth);
          a2 -> qzz -= t2;
          a1 -> qzz -= t2;

#ifdef CUBIC
          t2          = xt * xt;
          t1          = xt * t2;
          a2 -> qxxx -= k3  * (t1 - xt * ( 9.0f / 15.0f)) ;
          a2 -> qxxx -= ka3 * (t1 - xt * (24.0f / 80.0f)) ;
          a2 -> qxxx -= kb3 * (t1 - xt * (42.0f / (14.0f * 16.0f)));
          a1 -> qxxx += k3  * (t1 - xt * ( 9.0f / 15.0f)) ;
          a1 -> qxxx += ka3 * (t1 - xt * (24.0f / 80.0f)) ;
          a1 -> qxxx += kb3 * (t1 - xt * (42.0f / (14.0f * 16.0f)));
          t1          = yt * t2;
          a2 -> qxxy -= k3  * (t1 - yt * ( 6.0f / 15.0f));
          a2 -> qxxy -= ka3 * (t1 - yt * (11.0f / 80.0f));
          a2 -> qxxy -= kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
          a1 -> qxxy += k3  * (t1 - yt * ( 6.0f / 15.0f));
          a1 -> qxxy += ka3 * (t1 - yt * (11.0f / 80.0f));
          a1 -> qxxy += kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
          t1          = zt * t2;
          a2 -> qxxz -= k3  * (t1 - zt * ( 6.0f / 15.0f));
          a2 -> qxxz -= ka3 * (t1 - zt * (11.0f / 80.0f));
          a2 -> qxxz -= kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
          a1 -> qxxz += k3  * (t1 - zt * ( 6.0f / 15.0f));
          a1 -> qxxz += ka3 * (t1 - zt * (11.0f / 80.0f));
          a1 -> qxxz += kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
          t1          = yt* yt * xt;
          a2 -> qxyy -= k3  * (t1 - xt * ( 6.0f / 15.0f));
          a2 -> qxyy -= ka3 * (t1 - xt * (11.0f / 80.0f));
          a2 -> qxyy -= kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
          a1 -> qxyy += k3  * (t1 - xt * ( 6.0f / 15.0f));
          a1 -> qxyy += ka3 * (t1 - xt * (11.0f / 80.0f));
          a1 -> qxyy += kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
          t1          = (k3 + ka3 + kb3) * yt * zt * xt;
          a2 -> qxyz -= t1;
          a1 -> qxyz += t1;
          t1          = zt * zt * xt;
          a2 -> qxzz -= k3  * (t1 - xt * ( 6.0f / 15.0f));
          a2 -> qxzz -= ka3 * (t1 - xt * (11.0f / 80.0f));
          a2 -> qxzz -= kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
          a1 -> qxzz += k3  * (t1 - xt * ( 6.0f / 15.0f));
          a1 -> qxzz += ka3 * (t1 - xt * (11.0f / 80.0f));
          a1 -> qxzz += kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
          t2          = yt * yt;
          t1          = t2 * yt;
          a2 -> qyyy -= k3  * (t1 - yt * ( 9.0f / 15.0f));
          a2 -> qyyy -= ka3 * (t1 - yt * (24.0f / 80.0f));
          a2 -> qyyy -= kb3 * (t1 - yt * (42.0f / (14.0f * 16.0f)));
          a1 -> qyyy += k3  * (t1 - yt * ( 9.0f / 15.0f));
          a1 -> qyyy += ka3 * (t1 - yt * (24.0f / 80.0f));
          a1 -> qyyy += kb3 * (t1 - yt * (42.0f / (14.0f * 16.0f)));
          t1          = t2 * zt
          a2 -> qyyz -= k3  * (t1 - zt * ( 6.0f / 15.0f));
          a2 -> qyyz -= ka3 * (t1 - zt * (11.0f / 80.0f));
          a2 -> qyyz -= kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
          a1 -> qyyz += k3  * (t1 - zt * ( 6.0f / 15.0f));
          a1 -> qyyz += ka3 * (t1 - zt * (11.0f / 80.0f));
          a1 -> qyyz += kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
          t2          = zt * zt;
          t1          = t2 * yt;
          a2 -> qyzz -= k3  * (t1 - yt * ( 6.0f / 15.0f));
          a2 -> qyzz -= ka3 * (t1 - yt * (11.0f / 80.0f));
          a2 -> qyzz -= kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
          a1 -> qyzz += k3  * (t1 - yt * ( 6.0f / 15.0f));
          a1 -> qyzz += ka3 * (t1 - yt * (11.0f / 80.0f));
          a1 -> qyzz += kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
          t1          = t2 * zt;
          a2 -> qzzz -= k3  * (t1 - zt * ( 9.0f / 15.0f));
          a2 -> qzzz -= ka3 * (t1 - zt * (24.0f / 80.0f));
          a2 -> qzzz -= kb3 * (t1 - zt * (42.0f / (14.0f * 16.0f)));
          a1 -> qzzz += k3  * (t1 - zt * ( 9.0f / 15.0f));
          a1 -> qzzz += ka3 * (t1 - zt * (24.0f / 80.0f));
          a1 -> qzzz += kb3 * (t1 - zt * (42.0f / (14.0f * 16.0f)));
#endif
        } else {
          if (inclose == CloseSize) {
            CloseSize <<= 1;
            if ((Close = (AMMP_ATOM **)ReAlloca(Close, sizeof(AMMP_ATOM *) * CloseSize, Routine)) == NULL)
              return FALSE;
          }
  }
SKIPNEW:;
  }/* End of for (i) */

        i = sizeof(AMMP_ATOM *) * inclose;
        if (!a1 -> Close) {
          if ((a1 -> Close = (AMMP_ATOM **)Alloca(i + sizeof(AMMP_ATOM *), Routine)) == NULL)
            return FALSE;
          a1 -> nclose = inclose;
        } else if (a1 -> nclose < inclose) {
          free(a1 -> Close);
          if ((a1 -> Close = (AMMP_ATOM **)Alloca(i + sizeof(AMMP_ATOM *), Routine)) == NULL)
            return FALSE;
          a1 -> nclose = inclose;
        }

        memcpy(a1 -> Close, Close, i);
        a1 -> Close[inclose] = NULL;
      } /* End of if in close MM node */
    } /* End of for (inode) */

    /**** Set the position ****/

    if (lambda) {
      a1 -> px = a1 -> dx * lambda + a1 -> x;
      a1 -> py = a1 -> dy * lambda + a1 -> y;
      a1 -> pz = a1 -> dz * lambda + a1 -> z;
    } else {
      a1 -> px = a1 -> x;
      a1 -> py = a1 -> y;
      a1 -> pz = a1 -> z;
    }
  } /* End of for (ii) */

  free(Close);
  free(nodelist);

  return TRUE;
}

