/* restrain.c
*
* collection of routines to service restrain potentials
*
* POOP (Poor-mans Object Oriented Programming) using scope rules
*
* these routines hold a data base (in terms of array indeces)
* of restraints, with the associated length and force constant
* These are updateable - unlike bonds which are "permanent"
*
* (this could be table driven but what the hell memories cheap)
*
* the routines for potential value, force and (eventually) second
* derivatives are here also
*
* force and 2nd derivative routines assume zero'd arrays for output
* this allows for parralellization if needed (on a PC?)
*
* forces are symmetric 
*/
/*
*  copyright 1992 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"


/* function restrain adds a restrain to the restrain list
* returns 1 if ok
* returns 0 if not
*  is passed the atom serial numbers, length and constant
* allocates the new memory, initializes it and
* returns
*/

int AMMP_FASTCALL restrain(int p1, int p2, float bl, float fk)
{
  RESTRAIN *new;

  /**** Get the atom pointers for the two serial numbers ****/

  AMMP_ATOM *   ap1 = a_m_serial(p1);
  AMMP_ATOM *   ap2 = a_m_serial(p2);

  if ((!ap1) || (!ap2)) {
    aaerror("Undefined atom in restrain %d %d", p1, p2);
    return FALSE;
  }

  /**** Check to see if a restraint is already defined ****/

  if ((new = restrain_first) != NULL) {
    while(1) {
      if (!new) break;
      if (((new -> atom1 == ap1) && (new -> atom2 == ap2)) ||
          ((new -> atom1 == ap2) && (new -> atom2 == ap1))) {
        new -> length = bl;
        new -> k      = fk;
        return TRUE;
      }
      if (new == new -> next) break;
      new = new -> next;
    } /* End of while */
  }

  if ((new = (RESTRAIN *)Alloca(sizeof(RESTRAIN), "restrain()")) == NULL)
    return FALSE;

  /**** Initialize the pointers ****/

  if (restrain_first == NULL) restrain_first = new;
  if (restrain_last  == NULL) restrain_last  = new;

  new -> atom1          = ap1;
  new -> atom2          = ap2;
  new -> length         = bl;
  new -> k              = fk;
  new -> next           = new;
  restrain_last -> next = new;
  restrain_last         = new;

  return TRUE;
}


/* v_restrain()
* this function sums up the potentials
* for the atoms defined in the RESTRAIN data structure.
*/
/* standard returns 0 if error (any) 1 if ok
* V is the potential */

int AMMP_FASTCALL v_restrain(float *V, float lambda)
{
	RESTRAIN *bp;
	float r,xt,yt,zt;
	AMMP_ATOM *a1,*a2;


	bp = restrain_first;
       if( bp == NULL ) return 1;
       while(1)
       {
	if( bp == NULL) return 0;
	a1 = bp->atom1; a2 = bp->atom2;
	if( lambda == 0.)
	{
	r = (a1->x - a2->x)*(a1->x - a2->x);
	r = r + (a1->y - a2->y)*(a1->y - a2->y);
	r = r + (a1->z - a2->z)*(a1->z - a2->z);
	} else
	{
	xt = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
	yt = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
	zt = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
	r = xt*xt+yt*yt+zt*zt;
	}
	r = sqrt(r); *V += bp->k*( r - bp->length)*(r - bp->length);
	if( bp == bp->next ) return 1;
	bp = bp->next;
       }
}

/* f_restrain()
*
* f_restrain increments the forces in the atom structures by the force
* due to the restrain components.  NOTE THE WORD increment.
* the forces should first be zero'd.
* if not then this code will be invalid.  THIS IS DELIBERATE.
* on bigger (and better?) machines the different potential terms
* may be updated at random or in parrellel, if we assume that this routine
* will initialize the forces then we can't do this.
*/

int AMMP_FASTCALL f_restrain(float lambda)
/*  returns 0 if error, 1 if OK */
{
	RESTRAIN *bp;
	float r,k,ux,uy,uz;
	AMMP_ATOM *a1,*a2;


	bp = restrain_first;
       if( bp == NULL ) return 1;
       while(1)
       {
	if( bp == NULL) return 0;
	k = bp->k;
	a1 = bp->atom1; a2 = bp->atom2;
	if( lambda == 0.)
	{
	ux = (a2->x - a1->x);
	uy = (a2->y - a1->y);
	uz = (a2->z - a1->z);
	}else{
	ux = (a2->x -a1->x +lambda*(a2->dx-a1->dx));
	uy = (a2->y -a1->y +lambda*(a2->dy-a1->dy));
	uz = (a2->z -a1->z +lambda*(a2->dz-a1->dz));
	}
	r = ux*ux + uy*uy + uz*uz;
	 /* watch for FP errors*/
	 if( r <= 1.e-5)
	 { r = 0; ux = 1.; uy = 0.; uz = 0.; }else{
	r = sqrt(r); ux = ux/r; uy = uy/r; uz = uz/r;
	}
	ux = 2*k*(r-bp->length)*ux; 
	uy = 2*k*(r-bp->length)*uy; 
	uz = 2*k*(r-bp->length)*uz;
	a1->fx += ux; 
	a1->fy += uy; 
	a1->fz += uz; 
	a2->fx -= ux; 
	a2->fy -= uy; 
	a2->fz -= uz; 
	if( bp == bp->next ) return 1;
	bp = bp->next;
       }
}

/* function get_restrain( a1,restrained,10,inrestrain);
* check the RESTRAINS list for atoms restrained to a1
*/

void AMMP_FASTCALL get_restrain(AMMP_ATOM *a1, AMMP_ATOM *restrained[], int mrestrain, int *inrestrain)
{
	RESTRAIN *mine;
	mine = restrain_first;
	*inrestrain = 0;
	while(1)
	{
	if( (mine == NULL) )
	{
		return;
	}
	if( mine->atom1 == a1)
	{
		restrained[(*inrestrain)++] = mine->atom2;
	}
	if( mine->atom2 == a1)
	{
		restrained[(*inrestrain)++] = mine->atom1;
	}
	if( mine == mine->next) return;
	mine = mine->next;
	if( *inrestrain == mrestrain ) return;
	}		
}


/**** Dump the restrains ****/

void AMMP_FASTCALL dump_restrains(FILE *where)
{
  RESTRAIN      *b = restrain_first;

  while(b) {
    fprintf(where, "restrain %d %d %f %f;\n",
            b -> atom1 -> serial, b -> atom2 -> serial, b -> length, b -> k);
    if (b -> next == b) break;
    b = b -> next;
  } /* End of while */
}


/* a_restrain()
* this function sums up the potentials
* for the atoms defined in the RESTRAIN data structure.
*/
/* standard returns 0 if error (any) 1 if ok
* V is the potential */

int AMMP_FASTCALL a_restrain(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
	RESTRAIN *bp;
	float r,xt,yt,zt;
	AMMP_ATOM *a1,*a2;


	bp = restrain_first;
       if( bp == NULL ) return 1;
       while(1)
       {
	if( bp == NULL) return 0;
	a1 = bp->atom1; a2 = bp->atom2;
	if(( a1->serial >= ilow && a1->serial <=ihigh)
	 ||( a2->serial >= ilow && a2->serial <=ihigh))
	{
	if( lambda == 0.)
	{
	r = (a1->x - a2->x)*(a1->x - a2->x);
	r = r + (a1->y - a2->y)*(a1->y - a2->y);
	r = r + (a1->z - a2->z)*(a1->z - a2->z);
	} else
	{
	xt = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
	yt = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
	zt = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
	r = xt*xt+yt*yt+zt*zt;
	}
	r = sqrt(r); zt= bp->k*( r - bp->length)*(r - bp->length);
	*V += zt;
	fprintf(op,"Restrain %d %d E %f value %f error %f\n"
		,a1->serial,a2->serial,zt,r,r-bp->length);
	}
	if( bp == bp->next ) return 1;
	bp = bp->next;
       }
}
