/* richard.c
*
*  thermodynamic path integrals using the
*  Feynmann Weiner Kac approach (actually the Wiener version)
*
*
*  basically:
*
* follow the integral of the exp(lagrangian)
* over the path
*
*  exp( - ( 0.5 m Delta((v )^2)dt - Delta(V) dt) / (kT dt) )
*  where v is velocity, V potential energy and everything
* else is obvious
*
* for each atom therefor:
*
*  occupancy *=  exp( -delta(V)/kT ) *exp( 0.5 m Delta(v^2)/kT)
*  where the potential change is global, but the 
*   velocity changes are local 
*
*  the total state occupancy is therefore:
*
*  sum( atom occupancy) = exp( -delta(V)/kT)*sum( exp(0.5....) )
*
*  This can be seen either as a limit in small time steps
*  or a global potential (i.e. function of all atoms at once)
*
*  however in the absense of atom creation or anihilation
*  the sum( exp(0.5 ...)) must be constant (=Natom)
*  (if later we wish to use it this is a constraint)
*  (i.e. the integral over all paths of the free particle propagator
*   must be the total number of free particles)
*
* >>>>  Therefore in order to study path integral averages over
* >>>>  the system  it is only necessary to track 
* >>>>   exp( -delta(V)/kT)  as a product from step to step.
*
*  nifty?
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"


int AMMP_FASTCALL richard(FILE *op, int echo, AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, char *who, int niter, float p1, float p2, float p3)
{
  void *        touse;
  float         o, V, Vold, T;
  int           i, numatm;

  /**** Figure out which  function to use ****/

  touse = NULL;
  if (strcmp(who, "pac") == 0) {
    touse = (void *)pac;
  } else if (strcmp(who, "pacpac") == 0) {
    touse = (void *)pacpac;
  } else if (strcmp(who, "tpac") == 0) {
    touse = (void *)tpac;
  } else if (strcmp(who, "hpac") == 0) {
    touse = (void *)hpac;
  } else if (strcmp(who, "ptpac") == 0) {
    touse = (void *)ptpac;
  } else if (strcmp(who, "verlet") == 0){
    touse = (void *)verlet;
  }

  if (touse == NULL ) {
    aaerror("Usage richard <pac, pacpac, tpac, hpac, ptpac, verlet> nstep");
    return FALSE;
  }

  numatm = a_number();
  T = p2;
  V = 0.0f;
  if (touse == (void *)ptpac) T = p3;
  if (T <= 0.00001f) T = 300.0f;

  for(i = 0; i < nfs; i++) (*vfs[i])(&V, 0.0f);

  for(i = 0; i < 1; i++) {
    if (touse == (void *)pac) {
      pac(ffs,nfs,niter,p1);
    } else if (touse == (void *)pacpac) {
      pacpac(ffs,nfs,niter,p1);
    } else if (touse == (void *)verlet) {
      verlet(ffs,nfs,niter,p1);
    } else if (touse == (void *)tpac) {
      tpac(ffs, nfs, niter, p1, p2);
    } else if (touse == (void *)hpac) {
      hpac(ffs,vfs,nfs,niter,p1,p2);
    } else {
      ptpac(ffs,nfs,niter,p1,p2,p3);
    }
  } /* End of for (i) */

  Vold = V;
  V    = 0.0f;

  for(i = 0;i < nfs; i++) (*vfs[i])(&V,0.0f);

  o = get_f_variable("occup");
  if (o <= 0.0f) o = 1.0f;

  o *= exp( -(V - Vold) / (1.987e-3 * T * numatm));
  set_f_variable("occup", o);

  return 0;
}

