
/*************************************************
****     AMMP - Screen non-bond potential     ****
**** Copyright 1992-2012, Robert W. Harrison  ****
****    VEGA edition by Alessandro Pedretti   ****
*************************************************/


#include <stdio.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

#define  alpha          1.0f

/**** Global variables ****/

extern AMMP_ATOM **     NbAtomAll;
extern int              NbAtomAllNum;
extern int *            NbIndexes;
extern float *          NbVector;

/**** Sum the potential for a range of atoms ****/

int AMMP_FASTCALL a_screen(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
  AMMP_ATOM     *a1,*a2;
  float         r, r0, ra, k, xt, yt, zt;
  float         lcutoff, cutoff;
  float         dielectric, ve, va, vh;
  float         vel, val, vhl;
  float         vtint, vtout, vtt;
  int           i, ii;

  cutoff     = GetCutoff();
  dielectric =  Q_CONST / GetDielectric();
  lcutoff    = -cutoff;
  vtint      = zero;
  vtout      = zero;
  vtt        = zero;

  for(ii = ilow; ii <= ihigh; ii++) {
    a1 = a_m_serial(ii);
    if (a1 == NULL) goto NOTANATOM;
    ve  = zero;
    va  = zero;
    vh  = zero;
    vel = zero;
    val = zero;
    vhl = zero;
    a2  = a_next(-1);
    while(a2 -> next) {
      for(i = 0; i < a1 -> dontuse; i++)
        if (a2 == a1 -> excluded[i]) goto SKIP;

      /**** Non-bonded are only used when the atoms arent bonded ****/

      if (lambda) {
  xt = a1 -> x - a2 -> x + lambda *(a1 -> dx - a2 -> dx);
  if ((xt > cutoff) || (xt < lcutoff)) goto SKIP;
  yt = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
  if ((yt > cutoff) || (yt < lcutoff)) goto SKIP;
  zt = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
  if ((zt > cutoff) || (zt < lcutoff)) goto SKIP;
      } else {
        xt = a1 -> x - a2 -> x;
  if ((xt > cutoff) || (xt < lcutoff)) goto SKIP;
  yt = a1 -> y - a2 -> y;
  if ((yt > cutoff) || (yt < lcutoff)) goto SKIP;
  zt = a1 -> z - a2 -> z;
  if ((zt > cutoff) || (zt < lcutoff)) goto SKIP;
      }
      r   = xt * xt + yt * yt + zt * zt;
      if (r < 0.01f) r = 0.01f;
      r0  = sqrt(r);
      r   = r * r * r;
      k   = dielectric * a1 -> q * a2 -> q / r0;
      ra  = r0 * alpha;
      k  *= 1.0f - exp(-ra) * ((((ra + 9.0f) * ra + 33.0f) * ra) / 48.0f + 1.0f);
      ve += k;
      va -= a1 -> a * a2 ->a / r;
      vh += a1 -> b * a2 ->b / r / r;

      if ((a2 -> serial < ilow) || (a2 -> serial > ihigh)) {
        vel += k;
        val -= a1 -> a * a2 -> a / r;
  vhl += a1 -> b * a2 -> b / r / r;
      }

SKIP:
      if(a2 -> next == a2) break;
      a2 = a2 -> next;
    } /* End of while */

    fprintf(op, "Vnonbon internal %s %d Eq %f E6 %f E12 %f\n",
      a1 -> name, ii, ve - vel, va - val, vh - vhl);
    fprintf(op, "Vnonbon external %s %d Eq %f E6 %f E12 %f\n",
            a1 -> name, ii, vel, val, vhl);
    fprintf(op, "Vnonbon total    %s %d Eq %f E6 %f E12 %f\n",
            a1 -> name, ii, ve, va, vh);
    *V    += ve  + va  + vh;
    vtint += ve  - vel + va - val + vh - vhl;
    vtout += vel + val + vhl;
    vtt   += ve  + va  + vh;
NOTANATOM:;
  } /* End of for (ii) */

  fprintf(op," Vnonbon total internal %f \n",vtint);
  fprintf(op," Vnonbon total external %f \n",vtout);
  fprintf(op," Vnonbon total          %f \n",vtt);

  return TRUE;
}


/**** Calculate the non-bond force increment ****/

int AMMP_FASTCALL f_screen(float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         ux, uy, uz;
  float         k, r, r0, ra, xt, yt, zt;
  float         dielectric;
  float         fx, fy, fz;
#if defined(CUBIC) || defined(QUARTIC) || defined(QUINTIC)
  float         xt2, yt2, zt2;

#  if defined(QUARTIC) || defined(QUINTIC)
  float         xt3, yt3, zt3;

#    ifdef QUINTIC
  float         xt4, yt4, zt4;
#    endif
#  endif
#endif

  int           i, ii, jj;
  int           imax;

  /* First update the lists
   * this routine checks if any atom has
   * broken the mxdq barrier and updates the
   * forces, potentials and expansions thereof
   */

  fv_update_nonbon(lambda);

  dielectric = Q_CONST / GetDielectric();
  imax       = a_number();

  if (lambda) {
    for(i = 0; i < imax; i++) {
      a1   = NbAtomAll[i];
      xt   = a1 -> dx * lambda + a1 -> x - a1 -> px;
      yt   = a1 -> dy * lambda + a1 -> y - a1 -> py;
      zt   = a1 -> dz * lambda + a1 -> z - a1 -> pz;
      fx   = a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt;
      fy   = a1 -> qxy * xt + a1 -> qyy * yt + a1 -> qyz * zt;
      fz   = a1 -> qxz * xt + a1 -> qyz * yt + a1 -> qzz * zt;

#ifdef CUBIC
      xt2  = xt * xt;
      yt2  = yt * yt;
      zt2  = zt * zt;
      fx  += a1 -> qxxx * xt2 / 2.0f + a1 -> qxxy * xt * yt + a1 -> qxxz * xt * zt +
             a1 -> qxyy * yt / 2.0f + a1 -> qxyz * yt * zt + a1 -> qxzz * zt2 / 2.0f;
      fy  += a1 -> qxxy * xt2 / 2.0f + a1 -> qxyy * xt * yt + a1 -> qxyz * xt * zt +
             a1 -> qyyy * yt / 2.0f + a1 -> qyyz * yt * zt + a1 -> qyzz * zt2 / 2.0f;
      fz  += a1 -> qxxz * xt2 / 2.0f + a1 -> qxyz * xt * yt + a1 -> qxzz * xt * zt +
             a1 -> qyyz * yt / 2.0f + a1 -> qyzz * yt * zt + a1 -> qzzz * zt2 / 2.0f;
#endif

#ifdef QUARTIC
      xt3  = xt * xt2;
      yt3  = yt * yt2;
      zt3  = zt * zt2;
      fx  += a1 -> qxxxx * xt3 / 6.0f + a1 -> qxxxy * xt2 * yt / 2.0f +
             a1 -> qxxxz * xt2 * zt / 2.0f + a1 -> qxxyy * xt * yt / 2.0f +
             a1 -> qxxyz * xt * yt * zt + a1 -> qxxzz * xt * zt2 / 2.0f +
             a1 -> qxyyy * yt3 / 6.0f + a1 -> qxyyz * yt2 * zt / 2.0f +
             a1 -> qxyzz * yt * zt2 / 2.0f + a1 -> qxzzz * zt3 / 6.0f;
      fy  += a1 -> qxxxy * xt3 / 6.0f + a1 -> qxxyy * xt2 * yt / 2.0f +
             a1 -> qxxyz * xt2 * zt / 2.0f + a1 -> qxyyy * xt * yt / 2.0f +
             a1 -> qxyyz * xt * yt * zt + a1 -> qxyzz * xt * zt2 / 2.0f +
             a1 -> qyyyy * yt3 / 6.0f + a1 -> qyyyz * yt2 * zt / 2.0f +
             a1 -> qyyzz * yt * zt2 / 2.0f + a1 -> qyzzz * zt3 / 6.0f;
      fz  += a1 -> qxxxz * xt3 / 6.0f + a1 -> qxxyz * xt2 * yt / 2.0f +
             a1 -> qxxzz * xt2 * zt / 2.0f + a1 -> qxyyz * xt * yt / 2.0f +
             a1 -> qxyzz * xt * yt * zt + a1 -> qxzzz * xt * zt2 / 2.0f +
             a1 -> qyyyz * yt3 / 6.0f + a1 -> qyyzz * yt2 * zt / 2.0f +
             a1 -> qyzzz * yt * zt2 / 2.0f + a1 -> qzzzz * zt3 / 6.0f;
#endif

#ifdef QUINTIC
      xt4  = xt * xt3;
      yt4  = yt * yt3;
      zt4  = zt * zt3;
      fx  += a1 -> qxxxxx * xt4 / 24.0f + a1 -> qxxxxy * xt3 * yt /6.0f +
             a1 -> qxxxxz * xt3 * zt / 6.0f  + a1 -> qxxxyy * xt2 * yt2 / 4.0f +
             a1 -> qxxxyz * xt2 * yt * zt / 2.0f + a1 -> qxxxzz * xt2 * zt2 / 4.0f +
             a1 -> qxxyyy * xt * yt3 / 6.0f + a1 -> qxxyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxxyzz * xt * yt * zt2 / 2.0f + a1 -> qxxzzz * xt * zt3 / 6.0f +
             a1 -> qxyyyy * yt4 / 24.0f + a1 -> qxyyyz * yt3 * zt / 6.0f +
             a1 -> qxyyzz * yt2 * zt2 / 4.0f + a1 -> qxyzzz * yt * zt3 / 6.0f +
             a1 -> qxzzzz * zt4 / 24.0f;
      fy  += a1 -> qxxxxy * xt4 / 24.0f + a1 -> qxxxyy * xt3 * yt / 6.0f +
             a1 -> qxxxyz * xt3 * zt / 6.0f + a1 -> qxxyyy * xt2 * yt2 / 4.0f +
             a1 -> qxxyyz * xt2 * yt * zt / 2.0f + a1 -> qxxyzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyyy * xt * yt3 / 6.0f + a1 -> qxyyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxyyzz * xt * yt * zt2 / 2.0f + a1 -> qxyzzz * xt * zt3 / 6.0f +
             a1 -> qyyyyy * yt4 / 24.0f + a1 -> qyyyyz * yt3 * zt / 6.0f +
             a1 -> qyyyzz * yt2 * zt2 / 4.0f + a1 -> qyyzzz * yt * zt3 / 6.0f +
             a1 -> qyzzzz * zt4 / 24.0f;
      fz  += a1 -> qxxxxz * xt4 / 24.0f + a1 -> qxxxyz * xt3 * yt / 6.0f +
             a1 -> qxxxzz * xt3 * zt / 6.0f + a1 -> qxxyyz * xt2 * yt2 / 4.0f +
             a1 -> qxxyzz * xt2 * yt * zt / 2.0f + a1 -> qxxzzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyyz * xt * yt3 / 6.0f + a1 -> qxyyzz * xt * yt2 *zt / 2.0f +
             a1 -> qxyzzz * xt * yt * zt2 / 2.0f + a1 -> qxzzzz * xt * zt3 / 6.0f +
             a1 -> qyyyyz * yt4 / 24.0f + a1 -> qyyyzz * yt3 *zt / 6.0f +
             a1 -> qyyzzz * yt2 * zt2 / 4.0f + a1 -> qyzzzz * yt * zt3 / 6.0f +
             a1 -> qzzzzz * zt4 / 24.0f;
#endif

      a1 -> fx += fx + a1 -> dpx;
      a1 -> fy += fy + a1 -> dpy;
      a1 -> fz += fz + a1 -> dpz;

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];

        /**** Note ux is backwards from below ****/

        ux = (a2 -> dx - a1 -> dx) * lambda + (a2 -> x - a1 -> x);
        uy = (a2 -> dy - a1 -> dy) * lambda + (a2 -> y - a1 -> y);
        uz = (a2 -> dz - a1 -> dz) * lambda + (a2 -> z - a1 -> z);
        r  = ux * ux + uy * uy + uz * uz;
        if (!r) continue;
        r0 = sqrt(r);
        ux /= r0;
        uy /= r0;
        uz /= r0;
        k   = dielectric * a1 -> q * a2 -> q;
  ra  = r0 * alpha;
        k  *= -(1.0f - exp(-ra) * ((((ra + 9.0f) * ra + 33.0f) * ra) / 48.0f + 1.0f)) / r +
        (alpha * exp(-ra) * (((ra + 6.0f) * ra + 15.0f) * ra / 48.0f + 1.0f - 11.0f / 16)) / r0;
        r  = r * r * r;
        k += (a1 -> a * a2 -> a) / r / r0 * six;
        k -= (a1 -> b * a2 -> b) / r / r / r0 * twelve;
        a1 -> fx += ux * k;
        a1 -> fy += uy * k;
        a1 -> fz += uz * k;
        a2 -> fx -= ux * k;
        a2 -> fy -= uy * k;
        a2 -> fz -= uz * k;
      } /* End of for (ii) */
    } /* End of for (i) */
  } else {
    for(i = 0; i < imax; i++) {
      a1   = NbAtomAll[i];
      xt   = a1 -> x - a1 -> px;
      yt   = a1 -> y - a1 -> py;
      zt   = a1 -> z - a1 -> pz;
      fx   = a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt;
      fy   = a1 -> qxy * xt + a1 -> qyy * yt + a1 -> qyz * zt;
      fz   = a1 -> qxz * xt + a1 -> qyz * yt + a1 -> qzz * zt;

#ifdef CUBIC
      xt2  = xt * xt;
      yt2  = yt * yt;
      zt2  = zt * zt;
      fx  += a1 -> qxxx * xt2 / 2.0f + a1 -> qxxy * xt * yt + a1 -> qxxz * xt * zt +
             a1 -> qxyy * yt / 2.0f + a1 -> qxyz * yt * zt + a1 -> qxzz * zt2 / 2.0f;
      fy  += a1 -> qxxy * xt2 / 2.0f + a1 -> qxyy * xt * yt + a1 -> qxyz * xt * zt +
             a1 -> qyyy * yt / 2.0f + a1 -> qyyz * yt * zt + a1 -> qyzz * zt2 / 2.0f;
      fz  += a1 -> qxxz * xt2 / 2.0f + a1 -> qxyz * xt * yt + a1 -> qxzz * xt * zt +
             a1 -> qyyz * yt / 2.0f + a1 -> qyzz * yt * zt + a1 -> qzzz * zt2 / 2.0f;
#endif

#ifdef QUARTIC
      xt3  = xt * xt2;
      yt3  = yt * yt2;
      zt3  = zt * zt2;
      fx  += a1 -> qxxxx * xt3 / 6.0f + a1 -> qxxxy * xt2 * yt / 2.0f +
             a1 -> qxxxz * xt2 * zt / 2.0f + a1 -> qxxyy * xt * yt / 2.0f +
             a1 -> qxxyz * xt * yt * zt + a1 -> qxxzz * xt * zt2 / 2.0f +
             a1 -> qxyyy * yt3 / 6.0f + a1 -> qxyyz * yt2 * zt / 2.0f +
             a1 -> qxyzz * yt * zt2 / 2.0f + a1 -> qxzzz * zt3 / 6.0f;
      fy  += a1 -> qxxxy * xt3 / 6.0f + a1 -> qxxyy * xt2 * yt / 2.0f +
             a1 -> qxxyz * xt2 * zt / 2.0f + a1 -> qxyyy * xt * yt / 2.0f +
             a1 -> qxyyz * xt * yt * zt + a1 -> qxyzz * xt * zt2 / 2.0f +
             a1 -> qyyyy * yt3 / 6.0f + a1 -> qyyyz * yt2 * zt / 2.0f +
             a1 -> qyyzz * yt * zt2 / 2.0f + a1 -> qyzzz * zt3 / 6.0f;
      fz  += a1 -> qxxxz * xt3 / 6.0f + a1 -> qxxyz * xt2 * yt / 2.0f +
             a1 -> qxxzz * xt2 * zt / 2.0f + a1 -> qxyyz * xt * yt / 2.0f +
             a1 -> qxyzz * xt * yt * zt + a1 -> qxzzz * xt * zt2 / 2.0f +
             a1 -> qyyyz * yt3 / 6.0f + a1 -> qyyzz * yt2 * zt / 2.0f +
             a1 -> qyzzz * yt * zt2 / 2.0f + a1 -> qzzzz * zt3 / 6.0f;
#endif

#ifdef QUINTIC
      xt4  = xt * xt3;
      yt4  = yt * yt3;
      zt4  = zt * zt3;
      fx  += a1 -> qxxxxx * xt4 / 24.0f + a1 -> qxxxxy * xt3 * yt /6.0f +
             a1 -> qxxxxz * xt3 * zt / 6.0f  + a1 -> qxxxyy * xt2 * yt2 / 4.0f +
             a1 -> qxxxyz * xt2 * yt * zt / 2.0f + a1 -> qxxxzz * xt2 * zt2 / 4.0f +
             a1 -> qxxyyy * xt * yt3 / 6.0f + a1 -> qxxyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxxyzz * xt * yt * zt2 / 2.0f + a1 -> qxxzzz * xt * zt3 / 6.0f +
             a1 -> qxyyyy * yt4 / 24.0f + a1 -> qxyyyz * yt3 * zt / 6.0f +
             a1 -> qxyyzz * yt2 * zt2 / 4.0f + a1 -> qxyzzz * yt * zt3 / 6.0f +
             a1 -> qxzzzz * zt4 / 24.0f;
      fy  += a1 -> qxxxxy * xt4 / 24.0f + a1 -> qxxxyy * xt3 * yt / 6.0f +
             a1 -> qxxxyz * xt3 * zt / 6.0f + a1 -> qxxyyy * xt2 * yt2 / 4.0f +
             a1 -> qxxyyz * xt2 * yt * zt / 2.0f + a1 -> qxxyzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyyy * xt * yt3 / 6.0f + a1 -> qxyyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxyyzz * xt * yt * zt2 / 2.0f + a1 -> qxyzzz * xt * zt3 / 6.0f +
             a1 -> qyyyyy * yt4 / 24.0f + a1 -> qyyyyz * yt3 * zt / 6.0f +
             a1 -> qyyyzz * yt2 * zt2 / 4.0f + a1 -> qyyzzz * yt * zt3 / 6.0f +
             a1 -> qyzzzz * zt4 / 24.0f;
      fz  += a1 -> qxxxxz * xt4 / 24.0f + a1 -> qxxxyz * xt3 * yt / 6.0f +
             a1 -> qxxxzz * xt3 * zt / 6.0f + a1 -> qxxyyz * xt2 * yt2 / 4.0f +
             a1 -> qxxyzz * xt2 * yt * zt / 2.0f + a1 -> qxxzzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyyz * xt * yt3 / 6.0f + a1 -> qxyyzz * xt * yt2 *zt / 2.0f +
             a1 -> qxyzzz * xt * yt * zt2 / 2.0f + a1 -> qxzzzz * xt * zt3 / 6.0f +
             a1 -> qyyyyz * yt4 / 24.0f + a1 -> qyyyzz * yt3 *zt / 6.0f +
             a1 -> qyyzzz * yt2 * zt2 / 4.0f + a1 -> qyzzzz * yt * zt3 / 6.0f +
             a1 -> qzzzzz * zt4 / 24.0f;
#endif

      a1 -> fx += fx + a1 -> dpx;
      a1 -> fy += fy + a1 -> dpy;
      a1 -> fz += fz + a1 -> dpz;

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close [ii];

        /**** Note ux is backwards from below ****/

        ux = a2 -> x - a1 -> x;
        uy = a2 -> y - a1 -> y;
        uz = a2 -> z - a1 -> z;
        r  = ux * ux + uy * uy + uz * uz;
        if (!r) continue;
        r0  = sqrt(r);
        ux /= r0;
        uy /= r0;
        uz /= r0;
        k   = -dielectric * a1 -> q * a2 -> q;
        ra  = r0 * alpha;
        k  *= -(1.0f - exp(-ra) * ((((ra + 9.0f) * ra + 33.0f) * ra) / 48.0f + 1.0f)) / r +
        (alpha * exp(-ra) * (((ra + 6.0f) * ra + 15.0f) * ra / 48.0f + 1.0f - 11.0f / 16)) / r0;
        r  = r * r * r;
        k += (a1 -> a * a2 -> a) / r / r0 * six;
        k -= (a1 -> b * a2 -> b) / r / r / r0 * twelve;
        a1 -> fx += ux * k;
        a1 -> fy += uy * k;
        a1 -> fz += uz * k;
        a2 -> fx -= ux * k;
        a2 -> fy -= uy * k;
        a2 -> fz -= uz * k;
      } /* End of for (ii) */
    } /* End of for (i) */
  }

  /**** Clear the forces of inactive atoms ****/

  for(i = 0; i < imax; i++) {
    a1 = NbAtomAll[i];
    if (a1 -> active) continue;
    a1 -> fx = zero;
    a1 -> fy = zero;
    a1 -> fz = zero;
    a1 -> fw = zero;
  } /* End of for (i) */

  return TRUE;
}


/**** Calculate the non-bond potential ****/

int AMMP_FASTCALL v_screen(float *V, float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         r, r0, xt, yt, zt;
  float         k;
  float         vx, dielectric;

#if defined(CUBIC) || defined(QUARTIC) || defined(QUINTIC)
  float         xt2, xt3;
  float         yt2, yt3;
  float         zt2, zt3;

#  if defined(QUARTIC) || defined(QUINTIC)
  float         xt4, yt4, zt4;

#    ifdef QUINTIC
  float         xt5, yt5, zt5;
#    endif
#  endif
#endif

  int           i, ii, jj, imax;

  fv_update_nonbon(lambda);

  dielectric = Q_CONST / GetDielectric();
  imax       = a_number();

  if (lambda) {
    for(i = 0; i < imax; i++) {
      a1  = a_next(i);
      if (!a1 -> active) {
        a1 -> fx = zero;
        a1 -> fy = zero;
        a1 -> fz = zero;
        a1 -> fw = zero;
      }
      vx  = a1 -> VP;
      xt  = a1 -> dx * lambda + a1 -> x - a1 -> px;
      yt  = a1 -> dy * lambda + a1 -> y - a1 -> py;
      zt  = a1 -> dz * lambda + a1 -> z - a1 -> pz;
      vx -= (a1 -> dpx * xt + a1 -> dpy * yt + a1 -> dpz * zt);
      vx -= ((xt * (0.5f * a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt) +
              yt * (0.5f * a1 -> qyy * yt + a1 -> qyz * zt) +
              0.5f * zt * a1 -> qzz * zt));
#ifdef CUBIC
      xt2  = xt * xt;
      yt2  = yt * yt;
      zt2  = zt * zt;
      xt3  = xt2 * xt;
      yt3  = yt2 * yt;
      zt3  = zt2 * zt;
      vx  -= a1 -> qxxx * xt3 / 6.0f + a1 -> qxxy * xt2 * yt / 2.0f +
             a1 -> qxxz * xt2 * zt / 2.0f + a1 -> qxyy * xt * yt2 / 2.0f +
             a1 -> qxyz * xt * yt *zt + a1 -> qxzz * xt * zt2 / 2.0f +
             a1 -> qyyy * yt3 / 6.0f + a1 -> qyyz * yt2 * zt / 2.0f +
             a1 -> qyzz * yt * zt2 / 2.0f + a1 -> qzzz * zt3 / 6.0f;
#endif

#ifdef QUARTIC
      xt4  = xt3 * xt;
      yt4  = yt3 * yt;
      zt4  = zt3 * zt;
      vx  -= a1 -> qxxxx * xt4 / 24.0f + a1 -> qxxxy * xt3 * yt / 6.0f +
             a1 -> qxxxz * xt3 * yt / 6.0f + a1 -> qxxyy * xt2 * yt2 / 4.0f +
             a1 -> qxxyz * xt2 * yt * zt / 2.0f + a1 -> qxxzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyy * xt * yt3 / 6.0f + a1 -> qxyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxyzz * xt * yt * zt2 / 2.0f + a1 -> qxzzz * xt * zt3 / 6.0f +
             a1 -> qyyyy * yt4 / 24.0f + a1 -> qyyyz * yt3 * zt / 6.0f +
             a1 -> qyyzz * yt2 * zt2 / 4.0f + a1 -> qyzzz * yt * zt3 / 6.0f +
             a1 -> qzzzz * zt4 / 24.0f;
#endif

#ifdef QUINTIC
      xt5  = xt4 * xt;
      yt5  = yt4 * yt;
      zt5  = zt4 * zt;
      vx  -= a1 -> qxxxxx * xt5 / 120.0f + a1 -> qxxxxy * xt4 * yt / 24.0f +
             a1 -> qxxxxz * xt4 * zt / 24.0f + a1 -> qxxxyy * xt3 * yt2 / 12.0f +
             a1 -> qxxxyz * xt3 * yt * zt / 6.0f + a1 -> qxxxzz * xt3 * zt2 / 12.0f +
             a1 -> qxxyyy * xt2 * yt3 / 12.0f + a1 -> qxxyyz * xt2 * yt2 * zt / 4.0f +
             a1 -> qxxyzz * xt2 * yt * zt2 / 4.0f + a1 -> qxxzzz * xt2 * zt3 / 12.0f +
             a1 -> qxyyyy * xt * yt4 / 24.0f + a1 -> qxyyyz * xt * yt3 * zt / 6.0f +
             a1 -> qxyyzz * xt * yt2 * zt2 / 4.0f + a1 -> qxyzzz * xt * yt * zt3 / 6.0f +
             a1 -> qxzzzz * xt * zt4 / 24.0f + a1 -> qyyyyy * yt5 / 120.0f +
             a1 -> qyyyyz * yt4 * zt / 24.0f + a1 -> qyyyzz * yt3 * zt2 / 12.0f +
             a1 -> qyyzzz * yt2 * zt3 / 12.0f + a1 -> qyzzzz * yt * zt4 / 24.0f +
             a1 -> qzzzzz * zt5 / 120.0f;
#endif

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];
        xt = (a2 -> dx - a1 -> dx) * lambda + (a2 -> x - a1 -> x);
        yt = (a2 -> dy - a1 -> dy) * lambda + (a2 -> y - a1 -> y);
        zt = (a2 -> dz - a1 -> dz) * lambda + (a2 -> z - a1 -> z);
        r  = xt * xt + yt * yt + zt * zt;
        if (!r) continue;
        r0  = sqrt(r);
        k   = dielectric * a1 -> q * a2 -> q;
        r0 *= alpha;
        k  *= 1.0f - exp(-r0) * ((((r0 + 9.0f) * r0 + 33.0f) * r0) / 48.0f + 1.0f);
        r   = r * r * r;
        k  -= (a1 -> a * a2 -> a) / r;
        k  += (a1 -> b * a2 -> b) / r / r;
        vx += k;
      }  /* End of for (ii) */
      *V += vx;
    } /* End of for (i) */
  } else {
    for(i = 0; i < imax; i++) {
      a1  = a_next(i);
      if (!a1 -> active) {
        a1 -> fx = 0.0f;
        a1 -> fy = 0.0f;
        a1 -> fz = 0.0f;
        a1 -> fw = 0.0f;
      }
      vx  = a1 -> VP;
      xt  = a1 -> x - a1 -> px;
      yt  = a1 -> y - a1 -> py;
      zt  = a1 -> z - a1 -> pz;
      vx -= (a1 -> dpx * xt + a1 -> dpy * yt + a1 -> dpz * zt);
      vx -= ((xt * (0.5f * a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt) +
              yt * (0.5f * a1 -> qyy * yt + a1 -> qyz * zt) +
              0.5f * zt * a1 -> qzz * zt));
#ifdef CUBIC
      xt2  = xt * xt;
      yt2  = yt * yt;
      zt2  = zt * zt;
      xt3  = xt2 * xt;
      yt3  = yt2 * yt;
      zt3  = zt2 * zt;
      vx  -= a1 -> qxxx * xt3 / 6.0f + a1 -> qxxy * xt2 * yt / 2.0f +
             a1 -> qxxz * xt2 * zt / 2.0f + a1 -> qxyy * xt * yt2 / 2.0f +
             a1 -> qxyz * xt * yt *zt + a1 -> qxzz * xt * zt2 / 2.0f +
             a1 -> qyyy * yt3 / 6.0f + a1 -> qyyz * yt2 * zt / 2.0f +
             a1 -> qyzz * yt * zt2 / 2.0f + a1 -> qzzz * zt3 / 6.0f;
#endif

#ifdef QUARTIC
      xt4  = xt3 * xt;
      yt4  = yt3 * yt;
      zt4  = zt3 * zt;
      vx  -= a1 -> qxxxx * xt4 / 24.0f + a1 -> qxxxy * xt3 * yt / 6.0f +
             a1 -> qxxxz * xt3 * yt / 6.0f + a1 -> qxxyy * xt2 * yt2 / 4.0f +
             a1 -> qxxyz * xt2 * yt * zt / 2.0f + a1 -> qxxzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyy * xt * yt3 / 6.0f + a1 -> qxyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxyzz * xt * yt * zt2 / 2.0f + a1 -> qxzzz * xt * zt3 / 6.0f +
             a1 -> qyyyy * yt4 / 24.0f + a1 -> qyyyz * yt3 * zt / 6.0f +
             a1 -> qyyzz * yt2 * zt2 / 4.0f + a1 -> qyzzz * yt * zt3 / 6.0f +
             a1 -> qzzzz * zt4 / 24.0f;
#endif

#ifdef QUINTIC
      xt5  = xt4 * xt;
      yt5  = yt4 * yt;
      zt5  = zt4 * zt;
      vx  -= a1 -> qxxxxx * xt5 / 120.0f + a1 -> qxxxxy * xt4 * yt / 24.0f +
             a1 -> qxxxxz * xt4 * zt / 24.0f + a1 -> qxxxyy * xt3 * yt2 / 12.0f +
             a1 -> qxxxyz * xt3 * yt * zt / 6.0f + a1 -> qxxxzz * xt3 * zt2 / 12.0f +
             a1 -> qxxyyy * xt2 * yt3 / 12.0f + a1 -> qxxyyz * xt2 * yt2 * zt / 4.0f +
             a1 -> qxxyzz * xt2 * yt * zt2 / 4.0f + a1 -> qxxzzz * xt2 * zt3 / 12.0f +
             a1 -> qxyyyy * xt * yt4 / 24.0f + a1 -> qxyyyz * xt * yt3 * zt / 6.0f +
             a1 -> qxyyzz * xt * yt2 * zt2 / 4.0f + a1 -> qxyzzz * xt * yt * zt3 / 6.0f +
             a1 -> qxzzzz * xt * zt4 / 24.0f + a1 -> qyyyyy * yt5 / 120.0f +
             a1 -> qyyyyz * yt4 * zt / 24.0f + a1 -> qyyyzz * yt3 * zt2 / 12.0f +
             a1 -> qyyzzz * yt2 * zt3 / 12.0f + a1 -> qyzzzz * yt * zt4 / 24.0f +
             a1 -> qzzzzz * zt5 / 120.0f;
#endif

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];
        xt = a2 -> x - a1 -> x;
        yt = a2 -> y - a1 -> y;
        zt = a2 -> z - a1 -> z;
        r  = xt * xt + yt * yt + zt * zt;
        if (!r) continue;
        r0  = sqrt(r);
        k   = dielectric * a1 -> q * a2 -> q;
        r0 *= alpha;
        k  *= 1.0f - exp(-r0) * ((((r0 + 9.0f) * r0 + 33.0f) * r0) / 48.0f + 1.0f);
        r   = r * r * r;
        k  -= (a1 -> a * a2 -> a) / r;
        k  += (a1 -> b * a2 -> b) / r / r;
        vx += k;
      }  /* End of for (ii) */
      *V += vx;
    } /* End of for (i) */
  }

  return TRUE;
}

