
/*************************************************
****     AMMP - Shadow non-bond potential     ****
**** Copyright 1992-2012, Robert W. Harrison  ****
****    VEGA edition by Alessandro Pedretti   ****
*************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Global variables ****/

extern AMMP_ATOM **     NbAtomAll;
extern int              NbAtomAllNum;
extern int *            NbIndexes;
extern float *          NbVector;

/**** Local variables ****/

static float  fv_update_shadow_dielecold  = -1.0f;


/**** Reset local variables ****/

void AMMP_FASTCALL ResetShadow(void)
{
  NonBondFreeTable();
  fv_update_shadow_dielecold  = -1.0f;
}


/**** Update the non-bond parameters ****/

int AMMP_FASTCALL fv_update_shadow(float lambda)
{
  AMMP_ATOM **  Close;
  AMMP_ATOM     *a1, *a2;
  float         r, r0, xt, yt, zt, wt;
  float         k, k1, k2;
  float         ka2, kb2;
  float         t1, t2, t3;
  float *       VectorPtr;
  int           CloseSize;
  int           inindex, in;
  int           i, ii, j, imax, inclose;

#ifdef CUBIC
  float         k3, ka3, kb3;
#endif

  const char *  Routine    = "fv_update_shadow()";
  float         dielectric = GetDielectric();
  float         mxcut      = GetMxcut();
  float         mxdq       = GetMxdq2();

  if (fv_update_shadow_dielecold != dielectric) {
    fv_update_shadow_dielecold = dielectric;
    mxdq      = - one;
  }
  dielectric = Q_CONST / dielectric;

  /**** Allocate the memory for the array space ****/

  imax = a_number();
  if ((NbAtomAllNum != imax) &&
      (!NonBondAllocTable(imax, AMMP_NBTABLE_SHADOW))) return FALSE;

  /* First check if anyone's moved and update the lists
   * note that this must be a look-ahead rather than
   * look back search because
   * we cannot update -> px until we've used that atom !!!
   */

  if (NonBondCheck(mxdq, lambda)) return TRUE;

  CloseSize = NCLOSE;
  if ((Close = (AMMP_ATOM **)Alloca(sizeof(AMMP_ATOM *) * CloseSize, Routine)) == NULL)
    return FALSE;

  for(ii = 0; ii < imax; ii++) {
    a1        = NbAtomAll[ii];
    a1 -> VP  = zero;
    a1 -> dpx = zero;
    a1 -> dpy = zero;
    a1 -> dpz = zero;
    a1 -> dpw = zero;
    a1 -> qxx = zero;
    a1 -> qxy = zero;
    a1 -> qxz = zero;
    a1 -> qyy = zero;
    a1 -> qyz = zero;
    a1 -> qzz = zero;
    a1 -> qxw = zero;
    a1 -> qyw = zero;
    a1 -> qzw = zero;
    a1 -> qww = zero;

    /**** Clear the forces of inactive atoms ****/

    if (!a1 -> active) {
      a1 -> fx = zero;
      a1 -> fy = zero;
      a1 -> fz = zero;
      a1 -> fw = zero;
    }

#ifdef CUBIC
    a1 -> qxxx = zero;
    a1 -> qxxy = zero;
    a1 -> qxxz = zero;
    a1 -> qxyy = zero;
    a1 -> qxyz = zero;
    a1 -> qxzz = zero;
    a1 -> qyyy = zero;
    a1 -> qyyz = zero;
    a1 -> qyzz = zero;
    a1 -> qzzz = zero;
#endif
  } /* End of for (ii) */


  for(ii = 0; ii <  imax; ii++) {
    a1 = NbAtomAll[ii];

    /**** If this is met we update the expansion for this atom ****/

    i = ii + 1;
    VectorPtr = NbVector + (i << 2);
    if (lambda) {
      while(i < imax) {
        a2            = NbAtomAll[i];
        xt            = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
        yt            = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
        zt            = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
        wt            = a2 -> w - a1 -> w + lambda * (a2 -> dw - a1 -> dw);
        VectorPtr[0]  = xt;
        VectorPtr[1]  = yt;
        VectorPtr[2]  = zt;
        VectorPtr[3]  = wt;
        VectorPtr[4]  = sqrt(xt * xt + yt * yt + zt * zt + wt * wt);
        VectorPtr    += 5;
        ++i;
      } /* End of while */
    } else {
       while(i < imax) {
        a2            = NbAtomAll[i];
        xt            = a2 -> x - a1 -> x;
        yt            = a2 -> y - a1 -> y;
        zt            = a2 -> z - a1 -> z;
        wt            = a2 -> w - a1 -> w;
        VectorPtr[0]  = xt;
        VectorPtr[1]  = yt;
        VectorPtr[2]  = zt;
        VectorPtr[3]  = wt;
        VectorPtr[4]  = sqrt(xt * xt + yt * yt + zt * zt + wt * wt);
        VectorPtr    += 5;
        ++i;
      } /* End of while */
    }

    /**** Add the new components, first extract indexes ****/

    inindex = 0;
    inclose = 0;
    for(i = ii + 1; i < imax; i++) {
      a2 = NbAtomAll[i];
      for(j = 0; j < a1 -> dontuse; j++)
        if (a2 == a1 -> excluded[j]) goto SKIPNEW;
      if (NbVector[(i * 5) + 4] > mxcut) NbIndexes[inindex++] = i;
      else Close[inclose++] = NbAtomAll[i];
      if (inclose == CloseSize) {
        CloseSize <<= 1;
        if ((Close = (AMMP_ATOM **)ReAlloca(Close, sizeof(AMMP_ATOM *) * CloseSize, Routine)) == NULL)
          return FALSE;
      }
SKIPNEW:;
    } /* End of for (i) */

    i = sizeof(AMMP_ATOM *) * inclose;
    if (!a1 -> Close) {
      if ((a1 -> Close = (AMMP_ATOM **)Alloca(i + sizeof(AMMP_ATOM *), Routine)) == NULL)
        return FALSE;
      a1 -> nclose = inclose;
    } else if (a1 -> nclose < inclose) {
      free(a1 -> Close);
      if ((a1 -> Close = (AMMP_ATOM **)Alloca(i + sizeof(AMMP_ATOM *), Routine)) == NULL)
        return FALSE;
      a1 -> nclose = inclose;
    }
    memcpy(a1 -> Close, Close, i);
    a1 -> Close[inclose] = NULL;

    for(in = 0; in < inindex; in++) {
      i         = NbIndexes[in];
      a2        = NbAtomAll[i];
      VectorPtr = NbVector + (i * 5);
      r0        = VectorPtr[4];
      r         = r0 * r0;
      r         = r * r * r; /* r0^6 */
      xt        = a1 -> q * a2 -> q * dielectric / r0;
      yt        = a1 -> a * a2 -> a / r;
      zt        = a1 -> b * a2 -> b / r / r;
      k         = xt - yt + zt;
      xt        /= r0;
      yt        /= r0;
      zt        /= r0;
      k1         = xt - yt * six + zt * twelve;
      xt        /= r0;
      yt        /= r0;
      zt        /= r0;
      k2         = xt * 3.0f;
      ka2        = - yt * 48.f;
      kb2        =   zt * 168.0f;
#ifdef CUBIC
      xt        /= r0;
      yt        /= r0;
      zt        /= r0;
      k3         = -xt *  5.0f *  3.0f;
      ka3        =  yt *  6.0f *  8.0f * 10.0f;
      kb3        = -zt * 12.0f * 14.0f * 16.0f;
#endif
      k1         = -k1;
      xt         = VectorPtr[0] / r0;
      yt         = VectorPtr[1] / r0;
      zt         = VectorPtr[2] / r0;
      wt         = VectorPtr[3] / r0;

      a1 -> VP  += k;

      t1         = k1 * xt;
      a2 -> dpx -= t1;
      a1 -> dpx += t1;
      t1         = k1 * yt;
      a2 -> dpy -= t1;
      a1 -> dpy += t1;
      t1         = k1 * zt;
      a2 -> dpz -= t1;
      a1 -> dpz += t1;
      t1         = k1 * wt;
      a2 -> dpw -= t1;
      a1 -> dpw += t1;

      /****  Note that xt has the 1/r in it so k2*xt*xt is 1/r^5 ****/

      t1         = xt * xt;
      t2         = k2 * (t1 - third) + ka2 * (t1 - eightth) + kb2 * (t1 - fourteenth);
      a2 -> qxx -= t2;
      a1 -> qxx -= t2;
      t3         = k2 + ka2 + kb2;
      t2         = t3 * yt * xt;
      a2 -> qxy -= t2;
      a1 -> qxy -= t2;
      t2         = t3 * zt * xt;
      a2 -> qxz -= t2;
      a1 -> qxz -= t2;
      t2         = t3 * wt * xt;
      a2 -> qxw -= t2;
      a1 -> qxw -= t2;
      t1         = yt * yt;
      t2         = k2 * (t1 - third) + ka2 * (t1 - eightth) + kb2 * (t1 - fourteenth);
      a2 -> qyy -= t2;
      a1 -> qyy -= t2;
      t2         = t3 * yt * zt;
      a2 -> qyz -= t2;
      a1 -> qyz -= t2;
      t2         = t3 * yt * wt;
      a2 -> qyw -= t2;
      a1 -> qyw -= t2;
      t1         = zt * zt;
      t2         = k2 * (t1 - third) + ka2 * (t1 - eightth) + kb2 * (t1 - fourteenth);
      a2 -> qzz -= t2;
      a1 -> qzz -= t2;
      t2         = t3 * wt * zt;
      a2 -> qzw -= t2;
      a1 -> qzw -= t2;
      t1         = wt * wt;
      t2         = k2 * (t1 - third) + ka2 * (t1 - eightth) + kb2 * (t1 - fourteenth);
      a2 -> qww -= t2;
      a1 -> qww -= t2;

#ifdef CUBIC
      t2          = xt * xt;
      t1          = xt * t2;
      a2 -> qxxx -= k3  * (t1 - xt * ( 9.0f / 15.0f)) ;
      a2 -> qxxx -= ka3 * (t1 - xt * (24.0f / 80.0f)) ;
      a2 -> qxxx -= kb3 * (t1 - xt * (42.0f / (14.0f * 16.0f)));
      a1 -> qxxx += k3  * (t1 - xt * ( 9.0f / 15.0f)) ;
      a1 -> qxxx += ka3 * (t1 - xt * (24.0f / 80.0f)) ;
      a1 -> qxxx += kb3 * (t1 - xt * (42.0f / (14.0f * 16.0f)));
      t1          = yt * t2;
      a2 -> qxxy -= k3  * (t1 - yt * ( 6.0f / 15.0f));
      a2 -> qxxy -= ka3 * (t1 - yt * (11.0f / 80.0f));
      a2 -> qxxy -= kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
      a1 -> qxxy += k3  * (t1 - yt * ( 6.0f / 15.0f));
      a1 -> qxxy += ka3 * (t1 - yt * (11.0f / 80.0f));
      a1 -> qxxy += kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
      t1          = zt * t2;
      a2 -> qxxz -= k3  * (t1 - zt * ( 6.0f / 15.0f));
      a2 -> qxxz -= ka3 * (t1 - zt * (11.0f / 80.0f));
      a2 -> qxxz -= kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
      a1 -> qxxz += k3  * (t1 - zt * ( 6.0f / 15.0f));
      a1 -> qxxz += ka3 * (t1 - zt * (11.0f / 80.0f));
      a1 -> qxxz += kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
      t1          = yt* yt * xt;
      a2 -> qxyy -= k3  * (t1 - xt * ( 6.0f / 15.0f));
      a2 -> qxyy -= ka3 * (t1 - xt * (11.0f / 80.0f));
      a2 -> qxyy -= kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
      a1 -> qxyy += k3  * (t1 - xt * ( 6.0f / 15.0f));
      a1 -> qxyy += ka3 * (t1 - xt * (11.0f / 80.0f));
      a1 -> qxyy += kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
      t1          = (k3 + ka3 + kb3) * yt * zt * xt;
      a2 -> qxyz -= t1;
      a1 -> qxyz += t1;
      t1          = zt * zt * xt;
      a2 -> qxzz -= k3  * (t1 - xt * ( 6.0f / 15.0f));
      a2 -> qxzz -= ka3 * (t1 - xt * (11.0f / 80.0f));
      a2 -> qxzz -= kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
      a1 -> qxzz += k3  * (t1 - xt * ( 6.0f / 15.0f));
      a1 -> qxzz += ka3 * (t1 - xt * (11.0f / 80.0f));
      a1 -> qxzz += kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
      t2          = yt * yt;
      t1          = t2 * yt;
      a2 -> qyyy -= k3  * (t1 - yt * ( 9.0f / 15.0f));
      a2 -> qyyy -= ka3 * (t1 - yt * (24.0f / 80.0f));
      a2 -> qyyy -= kb3 * (t1 - yt * (42.0f / (14.0f * 16.0f)));
      a1 -> qyyy += k3  * (t1 - yt * ( 9.0f / 15.0f));
      a1 -> qyyy += ka3 * (t1 - yt * (24.0f / 80.0f));
      a1 -> qyyy += kb3 * (t1 - yt * (42.0f / (14.0f * 16.0f)));
      t1          = t2 * zt
      a2 -> qyyz -= k3  * (t1 - zt * ( 6.0f / 15.0f));
      a2 -> qyyz -= ka3 * (t1 - zt * (11.0f / 80.0f));
      a2 -> qyyz -= kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
      a1 -> qyyz += k3  * (t1 - zt * ( 6.0f / 15.0f));
      a1 -> qyyz += ka3 * (t1 - zt * (11.0f / 80.0f));
      a1 -> qyyz += kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
      t2          = zt * zt;
      t1          = t2 * yt;
      a2 -> qyzz -= k3  * (t1 - yt * ( 6.0f / 15.0f));
      a2 -> qyzz -= ka3 * (t1 - yt * (11.0f / 80.0f));
      a2 -> qyzz -= kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
      a1 -> qyzz += k3  * (t1 - yt * ( 6.0f / 15.0f));
      a1 -> qyzz += ka3 * (t1 - yt * (11.0f / 80.0f));
      a1 -> qyzz += kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
      t1          = t2 * zt;
      a2 -> qzzz -= k3  * (t1 - zt * ( 9.0f / 15.0f));
      a2 -> qzzz -= ka3 * (t1 - zt * (24.0f / 80.0f));
      a2 -> qzzz -= kb3 * (t1 - zt * (42.0f / (14.0f * 16.0f)));
      a1 -> qzzz += k3  * (t1 - zt * ( 9.0f / 15.0f));
      a1 -> qzzz += ka3 * (t1 - zt * (24.0f / 80.0f));
      a1 -> qzzz += kb3 * (t1 - zt * (42.0f / (14.0f * 16.0f)));
#endif
    } /* End of for (i) */

    /**** Set the position ****/

    if (lambda) {
      a1 -> px = a1 -> dx * lambda + a1 -> x;
      a1 -> py = a1 -> dy * lambda + a1 -> y;
      a1 -> pz = a1 -> dz * lambda + a1 -> z;
      a1 -> pw = a1 -> dw * lambda + a1 -> w;
    } else {
      a1 -> px = a1 -> x;
      a1 -> py = a1 -> y;
      a1 -> pz = a1 -> z;
      a1 -> pw = a1 -> w;
    }
  } /* End of for (ii) */

  free(Close);

  return TRUE;
}


/**** Calculate the non-bond force increment ****/

int AMMP_FASTCALL f_shadow(float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         ux, uy, uz, uw;
  float         k, r, r0, xt, yt, zt, wt;
  float         dielectric;
  float         fx, fy, fz, fw;

#if defined(CUBIC) || defined(QUARTIC) || defined(QUINTIC)
  float         xt2, yt2, zt2;

#  if defined(QUARTIC) || defined(QUINTIC)
  float         xt3, yt3, zt3;

#    ifdef QUINTIC
  float         xt4, yt4, zt4;
#    endif
#  endif
#endif

  int           i, ii, jj;
  int           imax;

  /* First update the lists
   * this routine checks if any atom has
   * broken the mxdq barrier and updates the
   * forces, potentials and expansions thereof
   */

  fv_update_shadow( lambda);

  dielectric = Q_CONST / GetDielectric();
  imax       = a_number();

  if (lambda) {
    for(i = 0; i < imax; i++) {
      a1   = NbAtomAll[i];
      xt   = a1 -> dx * lambda + a1 -> x - a1 -> px;
      yt   = a1 -> dy * lambda + a1 -> y - a1 -> py;
      zt   = a1 -> dz * lambda + a1 -> z - a1 -> pz;
      wt   = a1 -> dw * lambda + a1 -> w - a1 -> pw;
      fx   = a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt + a1 -> qxw * wt;
      fy   = a1 -> qxy * xt + a1 -> qyy * yt + a1 -> qyz * zt + a1 -> qyw * wt;
      fz   = a1 -> qxz * xt + a1 -> qyz * yt + a1 -> qzz * zt + a1 -> qzw * wt;
      fw   = a1 -> qxw * xt + a1 -> qyw * yt + a1 -> qzw * zt + a1 -> qww * wt;
#ifdef CUBIC
      xt2  = xt * xt;
      yt2  = yt * yt;
      zt2  = zt * zt;
      fx  += a1 -> qxxx * xt2 / 2.0f + a1 -> qxxy * xt * yt + a1 -> qxxz * xt * zt +
             a1 -> qxyy * yt / 2.0f + a1 -> qxyz * yt * zt + a1 -> qxzz * zt2 / 2.0f;
      fy  += a1 -> qxxy * xt2 / 2.0f + a1 -> qxyy * xt * yt + a1 -> qxyz * xt * zt +
             a1 -> qyyy * yt / 2.0f + a1 -> qyyz * yt * zt + a1 -> qyzz * zt2 / 2.0f;
      fz  += a1 -> qxxz * xt2 / 2.0f + a1 -> qxyz * xt * yt + a1 -> qxzz * xt * zt +
             a1 -> qyyz * yt / 2.0f + a1 -> qyzz * yt * zt + a1 -> qzzz * zt2 / 2.0f;
#endif
      a1 -> fx += fx + a1 -> dpx;
      a1 -> fy += fy + a1 -> dpy;
      a1 -> fz += fz + a1 -> dpz;
      a1 -> fw += fw + a1 -> dpw;

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];

        /**** Note ux is backwards from below ****/

        ux = (a2 -> dx - a1 -> dx) * lambda + (a2 -> x - a1 -> x);
        uy = (a2 -> dy - a1 -> dy) * lambda + (a2 -> y - a1 -> y);
        uz = (a2 -> dz - a1 -> dz) * lambda + (a2 -> z - a1 -> z);
        uw = (a2 -> dw - a1 -> dw) * lambda + (a2 -> w - a1 -> w);
        r  = ux * ux + uy * uy + uz * uz + uw * uw;
        if (!r) continue;
        r0  = sqrt(r);
        ux /= r0;
        uy /= r0;
        uz /= r0;
        uw /= r0;
        k   = (-dielectric * a1 -> q * a2 -> q) / r;
        r   = r * r * r;
        k  += (a1 -> a * a2 -> a) / r / r0 * six;
        k  -= (a1 -> b * a2 -> b) / r / r / r0 * twelve;
        a1 -> fx += ux * k;
        a1 -> fy += uy * k;
        a1 -> fz += uz * k;
        a1 -> fw += uw * k;
        a2 -> fx -= ux * k;
        a2 -> fy -= uy * k;
        a2 -> fz -= uz * k;
        a2 -> fw -= uw * k;
      } /* End of for (ii) */
    } /* End of for (i) */
  } else {
    for(i = 0; i < imax; i++) {
      a1   = NbAtomAll[i];
      xt   = a1 -> x - a1 -> px;
      yt   = a1 -> y - a1 -> py;
      zt   = a1 -> z - a1 -> pz;
      wt   = a1 -> w - a1 -> pw;
      fx   = a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt + a1 -> qxw * wt;
      fy   = a1 -> qxy * xt + a1 -> qyy * yt + a1 -> qyz * zt + a1 -> qyw * wt;
      fz   = a1 -> qxz * xt + a1 -> qyz * yt + a1 -> qzz * zt + a1 -> qzw * wt;
      fw   = a1 -> qxw * xt + a1 -> qyw * yt + a1 -> qzw * zt + a1 -> qww * wt;
#ifdef CUBIC
      xt2  = xt * xt;
      yt2  = yt * yt;
      zt2  = zt * zt;
      fx  += a1 -> qxxx * xt2 / 2.0f + a1 -> qxxy * xt * yt + a1 -> qxxz * xt * zt +
             a1 -> qxyy * yt / 2.0f + a1 -> qxyz * yt * zt + a1 -> qxzz * zt2 / 2.0f;
      fy  += a1 -> qxxy * xt2 / 2.0f + a1 -> qxyy * xt * yt + a1 -> qxyz * xt * zt +
             a1 -> qyyy * yt / 2.0f + a1 -> qyyz * yt * zt + a1 -> qyzz * zt2 / 2.0f;
      fz  += a1 -> qxxz * xt2 / 2.0f + a1 -> qxyz * xt * yt + a1 -> qxzz * xt * zt +
             a1 -> qyyz * yt / 2.0f + a1 -> qyzz * yt * zt + a1 -> qzzz * zt2 / 2.0f;
#endif
      a1 -> fx += fx + a1 -> dpx;
      a1 -> fy += fy + a1 -> dpy;
      a1 -> fz += fz + a1 -> dpz;
      a1 -> fw += fw + a1 -> dpw;

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];

        /**** Note ux is backwards from below ****/

        ux = a2 -> x - a1 -> x;
        uy = a2 -> y - a1 -> y;
        uz = a2 -> z - a1 -> z;
        uw = a2 -> w - a1 -> w;
        r  = ux * ux + uy * uy + uz * uz + uw * uw;
        if (!r) continue;
        r0  = sqrt(r);
        ux /= r0;
        uy /= r0;
        uz /= r0;
        uw /= r0;
        k   = (-dielectric * a1 -> q * a2 -> q) / r;
        r   = r*r*r;
        k  += (a1 -> a * a2 -> a) / r / r0 * six;
        k  -= (a1 -> b * a2 -> b) / r / r / r0 * twelve;
        a1 -> fx += ux * k;
        a1 -> fy += uy * k;
        a1 -> fz += uz * k;
        a1 -> fw += uw * k;
        a2 -> fx -= ux * k;
        a2 -> fy -= uy * k;
        a2 -> fz -= uz * k;
        a2 -> fw -= uw * k;
      } /* End of for (ii) */
    } /* End of for (i) */
  }

  /**** Clear the forces of inactive atoms ****/

  for(i = 0; i < imax; i++) {
    a1 = NbAtomAll[i];
    if (a1 -> active) continue;
    a1 -> fx = zero;
    a1 -> fy = zero;
    a1 -> fz = zero;
    a1 -> fw = zero;
  } /* End of for (i) */

  return TRUE;
}


/**** Calculate the non-bond potential ****/

int AMMP_FASTCALL v_shadow(float *V, float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         r, xt, yt, zt, wt;
  float         vx, dielectric;
  int           i, ii, jj, imax;

  fv_update_shadow(lambda);

  dielectric = Q_CONST / GetDielectric();
  imax       = a_number();

  if (lambda) {
    for(i = 0; i < imax; i++) {
      a1  = a_next(i);
      if (!a1 -> active) {
        a1 -> fx = zero;
        a1 -> fy = zero;
        a1 -> fz = zero;
        a1 -> fw = zero;
      }
      vx  = a1 -> VP;
      xt  = a1 -> dx * lambda + a1 -> x - a1 -> px;
      yt  = a1 -> dy * lambda + a1 -> y - a1 -> py;
      zt  = a1 -> dz * lambda + a1 -> z - a1 -> pz;
      wt  = a1 -> dw * lambda + a1 -> w - a1 -> pw;
      vx -= a1 -> dpx * xt + a1 -> dpy * yt + a1 -> dpz * zt + a1 -> dpw * wt;
      vx -= (xt * (0.5f * a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt + a1 -> qxw * wt) +
             yt * (0.5f * a1 -> qyy * yt + a1 -> qyz * zt + a1 -> qyw * wt) +
             zt * (0.5f * a1 -> qzz * zt + a1 -> qzw * wt) + 0.5f * wt * wt * a1 -> qww);

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];
        xt    = (a2 -> dx - a1 -> dx) * lambda + (a2 -> x - a1 -> x);
        yt    = (a2 -> dy - a1 -> dy) * lambda + (a2 -> y - a1 -> y);
        zt    = (a2 -> dz - a1 -> dz) * lambda + (a2 -> z - a1 -> z);
  wt    = (a2 -> dw - a1 -> dw) * lambda + (a2 -> w - a1 -> w);
        r     = xt * xt + yt * yt + zt * zt + wt * wt;
        if (!r) continue;
        vx  += (dielectric * a1 -> q * a2 -> q) / (float)sqrt(r);
        r   = r * r * r;
        vx += (a1 -> b * a2 -> b) / r / r - (a1 -> a * a2 -> a) / r;
      }  /* End of for (ii) */
      *V += vx;
    } /* End of for (i) */
  } else {
    for(i = 0; i < imax; i++) {
      a1  = a_next(i);
      if (!a1 -> active) {
        a1 -> fx = zero;
        a1 -> fy = zero;
        a1 -> fz = zero;
        a1 -> fw = zero;
      }
      vx  = a1 -> VP;
      xt  = a1 -> x - a1 -> px;
      yt  = a1 -> y - a1 -> py;
      zt  = a1 -> z - a1 -> pz;
      wt  = a1 -> w - a1 -> pw;
      vx -= a1 -> dpx * xt + a1 -> dpy * yt + a1 -> dpz * zt + a1 -> dpw * wt;
      vx -= (xt * (0.5f * a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt + a1 -> qxw * wt) +
             yt * (0.5f * a1 -> qyy * yt + a1 -> qyz * zt + a1 -> qyw * wt) +
             zt * (0.5f * a1 -> qzz * zt + a1 -> qzw * wt) + 0.5f * wt * wt * a1 -> qww);

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];
        xt    = a2 -> x - a1 -> x;
        yt    = a2 -> y - a1 -> y;
        zt    = a2 -> z - a1 -> z;
  wt    = a2 -> w - a1 -> w;
        r     = xt * xt + yt * yt + zt * zt + wt * wt;
        if (!r) continue;
        vx  += (dielectric * a1 -> q * a2 -> q) / (float)sqrt(r);
        r   = r * r * r;
        vx += (a1 -> b * a2 -> b) / r / r - (a1 -> a * a2 -> a) / r;
      }  /* End of for (ii) */
      *V += vx;
    } /* End of for (i) */
  }

  return TRUE;
}


/**** Initialize fourd (w) ****/

void AMMP_FASTCALL init_fourd(float howmuch)
{
  AMMP_ATOM *   ap;
  int           i, numatm;

  numatm = a_number();
  if (numatm <= 0) return;
  if (howmuch < 1.e-3f) howmuch = one;
  for(i = 0; i < numatm; i++) {
    ap = a_next(i);
    if (ap -> active)
      ap -> w = howmuch * two * randf() - one;
  } /* End of for (i) */
}


/* routines to force w to zero
* the idea is that after generating a 4-d structure you make it 3-d
* by slowly restraining w to zero
*/

int AMMP_FASTCALL f_fourd(float lambda)
{
  AMMP_ATOM *   ap;
  float         kfourd;
  int           i, numatm;

  kfourd = get_f_variable("kfourd");
  if (kfourd <= 0) return FALSE;

  numatm = a_number();
  if (numatm <= 0) return TRUE;

  if (lambda) {
    for(i = 0; i < numatm; i++) {
      ap        = a_next(i);
      ap -> fw -= kfourd * (ap -> w + lambda * ap -> dw);
    } /* End of for (i) */
  } else {
    for(i = 0; i < numatm; i++) {
      ap        = a_next(i);
      ap -> fw -= kfourd * ap -> w ;
    } /* End of for (i) */
  }

  return TRUE;
}


int AMMP_FASTCALL v_fourd(float *V, float lambda)
{
  AMMP_ATOM *   ap;
  float         kfourd, wt;
  int           i, numatm;


  kfourd = get_f_variable("kfourd");
  if (kfourd <= 0) return FALSE;

  numatm = a_number();
  if (numatm <= 0) return TRUE;

  if (lambda) {
    for (i = 0; i <numatm; i++) {
      ap  = a_next(i);
      wt  = ap -> w + lambda * ap -> dw;
      *V += 0.5f * kfourd * wt * wt;
    } /* End of for (i) */
  } else {
    for (i = 0; i <numatm; i++) {
      ap  = a_next(i);
      wt  = ap -> w;
      *V += 0.5f * kfourd * wt * wt;
    } /* End of for (i) */
  }

  return TRUE;
}

