/*  simplex.c
*
*  polytope simplex minimizer
*
* AMMP version
*  given a range of atom ID's and uses them to define hull
*  also uses the total potential,
*
*
*/
/*
*  copyright 1993 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify
*  this notice
*/

#include <stdio.h>
#include <stdlib.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"


int AMMP_FASTCALL simplex(FILE *op, float toler, int niter, float var, AMMP_VFUNC potent[],
                          int inpotent, int imin, int imax, int echo)
{
  AMMP_ATOM     *Atm, *(*ap)[];
  AMMP_FFUNC    ftem[2];
  AMMP_VFUNC    ptem[2];
  float         *polytope;
  float         x;
  int           i, j, k, l, natom, nargs, mycase;
  int           worst;

  const char *  Routine  = "simplex()";
  float         *fvals   = NULL;
  float         *reflect = NULL;
  int           best     = 0;

  if (!CheckAtoms()) return FALSE;
  if (niter < 1) niter = 1;

  /**** Check the parameters ****/

  if (imax <= 0 ) {
    natom = a_number();
    for(i = 0; i < natom; i++) {
      Atm = a_next(i);
      if (imax < Atm -> serial) imax = Atm -> serial;
    } /* End of for (i) */
  }

  if (imin > imax) {
    i    = imin;
    imin = imax;
    imax = i;
  }
  natom = 0;
  for(i = imin; i <= imax; i ++)
    if (a_m_serial(i) != NULL) natom++;

  if (!natom) {
    aaerror("No atoms in the selection range");
    return FALSE;
  }
  nargs = natom * 3;

  ptem[0] = v_bond;
  ftem[0] = f_bond;
  ptem[1] = v_angle;
  ftem[1] = f_angle;

  if (((polytope = (float *)Alloca(nargs * (nargs + 1) * sizeof(float), Routine)) == NULL) ||
      ((fvals    = (float *)Alloca(        (nargs + 1) * sizeof(float), Routine)) == NULL) ||
      ((reflect  = (float *)Alloca(nargs *               sizeof(float), Routine)) == NULL) ||
      ((ap       =          Alloca(natom * sizeof(AMMP_ATOM *)        , Routine)) == NULL)) {
    SafeFree(polytope);
    SafeFree(fvals   );
    SafeFree(reflect );
    return FALSE;
  }

  /**** Now gather up the atoms ****/

  for(i = 0; i < natom; i++) (*ap)[i] = NULL;

  j = 0;
  for(i = imin; i <= imax; i ++) {
    if (((*ap)[j] = a_m_serial(i)) != NULL) {
      (*ap)[j] -> gx = (*ap)[j] -> x;
      (*ap)[j] -> gy = (*ap)[j] -> y;
      (*ap)[j] -> gz = (*ap)[j] -> z;
      j++;
    }
    if (j == natom) break;
  } /* End of for (i) */

  for(i = 0 ; i < (nargs + 1); i++) {
    for(j = 0; j < nargs; j += 3) {
      k = j / 3;
      polytope[i * nargs + j    ] = (*ap)[k] -> gx + var * (2.0f * randf() - 1.0f);
      polytope[i * nargs + j + 1] = (*ap)[k] -> gy + var * (2.0f * randf() - 1.0f);
      polytope[i * nargs + j + 2] = (*ap)[k] -> gz + var * (2.0f * randf() - 1.0f);
      (*ap)[k] -> x = polytope[i * nargs + j    ];
      (*ap)[k] -> y = polytope[i * nargs + j + 1];
      (*ap)[k] -> z = polytope[i * nargs + j + 2];
    } /* End of for (j) */

    steep(ptem, ftem, 2, 5, 0.0f, 0);

    for(j = 0; j < nargs; j += 3) {
      k = j / 3;
      polytope[ i*nargs + j    ] = (*ap)[k] -> x;
      polytope[ i*nargs + j + 1] = (*ap)[k] -> y;
      polytope[ i*nargs + j + 2] = (*ap)[k] -> z;
    } /* End of for (j) */

    fvals[i] = 0.0f;
    for(k = 0; k < inpotent; k++) {
      if ((*potent[k] != v_nonbon) && (*potent[k] != u_v_nonbon))
        (*potent[k])(fvals + i, 0.0f);
      else
        zone_nonbon(fvals + i, 0.0f, ap, natom);
    } /* End of for (k) */
  } /* End of for (i) */

  for(k = 0; k < niter; k++ ) {
    best = simplex_get_best(fvals, nargs);
    if ((var = simplex_get_var(fvals, nargs + 1)) < toler ) goto DONE;
    worst = simplex_get_worst(fvals, nargs);
#ifndef VEGAZZ
    if (echo)
#endif
      fprintf(op, "  Polytope %3d: Best %d energy %f, worst %d energy %f\n",
              k, best, fvals[best], worst, fvals[worst]);
    mycase = 0;

NEW_HULL:

    simplex_reflect(polytope, reflect , nargs, worst, 2.0f);
    for( j = 0; j < nargs; j += 3) {
      i = j/3;
      (*ap)[i] -> x = reflect[j    ];
      (*ap)[i] -> y = reflect[j + 1];
      (*ap)[i] -> z = reflect[j + 2];
    } /* End of for (j) */

EVALUATE:

    steep(ptem, ftem, 2, 10, 0.0f, 0);

    for(j = 0; j< nargs; j += 3) {
      i = j/3;
      reflect[j    ] = (*ap)[i] -> x;
      reflect[j + 1] = (*ap)[i] -> y;
      reflect[j + 2] = (*ap)[i] -> z;
    } /* End of for (j) */

    x = 0.0f;
    for(j = 0; j < inpotent; j++) {
      if ((*potent[j] != v_nonbon) && (*potent[j] != u_v_nonbon))
        (*potent[j])(&x, 0.0f);
      else
        zone_nonbon(&x, 0.0f, ap, natom);
    } /* End of for (j) */

    if (x >= fvals[worst]) {
      if (mycase == 0) {
        mycase = 1;
	simplex_reflect(polytope, reflect , nargs, worst, 1.0f);
      }

      if (mycase == 1) {
        mycase = 2;
	simplex_reflect(polytope, reflect, nargs, worst, 0.25f);
      }

      if (mycase == 2) {
        mycase = 3;
        simplex_reflect(polytope, reflect, nargs, worst, 0.0f);
      }

      if (mycase == 3) {
        mycase = 4;
        simplex_reflect(polytope, reflect, nargs, worst, -0.25f);
      }

      if (mycase == 4 ) {
        mycase = 5;
        simplex_reflect(polytope, reflect, nargs, worst, -0.5f);
      }

      if (mycase == 5) {
        /* desparation !!! */
	simplex_contract( polytope, reflect, nargs, best, worst, 0.5f);
        mycase = 0;
        for(i = 0; i < nargs + 1; i++) {
          for(j = 0; j < nargs; j+=3) {
            l = j / 3;
            (*ap)[l] -> x = polytope[i * nargs + j    ];
            (*ap)[l] -> y = polytope[i * nargs + j + 1];
            (*ap)[l] -> z = polytope[i * nargs + j + 2];
          } /* End of for (j) */

          steep(ptem, ftem, 2, 5, 0.0f, 0);
          for(j = 0; j < nargs; j += 3) {
            l = j / 3;
            polytope[i * nargs + j     ] = (*ap)[l] -> x;
            polytope[i * nargs + j + 1 ] = (*ap)[l] -> y;
            polytope[i * nargs + j + 2 ] = (*ap)[l] -> z;
          } /* End of for (j) */

          fvals[i] = 0.0f;
          for(l = 0; l < inpotent; l++) {
            if ((*potent[l] != v_nonbon) && (*potent[l] != u_v_nonbon))
              (*potent[l])(fvals + i, 0.0f);
            else
              zone_nonbon(fvals + i , 0.0f, ap,natom);
          } /* End of for (l) */
        } /* End of for (i) */

        best = simplex_get_best(fvals, nargs);
        if ((var = simplex_get_var(fvals, nargs + 1)) < toler) goto DONE;
        worst = simplex_get_worst(fvals, nargs);
#ifndef VEGAZZ
        if (echo)
#endif
          fprintf(op, "  Polytope %3d:\n"
                      "    Reflect best %d energy %f\n"
                      "    Reflect worst %d energy %f\n"
                      "    Refelct %f var %f toler\n",
                      k, best, fvals[best],
                      worst, fvals[worst],
                      var, toler);
        goto NEW_HULL;
      }

      for(j = 0; j < nargs; j+=3) {
        i = j / 3;
        (*ap)[i] -> x = reflect[j    ];
        (*ap)[i] -> y = reflect[j + 1];
        (*ap)[i] -> z = reflect[j + 2];
      } /* End of for (j) */

      goto EVALUATE;
    }

    fvals[worst] = x;
    for(j = 0; j < nargs; j++)
      polytope[worst * nargs + j] = reflect[j];
  }

DONE:

#ifndef VEGAZZ
  if (echo)
#endif
    fprintf(op, "  Polytope    : Putting best %d into coordinates\n\n", best);

  for(i = 0; i < nargs; i += 3) {
    j = i / 3;
    (*ap)[j] -> x = polytope[best * nargs + i    ];
    (*ap)[j] -> y = polytope[best * nargs + i + 1];
    (*ap)[j] -> z = polytope[best * nargs + i + 2];
  } /* End of for (i) */

  free(ap      );
  free(reflect );
  free(fvals   );
  free(polytope);

  return TRUE;
}


/* simplex_get_best( fvals, number );
*  return the array index in fvals of the best (most negative)
* value of f
*/

int AMMP_FASTCALL simplex_get_best(float *fvals, int n)
{
  int           i ;

  float         x   = 10e10f;
  int           who = -1;

  ++n;
  for(i = 0; i < n; i++) {
    if (x > fvals[i]) {
      who = i;
      x = fvals[i];
    }
  } /* End of for (i) */

  if (who < 0) who = 0;

  return who;
}


/* simplex_get_worst( fvals, number );
*  return the array index in fvals of the worst (least negative)
* value of f
*/

int AMMP_FASTCALL simplex_get_worst(float *fvals, int n)
{
  int           i;

  int           who = -1;
  float         x   = -10e10f;

  ++n;
  for(i = 0; i < n; i++) {
    if (x < fvals[i]) {
      who = i;
      x   = fvals[i];
    }
  } /* End of for (i) */

  if (who < 0) who = 0;

  return who;
}


float AMMP_FASTCALL simplex_get_var(float *fvals, int n)
{
  int           i;

  float         sx  = 0.0f;
  float         sx2 = 0.0f;
  float         t   = 1.0f / (float)n;


  for(i = 0; i < n; i++) {
    sx  += fvals[i];
    sx2 += fvals[i] * fvals[i];
  } /* End of for (i) */

  sx *= t;

  return (sx2 * t - sx * sx);
}


/**** Reflect the worst around the mean ****/

void AMMP_FASTCALL simplex_reflect(float *pt, float *rf, int n, int worst, float wait)
{
  int           i, j, k;
  float         nf, t;

  for(j = 0; j < n; j++) rf[j] = 0.0f;

  for(i = 0, k = 0; i < worst; i++, k += n) {
    for(j = 0; j < n; j++)
      rf[j] += pt[k + j];
  } /* End of for (i) */

  for(i = worst + 1; i < (n + 1); i++) {
    k = i * n;
    for(j = 0; j < n; j++)
      rf[j] += pt[k + j];
  } /* End of for (i) */


  nf = (float)n;
  k  = worst * n;
  for(j = 0; j < n; j++) {
    t     = rf[j] / nf;
    rf[j] = t + wait * (t - pt[k  + j]);
  } /* End of for (j) */
}


void AMMP_FASTCALL simplex_contract(float *pt, float *rf, int n, int best, int worst, float howmuch)
{
  int           i, j, k, l;

  best *= n;
  for(i = 0, k = 0; i < n + 1; i++, k += n) {
    for(j = 0, l = k; j < n; ++j, ++l)
      pt[l] = (2.0f * pt[l] + pt[best + j]) * (1.0f / 3.0f);
  } /* End of for (i) */
}


/**** Simplex/quarternion rigid body solver ****/

int AMMP_FASTCALL rigid(FILE *op, float toler, int niter, float var, AMMP_VFUNC potent[],
                        int inpotent, int imin, int imax, int echo)
{
  AMMP_ATOM     *Atm, *(*ap)[];
  float         polytope[56];
  float         fvals[8];
  float         reflect[8];
  float         x;
  int           i,j,k,l,natom,nargs,mycase;
  int           worst;

  int           best     = 0;

  if (!CheckAtoms()) return FALSE;
  if (niter < 1) niter = 1;

  /**** Check the parameters ****/

  if (imax <= 0 ) {
    natom = a_number();
    for(i = 0; i < natom; i++) {
      Atm = a_next(i);
      if (imax < Atm -> serial) imax = Atm -> serial;
    } /* End of for (i) */
  }

  if (imin > imax) {
    i    = imin;
    imin = imax;
    imax = i;
  }

  natom = 0;
  for(i = imin; i <= imax; i++)
    if (a_m_serial(i) != NULL) natom++;

  if (!natom) return FALSE;
  nargs = 7;

  if ((ap = Alloca(natom * sizeof(AMMP_ATOM *), "rigid()")) == NULL)
    return FALSE;

  /**** Now gather up the atoms ****/

  for(i = 0; i < natom; i++) (*ap)[i] = NULL;
  j = 0;
  for(i = imin; i <= imax; i ++) {
    if (((*ap)[j] = a_m_serial(i)) != NULL) {
      (*ap)[j] -> gx = (*ap)[j] -> x;
      (*ap)[j] -> gy = (*ap)[j] -> y;
      (*ap)[j] -> gz = (*ap)[j] -> z;
      j++;
    }
    if (j == natom) break;
  } /* End of for (i) */

  /**** Initialize the hull ****/

  for(i = 0; i < 8; i++) {
    for(j = 0; j< 7; j++)
      polytope[i * nargs + j] = var * (2.0f * randf() - 1.0f);

    polytope[i * nargs + 3] += 1.0f / var; /* I = 1 0 0 0*/
    quarternion_rot_tran(&polytope[i * nargs], ap, natom);
    fvals[i] = 0.0f;
    for(j = 0; j < inpotent; j++) {
      if ((*potent[j] != v_nonbon) && (*potent[j] != u_v_nonbon))
        (*potent[j])(&fvals[i], 0.0f);
      else
        zone_nonbon(&fvals[i], 0.0f, ap, natom);
    } /* End of for (j) */
  } /* End of for (i) */

  for(k = 0; k < niter; k++) {
    best = simplex_get_best(fvals, nargs);
    if ((var = simplex_get_var(fvals, nargs + 1)) < toler) goto DONE;
    worst = simplex_get_worst(fvals, nargs);
#ifndef VEGAZZ
    if (echo)
#endif
      fprintf(op, "  Rigid  %5d: Best %d energy %f, worst %d energy %f\n",
              k, best, fvals[best], worst, fvals[worst]);
    mycase = 0;

NEW_HULL:
    simplex_reflect(polytope, reflect , nargs, worst , 2.0f);
    quarternion_rot_tran(&reflect[0], ap, natom);

EVALUATE:
    x = 0.0f;
    for(j = 0; j < inpotent; j++) {
      if ((*potent[j] != v_nonbon) && (*potent[j] != u_v_nonbon))
        (*potent[j])(&x, 0.0f);
      else
        zone_nonbon(&x, 0.0f, ap, natom);
    } /* End of for (j) */

    if (x >= fvals[worst]) {
      if (mycase == 0) {
        mycase = 1;
        simplex_reflect(polytope, reflect, nargs, worst, 1.0f);
      }

      if (mycase == 1) {
        mycase = 2;
        simplex_reflect(polytope, reflect, nargs, worst , 0.25f);
      }

      if (mycase == 2) {
        mycase = 3;
        simplex_reflect(polytope, reflect, nargs, worst, 0.0f);
      }

      if (mycase == 3) {
        mycase = 4;
        simplex_reflect(polytope, reflect, nargs, worst , -0.25f);
      }

      if (mycase == 4) {
        mycase = 5;
        simplex_reflect( polytope, reflect , nargs, worst , -0.5f);
      }

      if (mycase == 5) {
        /* desparation !!! */
        simplex_contract(polytope, reflect, nargs, best, worst, 0.5f);
        mycase = 0;
        for(i = 0; i < (nargs + 1); i++) {
          quarternion_rot_tran(&polytope[i * nargs], ap, natom);
          fvals[i] = 0.0f;
          for(l = 0; l < inpotent; l++) {
            if ((*potent[l] != v_nonbon) && (*potent[l] != u_v_nonbon))
              (*potent[l])(&fvals[i], 0.0f);
            else
              zone_nonbon(&fvals[i], 0.0f, ap, natom);
          } /* End of for (l) */
        } /* End of for (i) */
        best = simplex_get_best( fvals,nargs );
        if ((var = simplex_get_var(fvals, nargs + 1 )) < toler) goto DONE;
        worst = simplex_get_worst(fvals, nargs);

#ifndef VEGAZZ
        if (echo)
#endif
          fprintf(op, "  Rigid  %5d:\n"
                      "    Reflect best %d energy %f\n"
                      "    Reflect worst %d energy %f\n"
                      "    Refelct %f var %f toler\n",
                      k, best, fvals[best],
                      worst, fvals[worst],
                      var, toler);
        goto NEW_HULL;
      }

      quarternion_rot_tran(&reflect[0], ap, natom);
      goto EVALUATE;
    }
    fvals[worst] = x;

    for(j = 0; j < nargs ; j++)
      polytope[worst * nargs + j] = reflect[j];
  } /* End of for (k) */

DONE:
#ifndef VEGAZZ
  if (echo)
#endif
    fprintf(op, "  Rigid       : Putting best %d into coordinates\n\n", best);

  quarternion_rot_tran(&polytope[best*nargs],ap,natom);
  free(ap);

  return TRUE;
}


/**** Quarternion rototranslation ****/

void AMMP_FASTCALL quarternion_rot_tran(float what[], AMMP_ATOM *(*who)[], int howmany)
{
  float         rot[3][3];
  float         x, y, z;
  float         x1, x2, x3, x4, x5, x6;
  int           i;

  float         norm = sqrt((double)(what[3] * what[3] + what[4] * what[4] +
                                     what[5] * what[5] + what[6] * what [6]));

  if (!norm) return;

  norm      = one / norm;
  what[3]  *= norm;
  what[4]  *= norm;
  what[5]  *= norm;
  what[6]  *= norm;
  x1        = what[3];
  x2        = what[4];
  x3        = what[5];
  x4        = what[6];
  rot[0][0] = x1 * x1 + x2 * x2 - x3 * x3 - x4 * x4;
  rot[1][0] = two * (x2 * x3 + x1 * x4);
  rot[2][0] = two * (x2 * x4 - x1 * x3);
  rot[0][1] = two * (x2 * x3 - x1 * x4);
  rot[1][1] = x1 * x1 - x2 * x2 + x3 * x3 - x4 * x4;
  rot[2][1] = two * (x3 * x4 + x1 * x2);
  rot[0][2] = two * (x2 * x4 + x1 * x3);
  rot[1][2] = two * (x3 * x4 - x1 * x2);
  rot[2][2] = x1 * x1 - x2 * x2 - x3 * x3 + x4 * x4;
  x4        = zero;
  x5        = zero;
  x6        = zero;

  for(i = 0; i < howmany; i++) {
    x4 += (*who)[i]->gx;
    x5 += (*who)[i]->gy;
    x6 += (*who)[i]->gz;
  } /* End of for (i) */

  x   = (float)howmany;
  x4 /= x;
  x5 /= x;
  x6 /= x;

  for(i = 0; i < howmany; i++) {
    x               = what[0] + x4;
    y               = what[1] + x5;
    z               = what[2] + x6;
    x1              = (*who)[i] -> gx - x4;
    x2              = (*who)[i] -> gy - x5;
    x3              = (*who)[i] -> gz - x6;
    x              += rot[0][0] * x1 + rot[0][1] * x2 + rot[0][2] * x3;
    y              += rot[1][0] * x1 + rot[1][1] * x2 + rot[1][2] * x3;
    z              += rot[2][0] * x1 + rot[2][1] * x2 + rot[2][2] * x3;
    (*who)[i] -> x  = x;
    (*who)[i] -> y  = y;
    (*who)[i] -> z  = z;
  } /* End of for (i) */
}

