/* table.c
* support table driven potentials
*
*  ammp commands 
*	table id n r0 v0 r1 v1 .. r(n-1) v(n-1);
*       dump table ;
*	use tbond; 
*       tbond i j id scale;
*       dump tbond;
*	
*  table ... is implemented as 
*   create_table(), and add_pair_to_table;
*   tables are overwritable
*
*   alse have tbond(),v_tbond(), and f_tbond();
*
*
*
*/
/*
*  copyright 1993,1994,1995,1996 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"


/**** Local variables ****/

static DATATABLE *      add_pair_to_table_who = NULL;


/**** Reset local variables ****/

void AMMP_FASTCALL ResetTable(void)
{
  add_pair_to_table_who = NULL;
}


/*** Add a pait to table ****/

void AMMP_FASTCALL add_pair_to_table(int id, int count, float r, float v)
{
  if (first_datatable == NULL) return;
  if (add_pair_to_table_who == NULL)
    add_pair_to_table_who = first_datatable;

  if (add_pair_to_table_who -> id != id) {
    add_pair_to_table_who = first_datatable;
    while(1) {
      if (add_pair_to_table_who -> id == id) break;
      if (add_pair_to_table_who -> next == add_pair_to_table_who) return;
      add_pair_to_table_who = add_pair_to_table_who -> next;
    } /* End of while */
  }
  add_pair_to_table_who -> r[count] = r;
  add_pair_to_table_who -> v[count] = v;
}


/**** Create a new table ****/

int AMMP_FASTCALL create_table(int id, int inme)
{
  DATATABLE *   new;
  int           i;

  const char *  Routine = "create_table()";

  if (first_datatable == NULL) {
   if ((new = (DATATABLE *)Alloca(sizeof(DATATABLE), Routine)) == NULL)
     return FALSE;

   first_datatable        = new;
   last_datatable         = new;
   last_datatable -> next = new;
   new -> next            = new;
  } else {
    new = first_datatable;
    while(new -> id != id) {
      if (new -> next == new) break;
      new = new -> next;
    } /* End of while */
    if ((new -> next == new) && (new -> id != id)) {
      if ((new = (DATATABLE *)Alloca(sizeof(DATATABLE), Routine)) == NULL)
        return FALSE;
      last_datatable -> next = new;
      last_datatable         = new;
      new -> next            = new;
    }
  }
  new -> id   = id;
  new -> inme = inme;
  for(i = 0; i < inme; i++) {
    new -> r[i] = zero;
    new -> v[i] = zero;
  } /* End of for (i) */

  return TRUE;
}


/**** Dump the tables ****/

void AMMP_FASTCALL dump_table(FILE * where)
{
  int           i;

  DATATABLE *   who = first_datatable;

  while(who) {
    fprintf(where, "table %d %d;\n", who -> id, who -> inme);
    for(i = 0; i < who -> inme; i++)
      fprintf(where,"tableent %d %d %f %f;\n", who -> id, i, who -> r[i], who -> v[i]);
    fprintf(where, ";\n");
    if (who -> next == who) break;
    who = who -> next;
  } /* End of while */
}


/**** Dump tbonds ****/

void AMMP_FASTCALL dump_tbond(FILE *where)
{
  TBOND *       who = first_tbond;

  while(who) {
    fprintf(where,"tbond %d %d %d %f;\n",
	    who -> a1 -> serial, who -> a2 -> serial, who -> which -> id,
            who -> scale);
    who = who -> next;
  } /* End of while */
}


/**** Add a new tbond ****/

int AMMP_FASTCALL tbond(int s1, int s2, int id, float scale)
{
  TBOND *       new;
  DATATABLE *   table;

  AMMP_ATOM *   a1 = a_m_serial(s1);
  AMMP_ATOM *   a2 = a_m_serial(s2);

  if ((!a1) || (!a2)) {
    aaerror("Undefined atom in tbond()");
    return FALSE;
  }

  table = first_datatable;
  if (table == NULL) return FALSE;

  while(1) {
    if (table -> id == id) break;
    if (table -> next == table) {
      aaerror("Undefined bond table in tbond()");
      return FALSE;
    }
    table = table -> next;
  } /* End of while */

  if ((new = (TBOND *)Alloca(sizeof(TBOND), "tbond()")) == NULL)
    return FALSE;

  if (first_tbond == NULL) first_tbond = new;
  if (last_tbond == NULL) last_tbond = new;
  last_tbond -> next = new;
  last_tbond         = new;
  new -> next        = new;
  new -> a1          = a1;
  new -> a2          = a2;
  new -> scale       = scale;
  new -> which       = table;

  return TRUE;
}


/**** Increment the tbond force ****/

int AMMP_FASTCALL f_tbond(float lambda)
{
  AMMP_ATOM     *a1, *a2;
  DATATABLE *   tp;
  float         r, xt, yt, zt;
//  float         vl, dvl;
  float         dvl;
  float         rred, rx;
  float         matrix[3][3], vector[3];
  int           i;

  TBOND *       bp = first_tbond;

  while(bp) {
    a1 = bp -> a1;
    a2 = bp -> a2;
    if (lambda) {
      xt = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
      yt = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
      zt = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
    } else {
      xt = a1 -> x - a2 -> x;
      yt = a1 -> y - a2 -> y;
      zt = a1 -> z - a2 -> z;
    }
    r = sqrt(xt * xt + yt * yt + zt * zt);
    if (r < 2.5){
      dvl = one;
      if (r < 1.e-4) r = 1.e-4;
      goto SKIP;
    }
    tp = bp -> which;
    if ((r < tp -> r[0]) || (r > tp -> r[tp -> inme - 1])) {
      dvl = 0.0f;
      if (r < tp -> r[0]) dvl = - bp -> scale;
      if (r > tp -> r[tp -> inme - 1]) dvl = zero;
//      vl = one;
    } else {
      matrix[0][0] = zero;
      matrix[1][0] = zero;
      matrix[2][0] = zero;
      matrix[0][1] = zero;
      matrix[1][1] = zero;
      matrix[2][1] = zero;
      matrix[0][2] = zero;
      matrix[1][2] = zero;
      matrix[2][2] = zero;
      vector[0]    = zero;
      vector[1]    = zero;
      vector[2]    = zero;

      for(i = 0; i < tp -> inme; i++) {
        rred =  tp -> r[i] - r;
        if (fabs(rred) < 0.25) {
          matrix[0][0] += one;
          vector[0]    += tp -> v[i];
          matrix[0][1] += rred;
          vector[1]    += tp -> v[i] * rred;
          rx            = rred * rred;
          matrix[0][2] += rx;
          matrix[1][1] += rx;
          vector[2]    += rx * tp -> v[i];
          rx           *= rred;
          matrix[1][2] += rx;
          matrix[2][2] += rx * rred;
        }
      } /* End of for (i) */

      matrix[1][0] = matrix[0][1];
      matrix[2][0] = matrix[0][2];
      matrix[2][1] = matrix[1][2];

      mom_solve((float (*)[])&matrix[0][0], (float (*)[])&vector[0], 3, 3);
//      vl  = vector[0];
      dvl = vector[1];
    }

//    if (vl < 1.e-5f) vl = 1.e-5f;
    dvl = - dvl * bp -> scale;

SKIP:
    xt = - dvl * xt / r;
    yt = - dvl * yt / r;
    zt = - dvl * zt / r;

    if (a1 -> active) {
      a1 -> fx += xt;
      a1 -> fy += yt;
      a1 -> fz += zt;
    }
    if (a2 -> active) {
      a2 -> fx -= xt;
      a2 -> fy -= yt;
      a2 -> fz -= zt;
    }


    if (bp == bp->next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}


/**** Calculate the tbond potential ****/

int AMMP_FASTCALL v_tbond(float *V, float lambda)
{
  AMMP_ATOM     *a1,*a2;
  DATATABLE *   tp;
  float         r, xt, yt, zt;
  float         vl;
  float         rred, rx;
  float         matrix[3][3], vector[3];
  int           i;
	
  TBOND *       bp = first_tbond;

  while(bp) {
    a1 = bp -> a1;
    a2 = bp -> a2;

    if (lambda) {
      xt = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
      yt = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
      zt = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
    } else {
      xt = a1 -> x - a2 -> x;
      yt = a1 -> y - a2 -> y;
      zt = a1 -> z - a2 -> z;
    }
    r  = sqrt(xt * xt + yt * yt + zt * zt);
    tp = bp -> which;

    if ((r < tp -> r[0]) || (r > tp -> r[tp -> inme - 1])) {
      vl = 0.0f;
      if (r < tp -> r[0]) vl = tp -> v[0];
      if (r > tp -> r[tp -> inme - 1]) vl = tp -> v[tp -> inme - 1];
    } else {
      matrix[0][0] = zero;
      matrix[1][0] = zero;
      matrix[2][0] = zero;
      matrix[0][1] = zero;
      matrix[1][1] = zero;
      matrix[2][1] = zero;
      matrix[0][2] = zero;
      matrix[1][2] = zero;
      matrix[2][2] = zero;
      vector[0]    = zero;
      vector[1]    = zero;
      vector[2]    = zero;
      for(i = 0; i < tp -> inme; i++) {
        rred = tp -> r[i] - r;
        if (fabs(rred) < 0.25) {
          matrix[0][0] += one;
          vector[0]    += tp -> v[i];
          matrix[0][1] += rred;
          vector[1]    += tp -> v[i] * rred;
          rx            = rred * rred;
          matrix[0][2] += rx;
          matrix[1][1] += rx;
          vector[2]    += rx * tp -> v[i];
          rx           *= rred;
          matrix[1][2] += rx;
          matrix[2][2] += rx * rred;
        }
      } /* End of for (i) */

      matrix[1][0] = matrix[0][1];
      matrix[2][0] = matrix[0][2];
      matrix[2][1] = matrix[1][2];

      mom_solve((float (*)[])&matrix[0][0], (float (*)[])&vector[0],3,3);
      vl = vector[0];
    }

    if (vl < 1.e-5f) vl = 1.e-5f;
    *V -= bp -> scale * vl ;
    if (bp == bp -> next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}
