/* tether.c
*
* collection of routines to service tether  potentials
*  these fix an atom to a location
*
* POOP (Poor-mans Object Oriented Programming) using scope rules
*
* these routines hold a data base (in terms of array indeces)
* of tethers, with the associated position and force constant
*
* (this could be table driven but what the hell memories cheap)
*
* the routines for potential value, force and (eventually) second
* derivatives are here also
*
* force and 2nd derivative routines assume zero'd arrays for output
* this allows for parralellization if needed (on a PC?)
*
*/
/*
*  copyright 1992 Robert W. Harrison
*
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify
*  this notice
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"


/**** Sum the tether potential by the atom range ****/

int AMMP_FASTCALL a_tether(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
  AMMP_ATOM *   a1;
  float         r, xt, yt, zt;
  float         *x, *y, *z, *xx, *yy, *zz;
  float         matrix[3][3], delta[3];
  int           imax, numatm;

  const char *  Routine = "a_tether()";
  float         rms     = zero;
  float         rmax    = - one;
  int           tried   = 0;
  TETHER *      bp      = tether_first;

  if (!bp) return TRUE;

  numatm = a_number();
  imax   = numatm * sizeof(float);

  if ((x = (float *)Alloca(imax, Routine)) == NULL)
    return FALSE;

  if ((y = (float *)Alloca(imax, Routine)) == NULL) {
    free(x);
    return FALSE;
  }

  if ((z = (float *)Alloca(imax, Routine)) == NULL) {
    free(x);
    free(y);
    return FALSE;
  }

  if ((xx = (float *)Alloca(imax, Routine)) == NULL) {
    free(x);
    free(y);
    free(z);
    return FALSE;
  }

  if ((yy = (float *)Alloca(imax, Routine)) == NULL) {
    free(x);
    free(y);
    free(z);
    free(xx);
    return FALSE;
  }

  if ((zz = (float *)Alloca(imax, Routine)) == NULL) {
    free(x);
    free(y);
    free(z);
    free(xx);
    free(yy);
    return FALSE;
  }

  while(bp) {
    if (bp == NULL) return FALSE;
    a1 = bp -> atom1;
    if ((a1 -> serial >= ilow) && (a1 -> serial <= ihigh)) {
      if (lambda) {
        xt       = a1 -> x - bp -> x + lambda * a1 -> dx;
        yt       = a1 -> y - bp -> y + lambda * a1 -> dy;
        zt       = a1 -> z - bp -> z + lambda * a1 -> dz;
	x[tried] = a1 -> x + lambda * a1 -> dx;
	y[tried] = a1 -> y + lambda * a1 -> dy;
	z[tried] = a1 -> z + lambda * a1 -> dz;
      } else {
        xt       = a1 -> x - bp -> x;
        yt       = a1 -> y - bp -> y;
        zt       = a1 -> z - bp -> z;
	x[tried] = a1 -> x;
	y[tried] = a1 -> y;
	z[tried] = a1 -> z;
      }
      r  = xt * xt + yt * yt + zt * zt;
      xx[tried] = bp -> x;
      yy[tried] = bp -> y;
      zz[tried] = bp -> z;
      ++tried;
      rms += r;
      if (r > rmax) {
        rmax = r;
        imax = a1 -> serial;
      }

      zt  = bp -> k * r;
      *V += zt;
      fprintf(op, "Tether %d E %f error %f\n", a1 -> serial, zt, sqrt(r));
    }
    if (bp == bp -> next) break;
    bp = bp -> next;
  } /* End of while */

  if (tried > 0) {
    rms  = sqrt(rms / tried);
    rmax = sqrt(rmax);
    fprintf(op, " RMSD %f Maximum Deviation %f on atom %d\n"
                " RMSD after superposition %f\n",
                rms, rmax, imax,
		bstrot(x, y, z, xx, yy, zz, tried, matrix, delta));
  }

  free(zz);
  free(yy);
  free(xx);
  free(z);
  free(y);
  free(x);

  return TRUE;
}


/**** Tether all atoms ****/

int AMMP_FASTCALL alltether(float fk)
{
  AMMP_ATOM *        ap = atomfirst;

  if (ap == NULL) {
    aaerror("No atoms to tether");
    return 0;
  }

  while(ap) {
    if ((fabs(ap -> x) > 1.e-8) &&
        (fabs(ap -> y) > 1.e-8) &&
        (fabs(ap -> z) > 1.e-8) &&
        (!tether(ap -> serial, fk, ap -> x, ap -> y, ap -> z)))
      return 0;
    if (ap == ap -> next) return a_number();
    ap = ap -> next;
  } /* End of while */

  return 0;
}


/**** Copy a vector into another ****/

void AMMP_FASTCALL cpyvec(float orig[], float copy[], int n)
{
  register int  i;

  for(i = 0; i < n; i++) copy[i] = orig[i];
}


/**** Dunp the tethers ****/

void AMMP_FASTCALL dump_tethers(FILE *where)
{
  TETHER        *b = tether_first;

  while(b) {
    fprintf(where, "tether %d %f %f %f %f;\n",
            b -> atom1 -> serial, b -> k, b -> x, b -> y, b -> z);
    if (b -> next == b) break;
    b = b -> next;
  } /* End of while */
}


/**** Increment the tethering force (homotopy version) ****/

int AMMP_FASTCALL f_ho_tether(float lambda)
{
  AMMP_ATOM     *a1;
/*
  float         r;
*/
  float         k, ux, uy, uz;
  float         hol;

  TETHER        *bp = tether_first;

  if (!bp) return TRUE;

  hol = one - GetLambda();

  if (lambda) {
    while(bp) {
      a1 = bp -> atom1;
      if (a1 -> active) {
	ux = a1 -> x - bp -> x + lambda * a1 -> dx;
	uy = a1 -> y - bp -> y + lambda * a1 -> dy;
	uz = a1 -> z - bp -> z + lambda * a1 -> dz;
/*
	r  = ux * ux + uy * uy + uz * uz;
        if (r > 1.e-5) {
*/
          k         = two * bp -> k * hol;
          a1 -> fx -= ux * k;
          a1 -> fy -= uy * k;
          a1 -> fz -= uz * k;
/*
	}
*/
      }
      if (bp == bp -> next) return TRUE;
      bp = bp -> next;
    } /* End of while */
  } else {
    while(bp) {
      a1 = bp -> atom1;
      if (a1 -> active) {
	ux = a1 -> x - bp -> x;
	uy = a1 -> y - bp -> y;
	uz = a1 -> z - bp -> z;
/*
	r  = ux * ux + uy * uy + uz * uz;
        if (r > 1.e-5) {
*/
          k         = two * bp -> k * hol;
          a1 -> fx -= ux * k;
          a1 -> fy -= uy * k;
          a1 -> fz -= uz * k;
/*
	}
*/
      }
      if (bp == bp -> next) return TRUE;
      bp = bp -> next;
    } /* End of while */
  }

  return FALSE;
}


/**** Increment the tethering force ****/

int AMMP_FASTCALL f_tether(float lambda)
{
  AMMP_ATOM     *a1;
  float         k, ux, uy, uz;

  TETHER        *bp = tether_first;


  if (!bp) return TRUE;

  if (lambda) {
    while(bp) {
      a1 = bp -> atom1;
      if (a1 -> active){
        ux = a1 -> x - bp -> x + lambda * a1 -> dx;
        uy = a1 -> y - bp -> y + lambda * a1 -> dy;
        uz = a1 -> z - bp -> z + lambda * a1 -> dz;

        if ((ux * ux + uy * uy + uz * uz) > 1.e-5) {
          k         = two * bp -> k;
          a1 -> fx -= ux * k;
          a1 -> fy -= uy * k;
          a1 -> fz -= uz * k;
        }
      }

      if (bp == bp -> next) return TRUE;
      bp = bp -> next;
    } /* End of while */
  } else {
    while(bp) {
      a1 = bp -> atom1;
      if (a1 -> active){
        ux = a1 -> x - bp -> x;
        uy = a1 -> y - bp -> y;
        uz = a1 -> z - bp -> z;

        if ((ux * ux + uy * uy + uz * uz) > 1.e-5) {
          k         = two * bp -> k;
          a1 -> fx -= ux * k;
          a1 -> fy -= uy * k;
          a1 -> fz -= uz * k;
        }
      }

      if (bp == bp -> next) return TRUE;
      bp = bp -> next;
    } /* End of while */
  }

  return FALSE;
}

/**** Multiply two matrices ****/

void AMMP_FASTCALL matmul(float a[], float b[], float c[], int n, int m)
{
  int           i, j, k, ioff, koff;

  k = n * n;
  for(i = 0; i < k; i++) c[i] = zero;

  for(i = 0; i < n; i++) {
    ioff = i * n;
    for(j = 0; j < n; j++) {
      koff = zero;
      for(k = 0; k < m; k++) {
        c[ioff + j] += a[ioff + k] * b[j + koff];
        koff += m;
      } /* End of for (k) */
    } /* End of for (j) */
  } /* End of for (i) */
}


/**** Add a new tether ****/

int AMMP_FASTCALL tether(int p1, float fk, float x, float y, float z)
{
  TETHER *      new;

  AMMP_ATOM *        ap1 = a_m_serial(p1);

  if (!ap1) {
    aaerror("Undefined atom in tether %d", p1);
    return FALSE;
  }

  /**** Check to see if it exists and update if so ****/

  if( tether_first != NULL) {
    new = tether_first;
    while(1) {
      if (new -> next == NULL) break;
      if (new -> atom1 == ap1) {
        new -> k = fk;
        new -> x = x;
        new -> y = y;
        new -> z = z;
        return TRUE;
      }
      if (new -> next == new) break;
      new = new -> next;
    } /* End of while */
  }

  if ((new = (TETHER *)Alloca(sizeof(TETHER), "tether()")) == NULL)
    return FALSE;

  /**** Initialize the pointers ****/

  if (tether_first == NULL) tether_first = new;
  if (tether_last == NULL) tether_last = new;
  new -> atom1 = ap1;
  new -> k = fk;
  new -> x = x;
  new -> y = y;
  new -> z = z;
  new -> next = new;
  tether_last -> next = new;
  tether_last = new;

  return TRUE;
}


/**** Calculate the tethering potential (homotopy version) ****/

int AMMP_FASTCALL v_ho_tether(float *V, float lambda)
{
  AMMP_ATOM     *a1;
  float         xt, yt, zt;
  float         hol;

  TETHER        *bp = tether_first;

  if (!bp) return TRUE;

  hol  = one - GetLambda();
  hol *= hol;

  if (lambda) {
    while(bp) {
      a1 = bp -> atom1;
      if (a1 -> active) {
        xt  = a1 -> x + lambda * a1 -> dx - bp -> x;
        yt  = a1 -> y + lambda * a1 -> dy - bp -> y;
        zt  = a1 -> z + lambda * a1 -> dz - bp -> z;
        *V += bp -> k * (xt * xt + yt * yt + zt * zt) * hol;
      }
      if (bp == bp -> next) return TRUE;
      bp = bp -> next;
    } /* End of while */
  } else {
    while(bp) {
      a1 = bp -> atom1;
      if (a1 -> active) {
        xt = a1 -> x - bp -> x;
        yt = a1 -> y - bp -> y;
        zt = a1 -> z - bp -> z;
        *V += bp -> k * (xt * xt + yt * yt + zt * zt) * hol;
      }
      if (bp == bp -> next) return TRUE;
      bp = bp -> next;
    } /* End of while */
  }
  return FALSE;
}


/**** Calculate the tethering potential ****/

int AMMP_FASTCALL v_tether(float *V, float lambda)
{
  AMMP_ATOM     *a1;
  float         xt, yt, zt;

  TETHER        *bp = tether_first;

  if (!bp) return TRUE;

  if (lambda) {
    while(bp) {
      a1 = bp -> atom1;
      if (a1 -> active) {
        xt  = a1 -> x - bp -> x + lambda * a1 -> dx;
        yt  = a1 -> y - bp -> y + lambda * a1 -> dy;
        zt  = a1 -> z - bp -> z + lambda * a1 -> dz;
        *V += bp -> k * (xt * xt + yt * yt + zt * zt);
      }
      if (bp == bp -> next) return TRUE;
      bp = bp -> next;
    } /* End of while */
  } else {
    while(bp) {
      a1 = bp -> atom1;
      if (a1 -> active) {
        xt  = a1 -> x - bp -> x;
        yt  = a1 -> y - bp -> y;
        zt  = a1 -> z - bp -> z;
        *V += bp -> k * (xt * xt + yt * yt + zt * zt);
      }
      if (bp == bp -> next) return TRUE;
      bp = bp -> next;
    } /* End of while */
  }

  return FALSE;
}


/* routine to superimpose two sets of coordinates
*
* cadged from Larry Andrews' Newvector.f library
*
C  The algorithm is from Ferro and Hermans, Acta Cryst., A33,345-347
C  (1977).  Other algorithms give the same result, such as:
C
C  Kearsley, Acta Cryst., A45, 208-210 (1989)
C  Diamond,  Acta Cryst., A44, 211-216 (1988)
C  Kabsch,   Acta Cryst., A34, 827-828 (1978)
C  Kabsch,   Acta Cryst., A32, 922-923 (1976)
C
C  The algorithm of Ferro and Hermans has the advantage that the
C  rotational components are removed iteratively, so that residual
C  numerical errors are correctly removed.  This leaves no
C  possibility of poorly orthogonalized matrices.  The explicit
C  iteration is not a problem.  The other methods simply hide the
C  iteration in the eigenvector solver.  Ferro and Hermans
C  simply build what is a simple, special purpose version of the QR
C  method.
*/

/*void matmul(float[],float[],float[],int,int);
*/

/* return the RMS error, -1 on internal error */

float AMMP_FASTCALL bstrot(float x[], float y[], float z[], float xx[], float yy[], float zz[], int na, float matrix[3][3], float delta[3])
{
  float         tensor[3][3], cx, cy, cz, cxx, cyy, czz;
  float         tx, ty, tz, txx, tyy, tzz;
  float         rms;
  float         sx[3][3], sy[3][3], sz[3][3];
  float         sq[3][3];
  int           i, j, ipass;


  if (na < 1) return - one;

  /**** Find the centers of mass ****/

  cx  = zero;
  cy  = zero;
  cz  = zero;
  cxx = zero;
  cyy = zero;
  czz = zero;
  for(i = 0; i < na; i++) {
    cx  += x[i];
    cy  += y[i];
    cz  += z[i];
    cxx += xx[i];
    cyy += yy[i];
    czz += zz[i];
  } /* End of for (i) */

  rms  = one / (float)na;
  cx  *= rms;
  cy  *= rms;
  cz  *= rms;
  cxx *= rms;
  cyy *= rms;
  czz *= rms;

  /**** Make the metric tensor ****/

  for(i = 0; i < 3; i++) {
    for(j = 0; j < 3; j++) {
      tensor[i][j] = zero;
      matrix[i][j] = zero;
      sx[i][j]     = zero;
      sy[i][j]     = zero;
      sz[i][j]     = zero;
    } /* End of foe (j) */
  } /* End of for (i) */

  matrix[0][0] = one;
  matrix[1][1] = one;
  matrix[2][2] = one;
  sx[0][0]     = one;
  sy[1][1]     = one;
  sz[2][2]     = one;

  for(i = 0; i < na; i++) {
    tx            = x[i] - cx;
    ty            = y[i] - cy;
    tz            = z[i] - cz;
    txx           = xx[i] - cxx;
    tyy           = yy[i] - cyy;
    tzz           = zz[i] - czz;
    tensor[0][0] += tx * txx;
    tensor[0][1] += tx * tyy;
    tensor[0][2] += tx * tzz;
    tensor[1][0] += ty * txx;
    tensor[1][1] += ty * tyy;
    tensor[1][2] += ty * tzz;
    tensor[2][0] += tz * txx;
    tensor[2][1] += tz * tyy;
    tensor[2][2] += tz * tzz;
  } /* End of for (i) */

  /**** Find the linear orthogonal transformation which symetrizes the metric tensor ****/

  for(ipass = 0; ipass < 20; ipass ++) {
    rms = zero;

    /**** X ****/

    tx        = atan2(tensor[2][1] - tensor[1][2], tensor[1][1] + tensor[2][2]);
    rms      += fabs(tx);
    ty        = cos(tx);
    tz        = sin(tx);
    sx[1][1]  = ty;
    sx[2][1]  = - tz;
    sx[1][2]  = tz;
    sx[2][2]  = ty;
    matmul((float *)sx, (float *)tensor, (float *)sq, 3, 3);
    cpyvec((float *)sq, (float *)tensor, 9);
    matmul((float *)sx, (float *)matrix, (float *)sq, 3, 3);
    cpyvec((float *)sq, (float *)matrix, 9);

    /**** Y ****/

    tx        = atan2(tensor[2][0] - tensor[0][2], tensor[0][0] + tensor[2][2]);
    rms      += fabs(tx);
    ty        = cos(tx);
    tz        = sin(tx);
    sy[0][0]  = ty;
    sy[2][0]  = - tz;
    sy[0][2]  = tz;
    sy[2][2]  = ty;
    matmul((float *)sy, (float *)tensor, (float *)sq, 3, 3);
    cpyvec((float *)sq, (float *)tensor, 9);
    matmul((float *)sy, (float *)matrix, (float *)sq, 3, 3);
    cpyvec((float *)sq, (float *)matrix, 9);

    /**** Z ****/

    tx        = atan2(tensor[0][1] - tensor[1][0], tensor[1][1] + tensor[0][0]);
    rms      += fabs(tx);
    ty        = cos(tx);
    tz        = sin(tx);
    sz[1][1]  = ty;
    sz[0][1]  = -tz;
    sz[1][0]  = tz;
    sz[0][0]  = ty;
    matmul((float *)sz, (float *)tensor, (float *)sq, 3, 3);
    cpyvec((float *)sq, (float *)tensor, 9);
    matmul((float *)sz, (float *)matrix, (float *)sq, 3, 3);
    cpyvec((float *)sq, (float *)matrix, 9);

    /**** Termination critereon here ****/

    if (rms < 1.e-7) break;
  } /* End of for (ipass) */

  rms = zero;
  for(i = 0; i < na; i++) {
    txx  = xx[i] - cxx;
    tyy  = yy[i] - cyy;
    tzz  = zz[i] - czz;
    tx   = matrix[0][0] * txx + matrix[1][0] * tyy + matrix[2][0] * tzz;
    ty   = matrix[0][1] * txx + matrix[1][1] * tyy + matrix[2][1] * tzz;
    tz   = matrix[0][2] * txx + matrix[1][2] * tyy + matrix[2][2] * tzz;
    tx  += cx - x[i];
    ty  += cy - y[i];
    tz  += cz - z[i];
    rms += tx * tx + ty * ty + tz * tz;
  } /* End of for (i) */

  tx       = matrix[0][0] * cxx + matrix[1][0] * cyy + matrix[2][0] * czz;
  ty       = matrix[0][1] * cxx + matrix[1][1] * cyy + matrix[2][1] * czz;
  tz       = matrix[0][2] * cxx + matrix[1][2] * cyy + matrix[2][2] * czz;
  delta[0] = cx - tx;
  delta[1] = cy - ty;
  delta[2] = cz - tz;

  return sqrt(rms / na);
}




