
/*************************************************
****   AMMP - Conformational search routines  ****
**** Copyright 2006-2012, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Constants ****/

#define  AMMP_TSRC_CLONEID1             -1
#define  AMMP_TSRC_CLONEID2             -2

/**** Local variables ****/

#ifdef VEGAZZ
static int              nupdat;
static int              supdat;
#endif

static CLONE *          Clone1;
static CLONE *          Clone2;

static AMMP_ATOM **     newer     = NULL;
static AMMP_ATOM **     newest    = NULL;
static int              TgBufSize = 0;

/**** Local prototypes ****/

static int              AMMP_FASTCALL TgroupAllocBuf(void);
static void             AMMP_FASTCALL TgroupCheckActive(void);
static int              AMMP_FASTCALL TgroupCount(void);
static void             AMMP_FASTCALL TgroupGetVal(float *Val, TGROUP **Group, int Ngroup);
static void             AMMP_FASTCALL TgroupInit(TGROUP **gl, int *ng ,int deep);
static void             AMMP_FASTCALL TgroupRemove(TGROUP *tgp);
float                   AMMP_FASTCALL TgroupRmsd2(float *TorSet1, float *TorSet2, int Ngroup);
static void             AMMP_FASTCALL TgroupRotate(TGROUP *tgp, float off);
static void             AMMP_FASTCALL TgroupRotateOff(TGROUP *tgp, float phi);
static void             AMMP_FASTCALL TgroupRotateStep(TGROUP *tgp, int num);
static void                           TgroupSearch(int *Iter, float *vb, int nstep, TGROUP **GroupList, int igroup, int ngroup, AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int MinSteps, float MinToler);
static int              AMMP_FASTCALL TgroupUpdate(TGROUP *tgp);
static void             AMMP_FASTCALL TsearchFree(TGROUP **GroupList, float V);
static TGROUP **        AMMP_FASTCALL TsearchInit(int *ngroup, int *nstep, int Echo);


/**** Define a new torsion group ****/

int AMMP_FASTCALL Tgroup(int which, int context, int b1, int b2, int b3, float base,
                         int ntry, float TorWin)
{
  TGROUP *              tgp;

  if (!which) {
    aaerror("The tgroup 0 is unaccessible");
    return FALSE;
  }

  if ((base < zero) || (base > 360.0f)) {
    aaerror("Invalid base torsion value");
    return FALSE;
  }

  if ((TorWin < zero) || (TorWin > 360.0f)) {
    aaerror("Invalid torsion window");
    return FALSE;
  }

  /**** Find the torsion group ****/

  for(tgp = tg_first; (tgp) && (tgp -> which != which); tgp = tgp -> next);

  if (tgp) SafeFree(tgp -> group);
  else if ((tgp = (TGROUP *)ListAdd((void **)&tg_first, (void **)&tg_last, sizeof(TGROUP))) == NULL)
    return FALSE;

  /**** Remove the torsion group ****/

  if ((context < 0) || (b1 < 0) || (b2 < 0) || (b3 < 0)) {
    TgroupRemove(tgp);
    return TRUE;
  }

  tgp -> B1Id      = b1;
  tgp -> B2Id      = b2;
  tgp -> B3Id      = b3;
  tgp -> base      = AMMP_DEG_TO_RAD(base);
  tgp -> ContextId = context;
  tgp -> group     = NULL;
  tgp -> Flags     = AMMP_TGPF_ACTIVE;
  tgp -> ingroup   = 0;

  /**** Default values ****/

  if (!ntry  ) ntry   = 6;
  if (!TorWin) TorWin = 360.0f;

  tgp -> ntry      = ntry;
  tgp -> TorWin    = AMMP_DEG_TO_RAD(TorWin);
  tgp -> which     = which;

  return TRUE;
}


/**** Allocate the buffers ****/

static int AMMP_FASTCALL TgroupAllocBuf(void)
{
  int           Size;

  const char *  Routine = "TgroupAllocBuf()";
  int           MaxAtm  = a_number();

  if (MaxAtm < TgBufSize) return TRUE;

  Size = MaxAtm * sizeof(AMMP_ATOM *);
  SafeFree(newer);
  SafeFree(newest);
  if (((newer  = (AMMP_ATOM **)Alloca(Size, Routine)) == NULL) ||
      ((newest = (AMMP_ATOM **)Alloca(Size, Routine)) == NULL)) {
    SafeFree(newer);
    SafeFree(newest);
    newer  = NULL;
    newest = NULL;
    return FALSE;
  }
  TgBufSize = MaxAtm;

  return TRUE;
}


/**** Check if all torsions are activable ****/

static void AMMP_FASTCALL TgroupCheckActive(void)
{
  int           j;

  TGROUP *      tgp = tg_first;

  while(tgp) {
    if (tgp -> Flags & AMMP_TGPF_ACTIVE) {
      for(j = 0; j < tgp -> ingroup; j++) {
        if (!tgp -> group[j] -> active) {
          tgp -> Flags &= (~AMMP_TGPF_ACTIVE);
          break;
        }
      } /* End of for (j) */
    }
    tgp = tgp -> next;
  } /* End of while */
}


/**** Count the torsion groups ****/

static int AMMP_FASTCALL TgroupCount(void)
{
  int           Num = 0;
  TGROUP *      Tgp = tg_first;

  while(Tgp) {
    if (Tgp -> Flags & AMMP_TGPF_ACTIVE) ++Num;
    Tgp = Tgp -> next;
  } /* End of while */

  return Num;
}


/**** Dump tgroup ****/

void AMMP_FASTCALL TgroupDump(FILE *where)
{
  TGROUP *      tgp = tg_first;

  while(tgp) {
    fprintf(where, "tgroup %d %d %d %d %d %f %d;\n",
            tgp -> which,
            tgp -> context -> serial,
            tgp -> b1 -> serial,
            tgp -> b2 -> serial,
            tgp -> b3 -> serial,
            AMMP_RAD_TO_DEG(tgp -> base),
            tgp -> ntry);
    tgp = tgp -> next;
  } /* End of while */
}


/**** Get the torsion group values ****/

static void AMMP_FASTCALL TgroupGetVal(float *Val, TGROUP **Group, int Ngroup)
{
  int           i;

  for(i = 0; i < Ngroup; i++) {
    *Val++ = normalize_torsion(get_torsion_value((*Group) -> context,
                                                 (*Group) -> b1, (*Group) -> b2,
                                                 (*Group) -> b3));
    ++Group;
  } /* End of for (i) */
}


/**** Initialize the torsion groups ****/

static void AMMP_FASTCALL TgroupInit(TGROUP **gl, int *ng, int deep)
{
  AMMP_ATOM     *a1, *a2, *a3, *a4;
  float         x1, y1, z1, x2, y2, z2, x3, y3, z3;
  float         cx1, cy1, cz1, cx2, cy2, cz2;
  float         dp, r;
  int           i, j, ngl;

  if (!deep) {
    j = a_number();
    for(i = 0; i < j; i++) {
      a1       = a_next(i);
      a1 -> dx = a1 -> x;
      a1 -> dy = a1 -> y;
      a1 -> dz = a1 -> z;
    } /* End of for (i) */
  }

  ngl = *ng;
  for(i = ngl - 1; i > -1 ; i --) {
    if (!gl[i]) {
      if (!i) {
        *ng = 0;
        return;
      }
      *ng = i;
      TgroupInit(gl, ng, deep + 1);
      return;
    }

    for(j = 0; j < gl[i] -> ingroup; j++) {
      a1 = gl[i] -> group[j];
      if (!a1) return;
      a1 -> dx = a1 -> x;
      a1 -> dy = a1 -> y;
      a1 -> dz = a1 -> z;
    } /* End of for (j) */

    a1 = gl[i] -> context;
    a2 = gl[i] -> b1;
    a3 = gl[i] -> b2;
    a4 = gl[i] -> b3;

    a1 -> dx = a1 -> x;
    a1 -> dy = a1 -> y;
    a1 -> dz = a1 -> z;
    a2 -> dx = a2 -> x;
    a2 -> dy = a2 -> y;
    a2 -> dz = a2 -> z;
    a3 -> dx = a3 -> x;
    a3 -> dy = a3 -> y;
    a3 -> dz = a3 -> z;
    a4 -> dx = a4 -> x;
    a4 -> dy = a4 -> y;
    a4 -> dz = a4 -> z;

    x1 = a1 -> x - a2 -> x;
    y1 = a1 -> y - a2 -> y;
    z1 = a1 -> z - a2 -> z;
    x2 = a3 -> x - a2 -> x;
    y2 = a3 -> y - a2 -> y;
    z2 = a3 -> z - a2 -> z;
    x3 = a4 -> x - a3 -> x;
    y3 = a4 -> y - a3 -> y;
    z3 = a4 -> z - a3 -> z;

    /**** 1 cross 2 ****/

     cx1 =  y1 * z2 - y2 * z1;
     cy1 = -x1 * z2 + x2 * z1;
     cz1 =  x1 * y2 - x2 * y1;
     r = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;
     if( r < 1.e-4) goto SKIP;
     r   = sqrt(r);
     cx1 = cx1 / r;
     cy1 = cy1 / r;
     cz1 = cz1 / r;

     /**** 3 cross 2 ****/

     cx2 =  y3 * z2 - y2 * z3;
     cy2 = -x3 * z2 + x2 * z3;
     cz2 =  x3 * y2 - x2 * y3;
     r   = cx2 * cx2 + cy2 * cy2 + cz2 * cz2;
     if( r < 1.e-4) goto SKIP;
     r   = sqrt(r);
     cx2 = cx2 / r;
     cy2 = cy2 / r;
     cz2 = cz2 / r;

     /**** If here everything is well determined ****/

     dp = cx1 * cx2 + cy1 * cy2 + cz1 * cz2; /* cos( abs(theta)) */
     if (dp >   one) dp = one;
     if (dp < - one) dp = -one;
     dp = acos(dp);

     /**** Determine the sign by triple product ****/

     r = cx1 * x3 + cy1 * y3 + cz1 * z3;
     if (r > zero) dp = -dp;
     r =  gl[i] -> base - dp;
     TgroupRotate(gl[i], r);
     for(j = 0; j < gl[i] -> ingroup; j++) {
       a1      = gl[i] -> group[j];
       a1 -> x = a1 -> dx;
       a1 -> y = a1 -> dy;
       a1 -> z = a1 -> dz;
     } /* End of for (j) */
SKIP:;
  } /* End of for (i) */
}


/**** Remove a torsion group ****/

static void AMMP_FASTCALL TgroupRemove(TGROUP *Tgp)
{
  SafeFree(Tgp -> group);
  ListRemove((void **)&tg_first, (void **)&tg_last, Tgp);
}


/**** Reset the torsion groups ****/

void AMMP_FASTCALL TgroupReset(void)
{
  TGROUP *      tgp = tg_first;

  while(tgp) {
    SafeFree(tgp -> group);
    tgp = tgp -> next;
  } /* End of while */

  ListFree((void **)&tg_first, (void **)&tg_last);
  SafeFree(newer);
  SafeFree(newest);
  newer     = NULL;
  newest    = NULL;
  TgBufSize = 0;
}


/**** Compute the square RMS difference ****/

float AMMP_FASTCALL TgroupRmsd2(float *TorSet1, float *TorSet2, int Ngroup)
{
  float         T;
  int           i;
  float         Rmsd = zero;

  for(i = 0; i < Ngroup; i++) {
    T = TorSet1[i] - TorSet2[i];
    Rmsd += T * T;
  } /* End of for (i) */

  return (Rmsd / (float)Ngroup);
}


/**** Rotate a torsion group (arbitrary offset) ****/

static void AMMP_FASTCALL TgroupRotate(TGROUP *tgp, float off)
{
  float         phi, cphi, sphi;
  float         ry, rz, nnrx, nnry, nnrz, rnx, rny, rnz;
  int           i;

  AMMP_ATOM *   b1 = tgp -> b1;
  AMMP_ATOM *   b2 = tgp -> b2;
  float         nx = b2->dx - b1->dx;
  float         ny = b2->dy - b1->dy;
  float         nz = b2->dz - b1->dz;
  float         rx = sqrt(nx*nx + ny*ny + nz*nz);

  if (rx < 1.e-6) {
    aaerror("Bad torsion radius in tg_d_apply()");
    return;
  }

  nx /= rx;
  ny /= rx;
  nz /= rx;

  phi  = off;
  cphi = cos(phi);
  sphi = sin(phi);

  for(i = 0; i < tgp -> ingroup; i++) {
    rx   = tgp -> group[i] -> dx - b1 -> dx;
    ry   = tgp -> group[i] -> dy - b1 -> dy;
    rz   = tgp -> group[i] -> dz - b1 -> dz;
    phi  = nx * rx + ny * ry + nz * rz;
    nnrx = phi * nx;
    nnry = phi * ny;
    nnrz = phi * nz;
    rnx  =  ny * rz - nz * ry;
    rny  = -nx * rz + nz * rx;
    rnz  =  nx * ry - ny * rx;
    phi  = one - cphi;
    rx   = cphi * rx + phi * nnrx + sphi * rnx;
    ry   = cphi * ry + phi * nnry + sphi * rny;
    rz   = cphi * rz + phi * nnrz + sphi * rnz;
    tgp -> group[i] -> dx = rx + b1 -> dx;
    tgp -> group[i] -> dy = ry + b1 -> dy;
    tgp -> group[i] -> dz = rz + b1 -> dz;
  } /* End of for (i) */
}


/**** Rotate a torsion group ****/

static void AMMP_FASTCALL TgroupRotateOff(TGROUP *tgp, float phi)
{
  float         cphi, sphi;
  float         ry, rz, nnrx, nnry, nnrz, rnx, rny, rnz;
  int           i;

  AMMP_ATOM *   b1 = tgp -> b1;
  AMMP_ATOM *   b2 = tgp -> b2;
  float         nx = b2 -> x - b1 -> x;
  float         ny = b2 -> y - b1 -> y;
  float         nz = b2 -> z - b1 -> z;
  float         rx = sqrt(nx * nx + ny * ny + nz * nz);

  if (rx < 1.e-6) {
    aaerror("Bad torsion radius in TgroupRotateOff()");
    return;
  }
  nx  /= rx;
  ny  /= rx;
  nz  /= rx;
  cphi = cos(phi);
  sphi = sin(phi);
  for(i = 0; i < tgp -> ingroup; i++) {
    rx   = tgp -> group[i] -> x - b1 -> x;
    ry   = tgp -> group[i] -> y - b1 -> y;
    rz   = tgp -> group[i] -> z - b1 -> z;
    phi  = nx * rx + ny * ry + nz * rz;
    nnrx = phi * nx;
    nnry = phi * ny;
    nnrz = phi * nz;
    rnx  =  ny * rz - nz * ry;
    rny  = -nx * rz + nz * rx;
    rnz  =  nx * ry - ny * rx;
    phi  = one - cphi;
    rx   = cphi * rx + phi * nnrx + sphi * rnx;
    ry   = cphi * ry + phi * nnry + sphi * rny;
    rz   = cphi * rz + phi * nnrz + sphi * rnz;
    tgp -> group[i] -> x = rx + b1 -> x;
    tgp -> group[i] -> y = ry + b1 -> y;
    tgp -> group[i] -> z = rz + b1 -> z;
  } /* End of for (i) */
}


/**** Rotate a torsion group ****/

static void AMMP_FASTCALL TgroupRotateStep(TGROUP *tgp, int num)
{
  float         phi, cphi, sphi;
  float         ry, rz, nnrx, nnry, nnrz, rnx, rny, rnz;
  int           i;

  AMMP_ATOM *   b1 = tgp -> b1;
  AMMP_ATOM *   b2 = tgp -> b2;
  float         nx = b2 -> x - b1 -> x;
  float         ny = b2 -> y - b1 -> y;
  float         nz = b2 -> z - b1 -> z;
  float         rx = sqrt(nx * nx + ny * ny + nz * nz);

  if (rx < 1.e-6) {
    aaerror("Bad torsion radius in TgroupRotateStep()");
    return;
  }
  nx  /= rx;
  ny  /= rx;
  nz  /= rx;
  phi  = TWOPI / (float)tgp -> ntry * (float)num;
  cphi = cos(phi);
  sphi = sin(phi);
  for(i = 0; i < tgp -> ingroup; i++) {
    rx   = tgp -> group[i] -> x - b1 -> x;
    ry   = tgp -> group[i] -> y - b1 -> y;
    rz   = tgp -> group[i] -> z - b1 -> z;
    phi  = nx * rx + ny * ry + nz * rz;
    nnrx = phi * nx;
    nnry = phi * ny;
    nnrz = phi * nz;
    rnx  =  ny * rz - nz * ry;
    rny  = -nx * rz + nz * rx;
    rnz  =  nx * ry - ny * rx;
    phi  = one - cphi;
    rx   = cphi * rx + phi * nnrx + sphi * rnx;
    ry   = cphi * ry + phi * nnry + sphi * rny;
    rz   = cphi * rz + phi * nnrz + sphi * rnz;
    tgp -> group[i] -> x = rx + b1 -> x;
    tgp -> group[i] -> y = ry + b1 -> y;
    tgp -> group[i] -> z = rz + b1 -> z;
  } /* End of for (i) */
}


/**** Perform the recursive conformational search ****/

static void TgroupSearch(int *Iter, float *vb, int nstep, TGROUP **GroupList,
                         int igroup, int ngroup, AMMP_VFUNC vfs[], AMMP_FFUNC ffs[],
                         int nfs, int MinSteps, float MinToler)
{
  float         vl;
  int           i, ifs;

  for(i = 0; i < GroupList[igroup] -> ntry; i++) {
    if (igroup < (ngroup - 1)) {
      TgroupSearch(Iter, vb, nstep, GroupList, igroup + 1, ngroup, vfs, ffs, nfs,
                   MinSteps, MinToler);
      TgroupRotateStep(GroupList[igroup], 1);
      continue;
    }

    /**** Evaluate the energy ****/

    vl = 0.0f;
    for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vl, zero);
    if (!*Iter) *vb = vl;

    if (MinSteps) {
      if ((supdat) && (!(*Iter % supdat) || (*Iter == nstep)))
        printf("  Tsearch%5d: v %.3f, vb %.3f", *Iter + 1, vl, *vb);
      Clone2 = Clone(Clone2, AMMP_TSRC_CLONEID2);
      cngdel(vfs, ffs, nfs, MinSteps, MinSteps, MinToler, FALSE, 0);
      vl = 0.0f;
      for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vl, zero);
      if ((supdat) && (!(*Iter % supdat) || (*Iter == nstep)))
        printf(", vmin %.3f\n", vl);
    } else if ((supdat) && (!(*Iter % supdat) || (*Iter == nstep)))
      printf("  Tsearch%5d: v %.3f, vb %.3f\n", *Iter + 1, vl, *vb);

#ifdef VEGAZZ
    if ((nupdat) && (!(*Iter % nupdat))) DumpAtoms(vl);
#endif

    /**** Store the best conformation ****/

    if (vl < *vb) {
      *vb = vl;
      Clone1 = Clone(Clone1, AMMP_TSRC_CLONEID1);
    }

    if (MinSteps) CloneRestore(Clone2, AMMP_TSRC_CLONEID2);

//    if (GroupList[igroup] -> ntry > 1)
      TgroupRotateStep(GroupList[igroup], 1);
    ++*Iter;
  } /* End of for (i) */
}


/**** Update a torsion group ****/

static int AMMP_FASTCALL TgroupUpdate(TGROUP *tgp)
{
  AMMP_ATOM *   bonded[20];
  int           in_newest, in_newer;
  int           i, j, k, l, ll;

  const char *  ErrMsg1 = "Tgroup %d - atom %d not defined";
  const char *  ErrMsg2 = "Too many atoms in the tgroup %d";
  const char *  Routine = "TgroupUpdate()";

  if (!TgroupAllocBuf()) return FALSE;

  if ((tgp -> context = a_m_serial(tgp -> ContextId)) == NULL) {
    aaerror(ErrMsg1, tgp -> which, tgp -> ContextId);
    return FALSE;
  }

  if ((tgp -> b1 = a_m_serial(tgp -> B1Id)) == NULL) {
    aaerror(ErrMsg1, tgp -> which, tgp -> B1Id);
    return FALSE;
  }

  if ((tgp -> b2 = a_m_serial(tgp -> B2Id)) == NULL) {
    aaerror(ErrMsg1, tgp -> which, tgp -> B2Id);
    return FALSE;
  }

  if ((tgp -> b3 = a_m_serial(tgp ->  B3Id)) == NULL) {
    aaerror(ErrMsg1, tgp -> which, tgp -> B3Id);
    return FALSE;
  }

  /**** Allocate the buffers ****/

  if ((tgp -> group = (AMMP_ATOM **)Alloca(TgBufSize * sizeof(AMMP_ATOM *), Routine)) == NULL)
    return FALSE;

  for(k = 0; k < TgBufSize; k++) tgp -> group[k] = NULL;

  get_bond(tgp -> b2, bonded, 20, &j);
  for(k = 0; k < j; k++) {
    if ((bonded[k] != tgp -> context) && (bonded[k] != tgp -> b3) &&
        (bonded[k] != tgp -> b1     )) break;
  } /* End of for (k) */

  newest[0]       = tgp -> b3;
  tgp -> group[0] = tgp -> b3;
  tgp -> ingroup  = 1;
  in_newest       = 1;

  if (k != j) {
    newest[1]       = bonded[k];
    tgp -> group[1] = bonded[k];
    tgp -> ingroup  = 2;
    in_newest       = 2;
    for(i = 0; i < j; i++) {
      l = 1;
      for(k =0 ; k < in_newest; k++) {
        if (newest[k] == bonded[i]) {
          l = 0;
          break;
        }
      } /* End of for (k) */

      if ((bonded[i] == tgp -> context) ||
          (bonded[i] == tgp -> b1     ) ||
          (bonded[i] == tgp -> b2     ) ||
          (bonded[i] == tgp -> b3     )) l = 0;
      if (l == 1) {
        tgp -> group[tgp -> ingroup++] = bonded[i];
        newest[in_newest++] = bonded[i];
      }
    } /* End of for (i) */
  }

  while(in_newest > 0) {
    in_newer = 0;
    for(l = 0; l < in_newest; l++) {
      get_bond(newest[l], bonded, 20, &j);
      for(i = 0; i < j; i++) {
        ll = 1;
  for(k = 0; k < tgp -> ingroup ; k++) {
          if (tgp -> group[k] == bonded[i]) {
            ll = 0;
            break;
          }
        } /* End of for (k) */

        if ((bonded[i] == tgp -> context) ||
            (bonded[i] == tgp -> b1     ) ||
            (bonded[i] == tgp -> b2     ) ||
            (bonded[i] == tgp -> b3     )) ll = 0;
        if (ll == 1) {
          tgp -> group[tgp -> ingroup++] = bonded[i];
          newer[in_newer++] = bonded[i];
          if (tgp -> ingroup > TgBufSize) {
            aaerror(ErrMsg2, tgp -> which);
            return FALSE;
          }
        }
      } /* End of for (i) */
    } /* End of for (l) */

    for(i = 0; i < in_newer; i++) newest[i] = newer[i];
    in_newest = in_newer;
  } /* End of while */

  if (tgp -> ingroup > TgBufSize) {
    aaerror(ErrMsg2, tgp -> which);
    free(tgp -> group);
    return FALSE;
  }

  if ((tgp -> group = ReAlloca(tgp -> group, tgp -> ingroup * sizeof(AMMP_ATOM), Routine)) == NULL) {
    tgp -> group = NULL;
    return FALSE;
  }

/*
  for(i = 0; i < tgp -> ingroup; i++)
    printf(" atom %d in group %d\n", tgp -> group[i] -> serial, tgp -> which);
*/

  return TRUE;
}


/**** Update all torsion groups ****/

static int AMMP_FASTCALL TgroupUpdateAll(void)
{
  TGROUP *      tgp = tg_first;

  while(tgp) {
    if ((tgp -> Flags & AMMP_TGPF_ACTIVE) && (!TgroupUpdate(tgp))) return FALSE;
    tgp = tgp -> next;
  } /* End of while */

  return TRUE;
}


/**** Boltzmann jump ****/

#if 0
int AMMP_FASTCALL Tjump(AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int TotConf,
                        int MinSteps, float MinToler, float Temp, int Echo)
{
  int           i, ifs, Iter, Ngroup, Nstep, PrintEne;
  float         vl, vb, vtb;
  TGROUP **     Group;
  TGROUP **     GroupList;

  if (TotConf <= 0) {
    aaerror("Invalid number of conformations");
    return FALSE;
  }

  if (Temp <= zero) {
    aaerror("Invalid temperature");
    return FALSE;
  }

  if ((GroupList = TsearchInit(&Ngroup, &Nstep, Echo)) == NULL)
    return FALSE;

  /**** Check the minimization parameters ****/

  if (MinSteps < 0   ) MinSteps = 0;
  if (MinToler < zero) MinToler = zero;

  printf("  Starting Boltzmann jump search (%d steps, %.1f K)\n\n", TotConf, Temp);

  Clone2 = Clone(Clone2, AMMP_TSRC_CLONEID2);
  Clone1 = Clone(Clone1, AMMP_TSRC_CLONEID1);
  vb = zero;
  for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vb, zero);
  vtb = vb;

  /**** Boltzmann jump ****/

  for(Iter = 0; Iter < TotConf; ++Iter) {
    if ((supdat) && (!(Iter % supdat) || (Iter == TotConf))) PrintEne = TRUE;
    else PrintEne = FALSE;

    /**** Generate the new conformation ****/

    Group = GroupList;
    for(i = 0; i < Ngroup; i++) {
      TgroupRotateOff(*Group, randf() * (*Group) -> TorWin);
      ++Group;
    } /* End of for (i) */

    /**** Evaluate the energy ****/

    vl = zero;
    for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vl, zero);

    if (PrintEne)
      printf("  Tjump  %5d: v %.3f, vtb %.3f, vb %.3f",
             Iter + 1, vl, vtb, vb);

    if (vl < vtb) {
      if (MinSteps) {
        cngdel(vfs, ffs, nfs, MinSteps, MinSteps, MinToler, FALSE, 0);
        vl = 0.0f;
        for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vl, zero);
      }

      if (PrintEne) {
        if (MinSteps) printf(", vmin %.3f\n", vl);
        else printf("\n");
      }
      Clone(Clone2, AMMP_TSRC_CLONEID2);
      vtb = vl;
    } else {
      if ((exp((vtb - vl) / (Temp * 1.987f))) < randf()) {
        CloneRestore(Clone2, AMMP_TSRC_CLONEID2);
      } else {
        Clone(Clone2, AMMP_TSRC_CLONEID2);
        vtb = vl;
      }
      if (PrintEne) printf("\n");
    }

    if (vtb < vb) {
      Clone(Clone1, AMMP_TSRC_CLONEID1);
      vb = vtb;
    }

#ifdef VEGAZZ
    if ((nupdat) && (!(Iter % nupdat))) DumpAtoms(vl);
#endif

  } /* End of for (Iter) */

  TsearchFree(GroupList, vb);

  return TRUE;
}
#else
int AMMP_FASTCALL Tjump(AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int TotConf,
                        int MinSteps, float MinToler, float Temp, float RefRmsd,
                        int Echo)
{
  float         *CurTor, *MinTor;
  float         Rmsd, vl, vb, vtb;
  int           i, ifs, Iter, MxPerIter, Ngroup, Nstep, Psteps, PrintEne;
  TGROUP **     Group;
  TGROUP **     GroupList;

  const char *  RoutineStr = "Tjump()";

  if (TotConf <= 0) {
    aaerror("Invalid number of conformations");
    return FALSE;
  }

  if (Temp    <= zero) Temp    = 300.0f;
  if (RefRmsd <= zero) RefRmsd = 60.0f;

  /**** Initialize the conformational search ****/

  if ((GroupList = TsearchInit(&Ngroup, &Nstep, Echo)) == NULL)
    return FALSE;

  /**** Allocate the torsion value arrays ****/

  if ((CurTor = (float *)Alloca(sizeof(float) * Ngroup, RoutineStr)) == NULL) {
    free(GroupList);
    return FALSE;
  }

  if ((MinTor = (float *)Alloca(sizeof(float) * Ngroup, RoutineStr)) == NULL) {
    free(GroupList);
    free(CurTor);
    return FALSE;
  }

  /**** Check the minimization parameters ****/

  if (MinSteps < 0   ) MinSteps = 0;
  if (MinToler < zero) MinToler = zero;

  printf("  Starting Boltzmann jump search (%d steps, %.1f K, %.2f perturbation RMSD)\n\n",
         TotConf, Temp, RefRmsd);

  Clone2     = Clone(Clone2, AMMP_TSRC_CLONEID2);
  Clone1     = Clone(Clone1, AMMP_TSRC_CLONEID1);
  MxPerIter  = GetMxPerIter();
  RefRmsd    = AMMP_DEG_TO_RAD(RefRmsd);
  RefRmsd   *= RefRmsd;
  Rmsd       = RefRmsd;
  vb         = zero;
  for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vb, zero);
  vtb = vb;
  TgroupGetVal(MinTor, GroupList, Ngroup);

  /**** Boltzmann jump ****/

  for(Iter = 0; Iter < TotConf; ++Iter) {
    PrintEne = ((supdat) && (!(Iter % supdat) || (Iter == TotConf)));

    /**** Perturbation phase ****/

    Psteps = 0;
    for(i = 0; i < Ngroup; i++) CurTor[i] = MinTor[i];
    do {
      if (Psteps >= MxPerIter) break;

      /**** Generate the new conformation ****/

      Group = GroupList;
      for(i = 0; i < Ngroup; i++) {
        TgroupRotateOff(*Group, randf() * (*Group) -> TorWin);
        ++Group;
      } /* End of for (i) */

      /**** Evaluate the energy ****/

      vl = zero;
      for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vl, zero);

      /**** Boltzmann decision ****/

      if (vl > vtb) {
        if ((exp((vtb - vl) / (Temp * 1.987f))) < randf()) {
          CloneRestore(Clone2, AMMP_TSRC_CLONEID2);
          ++Psteps;
          continue;
        } else vtb = vl;
      } else vtb = vl;

      Clone(Clone2, AMMP_TSRC_CLONEID2);
      TgroupGetVal(CurTor, GroupList, Ngroup);
      Rmsd = TgroupRmsd2(MinTor, CurTor, Ngroup);
      ++Psteps;
    } while(Rmsd < RefRmsd);

    if (PrintEne)
      printf("  Tjump  %5d: ps %4d, vb %.3f, v %.3f", Iter + 1, Psteps, vb, vl);

    /**** Minimize the conformation ****/

    if (MinSteps) {
      cngdel(vfs, ffs, nfs, MinSteps, MinSteps, MinToler, FALSE, 0);
      TgroupGetVal(MinTor, GroupList, Ngroup);
      Clone(Clone2, AMMP_TSRC_CLONEID2);
      vl = 0.0f;
      for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vl, zero);
      if (PrintEne) printf(", vmin %.3f\n", vl);
      vtb = vl;
    } else {
      Clone(Clone2, AMMP_TSRC_CLONEID2);
      for(i = 0; i < Ngroup; i++) MinTor[i] = CurTor[i];
      if (PrintEne) printf("\n");
    }

    if (vl < vb) {
      Clone(Clone1, AMMP_TSRC_CLONEID1);
      vb = vl;
    }

#ifdef VEGAZZ
    if ((nupdat) && (!(Iter % nupdat))) DumpAtoms(vl);
#endif

  } /* End of for (Iter) */

  TsearchFree(GroupList, vb);
  free(CurTor);
  free(MinTor);

  return TRUE;
}
#endif

/**** Random conformational search ****/

int AMMP_FASTCALL Trandom(AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int TotConf,
                          int MinSteps, float MinToler, int KeepPrev, int KeepMin,
                          int Echo)
{
  int           i, ifs, Iter, Ngroup, Nstep;
  float         vl, vb;
  TGROUP **     Group;
  TGROUP **     GroupList;

  if (TotConf <= 0) {
    aaerror("Invalid number of conformations");
    return FALSE;
  }

  if ((GroupList = TsearchInit(&Ngroup, &Nstep, Echo)) == NULL)
    return FALSE;

  /**** Check the minimization parameters ****/

  if (MinSteps < 0   ) MinSteps = 0;
  if (MinToler < zero) MinToler = zero;

  if (MinSteps) {
    if (KeepMin) KeepPrev = TRUE;
  } else KeepMin = FALSE;

  /**** Set all the torsions to the ideal values ****/

  TgroupInit(GroupList, &Ngroup, 0);

  printf("  Starting random conformational search (%d steps)\n\n", TotConf);

  /**** Random search ****/

  Clone2 = Clone(Clone2, AMMP_TSRC_CLONEID2);
  for(Iter = 0; Iter < TotConf; ++Iter) {

    /**** Generate the new conformation ****/

    Group = GroupList;
    for(i = 0; i < Ngroup; i++) {
      TgroupRotateOff(*Group, randf() * (*Group) -> TorWin);
      ++Group;
    } /* End of for (i) */

    /**** Evaluate the energy ****/

    vl = 0.0f;
    for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vl, zero);
    if (!Iter) vb = vl;

    if (MinSteps) {
      if ((supdat) && (!(Iter % supdat) || (Iter == TotConf)))
        printf("  Trandom%5d: v %.3f, vb %.3f", Iter + 1, vl, vb);

      if (!KeepMin) Clone2 = Clone(Clone2, AMMP_TSRC_CLONEID2);
      cngdel(vfs, ffs, nfs, MinSteps, MinSteps, MinToler, FALSE, 0);
      vl = 0.0f;
      for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vl, zero);

      if ((supdat) && (!(Iter % supdat) || (Iter == TotConf)))
        printf(", vmin %.3f\n", vl);
    } else if ((supdat) && (!(Iter % supdat) || (Iter == TotConf)))
      printf("  Trandom%5d: v %.3f, vb %.3f\n", Iter + 1, vl, vb);

#ifdef VEGAZZ
    if ((nupdat) && (!(Iter % nupdat))) DumpAtoms(vl);
#endif

    /**** Store the best conformation ****/

    if (vl < vb) {
      Clone1 = Clone(Clone1, AMMP_TSRC_CLONEID1);
      vb     = vl;
    }

    if (!KeepPrev) CloneRestore(Clone2, AMMP_TSRC_CLONEID2);
  } /* End of for (Iter) */

  TsearchFree(GroupList, vb);

  return TRUE;
}


/**** Systematic conformational search ****/

int AMMP_FASTCALL Tsearch(int MinSteps, float MinToler, int Echo)
{
  float         V;
  int           i, ngroup, nstep;
  TGROUP **     grouplist;

  if ((grouplist = TsearchInit(&ngroup, &nstep, Echo)) == NULL)
    return FALSE;

  /**** Check the parameters ****/

  if (MinSteps < 0   ) MinSteps = 0;
  if (MinToler < zero) MinToler = zero;

  printf("  Starting systematic conformational search (%d steps)\n\n", nstep);

  /**** Set all the torsions to the ideal values ****/

  TgroupInit(grouplist, &ngroup, 0);

  /**** Do the recursive search */

  i = 0;
  TgroupSearch(&i, &V, nstep - 1, grouplist, 0, ngroup, potentials, forces, nused,
               MinSteps, MinToler);

  TsearchFree(grouplist, V);

  return TRUE;
}


/**** Free the conformational search resources ****/

static void AMMP_FASTCALL TsearchFree(TGROUP **GroupList, float V)
{

  CloneRestore(Clone1, AMMP_TSRC_CLONEID1);

  printf("\n  Best conformation selected (v %f)\n", V);

  if (Clone1) CloneDel(Clone1, AMMP_TSRC_CLONEID1);
  if (Clone2) CloneDel(Clone2, AMMP_TSRC_CLONEID2);
  free(GroupList);
}


/**** Initialize the conformational search ****/

static TGROUP ** AMMP_FASTCALL TsearchInit(int *ngroup, int *nstep, int Echo)
{
  int           i;
  TGROUP **     GroupList;
  TGROUP *      Tgp;

  if (!CheckAtoms()) return NULL;

  if (!tg_first) {
    aaerror("No tgroup defined");
    return NULL;
  }

  /**** Update the torsion groups ****/

  if (!TgroupUpdateAll()) return NULL;

  TgroupCheckActive();

  if ((*ngroup = TgroupCount()) == 0) {
    aaerror("No active or moveable tgroup");
    return NULL;
  }

  /**** Allocate the group list ****/

  if ((GroupList = (TGROUP **)Alloca(*ngroup * sizeof(TGROUP *), "TsearchInit()")) == NULL)
    return NULL;

  /**** Select the active torsions ****/

  i     = 0;
  *nstep = 1;
  for(Tgp = tg_first; Tgp; Tgp = Tgp -> next) {
    if (Tgp -> Flags & AMMP_TGPF_ACTIVE) {
      *nstep *= Tgp -> ntry;
      GroupList[i++] = Tgp;
    }
  } /* End of for (Tgp) */

#ifdef VEGAZZ
  nupdat = GetNupdat();
  if (!Echo) supdat = 0;
  else supdat = GetSupdat();
#endif

  Clone1 = NULL;
  Clone2 = NULL;

  return GroupList;
}

