/* torsion.c
*
* collection of routines to service bond torsion  potentials
*
* POOP (Poor-mans Object Oriented Programming) using scope rules
*
* these routines hold a data base (in terms of array indeces)
* of torsion, with the associated length and force constant
*
* (this could be table driven but what the hell memories cheap)
*
* the routines for potential value, force and (eventually) second
* derivatives are here also
*
* force and 2nd derivative routines assume zero'd arrays for output
* this allows for parralellization if needed (on a PC?)
*
* forces are bond wise symmetric - so we don't have to fuck around with
* s matrices and the like.
*/
/*
*  copyright 1992 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"
#include "mthread.h"

/**** Local prototypes ****/

static int f_torsion_thread(void *HdtGlobVars);
static int v_torsion_thread(void *HdtGlobVars);


/**** Dump the torsions ****/

void AMMP_FASTCALL dump_torsions(FILE *where)
{
  TORSION       *b = torsion_first;

  while(b) {
    fprintf(where, "torsion %d %d %d %d %f %d %f;\n",
            b -> atom1 -> serial, b -> atom2 -> serial,
            b -> atom3 -> serial, b -> atom4 -> serial,
            b -> k, b -> n, AMMP_RAD_TO_DEG(b -> offset));
    if (b -> next == b) break;
    b = b -> next;
  } /* End of while */
}


/**** Increment the torsion force (parallel) ****/

int AMMP_FASTCALL f_torsion_p(float lambda)
{
  MCL_BEGIN
    float               lambda;
  MCL_END

  MCL_ALLOCGLOBVARS;

  if (!torsion_first) return TRUE;

  MCL_GetVar(lambda) = lambda;
  MCL_CreateThreads(0, f_torsion_thread, &HdtGlobVars, MCL_THREADS_WAIT);

  return TRUE;
}


/**** Increment the torsion force ****/

int AMMP_FASTCALL f_torsion(float lambda)
{
  AMMP_ATOM     *a1, *a2, *a3, *a4;
  doublefloat   x1, y1, z1, x2, y2, z2, x3, y3, z3;
  doublefloat   r, cx1, cy1, cz1, cx2, cy2, cz2;
  doublefloat   dp, sdp;
  doublefloat   r1, r2, r3, c1, c2, s1, s2;
  doublefloat   ux, uy, uz;

  TORSION       *bp = torsion_first;

  if (!bp) return TRUE;

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    a3 = bp -> atom3;
    a4 = bp -> atom4;

    if ((a1 -> active) || (a2 -> active) ||
        (a3 -> active) || (a4 -> active)) {
      if (lambda) {
        x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
        x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
        y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
        z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
        x3 = a4 -> x - a3 -> x + lambda * (a4 -> dx - a3 -> dx);
        y3 = a4 -> y - a3 -> y + lambda * (a4 -> dy - a3 -> dy);
        z3 = a4 -> z - a3 -> z + lambda * (a4 -> dz - a3 -> dz);
      } else {
        x1 = a1 -> x - a2 -> x;
        y1 = a1 -> y - a2 -> y;
        z1 = a1 -> z - a2 -> z;
        x2 = a3 -> x - a2 -> x;
        y2 = a3 -> y - a2 -> y;
        z2 = a3 -> z - a2 -> z;
        x3 = a4 -> x - a3 -> x;
        y3 = a4 -> y - a3 -> y;
        z3 = a4 -> z - a3 -> z;
      }

      /**** Lengths for normalization ****/

      r1 = sqrt(x1 * x1 + y1 * y1 + z1 * z1);
      r2 = sqrt(x2 * x2 + y2 * y2 + z2 * z2);
      r3 = sqrt(x3 * x3 + y3 * y3 + z3 * z3);
      c1 =  (x1 * x2 + y1 * y2 + z1 * z2) / r1 / r2;
      c2 = -(x2 * x3 + y2 * y3 + z2 * z3) / r2 / r3;
      s1 = (1.0 - c1 * c1);
      if (s1 > TOO_SMALL) {
        s2 = (1.0 - c2 * c2);
        if (s2 > TOO_SMALL) {

          /**** 1 cross 2 ****/

          cx1 =  y1 * z2 - y2 * z1;
          cy1 = -x1 * z2 + x2 * z1;
          cz1 =  x1 * y2 - x2 * y1;
          r   = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;
	  if (r > TOO_SMALL) {
            r = 1.0 / sqrt(r);
            cx1 = cx1 * r;
            cy1 = cy1 * r;
            cz1 = cz1 * r;

            /**** 3 cross 2 ****/

            cx2 =  y3 * z2 - y2 * z3;
            cy2 = -x3 * z2 + x2 * z3;
            cz2 =  x3 * y2 - x2 * y3;
            r   = cx2 * cx2 + cy2 * cy2 + cz2 * cz2;
	    if (r > TOO_SMALL) {
              r = 1.0 / sqrt(r);
              cx2 = cx2 * r;
              cy2 = cy2 * r;
              cz2 = cz2 * r;

              /**** If here everything is well determined ****/

              dp = cx1 * cx2 + cy1 * cy2 + cz1 * cz2; /* cos( abs(theta)) */
              if (dp > 1.0) dp = 1.0;
              else if (dp < -1.0) dp = -1.0;
              dp = acos(dp);

              /**** Determine the sign by triple product ****/

              r = sqrt(x3 * x3 + y3 * y3 + z3 * z3) * sqrt(x2 * x2 + y2 * y2 + z2 * z2);
              sdp = x3 * x2 + y3 * y2 + z3 * z2;
              sdp = sdp / r;
              ux = x3 - sdp * x2;
              uy = y3 - sdp * y2;
              uz = z3 - sdp * z2;
              r = cx1 * ux + cy1 * uy + cz1 * uz;
              if (r >= 0.0) dp = -dp;
              r  = -0.5 * bp -> k * bp -> n * sin(bp -> n * dp - bp -> offset);
              s1 = 1.0 / s1;
              s2 = 1.0 / s2;

              if (a1 -> active) {
                a1 -> fx -= r * cx1 / r1 * s1;
                a1 -> fy -= r * cy1 / r1 * s1;
                a1 -> fz -= r * cz1 / r1 * s1;
              }

              if (a2 -> active) {
                a2 -> fx += r * cx1 * (r2 - c1 * r1) / r2 / r1 * s1;
                a2 -> fy += r * cy1 * (r2 - c1 * r1) / r2 / r1 * s1;
                a2 -> fz += r * cz1 * (r2 - c1 * r1) / r2 / r1 * s1;
                a2 -> fx -= r * cx2 * c2 / r2 * s2;
                a2 -> fy -= r * cy2 * c2 / r2 * s2;
                a2 -> fz -= r * cz2 * c2 / r2 * s2;
              }

              if (a3 -> active) {
                a3 -> fx -= r * cx2 * (r2 - c2 * r3) / r2 / r3 * s2;
                a3 -> fy -= r * cy2 * (r2 - c2 * r3) / r2 / r3 * s2;
                a3 -> fz -= r * cz2 * (r2 - c2 * r3) / r2 / r3 * s2;
                a3 -> fx += r * cx1 * c1 / r2 * s1;
                a3 -> fy += r * cy1 * c1 / r2 * s1;
                a3 -> fz += r * cz1 * c1 / r2 * s1;
              }

              if (a4 -> active) {
                a4 -> fx += r * cx2 / r3 * s2;
                a4 -> fy += r * cy2 / r3 * s2;
                a4 -> fz += r * cz2 / r3 * s2;
              }
            }
          }
        }
      }
    }
    if (bp == bp -> next) return TRUE;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}


/**** Increment the torsion force (parallel thread) ****/

static int f_torsion_thread(void *HdtGlobVars)
{
  MCL_BEGIN
    float               lambda;
  MCL_END

  AMMP_ATOM             *a1, *a2, *a3, *a4;
  doublefloat           x1, y1, z1, x2, y2, z2, x3, y3, z3;
  doublefloat           r, cx1, cy1, cz1, cx2, cy2, cz2;
  doublefloat           dp, sdp;
  doublefloat           r1, r2, r3, c1, c2, s1, s2;
  doublefloat           ux, uy, uz;

  unsigned int          ThreadID;

  float                 lambda = MCL_GetVarTh(lambda);
  TORSION               *bp    = torsion_first;
  unsigned int          j      = 1;

  MCL_ObtainThID(ThreadID);

  while(bp) {
    if (j == ThreadID) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      a3 = bp -> atom3;
      a4 = bp -> atom4;

      if ((a1 -> active) || (a2 -> active) ||
          (a3 -> active) || (a4 -> active)) {
        if (lambda) {
          x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
          y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
          z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
          x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
          y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
          z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
          x3 = a4 -> x - a3 -> x + lambda * (a4 -> dx - a3 -> dx);
          y3 = a4 -> y - a3 -> y + lambda * (a4 -> dy - a3 -> dy);
          z3 = a4 -> z - a3 -> z + lambda * (a4 -> dz - a3 -> dz);
        } else {
          x1 = a1 -> x - a2 -> x;
          y1 = a1 -> y - a2 -> y;
          z1 = a1 -> z - a2 -> z;
          x2 = a3 -> x - a2 -> x;
          y2 = a3 -> y - a2 -> y;
          z2 = a3 -> z - a2 -> z;
          x3 = a4 -> x - a3 -> x;
          y3 = a4 -> y - a3 -> y;
          z3 = a4 -> z - a3 -> z;
        }

        /**** Lengths for normalization ****/

        r1 = sqrt(x1 * x1 + y1 * y1 + z1 * z1);
        r2 = sqrt(x2 * x2 + y2 * y2 + z2 * z2);
        r3 = sqrt(x3 * x3 + y3 * y3 + z3 * z3);
        c1 =  (x1 * x2 + y1 * y2 + z1 * z2) / r1 / r2;
        c2 = -(x2 * x3 + y2 * y3 + z2 * z3) / r2 / r3;
        s1 = (1.0 - c1 * c1);
        if (s1 > TOO_SMALL) {
          s2 = (1.0 - c2 * c2);
          if (s2 > TOO_SMALL) {

            /**** 1 cross 2 ****/

            cx1 =  y1 * z2 - y2 * z1;
            cy1 = -x1 * z2 + x2 * z1;
            cz1 =  x1 * y2 - x2 * y1;
            r   = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;
	  if (r > TOO_SMALL) {
              r = 1.0 / sqrt(r);
              cx1 = cx1 * r;
              cy1 = cy1 * r;
              cz1 = cz1 * r;

              /**** 3 cross 2 ****/

              cx2 =  y3 * z2 - y2 * z3;
              cy2 = -x3 * z2 + x2 * z3;
              cz2 =  x3 * y2 - x2 * y3;
              r   = cx2 * cx2 + cy2 * cy2 + cz2 * cz2;
	    if (r > TOO_SMALL) {
                r = 1.0 / sqrt(r);
                cx2 = cx2 * r;
                cy2 = cy2 * r;
                cz2 = cz2 * r;

                /**** If here everything is well determined ****/

                dp = cx1 * cx2 + cy1 * cy2 + cz1 * cz2; /* cos( abs(theta)) */
                if (dp > 1.0) dp = 1.0;
                else if (dp < -1.0) dp = -1.0;
                dp = acos(dp);

                /**** Determine the sign by triple product ****/

                r = sqrt(x3 * x3 + y3 * y3 + z3 * z3) * sqrt(x2 * x2 + y2 * y2 + z2 * z2);
                sdp = x3 * x2 + y3 * y2 + z3 * z2;
                sdp = sdp / r;
                ux = x3 - sdp * x2;
                uy = y3 - sdp * y2;
                uz = z3 - sdp * z2;
                r = cx1 * ux + cy1 * uy + cz1 * uz;
                if (r >= 0.0) dp = -dp;
                r  = -0.5 * bp -> k * bp -> n * sin(bp -> n * dp - bp -> offset);
                s1 = 1.0 / s1;
                s2 = 1.0 / s2;

//                MCL_MutexOn();
                if (a1 -> active) {
                  a1 -> fx -= r * cx1 / r1 * s1;
                  a1 -> fy -= r * cy1 / r1 * s1;
                  a1 -> fz -= r * cz1 / r1 * s1;
                }

                if (a2 -> active) {
                  a2 -> fx += r * cx1 * (r2 - c1 * r1) / r2 / r1 * s1;
                  a2 -> fy += r * cy1 * (r2 - c1 * r1) / r2 / r1 * s1;
                  a2 -> fz += r * cz1 * (r2 - c1 * r1) / r2 / r1 * s1;
                  a2 -> fx -= r * cx2 * c2 / r2 * s2;
                  a2 -> fy -= r * cy2 * c2 / r2 * s2;
                  a2 -> fz -= r * cz2 * c2 / r2 * s2;
                }

                if (a3 -> active) {
                  a3 -> fx -= r * cx2 * (r2 - c2 * r3) / r2 / r3 * s2;
                  a3 -> fy -= r * cy2 * (r2 - c2 * r3) / r2 / r3 * s2;
                  a3 -> fz -= r * cz2 * (r2 - c2 * r3) / r2 / r3 * s2;
                  a3 -> fx += r * cx1 * c1 / r2 * s1;
                  a3 -> fy += r * cy1 * c1 / r2 * s1;
                  a3 -> fz += r * cz1 * c1 / r2 * s1;
                }

                if (a4 -> active) {
                  a4 -> fx += r * cx2 / r3 * s2;
                  a4 -> fy += r * cy2 / r3 * s2;
                  a4 -> fz += r * cz2 / r3 * s2;
                }
//                MCL_MutexOff();
              }
            }
          }
        }
      }
    }
    if (bp == bp -> next) return TRUE;
    if ((++j) > MCL_GetVarTh(Threads)) j = 1;
    bp = bp -> next;
  } /* End of while */

  return FALSE;
}


/**** Get the torsion ****/

void AMMP_FASTCALL get_torsion(AMMP_ATOM *a1, AMMP_ATOM *bonded[], int mbond, int *inbond)
{
  TORSION       *mine = torsion_first;

  *inbond = 0;

  while(mine) {
    if (mine -> atom1 == a1)
      bonded[(*inbond)++] = mine -> atom4;
    if (mine -> atom4 == a1)
      bonded[(*inbond)++] = mine->atom1;
    if (mine == mine -> next) return;
    mine = mine -> next;
    if (*inbond == mbond) return;
  } /* End of while */
}


/**** Add a new torsion ****/

int AMMP_FASTCALL torsion(int p1, int p2, int p3, int p4, float fk, int n, float off)
{
  AMMP_ATOM     *ap1, *ap2, *ap3, *ap4;
  TORSION       *new;

  if ((p1 == p2) || (p1 == p3) || (p1 == p4) ||
      (p2 == p3) || (p2 == p4) || (p3 == p4)) {
    aaerror("Same atom in torsion %d %d %d %d", p1, p2, p3, p4);
    return TRUE;
  }

  if (fk == 0.0f) return TRUE;

  /**** Get the atom pointers for the two serial numbers ****/

  ap1 = a_m_serial_table(p1);
  ap2 = a_m_serial_table(p2);
  ap3 = a_m_serial_table(p3);
  ap4 = a_m_serial_table(p4);

  if ((!ap1) || (!ap2) || (!ap3) || (!ap4)) {
    aaerror("Undefined atom in torsion %d %d %d %d", p1, p2, p3, p4);
    return FALSE;
  }

  if ((new = (TORSION *)Alloca(sizeof(TORSION), "torsion()")) == NULL)
    return FALSE;

  /**** Initialize the pointers ****/

  if( torsion_first == NULL) torsion_first = new;
  if( torsion_last == NULL) torsion_last = new;
  new -> atom1 = ap1;
  new -> atom2 = ap2;
  new -> atom3 = ap3;
  new -> atom4 = ap4;
  new -> offset = off;
  new -> k = fk;
  new -> n = n;
  new -> next = new;
  torsion_last -> next = new;
  torsion_last = new;

  return TRUE;
}


/**** Calculate the torsion potential (parallel) ****/

int AMMP_FASTCALL v_torsion_p(float *V, float lambda)
{
  MCL_BEGIN
    float               lambda;
    float               *V;
  MCL_END

  MCL_ALLOCGLOBVARS;

  if (!torsion_first) return TRUE;

  MCL_GetVar(lambda) = lambda;
  MCL_GetVar(V     ) = V;
  MCL_CreateThreads(0, v_torsion_thread, &HdtGlobVars, MCL_THREADS_WAIT);

  return TRUE;
}


/**** Calculate the torsion potential ****/

int AMMP_FASTCALL v_torsion(float *V, float lambda)
{
  AMMP_ATOM     *a1, *a2, *a3, *a4;
  doublefloat   x1, y1, z1, x2, y2, z2, x3, y3, z3;
  doublefloat   r, cx1, cy1, cz1, cx2, cy2, cz2;
  doublefloat   dp;

  TORSION *     bp = torsion_first;

  if (!bp) return TRUE;

  while(bp) {
    a1 = bp -> atom1;
    a2 = bp -> atom2;
    a3 = bp -> atom3;
    a4 = bp -> atom4;
    if ((a1 -> active) || (a2 -> active) || (a3 -> active) || (a4 -> active)) {
      if (lambda) {
        x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
        y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
        z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
        x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
        y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
        z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
        x3 = a4 -> x - a3 -> x + lambda * (a4 -> dx - a3 -> dx);
        y3 = a4 -> y - a3 -> y + lambda * (a4 -> dy - a3 -> dy);
        z3 = a4 -> z - a3 -> z + lambda * (a4 -> dz - a3 -> dz);
      } else {
        x1 = a1 -> x - a2 -> x;
        y1 = a1 -> y - a2 -> y;
        z1 = a1 -> z - a2 -> z;
        x2 = a3 -> x - a2 -> x;
        y2 = a3 -> y - a2 -> y;
        z2 = a3 -> z - a2 -> z;
        x3 = a4 -> x - a3 -> x;
        y3 = a4 -> y - a3 -> y;
        z3 = a4 -> z - a3 -> z;
      }

      /**** 1 cross 2 ****/

      cx1 =  y1 * z2 - y2 * z1;
      cy1 = -x1 * z2 + x2 * z1;
      cz1 =  x1 * y2 - x2 * y1;
      r   = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;
      if (r > TOO_SMALL) {
        r   = 1.0 / sqrt(r);
        cx1 *= r;
        cy1 *= r;
        cz1 *= r;

        /**** 3 cross 2 ****/

        cx2 =  y3 * z2 - y2 * z3;
        cy2 = -x3 * z2 + x2 * z3;
        cz2 =  x3 * y2 - x2 * y3;
        r   = cx2 * cx2 + cy2 * cy2 + cz2 * cz2;
        if (r > TOO_SMALL) {
          r   = 1.0 / sqrt(r);
          cx2 *= r;
          cy2 *= r;
          cz2 *= r;

          /**** If here everything is well determined ****/

          dp = cx1 * cx2 + cy1 * cy2 + cz1 * cz2; /* cos( abs(theta)) */
          if (dp > 1.0) dp = 0;
          else if (dp < -1.0) dp = PI_DP;
            else dp = acos(dp);

          /**** Determine the sign by triple product ****/

          r = cx1 * x3 + cy1 * y3 + cz1 * z3;
          if (r > zero) dp = -dp ;
          *V += 0.5 * bp -> k * (1.0 + cos(bp -> n * dp - bp -> offset));
        }
      }
    }
    if (bp == bp -> next) return TRUE;
    bp = bp -> next;
  } /* End of while */


  return FALSE;
}


/**** Calculate the torsion potential (parallel, thread) ****/

int v_torsion_thread(void *HdtGlobVars)
{
  MCL_BEGIN
    float               lambda;
    float               *V;
  MCL_END

  AMMP_ATOM             *a1, *a2, *a3, *a4;
  doublefloat           x1, y1, z1, x2, y2, z2, x3, y3, z3;
  doublefloat           r, cx1, cy1, cz1, cx2, cy2, cz2;
  doublefloat           dp;
  unsigned int          ThreadID;

  float                 V      = zero;
  float                 lambda = MCL_GetVarTh(lambda);
  TORSION *             bp     = torsion_first;
  unsigned int          j      = 1;

  MCL_ObtainThID(ThreadID);

  while(bp) {
    if (j == ThreadID) {
      a1 = bp -> atom1;
      a2 = bp -> atom2;
      a3 = bp -> atom3;
      a4 = bp -> atom4;
      if ((a1 -> active) || (a2 -> active) || (a3 -> active) || (a4 -> active)) {
        if (lambda) {
          x1 = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
          y1 = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
          z1 = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
          x2 = a3 -> x - a2 -> x + lambda * (a3 -> dx - a2 -> dx);
          y2 = a3 -> y - a2 -> y + lambda * (a3 -> dy - a2 -> dy);
          z2 = a3 -> z - a2 -> z + lambda * (a3 -> dz - a2 -> dz);
          x3 = a4 -> x - a3 -> x + lambda * (a4 -> dx - a3 -> dx);
          y3 = a4 -> y - a3 -> y + lambda * (a4 -> dy - a3 -> dy);
          z3 = a4 -> z - a3 -> z + lambda * (a4 -> dz - a3 -> dz);
        } else {
          x1 = a1 -> x - a2 -> x;
          y1 = a1 -> y - a2 -> y;
          z1 = a1 -> z - a2 -> z;
          x2 = a3 -> x - a2 -> x;
          y2 = a3 -> y - a2 -> y;
          z2 = a3 -> z - a2 -> z;
          x3 = a4 -> x - a3 -> x;
          y3 = a4 -> y - a3 -> y;
          z3 = a4 -> z - a3 -> z;
        }

        /**** 1 cross 2 ****/

        cx1 =  y1 * z2 - y2 * z1;
        cy1 = -x1 * z2 + x2 * z1;
        cz1 =  x1 * y2 - x2 * y1;
        r   = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;
        if (r > TOO_SMALL) {
          r   = 1.0 / sqrt(r);
          cx1 *= r;
          cy1 *= r;
          cz1 *= r;

          /**** 3 cross 2 ****/

          cx2 =  y3 * z2 - y2 * z3;
          cy2 = -x3 * z2 + x2 * z3;
          cz2 =  x3 * y2 - x2 * y3;
          r   = cx2 * cx2 + cy2 * cy2 + cz2 * cz2;
          if (r > TOO_SMALL) {
            r   = 1.0 / sqrt(r);
            cx2 *= r;
            cy2 *= r;
            cz2 *= r;

            /**** If here everything is well determined ****/

            dp = cx1 * cx2 + cy1 * cy2 + cz1 * cz2; /* cos( abs(theta)) */
            if (dp > 1.0) dp = 0;
            else if (dp < -1.0) dp = PI_DP;
              else dp = acos(dp);

            /**** Determine the sign by triple product ****/

            r = cx1 * x3 + cy1 * y3 + cz1 * z3;
            if (r > zero) dp = -dp ;
            V += 0.5 * bp -> k * (1.0 + cos(bp -> n * dp - bp -> offset));
          }
        }
      }
    }
    if (bp == bp -> next) break;
    if ((++j) > MCL_GetVarTh(Threads)) j = 1;
    bp = bp -> next;
  } /* End of while */

  MCL_MutexOn();
  *MCL_GetVarTh(V) += V;
  MCL_MutexOff();


  return FALSE;
}


/* a_torsion()
* this function sums up the potentials
* for the atoms defined in the torsion data structure.
*/
/* standard returns 0 if error (any) 1 if ok
* V is the potential */

int AMMP_FASTCALL a_torsion(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
	TORSION *bp;
/* difference vectors */
	float x1,y1,z1,x2,y2,z2,x3,y3,z3;
/* cross products and storage for normalizing */
	float r,cx1,cy1,cz1,cx2,cy2,cz2;
	float dp;
	AMMP_ATOM *a1,*a2,*a3,*a4;


	bp = torsion_first;
       if( bp == NULL ) return 1;
       while(1)
       {
	if( bp == NULL) return 0;
	a1 = bp->atom1; a2 = bp->atom2; a3 = bp->atom3;
	a4 = bp->atom4;
	if( (a1->serial >= ilow && a1->serial <= ihigh)
	||  (a2->serial >= ilow && a2->serial <= ihigh)
	||  (a3->serial >= ilow && a3->serial <= ihigh) 
	||  (a4->serial >= ilow && a4->serial <= ihigh) )
	{
	x1 = (a1->x -a2->x +lambda*(a1->dx-a2->dx));
	y1 = (a1->y -a2->y +lambda*(a1->dy-a2->dy));
	z1 = (a1->z -a2->z +lambda*(a1->dz-a2->dz));
	x2 = (a3->x -a2->x +lambda*(a3->dx-a2->dx));
	y2 = (a3->y -a2->y +lambda*(a3->dy-a2->dy));
	z2 = (a3->z -a2->z +lambda*(a3->dz-a2->dz));
	x3 = (a4->x -a3->x +lambda*(a4->dx-a3->dx));
	y3 = (a4->y -a3->y +lambda*(a4->dy-a3->dy));
	z3 = (a4->z -a3->z +lambda*(a4->dz-a3->dz));
/* 1 cross 2 */
	cx1 = y1*z2 - y2*z1;
	cy1 = -x1*z2 + x2*z1;
	cz1 = x1*y2 - x2*y1;
	r = cx1*cx1 + cy1*cy1 + cz1*cz1;
	if( r < TOO_SMALL) goto SKIP;
	r = sqrt(r);
	cx1 = cx1/r;
	cy1 = cy1/r;
	cz1 = cz1/r;
/* 3 cross 2 */
	cx2 = y3*z2 - y2*z3;
	cy2 = -x3*z2 + x2*z3;
	cz2 = x3*y2 - x2*y3;
	r = cx2*cx2 + cy2*cy2 + cz2*cz2;
	if( r < TOO_SMALL) goto SKIP;
	r = sqrt(r);
	cx2 = cx2/r;
	cy2 = cy2/r;
	cz2 = cz2/r;
/* if here everything is well determined */
	dp = cx1*cx2 + cy1*cy2 + cz1*cz2; /* cos( abs(theta)) */
	if( dp > 1.) dp = 1.; if( dp < -1.) dp = -1.;
	dp = acos(dp);
/* determine the sign by triple product */
	r = cx1*x3 + cy1*y3 + cz1*z3;
	if( r > 0 ) dp =  -dp ;
	z2 = .5*(bp->k)*( 1. + cos( bp->n * dp - bp->offset   )) ;
	*V += z2;
	fprintf(op,"Torsion %s %d %s %d %s %d %s %d E %f Angle %f error %f\n",
                a1 -> name, a1 -> serial, a2 -> name, a2 -> serial,
                a3 -> name, a3 -> serial, a4 -> name, a4 -> serial,
                z2, AMMP_RAD_TO_DEG(dp),
                AMMP_RAD_TO_DEG(acos(-(cos(bp -> n * dp - bp -> offset)))));
	}
SKIP:
	if( bp == bp->next ) return 1;
	bp = bp->next;
       }
}

void AMMP_FASTCALL gsdg_torsion(AMMP_ATOM *ap)
{
TORSION *tp;
float b1,b2,b3;

	tp =torsion_first;
	while( 1)
	{ 
		if( tp == NULL ) return;
		if( tp->atom1 == ap)
		{
			b1 = bond_length(ap,tp->atom2);
			b2 = bond_length(tp->atom2,tp->atom3);
			b3 = bond_length(tp->atom3,tp->atom4);
			ap->vx = b2 + .75*(b1+b3);
			ap->vx *= ap->vx;
			ap->vy = 10.;
		}
		if( tp->atom4 == ap)
		{
			b1 = bond_length(ap,tp->atom3);
			b2 = bond_length(tp->atom2,tp->atom3);
			b3 = bond_length(tp->atom2,tp->atom1);
			ap->vx = b2 + .75*(b1+b3);
			ap->vx *= ap->vx;
			ap->vy = 10.;
		}
		if( tp == tp->next) return;
		tp = tp->next;
	}
}


