/*
* tncnewt.c
*
* truncated newtons method
* optimiser routine for AMMP
*
* theory is to use a finite difference appx to the
* hessian step product and find the optimal step vector
* that way.
*
* basically this is fixed point iteration for the
* difference in gradients being colinear and 
* descent direction along the step vector.
* 
*/
/*
*  copyright 1992 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/* function tncnewt()
* truncated newtons method optimiser
*
* passed a set of functions vfs for the potential
* passed a set of functions ffs for the force
*  how many functions  nfs
*  how many steps to try
*  when to stop
*/

int AMMP_FASTCALL tncnewt(FILE *op, AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs,
                          int nstep, float dtoler, float toler, int echo)
{
  AMMP_ATOM *   ap;
  float         vb, vt, alpha, atop;
  float         dfx, dfy, dfz;
  float         lam;
  int           i, ifs, istep, Ret;

  const char *  StepStr = "  Trunc  %5d: vb %f, maxf %f\n";

#ifdef VEGAZZ
  float         lastlam = 0.0f;
  int           nupdat  = GetNupdat();
  int           supdat  = GetSupdat();
#endif

  if (!CheckAtoms()) return FALSE;
  if (nstep < 1) nstep = 1;
  if (dtoler < 1.e-3f) dtoler = 0.001f;

  Ret = FALSE;
  for(istep = 0; istep < nstep ; istep++) {

#ifdef VEGAZZ
    if ((nupdat) && (!(istep % nupdat))) send_all_atoms();
#endif

    vb = 0.0f;
    for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vb,  0.0f);
    a_f_zero();
    for(ifs = 0; ifs < nfs; ifs++) (*ffs[ifs])(0.0f);
    lam = a_max_f();

#ifdef VEGAZZ
    lastlam = lam;
    if ((supdat) && (!(istep % supdat)))
#else
    if (echo)
#endif
      fprintf(op, StepStr, istep, vb, lam);

    if (lam <= toler) {
      Ret = TRUE;
      break;
    }
    if (lam <= 1.0f) a_ftodx(1.0f , 0.0f);
    else a_ftodx(1.0 / sqrt(lam), 0.0f);
    a_ftovx(1.0f, 0.0f);

    /* Initial iteration to find the step vector
     * using a small number of trial searches
     * rather than the largish number in tnpack
     * but not checking for convergence
     */

    a_v_zero();
    for(i = 0; i < 5; i++) {
      a_f_zero();
      for(ifs = 0; ifs < nfs; ifs++) (*ffs[ifs])(dtoler);
      ap = a_next(-1);

      /**** Form the Hd product and alpha ****/

      alpha = 0.0f;
      atop  = 0.0f;
      while(ap != NULL) {
        dfx    = (ap -> fx - ap -> dx) / dtoler;
        dfy    = (ap -> fy - ap -> dy) / dtoler;
        dfz    = (ap -> fz - ap -> dz) / dtoler;
        alpha += ap -> dx * dfx;
        alpha += ap -> dy * dfy;
        alpha += ap -> dz * dfz;
        atop  += (dfx - ap -> fx) * ap -> dx;
        atop  += (dfy - ap -> fy) * ap -> dy;
        atop  += (dfz - ap -> fz) * ap -> dz;
        if (ap == ap -> next) break;
        ap = ap -> next;
      } /* End of while */
      lam = sqrt(a_max_f());
      if (lam < 1.0f) lam = 1.0f;
      ap = a_next(-1);
      dfx = atop / alpha;
      while(ap != NULL) {
        ap -> vx += dfx * ap -> dx;
        ap -> dx  = ap -> fx / lam;
        ap -> vy += dfx * ap -> dy;
        ap -> dy  = ap -> fy / lam;
        ap -> vz += dfx * ap -> dz;
        ap -> dz = ap -> fz / lam ;
        if (ap == ap -> next) break;
        ap = ap -> next;
      } /* End of while */
    }

    ap = a_next(-1);
    while(ap != NULL) {
      ap -> dx = ap -> vx;
      ap -> dy = ap -> vy;
      ap -> dz = ap -> vz;
      if (ap == ap -> next) break;
      ap = ap -> next;
    } /* End of while */

    lam = linmin(vfs, nfs, 0.01f);
    if (lam < 1.e-6f) {
      a_f_zero();
      for(ifs = 0; ifs < nfs; ifs++) (*ffs[ifs])( 0.0f);
      lam = sqrt(a_max_f());
      if (lam < 1.0f) lam = 1.0f;
      a_ftodx(1.0f / lam, 0.0f);
      lam = linmin(vfs, nfs, 0.01f);
      if (lam < 1.e-8f) {
        Ret = TRUE;
        break;
      }
      vt = 0.0f;
      for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vt, lam);

#ifndef VEGAZZ
      if (supdat)
#else
      if (echo)
#endif
        fprintf(op, "  Trunc  %5d: Steep sub-iter %f %f\n", istep, vt, vb);
      if (vt > vb) break;
    }
    a_inc_d(lam);
  } /* End of for (istep) */

#ifdef VEGAZZ
  if ((supdat) && ((istep % supdat) || (istep == nstep)))
    fprintf(op, StepStr, istep, vb, lastlam);
#else
  if (echo)
#endif
    fprintf(op, "\n");

  return Ret;
}
