/*
* trust.c
*
* trust based
* optimiser routine for AMMP
*
* theory is to move in a steepest descent direction until
* the disagreement between observed and calculated energies 
* is too high  ( use a .1 v_init tolerance zB) 
* 
*/
/*
*  copyright 1992 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/* function trust()
* trust region based optimiser
*
* passed a set of functions vfs for the potential
* passed a set of functions ffs for the force
*  how many functions  nfs
*  how many steps to try
*  when to stop
*/

int AMMP_FASTCALL trust(AMMP_VFUNC vfs[], AMMP_FFUNC ffs[], int nfs, int nstep, float dtoler, float toler)
{
  int           Abort, i, j, ifs;
  float         vb, vt, vto, vtk;
  float         lam, vtoler;

  const char *  StepStr = "  Trust  %5d: vb %f, maxf %f\n";
  float         lastvb  = 0.0f;
  int           vbcheck = 0;

#ifdef VEGAZZ
  float         lastlam = 0.0f;
  int           nupdat  = GetNupdat();
  int           supdat  = GetSupdat();
#endif

  if (!CheckAtoms()) return FALSE;
  if (nstep < 1) nstep = 1;

  a_v_zero();
  if (dtoler < 1.e-3) dtoler = 0.1f;

  Abort = FALSE;
  for(i = 0; i < nstep; i++) {
#ifdef VEGAZZ
    if ((nupdat) && (!(i % nupdat))) send_all_atoms();
#endif

    vb = 0.0f;
    for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vb, 0.0f);
    a_f_zero();
    for(ifs = 0; ifs < nfs; ifs++) (*ffs[ifs])(0.0f);
    lam = a_max_f();

#ifdef VEGAZZ
    if ((supdat) && (!(i % supdat)))
#else
    if (echo)
#endif
      printf(StepStr, i, vb, lam);

    if (vb == lastvb) {
      if (vbcheck++ == 5) {
        Abort = TRUE;
        break;
      }
    } else vbcheck = 0;
    lastvb  = vb;
#ifdef VEGAZZ
    lastlam = lam;
#endif

    if (lam <= toler) break;
    if (lam <= 1.0f) a_ftodx(1.0f, 0.0f);
    else a_ftodx(0.1 / sqrt(lam), 0.0f);
    vtk = vb;
    lam = 0.0f;

    /* This is a Fibonacci line search rather odd, the steps keep getting
     * bigger until either the end or we go up.
     */

    vtoler = fabs(vb * dtoler);
    if (vtoler < 0.1f) vtoler = 0.1f;
    for(j = 0; j < 200; j++) {
      vt   = 0.0f;
      lam += 0.001f * (float)j;
      for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vt, lam);
      vto = v_predict(lam, vb);
      if (((vt - vto) > vtoler) || ((vt > vtk) && (j != 0))) {
        lam -= 0.001f * (float)j;
        break;
      }
      vtk = vt;
    } /* End of for (j) */
    a_inc_d(lam);
  } /* End of for (i) */

#ifdef VEGAZZ
  if ((supdat) && ((i % supdat) || (i == nstep)))
    printf(StepStr, i, vb, lastlam);
#endif

  if (Abort)
    printf("  Trust       : Unable to converge. Minimization stopped\n");

#ifdef VEGAZZ
  if (supdat) printf("\n");
#else
  if (echo  ) printf("\n");
#endif

  return TRUE;
}


/* function v_predict( lam,vinit )
*
* predicts the potential based on the initial force
* vector
*
* dv =  -F dx 
*/

float AMMP_FASTCALL v_predict(float lambda, float vinit)
{
  AMMP_ATOM *   ap = a_next(-1);
  float         vp = vinit;

  if( ap == NULL) return vp;
  while(ap->next) {
    vp -= ap -> fx * ap -> dx * lambda;
    vp -= ap -> fy * ap -> dy * lambda;
    vp -= ap -> fz * ap -> dz * lambda;
    if (ap == ap -> next) return vp;
    ap = ap -> next;
  } /* End of while */

  return vp;
}
