/* tset.c
*
* collection of routines to service  torsion searching
*
* POOP (Poor-mans Object Oriented Programming) using scope rules
*
*
*  tset forces a given torsion to be a given value
*  unlike tgroup tset is one deep and does not check energy.
*
*  tmin  searches one and only one torsion angle for an energy minimum
*
*
*  tmap i j k l ii jj kk ll nstep1 nstep2 ;
*   step through the two torsions and produce a map of the energies
*   the atoms are stored in the vx registers
*
*  tgroup/tsearch define a structure and use it
*  tset/tmin   do it on the fly.
*
*  tmin can be quite expensive
*
*/
/*
*  copyright 1993,1995 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <ctype.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Local variables ****/

static const char *     NdefStr = "An atom in tset is not defined";

#ifdef __BORLANDC__

/**** Single precision assembly floor ****/

__inline float floorf(float Val)
{
  if (Val >= 0) return (float)((int)Val);

  return (float)(-1 + (int)Val);
}
#endif


/**** Normalize a torsion in the 0 2Pi range ****/

float AMMP_FASTCALL normalize_torsion(register float Val)
{
  if ((Val >= zero) && (Val <= TWOPI)) return Val;

  return (Val - TWOPI * floorf(Val / TWOPI));
}


/**** Set a torsion angle ****/

void AMMP_FASTCALL tset(FILE *op, int echo, int i1, int i2, int i3,int i4, float alpha)
{
  AMMP_ATOM     *ap1, *ap2, *ap3, *ap4;
  float         original, delta;

  if (!CheckAtoms()) return;

  if (((ap1 = a_m_serial(i1)) == NULL) ||
      ((ap2 = a_m_serial(i2)) == NULL) ||
      ((ap3 = a_m_serial(i3)) == NULL) ||
      ((ap4 = a_m_serial(i4)) == NULL)) {
    aaerror(NdefStr);
    return;
  }

  /**** Get the torsion value and check if we have to do the work ****/

  original = get_torsion_value( ap1,ap2,ap3,ap4);
  if (fabs(original - alpha) < 1.e-3f) return;

  /**** Set up the bond structure flags in ap->gx ****/

  tset_bond_build(ap1, ap2, ap3, ap4);

  delta = alpha - original;
  set_torsion(ap1, ap2, ap3, ap4, delta);

#ifndef VEGAZZ
  if (echo)
#endif
    fprintf(op, "  Tset        : Original %f, delta %f, final %f\n",
            AMMP_RAD_TO_DEG(original), AMMP_RAD_TO_DEG(delta),
	    AMMP_RAD_TO_DEG(get_torsion_value(ap1, ap2, ap3, ap4)));
}


/**** Torsion minimizier ****/

void AMMP_FASTCALL tmin(FILE *op, int echo, int i1, int i2, int i3, int i4,
                        int nstep, AMMP_VFUNC vfs[], int nfs)
{
  float         angle, original, delta;
  float         vtemp, vmax;
  AMMP_ATOM     *ap1, *ap2, *ap3, *ap4;
  int           i, imax, ifs;

#ifdef VEGAZZ
  int           laststep;

  int           nupdat = GetNupdat();
  int           supdat = GetSupdat();
#endif

  if (!CheckAtoms()) return;
  if (nstep < 1) nstep = 12;

#ifdef VEGAZZ
  laststep = nstep - 1;
#endif

  if (((ap1 = a_m_serial(i1)) == NULL) ||
      ((ap2 = a_m_serial(i2)) == NULL) ||
      ((ap3 = a_m_serial(i3)) == NULL) ||
      ((ap4 = a_m_serial(i4)) == NULL)) {
    aaerror(NdefStr);
    return;
  }

  /**** Get the torsion value and check if we have to do the work ****/

  original = get_torsion_value(ap1, ap2, ap3, ap4);

  fprintf(op, "  Starting the torsion minimizer (angle %.3f, steps %d)\n\n",
          AMMP_RAD_TO_DEG(original), nstep);

  /**** Set up the bond structure flags in ap->gx ****/

  tset_bond_build(ap1, ap2, ap3, ap4);

  delta =  -original;
  set_torsion(ap1, ap2, ap3, ap4, delta);

  imax  = -1;
  vmax  = 10e20f;
  delta = TWOPI / (float)nstep;
  angle = delta;

  for(i = 0; i < nstep; i++) {
    vtemp = 0.0f;
    for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&vtemp, 0.0f);
    if (vtemp < vmax) {
      vmax = vtemp;
      imax = i;
    }
    set_torsion(ap1, ap2, ap3, ap4, delta);

#ifdef VEGAZZ
    if ((nupdat) && ((!(i % nupdat)) || (i == laststep))) send_all_atoms();
    if ((supdat) && ((!(i % supdat)) || (i == laststep)))
#else
    if (echo)
#endif
      fprintf(op, "  Tmin   %5d: angle %.3f, v %.3f, vb %.3f\n",
              i + 1, AMMP_RAD_TO_DEG(angle), vtemp, vmax);
    angle += delta;
  } /* End of for (i) */

  set_torsion(ap1, ap2, ap3, ap4, (imax + 1) * delta);

#ifdef VEGAZZ
  if (nupdat) send_all_atoms();
#endif

  fprintf(op, "\n  Best conformation selected (angle %.3f, v %f)\n",
          AMMP_RAD_TO_DEG(get_torsion_value(ap1, ap2, ap3, ap4)),
          vmax);
}


/**** Set the atoms to move the torsion ****/

void AMMP_FASTCALL tset_bond_build(AMMP_ATOM *ap1, AMMP_ATOM *ap2, AMMP_ATOM *ap3, AMMP_ATOM *ap4)
{
  AMMP_ATOM     *ap, *bonded[20];
  int           i, inbond;
  int           j, tobonded;

  int           numatom = a_number();

  for(i = 0; i < numatom; i++) {
    ap       = a_next(i);
    ap -> gx = -1.0f;
  } /* End of for (i) */

  tobonded = 0;
  get_bond(ap3, bonded, 20, &inbond);
  for(i = 0; i < inbond; i++) {
    if ((bonded[i] != ap1) && (bonded[i] != ap2)) {
      bonded[i] -> gx = 0.0f;
      tobonded++;
    }
  } /* End of for (i) */

  get_bond(ap4, bonded, 20, &inbond);
  for(i = 0; i < inbond; i++) {
    if ((bonded[i] != ap1) && (bonded[i] != ap2)) {
      bonded[i] -> gx = 0.0f;
      tobonded++;
    }
  } /* End of for (i) */
  ap3 -> gx = 1.0f;
  ap4 -> gx = 1.0f;

  while(tobonded > 0) {
    tobonded = 0;
    for(i = 0; i < numatom;i++) {
      ap = a_next(i);
      if ((ap != ap1) && (ap != ap2)) {
        if (ap -> gx == 0.0f) {
          ap -> gx = 1.0f;
          get_bond(ap, bonded, 20, &inbond);
          for(j = 0; j < inbond; j++) {
            if (bonded[j] -> gx < 1.0f) {
              bonded[j] -> gx = 0.0f;
              tobonded++;
            }
          } /* End of for (j) */
        }
      }
    } /* End of for (i) */
  } /* End of while */
}


/**** Get the torsion value ****/

float AMMP_FASTCALL get_torsion_value(AMMP_ATOM *a1, AMMP_ATOM *a2, AMMP_ATOM *a3,
                                      AMMP_ATOM *a4)
{
  float         cx2,cy2,cz2;
  float         dp;
	
  float         x1  = a1 -> x - a2 -> x;
  float         y1  = a1 -> y - a2 -> y;
  float         z1  = a1 -> z - a2 -> z;
  float         x2  = a3 -> x - a2 -> x;
  float         y2  = a3 -> y - a2 -> y;
  float         z2  = a3 -> z - a2 -> z;
  float         x3  = a4 -> x - a3 -> x;
  float         y3  = a4 -> y - a3 -> y;
  float         z3  = a4 -> z - a3 -> z;
  float         cx1 =  y1 * z2 - y2 * z1;
  float         cy1 = -x1 * z2 + x2 * z1;
  float         cz1 =  x1 * y2 - x2 * y1;
  float         r   = cx1 * cx1 + cy1 * cy1 + cz1 * cz1;

  if (r < 1.e-4f) return 0.0f;

  r   = 1.0f / (float)sqrt(r);
  cx1 = cx1 * r;
  cy1 = cy1 * r;
  cz1 = cz1 * r;
  cx2 =  y3 * z2 - y2 * z3;
  cy2 = -x3 * z2 + x2 * z3;
  cz2 =  x3 * y2 - x2 * y3;
  r   = cx2 * cx2 + cy2 * cy2 + cz2 * cz2;
  if (r < 1.e-4f) return 0.0f;

  r   = 1.0f / (float)sqrt(r);
  cx2 = cx2 * r;
  cy2 = cy2 * r;
  cz2 = cz2 * r;

  /**** if here everything is well determined ****/

  dp = cx1 * cx2 + cy1 * cy2 + cz1 * cz2; /* cos( abs(theta)) */
  if (dp > 1.0f) dp = 1.0f;
  else if (dp < -1.0f) dp = -1.0f;
  dp = (float)acos(dp);

  /**** Determine the sign by triple product ****/

  if ((cx1 * x3 + cy1 * y3 + cz1 * z3) > 0.0f) dp = -dp ;

  return dp;
}


/**** Set the torsion value ****/

void AMMP_FASTCALL set_torsion(AMMP_ATOM *ap1, AMMP_ATOM *ap2, AMMP_ATOM *ap3,
                               AMMP_ATOM *ap4, float howmuch)
{
  AMMP_ATOM *   ap;
  float         phi, cphi, sphi, xphi;
  float         ry, rz, nnrx, nnry, nnrz, rnx, rny, rnz;
  int           i;

  AMMP_ATOM *   b1      = ap2;
  AMMP_ATOM *   b2      = ap3;
  float         nx      = b2 -> x - b1 -> x;
  float         ny      = b2 -> y - b1 -> y;
  float         nz      = b2 -> z - b1 -> z;
  float         rx      = sqrt(nx * nx + ny * ny + nz * nz);
  int           numatom = a_number();

  if (rx < 1.e-6f) {
    aaerror("Bad torsion radius in set_torsion()");
    return ;
  }

  nx /= rx;
  ny /= rx;
  nz /= rx;

  cphi = (float)cos(howmuch);
  sphi = (float)sin(howmuch);
  phi  = 1.0f - cphi;

  for(i = 0; i < numatom; i++) {
    ap = a_next(i);
    if((ap -> gx > 0.0f) && (ap != b2)) {
      rx   = ap -> x - b1 -> x;
      ry   = ap -> y - b1 -> y;
      rz   = ap -> z - b1 -> z;
      xphi = nx * rx + ny * ry + nz * rz;
      nnrx = xphi * nx;
      nnry = xphi * ny;
      nnrz = xphi * nz;
      rnx  = ny * rz - nz * ry;
      rny  = -nx * rz + nz * rx;
      rnz  = nx * ry - ny * rx;
      rx   = cphi * rx + phi * nnrx + sphi * rnx;
      ry   = cphi * ry + phi * nnry + sphi * rny;
      rz   = cphi * rz + phi * nnrz + sphi * rnz;
      ap -> x = rx + b1 -> x;
      ap -> y = ry + b1 -> y;
      ap -> z = rz + b1 -> z;
    }
  } /* End of for (i) */
}


void AMMP_FASTCALL tmap(FILE *op, int echo, AMMP_VFUNC vfs[], int nfs, int i1, int i2,
                        int i3, int i4, int j1 , int j2 ,int j3, int j4, int nistep,
                        int njstep)
{
  AMMP_ATOM *   ap;
  float         V;
  float         x, dx, y, dy;
  int           i, j, ifs;

  int           numatm = a_number();

  if (!CheckAtoms()) return;
  if (numatm < 5) {
    aaerror("Not enough atoms for Tmap (min. 5)");
    return;
  }

  if (nistep < 1) nistep = 12;
  if (njstep < 1) njstep = 12;

  /**** Save the atoms ****/

  for(i = 0; i < numatm; i++) {
    ap       = a_next(i);
    ap -> vx = ap -> x;
    ap -> vy = ap -> y;
    ap -> vz = ap -> z;
  } /* End of for (i) */

  x  = - PI;
  dx = 0.0f;
  dy = 0.0f;
  if (nistep > 1) dx = TWOPI / (float)nistep;
  if (njstep > 1) dy = TWOPI / (float)njstep;
  fprintf(op, "  Tmap        : %d %d torsion map %f %f steps\n",
          nistep, njstep, AMMP_RAD_TO_DEG(dx),
          AMMP_RAD_TO_DEG(dy));

  for(i = 0; i < nistep; i++) {
    tset(op, 0, i1, i2, i3, i4, x);
    y = - PI;
    for(j = 0; j< njstep; j++) {
      tset(op, 0, j1, j2, j3, j4, y);
      V = 0.0f;
      for(ifs = 0; ifs < nfs; ifs++) (*vfs[ifs])(&V, 0.0f);
      fprintf(op, "  Tmap %3d %3d: Energy %f\n", i, j, V);
      y = y + dy;
    } /* End of for (j) */
    x = x + dx;
  } /* End of for (i) */

  /**** Restore the atoms ****/

  for(i = 0; i < numatm; i++) {
    ap      = a_next(i);
    ap -> x = ap -> vx;
    ap -> y = ap -> vy;
    ap -> z = ap -> vz;
  } /* End of for (i) */

#ifndef VEGAZZ
  if (echo)
#endif
    fprintf(op, "\n");
}


