
/*************************************************
****        AMMP - Non-bond potential         ****
**** Copyright 1992-2012, Robert W. Harrison  ****
****    VEGA edition by Alessandro Pedretti   ****
*************************************************/

/*
 * unonbon uses a 'use list' for the interactions
 *  this is physically incorrect, but often done.
 *  the use list is coded as ap ... bp's for interaction ... ap
 *  a single array is alloca'd, it is 20 a_number() long
 *  the global variable (in variable storage) nbdeep will allowriding
 *  every nsteps ( 10 default ) will recalculate the list
 *  again this may be over ridden with  nbstep
 *  and if cutoff is not set (== 0) these routines silently call the
 *  regular routines which don't care about cutoff
 *  will always redo the list if a_number() changes
 *
 */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Local variables ****/

static AMMP_ATOM **     uselist_local;
static int              uselist_lsize;

static float            uselist_oldcutoff     = -1.0f;
static int              uselist_oldatomnumber = 0;
static int              uselist_since         = 0;

/**** Local prototypes ****/

static int              AMMP_FASTCALL SetSize(int Size);
static int              AMMP_FASTCALL uselist(AMMP_ATOM ***List, int *Size, float Cutoff);

/**** Reset local variables ****/

void AMMP_FASTCALL ResetUnonbon(void)
{
  uselist_oldcutoff     = -1.0f;
  uselist_oldatomnumber = 0;
  uselist_since         = 0;
}


/**** Calculate the non-bond potential ****/

int AMMP_FASTCALL u_v_nonbon(float *V, float lambda)
{
  AMMP_ATOM     *a1;
  AMMP_ATOM     *bp;
  AMMP_ATOM     **use;
  AMMP_ATOM     **atms;
  float         r, r0, xt, yt, zt;
  float         dielectric, rdebye;
  float         (*buffer)[],xx,yy,zz;
  float         (*vector)[];
  int           i, nuse, used;
  int           inbuffer, imax, jj;

  const char *  Routine = "u_v_nonbon()";
  float         cutoff  = GetCutoff();


  if ((cutoff > 1000.0f) || (cutoff == 0.0f)) return v_nonbon(V, lambda);


  rdebye     = cutoff * 0.5f;
  dielectric = Q_CONST / GetDielectric();

  if (!uselist(&use, &nuse, cutoff)) return FALSE;

  /**** Allocate the memory for the array space ****/

  i = a_number();
  if ((buffer = Alloca(3 * i * sizeof(float), Routine)) == NULL)
    return FALSE;

  if ((vector = Alloca(i * sizeof(float), Routine)) == NULL) {
    free(buffer);
    return FALSE;
  }

  if ((atms = Alloca(i * sizeof(AMMP_ATOM *), Routine)) == NULL) {
    free(buffer);
    free(vector);
    return FALSE;
  }

  a1   = a_next(-1);
  a1   = a1 -> next;
  imax = a_number();
  used = 0;

  for(jj = 1; jj < imax; jj++, a1 = bp) {
    bp       = a1 -> next;
    inbuffer = 0;
    if (use[used] == a1) {
      used += 1;
    } else {
      aaerror("In uselist - must abort");
      return FALSE;
    }
    while(use[used] != a1) {
      atms[inbuffer++] = use[used];
      used += 1;
    } /* End of while */
    used += 1;

/* (*atms) now contains the list of atoms to be  done
*  there are inbuffer of them
*  of course inbuffer can be zero so we must check for that
*/

  if (inbuffer > 0) {
    for(i = 0; i < inbuffer; i++) {
      (*buffer)[3*i  ] = atms[i] -> x;
      (*buffer)[3*i+1] = atms[i] -> y;
      (*buffer)[3*i+2] = atms[i] -> z;
    } /* End of for (i) */

    if (lambda) {
      for(i = 0; i< inbuffer; i++) {
        (*buffer)[3*i  ] = atms[i] -> x + atms[i] -> dx * lambda;
  (*buffer)[3*i+1] = atms[i] -> y + atms[i] -> dy * lambda;
  (*buffer)[3*i+2] = atms[i] -> z + atms[i] -> dz * lambda;
      } /* End of for (i) */
    }
    xx = a1 -> x + lambda * a1 -> dx;
    yy = a1 -> y + lambda * a1 -> dy;
    zz = a1 -> z + lambda * a1 -> dz;

    /**** Now for the work ****/

    for(i = 0; i < inbuffer; i++) {
  xt = xx - (*buffer)[3*i];
  yt = yy - (*buffer)[3*i+1];
  zt = zz - (*buffer)[3*i+2];
  r = xt*xt+yt*yt+zt*zt;
  if( r < 2.) r = 2.;
  r0 = sqrt(r); r = r*r*r ;
/* the standard which follows is recursive */
/*   *V += Q_CONST *a1->q*a2->q/r0;
  *V -= a1->a*a2->a/r;
  *V += a1->b*a2->b/r/r;
*/
/* use debye screen e(-r0/rdebye) */
  (*vector)[i] = a1 -> q * atms[i] -> q / r0 * dielectric * exp(-r0 / rdebye)
         - a1 -> a * atms[i] -> a / r
         + a1 -> b * atms[i] -> b / r / r;
  }

      for(i = 0; i< inbuffer; i++) *V += (*vector)[i];
    } /* end of the inbuffer if check many lines ago */
  }

  free(atms  );
  free(buffer);
  free(vector);
  return TRUE;
}


/**** Calculate the non-bond force increment ****/

int AMMP_FASTCALL u_f_nonbon(float lambda)
{
  AMMP_ATOM     *a1;
  AMMP_ATOM     **use;
  AMMP_ATOM     *bp;
  AMMP_ATOM **  atms;
  float         r, r0, xt, yt, zt;
  float         cutoff;
  float         dielectric, rdebye;
  float         (*buffer)[], xx, yy, zz, k;
  float         (*vector)[];
  int           i, jj, nuse, used;
  int           imax, inbuffer;

  const char *  Routine = "u_f_nonbon()";

  /* nonbonded potentials
   * do a double loop starting from the first atom to the
   * last then from the second to the last, etc
   *
   * also check to avoid bonded and 1-3 bonded atoms
   */

  cutoff = GetCutoff();
  if ((cutoff > 1000.0f) || (cutoff == 0.0f)) return f_nonbon(lambda);

  rdebye     = cutoff * 0.5f;
  dielectric = Q_CONST / GetDielectric();

  if (!uselist(&use, &nuse, cutoff))return FALSE;

  /****  Allocate the memory for the array space ****/

  i = a_number();
  if ((buffer = Alloca(3 * i * sizeof(float), Routine)) == NULL)
    return FALSE;

  if ((vector = Alloca(3 * i * sizeof(float), Routine)) == NULL) {
    free(buffer);
    return FALSE;
  }

  if ((atms = Alloca(i * sizeof(AMMP_ATOM *), Routine)) == NULL) {
    free(buffer);
    free(vector);
    return FALSE;
  }

  a1   = a_next(-1);
  a1   = a1 -> next;
  imax = a_number();
  used = 0;
  for(jj = 1; jj < imax; jj++, a1 = bp) {
    bp       = a1 -> next;
    inbuffer = 0;
    if (use[used] == a1) {
      used += 1;
    } else {
      aaerror("In uselist - must abort");
      return FALSE;
    }
    while(use[used] != a1) {
      atms[inbuffer++] = use[used];
      used += 1;
    } /* End of while */
    used += 1;

    /* (*atms) now contains the list of atoms to be done
     *  there are inbuffer of them
     *  of course inbuffer can be zero so we must check for that
     */

    if (inbuffer > 0) {
      for(i = 0; i < inbuffer; i++) {
        (*buffer)[3 * i    ] = atms[i] -> x;
        (*buffer)[3 * i + 1] = atms[i] -> y;
        (*buffer)[3 * i + 2] = atms[i] -> z;
      } /* End of for (i) */

      if (lambda) {
        for(i = 0; i < inbuffer; i++) {
          (*buffer)[3 * i    ] = atms[i] -> x + atms[i] -> dx * lambda;
          (*buffer)[3 * i + 1] = atms[i] -> y + atms[i] -> dy * lambda;
    (*buffer)[3 * i + 2] = atms[i] -> z + atms[i] -> dz * lambda;
        } /* End of for (i) */
      }

      xx = a1 -> x + lambda * a1 -> dx;
      yy = a1 -> y + lambda * a1 -> dy;
      zz = a1 -> z + lambda * a1 -> dz;

      /**** Now for the work ****/

      for(i = 0; i < inbuffer; i++) {
        xt = xx - (*buffer)[3 * i    ];
        yt = yy - (*buffer)[3 * i + 1];
        zt = zz - (*buffer)[3 * i + 2];
        r  = xt * xt + yt * yt + zt * zt;

        /**** Watch for FP errors ****/

        if (r <= 1.0f) r = 1.0f;

        r0 = sqrt(r);
        xt = xt / r0;
        yt = yt / r0;
        zt = zt / r0;

        /* use debye screen e(-r0/rdebye) */
        /* d/dx(e(-r0/rdebye)/r0  = e(-r0/rdebye)*(-1/rdebye)/r0 + e(-r0/rdebye)/r) */
        k  = -a1 -> q * atms[i] -> q * dielectric * exp(-r0 / rdebye) *
       (1.0 / (rdebye * r0) + 1.0 / r) ;
        r  = r * r * r;
        k += a1 -> a * atms[i] -> a / r / r0 * 6.0f;
        k -= a1 -> b * atms[i] -> b / r / r / r0 * 12.0f;
        (*vector)[3 * i    ] = xt * k;
        (*vector)[3 * i + 1] = yt * k;
        (*vector)[3 * i + 2] = zt * k;
      } /* End of for (i) */

      for(i = 0; i < inbuffer; i++) {
        a1 -> fx -= (*vector)[3 * i    ];
        a1 -> fy -= (*vector)[3 * i + 1];
        a1 -> fz -= (*vector)[3 * i + 2];
        atms[i] -> fx += (*vector)[3 * i    ];
        atms[i] -> fy += (*vector)[3 * i + 1];
        atms[i] -> fz += (*vector)[3 * i + 2];
      } /* End of for (i) */
    }
  } /* End of for (jj) */

  free(atms );
  free(buffer);
  free(vector);

  return TRUE;
}


/* uselist()
*  returns a pointer to an array of AMMP_ATOM structure pointers
*  these are encoded as
*  a,bcdsfg,a where a is the outer most atom in the N^2 -N tree.
*
* not over brilliant
*
*  checks for change in atom number
*  other wise redoes the list every (nbstep (default= 10)) steps
*  makes a list with total storage (nbdeep (default = 20)) time a_number
*/

static int AMMP_FASTCALL uselist(AMMP_ATOM ***List, int *Size, float Cutoff)
{
  AMMP_ATOM             *a1,*a2,*ap,*bp;
  float                 lcutoff;
  float                 x,y,z,r,rcut;
  int                   i,j,k,max;

  /**** Check on wether to redo it or not ****/

  i = a_number();
  j = GetNbstep();

  if ((i == uselist_oldatomnumber) && (uselist_since < j) &&
      (Cutoff == uselist_oldcutoff)) {
    *List = uselist_local;
    *Size = uselist_lsize;
    ++uselist_since;
    return TRUE;
  }

  /**** free and Alloca are used because nbdeep may change ****/

RESET:

  if (uselist_oldatomnumber > 0) free(uselist_local);
  uselist_oldcutoff     = Cutoff;
  lcutoff               = -Cutoff;
  uselist_since         = 0;
  uselist_oldatomnumber = i;
  j                     = GetNbdeep();

  max = i * j;

  if ((uselist_local = Alloca(max * sizeof(AMMP_ATOM *), "uselist()")) == NULL)
    return FALSE;

  /**** Now have the uselist allocated ****/

  *List = uselist_local;
  *Size = 0;
  rcut  = Cutoff * Cutoff;
  a1    = a_next(-1);
  a1    = a1 -> next;

  for(i = 1; i < uselist_oldatomnumber; i++, a1 = ap) {
    ap = a1 -> next;
    uselist_local[*Size] = a1;
    ++*Size;
    if (*Size == max) {
      if (SetSize(*Size)) goto RESET;
      else return FALSE;
    }
    a2 = a_next(-1);
    for(j = 0; j < i; j++, a2 = bp) {
      for(k = 0; k < a1 -> dontuse; k++) {
        if(a2 == a1 -> excluded[k]) goto SKIP;
      } /* End of for (k) */
      if ((a2 -> x - a1 -> x) > Cutoff ) goto SKIP;
      if ((a2 -> x - a1 -> x) < lcutoff) goto SKIP;
      if ((a2 -> y - a1 -> y) > Cutoff ) goto SKIP;
      if ((a2 -> y - a1 -> y) < lcutoff) goto SKIP;
      if ((a2 -> z - a1 -> z) > Cutoff ) goto SKIP;
      if ((a2 -> z - a1 -> z) < lcutoff) goto SKIP;

      /**** Now calculate the radius ****/

      x = a2 -> x - a1 -> x;
      y = a2 -> y - a1 -> y;
      z = a2 -> z - a1 -> z;
      r = x * x + y * y + z * z;
      if (r > rcut) goto SKIP;

      uselist_local[*Size] = a2;
      ++*Size;
      if (*Size == max) {
        if (SetSize(*Size)) goto RESET;
        else return FALSE;
      }

SKIP:

      bp = a_next(1);
    } /* End of for (j) */

    uselist_local[*Size] = a1;
    ++*Size;
    if (*Size == max) {
      if (SetSize(*Size)) goto RESET;
      else return FALSE;
    }
    uselist_lsize = *Size;
  } /* End of for (i) */

  return TRUE;
}


/**** Set the list size ***/

static int AMMP_FASTCALL SetSize(int Size)
{
  int           i = a_number();
  int           j = GetNbdeep();

  aaerror("Please increase nbdeep (seti nbdeep (>20);)");
  if (j == (i + 2)) {
    aaerror("Terrible error in uselist, too many interactions");
    return FALSE;
  }
  j = 2 * j;
  if (j > (i + 2)) j = i + 2;
  set_i_variable("nbdeep", j);

  return TRUE;
}

