
/*************************************************
****             AMMP - Utilities             ****
**** Copyright 2006-2012, Alessandro Pedretti ****
*************************************************/


#ifdef WIN32
#  include <windows.h>
#else
#  include <unistd.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <limits.h>
#include <string.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "ammp.h"
#include "base64.h"
#include "inline.h"
#include "mthread.h"

/**** Default values ****/

#define  AMMP_DEF_MXCUT                 six
#define  AMMP_DEF_MXPERITER             1000
#define  AMMP_DEF_NBDEEP                20
#define  AMMP_DEF_NBSTEP                10
#define  AMMP_DEF_NUPDAT                0
#define  AMMP_DEF_RDEBYE                0.25f
#define  AMMP_DEF_SUPDAT                4

/**** Global variables ****/

#ifdef WIN32
extern HANDLE           ApiStdOut;
#endif

/**** dlmalloc ****/

#ifdef AMMP_USE_DLMALLOC
#  define  USE_DL_PREFIX
#  define  MALLOC_ALIGNMENT             8
#  include "malloc.c"
#endif


/**** Local variables ****/

static VARIABLE *       VarPtr_B64enc;
static VARIABLE *       VarPtr_Cutoff;
static VARIABLE *       VarPtr_Dddr;
static VARIABLE *       VarPtr_Dend;
static VARIABLE *       VarPtr_Dielectric;
static VARIABLE *       VarPtr_Lambda;
static VARIABLE *       VarPtr_Mmbox;
static VARIABLE *       VarPtr_MxPerIter;
static VARIABLE *       VarPtr_Mxcut;
static VARIABLE *       VarPtr_Mxdq;
static VARIABLE *       VarPtr_Nbdeep;
static VARIABLE *       VarPtr_Nbstep;
static VARIABLE *       VarPtr_Nostep;
static VARIABLE *       VarPtr_Nupdat;
static VARIABLE *       VarPtr_Rdebye;
static VARIABLE *       VarPtr_Supdat;
static VARIABLE *       VarPtr_Threads;
static VARIABLE *       VarPtr_Trace;

/**** Base64 tables ****/

const char cb64[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";


/**** Allocate the memory ****/

void * AMMP_FASTCALL Alloca(unsigned int Size, const char *Routine)
{
  void *        Mem;

  if ((Mem = (void *)malloc(Size)) == NULL) {
    if (Routine)
      aaerror("Can't allocate the memory in %s", Routine);
    else
      aaerror("Can't allocate the memory");
  }

  return Mem;
}


/**** Allocate the memory ****/

void * AMMP_FASTCALL AllocaC(unsigned int Size, const char *Routine)
{
  void *        Mem = Alloca(Size, Routine);

  if (Mem) memset(Mem, 0, Size);

  return Mem;
}


/**** Ceil (float version) ****/

float AMMP_FASTCALL CeilF(float x)
{
  register int  r = x;

  if (r < 0) return r;
  return (r + ((r < x) ? 1 : 0));
}


/**** Compute the total charge ****/

float AMMP_FASTCALL Charge(void)
{
  float                 TotCharge  = 0.0f;
  AMMP_ATOM *           Atm        = a_next(-1);

  if (!Atm) return 0.0f;

  while(Atm) {
    TotCharge += Atm -> q;
    if (Atm -> next == Atm) break;
    Atm = Atm -> next;
  } /* End of while */

  return TotCharge;
}


/**** Check atoms ****/

int AMMP_FASTCALL CheckAtoms(void)
{
  if (a_number()) return TRUE;

  aaerror("No atoms defined - nothing to calculate");
  return FALSE;
}


/**** Check the machine endian ****/

int AMMP_FASTCALL CheckEndian(void)
{
  int           i = 1;
  char *        p = (char *)&i;

  if (*p == 1) return AMMP_LITTLE_ENDIAN;

  return AMMP_BIG_ENDIAN;
}


/**** Set the default ****/

void AMMP_FASTCALL Default(void)
{
  int           k;
  VARIABLE *    V;

  if ((V = set_f_variable("charge", zero)) != NULL) {
    V -> GetVal.f = Charge;
    V -> flags    = AMMP_VAR_FLAG_READONLY;
  }

  if ((V = set_i_variable("hend", AmmpEndian)) != NULL) {
    V -> flags    = AMMP_VAR_FLAG_READONLY;
  }

  if ((V = set_f_variable("l2f", zero)) != NULL) {
    V -> GetVal.f = a_l2_f;
    V -> flags    = AMMP_VAR_FLAG_READONLY;
  }

  if ((V = set_f_variable("lmaxf", zero)) != NULL) {
    V -> GetVal.f = a_max_f;
    V -> flags    = AMMP_VAR_FLAG_READONLY;
  }

  if ((V = set_i_variable("numatm", 0)) != NULL) {
    V -> GetVal.i = a_number;
    V -> flags    = AMMP_VAR_FLAG_READONLY;
  }

  if ((V = set_f_variable("pi", PI)) != NULL) {
    V -> flags    = AMMP_VAR_FLAG_READONLY;
  }

  VarPtr_B64enc    = set_i_variable("b64enc"    , 0                 );
  VarPtr_Dend      = set_i_variable("dend"      , AmmpEndian        );
  VarPtr_Mxdq      = set_f_variable("mxdq"      , 0.05f             );
  VarPtr_Mxcut     = set_f_variable("mxcut"     , AMMP_DEF_MXCUT    );
  VarPtr_MxPerIter = set_f_variable("mxperiter" , AMMP_DEF_MXPERITER);
  VarPtr_Nostep    = set_i_variable("nostep"    , 1                 );
  VarPtr_Nupdat    = set_i_variable("nupdat"    , AMMP_DEF_NUPDAT   );
  VarPtr_Supdat    = set_i_variable("supdat"    , AMMP_DEF_SUPDAT   );

  /**** Multithreading ****/

  k = MCL_GetCPUs();
  set_i_variable("cpus"  , k);
  set_flags_variable("cpus", AMMP_VAR_FLAG_READONLY);

  VarPtr_Threads    = set_i_variable("threads", k);
  VarPtr_Cutoff     = NULL;
  VarPtr_Dddr       = NULL;
  VarPtr_Dielectric = NULL;
  VarPtr_Lambda     = NULL;
  VarPtr_Mmbox      = NULL;
  VarPtr_Rdebye     = NULL;
  VarPtr_Trace      = NULL;

  /**** Potential ****/

  forces[0]     = f_bond;
  forces[1]     = f_angle;
  forces[2]     = u_f_nonbon;
  forces[3]     = f_torsion;
  forces[4]     = f_hybrid;
  potentials[0] = v_bond;
  potentials[1] = v_angle;
  potentials[2] = u_v_nonbon;
  potentials[3] = v_torsion;
  potentials[4] = v_hybrid;
  nused         = 5;
}


/**** Get the cutoff ****/

float AMMP_FASTCALL GetCutoff(void)
{
  float         Cutoff;

  if (!VarPtr_Cutoff) {
    VarPtr_Cutoff = match_variable("cutoff");
    if (!VarPtr_Cutoff) return 1.0e10f;
  }

  Cutoff = VarPtr_Cutoff -> value.f;
  if (Cutoff < one) return 1.0e10f;

  return Cutoff;
}


/**** Get the dump endian ****/

int AMMP_FASTCALL GetDend(void)
{
  return VarPtr_Dend -> value.i;
}


/**** Get the dddr ****/

float AMMP_FASTCALL GetDddr(void)
{
  float         Dddr;

  if (!VarPtr_Dddr) {
    VarPtr_Dddr = match_variable("dddr");
    if (!VarPtr_Dddr) return one;
  }

  Dddr = VarPtr_Dddr -> value.f;
  if (Dddr < 1.e-7) return one;

  return Dddr;
}


/**** Get the dielectric ****/

float AMMP_FASTCALL GetDielectric(void)
{
  float         Dielectric;

  if (!VarPtr_Dielectric) {
    VarPtr_Dielectric = match_variable("dielec");
    if (!VarPtr_Dielectric) return one;
  }

  Dielectric = VarPtr_Dielectric -> value.f;
  if (Dielectric < one) return one;

  return Dielectric;
}


/**** Get Lambda ****/

float AMMP_FASTCALL GetLambda(void)
{
  float *       Lambda;

  if (!VarPtr_Lambda) {
    VarPtr_Lambda = match_variable("lambda");
    if (!VarPtr_Lambda) return zero;
  }
  Lambda = &VarPtr_Lambda -> value.f;

  if (*Lambda > one ) *Lambda = one;
  if (*Lambda < zero) *Lambda = zero;

  return *Lambda;
}


/**** Get Nbdeep ****/

int AMMP_FASTCALL GetNbdeep(void)
{
  int *         Nbdeep;

  if (!VarPtr_Nbdeep) {
    VarPtr_Nbdeep = match_variable("nbdeep");
    if (!VarPtr_Nbdeep) return AMMP_DEF_NBDEEP;
  }
  Nbdeep = &VarPtr_Nbdeep -> value.i;
  if (*Nbdeep <= 0) *Nbdeep = AMMP_DEF_NBDEEP;

  return *Nbdeep;
}


/**** Get Nbstep ****/

int AMMP_FASTCALL GetNbstep(void)
{
  int *         Nbstep;

  if (!VarPtr_Nbstep) {
    VarPtr_Nbstep = match_variable("nbstep");
    if (!VarPtr_Nbstep) return AMMP_DEF_NBSTEP;
  }
  Nbstep = &VarPtr_Nbstep -> value.i;
  if (*Nbstep <= 0) *Nbstep = AMMP_DEF_NBSTEP;

  return *Nbstep;
}


/**** Get  Mmbox ****/

float AMMP_FASTCALL GetMmbox(void)
{
  if (!VarPtr_Mmbox) {
    VarPtr_Mmbox = match_variable("mmbox");
    if (!VarPtr_Mmbox) return 0.0f;
  }

  return VarPtr_Mmbox -> value.f;
}


/**** Get the mxcut ****/

float AMMP_FASTCALL GetMxcut(void)
{
  float         Mxcut;

  if (!VarPtr_Mxcut) {
    VarPtr_Mxcut = match_variable("mxcut");
    if (!VarPtr_Mxcut) return AMMP_DEF_MXCUT;
  }

  Mxcut = VarPtr_Mxcut -> value.f;
  if (Mxcut < zero) return AMMP_DEF_MXCUT;

  return Mxcut;
}


/**** Get the mxdq ****/

float AMMP_FASTCALL GetMxdq(void)
{
  if (!VarPtr_Mxdq) {
    VarPtr_Mxdq = match_variable("mxdq");
    if (!VarPtr_Mxdq) return zero;
  }

  return VarPtr_Mxdq -> value.f;
}


/**** Get the mxdq square ****/

float AMMP_FASTCALL GetMxdq2(void)
{
  float         Mxdq;

  if (!VarPtr_Mxdq) {
    VarPtr_Mxdq = match_variable("mxdq");
    if (!VarPtr_Mxdq) return zero;
  }

  Mxdq = VarPtr_Mxdq -> value.f;
  if (Mxdq > zero) return (Mxdq * Mxdq);

  return Mxdq;
}


/**** Get the mxperiter ****/

float AMMP_FASTCALL GetMxPerIter(void)
{
  if (VarPtr_MxPerIter -> value.f < 1) VarPtr_MxPerIter -> value.f = 1;

  return VarPtr_MxPerIter -> value.f;
}


/**** Get the nostep ****/

int AMMP_FASTCALL GetNostep(void)
{
  if (!VarPtr_Nostep) {
    VarPtr_Nostep = match_variable("nostep");
    if (!VarPtr_Nostep) return 0;
  }

  return VarPtr_Nupdat -> value.i;
}


/**** Get the nupdat ****/

int AMMP_FASTCALL GetNupdat(void)
{
  float         Nupdat;

  if (!VarPtr_Nupdat) {
    VarPtr_Nupdat = match_variable("nupdat");
    if (!VarPtr_Nupdat) return AMMP_DEF_NUPDAT;
  }

  Nupdat = VarPtr_Nupdat -> value.i;
  if (Nupdat < 0) {
    VarPtr_Nupdat -> value.i = AMMP_DEF_NUPDAT;
    return AMMP_DEF_NUPDAT;
  }

  return Nupdat;
}


/**** Get the rdebye ****/

float AMMP_FASTCALL GetRdebye(void)
{
  float         Rdebye;

  if (!VarPtr_Rdebye) {
    VarPtr_Rdebye = match_variable("rdebye");
    if (!VarPtr_Rdebye) return AMMP_DEF_RDEBYE;
  }

  Rdebye = VarPtr_Rdebye -> value.f;
  if (Rdebye >= one) {
    VarPtr_Supdat -> value.i = AMMP_DEF_RDEBYE;
    return AMMP_DEF_RDEBYE;
  }

  return Rdebye;
}


/**** Get the supdat ****/

int AMMP_FASTCALL GetSupdat(void)
{
  float         Supdat;

  if (!VarPtr_Supdat) {
    VarPtr_Supdat = match_variable("supdat");
    if (!VarPtr_Supdat) return AMMP_DEF_SUPDAT;
  }

  Supdat = VarPtr_Supdat -> value.i;
  if (Supdat < 0) {
    VarPtr_Supdat -> value.i = AMMP_DEF_SUPDAT;
    return AMMP_DEF_SUPDAT;
  }

  return Supdat;
}


/**** Get the number of threads ****/

int AMMP_FASTCALL GetThreads(void)
{
  return (VarPtr_Threads -> value.i < 1) ? 1: VarPtr_Threads -> value.i;
}


/**** Get the trace ****/

float AMMP_FASTCALL GetTrace(void)
{
  if (!VarPtr_Trace) {
    VarPtr_Trace = match_variable("trace");
    if (!VarPtr_Trace) return zero;
  }

  return VarPtr_Trace -> value.f;
}


/**** Print a string ****/

void AMMP_FASTCALL PrintStr(FILE *FH, char *Str)
{
  char                  NumStr[4];
  register const char * Cod;
  int                   c;

  static const char     Codes[] = { 'a' , 0x07, 'b' , 0x08, 'f' , 0x0c, 'n', 0x0a,
                                    'r' , 0x0d, 't' , 0x09, 'v' , 0x0b, '\'', 0x27,
                                    '\"', 0x22, '\?', 0x3f, '\\', 0x5c,
                                     0 , 0x00
                                  };
  while(*Str) {
    if (*Str == '\\') {
      ++Str;
      if (*Str) {
        for(Cod = Codes; *Cod; Cod += 2) {
          if (*Cod == *Str) {
            fputc(Cod[1], FH);
            goto ExitLoop;
          }
        } /* End of for (Cod) */

        /**** Hex code ****/

        if ((*Str == 'x') || (*Str == 'X')) {
          if (isxdigit(Str[1])) {
            NumStr[0] = *++Str;
            if (isxdigit(Str[1])) {
              NumStr[1] = *++Str;
              NumStr[2] = 0;
            } else NumStr[1] = 0;
            sscanf(NumStr, "%x", &c);
            fputc(c, FH);
          }
        } else {

          /**** Octal code ****/

          c = 0;
          if ((isdigit(*Str)) && (*Str < '8')) {
            NumStr[0] = *Str;
            if ((isdigit(Str[1])) && (Str[1] < '8')) {
              NumStr[1] = *++Str;
              if ((isdigit(Str[1])) && (Str[1] < '8')) {
                NumStr[2] = *++Str;
                NumStr[3] = 0;
              } else NumStr[2] = 0;
            } else NumStr[1] = 0;
            sscanf(NumStr, "%o", &c);
            fputc(c, FH);
          }
        }
      }
    } else fputc(*Str, FH);
ExitLoop:
    ++Str;
  } /* End of while */
}


/**** Reallocate the memory ****/

void * AMMP_FASTCALL ReAlloca(void *OldMem, unsigned int Size, const char *Routine)
{
  void *        Mem;

  if ((Mem = (void *)realloc(OldMem, Size)) == NULL) {
    if (Routine)
      aaerror("Can't reallocate the memory in %s", Routine);
    else
      aaerror("Can't reallocate the memory");
  }

  return Mem;
}


/**** Reset all resources ****/

void AMMP_FASTCALL Reset(void)
{
  ListFree((void **)&abc_first      , (void **)&abc_last      );
  ListFree((void **)&angle_first    , (void **)&angle_last    );
  ListFree((void **)&av5_first      , (void **)&av5_last      );
  ListFree((void **)&bond_first     , (void **)&bond_last     );
  ListFree((void **)&first_datatable, (void **)&last_datatable);
  ListFree((void **)&hybrid_first   , (void **)&hybrid_last   );
  ListFree((void **)&morse_first    , (void **)&morse_last    );
  ListFree((void **)&noel_first     , (void **)&noel_last     );
  ListFree((void **)&restrain_first , (void **)&restrain_last );
  ListFree((void **)&STEP_first     , (void **)&STEP_last     );
  ListFree((void **)&SWARM_first    , (void **)&SWARM_last    );
  ListFree((void **)&first_tbond    , (void **)&first_tbond   );
  ListFree((void **)&tether_first   , (void **)&tether_last   );
  ListFree((void **)&torsion_first  , (void **)&torsion_last  );
  ListFree((void **)&ttarget_first  , (void **)&ttarget_last  );
  ListFree((void **)&variableFIRST  , (void **)&variableLAST  );

  ResetAtom();
  ResetClone();
  ResetDebye();
  ResetMom();
  ResetN3();
  ResetNoel();
  ResetOrbit();
  ResetRandom();
  ResetReact();
  ResetShadow();
  ResetTable();
  TgroupReset();
  ResetUnonbon();
  ResetVnonbon();

  atomNUMBER  = 0;
  atomUPDATE  = FALSE;
  in_mom_list = 0;
}


/**** Safe free ****/

void AMMP_FASTCALL SafeFree(void *Mem)
{
  if (Mem) free(Mem);
}


/**** Set the mxdq cvalue ****/

void AMMP_FASTCALL SetMxdq(float Val)
{
  if (!VarPtr_Mxdq) {
    set_f_variable("mxdq" , Val);
    return;
  }

  VarPtr_Mxdq -> value.f = Val;
}


#ifdef AMMP_SENDTOGFX

/**** Dump the charges ****/

int AMMP_FASTCALL BinDumpCharge(void)
{
  unsigned char         EncBuf[16];
#ifdef WIN32
  DWORD                 Nbytes;
#endif
  float                 Chg;
  int                   i;
  union {
    float               F[3];
    char                UC[3 * sizeof(float)];
  } Charge;

  AMMP_ATOM *           Atm        = a_next(-1);
  int                   SwapEndian = (GetDend() != AmmpEndian);

  if (!Atm) return FALSE;
  if (VarPtr_B64enc -> value.i) {
    printf("#BASE64CHARGE\n");
    i = 0;
    while(Atm) {
      if (SwapEndian) Charge.F[i] = AmmpSwapf(Atm -> q);
      else Charge.F[i] = Atm -> q;
      if (i == 2) {
        Base64EncodeBlock(Charge.UC    , EncBuf     , 3);
        Base64EncodeBlock(Charge.UC + 3, EncBuf +  4, 3);
        Base64EncodeBlock(Charge.UC + 6, EncBuf +  8, 3);
        Base64EncodeBlock(Charge.UC + 9, EncBuf + 12, 3);
#ifdef WIN32
        WriteFile(ApiStdOut, EncBuf, 16, &Nbytes, 0);
#else
        write(STDOUT_FILENO, EncBuf, 16);
#endif
        i = 0;
      } else ++i;
      if (Atm -> next == Atm) break;
      Atm = Atm -> next;
    } /* End of while */
    if (i == 1) {
      Charge.F[1] = 0.0f;
      Base64EncodeBlock(Charge.UC    , EncBuf     , 3);
      Base64EncodeBlock(Charge.UC + 3, EncBuf +  4, 1);
#ifdef WIN32
      WriteFile(ApiStdOut, EncBuf, 8, &Nbytes, 0);
#else
      write(STDOUT_FILENO, EncBuf, 8);
#endif
    } else if (i == 2) {
      Charge.F[2] = 0.0f;
      Base64EncodeBlock(Charge.UC    , EncBuf     , 3);
      Base64EncodeBlock(Charge.UC + 3, EncBuf +  4, 3);
      Base64EncodeBlock(Charge.UC + 6, EncBuf +  8, 3);
#ifdef WIN32
      WriteFile(ApiStdOut, EncBuf, 12, &Nbytes, 0);
#else
      write(STDOUT_FILENO, EncBuf, 12);
#endif
    }
  } else {
    printf("#BINCHARGE\n");
    while(Atm) {
      if (SwapEndian) Chg = AmmpSwapf(Atm -> q);
      else Chg = Atm -> q;

#ifdef WIN32
      WriteFile(ApiStdOut, &Chg, sizeof(float), &Nbytes, 0);
#else
      write(STDOUT_FILENO, &Chg, sizeof(float));
#endif
      if (Atm -> next == Atm) break;
      Atm = Atm -> next;
    } /* End of while */
  }
  printf("#END\n");

  return TRUE;
}


/**** Dump the atoms and the energy ****/

int AMMP_FASTCALL DumpAtoms(float Ene)
{
  if (send_all_atoms()) return DumpVal("ENE", &Ene, AMMP_VAR_TYPE_FLOAT);

  return FALSE;
}


/**** Dump a variable ****/

int AMMP_FASTCALL DumpVal(const char *ValName, void *Val, int ValType)
{
  const char *          ValTypeStr = "IF";

  printf("#VAL %s %c ", ValName, ValTypeStr[ValType]);
  if (ValType == AMMP_VAR_TYPE_INTEGER) printf("%d\n", *(int *)Val);
  else printf("%f\n", *(float *)Val);

  return TRUE;
}


/**** Send the atom coordinates to VEGA ZZ ****/

int AMMP_FASTCALL send_all_atoms(void)
{
  unsigned char         EncBuf[16];
#ifdef WIN32
  DWORD                 Nbytes;
#endif
  union {
    float               Xyz[3];
    char                UC[3 * sizeof(float)];
  } Coord;

  AMMP_ATOM *           Atm        = a_next(-1);
  int                   SwapEndian = (GetDend() != AmmpEndian);

  if (!Atm) return FALSE;
  if (VarPtr_B64enc -> value.i) {

    /**** Base64 dump ****/

    printf("#BASE64COORD\n");
    while(Atm) {
#ifdef WIN32
      ZeroMemory(EncBuf, 16);
#else
      memset(EncBuf, 0, 16);
#endif
      if (SwapEndian) {
        Coord.Xyz[0] = AmmpSwapf(Atm -> x);
        Coord.Xyz[1] = AmmpSwapf(Atm -> y);
        Coord.Xyz[2] = AmmpSwapf(Atm -> z);
      } else {
        Coord.Xyz[0] = Atm -> x;
        Coord.Xyz[1] = Atm -> y;
        Coord.Xyz[2] = Atm -> z;
      }
      Base64EncodeBlock(Coord.UC    , EncBuf     , 3);
      Base64EncodeBlock(Coord.UC + 3, EncBuf +  4, 3);
      Base64EncodeBlock(Coord.UC + 6, EncBuf +  8, 3);
      Base64EncodeBlock(Coord.UC + 9, EncBuf + 12, 3);
#ifdef WIN32
      WriteFile(ApiStdOut, EncBuf, 16, &Nbytes, 0);
#else
      write(STDOUT_FILENO, EncBuf, 16);
#endif

      if (Atm -> next == Atm) break;
      Atm = Atm -> next;
    } /* End of while */
  } else {

    /**** Binary dump ****/

    printf("#BINCOORD\n");
    while(Atm) {
      if (SwapEndian) {
        Coord.Xyz[0] = AmmpSwapf(Atm -> x);
        Coord.Xyz[1] = AmmpSwapf(Atm -> y);
        Coord.Xyz[2] = AmmpSwapf(Atm -> z);
#ifdef WIN32
        WriteFile(ApiStdOut, Coord.UC, sizeof(float) * 3, &Nbytes, 0);
#else
        write(STDOUT_FILENO, Coord.UC, sizeof(float) * 3);
#endif
      } else {
#ifdef WIN32
        WriteFile(ApiStdOut, &Atm -> x, sizeof(float) * 3, &Nbytes, 0);
#else
        write(STDOUT_FILENO, &Atm -> x, sizeof(float) * 3);
#endif
      }
      if (Atm -> next == Atm) break;
      Atm = Atm -> next;
    } /* End of while */
  }
  printf("#END\n");

  return TRUE;
}

#endif



