/* variable.c
*
* variable storage and retreival routines for AMMP
*
* using scope rules for structuring
*
*
*  variables are stored in linked list form
*
*   get_f_variable( char *name, float *fvalue )
*   get_i_variable( char *name,  int *ivalue )
*	returns variable who matches name (all lower case )
*   set_f_variable( char *name, float fvalue )
*   set_i_variable( char *name, int ivalue )
*	sets variable who matches name (all lower case )
*   match_variable( char *name ) returns pointer to name if there NULL if not
*   dump_variable(FILE *output  )
*	 dumps variables to  file 
*/
/*
*  copyright 1992 Robert W. Harrison
*  
*  This notice may not be removed
*  This program may be copied for scientific use
*  It may not be sold for profit without explicit
*  permission of the author(s) who retain any
*  commercial rights including the right to modify 
*  this notice
*/

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "ammp.h"

/**** Local proptotypes ****/

static VARIABLE *       AMMP_FASTCALL alloc_variable(char *name);
static int              AMMP_FASTCALL chk_set_variable(VARIABLE *var);


/**** Alloc a new variable ****/

static VARIABLE * AMMP_FASTCALL alloc_variable(register char *name)
{
  register int                  i;
  register VARIABLE *           new;

  if ((new = match_variable(name)) == NULL) {
    if ((new = (VARIABLE *)Alloca(sizeof(VARIABLE), "alloc_variable()")) == NULL)
      return FALSE;

    new -> flags = AMMP_VAR_FLAG_NONE;
    new -> next  = NULL;
    if (!variableFIRST) {
      variableFIRST = new;
      variableLAST  = new;
    } else {
      variableLAST -> next = new;
      variableLAST = new;
    }

    for(i = 0; i < NUM_TOT; i++) {
      new -> name[i] = *name;
      if (*name == '\0') break;
      name++;
    } /* End of for (i) */
  }

  return new;
}


/**** Check if a variable is writable ****/

static int AMMP_FASTCALL chk_set_variable(VARIABLE *var)
{
  if (var -> flags & AMMP_VAR_FLAG_READONLY) {
    aaerror("Read-only variable");
    return FALSE;
  }

  return TRUE;
}


/**** Dump the variable ****/

void AMMP_FASTCALL dump_variable(FILE *where)
{
  register VARIABLE *           vp;

  for(vp = variableFIRST; vp; vp = vp -> next) {
    if (vp -> type == AMMP_VAR_TYPE_INTEGER)
      fprintf(where, "seti %s %d;\n", vp -> name, vp->value.i);
    else
      fprintf(where, "setf %s %f;\n", vp -> name, vp -> value.f);
  } /* End of for (vp) */
}


/* function match_variable( char *name)
* tries to find the variable in the list returns NULL if not there *
*/

VARIABLE * AMMP_FASTCALL match_variable(register char *name)
{

  register VARIABLE *           vp;

  for(vp = variableFIRST; vp; vp = vp -> next)
    if (!strcmp(vp -> name, name)) return vp;

  return NULL;
}



/* function get_f_variable(char * name )
*
* returns 0. or the float referenced or converted by name
*/

float AMMP_FASTCALL get_f_variable(char *name)
{
  VARIABLE *    vp = match_variable(name);

  if (vp == NULL) return 0.0f;
  if (vp -> type == AMMP_VAR_TYPE_FLOAT) {
    if (vp -> GetVal.f) vp -> value.f = vp -> GetVal.f();
    return vp -> value.f;
  }
  if (vp -> GetVal.i) return vp -> value.i = vp -> GetVal.i();

  return (float)vp -> value.i;
}


/* function get_i_variable(char * name )
*
* returns 0 or the integer referenced or converted by name
*/

int AMMP_FASTCALL get_i_variable(char *name)
{
  VARIABLE *    vp = match_variable(name);

  if (vp == NULL) return 0;
  if (vp -> type == AMMP_VAR_TYPE_INTEGER) {
    if (vp -> GetVal.i) vp -> value.i = vp -> GetVal.i();
    return vp -> value.i;
  }
  if (vp -> GetVal.f) vp -> value.f = vp -> GetVal.f();

  return (int)vp -> value.f;
}


/**** Obtain float and integer value at the same time ****/

VARIABLE * AMMP_FASTCALL get_fi_variable(char *name, float *f, int *i)
{
  VARIABLE *    vp = match_variable(name);

  if (vp == NULL) {
    *f = 0.0f;
    *i = 0.0f;
    return NULL;
  }

  /**** Float ****/

  if (vp -> type == AMMP_VAR_TYPE_FLOAT) {
    if (vp -> GetVal.f) vp -> value.f = vp -> GetVal.f();
    *f = vp -> value.f;
    *i = (int)vp -> value.f;
    return vp;
  }

  /**** Integer ****/

  if (vp -> GetVal.i) vp -> value.i = vp -> GetVal.i();

  *f = (float)vp -> value.i;
  *i = vp -> value.i;

  return vp;
}



/* function set_f_variable( char *name, float f)
*
*   allocates storage if needed
*   atempts to match name and update, otherwise adds to the list
*/

VARIABLE * AMMP_FASTCALL set_f_variable(char *name, float f)
{
  VARIABLE *            new;

  if (((new = alloc_variable(name)) == NULL) ||
      (!chk_set_variable(new)))
    return NULL;

  new -> value.f  = f;
  new -> GetVal.f = NULL;
  new -> type     = AMMP_VAR_TYPE_FLOAT;

  return new;
}


/**** Set the variable flags ****/

int AMMP_FASTCALL set_flags_variable(char *name, int flags)
{
  VARIABLE *    vp = match_variable(name);

  if (vp == NULL) return FALSE;
  vp -> flags = flags;

  return TRUE;
}


/* function set_i_variable( char *name, int f)
*
*   allocates storage if needed
*   atempts to match name and update, otherwise adds to the list
*/

VARIABLE * AMMP_FASTCALL set_i_variable(char *name, int f)
{
  VARIABLE *            new;

  if (((new = alloc_variable(name)) == NULL) ||
      (!chk_set_variable(new)))
    return NULL;

  new -> value.i  = f;
  new -> GetVal.i = NULL;
  new -> type     = AMMP_VAR_TYPE_INTEGER;

  return new;
}


