
/*************************************************
****        AMMP - Non-bond potential         ****
**** Copyright 1992-2012, Robert W. Harrison  ****
****    VEGA edition by Alessandro Pedretti   ****
*************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  include <fastmath.h>
#else
#  include <math.h>
#endif

#include "ammp.h"

/**** Global variables ****/

extern AMMP_ATOM **     NbAtomAll;
extern int              NbAtomAllNum;
extern int *            NbIndexes;
extern float *          NbVector;

/**** Local variables ****/

static float     fv_update_nonbon_dielecold = -1.0f;


/**** Reset local variables ****/

void AMMP_FASTCALL ResetVnonbon(void)
{
  NonBondFreeTable();
  fv_update_nonbon_dielecold = -1.0f;
}


/**** Sum the potential for a range of atoms ****/

int AMMP_FASTCALL a_nonbon(float *V, float lambda, int ilow, int ihigh, FILE *op)
{
  AMMP_ATOM     *a1,*a2;
  float         r, r0, xt, yt, zt;
  float         lcutoff, cutoff;
  float         dielectric, ve, va, vh;
  float         vel, val, vhl;
  float         vtint, vtout, vtt;
  int           i, ii;


  cutoff     = GetCutoff();
  dielectric =  Q_CONST / GetDielectric();
  lcutoff    = -cutoff;
  vtint      = zero;
  vtout      = zero;
  vtt        = zero;

  for(ii = ilow; ii <= ihigh; ii++) {
    a1 = a_m_serial(ii);
    if (a1 == NULL) goto NOTANATOM;
    ve  = zero;
    va  = zero;
    vh  = zero;
    vel = zero;
    val = zero;
    vhl = zero;
    a2  = a_next(-1);
    while(a2 -> next) {
      for(i = 0; i < a1 -> dontuse; i++)
        if (a2 == a1 -> excluded[i]) goto SKIP;

      /**** Non-bonded are only used when the atoms arent bonded ****/

      if (lambda) {
  xt = a1 -> x - a2 -> x + lambda *(a1 -> dx - a2 -> dx);
  if ((xt > cutoff) || (xt < lcutoff)) goto SKIP;
  yt = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
  if ((yt > cutoff) || (yt < lcutoff)) goto SKIP;
  zt = a1 -> z - a2 -> z + lambda * (a1 -> dz - a2 -> dz);
  if ((zt > cutoff) || (zt < lcutoff)) goto SKIP;
      } else {
        xt = a1 -> x - a2 -> x;
  if ((xt > cutoff) || (xt < lcutoff)) goto SKIP;
  yt = a1 -> y - a2 -> y;
  if ((yt > cutoff) || (yt < lcutoff)) goto SKIP;
  zt = a1 -> z - a2 -> z;
  if ((zt > cutoff) || (zt < lcutoff)) goto SKIP;
      }
      r   = xt * xt + yt * yt + zt * zt;
      r0  = sqrt(r);
      r   = r * r * r;
      ve += dielectric * a1 -> q * a2 -> q / r0;
      va -= a1 -> a * a2 -> a / r;
      vh += a1 -> b * a2 -> b / r / r;
      if ((a2 -> serial < ilow) || (a2 -> serial > ihigh)) {
        vel += dielectric * a1 -> q * a2 -> q / r0;
        val -= a1 -> a * a2 -> a / r;
  vhl += a1 -> b * a2 -> b / r / r;
      }

SKIP:
      if(a2 -> next == a2) break;
      a2 = a2->next;
    } /* End of while */

    fprintf(op, "Vnonbon internal %s %d Eq %f E6 %f E12 %f\n",
      a1 -> name, ii, ve - vel, va - val, vh - vhl);
    fprintf(op, "Vnonbon external %s %d Eq %f E6 %f E12 %f\n",
            a1 -> name, ii, vel, val, vhl);
    fprintf(op, "Vnonbon total    %s %d Eq %f E6 %f E12 %f\n",
            a1 -> name, ii, ve, va, vh);
    *V    += ve  + va  + vh;
    vtint += ve  - vel + va - val + vh - vhl;
    vtout += vel + val + vhl;
    vtt   += ve  + va  + vh;
NOTANATOM:;
  } /* End of for (ii) */

  fprintf(op," Vnonbon total internal %f \n",vtint);
  fprintf(op," Vnonbon total external %f \n",vtout);
  fprintf(op," Vnonbon total          %f \n",vtt);

  return TRUE;
}


/**** Update the non-bond parameters ****/

int AMMP_FASTCALL fv_update_nonbon(float lambda)
{
  AMMP_ATOM **  Close;
  AMMP_ATOM     *a1, *a2;
  float         r, r0, xt, yt, zt;
  float         k, k1, k2;
  float         ka2, kb2;
  float         t1, t2, t3;
  float *       VectorPtr;
  int           CloseSize;
  int           inindex, in;
  int           i, ii, j, imax, inclose;

#ifdef CUBIC
  float         k3, ka3, kb3;
#endif

  const char *  Routine    = "fv_update_nonbon()";
  float         dielectric = GetDielectric();
  float         mmbox      = GetMmbox();
  float         mxcut      = GetMxcut();
  float         mxdq       = GetMxdq2();

  if (fv_update_nonbon_dielecold != dielectric) {
    fv_update_nonbon_dielecold = dielectric;
    mxdq = - one;
  }
  dielectric = Q_CONST / dielectric;

  /**** Allocate the memory for the array space ****/

  imax = a_number();
  if ((NbAtomAllNum != imax) &&
      (!NonBondAllocTable(imax, (mmbox > zero) ? AMMP_NBTABLE_RECTMM : AMMP_NBTABLE_NORMAL))) return FALSE;

  /* First check if anyone's moved and update the lists
   * note that this must be a look-ahead rather than
   * look back search because
   * we cannot update -> px until we've used that atom !!!
   */

  if (NonBondCheck(mxdq, lambda)) return TRUE;

  if (mmbox > zero) {
    mm_fv_update_nonbon(lambda);
    return TRUE;
  }

  CloseSize = NCLOSE;
  if ((Close = (AMMP_ATOM **)Alloca(sizeof(AMMP_ATOM *) * CloseSize, Routine)) == NULL)
    return FALSE;

  for(ii = 0; ii < imax; ii++) {
    a1 = NbAtomAll[ii];
    a1 -> VP  = zero;
    a1 -> dpx = zero;
    a1 -> dpy = zero;
    a1 -> dpz = zero;
    a1 -> qxx = zero;
    a1 -> qxy = zero;
    a1 -> qxz = zero;
    a1 -> qyy = zero;
    a1 -> qyz = zero;
    a1 -> qzz = zero;

    /**** Clear the forces of inactive atoms ****/

    if (!a1 -> active) {
      a1 -> fx = zero;
      a1 -> fy = zero;
      a1 -> fz = zero;
      a1 -> fw = zero;
    }

#ifdef CUBIC
    a1 -> qxxx = zero;
    a1 -> qxxy = zero;
    a1 -> qxxz = zero;
    a1 -> qxyy = zero;
    a1 -> qxyz = zero;
    a1 -> qxzz = zero;
    a1 -> qyyy = zero;
    a1 -> qyyz = zero;
    a1 -> qyzz = zero;
    a1 -> qzzz = zero;
#endif
  } /* End of for (ii) */


  for(ii = 0; ii <  imax; ii++) {
    a1 = NbAtomAll[ii];

    /**** If this is met we update the expansion for this atom ****/

    i  = ii + 1;
    VectorPtr = NbVector + (i << 2);
    if (lambda) {
      while(i < imax) {
        a2            = NbAtomAll[i];
        xt            = a2 -> x - a1 -> x + lambda * (a2 -> dx - a1 -> dx);
        yt            = a2 -> y - a1 -> y + lambda * (a2 -> dy - a1 -> dy);
        zt            = a2 -> z - a1 -> z + lambda * (a2 -> dz - a1 -> dz);
        VectorPtr[0]  = xt;
        VectorPtr[1]  = yt;
        VectorPtr[2]  = zt;
        VectorPtr[3]  = sqrt(xt * xt + yt * yt + zt * zt);
        VectorPtr    += 4;
        ++i;
      } /* End of while */
    } else {
      while(i < imax) {
        a2            = NbAtomAll[i];
        xt            = a2 -> x - a1 -> x;
        yt            = a2 -> y - a1 -> y;
        zt            = a2 -> z - a1 -> z;
        VectorPtr[0]  = xt;
        VectorPtr[1]  = yt;
        VectorPtr[2]  = zt;
        VectorPtr[3]  = sqrt(xt * xt + yt * yt + zt * zt);
        VectorPtr    += 4;
        ++i;
      } /* End of while */
    }

    /**** Add the new components, first extract indexes ****/

    inindex = 0;
    inclose = 0;
    for(i = ii + 1; i < imax; i++) {
      a2 = NbAtomAll[i];
      for(j = 0; j < a1 -> dontuse; j++)
        if (a2 == a1 -> excluded[j]) goto SKIPNEW;
      if (NbVector[(i << 2) + 3] > mxcut) NbIndexes[inindex++] = i;
      else Close[inclose++] = NbAtomAll[i];
      if (inclose == CloseSize) {
        CloseSize <<= 1;
        if ((Close = (AMMP_ATOM **)ReAlloca(Close, sizeof(AMMP_ATOM *) * CloseSize, Routine)) == NULL)
          return FALSE;
      }
SKIPNEW:;
    } /* End of for (i) */

    i = sizeof(AMMP_ATOM *) * inclose;
    if (!a1 -> Close) {
      if ((a1 -> Close = (AMMP_ATOM **)Alloca(i + sizeof(AMMP_ATOM *), Routine)) == NULL)
        return FALSE;
      a1 -> nclose = inclose;
    } else if (a1 -> nclose < inclose) {
      free(a1 -> Close);
      if ((a1 -> Close = (AMMP_ATOM **)Alloca(i + sizeof(AMMP_ATOM *), Routine)) == NULL)
        return FALSE;
      a1 -> nclose = inclose;
    }
    memcpy(a1 -> Close, Close, i);
    a1 -> Close[inclose] = NULL;

    for(in = 0; in < inindex; in++) {
      i   = NbIndexes[in];
      a2  = NbAtomAll[i];
      j   = i * 4;
      r0  = one / NbVector[j + 3];
      r   = r0 * r0;
      r   = r * r * r; /* r0^-6 */
      xt  = a1 -> q * a2 -> q * dielectric * r0;
      yt  = a1 -> a * a2 -> a * r;
      zt  = a1 -> b * a2 -> b * r * r;
      k   = xt - yt + zt;
      xt  = xt * r0;
      yt  = yt * r0;
      zt  = zt * r0;
      k1  = xt - yt * six + zt * twelve;
      xt  = xt * r0;
      yt  = yt * r0;
      zt  = zt * r0;
      k2  = xt * three;
      ka2 = - yt * 48.0f;
      kb2 = zt * 168.0f;

#ifdef CUBIC
      xt  = xt * r0;
      yt  = yt * r0;
      zt  = zt * r0;
      k3  = -xt * 5.0f * 3.0f;
      ka3 = yt * 6.0f * 8.0f * 10.0f;
      kb3 = -zt * 12.0f * 14.0f * 16.0f;
#endif

      k1  = -k1;
      xt  = NbVector[j    ] * r0;
      yt  = NbVector[j + 1] * r0;
      zt  = NbVector[j + 2] * r0;
      a1 -> VP  += k;

      t1         = k1 * xt;
      a2 -> dpx -= t1;
      a1 -> dpx += t1;
      t1         = k1 * yt;
      a2 -> dpy -= t1;
      a1 -> dpy += t1;
      t1         = k1 * zt;
      a2 -> dpz -= t1;
      a1 -> dpz += t1;

      /****  Note that xt has the 1/r in it so k2*xt*xt is 1/r^5 ****/

      t1         = xt * xt;
      t2         = k2 * (t1 - third) + ka2 * (t1 - eightth) + kb2 * (t1 - fourteenth);
      a2 -> qxx -= t2;
      a1 -> qxx -= t2;
      t3         = k2 + ka2 + kb2;
      t2         = t3 * yt * xt;
      a2 -> qxy -= t2;
      a1 -> qxy -= t2;
      t2         = t3 * zt * xt;
      a2 -> qxz -= t2;
      a1 -> qxz -= t2;
      t1         = yt * yt;
      t2         = k2 * (t1 - third) + ka2 * (t1 - eightth) + kb2 * (t1 - fourteenth);
      a2 -> qyy -= t2;
      a1 -> qyy -= t2;
      t2         = t3 * yt * zt;
      a2 -> qyz -= t2;
      a1 -> qyz -= t2;
      t1         = zt * zt;
      t2         = k2 * (t1 - third) + ka2 * (t1 - eightth) + kb2 * (t1 - fourteenth);
      a2 -> qzz -= t2;
      a1 -> qzz -= t2;

#ifdef CUBIC
      t2          = xt * xt;
      t1          = xt * t2;
      a2 -> qxxx -= k3  * (t1 - xt * ( 9.0f / 15.0f)) ;
      a2 -> qxxx -= ka3 * (t1 - xt * (24.0f / 80.0f)) ;
      a2 -> qxxx -= kb3 * (t1 - xt * (42.0f / (14.0f * 16.0f)));
      a1 -> qxxx += k3  * (t1 - xt * ( 9.0f / 15.0f)) ;
      a1 -> qxxx += ka3 * (t1 - xt * (24.0f / 80.0f)) ;
      a1 -> qxxx += kb3 * (t1 - xt * (42.0f / (14.0f * 16.0f)));
      t1          = yt * t2;
      a2 -> qxxy -= k3  * (t1 - yt * ( 6.0f / 15.0f));
      a2 -> qxxy -= ka3 * (t1 - yt * (11.0f / 80.0f));
      a2 -> qxxy -= kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
      a1 -> qxxy += k3  * (t1 - yt * ( 6.0f / 15.0f));
      a1 -> qxxy += ka3 * (t1 - yt * (11.0f / 80.0f));
      a1 -> qxxy += kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
      t1          = zt * t2;
      a2 -> qxxz -= k3  * (t1 - zt * ( 6.0f / 15.0f));
      a2 -> qxxz -= ka3 * (t1 - zt * (11.0f / 80.0f));
      a2 -> qxxz -= kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
      a1 -> qxxz += k3  * (t1 - zt * ( 6.0f / 15.0f));
      a1 -> qxxz += ka3 * (t1 - zt * (11.0f / 80.0f));
      a1 -> qxxz += kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
      t1          = yt* yt * xt;
      a2 -> qxyy -= k3  * (t1 - xt * ( 6.0f / 15.0f));
      a2 -> qxyy -= ka3 * (t1 - xt * (11.0f / 80.0f));
      a2 -> qxyy -= kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
      a1 -> qxyy += k3  * (t1 - xt * ( 6.0f / 15.0f));
      a1 -> qxyy += ka3 * (t1 - xt * (11.0f / 80.0f));
      a1 -> qxyy += kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
      t1          = (k3 + ka3 + kb3) * yt * zt * xt;
      a2 -> qxyz -= t1;
      a1 -> qxyz += t1;
      t1          = zt * zt * xt;
      a2 -> qxzz -= k3  * (t1 - xt * ( 6.0f / 15.0f));
      a2 -> qxzz -= ka3 * (t1 - xt * (11.0f / 80.0f));
      a2 -> qxzz -= kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
      a1 -> qxzz += k3  * (t1 - xt * ( 6.0f / 15.0f));
      a1 -> qxzz += ka3 * (t1 - xt * (11.0f / 80.0f));
      a1 -> qxzz += kb3 * (t1 - xt * (17.0f / (14.0f * 16.0f)));
      t2          = yt * yt;
      t1          = t2 * yt;
      a2 -> qyyy -= k3  * (t1 - yt * ( 9.0f / 15.0f));
      a2 -> qyyy -= ka3 * (t1 - yt * (24.0f / 80.0f));
      a2 -> qyyy -= kb3 * (t1 - yt * (42.0f / (14.0f * 16.0f)));
      a1 -> qyyy += k3  * (t1 - yt * ( 9.0f / 15.0f));
      a1 -> qyyy += ka3 * (t1 - yt * (24.0f / 80.0f));
      a1 -> qyyy += kb3 * (t1 - yt * (42.0f / (14.0f * 16.0f)));
      t1          = t2 * zt
      a2 -> qyyz -= k3  * (t1 - zt * ( 6.0f / 15.0f));
      a2 -> qyyz -= ka3 * (t1 - zt * (11.0f / 80.0f));
      a2 -> qyyz -= kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
      a1 -> qyyz += k3  * (t1 - zt * ( 6.0f / 15.0f));
      a1 -> qyyz += ka3 * (t1 - zt * (11.0f / 80.0f));
      a1 -> qyyz += kb3 * (t1 - zt * (17.0f / (14.0f * 16.0f)));
      t2          = zt * zt;
      t1          = t2 * yt;
      a2 -> qyzz -= k3  * (t1 - yt * ( 6.0f / 15.0f));
      a2 -> qyzz -= ka3 * (t1 - yt * (11.0f / 80.0f));
      a2 -> qyzz -= kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
      a1 -> qyzz += k3  * (t1 - yt * ( 6.0f / 15.0f));
      a1 -> qyzz += ka3 * (t1 - yt * (11.0f / 80.0f));
      a1 -> qyzz += kb3 * (t1 - yt * (17.0f / (14.0f * 16.0f)));
      t1          = t2 * zt;
      a2 -> qzzz -= k3  * (t1 - zt * ( 9.0f / 15.0f));
      a2 -> qzzz -= ka3 * (t1 - zt * (24.0f / 80.0f));
      a2 -> qzzz -= kb3 * (t1 - zt * (42.0f / (14.0f * 16.0f)));
      a1 -> qzzz += k3  * (t1 - zt * ( 9.0f / 15.0f));
      a1 -> qzzz += ka3 * (t1 - zt * (24.0f / 80.0f));
      a1 -> qzzz += kb3 * (t1 - zt * (42.0f / (14.0f * 16.0f)));
#endif
    } /* End of for (i) */

    /**** Set the position ****/

    if (lambda) {
      a1 -> px = a1 -> dx * lambda + a1 -> x;
      a1 -> py = a1 -> dy * lambda + a1 -> y;
      a1 -> pz = a1 -> dz * lambda + a1 -> z;
    } else {
      a1 -> px = a1 -> x;
      a1 -> py = a1 -> y;
      a1 -> pz = a1 -> z;
    }
  } /* End of for (ii) */

  free(Close);

  return TRUE;
}


/**** Calculate the non-bond force increment ****/

int AMMP_FASTCALL f_nonbon(float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         ux, uy, uz;
  float         k, r, r0, xt, yt, zt;
  float         dielectric;
  float         fx, fy, fz;
#if defined(CUBIC) || defined(QUARTIC) || defined(QUINTIC)
  float         xt2, yt2, zt2;

#  if defined(QUARTIC) || defined(QUINTIC)
  float         xt3, yt3, zt3;

#    ifdef QUINTIC
  float         xt4, yt4, zt4;
#    endif
#  endif
#endif

  int           i, ii, jj;
  int           imax;

  /* First update the lists
   * this routine checks if any atom has
   * broken the mxdq barrier and updates the
   * forces, potentials and expansions thereof
   */

  fv_update_nonbon(lambda);

  dielectric = Q_CONST / GetDielectric();
  imax       = a_number();

  if (lambda) {
    for(i = 0; i < imax; i++) {
      a1   = NbAtomAll[i];
      xt   = a1 -> dx * lambda + a1 -> x - a1 -> px;
      yt   = a1 -> dy * lambda + a1 -> y - a1 -> py;
      zt   = a1 -> dz * lambda + a1 -> z - a1 -> pz;
      fx   = a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt;
      fy   = a1 -> qxy * xt + a1 -> qyy * yt + a1 -> qyz * zt;
      fz   = a1 -> qxz * xt + a1 -> qyz * yt + a1 -> qzz * zt;

#ifdef CUBIC
      xt2  = xt * xt;
      yt2  = yt * yt;
      zt2  = zt * zt;
      fx  += a1 -> qxxx * xt2 / 2.0f + a1 -> qxxy * xt * yt + a1 -> qxxz * xt * zt +
             a1 -> qxyy * yt / 2.0f + a1 -> qxyz * yt * zt + a1 -> qxzz * zt2 / 2.0f;
      fy  += a1 -> qxxy * xt2 / 2.0f + a1 -> qxyy * xt * yt + a1 -> qxyz * xt * zt +
             a1 -> qyyy * yt / 2.0f + a1 -> qyyz * yt * zt + a1 -> qyzz * zt2 / 2.0f;
      fz  += a1 -> qxxz * xt2 / 2.0f + a1 -> qxyz * xt * yt + a1 -> qxzz * xt * zt +
             a1 -> qyyz * yt / 2.0f + a1 -> qyzz * yt * zt + a1 -> qzzz * zt2 / 2.0f;
#endif

#ifdef QUARTIC
      xt3  = xt * xt2;
      yt3  = yt * yt2;
      zt3  = zt * zt2;
      fx  += a1 -> qxxxx * xt3 / 6.0f + a1 -> qxxxy * xt2 * yt / 2.0f +
             a1 -> qxxxz * xt2 * zt / 2.0f + a1 -> qxxyy * xt * yt / 2.0f +
             a1 -> qxxyz * xt * yt * zt + a1 -> qxxzz * xt * zt2 / 2.0f +
             a1 -> qxyyy * yt3 / 6.0f + a1 -> qxyyz * yt2 * zt / 2.0f +
             a1 -> qxyzz * yt * zt2 / 2.0f + a1 -> qxzzz * zt3 / 6.0f;
      fy  += a1 -> qxxxy * xt3 / 6.0f + a1 -> qxxyy * xt2 * yt / 2.0f +
             a1 -> qxxyz * xt2 * zt / 2.0f + a1 -> qxyyy * xt * yt / 2.0f +
             a1 -> qxyyz * xt * yt * zt + a1 -> qxyzz * xt * zt2 / 2.0f +
             a1 -> qyyyy * yt3 / 6.0f + a1 -> qyyyz * yt2 * zt / 2.0f +
             a1 -> qyyzz * yt * zt2 / 2.0f + a1 -> qyzzz * zt3 / 6.0f;
      fz  += a1 -> qxxxz * xt3 / 6.0f + a1 -> qxxyz * xt2 * yt / 2.0f +
             a1 -> qxxzz * xt2 * zt / 2.0f + a1 -> qxyyz * xt * yt / 2.0f +
             a1 -> qxyzz * xt * yt * zt + a1 -> qxzzz * xt * zt2 / 2.0f +
             a1 -> qyyyz * yt3 / 6.0f + a1 -> qyyzz * yt2 * zt / 2.0f +
             a1 -> qyzzz * yt * zt2 / 2.0f + a1 -> qzzzz * zt3 / 6.0f;
#endif

#ifdef QUINTIC
      xt4  = xt * xt3;
      yt4  = yt * yt3;
      zt4  = zt * zt3;
      fx  += a1 -> qxxxxx * xt4 / 24.0f + a1 -> qxxxxy * xt3 * yt /6.0f +
             a1 -> qxxxxz * xt3 * zt / 6.0f  + a1 -> qxxxyy * xt2 * yt2 / 4.0f +
             a1 -> qxxxyz * xt2 * yt * zt / 2.0f + a1 -> qxxxzz * xt2 * zt2 / 4.0f +
             a1 -> qxxyyy * xt * yt3 / 6.0f + a1 -> qxxyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxxyzz * xt * yt * zt2 / 2.0f + a1 -> qxxzzz * xt * zt3 / 6.0f +
             a1 -> qxyyyy * yt4 / 24.0f + a1 -> qxyyyz * yt3 * zt / 6.0f +
             a1 -> qxyyzz * yt2 * zt2 / 4.0f + a1 -> qxyzzz * yt * zt3 / 6.0f +
             a1 -> qxzzzz * zt4 / 24.0f;
      fy  += a1 -> qxxxxy * xt4 / 24.0f + a1 -> qxxxyy * xt3 * yt / 6.0f +
             a1 -> qxxxyz * xt3 * zt / 6.0f + a1 -> qxxyyy * xt2 * yt2 / 4.0f +
             a1 -> qxxyyz * xt2 * yt * zt / 2.0f + a1 -> qxxyzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyyy * xt * yt3 / 6.0f + a1 -> qxyyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxyyzz * xt * yt * zt2 / 2.0f + a1 -> qxyzzz * xt * zt3 / 6.0f +
             a1 -> qyyyyy * yt4 / 24.0f + a1 -> qyyyyz * yt3 * zt / 6.0f +
             a1 -> qyyyzz * yt2 * zt2 / 4.0f + a1 -> qyyzzz * yt * zt3 / 6.0f +
             a1 -> qyzzzz * zt4 / 24.0f;
      fz  += a1 -> qxxxxz * xt4 / 24.0f + a1 -> qxxxyz * xt3 * yt / 6.0f +
             a1 -> qxxxzz * xt3 * zt / 6.0f + a1 -> qxxyyz * xt2 * yt2 / 4.0f +
             a1 -> qxxyzz * xt2 * yt * zt / 2.0f + a1 -> qxxzzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyyz * xt * yt3 / 6.0f + a1 -> qxyyzz * xt * yt2 *zt / 2.0f +
             a1 -> qxyzzz * xt * yt * zt2 / 2.0f + a1 -> qxzzzz * xt * zt3 / 6.0f +
             a1 -> qyyyyz * yt4 / 24.0f + a1 -> qyyyzz * yt3 *zt / 6.0f +
             a1 -> qyyzzz * yt2 * zt2 / 4.0f + a1 -> qyzzzz * yt * zt3 / 6.0f +
             a1 -> qzzzzz * zt4 / 24.0f;
#endif

      a1 -> fx += fx + a1 -> dpx;
      a1 -> fy += fy + a1 -> dpy;
      a1 -> fz += fz + a1 -> dpz;

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];

        /**** Note ux is backwards from below ****/

        ux = (a2 -> dx - a1 -> dx) * lambda + (a2 -> x - a1 -> x);
        uy = (a2 -> dy - a1 -> dy) * lambda + (a2 -> y - a1 -> y);
        uz = (a2 -> dz - a1 -> dz) * lambda + (a2 -> z - a1 -> z);
        r  = ux * ux + uy * uy + uz * uz;
        if (!r) continue;
        r  = one / r;
        r0 = sqrt(r);
        ux = ux * r0;
        uy = uy * r0;
        uz = uz * r0;
        k  = -dielectric * a1 -> q * a2 -> q * r;
        r  = r * r * r;
        k += a1 -> a * a2 -> a * r * r0 * six;
        k -= a1 -> b * a2 -> b * r * r * r0 * twelve;
        a1 -> fx += ux * k;
        a1 -> fy += uy * k;
        a1 -> fz += uz * k;
        a2 -> fx -= ux * k;
        a2 -> fy -= uy * k;
        a2 -> fz -= uz * k;
      } /* End of for (ii) */
    } /* End of for (i) */
  } else {
    for(i = 0; i < imax; i++) {
      a1   = NbAtomAll[i];
      xt   = a1 -> x - a1 -> px;
      yt   = a1 -> y - a1 -> py;
      zt   = a1 -> z - a1 -> pz;
      fx   = a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt;
      fy   = a1 -> qxy * xt + a1 -> qyy * yt + a1 -> qyz * zt;
      fz   = a1 -> qxz * xt + a1 -> qyz * yt + a1 -> qzz * zt;

#ifdef CUBIC
      xt2  = xt * xt;
      yt2  = yt * yt;
      zt2  = zt * zt;
      fx  += a1 -> qxxx * xt2 / 2.0f + a1 -> qxxy * xt * yt + a1 -> qxxz * xt * zt +
             a1 -> qxyy * yt / 2.0f + a1 -> qxyz * yt * zt + a1 -> qxzz * zt2 / 2.0f;
      fy  += a1 -> qxxy * xt2 / 2.0f + a1 -> qxyy * xt * yt + a1 -> qxyz * xt * zt +
             a1 -> qyyy * yt / 2.0f + a1 -> qyyz * yt * zt + a1 -> qyzz * zt2 / 2.0f;
      fz  += a1 -> qxxz * xt2 / 2.0f + a1 -> qxyz * xt * yt + a1 -> qxzz * xt * zt +
             a1 -> qyyz * yt / 2.0f + a1 -> qyzz * yt * zt + a1 -> qzzz * zt2 / 2.0f;
#endif

#ifdef QUARTIC
      xt3  = xt * xt2;
      yt3  = yt * yt2;
      zt3  = zt * zt2;
      fx  += a1 -> qxxxx * xt3 / 6.0f + a1 -> qxxxy * xt2 * yt / 2.0f +
             a1 -> qxxxz * xt2 * zt / 2.0f + a1 -> qxxyy * xt * yt / 2.0f +
             a1 -> qxxyz * xt * yt * zt + a1 -> qxxzz * xt * zt2 / 2.0f +
             a1 -> qxyyy * yt3 / 6.0f + a1 -> qxyyz * yt2 * zt / 2.0f +
             a1 -> qxyzz * yt * zt2 / 2.0f + a1 -> qxzzz * zt3 / 6.0f;
      fy  += a1 -> qxxxy * xt3 / 6.0f + a1 -> qxxyy * xt2 * yt / 2.0f +
             a1 -> qxxyz * xt2 * zt / 2.0f + a1 -> qxyyy * xt * yt / 2.0f +
             a1 -> qxyyz * xt * yt * zt + a1 -> qxyzz * xt * zt2 / 2.0f +
             a1 -> qyyyy * yt3 / 6.0f + a1 -> qyyyz * yt2 * zt / 2.0f +
             a1 -> qyyzz * yt * zt2 / 2.0f + a1 -> qyzzz * zt3 / 6.0f;
      fz  += a1 -> qxxxz * xt3 / 6.0f + a1 -> qxxyz * xt2 * yt / 2.0f +
             a1 -> qxxzz * xt2 * zt / 2.0f + a1 -> qxyyz * xt * yt / 2.0f +
             a1 -> qxyzz * xt * yt * zt + a1 -> qxzzz * xt * zt2 / 2.0f +
             a1 -> qyyyz * yt3 / 6.0f + a1 -> qyyzz * yt2 * zt / 2.0f +
             a1 -> qyzzz * yt * zt2 / 2.0f + a1 -> qzzzz * zt3 / 6.0f;
#endif

#ifdef QUINTIC
      xt4  = xt * xt3;
      yt4  = yt * yt3;
      zt4  = zt * zt3;
      fx  += a1 -> qxxxxx * xt4 / 24.0f + a1 -> qxxxxy * xt3 * yt /6.0f +
             a1 -> qxxxxz * xt3 * zt / 6.0f  + a1 -> qxxxyy * xt2 * yt2 / 4.0f +
             a1 -> qxxxyz * xt2 * yt * zt / 2.0f + a1 -> qxxxzz * xt2 * zt2 / 4.0f +
             a1 -> qxxyyy * xt * yt3 / 6.0f + a1 -> qxxyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxxyzz * xt * yt * zt2 / 2.0f + a1 -> qxxzzz * xt * zt3 / 6.0f +
             a1 -> qxyyyy * yt4 / 24.0f + a1 -> qxyyyz * yt3 * zt / 6.0f +
             a1 -> qxyyzz * yt2 * zt2 / 4.0f + a1 -> qxyzzz * yt * zt3 / 6.0f +
             a1 -> qxzzzz * zt4 / 24.0f;
      fy  += a1 -> qxxxxy * xt4 / 24.0f + a1 -> qxxxyy * xt3 * yt / 6.0f +
             a1 -> qxxxyz * xt3 * zt / 6.0f + a1 -> qxxyyy * xt2 * yt2 / 4.0f +
             a1 -> qxxyyz * xt2 * yt * zt / 2.0f + a1 -> qxxyzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyyy * xt * yt3 / 6.0f + a1 -> qxyyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxyyzz * xt * yt * zt2 / 2.0f + a1 -> qxyzzz * xt * zt3 / 6.0f +
             a1 -> qyyyyy * yt4 / 24.0f + a1 -> qyyyyz * yt3 * zt / 6.0f +
             a1 -> qyyyzz * yt2 * zt2 / 4.0f + a1 -> qyyzzz * yt * zt3 / 6.0f +
             a1 -> qyzzzz * zt4 / 24.0f;
      fz  += a1 -> qxxxxz * xt4 / 24.0f + a1 -> qxxxyz * xt3 * yt / 6.0f +
             a1 -> qxxxzz * xt3 * zt / 6.0f + a1 -> qxxyyz * xt2 * yt2 / 4.0f +
             a1 -> qxxyzz * xt2 * yt * zt / 2.0f + a1 -> qxxzzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyyz * xt * yt3 / 6.0f + a1 -> qxyyzz * xt * yt2 *zt / 2.0f +
             a1 -> qxyzzz * xt * yt * zt2 / 2.0f + a1 -> qxzzzz * xt * zt3 / 6.0f +
             a1 -> qyyyyz * yt4 / 24.0f + a1 -> qyyyzz * yt3 *zt / 6.0f +
             a1 -> qyyzzz * yt2 * zt2 / 4.0f + a1 -> qyzzzz * yt * zt3 / 6.0f +
             a1 -> qzzzzz * zt4 / 24.0f;
#endif

      a1 -> fx += fx + a1 -> dpx;
      a1 -> fy += fy + a1 -> dpy;
      a1 -> fz += fz + a1 -> dpz;

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];

        /**** Note ux is backwards from below ****/

        ux = a2 -> x - a1 -> x;
        uy = a2 -> y - a1 -> y;
        uz = a2 -> z - a1 -> z;
        r  = ux * ux + uy * uy + uz * uz;
        if (!r) continue;
        r  = one / r;
        r0 = sqrt(r);
        ux = ux * r0;
        uy = uy * r0;
        uz = uz * r0;
        k  = -dielectric * a1 -> q * a2 -> q * r;
        r  = r * r * r;
        k += a1 -> a * a2 -> a * r * r0 * six;
        k -= a1 -> b * a2 -> b * r * r * r0 * twelve;
        a1 -> fx += ux * k;
        a1 -> fy += uy * k;
        a1 -> fz += uz * k;
        a2 -> fx -= ux * k;
        a2 -> fy -= uy * k;
        a2 -> fz -= uz * k;
      } /* End of for (ii) */
    } /* End of for (i) */
  }

  a_inactive_f_zero();

  return TRUE;
}


/**** Calculate the non-bond potential ****/

int AMMP_FASTCALL v_nonbon(float *V, float lambda)
{
  AMMP_ATOM     *a1, *a2;
  float         r, r0, xt, yt, zt;
  float         k;
  float         vx, dielectric;

#if defined(CUBIC) || defined(QUARTIC) || defined(QUINTIC)
  float         xt2, xt3;
  float         yt2, yt3;
  float         zt2, zt3;

#  if defined(QUARTIC) || defined(QUINTIC)
  float         xt4, yt4, zt4;

#    ifdef QUINTIC
  float         xt5, yt5, zt5;
#    endif
#  endif
#endif

  int           i, ii, jj, imax;

  fv_update_nonbon(lambda);

  dielectric = Q_CONST / GetDielectric();
  imax       = a_number();

  if (lambda) {
    for(i = 0; i < imax; i++) {
      a1  = a_next(i);
      vx  = a1 -> VP;
      xt  = a1 -> dx * lambda + a1 -> x - a1 -> px;
      yt  = a1 -> dy * lambda + a1 -> y - a1 -> py;
      zt  = a1 -> dz * lambda + a1 -> z - a1 -> pz;
      vx -= (a1 -> dpx * xt + a1 -> dpy * yt + a1 -> dpz * zt);
      vx -= ((xt * (0.5f * a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt) +
              yt * (0.5f * a1 -> qyy * yt + a1 -> qyz * zt) +
              0.5f * zt * a1 -> qzz * zt));
#ifdef CUBIC
      xt2  = xt * xt;
      yt2  = yt * yt;
      zt2  = zt * zt;
      xt3  = xt2 * xt;
      yt3  = yt2 * yt;
      zt3  = zt2 * zt;
      vx  -= a1 -> qxxx * xt3 / 6.0f + a1 -> qxxy * xt2 * yt / 2.0f +
             a1 -> qxxz * xt2 * zt / 2.0f + a1 -> qxyy * xt * yt2 / 2.0f +
             a1 -> qxyz * xt * yt *zt + a1 -> qxzz * xt * zt2 / 2.0f +
             a1 -> qyyy * yt3 / 6.0f + a1 -> qyyz * yt2 * zt / 2.0f +
             a1 -> qyzz * yt * zt2 / 2.0f + a1 -> qzzz * zt3 / 6.0f;
#endif

#ifdef QUARTIC
      xt4  = xt3 * xt;
      yt4  = yt3 * yt;
      zt4  = zt3 * zt;
      vx  -= a1 -> qxxxx * xt4 / 24.0f + a1 -> qxxxy * xt3 * yt / 6.0f +
             a1 -> qxxxz * xt3 * yt / 6.0f + a1 -> qxxyy * xt2 * yt2 / 4.0f +
             a1 -> qxxyz * xt2 * yt * zt / 2.0f + a1 -> qxxzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyy * xt * yt3 / 6.0f + a1 -> qxyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxyzz * xt * yt * zt2 / 2.0f + a1 -> qxzzz * xt * zt3 / 6.0f +
             a1 -> qyyyy * yt4 / 24.0f + a1 -> qyyyz * yt3 * zt / 6.0f +
             a1 -> qyyzz * yt2 * zt2 / 4.0f + a1 -> qyzzz * yt * zt3 / 6.0f +
             a1 -> qzzzz * zt4 / 24.0f;
#endif

#ifdef QUINTIC
      xt5  = xt4 * xt;
      yt5  = yt4 * yt;
      zt5  = zt4 * zt;
      vx  -= a1 -> qxxxxx * xt5 / 120.0f + a1 -> qxxxxy * xt4 * yt / 24.0f +
             a1 -> qxxxxz * xt4 * zt / 24.0f + a1 -> qxxxyy * xt3 * yt2 / 12.0f +
             a1 -> qxxxyz * xt3 * yt * zt / 6.0f + a1 -> qxxxzz * xt3 * zt2 / 12.0f +
             a1 -> qxxyyy * xt2 * yt3 / 12.0f + a1 -> qxxyyz * xt2 * yt2 * zt / 4.0f +
             a1 -> qxxyzz * xt2 * yt * zt2 / 4.0f + a1 -> qxxzzz * xt2 * zt3 / 12.0f +
             a1 -> qxyyyy * xt * yt4 / 24.0f + a1 -> qxyyyz * xt * yt3 * zt / 6.0f +
             a1 -> qxyyzz * xt * yt2 * zt2 / 4.0f + a1 -> qxyzzz * xt * yt * zt3 / 6.0f +
             a1 -> qxzzzz * xt * zt4 / 24.0f + a1 -> qyyyyy * yt5 / 120.0f +
             a1 -> qyyyyz * yt4 * zt / 24.0f + a1 -> qyyyzz * yt3 * zt2 / 12.0f +
             a1 -> qyyzzz * yt2 * zt3 / 12.0f + a1 -> qyzzzz * yt * zt4 / 24.0f +
             a1 -> qzzzzz * zt5 / 120.0f;
#endif

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];
        xt = (a2 -> dx - a1 -> dx) * lambda + (a2 -> x - a1 -> x);
        yt = (a2 -> dy - a1 -> dy) * lambda + (a2 -> y - a1 -> y);
        zt = (a2 -> dz - a1 -> dz) * lambda + (a2 -> z - a1 -> z);
        r  = xt * xt + yt * yt + zt * zt;
        if (!r) continue;
        r   = 1.0f / r;
        r0  = sqrt(r);
        k   = dielectric * a1 -> q * a2 -> q * r0;
        r   = r * r * r;
        k  -= a1 -> a * a2 -> a * r;
        k  += a1 -> b * a2 -> b * r * r;
        vx += k;
      }  /* End of for (ii) */
      *V += vx;
    } /* End of for (i) */
  } else {
    for(i = 0; i < imax; i++) {
      a1  = a_next(i);
      vx  = a1 -> VP;
      xt  = a1 -> x - a1 -> px;
      yt  = a1 -> y - a1 -> py;
      zt  = a1 -> z - a1 -> pz;
      vx -= (a1 -> dpx * xt + a1 -> dpy * yt + a1 -> dpz * zt);
      vx -= ((xt * (0.5f * a1 -> qxx * xt + a1 -> qxy * yt + a1 -> qxz * zt) +
              yt * (0.5f * a1 -> qyy * yt + a1 -> qyz * zt) +
              0.5f * zt * a1 -> qzz * zt));
#ifdef CUBIC
      xt2  = xt * xt;
      yt2  = yt * yt;
      zt2  = zt * zt;
      xt3  = xt2 * xt;
      yt3  = yt2 * yt;
      zt3  = zt2 * zt;
      vx  -= a1 -> qxxx * xt3 / 6.0f + a1 -> qxxy * xt2 * yt / 2.0f +
             a1 -> qxxz * xt2 * zt / 2.0f + a1 -> qxyy * xt * yt2 / 2.0f +
             a1 -> qxyz * xt * yt *zt + a1 -> qxzz * xt * zt2 / 2.0f +
             a1 -> qyyy * yt3 / 6.0f + a1 -> qyyz * yt2 * zt / 2.0f +
             a1 -> qyzz * yt * zt2 / 2.0f + a1 -> qzzz * zt3 / 6.0f;
#endif

#ifdef QUARTIC
      xt4  = xt3 * xt;
      yt4  = yt3 * yt;
      zt4  = zt3 * zt;
      vx  -= a1 -> qxxxx * xt4 / 24.0f + a1 -> qxxxy * xt3 * yt / 6.0f +
             a1 -> qxxxz * xt3 * yt / 6.0f + a1 -> qxxyy * xt2 * yt2 / 4.0f +
             a1 -> qxxyz * xt2 * yt * zt / 2.0f + a1 -> qxxzz * xt2 * zt2 / 4.0f +
             a1 -> qxyyy * xt * yt3 / 6.0f + a1 -> qxyyz * xt * yt2 * zt / 2.0f +
             a1 -> qxyzz * xt * yt * zt2 / 2.0f + a1 -> qxzzz * xt * zt3 / 6.0f +
             a1 -> qyyyy * yt4 / 24.0f + a1 -> qyyyz * yt3 * zt / 6.0f +
             a1 -> qyyzz * yt2 * zt2 / 4.0f + a1 -> qyzzz * yt * zt3 / 6.0f +
             a1 -> qzzzz * zt4 / 24.0f;
#endif

#ifdef QUINTIC
      xt5  = xt4 * xt;
      yt5  = yt4 * yt;
      zt5  = zt4 * zt;
      vx  -= a1 -> qxxxxx * xt5 / 120.0f + a1 -> qxxxxy * xt4 * yt / 24.0f +
             a1 -> qxxxxz * xt4 * zt / 24.0f + a1 -> qxxxyy * xt3 * yt2 / 12.0f +
             a1 -> qxxxyz * xt3 * yt * zt / 6.0f + a1 -> qxxxzz * xt3 * zt2 / 12.0f +
             a1 -> qxxyyy * xt2 * yt3 / 12.0f + a1 -> qxxyyz * xt2 * yt2 * zt / 4.0f +
             a1 -> qxxyzz * xt2 * yt * zt2 / 4.0f + a1 -> qxxzzz * xt2 * zt3 / 12.0f +
             a1 -> qxyyyy * xt * yt4 / 24.0f + a1 -> qxyyyz * xt * yt3 * zt / 6.0f +
             a1 -> qxyyzz * xt * yt2 * zt2 / 4.0f + a1 -> qxyzzz * xt * yt * zt3 / 6.0f +
             a1 -> qxzzzz * xt * zt4 / 24.0f + a1 -> qyyyyy * yt5 / 120.0f +
             a1 -> qyyyyz * yt4 * zt / 24.0f + a1 -> qyyyzz * yt3 * zt2 / 12.0f +
             a1 -> qyyzzz * yt2 * zt3 / 12.0f + a1 -> qyzzzz * yt * zt4 / 24.0f +
             a1 -> qzzzzz * zt5 / 120.0f;
#endif

      /**** Do the close atoms ****/

      for(jj = 0; a1 -> Close[jj]; jj++);
      for(ii = 0; ii < jj; ii++) {
        a2 = a1 -> Close[ii];
        xt = a2 -> x - a1 -> x;
        yt = a2 -> y - a1 -> y;
        zt = a2 -> z - a1 -> z;
        r  = xt * xt + yt * yt + zt * zt;
        if (!r) continue;
        r   = 1.0f / r;
        r0  = sqrt(r);
        k   = dielectric * a1 -> q * a2 -> q * r0;
        r   = r * r * r;
        k  -= a1 -> a * a2 -> a * r;
        k  += a1 -> b * a2 -> b * r * r;
        vx += k;
      }  /* End of for (ii) */
      *V += vx;
    } /* End of for (i) */
  }

  a_inactive_f_zero();

  return TRUE;
}


/**** Sum the potentials ****/

int AMMP_FASTCALL zone_nonbon(float *V, float lambda, AMMP_ATOM *(*alist)[], int inalist)
{
  AMMP_ATOM     *a1, *a2;
  float         r, r0, xt, yt, zt;
  float         lcutoff, cutoff;
  float         dielectric, ve, va, vh;
  int           i, ii;

  if (inalist <= 0) return TRUE;

  dielectric = Q_CONST / GetDielectric();
  cutoff     = GetCutoff();
  lcutoff    = -cutoff;

  for(ii = 0; ii < inalist; ii++) {
    a1 = (*alist)[ii];
    if (a1 == NULL) goto NOTANATOM;
    ve = zero;
    va = zero;
    vh = zero;

    a2 = a_next(-1);
    while((a2 != NULL) && (a2 -> next != NULL) && (a2 -> next != a2)) {
      if (a2 == a1) goto SKIP;
      for(i = 0; i < a1 -> dontuse; i++)
        if (a2 == a1 -> excluded[i]) goto SKIP;

      /**** Non-bonded are only used when the atoms arent bonded ****/

      xt = a1 -> x - a2 -> x + lambda * (a1 -> dx - a2 -> dx);
      if ((xt > cutoff) || (xt < lcutoff)) goto SKIP;
      yt = a1 -> y - a2 -> y + lambda * (a1 -> dy - a2 -> dy);
      if ((yt > cutoff) || (yt < lcutoff)) goto SKIP;
      zt = a1 -> z - a2 -> z + lambda * a1 -> dz - a2 -> dz;
      if ((zt > cutoff) || (zt < lcutoff)) goto SKIP;

      r = xt * xt + yt * yt + zt * zt;
      if (r < one) r = one;
      r0  = sqrt(r);
      r   = r * r * r ;
      ve += dielectric * a1 -> q * a2 -> q / r0;
      va -= a1 -> a * a2 -> a / r;
      vh += a1 -> b * a2 -> b / r / r;

SKIP:
      if (a2 -> next == a2) break;
      a2 = a2 -> next;
    } /* End of while */
    *V += ve + va + vh;
NOTANATOM:;
  } /* End of for (ii) */

  return TRUE;
}


