
/*************************************************
****        VEGA - x86 CPU utilities          ****
**** Copyright 1996-2012, Alessandro Pedretti ****
*************************************************/


#if defined(__BORLANDC__) && !defined(WIN32)
#  define  WIN32
#endif

#ifdef WIN32
#  include <windows.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  pragma inline
#endif

#include "dxwcpu.h"

#define  NUM_LOGICAL_BITS      0x00FF0000  // Bit 16-23 in ebx contains the number of logical
                                           // processors per physical processor when execute cpuid with
                                           // eax set to 1
#ifndef WIN32
#  define  ZeroMemory(A, S)    memset((A), 0, (S))
#endif

/**** Cpuid wrapper ****/

#ifndef WIN64
#  ifdef __GNUC__
#    ifdef DXWCPU_64BIT

/**** Cpuid - GNU 64 bit ****/

static void Cpuid(CPUID_ARGS *p)
{
  __asm__ ("movl  (%%rdi), %%eax\n"
           "movl 8(%%rdi), %%ecx\n"
           "cpuid\n"
           "movl %%eax,   (%%rdi)\n"
           "movl %%ebx,  4(%%rdi)\n"
           "movl %%ecx,  8(%%rdi)\n"
           "movl %%edx, 12(%%rdi)\n"
           :
           :"D" (p)
           :"ax", "bx", "cx", "dx", "memory");
}

#    else

/**** Cpuid - GNU 32 bit ****/

static void Cpuid(CPUID_ARGS *p)
{
  __asm__ ("movl  (%%edi), %%eax\n"
           "movl 8(%%edi), %%ecx\n"
           "cpuid\n"
           "movl %%eax,   (%%edi)\n"
           "movl %%ebx,  4(%%edi)\n"
           "movl %%ecx,  8(%%edi)\n"
           "movl %%edx, 12(%%edi)\n"
           :
           :"D" (p)
           :"ax", "bx", "cx", "dx", "memory");

}

#    endif
#  else

/**** Cpuid - Borland 32 bit ****/

static void Cpuid(CPUID_ARGS *p)
{
  __asm {
    mov edi, p
    mov eax, [edi].eax
    mov ecx, [edi].ecx
    cpuid
    mov [edi].eax, eax
    mov [edi].ebx, ebx
    mov [edi].ecx, ecx
    mov [edi].edx, edx
  }
}
#  endif
#else

/**** Cpuid - GNU 64 bit ****/

static void Cpuid(CPUID_ARGS *p)
{
  __asm__ ("movl  (%%rdi), %%eax\n"
           "movl 8(%%rdi), %%ecx\n"
           "cpuid\n"
           "movl %%eax,   (%%rdi)\n"
           "movl %%ebx,  4(%%rdi)\n"
           "movl %%ecx,  8(%%rdi)\n"
           "movl %%edx, 12(%%rdi)\n"
           :
           :"D" (p)
           :"ax", "bx", "cx", "dx", "memory");
}

#  ifdef __GNUC__
#  else
/**** Cpuid - Visual C 64 bit ****/

#    ifdef __cplusplus
extern "C" {
#    endif
void Cpuid64(CPUID_ARGS *p);
#    ifdef __cplusplus
}
#    endif
#    define Cpuid         Cpuid64
#  endif
#endif


/**** Constructor ****/

DXWCPU::DXWCPU(void)
{
  char          *Tmp;
  uint32        k, j;

  m_CPU_Features           = 0;
  m_CPU_GotInfo            = false;
  m_CPU_HtActive           = false;
  m_CPU_LogicalPerPhysical = 1;
  m_MaxCPUIDVal            = 0;
  m_ProcBrandID            = 0;
  m_ProcType               = 0;
  m_ProcFamily             = 0;
  m_ProcExtendedFamily     = 0;
  m_ProcModel              = 0;
  m_ProcExtendedModel      = 0;
  m_ProcStepping           = 0;

  /**** Set the suffixes ****/

  for(k = 0; k < CPU_SUFF_MAXNUM; ++k) m_CPU_Suffix[k] = NULL;
  ZeroMemory(m_CPU_Suffix, sizeof(char *) * CPU_SUFF_MAXNUM);
  m_CPU_Suffix[0]      = (char *)CPU_SUFF_INTEL_586;
  m_CPU_Suffixes       = 1;

  GetInfo();

  for(k = 0, j = m_CPU_Suffixes - 1; k < j; ++k, --j) {
    Tmp             = m_CPU_Suffix[k];
    m_CPU_Suffix[k] = m_CPU_Suffix[j];
    m_CPU_Suffix[j] = Tmp;
  }
}


/**** Find a file ****/

bool DXWCPU::FindFile(char *FileName)
{
  char          *Prefix, **Suffix;

  FILE          *FH = NULL;

  Prefix = FileName + strlen(FileName) - 4;
  for(Suffix = m_CPU_Suffix; (FH == NULL)&& (*Suffix); ++Suffix) {
    *Prefix = 0;
    strcat(FileName, *Suffix);
    strcat(FileName, ".exe");
    if ((FH = fopen(FileName, "r")) != NULL) fclose(FH);
  } /* End of for */

  return (FH != NULL);
}


/**** Execute CPUID function EAX=funcNum and get EAX result ****/

uint32 DXWCPU::GetCPUID_EAX(uint32 funcNum)
{
  CPUID_ARGS    Args;

  Args.eax = funcNum;
  Cpuid(&Args);

  return Args.eax;
}


/**** Execute CPUID function EAX=funcNum and get EDX result ****/

uint32 DXWCPU::GetCPUID_EDX(uint32 funcNum)
{
  CPUID_ARGS    Args;

  Args.eax = funcNum;
  Cpuid(&Args);

  return Args.edx;
}


/**** Execute CPUID EAX=funcNum and get EAX result and ****/
/**** the string EBX:EDX:ECX                           ****/

uint32 DXWCPU::GetCPUIDString(uint32 funcNum, char *String)
{
  CPUID_ARGS    Args;

  Args.eax = funcNum;
  Cpuid(&Args);
  ((uint32 *)String)[0] = Args.ebx;
  ((uint32 *)String)[1] = Args.edx;
  ((uint32 *)String)[2] = Args.ecx;

  return Args.eax;

#if 0
  uint32    retval;

#ifdef __GNUC__
  uint32    EbxReg, EcxReg, EdxReg;
  try {
    __asm__("cpuid"
            : "=a" (retval),
              "=b" (EbxReg),
              "=c" (EcxReg),
              "=d" (EdxReg)
            : "0" (funcNum));

    ((uint32 *)String)[0] = EbxReg;
    ((uint32 *)String)[1] = EdxReg;
    ((uint32 *)String)[2] = EcxReg;
  } catch (...) {
    retval = 0;
  }
#else
  __try {
    _asm {
      mov   eax,funcNum
      CPUID
      mov   retval,eax
      mov   eax,String
      mov   dword ptr[eax],ebx
      mov   dword ptr[eax+4],edx
      mov   dword ptr[eax+8],ecx
    }
  } __except(EXCEPTION_EXECUTE_HANDLER) {
    retval = 0;
  }
#endif

  return retval;
#endif
}


/**** AMD Specific ****/

void DXWCPU::GetCPUIDStringAMD(uint32 funcNum, char *String)
{
  CPUID_ARGS    Args;

  Args.eax = funcNum;
  Cpuid(&Args);
  ((uint32 *)String)[0] = Args.eax;
  ((uint32 *)String)[1] = Args.ebx;
  ((uint32 *)String)[2] = Args.ecx;
  ((uint32 *)String)[3] = Args.edx;

#if 0
  uint32    retval;

#ifdef __GNUC__
  uint32    EbxReg, EcxReg, EdxReg;
  try {
    __asm__("cpuid"
            : "=a" (retval),
              "=b" (EbxReg),
              "=c" (EcxReg),
              "=d" (EdxReg)
            : "0" (funcNum));

    ((uint32 *)String)[0] = retval;
    ((uint32 *)String)[1] = EbxReg;
    ((uint32 *)String)[2] = EcxReg;
    ((uint32 *)String)[3] = EdxReg;
      } catch (...) {
    retval = 0;
  }
#else
  __try {
    _asm {
      mov       eax,funcNum
      CPUID
      mov   retval,eax
      mov   eax,String
      mov   dword ptr[eax+4],ebx
      mov   dword ptr[eax+8],ecx
      mov   ebx,retval
      mov   dword ptr[eax+12],edx
      mov   dword ptr[eax],ebx
    }
  } __except(EXCEPTION_EXECUTE_HANDLER) {
    retval = 0;
  }
#endif
#endif
}


/**** Get the information about the CPU ****/

bool DXWCPU::GetInfo(void)
{
  CPUID_ARGS    Args;

  if (m_CPU_GotInfo) return true;
  m_CPU_GotInfo = true;

  ZeroMemory(m_ProcVendorString, sizeof(m_ProcVendorString));
  m_MaxCPUIDVal = GetCPUIDString(0, m_ProcVendorString);

  /**** CPUID 1 ****/

  Args.eax = 1;
  Cpuid(&Args);

  m_ProcType     = (Args.eax >> 12) & 0x3;
  m_ProcFamily   = (Args.eax >> 8 ) & 0xf;
  m_ProcModel    = (Args.eax >> 4 ) & 0xf;
  m_ProcStepping = (Args.eax      ) & 0x7;
  m_CPU_Features = Args.edx;

  /**** Get the number of CPUs ****/

#ifdef WIN32
  SYSTEM_INFO SysInfo;
  ZeroMemory(&SysInfo, sizeof(SYSTEM_INFO));
  GetSystemInfo(&SysInfo);
  m_CPU_Number = SysInfo.dwNumberOfProcessors;
#else
  FILE * FH = popen("grep '^processor' /proc/cpuinfo| wc -l", "r");
  if (FH) {
    fscanf(FH, "%d", &m_CPU_Number);
    if (m_CPU_Number < 1) m_CPU_Number = 1;
    pclose(FH);
  }
#endif

  /**** Check the virtual CPUs ****/

  if (m_CPU_Features & INTEL_HT_FLAG)
    m_CPU_LogicalPerPhysical = ((Args.ebx & NUM_LOGICAL_BITS) >> 16);

  /**** Intel CPU ****/

  if ((strncmp(m_ProcVendorString, VENDOR_INTEL, 12)) == 0) {
    m_CPU_Type           = Args.eax;
    m_ProcBrandID        = Args.ebx & 0xF;
    strcpy(m_ProcName, GetIntelProcName());

    if (m_ProcFamily >= 6) {                                    /* Pentium Pro / 2 */
      m_CPU_Suffix[m_CPU_Suffixes++] = (char *)CPU_SUFF_INTEL_686;
      if (m_ProcModel >= 7)                                     /* Pentium 3       */
        m_CPU_Suffix[m_CPU_Suffixes++] = (char *)CPU_SUFF_INTEL_P3;
    }

    if (m_ProcFamily >= 15)                                     /* Pentium 4       */
      m_CPU_Suffix[m_CPU_Suffixes++] = (char *)CPU_SUFF_INTEL_P4;

    if ((m_ProcFamily == 6) && (m_ProcModel >= 14)) {           /* Core 2          */
      m_CPU_Suffix[m_CPU_Suffixes++] = (char *)CPU_SUFF_INTEL_P4;
      m_CPU_Suffix[m_CPU_Suffixes++] = (char *)CPU_SUFF_INTEL_CORE2;
    }

    /**** AMD CPU ****/

  } else if ((strncmp(m_ProcVendorString, VENDOR_AMD, 12))==0) {
    if (GetCPUID_EAX(0x80000000)) {
      m_CPU_ExtendedFeatures = GetCPUID_EDX(0x80000001);
      GetCPUIDStringAMD(0x80000002, m_ProcName);
      GetCPUIDStringAMD(0x80000003, m_ProcName + 16);
      GetCPUIDStringAMD(0x80000004, m_ProcName + 32);
    } else m_CPU_ExtendedFeatures = Args.edx;

    if (m_ProcFamily >= 6)
      m_CPU_Suffix[m_CPU_Suffixes++] = (char *)CPU_SUFF_INTEL_686;
    if (m_ProcFamily >= 5)
      m_CPU_Suffix[m_CPU_Suffixes++] = (char *)CPU_SUFF_AMD_K6;
    if (m_ProcFamily >= 6) {
      m_CPU_Suffix[m_CPU_Suffixes++] = (char *)CPU_SUFF_AMD_K7;
      if (m_CPU_Features & INTEL_SSE_FLAG)
        m_CPU_Suffix[m_CPU_Suffixes++] = (char *)CPU_SUFF_AMD_K7SSE;
    }
    if (m_ProcFamily >= 15)
      m_CPU_Suffix[m_CPU_Suffixes++] = (char *)CPU_SUFF_AMD_A64;
  }

  return true;
}


/**** Obtain the Intel's processor name ****/

char *DXWCPU::GetIntelProcName(void)
{
  char          *CPUName = (char *)PROCESSOR_UNKNOWN;

  switch (m_ProcFamily) {
/*
  case 4:       // 486 family
    if (Model == 4) CPUName = (char *)INTEL_486SL;
    if (Model == 7) CPUName = (char *)INTEL_DX2WB;
    if (Model == 8) CPUName = (char *)INTEL_DX4;
    if ((Model == 8)&& (Type == 1)) CPUName = (char *)INTEL_DX4O;
    break;
*/

  case 5:       // 586 family
    if (m_ProcType == 1) CPUName = (char *)INTEL_PO;
    else CPUName = (char *)INTEL_P;
    break;

  case 6:       // 686 family (0110)
    switch(m_ProcModel) {
    case 1:     /* 0001 Pentium Pro */
      CPUName = (char *)INTEL_PPRO;
      break;

    case 3:     /* 0011 Pentium II */
      if (m_ProcType == 1) CPUName = (char *)INTEL_PIIO;
      else CPUName = (char *)INTEL_PII;
      break;

    case 4:
      CPUName = (char *)INTEL_P55CT;
      break;

    case 5:     /* 0101 */
      CPUName = (char *)INTEL_PII;
      break;

    case 6:     /* 0110 */
      CPUName = (char *)INTEL_CELERON;
      break;

    case 7:     /* 0111 Pentium III */
    case 8:     /* 1000 Pentium III */
      CPUName = (char *)INTEL_PIII;
      break;

    case 9:     /* 1001 Pentium M */
      CPUName = (char *)INTEL_PMC;
      break;

    case 10:    /* 1010 Pentium III Xeon */
      CPUName = (char *)INTEL_XEON;
      break;

    case 11:    /* 1011 Pentium III */
      CPUName = (char *)INTEL_PIIIM;
      break;

    case 12:    /* 1100 Atom */
      CPUName = (char *)INTEL_ATOM;
      break;

    case 13:    /* 1101 Pentium M, Celeron M */
      CPUName = (char *)INTEL_PMC;
      break;

    case 14:    /* 1110 Core 2/ Core Solo */
      CPUName = (char *)INTEL_PMDC;
      break;

    case 15:    /* 1111 Core 2 Duo, Mobile, Quad */
    case 17:
      CPUName = (char *)INTEL_CORE2DC;
      break;

    case 16:
      CPUName = (char *)INTEL_CORE2SC;
      break;
    }
    break;

  case 15:      // Pentium 4 family
    m_ProcExtendedFamily = (m_CPU_Type >> 20) & 0xff;
    switch(m_ProcExtendedFamily) {
    case 0:     // Family = 15, Ext. Family = 0:  Pentium 4 (80786 ??) processor family
      m_ProcExtendedModel = (m_CPU_Type >> 16) & 0xFF;
      switch(m_ProcModel) {
      case 0:
      case 1:
        if (m_ProcBrandID == 8) CPUName = (char *)INTEL_PIV;
        else CPUName = (char *)INTEL_PIVXEON;
        break;

      case 2:
        if ((m_ProcBrandID == 9) || (m_ProcBrandID == 0xA)) CPUName = (char *)INTEL_PIV;
        else CPUName = (char *)INTEL_PIVXEON;
        break;

      default:
        CPUName = (char *)INTEL_PIV;
      } /* End of switch */
      break;

    case 1:     // Family = 15, Ext. Family = 1:  McKinley (64-bit) processor family
      CPUName = (char *)INTEL_MCKINLEY;
      break;

    default:
      CPUName = (char *)"Unknown Pentium 4+";
      break;
    } /* End of switch */
    break;
  } /* End of switch */

  return CPUName;
}


/**** Load library ****/

#if defined(WIN32) || defined(WIN64)
HINSTANCE DXWCPU::LoadLibrary(LPCTSTR lpszLibrary)
{
  char          Buf[MAX_PATH];
  char          *Prefix, **Suffix;

  HINSTANCE     hLib = NULL;

  strcpy(Buf, lpszLibrary);
  Prefix = Buf + strlen(Buf) - 4;
  for(Suffix = m_CPU_Suffix; (hLib == NULL) && (*Suffix); ++Suffix) {
    *Prefix = 0;
    strcat(Buf, *Suffix);
    strcat(Buf, ".dll");
    hLib = ::LoadLibrary(Buf);
  } /* End of for */

  return hLib;
}
#endif

