
/*********************************************************************
****         ESCHER Next Generation - Molecule routines           ****
**** (c) 1997-2012, G. Ausiello, G. Cesareni, M. Helmer Citterich ****
****               NG version by Alessandro Pedretti              ****
*********************************************************************/


#include "my_macros.h"
#include <ctype.h>

#include "brk.h"
#include "globstr.h"
#include "utils.h"

#ifdef __BORLAND_C__
#  pragma hdrstop
#endif

#include "param.h"

/**** Local variables ****/

static const char       *PDBAtmRec[] = { "ATOM", "HETATM" };

/*
 * Residue name array
 * ==================
 *
 * WARNING: Don't change the order of the residue names and add the new entries
 *          at the end of the array, otherwise the charge array must be renumbered
 *          (i and j fields).
 */

static const ES_QCHAR   Residues[]   = { /*   0 */ {"ALA"   }, /**** Aminoacids ****/
                                         /*   1 */ {"CYS"   },
                                         /*   2 */ {"ASP"   },
                                         /*   3 */ {"GLU"   },
                                         /*   4 */ {"PHE"   },
                                         /*   5 */ {"GLY"   },
                                         /*   6 */ {"HIS"   },
                                         /*   7 */ {"ILE"   },
                                         /*   8 */ {"LYS"   },
                                         /*   9 */ {"LEU"   },
                                         /*  10 */ {"MET"   },
                                         /*  11 */ {"ASN"   },
                                         /*  12 */ {"PRO"   },
                                         /*  13 */ {"GLN"   },
                                         /*  14 */ {"ARG"   },
                                         /*  15 */ {"SER"   },
                                         /*  16 */ {"THR"   },
                                         /*  17 */ {"VAL"   },
                                         /*  18 */ {"TRP"   },
                                         /*  19 */ {"TYR"   },
                                         /*  20 */ {"PCA"   },

                                         /*  21 */ {"A\0\0" }, /**** DNA/RNA ****/
                                         /*  22 */ {"T\0\0" },
                                         /*  23 */ {"G\0\0" },
                                         /*  24 */ {"C\0\0" },
                                         /*  25 */ {"U\0\0" },
                                         /*  26 */ {"DA\0" },
                                         /*  27 */ {"DT\0" },
                                         /*  28 */ {"DG\0" },
                                         /*  29 */ {"DC\0" },

                                         /*  30 */ {"SAH"   }, /**** Cofactors ****/

                                         /*  31 */ {"MG\0"  }, /**** Metals ****/
                                         /*  32 */ {"MG2"   },
                                         /*  33 */ {"CA\0"  },
                                         /*  34 */ {"CA2"   },
                                         /*  35 */ {"K\0\0" },
                                         /*  36 */ {"K1\0"  },
                                         /*  37 */ {"K+\0"  },

                                         /* 999 */ {"\0\0\0"}  /**** End ****/
                                       };

/*
 * Charge array
 * ============
 *
 * Field descriptions:
 * i = First residue number in the Residue array.
 * j = Last residue number in the Residue array.
 * t = atom name.
 * c = charge score.
 */

static const ES_ATMTYPE  tab[]      = {{14, 14, {"NH1"}, 1},
                                       {14, 14, {"NH2"}, 1},
                                       { 8,  8, {"NZ" }, 1},
                                       { 2,  2, {"OD1"}, 2},
                                       { 2,  2, {"OD2"}, 2},
                                       { 3,  3, {"OE1"}, 2},
                                       { 3,  3, {"OE2"}, 2},
                                       { 0, 20, {"N"  }, 3},
                                       {14, 14, {"NE" }, 3},
                                       {11, 11, {"ND2"}, 3},
                                       {13, 13, {"NE2"}, 3},
                                       {0 , 20, {"O"  }, 4},
                                       {11, 11, {"OD1"}, 4},
                                       {13, 13, {"OE1"}, 4},
                                       {15, 15, {"OG" }, 5},
                                       {16, 16, {"OG1"}, 5},
                                       {19, 19, {"OH "}, 5},
                                       { 6,  6, {"ND1"}, 5},
                                       { 6,  6, {"NE2"}, 5},

                                       /**** DNA/RNA ****/

                                       {21, 29, {"O1P"}, 2},
                                       {21, 29, {"O2P"}, 2},

                                       /**** Cofactors ****/

                                       {30, 30, {"OXT"}, 2},
                                       {30, 30, {"O"  }, 2},
                                       {30, 30, {"O2*"}, 5},
                                       {30, 30, {"O3*"}, 5},
                                       {30, 30, {"N"  }, 1},
                                       {30, 30, {"N6" }, 3},

                                       /**** Metals ****/

                                       {31, 32, {"MG" }, 1},
                                       {31, 32, {"MG1"}, 1},
                                       {33, 34, {"CA" }, 1},
                                       {33, 34, {"CA1"}, 1},
                                       {35, 37, {"K"  }, 1},
                                       {35, 37, {"K1" }, 1},
                                       {35, 37, {"K+" }, 1},

                                       /**** Anions ****/

                                       {0 ,  0, {"\0\0\0"}, 0}
                                      };

struct tipo_carica {
  char          car;
  char          val;
} car[7] = { {0, 0}, {1, 2}, {3, 2}, {1, 1}, {3, 1}, {2, 1}, {0, 0}};


/**** Allocate a new molecule ****/

ES_BRK *BrkNew(void)
{
  ES_BRK        *Brk;

  if ((Brk = (ES_BRK *)Alloca(sizeof(ES_BRK))) != NULL)
    Brk -> n_atoms = 0;

  return Brk;
}


/**** Copy the molecule ****/

void brk_copy(ES_BRK *dest, const ES_BRK *source)
{
  dest -> n_atoms = source -> n_atoms;
  dest -> atom = (ES_ATOM *)Alloca(dest -> n_atoms * sizeof(ES_ATOM));
  if (dest -> atom == NULL) {
    fprintf(stderr, "brk_copy\n");
    exit(1);
  }
  memcpy(dest -> atom, source -> atom, dest -> n_atoms * sizeof(ES_ATOM));
}


/**** Remove the molecule from the memory ****/

void BrkDelete(ES_BRK *Brk)
{
  FREE(Brk -> atom);
}


/**** Read the PDB file ****/

ES_BOOL BrkRead(char *name, ES_BRK *brk)
{
  char                  *j, ResSeq[20];
  const ES_ATMTAB       *AtmTab;
  ES_RECORD             PDBLin;
  int                   i, k, l;
  int                   aminoacido, carica;
  FILE                  *f;

  ES_ULONG              LineNum = 1;

  if ((f = fopen(name, "rb")) == NULL)
    return CatErr(MSG_ERR_FILE_NOTFOUND, name);

  brk -> n_atoms = 0;

  /**** Count the atoms ****/

  while(fgets(PDBLin.Line, ES_LINELEN, f)) {
    if ((PDBLin.Hdr == *(ES_LONG *)PDBAtmRec[0]) ||
        (PDBLin.Hdr == *(ES_LONG *)PDBAtmRec[1]))
      brk -> n_atoms++;
  } /* End of while */

  /**** Allocate the memory ****/

  if ((brk -> atom = (ES_ATOM *)Alloca(brk -> n_atoms * sizeof(ES_ATOM))) == NULL) {
    fclose(f);
    return FALSE;
  }

  rewind(f);
  i = 0;
  while(fgets(PDBLin.Line, ES_LINELEN, f)) {
    if ((PDBLin.Hdr == *(ES_LONG *)PDBAtmRec[0]) ||
        (PDBLin.Hdr == *(ES_LONG *)PDBAtmRec[1])) {

      /**** Decode the element ****/

      sscanf(PDBLin.Line + 6, "%d %4s", &brk -> atom[i].Num, brk -> atom[i].Name.C);

      l = *brk -> atom[i].Name.C;
      if (isdigit(l)) {
        for(k = 0; (k < 4) && (brk -> atom[i].Name.C[k]); ++k);
        for(j = brk -> atom[i].Name.C; j < (brk -> atom[i].Name.C + sizeof(brk -> atom[i].Name.C) - 1); ++j)
          *j = j[1];
        brk -> atom[i].Name.C[--k] = (char)l;
      }
      *brk -> atom[i].Elem.C = (char)toupper(*brk -> atom[i].Name.C);
      if ((isalpha(PDBLin.Line[12])) && (!isdigit(brk -> atom[i].Name.C[1])))
        brk -> atom[i].Elem.C[1] = (char)tolower(brk -> atom[i].Name.C[1]);

      /**** if present, decode the element name ****/

      if ((strlen(PDBLin.Line) >= 79) && (PDBLin.Line[77] != ' ') &&
          (!isdigit(PDBLin.Line[76])) && (!isdigit(PDBLin.Line[77]))) {
        if (PDBLin.Line[76] != ' ') {
          brk -> atom[i].Elem.C[0] = PDBLin.Line[76];
          brk -> atom[i].Elem.C[1] = PDBLin.Line[77];
        } else {
          brk -> atom[i].Elem.C[0] = PDBLin.Line[77];
          brk -> atom[i].Elem.C[1] = 0;
        }
      }

      sscanf(PDBLin.Line + 17, "%3s", brk -> atom[i].ResName.C);
      brk -> atom[i].ChainID = PDBLin.Line[21];
      sscanf(PDBLin.Line + 22, "%4s", ResSeq);
      Str2Qchar(&(brk -> atom[i].ResSeq), ResSeq);
      sscanf(PDBLin.Line + 30, "%8f%8f%8f",
             &brk -> atom[i].x,
             &brk -> atom[i].y,
             &brk -> atom[i].z);


      /**** Adjust the ATOM/HETATM flags ****/

      if (PDBLin.Hdr == *(ES_LONG *)PDBAtmRec[0])
        brk -> atom[i].Flags = ES_ATMF_NONE;
      else
        brk -> atom[i].Flags = ES_ATMF_HETATM;

      /**** Assign the Van der Waals radius ****/

      for(AtmTab = AtmTable; *AtmTab -> Elem.C; ++AtmTab) {
        if (AtmTab -> Elem.S == brk -> atom[i].Elem.S) {
          brk -> atom[i].r = AtmTab -> VdwRad;
          break;
        }
      } /* End of for */
      if (!*AtmTab -> Elem.C) {
        CatWarn(MSG_WARN_UNKELEM, brk -> atom[i].Elem.C, LineNum);
        brk -> atom[i].r = ES_DEFAULT_VDWRAD;
      }


      /**** Find the residues ****/

      for(aminoacido = 0; (Residues[aminoacido].L) &&
                          (Residues[aminoacido].L != brk -> atom[i].ResName.L); ++aminoacido);

      if (Residues[aminoacido].L == 0) {
        CatWarn(MSG_WARN_UNKRES, brk -> atom[i].ResName.C, LineNum);
        aminoacido = 5;
      }

      /**** Assign the charges ****/

      carica = 6;
      for(k = 0; tab[k].t.L; ++k) {
        if ((brk -> atom[i].Name.L == tab[k].t.L) && (tab[k].i <= aminoacido) &&
            (tab[k].f >= aminoacido)) {
          carica = tab[k].c;
          break;
        }
      } /* End of for */

      brk -> atom[i].carica = car[carica].car;
      brk -> atom[i].valore = car[carica].val;
      ++i;
    } else if ((i) && (PDBLin.Hdr == *(ES_LONG *)"TER ")) {
      brk -> atom[i - 1].Flags |= ES_ATMF_SEGEND;
    }
    ++LineNum;
  } /* End of while */

  fclose(f);

  return TRUE;
}


/**** Write the PDB file ****/

ES_BOOL BrkWrite(char* name, ES_BRK *brk)
{
  char          *j;
  ES_QCHAR      Name;
  ES_ULONG      i, k, l;
  FILE          *OUT;

  ES_BOOL       Ret = TRUE;


  if ((OUT = fopen(name, "w")) != NULL) {
    if (fprintf(OUT, "REMARK   4\n" \
                     "REMARK   4 File generated by ESCHER NG\n"\
                     "REMARK   4\n") >= 0) {
      for(i = 0, k = 1; (Ret) && (i < (ES_ULONG)brk -> n_atoms); ++i, ++k) {
        if (fprintf(OUT, "%-6s%5d ",
                    PDBAtmRec[brk -> atom[i].Flags & ES_ATMF_HETATM],
                    k) >= 0) {
          l = brk -> atom[i].Name.C[3];
          if (brk -> atom[i].Elem.C[1]) {
            if (fprintf(OUT, "%-4.3s", brk -> atom[i].Name.C) < 0) {
              Ret = PrintDosErr();
              break;
            }
          } else if (isdigit(l)) {
            Name.L = brk -> atom[i].Name.L;
            for(j = Name.C + 3;j > Name.C; --j)
              *j = *(j - 1);
            *Name.C = (char)l;
            if (fprintf(OUT, "%-4.4s", Name.C) < 0) {
              Ret = PrintDosErr();
              break;
            }
          } else if (fprintf(OUT, " %-3.3s", brk -> atom[i].Name.C) < 0) {
            Ret = PrintDosErr();
            break;
          }
          if (fprintf(OUT, " %-3.3s %c%4.4s    %8.3f%8.3f%8.3f  1.00  0.00\n",
                      brk -> atom[i].ResName.C, brk -> atom[i].ChainID, brk -> atom[i].ResSeq.C,
                      brk -> atom[i].x, brk -> atom[i].y, brk -> atom[i].z) >= 0) {
            if (brk -> atom[i].Flags & ES_ATMF_SEGEND) {
              ++k;
              if (fprintf(OUT, "TER   %5d      %-3.3s %c%4.4s\n",
                          k, brk -> atom[i].ResName.C, brk -> atom[i].ChainID, brk -> atom[i].ResSeq.C) <= 0) {
                Ret = PrintDosErr();
              }
            }
          } else Ret = PrintDosErr();
        } else Ret = PrintDosErr();
      } /* End of for */

      if (Ret)
        fprintf(OUT, "END\n");
    }
    fclose(OUT);
  }

  return Ret;
}


/**** Translate the molecule ****/

void brk_translate(ES_BRK *brk, float x, float y, float z)
{
  ES_ULONG              i;

  for(i = 0; i < brk -> n_atoms; i++) {
    brk -> atom[i].x += x;
    brk -> atom[i].y += y;
    brk -> atom[i].z += z;
  } /* End of for */
}


/**** X rotate the molecule ****/

void brk_x_rotate(ES_BRK *brk, float rx)
{
  ES_ULONG              i;
  float                 raggio, angolo;

  for(i = 0; i < brk -> n_atoms; i++) {
    raggio           = sqrtf(brk -> atom[i].z * brk -> atom[i].z + brk -> atom[i].y * brk -> atom[i].y);
    angolo           = rx + atan2f(brk -> atom[i].y, brk -> atom[i].z);
    brk -> atom[i].z = raggio * cosf(angolo);
    brk -> atom[i].y = raggio * sinf(angolo);
  } /* End of for */
}


/**** Y rotate the molecule ****/

void brk_y_rotate(ES_BRK *brk, float ry)
{
  ES_ULONG              i;
  float                 raggio, angolo;

  for(i = 0; i < brk -> n_atoms; i++) {
    raggio           = sqrtf(brk -> atom[i].z * brk -> atom[i].z + brk -> atom[i].x * brk -> atom[i].x);
    angolo           = ry + atan2f(brk -> atom[i].x, brk -> atom[i].z);
    brk -> atom[i].z = raggio * cosf(angolo);
    brk -> atom[i].x = raggio * sinf(angolo);
  } /* End of for */
}


/**** Z rotate the molecule ****/

void brk_z_rotate(ES_BRK *brk, float rz)
{
  ES_ULONG               i;
  float                  raggio, angolo;

  for(i = 0; i < brk -> n_atoms; i++) {
    raggio           = sqrtf(brk -> atom[i].x * brk -> atom[i].x + brk -> atom[i].y * brk -> atom[i].y);
    angolo           = rz + atan2f(brk -> atom[i].y, brk -> atom[i].x);
    brk -> atom[i].x = raggio * cosf(angolo);
    brk -> atom[i].y = raggio * sinf(angolo);
  } /* End of for */
}


/**** Calculate the RMS between two structures ****/

float brk_rms(ES_BRK *b1, ES_BRK *b2)
{
  ES_ULONG      i;

  float         rms = 0.0f;

  if (b1 -> n_atoms != b2 -> n_atoms) {
    fprintf(stderr, "Errore Rms con differente numero di atomi\n");
    exit(1);
  }

  for(i = 0; i < b1 -> n_atoms; i++) {
    rms += sqrtf(square(b1 -> atom[i].x - b2 -> atom[i].x)+
                 square(b1 -> atom[i].y - b2 -> atom[i].y)+
                 square(b1 -> atom[i].z - b2 -> atom[i].z));
  } /* End of for */

  return (rms / b1->n_atoms);
}


/**** Calculate the contacts ****/

float brk_contacts(ES_BRK *b1, ES_BRK *b2, float maxdist)
{
  ES_ULONG      i, j;
  float         r1, dist;
  float         dx, dy, dz;

  ES_ATOM       *a1      = b1 -> atom;
  ES_ATOM       *a2      = b2 -> atom;
  float         contacts = 0.0f;

  for(i = 0; i < b1 -> n_atoms; i++) {
    r1 = a1[i].r + maxdist;
    j  = 0;
    do {
      dx   = a1[i].x - a2[j].x;
      dy   = a1[i].y - a2[j].y;
      dz   = a1[i].z - a2[j].z;
      dist = dx * dx + dy * dy + dz * dz;
      if (dist < square(r1 + a2[j].r)) contacts += 1;
      j++;
    } while(j < b2 -> n_atoms);
  }

  return contacts;
}

