
/*********************************************************************
****             ESCHER Next Generation - Main code               ****
**** (c) 1997-2012, G. Ausiello, G. Cesareni, M. Helmer Citterich ****
****               NG version by Alessandro Pedretti              ****
*********************************************************************/


#ifdef __WIN32__
#  define  DXWCPU_C
#  include <windows.h>
#  include <dxwcpu.h>

#  ifdef __GNUC__
#    include <unistd.h>
#  endif
#endif

#ifdef __BORLANDC__
#  include <dir.h>
#endif

#include "my_macros.h"

#ifndef __WIN32__
#  include <unistd.h>
#  include <errno.h>
#endif

#include "sol.h"
#include "brk.h"
#include "cpu.h"
#include "charges.h"
#include "getopt.h"
#include "main.h"
#include "globstr.h"
#include "mthread.h"
#include "prot.h"
#include "surf.h"
#include "surf2prot.h"
#include "shapes.h"
#include "utils.h"

static ES_PREFS Prefs        = {{ "auto" },          /* Default language           */
                               };

#ifdef AMIGA
static char     *AmigaVer    = "$VER:ESCHER "ES_VER" ("ES_DATE")";
#endif


/**** Global variables ****/

char            nome_p1[MAX_PATH];
char            nome_p2[MAX_PATH];

ES_LONG         CpuToUse;                       /* CPU to use in parallel calc. */
ES_LONG         col_attivo       = 120;         /* Surface active color         */
FILE            *OutFH           = NULL;        /* Default output handle        */
float           lunghezza_lato   = 1.5f;
float           spessore_sezione = 1.5f;        /* Slice tickness               */
void            *Catalog         = NULL;        /* Catalog translation handle   */

#if defined(WIN32) || defined(Linux)
DXWCPU                  EsCpu;                  /* Cpu information              */
#endif

/**** Local variables ****/

static char     nome_sol1[MAX_PATH];
static char     nome_sol2[MAX_PATH];

static ES_LONG  startx           = -175;        /* Starting X rotation            */
static ES_LONG  endx             = 175;         /* Ending X rotation              */
static ES_LONG  starty           = -85;         /* Starting Y rotation            */
static ES_LONG  endy             = 85;          /* Ending Y rotation              */
static ES_LONG  stepx            = 10;          /* X rotation step                */
static ES_LONG  stepy            = 10;          /* Y rotation step                */
static ES_LONG  f_shapes         = 1;           /* Activate shapes calculation    */
static ES_LONG  only_charges     = 0;         /* 1 = electrostatic evaluation   */
static ES_LONG  n_soluzioni      = 1000;        /* Number of the solutions        */
static ES_LONG  sol_gen          = -1;          /* Solution number to extract     */
static float    larghezza_maglie = 1.0f;
static float    perimetro_min    = 20.0f;
static float    maxdist_polar    = 3.4f;        /* Polar cutoff distance          */
static float    maxdist_apolar   = 4.0f;        /* Apolar cutoff distance         */
static float    overlap          = 0.1f;        /* Van Der Waal overlapping       */
static float    rms_rag          = 2.5f;        /* Max Rms to cluster the sol.    */
static float    bu_max           = 200.0f;      /* Bump treshold                  */
static float    ca_max           = -200.0f;     /* Charge treshold                */
static ES_ULONG SrfDots          = ES_DEFDOTS;  /* Dot density for surface calc.  */
static float    SrfProbeRad      = ES_DEFPRAD;  /* Probe radius for surface calc. */
static ES_BOOL  SrfForceCalc     = FALSE;       /* Force the surface calculation  */

/**** Parallel execution variables ****/

static ES_BRK                   MthBrk1, MthBrk2;
static ES_LONG                  MthK;
static ES_SOLUTIONS             *MthS;
static struct Proteina          *MthP1;
static struct surf              *MthS2;

/**** Local prototypes ****/

static ES_BOOL          ReadParam(int, char **);
static MCL_THREAD       ChargesThread(void *);
static MCL_THREAD       DockingThread(void *);


/**** Main code ****/

int main(int argc, char* argv[])
{
  char                  nome_brksol[MAX_PATH];
  char                  nome_brk2[MAX_PATH];
  ES_BRK                *BrkSol;
  ES_BRK                brk2;
  ES_BOOL               Ret;
  ES_SOLUTIONS          *s, *s2, *s3;
  ES_ULONG              k;

  OutFH = stdout;

#ifdef WIN32
  DXWCPU_GetInfo(&EsCpu);
  CpuToUse = k = EsCpu.m_CPU_Number;
#else
  CpuToUse = k = MCL_GetCPUs();
#endif

  if (GetPrgPath(nome_brksol, TRUE)) {
    strcat(nome_brksol, ES_PRF_FILE);
    LoadPrefs(&Prefs, nome_brksol);

    CatInit(Prefs.Language);

    printf("\nESCHER NG " ES_VER " - (c) " ES_COPYRIGHT \
         ", G. Ausiello, G. Cesareni, M. Helmer Citterich\n");
    CatPrintf(stdout, MSG_VERSION,
              "A. Pedretti and G. Vistoli",
              ES_VERSION " - " MYCPU);
    if (Catalog) printf("%s\n\n", GetStr(MSG_TRANSLATION));
    else printf("\n");

    /**** Read the command line parameters ****/

    if ((Ret = ReadParam(argc, argv)) == TRUE) {

      /**** Cpu detection ****/

#if defined(WIN32)
      CatPrintf(stdout, MSG_CPU_TYPEWIN, k, EsCpu.m_ProcVendorString, EsCpu.m_ProcName);
#else
      CatPrintf(stdout, MSG_CPU_TYPE, k);
#endif

      if (CpuToUse > 1) CatPrintf(stdout, MSG_CPU_PARALLEL, CpuToUse);
      else printf("\n");

      /**** File names ****/

      sprintf(nome_sol1, "%s-%s", nome_p1, nome_p2);
      strcpy(nome_sol2, nome_sol1);
      strcat(nome_sol1, "_1.sol");
      strcat(nome_sol2, "_2.sol");

      /**** Generates a PDB file starting from a solution ****/

      if (sol_gen > 0) {
        sprintf(nome_brk2, "%s.pdb", nome_p2);
        if ((Ret = BrkRead(nome_brk2, &brk2)) == TRUE) {
          CatPrintf(stdout, MSG_READ_SOL);
          if ((s = SolutionsRead(nome_sol2)) != NULL) {
            if ((ES_LONG)s -> n_sol >= sol_gen) {
              CatPrintf(stdout, MSG_WRITE_SOL, sol_gen);
              if ((BrkSol = BrkSolutions(&brk2, s, sol_gen - 1)) != NULL) {
                sprintf(nome_brksol, "%s-%s_%d.pdb",
                        GetFileName(nome_p1), GetFileName(nome_p2), sol_gen);
                Ret = BrkWrite(nome_brksol, BrkSol);
              } else Ret = FALSE;
            } else Ret = CatErr(MSG_ERR_SOL_IDEXC, s -> n_sol);
            SolutionsDelete(&s);
          } else Ret = FALSE;
        }
        if (Ret) CatPrintf(stdout, MSG_DONE);
        CatClose();
        exit(0);
      }

      /**** Save the solutions ****/

      if (f_shapes == 1) {
        if ((s = SolutionsNew(n_soluzioni)) != NULL) {
          if (only_charges == 0) Ret = Shapes(s);
          else Ret = TRUE;
          if (Ret) {
            CatPrintf(stdout, MSG_EVALELECTINT);
            if ((Ret = Charges(s)) == TRUE) {
              CatPrintf(stdout, MSG_SAVINGSOL);
              if ((s2 = SolutionsSort(s)) != NULL) {
                Ret = SolutionsWrite(s2, nome_sol1);
                SolutionsDelete(&s2);
              } else Ret = FALSE;
            }
            SolutionsDelete(&s);
          }
        } else Ret = FALSE;
      }

      /**** Sort the solutions ****/

      if ((Ret) && (only_charges == 0)) {
        CatPrintf(stdout, MSG_SORTINGSOL);
        if ((s = SolutionsRead(nome_sol1)) != NULL) {
          if ((s2 = SolutionsSort(s)) != NULL) {
            if ((s3 = raggruppa(s2, rms_rag, bu_max, ca_max)) != NULL) {
              Ret = SolutionsWrite(s3, nome_sol2);
              SolutionsDelete(&s3);
            } else Ret = FALSE;
            SolutionsDelete(&s2);
          } else Ret = FALSE;
          SolutionsDelete(&s);
        } else Ret = FALSE;
      }

      if (Ret) CatPrintf(stdout, MSG_DONE);
    }

    CatClose();
  }

  return 0;
}


/**** Update the solutions with bumbs and electrostatic data ****/

ES_BOOL Charges(ES_SOLUTIONS *s)
{
  ES_BOOL       Ret;
  char          nome_brk1[MAX_PATH], nome_brk2[MAX_PATH];

  /**** Read the target and the probe protein ****/

  sprintf(nome_brk2, "%s.pdb", nome_p2);
  sprintf(nome_brk1, "%s.pdb", nome_p1);

  if ((Ret = BrkRead(nome_brk2, &MthBrk2)) == TRUE) {
    if ((Ret = BrkRead(nome_brk1, &MthBrk1)) == TRUE) {
      MthS = s;
      MthK = -1;
      if (CpuToUse > 1) {
        if ((Ret = (ES_BOOL)MCL_CreateMutex()) != 0) {
          if (MCL_CreateThreads(CpuToUse, (int (*)(void *))ChargesThread, NULL, MCL_THREADS_WAIT))
            Ret = FALSE;
          MCL_CloseMutex();
        }
      } else Ret = (ES_BOOL)!ChargesThread(NULL);
      BrkDelete(&MthBrk1);
    }
    BrkDelete(&MthBrk2);
  }

  return Ret;
}


/**** Charges thread ****/

static MCL_THREAD ChargesThread(void *Param)
{
  ES_BRK        *BrkSol;
  ES_ULONG      i;
  ES_ATMFLAG    *FlBrk1, *FlBrkSol;

  int           Err = 0;

  if ((FlBrk1 = (ES_ATMFLAG *)Alloca(MthBrk1.n_atoms * sizeof(ES_ATMFLAG))) != NULL) {
    if ((FlBrkSol = (ES_ATMFLAG *)Alloca(MthBrk2.n_atoms * sizeof(ES_ATMFLAG))) != NULL) {
      for(;;) {
        MCL_MutexOn();
        if ((MthK + 1) >= (ES_LONG)MthS -> n_sol) {
          MCL_MutexOff();
          break;
        } else ++MthK;
        i = MthK;
        MCL_MutexOff();

        if ((BrkSol = BrkSolutions(&MthBrk2, MthS, i)) != NULL) {

          /**** RMS evaluation between probe and solution ****/

          MthS -> sol[i].rms     = brk_rms(&MthBrk2, BrkSol);

          /**** Bump-check (target-solution) ****/

          MthS -> sol[i].bumps   = brk_bumps(&MthBrk1, BrkSol, FlBrk1, FlBrkSol, overlap);

          /**** Actractive electrostatic interactions ****/

          MthS -> sol[i].pos_neg = brk_pos_neg(&MthBrk1, BrkSol, FlBrk1, FlBrkSol, maxdist_polar);

          /**** Apolar interactions ****/

          MthS -> sol[i].apo_apo = brk_apo_apo(&MthBrk1, BrkSol, FlBrk1, FlBrkSol, maxdist_apolar);

          /**** Repulsive interactions ****/

          MthS -> sol[i].pos_pos = brk_pos_pos(&MthBrk1, BrkSol, FlBrk1, FlBrkSol, maxdist_polar);
          MthS -> sol[i].apo_pos = brk_apo_pos(&MthBrk1, BrkSol, FlBrk1, FlBrkSol, maxdist_polar);

          /**** Total charge score ****/

          MthS -> sol[i].charge  = MthS -> sol[i].pos_neg + MthS -> sol[i].apo_apo;
          MthS -> sol[i].charge -= MthS -> sol[i].pos_pos + MthS -> sol[i].apo_pos;
          BrkDelete(BrkSol);
        } else {
          Err = 1;
          break;
        }
      } /* End of for */
      FREE(FlBrkSol);
    } else Err = 1;
    FREE(FlBrk1);
  } else Err = 1;

  return Err;
}


/**** Docking thread ****/

static MCL_THREAD DockingThread(void *Param)
{
  struct surf           s2r;
  struct Proteina       *p1ext, *p2;
  int                   angolox, angoloy;

  int                   Err = 0;

  /**** Rotations ****/

    for(;;) {
      MCL_MutexOn();
      if ((MthS -> angoloy + stepy) >= endy) {
        MthS -> angoloy = starty;
        MthS -> angolox += stepx;
      } else MthS -> angoloy += stepy;
      if (MthS -> angolox > endx) {
        MCL_MutexOff();
        break;
      }
      angolox = MthS -> angolox;
      angoloy = MthS -> angoloy;
      MCL_MutexOff();

      /**** Probe orientation ****/

      CatPrintf(stdout, MSG_DOCK_UPDATE, angolox, angoloy);
      if (!SrfCopy(&s2r, MthS2)) break;
      SrfTranslate(&s2r, - MthS -> centrox, - MthS -> centroy, - MthS -> centroz); /* sovrapposizione dei baricentri del target e del probe*/
      SrfRotX(&s2r, angolox * (M_PI / 180.0f));              /* orientamento del probe in x*/
      SrfRotY(&s2r, angoloy * (M_PI / 180.0f));                   /* orientamento del probe in y*/
      SrfTranslate(&s2r, MthS -> centrox, MthS -> centroy, MthS -> centroz);       /* il probe ritorna nella posizione originale*/

      p2 = Surf2Prot(&s2r, spessore_sezione);                   /* trasformazione di target.surf in poligonali*/
      prot_ordina(p2, larghezza_maglie, perimetro_min);                     /* ordinamento dei lati delle poligonali*/
      prot_lati(p2, lunghezza_lato);                          /* omogeneizzazione della lunghezza dei lati delle poligonali*/
      prot_angoli(p2);                                                /* calcolo degli angoli interni della poligonale (necessari per il confronto*/

      p1ext = proteina_copy(MthP1);                               /* copia del target da estendere mantenedo intatto l'originale*/
      prot_estendi(p1ext, estensione(MthP1, p2));                             /* estensione dei lati del target necessario al confronto*/
      prot_estendi(p2, estensione(MthP1, p2));                              /* estensione dei lati del probe necessario al confronto*/

      ShapeAnalyze(p1ext, p2, MthS);                                        /* confronto delle proteine e salvataggio delle soluzionin nella struct 's' */
      proteina_delete(p1ext);
      SrfDelete(&s2r);
      proteina_delete(p2);
    } /* End of for */

  return Err;
}


/**** Read the command parameters ****/

ES_BOOL ReadParam(int argc, char **argv)
{
  char          *Ptr;
  char          Path[MAX_PATH];
  ES_BOOL       Skip;
  ES_LONG       TmpI;
  ES_LONG       i;
  float         TmpF;

  const char    *OptStr = "a:b:c:d:efgi:l:p:r:s:z";
  ES_BOOL       Ret     = TRUE;

  if (argc == 1) {
    CatPrintf(stderr, MSG_OPTIONS);
    return FALSE;
  }

  if (argc < 3)        return CatErr(MSG_ERR_OPT_TOOFEWPAR);
  if (*argv[1] == '-') return CatErr(MSG_ERR_OPT_TARGET);
  if (*argv[2] == '-') return CatErr(MSG_ERR_OPT_PROBE);

  Skip = FALSE;
  TmpI = 0;
  for(i = 1; i < argc; ++i) {
    if (*argv[i] == '-') {
      Skip = TRUE;
      if (((Ptr = strchr(OptStr, argv[i][1])) != NULL) && (Ptr[1] != ':'))
        Skip = FALSE;
    } else {
      if (Skip) Skip = FALSE;
      else ++TmpI;
    }
  } /* End of for */
  if (TmpI > 2) return CatErr(MSG_ERR_OPT_TOOMANYFILES);

  /**** Change the current directory to target molecule ****/

  GetFilePath(Path, argv[1]);
  if ((*Path) && (chdir(Path) == -1))
    return PrintDosErr();

  strcpy(nome_p1, GetFileName(argv[1]));
  strcpy(nome_p2, GetFileName(argv[2]));

  /**** Parse the options ****/

  opterr = 0;
  while ((i = getopt(argc, argv, OptStr)) != EOF) {
    TmpI = 0;
    if (optarg)
      sscanf(optarg, "%d", &TmpI);
    switch(i) {
    case 'a':           /* Active color */
      if ((TmpI > 0) && (TmpI < 256))
        col_attivo = TmpI;
      else Ret = CatErr(MSG_ERR_OPT_ACTCOLOR);
      break;

    case 'b':           /* Set the max bumps value */
      if (TmpI >= 0)
        bu_max = TmpI;
      else Ret = CatErr(MSG_ERR_OPT_BUMPS);
      break;

    case 'c':           /* Set the min charge value */
      ca_max = TmpI;
      break;

    case 'd':           /* Dot density for surface calculation */
      if (TmpI >= ES_MINSRFDOTS)
        SrfDots = TmpI;
      else Ret = CatErr(MSG_ERR_OPT_SRFDOTS, ES_MINSRFDOTS);
      break;

    case 'e':           /* Elecreostatic evaluation only */
      only_charges = 1;
      break;

    case 'f':           /* Force the surface calculation */
      SrfForceCalc = TRUE;
      break;

    case 'g':           /* Run the clustering only */
      f_shapes = 0;
      break;

    case 'i':           /* Probe radius for surface calculation */
      sscanf(optarg, "%f", &TmpF);
      if (TmpF >= 0.0f)
        SrfProbeRad = TmpF;
      else Ret = CatErr(MSG_ERR_OPT_SRFPRORAD);
      break;

    case 'l':           /* Number of solution to calculate */
      if (TmpI > 0)
        n_soluzioni = TmpI;
      else Ret = CatErr(MSG_ERR_OPT_SOL);
      break;

    case 'p':           /* Number of CPUs */
      if ((TmpI > 0) && (TmpI <= CpuToUse))
        CpuToUse = TmpI;
      else Ret = CatErr(MSG_ERR_OPT_CPUS, CpuToUse);
      break;

    case 'r':           /* Rotation step */
      if ((TmpI > 0) && (TmpI < 180)) {
        stepx = TmpI;
        stepy = TmpI;
      } else Ret = CatErr(MSG_ERR_OPT_ROTSTEP);
      break;


    case 's':           /* Solution to extract */
      if (TmpI >= 1)
        sol_gen = TmpI;
      else Ret = CatErr(MSG_ERR_OPT_SOLEXT);
      break;

    case 'z':           /* Z rotation only */
      startx = 0;
      starty = 0;
      endx   = 0;
      endy   = 0;
      break;

    case '?':
      if (strchr(OptStr, optopt) != NULL)
        Ret = CatErr(MSG_ERR_OPT_ARGMIS, optopt);
      else
        Ret = CatErr(MSG_ERR_OPT_UNKNOWN, optopt);
      break;
    } /* End of switch */
  } /* End of while */

  return Ret;
}


/**** Generate the solutions (parallel execution) ****/

ES_BOOL Shapes(ES_SOLUTIONS *s)
{
  ES_BOOL               Ret;
  ES_BRK                Brk;
  char                  nome_surf1[MAX_PATH], nome_surf2[MAX_PATH];
  struct surf           s1, s2;
  struct Proteina       *p1;

  /**** Target protein ****/

  strcpy(nome_surf1, nome_p1);
  strcat(nome_surf1, ".srf");
  if ((SrfForceCalc) || (!FileExists(nome_surf1))) {
    CatPrintf(stdout, MSG_SRF_CALCTARGET);
    strcpy(nome_surf2, nome_p1);
    strcat(nome_surf2, ".pdb");
    if (BrkRead(nome_surf2, &Brk)) {
      Ret = SrfCalc(nome_surf1, &Brk, SrfDots, SrfProbeRad);
      BrkDelete(&Brk);
      if (!Ret) return FALSE;
    } else return FALSE;
  }

  CatPrintf(stdout, MSG_SRF_READTARGET);
  if ((Ret = SrfRead(nome_surf1, &s1)) == TRUE) { /* lettura del file target.surf*/
    p1 = Surf2Prot(&s1, spessore_sezione);    /* trasformazione di target.surf in poligonali*/
    prot_ordina(p1, larghezza_maglie, perimetro_min); /* ordinamento dei lati delle poligonali*/
    prot_lati(p1, lunghezza_lato);      /* omogeneizzazione della lunghezza dei lati delle poligonali*/
    prot_inverti(p1);         /* l'ordine dei lati del target viene invertito di senso
                    per permettere il confronto con i lati del probe*/
    prot_angoli(p1);                /* calcolo degli angoli interni della poligonale (necessari per il confronto*/

    /**** Probe protein ****/

    strcpy(nome_surf2, nome_p2);
    strcat(nome_surf2, ".srf");
    if ((SrfForceCalc) || (!FileExists(nome_surf2))) {
      CatPrintf(stdout, MSG_SRF_CALCPROBE);
      strcpy(nome_surf1, nome_p2);
      strcat(nome_surf1, ".pdb");
      if ((Ret = BrkRead(nome_surf1, &Brk)) == TRUE) {
        Ret = SrfCalc(nome_surf2, &Brk, SrfDots, SrfProbeRad);
        BrkDelete(&Brk);
      }
      if (!Ret) {
        SrfDelete(&s1);
        return FALSE;
      }
    }

    CatPrintf(stdout, MSG_SRF_READPROBE);
    if ((Ret = SrfRead(nome_surf2, &s2)) == TRUE) {   /* lettura del file probe.surf*/
      CatPrintf(stdout, MSG_DOCK_STARTED);
      if ((Ret = InitDock(&s1, &s2, lunghezza_lato, s)) == TRUE) {
        MthS = s;
        MthS -> angolox = startx;
        MthS -> angoloy = starty - stepy;
        MthP1 = p1;
        MthS2 = &s2;
        if (CpuToUse > 1) {
          if ((Ret = (ES_BOOL)MCL_CreateMutex()) != 0) {
            if (MCL_CreateThreads(CpuToUse, (int (*)(void *))DockingThread, NULL, MCL_THREADS_WAIT))
              Ret = FALSE;
            MCL_CloseMutex();
          }
        } else Ret = (ES_BOOL)!DockingThread(NULL);
        printf("\n");
        FreeDock();
      }
    }
  }

  return Ret;
}


/**** Clusterize the solutions ****/

ES_SOLUTIONS *raggruppa(ES_SOLUTIONS *s, float rms_rag, float bu_max, float ca_max)
{
  char                  nome_brk2[MAX_PATH];
  ES_BRK                brk;
  ES_BRK                *BrkSol;
  ES_SOLUTIONS          *s2;
  ES_ULONG              i, j, k, good;
  FILE                  *g;
  float                 dx, dy, dz, val;
  float                 *gab1, *gab2;

  ES_BOOL               Ret = TRUE;

  if ((s2 = SolutionsCopy(s)) != NULL) {
    s2 -> n_sol = 0;
    strcpy(nome_brk2, nome_p2);
    strcat(nome_brk2, ".pdb");
    if (BrkRead(nome_brk2, &brk)) {
      if ((gab1 = (float*)Alloca(brk.n_atoms * 3 * sizeof(float))) != NULL) {
        if ((gab2 = (float*)Alloca(brk.n_atoms * 3 * sizeof(float))) != NULL) {
          for(i = 0; (Ret) && (i < s -> n_sol); i++) {
            if ((s -> sol[i].charge > ca_max) && (s -> sol[i].bumps < bu_max)) {
              if ((BrkSol = BrkSolutions(&brk, s, i)) == NULL) {
                Ret = FALSE;
                break;
              }
              for(j = 0; j < BrkSol -> n_atoms; j++) {
                gab1[j * 3]     = BrkSol -> atom[j].x;
          gab1[j * 3 + 1] = BrkSol -> atom[j].y;
          gab1[j * 3 + 2] = BrkSol -> atom[j].z;
              } /* End of for */

              BrkDelete(BrkSol);

              good = 1;
              if ((g = fopen("raggruppa.dat", "rb")) != NULL) {
                for(j = 0; (Ret) && (j < s2 -> n_sol); j++) {
                  if (fread(gab2, brk.n_atoms * 3, sizeof(float), g) == sizeof(float)) {
                    val = 0;
                    for(k = 0; k < brk.n_atoms * 3;) {
                      dx   = gab1[k] - gab2[k];k++;
                      dy   = gab1[k] - gab2[k];k++;
                      dz   = gab1[k] - gab2[k];k++;
                      val += dx * dx + dz * dz + dy * dy;
                    } /* End of for */

                    if ((val / brk.n_atoms) < (rms_rag*rms_rag)) {
                      good = 0;
                      break;
                    }
                  } else Ret = PrintDosErr();
                } /* End of for */
                fclose(g);
              }

              if ((Ret) && (good)) {
                SolutionCopy(&s2 -> sol[s2 -> n_sol++], &s -> sol[i]);
                if ((g = fopen("raggruppa.dat", "ab")) != NULL) {
                  if (fwrite(gab1, brk.n_atoms * 3, sizeof(float), g) != sizeof(float))
                    Ret = PrintDosErr();
                  fclose(g);
                } else Ret = PrintDosErr();
              }
            }
          } /* End of for */
          remove("raggruppa.dat");
          FREE(gab1);
        }
        FREE(gab2);
      }
    }
    if (!Ret) SolutionsDelete(&s2);
  }

  return s2;
}

