
/*************************************************
**** Universal Multithreading Control Library ****
****    (c) 2003-2012, Alessandro Pedretti    ****
*************************************************/


#include "mthread.h"

#ifdef MCL_PTHREAD
#  include <pthread.h>

static pthread_mutex_t  MthH;
static pthread_cond_t   *StartEvent = NULL;
#endif

#ifdef MCL_LINUX
#  include <stdio.h>
#endif

#ifdef MCL_SGI
#  include <unistd.h>
#  include <invent.h>
#  include <abi_mutex.h>
#  include <ulocks.h>
#  include <sys/prctl.h>
#  include <sys/types.h>
#  include <sys/wait.h>

static abilock_t        MthH;
static barrier_t        *BarrierH;
static int              ThreadsToWait;
static usptr_t          *UsH;
#endif

#ifdef MCL_SUN
#  include <unistd.h>
#endif

#ifdef MCL_WIN32
#  include <windows.h>

static HANDLE           MthH       = NULL;
static HANDLE           StartEvent = NULL;
#endif


/**** Close mutex ****/

void MCL_CloseMutex(void)
{
#ifdef WIN32
  if (MthH) {
    CloseHandle(MthH);
    MthH = NULL;
  }
#endif
}


/**** Create mutex ****/

int MCL_CreateMutex(void)
{

#ifdef MCL_PTHREAD
  pthread_mutex_init(&MthH, NULL);
#endif

#ifdef sgi
  if (init_lock(&MthH)) return 0;
#endif

#ifdef WIN32
  if ((MthH = CreateMutex(NULL, FALSE, NULL)) == NULL)
    return 0;
#endif

  return 1;
}


/**** Creates paralle threads ****/

int MCL_CreateThreads(int Cpu, int (*Routine)(void *), void *Args, int Flags)
{
  int                           Err = 0;

#ifdef MCL_PTHREAD
  int                           k, ThreadId;
  pthread_t                     ThreadHandle[MCL_MAX_CPU];
#endif

#ifdef MCL_SGI
  int                           k, WaitAddr;
  int                           ThreadHandle[MCL_MAX_CPU];
#endif

#ifdef MCL_WIN32
  LPTHREAD_START_ROUTINE        MyRoutine = (LPTHREAD_START_ROUTINE)Routine;
  DWORD                         ThreadId;
  int                           k;
  HANDLE                        ThreadHandle[MCL_MAX_CPU];
#else
  int                           (*MyRoutine)(void *) = (void *)Routine;
#endif


  if (!Cpu) Cpu = MCL_GetCPUs();
  if (Cpu > 1) {

#ifdef MCL_PTHREAD
    for(k = 0; k < Cpu; ++k) {
      if ((ThreadId = pthread_create(&ThreadHandle[k], NULL, (void * (*)(void *))Routine, Args)) != 0) {
        return 1;
      }
    } /* End of for */

    if (Flags & MCL_THREADS_WAIT) {
      for(k = 0; k < Cpu; ++k) Err |= pthread_join(ThreadHandle[k], NULL);
    }

#endif

#ifdef MCL_SGI
    BarrierH      = NULL;
    ThreadsToWait = 0;
    UsH           = NULL;
    if (Flags & MCL_THREADS_SYNC) {
      UsH = usinit("dumb1");
      BarrierH = new_barrier(UsH);
      init_barrier(BarrierH);
      ThreadsToWait = Cpu;
    }

    for(k = 0; k < Cpu; ++k) {
      if ((ThreadHandle[k] = sproc((void (*)(void *))Routine, PR_SADDR, Args)) == 0) {
        return 1;
      }
    } /* End of for */

    if (Flags & MCL_THREADS_WAIT) while(wait(&WaitAddr) > 0);

#endif

#ifdef MCL_WIN32
    if (Flags & MCL_THREADS_SYNC) {
      if ((StartEvent = CreateEvent(NULL, TRUE, FALSE, NULL)) == NULL)
        return 1;
      ResetEvent(StartEvent);
    }
    for(k = 0; k < Cpu; ++k) {
      if ((ThreadHandle[k] = CreateThread(NULL,
                                          0,
                                          (LPTHREAD_START_ROUTINE)Routine,
                                          Args, 0, &ThreadId)) == NULL) {
        return 1;
      }
    } /* End of for */
    if (Flags & MCL_THREADS_SYNC)
      SetEvent(StartEvent);
    if (Flags & MCL_THREADS_WAIT) {
      WaitForMultipleObjects(Cpu, ThreadHandle, TRUE, INFINITE);
      for(k = 0; k < Cpu; ++k) {
        GetExitCodeThread(ThreadHandle[k], &ThreadId);
        Err |= ThreadId;
      } /* End of for */
    }
#endif

  } else Err = MyRoutine(Args);

  return Err;
}


/**** Get the number of CPUs ****/

int MCL_GetCPUs(void)
{

#ifdef MCL_WIN32
  SYSTEM_INFO   SysInfo;

  ZeroMemory(&SysInfo, sizeof(SYSTEM_INFO));
  GetSystemInfo(&SysInfo);

  return SysInfo.dwNumberOfProcessors;
#else

#  ifdef MCL_SGI
  return sysconf(_SC_NPROC_ONLN);
#  else

#    ifdef MCL_SUN
  return sysconf(_SC_NPROCESSORS_ONLN);
#    else
#      ifdef MCL_LINUX
  FILE          *FH;
  int           Cpus = 1;

  if ((FH = popen("grep '^processor' /proc/cpuinfo| wc -l", "r")) != NULL) {
    fscanf(FH, "%d", &Cpus);
    if (Cpus < 1) Cpus = 1;
    pclose(FH);
  }

  return Cpus;
#      else
  return MCL_MAX_CPU;
#      endif
#    endif
#  endif
#endif
}


/**** Disable mutex ****/

void MCL_MutexOff(void)
{
#ifdef MCL_PTHREAD
  pthread_mutex_unlock(&MthH);
#endif

#ifdef MCL_SGI
  release_lock(&MthH);
#endif

#ifdef MCL_WIN32
  if (MthH) ReleaseMutex(MthH);
#endif
}


/**** Activate mutex ****/

void MCL_MutexOn(void)
{
#ifdef MCL_PTHREAD
  pthread_mutex_lock(&MthH);
#endif

#ifdef MCL_SGI
  spin_lock(&MthH);
#endif

#ifdef MCL_WIN32
  if (MthH) WaitForSingleObject(MthH, INFINITE);
#endif
}


/**** Wait the start signal ****/

void MCL_StartWait(void)
{

#ifdef MCL_SGI
  if (ThreadsToWait) barrier(BarrierH, ThreadsToWait);
#endif

#ifdef MCL_WIN32
  WaitForSingleObject(StartEvent, INFINITE);
#endif

}
