
/*************************************************
****        VEGA - x86 CPU utilities          ****
**** Copyright 1996-2012, Alessandro Pedretti ****
*************************************************/


#ifndef DXWCPU_H
#  define DXWCPU_H

#  if defined(__BORLANDC__) && !defined(WIN32)
#    define  WIN32
#  endif

#  ifdef WIN32
#    include <windows.h>
#  endif

#  ifdef __GNUC__
typedef long long               int64;
typedef int                     int32;
typedef short                   int16;
typedef char                    int8;
typedef unsigned long long      uint64;
typedef unsigned int            uint32;
typedef unsigned short          uint16;
typedef unsigned char           uint8;
#  else
typedef __int64                 int64;
typedef __int32                 int32;
typedef __int16                 int16;
typedef __int8                  int8;
typedef unsigned __int64        uint64;
typedef unsigned __int32        uint32;
typedef unsigned __int16        uint16;
typedef unsigned __int8         uint8;
#  endif

/**** Arguments for Cpuid() ****/

typedef struct {
  uint32        eax;
  uint32        ebx;
  uint32        ecx;
  uint32        edx;
} CPUID_ARGS;

/**** CPU Suffixes ****/

#define  CPU_SUFF_AMD_K6        "_k6"
#define  CPU_SUFF_AMD_K7        "_k7"
#define  CPU_SUFF_AMD_K7SSE     "_k7sse"
#define  CPU_SUFF_AMD_A64       "_k8"
#define  CPU_SUFF_INTEL_586     ""
#define  CPU_SUFF_INTEL_686     "_686"
#define  CPU_SUFF_INTEL_P3      "_p3"
#define  CPU_SUFF_INTEL_P4      "_p4"
#define  CPU_SUFF_INTEL_CORE2   "_core2"

#define  CPU_SUFF_MAXNUM        7

/**** CPU Vendors ****/

#define  VENDOR_AMD             "AuthenticAMD"
#define  VENDOR_CENTAUR         "CentaurHauls"
#define  VENDOR_CYRIX           "CyrixInstead"
#define  VENDOR_GEODE           "Geode by NSC"
#define  VENDOR_INTEL           "GenuineIntel"
#define  VENDOR_NEXGEN          "NexGenDriven"
#define  VENDOR_RISE            "RiseRiseRise"
#define  VENDOR_SIS             "SiS SiS SiS "
#define  VENDOR_TRANSMETA       "GenuineTMx86"
#define  VENDOR_UMC             "UMC UMC UMC "

/**** Standard (Intel) Feature Flags ****/

#define INTEL_FPU_FLAG     0x00000001 // Bit 0 - Floating-Point unit on chip
#define INTEL_VME_FLAG     0x00000002 // Bit 1 - Virtual Mode Extention
#define INTEL_DE_FLAG      0x00000004 // Bit 2 - Debugging Extention
#define INTEL_PSE_FLAG     0x00000008 // Bit 3 - Page Size Extention
#define INTEL_TSC_FLAG     0x00000010 // Bit 4 - Time Stamp Counter
#define INTEL_MSR_FLAG     0x00000020 // Bit 5 - Model Specific Registers
#define INTEL_PAE_FLAG     0x00000040 // Bit 6 - Physical Address Extention
#define INTEL_MCE_FLAG     0x00000080 // Bit 7 - Machine Check Exception
#define INTEL_CX8_FLAG     0x00000100 // Bit 8 - CMPXCHG8 Instruction
#define INTEL_APIC_FLAG    0x00000200 // Bit 9 - Software-accessible local APIC on Chip
#define INTEL_BIT_10       0x00000400 // Bit 10 - Reserved, do not count on value
#define INTEL_SEP_FLAG     0x00000800 // Bit 11 - Fast System Call
#define INTEL_MTRR_FLAG    0x00001000 // Bit 12 - Memory Type Range Registers
#define INTEL_PGE_FLAG     0x00002000 // Bit 13 - Page Global Enable
#define INTEL_MCA_FLAG     0x00004000 // Bit 14 - Machine Check Architecture
#define INTEL_CMOV_FLAG    0x00008000 // Bit 15 - Conditional Move Instruction
#define INTEL_PAT_FLAG     0x00010000 // Bit 16 - Page Attribute Table
#define INTEL_PSE36_FLAG   0x00020000 // Bit 17 - 36-bit Page Size Extention
#define INTEL_PN_FLAG      0x00040000 // Bit 18 - Processor Number
#define INTEL_BIT_19       0x00080000 // Bit 19 - Reserved, do not count on value
#define INTEL_BIT_20       0x00100000 // Bit 20 - Reserved, do not count on value
#define INTEL_BIT_21       0x00200000 // Bit 21 - Reserved, do not count on value
#define INTEL_BIT_22       0x00400000 // Bit 22 - Reserved, do not count on value
#define INTEL_MMX_FLAG     0x00800000 // Bit 23 - MMX technology
#define INTEL_FXSR_FLAG    0x01000000 // Bit 24 - Fast Floating Point Save and Restore
#define INTEL_SSE_FLAG     0x02000000 // Bit 25 - Streaming SIMD Extension
#define INTEL_SSE2_FLAG    0x04000000 // Bit 26 - SSE2
#define INTEL_BIT_27       0x08000000 // Bit 27 - Reserved, do not count on value
#define INTEL_HT_FLAG      0x10000000 // Bit 28 - Hyper-threading
#define INTEL_BIT_29       0x20000000 // Bit 29 - Reserved, do not count on value
#define INTEL_64BIT_FLAG   0x40000000 // Bit 30 - 64bit Intel Architecture
#define INTEL_BIT_31       0x80000000 // Bit 31 - Reserved, do not count on value

/**** AMD Standard Feature Flags ****/

#define AMD_FPU_FLAG       0x00000001 // Bit 0 - Floating-Point unit on chip
#define AMD_VME_FLAG       0x00000002 // Bit 1 - Virtual Mode Extention
#define AMD_DE_FLAG        0x00000004 // Bit 2 - Debugging Extention
#define AMD_PSE_FLAG       0x00000008 // Bit 3 - Page Size Extention
#define AMD_TSC_FLAG       0x00000010 // Bit 4 - Time Stamp Counter
#define AMD_MSR_FLAG       0x00000020 // Bit 5 - Model Specific Registers
#define AMD_PEA_FLAG       0x00000040 // Bit 6 - Page Address Extension
#define AMD_MCE_FLAG       0x00000080 // Bit 7 - Machine Check Exception
#define AMD_CX8_FLAG       0x00000100 // Bit 8 - CMPXCHG8 Instruction
#define AMD_APIC_FLAG      0x00000200 // Bit 9 - On-chip APIC
#define AMD_BIT_10         0x00000400 // Bit 10 - Reserved, do not count on value
#define AMD_BIT_11         0x00000800 // Bit 11 - Reserved, do not count on value
#define AMD_MTRR_FLAG      0x00001000 // Bit 12 - Memory Type Range Registers
#define AMD_PGE_FLAG       0x00002000 // Bit 13 - Page Global Enable
#define AMD_MCA_FLAG       0x00004000 // Bit 14 - Machine Check Architecture
#define AMD_ICMOV_FLAG     0x00008000 // Bit 15 - Integer Conditional Move Instruction
#define AMD_PAT_FLAG       0x00010000 // Bit 16 - Page Attribute Table
#define AMD_PSE36_17       0x00020000 // Bit 17 - 36-bit Page Size Extension
#define AMD_BIT_18         0x00040000 // Bit 18 - Reserved, do not count on value
#define AMD_BIT_19         0x00080000 // Bit 19 - Reserved, do not count on value
#define AMD_BIT_20         0x00100000 // Bit 20 - Reserved, do not count on value
#define AMD_BIT_21         0x00200000 // Bit 21 - Reserved, do not count on value
#define AMD_BIT_22         0x00400000 // Bit 22 - Reserved, do not count on value
#define AMD_MMX_FLAG       0x00800000 // Bit 23 - MMX technology
#define AMD_FXSR_FLAG      0x01000000 // Bit 24 - Fast floating point save and restore
#define AMD_BIT_25         0x02000000 // Bit 25 - Reserved, do not count on value
#define AMD_BIT_26         0x04000000 // Bit 26 - Reserved, do not count on value
#define AMD_BIT_27         0x08000000 // Bit 27 - Reserved, do not count on value
#define AMD_BIT_28         0x10000000 // Bit 28 - Reserved, do not count on value
#define AMD_BIT_29         0x20000000 // Bit 29 - Reserved, do not count on value
#define AMD_BIT_30         0x40000000 // Bit 30 - Reserved, do not count on value
#define AMD_BIT_31         0x80000000 // Bit 31 - Reserved, do not count on value

/**** AMD Enhanced Feature Flags ****/

#define EAMD_FPU_FLAG      0x00000001 // Bit 0 - Floating-Point unit on chip
#define EAMD_VME_FLAG      0x00000002 // Bit 1 - Virtual Mode Extention
#define EAMD_DE_FLAG       0x00000004 // Bit 2 - Debugging Extention
#define EAMD_PSE_FLAG      0x00000008 // Bit 3 - Page Size Extention
#define EAMD_TSC_FLAG      0x00000010 // Bit 4 - Time Stamp Counter
#define EAMD_MSR_FLAG      0x00000020 // Bit 5 - Model Specific Registers
#define EAMD_PEA_FLAG      0x00000040 // Bit 6 - Page Address Extension
#define EAMD_MCE_FLAG      0x00000080 // Bit 7 - Machine Check Exception
#define EAMD_CX8_FLAG      0x00000100 // Bit 8 - CMPXCHG8 Instruction
#define EAMD_APIC_FLAG     0x00000200 // Bit 9 - On-chip APIC
#define EAMD_BIT_10        0x00000400 // Bit 10 - Reserved, do not count on value
#define EAMD_SEP_FLAG      0x00000800 // Bit 11 - Fast System Call
#define EAMD_MTRR_FLAG     0x00001000 // Bit 12 - Memory Type Range Registers
#define EAMD_PGE_FLAG      0x00002000 // Bit 13 - Page Global Enable
#define EAMD_MCA_FLAG      0x00004000 // Bit 14 - Machine Check Architecture
#define EAMD_ICMOV_FLAG    0x00008000 // Bit 15 - Integer Conditional Move Instruction
#define EAMD_PAT_FLAG      0x00010000 // Bit 16 - Page Attribute Table
#define EAMD_PSE36_17      0x00020000 // Bit 17 - 36-bit Page Size Extension
#define EAMD_BIT_18        0x00040000 // Bit 18 - Reserved, do not count on value
#define EAMD_BIT_19        0x00080000 // Bit 19 - Reserved, do not count on value
#define EAMD_BIT_20        0x00100000 // Bit 20 - Reserved, do not count on value
#define EAMD_BIT_21        0x00200000 // Bit 21 - Reserved, do not count on value
#define EAMD_AMIE_FLAG     0x00400000 // Bit 22 - AMD Multimedia Instructions Extensions
#define EAMD_MMX_FLAG      0x00800000 // Bit 23 - MMX technology
#define EAMD_FXSR_FLAG     0x01000000 // Bit 24 - Fast floating point save and restore
#define EAMD_BIT_25        0x02000000 // Bit 25 - Reserved, do not count on value
#define EAMD_BIT_26        0x04000000 // Bit 26 - Reserved, do not count on value
#define EAMD_BIT_27        0x08000000 // Bit 27 - Reserved, do not count on value
#define EAMD_BIT_28        0x10000000 // Bit 28 - Reserved, do not count on value
#define EAMD_BIT_29        0x20000000 // Bit 29 - Reserved, do not count on value
#define EAMD_3DNOWEXT_FLAG 0x40000000 // Bit 30 - AMD 3DNOW! Extensions
#define EAMD_3DNOW_FLAG    0x80000000 // Bit 31 - AMD 3DNOW! Technology

/**** Intel processor names ****/

#define INTEL_486SL     "486(TM) SL processor"
#define INTEL_ATOM      "Atom processor"
#define INTEL_DX2WB     "Write-Back Enhanced DX2(TM) processor"
#define INTEL_DX4       "DX4(TM) processor"
#define INTEL_DX4O      "DX4(TM) OverDrive(R) processor"
#define INTEL_P         "Pentium(R) processor"
#define INTEL_PO        "Pentium(R) OverDrive(R) processor"
#define INTEL_P55CT     "Pentium(R) P55CT OverDrive(R) processor"
#define INTEL_PPRO      "Pentium(R) Pro processor"
#define INTEL_PII       "Pentium(R) II processor"
#define INTEL_CELERON   "Celeron(TM) processor"
#define INTEL_PIII      "Pentium(R) III processor"
#define INTEL_PIIIM     "Pentium(R) III-M processor"
#define INTEL_PMC       "Pentium(R) M processor"
#define INTEL_PMDC      "Pentium(R) M Dual Core processor"
#define INTEL_XEON      "Pentium(R) III Xeon processor"
#define INTEL_PIIO      "Pentium(R) II OverDrive(R) processor"
#define INTEL_PIV       "Pentium(R) IV processor"
#define INTEL_PIVXEON   "Pentium(R) IV Xeon processor"
#define INTEL_MCKINLEY  "McKinley (IA-64)"
#define INTEL_CORE2DC   "Core 2 Duo"
#define INTEL_CORE2SC   "Core 2 Solo"
#define INTEL_COREI7    "Core i7"
#define INTEL_XEONMP    "Xeon processor MP"

#define PROCESSOR_UNKNOWN       "Unknown processor"

#  ifdef DXWCPU_C

/**** Stabdard C definitions ****/

#    ifndef true
#      define  true               1
#    endif

#    ifndef false
#      define  false              0
#    endif

/**** Standard C structure ****/

typedef struct {
  uint32  m_CPU_Type;                      /* Type (eax register)                */
  uint32  m_CPU_Features;                  /* Features (edx register)            */
  uint32  m_CPU_ExtendedFeatures;          /* Extended features                  */
  uint32  m_CPU_Suffixes;                  /* Current suffix                     */
  char   *m_CPU_Suffix[CPU_SUFF_MAXNUM];   /* Cpu suffix                         */
  uint32  m_CPU_Number;                    /* Number of total CPUs               */
  uint32  m_CPU_LogicalPerPhysical;        /* Number of logical CPU per physic.  */
  uint32  m_MaxCPUIDVal;                   /* Highest value recodnized by CPUID  */
  char    m_ProcVendorString[13];          /* Vendor name                        */
  char    m_ProcName[48];                  /* Processor Name                     */
  uint32  m_ProcBrandID;                   /* Brand ID                           */
  uint32  m_ProcType;                      /* Processor Type                     */
  uint32  m_ProcFamily;                    /* Processor Family                   */
  uint32  m_ProcExtendedFamily;            /* Extended Processor Family          */
  uint32  m_ProcModel;                     /* Processor Model                    */
  uint32  m_ProcExtendedModel;             /* Extended Processor Model           */
  uint32  m_ProcStepping;                  /* Processor Stepping                 */
} DXWCPU;

/**** Standard C funtions ****/

#    ifdef __cplusplus
extern "C" {
#    endif

uint32        DXWCPU_FindFile(DXWCPU *, char *);
uint32        DXWCPU_GetInfo(DXWCPU *);
uint32        DXWCPU_GetCPUID_EAX(uint32);
uint32        DXWCPU_GetCPUID_EDX(uint32);
uint32        DXWCPU_GetCPUIDString(uint32, char *);
void          DXWCPU_GetCPUIDStringAMD(uint32 , char *);
char *        DXWCPU_GetIntelProcName(DXWCPU *);

#ifdef WIN32
HINSTANCE     DXWCPU_LoadLibrary(DXWCPU *, LPCTSTR);
#endif

#    ifdef __cplusplus
}
#    endif
#  else

class DXWCPU {
public:

  /**** Constructor/Destructor ****/

  DXWCPU(void);

  /**** Public methods ****/

  bool          FindFile(char *);
  bool          GetInfo(void);

#ifdef WIN32
  HINSTANCE     LoadLibrary(LPCTSTR);
#endif

  /**** Standard CPU Features ****/

  bool          CPU_Has_FPU()             { return ((m_CPU_Features & INTEL_FPU_FLAG)   == INTEL_FPU_FLAG  );}
  bool          CPU_Has_VME()             { return ((m_CPU_Features & INTEL_VME_FLAG)   == INTEL_VME_FLAG  );}
  bool          CPU_Has_DE()              { return ((m_CPU_Features & INTEL_DE_FLAG)    == INTEL_DE_FLAG   );}
  bool          CPU_Has_PSE()             { return ((m_CPU_Features & INTEL_PSE_FLAG)   == INTEL_PSE_FLAG  );}
  bool          CPU_Has_TSC()             { return ((m_CPU_Features & INTEL_TSC_FLAG)   == INTEL_TSC_FLAG  );}
  bool          CPU_Has_MSR()             { return ((m_CPU_Features & INTEL_MSR_FLAG)   == INTEL_MSR_FLAG  );}
  bool          CPU_Has_PAE()             { return ((m_CPU_Features & INTEL_PAE_FLAG)   == INTEL_PAE_FLAG  );}
  bool          CPU_Has_MCE()             { return ((m_CPU_Features & INTEL_MCE_FLAG)   == INTEL_MCE_FLAG  );}
  bool          CPU_Has_CX8()             { return ((m_CPU_Features & INTEL_CX8_FLAG)   == INTEL_CX8_FLAG  );}
  bool          CPU_Has_SEP()             { return ((m_CPU_Features & INTEL_CX8_FLAG)   == INTEL_SEP_FLAG  );}
  bool          CPU_Has_APIC()            { return ((m_CPU_Features & INTEL_APIC_FLAG)  == INTEL_APIC_FLAG );}
  bool          CPU_Has_MTRR()            { return ((m_CPU_Features & INTEL_MTRR_FLAG)  == INTEL_MTRR_FLAG );}
  bool          CPU_Has_PGE()             { return ((m_CPU_Features & INTEL_PGE_FLAG)   == INTEL_PGE_FLAG  );}
  bool          CPU_Has_MCA()             { return ((m_CPU_Features & INTEL_MCA_FLAG)   == INTEL_MCA_FLAG  );}
  bool          CPU_Has_CMOV()            { return ((m_CPU_Features & INTEL_CMOV_FLAG)  == INTEL_CMOV_FLAG );}
  bool          CPU_Has_PAT()             { return ((m_CPU_Features & INTEL_PAT_FLAG)   == INTEL_PAT_FLAG  );}
  bool          CPU_Has_PSE36()           { return ((m_CPU_Features & INTEL_PSE36_FLAG) == INTEL_PSE36_FLAG);}

  bool          CPU_Has_MMX()             { return ((m_CPU_Features & INTEL_MMX_FLAG)   == INTEL_MMX_FLAG  );}
  bool          CPU_Has_FXSR()            { return ((m_CPU_Features & INTEL_FXSR_FLAG)  == INTEL_FXSR_FLAG );}
  bool          CPU_Has_SSE()             { return ((m_CPU_Features & INTEL_SSE_FLAG)   == INTEL_SSE_FLAG  );}
  bool          CPU_Has_SSE2()            { return ((m_CPU_Features & INTEL_SSE2_FLAG)  == INTEL_SSE2_FLAG );}
  bool          CPU_Has_HT()              { return ((m_CPU_Features & INTEL_HT_FLAG)    == INTEL_HT_FLAG   );}
  bool          CPU_Has_HT_Active()       { return (m_CPU_HtActive);}
  bool          CPU_Has_64BIT()           { return ((m_CPU_Features & INTEL_64BIT_FLAG) == INTEL_64BIT_FLAG);}

  /**** AMD Specific ****/

  bool          CPU_Has_AMD3DNOW()        { return ((m_CPU_ExtendedFeatures & EAMD_3DNOW_FLAG)    == EAMD_3DNOW_FLAG   );}
  bool          CPU_Has_AMD3DNOWEXT()     { return ((m_CPU_ExtendedFeatures & EAMD_3DNOWEXT_FLAG) == EAMD_3DNOWEXT_FLAG);}
  bool          CPU_Has_AMIE()            { return ((m_CPU_ExtendedFeatures & EAMD_3DNOWEXT_FLAG) == EAMD_3DNOWEXT_FLAG);}

  char          *GetVendorString()        { return m_ProcVendorString;}
  char          *GetCPUName()           { return m_ProcName;}
  uint32        GetCPUFeatures()          { return m_CPU_Features;}
  uint32        GetCPUExtendedFeatures()  { return m_CPU_ExtendedFeatures;}
  uint32        GetType()                 { return m_ProcType;}
  uint32        GetFamily()               { return m_ProcFamily;}
  uint32        GetExtendedFamily()       { return m_ProcExtendedFamily;}
  uint32        GetModel()                { return m_ProcModel;}
  uint32        GetNumberOfCPUs()         { return m_CPU_Number;}
  uint32        GetNumberOfLogicalCPUs()  { return m_CPU_LogicalPerPhysical;}
  uint32        GetExtendedModel()        { return m_ProcExtendedModel;}
  uint32        GetStepping()             { return m_ProcStepping;}
  char          *GetDllSuffix()           { return m_CPU_Suffix[0];}
  char          **GetSuffix()             { return m_CPU_Suffix;}

private:

  /**** Functions ****/

  uint32 GetCPUID_EAX(uint32 funcNum);
  uint32 GetCPUID_EDX(uint32 funcNum);
  uint32 GetCPUIDString(uint32 funcNum, char *String);
  void   GetCPUIDStringAMD(uint32 funcNum, char *String);
  char * GetIntelProcName(void);

  /**** Data ****/

  bool          m_CPU_GotInfo;                   /* Already checked ?                  */
  uint32        m_CPU_Type;                      /* Type (eax register)                */
  uint32        m_CPU_Features;                  /* Features (edx register)            */
  uint32        m_CPU_ExtendedFeatures;          /* Extended features                  */
  bool          m_CPU_HtActive;                  /* HyperThreading active              */
  uint32        m_CPU_Suffixes;                  /* Current suffix                     */
  char *        m_CPU_Suffix[CPU_SUFF_MAXNUM];   /* Cpu suffix                         */
  uint32        m_CPU_Number;                    /* Number of total CPUs               */
  uint32        m_CPU_LogicalPerPhysical;        /* Number of logical CPUs per physic. */
  uint32        m_MaxCPUIDVal;                   /* Highest value recodnized by CPUID  */
  char          m_ProcVendorString[13];          /* Vendor name                        */
  char          m_ProcName[48];                  /* Processor Name                     */
  uint32        m_ProcBrandID;                   /* Brand ID                           */
  uint32        m_ProcType;                      /* Processor Type                     */
  uint32        m_ProcFamily;                    /* Processor Family                   */
  uint32        m_ProcExtendedFamily;            /* Extended Processor Family          */
  uint32        m_ProcModel;                     /* Processor Model                    */
  uint32        m_ProcExtendedModel;             /* Extended Processor Model           */
  uint32        m_ProcStepping;                  /* Processor Stepping                 */
};
#  endif
#endif
