
/*************************************************
****        VEGA - x86 CPU utilities          ****
**** Copyright 1996-2012, Alessandro Pedretti ****
*************************************************/


#if defined(__BORLANDC__) && !defined(WIN32)
#  define  WIN32
#endif

#ifdef WIN32
#  include <windows.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#  pragma inline
#endif

#ifndef DXWCPU_C
#  define DXWCPU_C
#endif

#include "dxwcpu.h"


#define  NUM_LOGICAL_BITS      0x00FF0000  // Bit 16-23 in ebx contains the number of logical
                                           // processors per physical processor when execute cpuid with
                                           // eax set to 1
#ifndef WIN32
#  define  ZeroMemory(A, S)    memset((A), 0, (S))
#endif


/**** Cpuid wrapper ****/

#ifndef WIN64
#  ifdef __GNUC__
#    ifdef DXWCPU_64BIT

/**** Cpuid - GNU 64 bit ****/

static void Cpuid(CPUID_ARGS *p)
{
  __asm__ ("movl  (%%rdi), %%eax\n"
           "movl 8(%%rdi), %%ecx\n"
           "cpuid\n"
           "movl %%eax,   (%%rdi)\n"
           "movl %%ebx,  4(%%rdi)\n"
           "movl %%ecx,  8(%%rdi)\n"
           "movl %%edx, 12(%%rdi)\n"
           :
           :"D" (p)
           :"ax", "bx", "cx", "dx", "memory");
}

#    else

/**** Cpuid - GNU 32 bit ****/

static void Cpuid(CPUID_ARGS *p)
{
  __asm__ ("movl  (%%edi), %%eax\n"
           "movl 8(%%edi), %%ecx\n"
           "cpuid\n"
           "movl %%eax,   (%%edi)\n"
           "movl %%ebx,  4(%%edi)\n"
           "movl %%ecx,  8(%%edi)\n"
           "movl %%edx, 12(%%edi)\n"
           :
           :"D" (p)
           :"ax", "bx", "cx", "dx", "memory");

}

#    endif
#  else

/**** Cpuid - Borland 32 bit ****/

static void Cpuid(CPUID_ARGS *p)
{
  __asm {
    mov edi, p
    mov eax, [edi].eax
    mov ecx, [edi].ecx
    cpuid
    mov [edi].eax, eax
    mov [edi].ebx, ebx
    mov [edi].ecx, ecx
    mov [edi].edx, edx
  }
}
#  endif
#else

/**** Cpuid - GNU 64 bit ****/

static void Cpuid(CPUID_ARGS *p)
{
  __asm__ ("movl  (%%rdi), %%eax\n"
           "movl 8(%%rdi), %%ecx\n"
           "cpuid\n"
           "movl %%eax,   (%%rdi)\n"
           "movl %%ebx,  4(%%rdi)\n"
           "movl %%ecx,  8(%%rdi)\n"
           "movl %%edx, 12(%%rdi)\n"
           :
           :"D" (p)
           :"ax", "bx", "cx", "dx", "memory");
}

#  ifdef __GNUC__
#  else
/**** Cpuid - Visual C 64 bit ****/

#    ifdef __cplusplus
extern "C" {
#    endif
void Cpuid64(CPUID_ARGS *p);
#    ifdef __cplusplus
}
#    endif
#    define Cpuid         Cpuid64
#  endif
#endif


/**** Find a file ****/

uint32 DXWCPU_FindFile(DXWCPU *Cpu, char *FileName)
{
  char          *Prefix, **Suffix;

  FILE          *FH = NULL;

  Prefix = FileName + strlen(FileName) - 4;
  for(Suffix = Cpu -> m_CPU_Suffix; (FH == NULL)&& (*Suffix); ++Suffix) {
    *Prefix = 0;
    strcat(FileName, *Suffix);
    strcat(FileName, ".exe");
    if ((FH = fopen(FileName, "r")) != NULL) fclose(FH);
  } /* End of for */

  return (FH != NULL);
}


/**** Execute CPUID function EAX=funcNum and get EAX result ****/

uint32 DXWCPU_GetCPUID_EAX(uint32 funcNum)
{
  CPUID_ARGS    Args;

  Args.eax = funcNum;
  Cpuid(&Args);

  return Args.eax;
}


/**** Execute CPUID function EAX=funcNum and get EDX result ****/

uint32 DXWCPU_GetCPUID_EDX(uint32 funcNum)
{
  CPUID_ARGS    Args;

  Args.eax = funcNum;
  Cpuid(&Args);
  return Args.edx;
}


/**** Execute CPUID EAX=funcNum and get EAX result and ****/
/**** the string EBX:EDX:ECX                           ****/

uint32 DXWCPU_GetCPUIDString(uint32 funcNum, char *String)
{
  CPUID_ARGS    Args;

  Args.eax = funcNum;
  Cpuid(&Args);
  ((uint32 *)String)[0] = Args.ebx;
  ((uint32 *)String)[1] = Args.edx;
  ((uint32 *)String)[2] = Args.ecx;

  return Args.eax;
}


/**** AMD Specific ****/

void DXWCPU_GetCPUIDStringAMD(uint32 funcNum, char *String)
{
  CPUID_ARGS    Args;

  Args.eax = funcNum;
  Cpuid(&Args);
  ((uint32 *)String)[0] = Args.eax;
  ((uint32 *)String)[1] = Args.ebx;
  ((uint32 *)String)[2] = Args.ecx;
  ((uint32 *)String)[3] = Args.edx;
}


/**** Get the information about the CPU ****/

uint32 DXWCPU_GetInfo(DXWCPU *Cpu)
{
  CPUID_ARGS    Args;
  char *        Tmp;
  uint32        k, j;

#ifdef WIN32
  SYSTEM_INFO   SysInfo;
#endif

  ZeroMemory(Cpu, sizeof(DXWCPU));

  /**** Get the number of CPUs ****/

#ifdef WIN32
  ZeroMemory(&SysInfo, sizeof(SYSTEM_INFO));
  GetSystemInfo(&SysInfo);
  Cpu -> m_CPU_Number             = SysInfo.dwNumberOfProcessors;
#else
  FILE * FH = popen("grep '^processor' /proc/cpuinfo| wc -l", "r");
  if (FH) {
    fscanf(FH, "%d", &Cpu -> m_CPU_Number);
    if (Cpu -> m_CPU_Number < 1) Cpu -> m_CPU_Number = 1;
    pclose(FH);
  }
#endif

  Cpu -> m_CPU_Suffix[0]          = CPU_SUFF_INTEL_586;
  Cpu -> m_CPU_Suffixes           = 1;
  Cpu -> m_CPU_LogicalPerPhysical = 1;

  Cpu -> m_MaxCPUIDVal  = DXWCPU_GetCPUIDString(0, Cpu -> m_ProcVendorString);

  /**** CPUID 1 ****/

  Args.eax = 1;
  Cpuid(&Args);

  Cpu -> m_ProcType     = (Args.eax >> 12) & 0x3;
  Cpu -> m_ProcFamily   = (Args.eax >> 8 ) & 0xf;
  Cpu -> m_ProcModel    = (Args.eax >> 4 ) & 0xf;
  Cpu -> m_ProcStepping = (Args.eax      ) & 0x7;
  Cpu -> m_CPU_Features = Args.edx;

  /**** Check the virtual CPUs ****/

  if (Cpu -> m_CPU_Features & INTEL_HT_FLAG)
    Cpu -> m_CPU_LogicalPerPhysical = ((Args.ebx & NUM_LOGICAL_BITS) >> 16);

  /**** Intel CPU ****/

  if ((strncmp(Cpu -> m_ProcVendorString, VENDOR_INTEL, 12)) == 0) {
    Cpu -> m_CPU_Type           = Args.eax;
    Cpu -> m_ProcBrandID        = Args.ebx & 0xF;
    strcpy(Cpu -> m_ProcName, DXWCPU_GetIntelProcName(Cpu));

    if (Cpu -> m_ProcFamily >= 6) {                                    /* Pentium Pro / 2 */
      Cpu -> m_CPU_Suffix[Cpu -> m_CPU_Suffixes++] = CPU_SUFF_INTEL_686;
      if (Cpu -> m_ProcModel >= 7)                                     /* Pentium 3       */
        Cpu -> m_CPU_Suffix[Cpu -> m_CPU_Suffixes++] = CPU_SUFF_INTEL_P3;
    }

    if (Cpu -> m_ProcFamily >= 15)                                     /* Pentium 4       */
      Cpu -> m_CPU_Suffix[Cpu -> m_CPU_Suffixes++] = CPU_SUFF_INTEL_P4;

    if ((Cpu -> m_ProcFamily == 6) && (Cpu -> m_ProcModel >= 14)) {    /* Core 2          */
      Cpu -> m_CPU_Suffix[Cpu -> m_CPU_Suffixes++] = CPU_SUFF_INTEL_P4;
      Cpu -> m_CPU_Suffix[Cpu -> m_CPU_Suffixes++] = CPU_SUFF_INTEL_CORE2;
    }

    /**** AMD CPU ****/

  } else if ((strncmp(Cpu -> m_ProcVendorString, VENDOR_AMD, 12))==0) {
    if (DXWCPU_GetCPUID_EAX(0x80000000)) {
      Cpu -> m_CPU_ExtendedFeatures = DXWCPU_GetCPUID_EDX(0x80000001);
      DXWCPU_GetCPUIDStringAMD(0x80000002, Cpu -> m_ProcName     );
      DXWCPU_GetCPUIDStringAMD(0x80000003, Cpu -> m_ProcName + 16);
      DXWCPU_GetCPUIDStringAMD(0x80000004, Cpu -> m_ProcName + 32);
    } else Cpu -> m_CPU_ExtendedFeatures = Args.edx;
    if (Cpu -> m_ProcFamily >= 6)
      Cpu -> m_CPU_Suffix[Cpu -> m_CPU_Suffixes++] = CPU_SUFF_INTEL_686;
    if (Cpu -> m_ProcFamily >= 5)
      Cpu -> m_CPU_Suffix[Cpu -> m_CPU_Suffixes++] = CPU_SUFF_AMD_K6;
    if (Cpu -> m_ProcFamily >= 6) {
      Cpu -> m_CPU_Suffix[Cpu -> m_CPU_Suffixes++] = CPU_SUFF_AMD_K7;
      if (Cpu -> m_CPU_Features & INTEL_SSE_FLAG)
        Cpu -> m_CPU_Suffix[Cpu -> m_CPU_Suffixes++] = CPU_SUFF_AMD_K7SSE;
    }
    if (Cpu -> m_ProcFamily >= 15)
      Cpu -> m_CPU_Suffix[Cpu -> m_CPU_Suffixes++] = CPU_SUFF_AMD_A64;
  }

  /**** Set the suffixes ****/

  for(k = 0, j = Cpu -> m_CPU_Suffixes - 1; k < j; ++k, --j) {
    Tmp                    = Cpu -> m_CPU_Suffix[k];
    Cpu -> m_CPU_Suffix[k] = Cpu -> m_CPU_Suffix[j];
    Cpu -> m_CPU_Suffix[j] = Tmp;
  }

  return true;
}


/**** Obtain the Intel's processor name ****/

char *DXWCPU_GetIntelProcName(DXWCPU *Cpu)
{
  char          *CPUName = PROCESSOR_UNKNOWN;

  switch (Cpu -> m_ProcFamily) {
/*
  case 4:       // 486 family
    if (Model == 4) CPUName = INTEL_486SL;
    if (Model == 7) CPUName = INTEL_DX2WB;
    if (Model == 8) CPUName = INTEL_DX4;
    if ((Model == 8)&& (Type == 1)) CPUName = INTEL_DX4O;
    break;
*/

  case 5:       // 586 family
    if (Cpu -> m_ProcType == 1) CPUName = INTEL_PO;
    else CPUName = INTEL_P;
    break;

  case 6:       // 686 family (0110)
    switch(Cpu -> m_ProcModel) {
    case 1:     /* 0001 Pentium Pro */
      CPUName = INTEL_PPRO;
      break;

    case 3:     /* 0011 Pentium II */
      if (Cpu -> m_ProcType == 1) CPUName = INTEL_PIIO;
      else CPUName = INTEL_PII;
      break;

    case 4:
      CPUName = INTEL_P55CT;
      break;

    case 5:     /* 0101 */
      CPUName = INTEL_PII;
      break;

    case 6:     /* 0110 */
      CPUName = INTEL_CELERON;
      break;

    case 7:     /* 0111 Pentium III */
    case 8:     /* 1000 Pentium III */
      CPUName = INTEL_PIII;
      break;

    case 9:     /* 1001 Pentium M */
      CPUName = INTEL_PMC;
      break;

    case 10:    /* 1010 Pentium III Xeon */
      CPUName = INTEL_XEON;
      break;

    case 11:    /* 1011 Pentium III */
      CPUName = INTEL_PIIIM;
      break;

    case 12:    /* 1100 Atom */
      CPUName = INTEL_ATOM;
      break;

    case 13:    /* 1101 Pentium M, Celeron M */
      CPUName = INTEL_PMC;
      break;

    case 14:    /* 1110 Core 2/ Core Solo */
      CPUName = INTEL_PMDC;
      break;

    case 15:    /* 1111 Core 2 Duo, Mobile, Quad */
    case 17:
      CPUName = INTEL_CORE2DC;
      break;

    case 16:
      CPUName = INTEL_CORE2SC;
      break;
    }
    break;

  case 15:      // Pentium 4 family
    Cpu -> m_ProcExtendedFamily = (Cpu -> m_CPU_Type >> 20) & 0xff;
    switch(Cpu -> m_ProcExtendedFamily) {
    case 0:     // Family = 15, Ext. Family = 0:  Pentium 4 (80786 ??) processor family
      Cpu -> m_ProcExtendedModel = (Cpu -> m_CPU_Type >> 16) & 0xFF;
      switch(Cpu -> m_ProcModel) {
      case 0:
      case 1:
        if (Cpu -> m_ProcBrandID == 8) CPUName = INTEL_PIV;
        else CPUName = INTEL_PIVXEON;
        break;

      case 2:
        if ((Cpu -> m_ProcBrandID == 9) || (Cpu -> m_ProcBrandID == 0xA)) CPUName = INTEL_PIV;
        else CPUName = INTEL_PIVXEON;
        break;

      default:
        CPUName = INTEL_PIV;
      } /* End of switch */
      break;

    case 1:     // Family = 15, Ext. Family = 1:  McKinley (64-bit) processor family
      CPUName = INTEL_MCKINLEY;
      break;

    default:
      CPUName = "Unknown Pentium 4+";
      break;
    } /* End of switch */
    break;
  } /* End of switch */

  return CPUName;
}


/**** Load library ****/

#if defined(WIN32) || defined(WIN64)
HINSTANCE DXWCPU_LoadLibrary(DXWCPU *Cpu, LPCTSTR lpszLibrary)
{
  char          Buf[MAX_PATH];
  char          *Prefix, **Suffix;

  HINSTANCE     hLib = NULL;

  strcpy(Buf, lpszLibrary);
  Prefix = Buf + strlen(Buf) - 4;
  for(Suffix = Cpu -> m_CPU_Suffix; (hLib == NULL) && (*Suffix); ++Suffix) {
    *Prefix = 0;
    strcat(Buf, *Suffix);
    strcat(Buf, ".dll");
    hLib = LoadLibrary(Buf);
  } /* End of for */

  return hLib;
}
#endif
