
/*********************************************************************
****        ESCHER Next Generation - Language localization        ****
**** (c) 1997-2012, G. Ausiello, G. Cesareni, M. Helmer Citterich ****
****               NG version by Alessandro Pedretti              ****
*********************************************************************/


#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <locale.h>

#ifdef __WIN32__
#  include <windows.h>
# else
#  include <unistd.h>
#  include <errno.h>
#endif

#include "my_macros.h"
#include "utils.h"

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#define  ESCHER_ARRAY
#include "globstr.h"

/**** Constants ****/

#define  ERROR_PROMPT           "\n%s %s.\n\n"

/**** Local variables ****/

static char     Buf[4096];

/**** Global variables ****/

extern void             *Catalog;
extern FILE             *OutFH;

#ifdef AMIGA
#  include <exec/libraries.h>
#  include <libraries/locale.h>

struct Library          *LocaleBase  = NULL;
#endif

#ifdef WIN32
#  include <windows.h>

#  define  BD_LOCALE_DLL      "locale.dll"

/**** Prototypes ****/

ES_BOOL      GetProcAddresses(HINSTANCE *, LPCSTR , INT, ... );

/**** Library function pointers ****/

void           (WINAPI *BD_LOCALE_LocCloseCat)(void *)                                      = NULL;
char          *(WINAPI *BD_LOCALE_LocGetCatStr)(void *, unsigned int, char *)               = NULL;
int            (WINAPI *BD_LOCALE_LocGetErr)(void)                                          = NULL;
void           (WINAPI *BD_LOCALE_LocGetErrStr)(char *, int)                                = NULL;
void           (WINAPI *BD_LOCALE_LocIso2Dos)(char *, short)                                = NULL;
void          *(WINAPI *BD_LOCALE_LocOpenCat)(char *, char *, char *, unsigned int, char *) = NULL;

/**** Library definitions ****/

#define  LocCloseCat            BD_LOCALE_LocCloseCat
#define  LocGetCatStr           BD_LOCALE_LocGetCatStr
#define  LocGetErr              BD_LOCALE_LocGetErr
#define  LocGetErrStr           BD_LOCALE_LocGetErrStr
#define  LocIso2Dos             BD_LOCALE_LocIso2Dos
#define  LocOpenCat             BD_LOCALE_LocOpenCat

static HINSTANCE                hLocale;
#endif


/**** Initialize the localization ****/

ES_BOOL CatInit(char *Lang)
{
#ifdef AMIGA

  /**** Open the locale.library (only for AmigaOS) ****/
  /**** and the language catalog for translation   ****/

  if ((LocaleBase = (struct Library *)OpenLibrary("locale.library", 38))) {
    Catalog  = (void *)OpenCatalog(NULL, ES_CAT_FILE, OC_BuiltInLanguage,
               "english", OC_Version, ES_CAT_VERSION, TAG_DONE);
  } else return FALSE;
#else

#  ifdef WIN32
  if ((GetProcAddresses(&hLocale, BD_LOCALE_DLL, 6,
                        &BD_LOCALE_LocCloseCat       , "LocCloseCat",
                        &BD_LOCALE_LocGetCatStr      , "LocGetCatStr",
                        &BD_LOCALE_LocGetErr         , "LocGetErr",
                        &BD_LOCALE_LocGetErrStr      , "LocGetErrStr",
                        &BD_LOCALE_LocIso2Dos        , "LocIso2Dos",
                        &BD_LOCALE_LocOpenCat        , "LocOpenCat"
                       ) == FALSE))
    return FALSE;
#  endif

  /**** Open the catalog for language translation ****/

  GetPrgPath(Buf, FALSE);

  if ((Catalog = (void *)LocOpenCat(ES_CAT_FILE,
                                    Buf,
                                    Lang,
                                    ES_CAT_VERSION,
                                    "english")) == NULL) {
    if ((LocGetErr()) && (LocGetErr() != LOCERR_NOTFOUND) &&
        (LocGetErr() != LOCERR_AUTOWIN)) {
#  ifndef WIN32
      if (LocGetErr() == LOCERR_AUTONOTSUP) {
        fprintf(stderr, "Your operating system doesn't support the automatic language\n" \
                        "detection. Please set the LANGUAGE key in the \"esprefs\" file\n" \
                        "with your preferred language\n");
      } else {
#  endif
        LocGetErrStr(Buf, 255);
        fprintf(OutFH, ERROR_PROMPT, GetStr(MSG_ERROR), Buf);
#  ifndef WIN32
      }
#  endif
    }
  }
#endif

  return TRUE;
}


/**** Close the catalog ****/

void CatClose(void)
{
#ifdef AMIGA
  if (LocaleBase) {
    CloseCatalog(Catalog);
    CloseLibrary(LocaleBase);
  }
#else
  if (Catalog) LocCloseCat(Catalog);
#endif

#ifdef WIN32
  if (hLocale) FreeLibrary(hLocale);
#endif
}


/**** Shows an error ****/

ES_BOOL CatErr(ES_ULONG Err, ...)
{
  va_list       vl;

  sprintf(Buf, ERROR_PROMPT, GetStr(MSG_ERROR), GetStr(Err));

  va_start(vl, Err);
  if (OutFH != stdout) vfprintf(OutFH, Buf, vl);

#ifdef WIN32
  if (hLocale) LocIso2Dos(Buf, FALSE);
#endif

  vfprintf(stderr, Buf, vl);
  va_end(vl);

  return FALSE;
}


/**** fprintf replacement with string localization ****/

int CatPrintf(FILE *FH, ES_ULONG Str, ...)
{
  int           Ret;
  va_list       vl;

  va_start(vl, Str);

#ifdef WIN32
  if ((FH == stdout) || (FH == stderr)) {
    vsprintf(Buf, GetStr(Str), vl);
    if (hLocale) LocIso2Dos(Buf, FALSE);
    Ret = fprintf(FH, Buf, vl);
  } else Ret = vfprintf(FH, GetStr(Str), vl);
#else
  Ret = vfprintf(FH, GetStr(Str), vl);
#endif
  va_end(vl);

  return Ret;
}


/**** Show a warning ****/

ES_BOOL CatWarn(ES_ULONG Warn, ...)
{
  ES_BOOL       Ret;
  va_list       vl;

  sprintf(Buf, ERROR_PROMPT, GetStr(MSG_WARNING), GetStr(Warn));

#ifdef WIN32
    if ((OutFH == stdout) && (hLocale)) LocIso2Dos(Buf, FALSE);
#endif

  va_start(vl, Warn);
  if (vfprintf(OutFH, Buf, vl) <= 0) Ret = FALSE;
  else Ret = TRUE;
  va_end(vl);

  return Ret;
}


/**** Return a string and if possible, translate it ****/

char *GetStr(ES_ULONG Id)
{
  char                  *BuiltIn = "Unknown localized string";
  register ES_ULONG     k        = sizeof(ESCHER_Str) / sizeof(LOCARRAY);
  register LOCARRAY     *Ptr     = (LOCARRAY *)ESCHER_Str;

  while(k--) {
    if (Ptr -> Id == Id) {
      BuiltIn = Ptr -> Str;
      break;
    }
    ++Ptr;
  }

  /**** String localization ****/

  if (Catalog)
#ifdef AMIGA
    return((char *)GetCatalogStr((void *)Catalog, Id, BuiltIn));
#else
    return((char *)LocGetCatStr((void *)Catalog, Id, BuiltIn));
#endif


  return BuiltIn;
}


/**** Shows a system error ****/

ES_BOOL PrintDosErr(void)
{
  CatPrintf(stderr, MSG_ERROR_DOS, strerror(errno));
  if (OutFH != stdout)
    CatPrintf(OutFH, MSG_ERROR_DOS, strerror(errno));

  return FALSE;
}


