
/*********************************************************************
****                ESCHER Next Generation - Shapes               ****
**** (c) 1997-2012, G. Ausiello, G. Cesareni, M. Helmer Citterich ****
****               NG version by Alessandro Pedretti              ****
*********************************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include "my_macros.h"

#include "sol.h"
#include "main.h"
#include "mthread.h"
#include "prot.h"
#include "shapes.h"
#include "utils.h"

/**** Constants ****/

#define  areaxarea              (1.6f * 1.6f)
#define  lungmin1               5.0f
#define  lungmin2               5.0f
#define  LUR                    120
#define  lut                    3.15f
#define  MAX_HASH               2000
#define  mincurv                0.45f

/**** Definitions ****/

typedef struct _element {
  ES_LONG               sol;
  struct _element       *next;
} ES_ELEMENT;

/**** Local variables ****/

static float            raggio1, raggio2;
static ES_LONG          maxr, maxt, cubo_size;
static struct point     centro1, centro2;

static float    *gcos    = NULL;
static float    *gsin    = NULL;

/**** Global variables ****/

extern float    lunghezza_lato;
extern float    spessore_sezione;
extern int      col_attivo;

/**** Local prototypes ****/

static void     ShapeCompare(struct Proteina *, ES_LONG nf1, struct Proteina *, ES_LONG nf2, ES_CUBE *, ES_CUBE *);
static void     ShapeFind(int, struct Proteina *, struct Proteina *, ES_SOLUTIONS *, ES_CUBE *);
static void     ShapeInsert(struct Proteina *, struct Proteina *, int, int, int,
                            int, int, int, int, ES_CUBE *, ES_CUBE *);
static void     ZeroCube(ES_CUBE *, int);


/**** Free all resources ****/

void FreeDock(void)
{
  if (gcos) {
    FREE(gcos);
    gcos = NULL;
  }

  if (gsin) {
    FREE(gsin);
    gsin = NULL;
  }
}


/**** Initialization ****/

ES_BOOL InitDock(struct surf *s1, struct surf *s2, float lunghezza_lato,
                 ES_SOLUTIONS *s)
{
  ES_ULONG              i;

  /**** Sin and cos table ****/

  if ((gsin = Alloca(sizeof(float) * 3600 * 3)) != NULL) {
    if ((gcos = Alloca(sizeof(float) * 3600 * 3)) == NULL) {
      FreeDock();
      return FALSE;
    }
    for(i = 0; i < (3600 * 3); ++i) {
      gcos[i] = lunghezza_lato * cosf((M_PI / 1800.0)*(float)i);
      gsin[i] = lunghezza_lato * sinf((M_PI / 1800.0)*(float)i);
    } /* End of for */

    centro1   = surf_center(s1);
    centro2   = surf_center(s2);
    raggio1   = surf_radius_max(s1, centro1);
    raggio2   = surf_radius_max(s2, centro2);
    maxr      = 3600 / LUR;

/*
    Original expression:

    maxt      = 1 + (2 * raggio1 + 2 * raggio2) / lut;

    New expression:
*/

    maxt      = 1 + (int)ceil((2.0f * raggio1 + 2.0f * raggio2) / lut);
    cubo_size = (maxr + 1) * maxt * maxt;

    s -> centrox = centro2.x;
    s -> centroy = centro2.y;
    s -> centroz = centro2.z;
  } else return FALSE;

  return TRUE;
}


/**** Analyze the shape ****/

ES_BOOL ShapeAnalyze(struct Proteina *p1, struct Proteina *p2, ES_SOLUTIONS *s)
{
  ES_CUBE       *c, *cl;
  ES_LONG       i, k, nf1, nf2, ok;

  ES_BOOL       Ret   = TRUE;
  ES_LONG       rxtxt = cube(maxr);
  ES_LONG       txt   = square(maxt);

  if ((c = (ES_CUBE *)Alloca(cubo_size * sizeof(ES_CUBE))) != NULL) {
    if ((cl = (ES_CUBE *)Alloca(cubo_size * sizeof(ES_CUBE))) != NULL) {
      for(i = p2 -> n_poligonali; i >= - p1 -> n_poligonali; i--) {
        ok = 0;
        for(nf1 = 0, nf2 = i; nf1 < p1 -> n_poligonali; nf1++, nf2++) {
          if ((nf2 >= 0) && (nf2 < p2 -> n_poligonali)) {
            if ((p1 -> poligonale[nf1].n_vertici) &&
                (p2 -> poligonale[nf2].n_vertici)) {
              if (!ok) ZeroCube(c, cubo_size);
              ok++;
              ZeroCube(cl, cubo_size);
              ShapeCompare(p1, nf1, p2, nf2, c, cl);
              for(k = 0; k < txt; k++) {
                if (cl[rxtxt + k].val > cl[k].val) {
                  c[k].val += cl[rxtxt + k].val - cl[k].val;
            c[k].tx  += cl[rxtxt + k].tx  - cl[k].tx;
            c[k].ty  += cl[rxtxt + k].ty  - cl[k].ty;
            c[k].az  += cl[rxtxt + k].az  - cl[k].az - 3600;
            c[k].n   += cl[rxtxt + k].n   - cl[k].n;
                }
              } /* End of for */
            }
          }
        } /* End of for */
        if (ok) {
          MCL_MutexOn();
          ShapeFind(i, p1, p2, s, c);
          MCL_MutexOff();
        }
      } /* End of for */
      FREE(c);
    } else Ret = FALSE;
    FREE(cl);
  } else Ret = FALSE;

  return Ret;
}


/**** Compare the shapes ****/

static void ShapeCompare(struct Proteina* p1, ES_LONG nf1, struct Proteina* p2, ES_LONG nf2,
                         ES_CUBE *c, ES_CUBE *cl)
{
  ES_LONG       a1, a2, del, val1, val2;
  ES_LONG       i, j, k;
  float         dx, dy;

  ES_LONG       dot1 = p1 -> poligonale[nf1].n_vertici;
  ES_LONG       dot2 = p2 -> poligonale[nf2].n_vertici;
  float         *gc  = gcos;
  float         *gs  = gsin;

  struct Vertice        *v1 = p1->poligonale[nf1].vertice;
  struct Vertice        *v2 = p2->poligonale[nf2].vertice;


  for(j = dot1; j < (dot1 * 2); j++) {
    for(i = dot2; i < (dot2 * 2); i++) {
      if((v1[j+i].col == col_attivo) && (v2[i].col == col_attivo)) {
        del = v1[j + i].a - v2[i].a + 3600;
        k   = i;
        dx  = 0.0f;
        dy  = 0.0f;
        do {
          k++;
          a1  = v1[j + k].a;
          a2  = v2[k].a + del;
          dx += gc[a1] - gc[a2];
          dy += gs[a1] - gs[a2];
        } while((dx * dx + dy * dy) <= areaxarea);
        val1 = 1 + (k - i) - 1;
        k    = i;
        dx   = 0.0f;
        dy   = 0.0f;
        do {
          k--;
          a1  = v1[j + k].a;
          a2  = v2[k].a + del;
          dx += gc[a1] - gc[a2];
          dy += gs[a1] - gs[a2];
        } while((dx * dx + dy * dy) <= areaxarea);
        val2 = (i - k) + 1;
        if((val1 + val2) >= lungmin1)
          ShapeInsert(p1, p2, nf1, nf2, i + j, i, val1, val2, del, c, cl);
      }
    } /* End of for */
  } /* End of for */
}


/**** Find ****/

static void ShapeFind(int shift, struct Proteina *p1, struct Proteina *p2,
                      ES_SOLUTIONS *s, ES_CUBE *c)
{
  ES_ELEMENT            *nuovo, *old;
  int                   i, val, solfree;

  static ES_ELEMENT     ele[MAX_HASH];

  static int            minval = 100000;
  static ES_ULONG       n_ele  = 0;

  /**** Clear the element matrix ****/

  if (n_ele == 0) {
    for (i = 0; i < MAX_HASH; i++)
      ele[i].next = NULL;
  }

  for(i = 0; i < (maxr * maxt * maxt); i++) {
    if (((val = c[i].val) > 0) && ((n_ele < s -> n_sol) || (val > minval))) {
      solfree = n_ele;
      if (n_ele == s -> n_sol) {
        while((old = ele[minval].next) == NULL) minval++;
        ele[minval].next = old -> next;
        solfree          = old -> sol;
        FREE(old);
        n_ele--;
      }
      n_ele++;
      minval = min(val, minval);
      nuovo  = (ES_ELEMENT *)Alloca(sizeof(ES_ELEMENT));
      if (!nuovo) exit(1);

      nuovo -> sol  = solfree;
      nuovo -> next = ele[val].next;
      ele[val].next = nuovo;
      s -> sol[nuovo -> sol].ax  = s -> angolox;
      s -> sol[nuovo -> sol].ay  = s -> angoloy;
      s -> sol[nuovo -> sol].val = c[i].val;
      s -> sol[nuovo -> sol].az  = (int)((c[i].az / (float)c[i].n) / 10.0f);
      s -> sol[nuovo -> sol].tx  = c[i].tx / c[i].n;
      s -> sol[nuovo -> sol].ty  = c[i].ty / c[i].n;
      s -> sol[nuovo -> sol].tz  = p1 -> z_min - p2 -> z_min - shift * spessore_sezione;
    }
  } /* End of for */
}


/**** Insert a shape ****/

static void ShapeInsert(struct Proteina* p1, struct Proteina* p2, int nf1, int nf2,
                 int pos1, int pos2, int forw, int back, int del, ES_CUBE *c, ES_CUBE *cl)
{
  float         corr1, ix1, iy1, fx1, fy1, x2, x1, y2, y1, angolo, r;
  float         corr, corr2, iy2, ix2, fx2, fy2;
  int           a, x, y, ia, ix, iy;

  float         val  = forw + back;
  int           dot1 = p1 -> poligonale[nf1].n_vertici;
  int           dot2 = p2 -> poligonale[nf2].n_vertici;

  ix1   = p1 -> poligonale[nf1].vertice[pos1-back].x;
  fx1   = p1 -> poligonale[nf1].vertice[pos1+forw].x;
  iy1   = p1 -> poligonale[nf1].vertice[pos1-back].y;
  fy1   = p1 -> poligonale[nf1].vertice[pos1+forw].y;
  corr1 = sqrtf((ix1 - fx1) * (ix1 - fx1) + (iy1 - fy1) * (iy1 - fy1)) / (lunghezza_lato * val);
  ix2   = p2 -> poligonale[nf2].vertice[pos2-back].x;
  fx2   = p2 -> poligonale[nf2].vertice[pos2+forw].x;
  iy2   = p2 -> poligonale[nf2].vertice[pos2-back].y;
  fy2   = p2 -> poligonale[nf2].vertice[pos2+forw].y;
  corr2 = sqrtf((ix2 - fx2) * (ix2 - fx2) + (iy2 - fy2) * (iy2 - fy2)) / (lunghezza_lato * val);
  corr  = (corr1 + corr2) / 2.0f;

  if (corr) val = val / corr;
  else return;

  if ((corr > mincurv) && (val > lungmin2)) {
    x2     = p2 -> poligonale[nf2].vertice[pos2 % dot2].x - centro2.x;
    y2     = p2 -> poligonale[nf2].vertice[pos2 % dot2].y - centro2.y;
    x1     = p1 -> poligonale[nf1].vertice[pos1 % dot1].x;
    y1     = p1 -> poligonale[nf1].vertice[pos1 % dot1].y;
    r      = sqrtf(x2 * x2 + y2 * y2);
    angolo = (float)(del % 3600) * (M_PI / 1800.0f) + atan2f(y2, x2);
    x2     = r * cosf(angolo) + centro2.x;
    y2     = r * sinf(angolo) + centro2.y;
    x      = maxt * (int)((x1 - x2 - centro1.x + centro2.x + raggio1 + raggio2) / lut);
    y      = (int)((y1 - y2 - centro1.y + centro2.y + raggio1 + raggio2) / lut);
    a      = maxt * maxt * (int)((del % 3600) / LUR);

    for(ia = a; ia <= a + maxt * maxt; ia += maxt * maxt) {
      for(ix = ia + x; ix <= ia + x + maxt; ix += maxt) {
        for(iy = ix + y; iy <= ix + y + 1; iy++) {
          if (cl[iy].val < val) {
            c[iy].n    += 1 - cl[iy].n;
            c[iy].val  += (int)(val - cl[iy].val);
            c[iy].tx   += x1 - x2 - cl[iy].tx;
            c[iy].ty   += y1 - y2 - cl[iy].ty;
            c[iy].az   += del % 3600 - cl[iy].az;
            cl[iy].val  = (int)val;
            cl[iy].tx   = x1 - x2;
            cl[iy].ty   = y1 - y2;
            cl[iy].az   = del % 3600;
            cl[iy].n    = 1;
          }
        } /* End of for */
      } /* End of for */
    } /* End of for */
  }
}


/**** Reset the cube ****/

static void ZeroCube(ES_CUBE *c, int dim)
{
#ifdef WIN32
  ZeroMemory(c, dim * sizeof(ES_CUBE));
#else
  memset(c, 0, dim * sizeof(ES_CUBE));
#endif
}
