
/*********************************************************************
****              ESCHER Next Generation - Solutions              ****
**** (c) 1997-2012, G. Ausiello, G. Cesareni, M. Helmer Citterich ****
****               NG version by Alessandro Pedretti              ****
*********************************************************************/


#include "my_macros.h"

#include <time.h>

#include "globstr.h"
#include "sol.h"
#include "brk.h"
#include "utils.h"

/**** Global variables ****/

extern char     nome_p1[], nome_p2[];


/**** Obtain the solution coordinates ****/

ES_BRK *BrkSolutions(ES_BRK *Source, ES_SOLUTIONS *s, ES_ULONG NSol)
{
  ES_BRK        *BrkSol;

  if ((BrkSol = BrkNew()) != NULL) {
    brk_copy(BrkSol, Source);
    brk_translate(BrkSol, - s -> centrox, - s -> centroy, - s -> centroz);
    brk_x_rotate(BrkSol, (float)s -> sol[NSol].ax * M_PI / 180.0f);
    brk_y_rotate(BrkSol, (float)s -> sol[NSol].ay * M_PI / 180.0f);
    brk_z_rotate(BrkSol, (float)s -> sol[NSol].az * M_PI / 180.0f);
    brk_translate(BrkSol, s -> sol[NSol].tx + s -> centrox,
                  s -> sol[NSol].ty + s->centroy, s -> sol[NSol].tz + s->centroz);
  }

  return BrkSol;
}


/**** Create a new solution array ****/

ES_SOLUTIONS *SolutionsNew(ES_ULONG Size)
{
  ES_ULONG              i;
  ES_SOLUTIONS          *s;

  if ((s = (ES_SOLUTIONS *)Alloca(sizeof(ES_SOLUTIONS))) != NULL) {
    if ((s -> sol = (ES_SOLUTION *)Alloca(Size * sizeof(ES_SOLUTION))) != NULL) {
      s -> n_sol   = Size;
      s -> centrox = 0.0f;
      s -> centroy = 0.0f;
      s -> centroz = 0.0f;

      for(i = 0; i < s -> n_sol; i++) {
        s -> sol[i].val     = 0;
        s -> sol[i].ax      = 0;
        s -> sol[i].ay      = 0;
        s -> sol[i].az      = 0;
        s -> sol[i].tx      = 0.0f;
        s -> sol[i].ty      = 0.0f;
        s -> sol[i].tz      = 0.0f;
        s -> sol[i].rms     = 0.0f;
        s -> sol[i].bumps   = 0.0f;
        s -> sol[i].charge  = 0.0f;
        s -> sol[i].pos_neg = 0.0f;
        s -> sol[i].apo_apo = 0.0f;
        s -> sol[i].pos_pos = 0.0f;
        s -> sol[i].apo_pos = 0.0f;
      } /* End of for */
    } else {
     FREE(s);
     s = NULL;
    }
  }

  return s;
}


/**** Copy a single solution ****/

void SolutionCopy(ES_SOLUTION *dest, ES_SOLUTION *source)
{
  dest -> val     = source -> val;
  dest -> ax      = source -> ax;
  dest -> ay      = source -> ay;
  dest -> az      = source -> az;
  dest -> tx      = source -> tx;
  dest -> ty      = source -> ty;
  dest -> tz      = source -> tz;
  dest -> rms     = source -> rms;
  dest -> bumps   = source -> bumps;
  dest -> charge  = source -> charge;
  dest -> pos_neg = source -> pos_neg;
  dest -> apo_apo = source -> apo_apo;
  dest -> pos_pos = source -> pos_pos;
  dest -> apo_pos = source -> apo_pos;
}


/**** Copy all solutions ****/

ES_SOLUTIONS *SolutionsCopy(ES_SOLUTIONS *s)
{
  ES_SOLUTIONS          *s2;
  ES_ULONG              i;

  if ((s2 = SolutionsNew(s -> n_sol)) != NULL) {
    s2 -> n_sol   = s -> n_sol;
    s2 -> centrox = s -> centrox;
    s2 -> centroy = s -> centroy;
    s2 -> centroz = s -> centroz;
    for(i = 0; i < s -> n_sol; i++)
      SolutionCopy(&s2->sol[i], &s->sol[i]);
  }

  return s2;
}


/**** Delete the solution array ****/

void SolutionsDelete(ES_SOLUTIONS **s)
{
  if ((*s) -> sol) FREE((*s) -> sol);
  FREE(*s);
  *s = NULL;
}


/**** Read the solutions ****/

ES_SOLUTIONS *SolutionsRead(char *FileName)
{
  char                  Buf[ES_LINELEN + 1];
  char                  *Ptr;
  ES_ULONG              i;
  FILE                  *FH;

  ES_ULONG              Line = 0;
  ES_ULONG              Size = 0;
  ES_SOLUTIONS          *s   = NULL;
  ES_BOOL               Ret  = TRUE;

  if ((FH = fopen(FileName, "r")) != NULL) {
    while((Ret) && (fgets(Buf, ES_LINELEN, FH))) {
      if (++Line == 1) {
        if (strncmp(Buf, "#ESCHERNG_VER", 13))
          Ret = CatErr(MSG_ERR_SOL_UNKFORMAT);
      } else if (!strncmp(Buf, "#DOCKING_INFO", 13)) {
        if ((Ret = SkipLines(FH, Buf, 3)) == TRUE) {
          if ((Ptr = strchr(Buf, ':')) != NULL) {
            sscanf(Ptr + 1, "%d", &Size);
            if (Size > 0) {
              if ((s = SolutionsNew(Size)) != NULL) {
                if ((fgets(Buf, ES_LINELEN, FH)) && ((Ptr = strchr(Buf, ':')) != NULL)) {
                  sscanf(Ptr + 1, "%f %f %f", &s -> centrox, &s -> centroy, &s -> centroz);
                } else Ret = CatErr(MSG_ERR_SOL_CORRFILE);
              }
            } else Ret = CatErr(MSG_ERR_SOL_NUMSOL);
          } else Ret = CatErr(MSG_ERR_SOL_CORRFILE);
        }
      } else if (!strncmp(Buf, "#SOLUTIONS", 10)) {
        if ((s) && ((Ret = SkipLines(FH, Buf, 2)) == TRUE)) {
          i = 0;
          while((Ret) && (fgets(Buf, ES_LINELEN, FH)) && (*Buf != '#')) {
            if (i < Size) {
              sscanf(Buf, "%*s %d %f %f %f "
                          "%f %f %f %f "
                          "%4d %4d %4d "
                          "%f %f %f\n",
                          &s -> sol[i].val, &s -> sol[i].rms, &s -> sol[i].bumps, &s -> sol[i].charge,
                          &s -> sol[i].pos_neg, &s -> sol[i].pos_pos, &s -> sol[i].apo_apo, &s -> sol[i].apo_pos,
                          &s -> sol[i].ax, &s -> sol[i].ay, &s -> sol[i].az,
                          &s -> sol[i].tx, &s -> sol[i].ty, &s -> sol[i].tz);
              ++i;
            } else Ret = CatErr(MSG_ERR_SOL_CORRFILE);
          } /* End of while */
        } else Ret = CatErr(MSG_ERR_SOL_CORRFILE);
      }
    } /* End of while */
    if ((!Ret) && (s)) SolutionsDelete(&s);
    fclose(FH);
  } else PrintDosErr();

  return s;
}


/**** Sort the solutions ****/

ES_SOLUTIONS *SolutionsSort(ES_SOLUTIONS *s)
{
  ES_ULONG              i, j;
  ES_SOLUTION           Tmp, *best, *csol, *dest;
  ES_SOLUTIONS          *s2;
  float                 maxcharge, maxrms, minbumps;
  ES_LONG               maxval;

  if ((s2 = SolutionsCopy(s)) != NULL) {
    dest = s2 -> sol;
    for(i = 0; i < s -> n_sol; i++) {
      best      = dest;
      maxval    = dest -> val;
      maxcharge = dest -> charge;
      maxrms    = dest -> rms;
      minbumps  = dest -> bumps;
      csol      = dest + 1;
      for(j = i + 1; j < s -> n_sol; j++) {
        if ((maxval < csol -> val) ||
            ((maxval == csol -> val) && (maxcharge < csol -> charge)) ||
            ((maxval == csol -> val) && (maxcharge == csol -> charge) && (minbumps > csol -> bumps)) ||
            ((maxval == csol -> val) && (maxcharge == csol -> charge) && (minbumps == csol -> bumps) && (maxrms < csol -> rms))) {
          maxval    = csol -> val;
          maxcharge = csol -> charge;
          maxrms    = csol -> rms;
          minbumps  = csol -> bumps;
          best      = csol;
        }
        ++csol;
      } /* End of for */
      SolutionCopy(&Tmp, dest);
      SolutionCopy(dest, best);
      SolutionCopy(best, &Tmp);
      ++dest;
    } /* End of for */
  }

  return s2;
}


/**** Write the solutions ****/

ES_BOOL SolutionsWrite(ES_SOLUTIONS *s, char *FileName)
{
  ES_ULONG      i;
  FILE          *FH;

  ES_BOOL       Ret = TRUE;

  if ((FH = fopen(FileName, "w")) != NULL) {
    i = time(NULL);
    if (fprintf(FH, "#ESCHERNG_VER " ES_OUT_VERSTR "\n"
                    "\n%s\n\n"
                    "#DOCKING_INFO\n"
                    "Target file name..: \"%s.pdb\"\n"
                    "Probe file name...: \"%s.pdb\"\n"
                    "Solutions.........: %d\n"
                    "Center (x, y, z)..: %f %f %f\n"
                    "#END\n\n"
                    "#SOLUTIONS\n"
                    "Sol. Score   Rms Bumps Chg. Pos. Neg. Apo. Pol. RotX RotY RotZ TransX TransY TransZ\n"
                    "===================================================================================\n",
                    ctime((time_t *)&i),
                    nome_p1, nome_p2,
                    s -> n_sol,
                    s -> centrox, s -> centroy, s -> centroz) > 0) {

        for(i = 0; (Ret) && (i < s -> n_sol); i++) {
          if (fprintf(FH, "%4d %5d %5.1f %5.0f %4.0f "
                          "%4.0f %4.0f %4.0f %4.0f "
                          "%4d %4d %4d "
                          "%6.1f %6.1f %6.1f\n",
                          i + 1, s -> sol[i].val, s -> sol[i].rms, s -> sol[i].bumps, s -> sol[i].charge,
                          s -> sol[i].pos_neg, s -> sol[i].pos_pos, s -> sol[i].apo_apo, s -> sol[i].apo_pos,
                          s -> sol[i].ax, s -> sol[i].ay, s -> sol[i].az,
                          s -> sol[i].tx, s -> sol[i].ty, s -> sol[i].tz) <= 0) {
            Ret = PrintDosErr();
          }
        } /* End of for */
      if (fprintf(FH, "#END\n") <= 0) Ret = PrintDosErr();
    } else Ret = PrintDosErr();
    fclose(FH);
  } else Ret = PrintDosErr();

  return Ret;
}



